/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developpers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
/*!
 *\file Render_area.cpp
 */
 

#include "Render_area.hpp"
#include "../interface/interface_console.hpp"

#include "Render_opengl.hpp"


using namespace std;


#define message *Message::get_instance() << "(" << __FILE__ << " l." << __LINE__ << "): "


const int Render_area::DRAWING_STATE_WAITING = 1;
const int Render_area::DRAWING_STATE_DRAWING = 2;
const int Render_area::_ctrl_scroll_factor = 10;
const int Render_area::_ctrl_zoom_factor = 3;
const int Render_area::DRAWING_TIMER_DEFAULT = 10;



/***********************************
 *
 *
 *
 * Constructor and destructor.
 *
 *
 *
 **********************************/

Render_area::Render_area(Render_opengl* render_instance, Interface_console* core, QWidget *parent)
    : QGLWidget(parent){    
    

    //  GLenum glew_code;


    if (!QGLFormat::hasOpenGL()){
        QMessageBox::information(0, "ViTE: Fatal OpenGL error","This system does not support OpenGL.");
    }

   //  makeCurrent();/* need to "enable" glew */

//     glew_code = glewInit();/* use for VBO and shaders */
    
//     if(GLEW_OK != glew_code){
//         message << "Cannot initialize GLEW: " << glewGetErrorString(glew_code) << Message::ende;
//     }
    
//     /*Check if VBO is supported */
//     if (GL_FALSE == glewIsSupported("GL_ARB_vertex_buffer_object")){
//         message << "VBO OpenGL extension is not supported by your graphic card." << Message::ende;
//     }   


    _render_instance = render_instance;    
    _core = core;
    
    
    _state = DRAWING_STATE_WAITING;/* At the beginning, no trace is drawing */

    /* init the wait animation */
    _wait_list    = 0;
    _wait_angle   = 0.0f; /* begin with 0 rad angle */
    _wait_angle_y = 0.0f; /* begin with 0 rad angle */
    _wait_timer   = NULL;
    _wait_spf     = DRAWING_TIMER_DEFAULT;/* DRAWING_TIMER_DEFAULT milliseconds per frame */
    
    _counter_last_x = 0.0;
    _counter_last_y = 0.0;

    _line_already_begun = false;


    /* init main informations about OpenGL scene and QGLWidget size */

    _screen_width  = width();  /* get the QGLWidget visibled width (in pixel)  */
    _screen_height = height(); /* get the QGLWidget visibled height (in pixel) */

    _render_width  = 100; /* 100 OpenGL units for the render scene width  */
    _render_height = 100; /* 100 OpenGL units for the render scene height */

    _x_scale_container_state = 0.2;/* 20% of screen is used for containers then the other part for states */

    _ruler_height = 4.0; /* height of the ruler   */
    _ruler_y      = 0.0; /* highness of the ruler */

    _default_entity_x_translate =  20;/* _render_width * _x_scale_container_state */

    _container_x_max = 0;
    _container_y_max = 0;
    _container_x_min = 0;
    _container_y_min = 0;

    _state_x_max = 0;
    _state_y_max = 0;
    _state_x_min = 0;
    _state_y_min = 0;

    _ruler_distance = 5;

    _mouse_x = 0;
    _mouse_y = 0;
    _new_mouse_x = 0;
    _new_mouse_y = 0;
    _mouse_pressed = false;
    _key_scrolling = false;

    _minimum_distance_for_selection = 5;/* 5 pixels */

    _selection_rectangle_alpha = 0.5;


    /* Info init */
    // Info::Render::_key_alt = false;/* no CTRL key pushed */

    _list_ruler      = 0;

    _x_state_scale     = 1;/* for states scaling */
    _x_state_translate = 0;/* for states translation */
    _y_state_scale     = 1;/* for states scaling */
    _y_state_translate = 0;/* for states translation */
    _x_scroll_pos      = 0;/* horizontal bar placed on 0 */
    _y_scroll_pos      = 0;/* vertical bar placed on 0 */
  
    /* Camera is placed on (0,0,0) and looks to (0,0,-1) */

    _z_ruler_over      = -0.5; 
    _z_container       = -1.0;
    _z_container_under = -1.2;
    _z_ruler           = -1.5;
    _z_ruler_under     = -1.6;
    _z_arrow           = -2.0;/* closer to camera than containers or states (MUST be negative)*/
    _z_event           = -3.0;/* closer to camera than containers or states (MUST be negative)*/
    _z_counter         = -4.0;    
    _z_state           = -5.0;

    setAutoFillBackground(false);

    setMouseTracking (true);/* to catch mouse events */
    setFocusPolicy(Qt::StrongFocus);/* to catch keybord events (focus by tabbing or clicking) */

    //_render_instance->updateGL();
   
}


Render_area::~Render_area(){

    /* Release the Rabbit and ruler lists */
    if (glIsList(_wait_list) == GL_TRUE)
        glDeleteLists(_wait_list, 1);

    if (glIsList(_list_ruler) == GL_TRUE)
        glDeleteLists(_list_ruler, 1);

    /* Release timer (for Rabbit rotate) */
    if (_wait_timer != NULL){
        delete _wait_timer;
        _wait_timer=NULL;
    }
}




/***********************************
 *
 *
 *
 * Scaling and scrolling functions.
 *
 *
 *
 **********************************/
     


void Render_area::mousePressEvent(QMouseEvent * event){
 
    /* If a right click was triggered, just restore the previous zoom */
    if( Qt::RightButton == event->button() ){

        /* Just a special case: if a zoom box is currently drawing, the right click is used
           to cancel the drawing */
        if ( true == _mouse_pressed){
            _mouse_pressed = false;
            _render_instance->updateGL();
            return;
        }

        /* if there is no previous zoom box registered, return */
        if (true == _previous_selection.empty())
            return;

        /* restore the previous values */
        _x_state_scale = _previous_selection.top().x_scale;
        _y_state_scale = _previous_selection.top().y_scale;
        _x_state_translate = _previous_selection.top().x_translate;
        _y_state_translate = _previous_selection.top().y_translate;


        /* remove the previous value */
        _previous_selection.pop();
	
	refresh_scroll_bars(true);
       _render_instance->updateGL();
	

#ifdef DEBUG_MODE_RENDER_AREA_
             
        /* if there is no previous zoom box registered, return */
        if (true == _previous_selection.empty())
            return;

        cerr << __FILE__ << " l." << __LINE__ << ":" << endl;  
        cerr << _previous_selection.top().x_scale << " " 
             << _previous_selection.top().y_scale << " "
             << _previous_selection.top().x_translate << " "
             << _previous_selection.top().y_translate <<  endl << endl;
#endif

        int buf;

        buf = (int)100*_x_state_scale;
        _core->launch_action(_core->_STATE_ZOOM_BOX_VALUE, &buf);

        /* Then, return */
        return;
    }

    /* else, registered current X and Y mouse position to draw a zoom box */
   
    _mouse_x = event->x();
    _new_mouse_x = _mouse_x;
    _mouse_y = event->y();
    _new_mouse_y = _mouse_y;
    _mouse_pressed = true;
}


void Render_area::mouseMoveEvent(QMouseEvent * event){
    if (!_mouse_pressed) return;

    _new_mouse_x = event->x();
    _new_mouse_y = event->y();

    _render_instance->updateGL();
}


void Render_area::mouseReleaseEvent(QMouseEvent * event){
    Element_pos x_click, y_click;/* Store the OpenGL scene point where user had clicked */
    Element_pos x_result, y_result;/* The click coordinates for the Data Structure. */
    Element_pos invert_buf_x;
    Element_pos invert_buf_y;

    Selection_ selection_stack_buf;


    /* If a right click was triggered, just restore the previous zoom */
    if( Qt::RightButton == event->button() )
        return;

    if (_new_mouse_x < _mouse_x){
        invert_buf_x = _mouse_x;
        _mouse_x = _new_mouse_x;
        _new_mouse_x = invert_buf_x;
    }
    
    if (_new_mouse_y < _mouse_y){
        invert_buf_y = _mouse_y;
        _mouse_y = _new_mouse_y;
        _new_mouse_y = invert_buf_y;
    }

    /*
     * When the mouse is released:
     *
     * First, check if there is a significant difference between mouse coordinates. Prevent bad manipulations.
     */

    if(false==_mouse_pressed)
        return;

    if (((_new_mouse_x-_mouse_x) < _minimum_distance_for_selection)
        && ((_new_mouse_y-_mouse_y) < _minimum_distance_for_selection)) {/* selection is too thin to draw a box. So, it must be a user click to displau entity information */

        /* 
         * Now determine where the user had clicked 
         */
        x_click = _mouse_x * (Element_pos)_render_width/(Element_pos)_screen_width;/* x position of the click in the OpenGL scene */
        y_click = _mouse_y * (Element_pos)_render_height/(Element_pos)_screen_height;/* y position of the click in the OpenGL scene */
        
        /* 
         * First, check the y position if user clicks on the ruler or other 
         */
        if (y_click <= (_ruler_height+_ruler_y))/* user had clicked on the ruler */
            return;
     
        /* Work out the y value in Data Structure coordinates */
        y_result = y_click-(_ruler_y+_ruler_height) + _y_state_translate;
        y_result *= (_container_y_max-_container_y_min)/((_render_height-_ruler_height)*_y_state_scale);
        
        /* 
         * Second, check the x position if user click on the container or other 
         */
        if (x_click <= (_x_scale_container_state*_render_width) ){/* user had clicked on the container */
            return;
        }
        else{
            /* 
             * Determine the time corresponding to the x click position.
             * It is equal to the x click position in the OpenGL render area minus the default translation of entities.
             * Also, near the container area time is 0 except if a translation had occured.
             * So add the state translate and the x_result will be obtained.
             *
             * Take care that x_result can be negative.
             */
            x_result = (x_click - _default_entity_x_translate + _x_state_translate); 
            
            /*
             * To convert from [0;_render_width*_state_scale] to [_state_x_min;_state_x_max]
             */
            x_result *= (_state_x_max - _state_x_min)/((_render_width-_default_entity_x_translate)*_x_state_scale);
            
            Info::Render::_info_x = x_result;
            Info::Render::_info_y = y_result;
            Info::Render::_info_accurate = (_state_x_max - _state_x_min)/((_render_width-_default_entity_x_translate)*_x_state_scale);//_state_x_max/_render_width;
            _core->launch_action(_core->_STATE_RENDER_DISPLAY_INFORMATION, NULL);
        }/* end else of if (x_click...) */

        _mouse_pressed = false;
        _render_instance->updateGL();
        return;
    }

    /*
     * Now, user was drawing a box. Zoom in it!
     */

    /* Now, register this position which will be used by a right clic */
    selection_stack_buf.x_scale = _x_state_scale;
    selection_stack_buf.y_scale = _y_state_scale;
    selection_stack_buf.x_translate = _x_state_translate;
    selection_stack_buf.y_translate = _y_state_translate;

    _previous_selection.push(selection_stack_buf);

#ifdef DEBUG_MODE_RENDER_AREA_
             
        cerr << __FILE__ << " l." << __LINE__ << ":" << endl;  
        cerr << _previous_selection.top().x_scale << " " 
             << _previous_selection.top().y_scale << " "
             << _previous_selection.top().x_translate << " "
             << _previous_selection.top().y_translate << endl << endl;
#endif


    apply_zoom_box(_mouse_x, _new_mouse_x, _mouse_y, _new_mouse_y);
}



void Render_area::apply_zoom_box(Element_pos x_min, Element_pos x_max, Element_pos y_min, Element_pos y_max){

    Element_pos x_middle_selection;
    Element_pos y_middle_selection;
    Element_pos x_scaled_middle_selection;
    Element_pos y_scaled_middle_selection;
    Element_pos x_distance_between_state_origin_and_render_middle;
    Element_pos y_distance_between_state_origin_and_render_middle;


    /*
     * Now, we try to zoom on the selection rectangle. To perform this, the left of the selection rectangle must be fit with the left
     * of the render area. Idem for the right, the top and the bottom.
     *
     * Thus, we need to know the difference the scale between the rectangle width and height and the render area width and height.
     * Results are given by the scale: _screen_width/(x_max - x_min) for the horizontal dimension.
     *
     * Then, our selection rectangle is scaled. Nevertheless, it should not be centered inside the render area. So, we work out
     * the correct x and y translation to make both the render area middle and the selection rectangle middle fit.
     * There are 3 steps for the middle x position :
     *  - First, find distance between the scaled selection rectangle middle and the state origin.
     *  - Second, find the distance between the state origin and the render area middle.
     *  - Finally, translate the scaled selection middle to the difference.
     */


    /*
     * Work out the horizontal middle of the selection area (the selection area width divide by 2 plus the x left coordinate)
     * Note: mouse positions are converted to render area coordinate with the coeff _render_width/_screen_width.
     */
    x_middle_selection = (x_min + (x_max - x_min)/2 )*(_render_width/_screen_width);
    y_middle_selection = (y_min + (y_max - y_min)/2 )*(_render_height/_screen_height);
   

    /*
     * 1st step:
     *
     * Work out the new selection middle position after applying the scale.
     */
    x_scaled_middle_selection = (x_middle_selection + _x_state_translate - _default_entity_x_translate)*(_screen_width/((x_max - x_min)));
    y_scaled_middle_selection = (y_middle_selection + _y_state_translate - _ruler_height )*(_screen_height/((y_max - y_min)));

    /*
     * 2nd step:
     *
     * Work out the distance between the state origin and the render area middle (_render_width/2).
     */
    x_distance_between_state_origin_and_render_middle =  _x_state_translate - _default_entity_x_translate + _render_width/2;
    y_distance_between_state_origin_and_render_middle =  _y_state_translate /*- _ruler_height*/ + _render_height/2;

    /*
     * 3rd step:
     *
     * Translate entities.
     */
    _x_state_translate +=  x_scaled_middle_selection -  x_distance_between_state_origin_and_render_middle;
    _y_state_translate +=  y_scaled_middle_selection -  y_distance_between_state_origin_and_render_middle;


    /* 
     * Finally, perform the scale. 
     */
   
     /* NOTE: do not use replace_scale() because the translate will be also modified */
     int buf;

     _x_state_scale *= _screen_width/(x_max - x_min);
     buf = (int)100*_x_state_scale;
     _core->launch_action(_core->_STATE_ZOOM_BOX_VALUE, &buf);


    _y_state_scale *= _screen_height/(y_max - y_min);

    _mouse_pressed = false;
    _render_instance->updateGL();
    refresh_scroll_bars(true);
}


void Render_area::wheelEvent(QWheelEvent * event){
     int num_degrees = event->delta() / 8;
     int num_steps = num_degrees / 15;
     Element_pos scale_coeff = num_steps;

     int ctrl_factor = 1;
     
     if (event->modifiers() == Qt::CTRL) ctrl_factor *= _ctrl_zoom_factor;

     //   cerr << Info::Render::_key_alt << endl;
     if (true ==  Info::Render::_key_alt){/* Zoom on height */
         change_scale_y(scale_coeff);
     }else{ /* Zoom on time */
         change_scale(scale_coeff*ctrl_factor);
     }
     
    event->accept();/* accept the event */
}


void Render_area::keyPressEvent(QKeyEvent * event) {
    
    int ctrl_factor = 1;
    
    if (event->key() == Qt::Key_Alt) {
	//    cerr << "Push" << endl;
        Info::Render::_key_alt = true;
    }
    
    if (event->modifiers() == Qt::CTRL) {
        ctrl_factor *= _ctrl_scroll_factor;
	Info::Render::_key_ctrl = true;
    }
    
    switch (event->key()) {
    case Qt::Key_Left:
        /*
         * Key 'left' pressed.
         */
        change_translate(-1*ctrl_factor);
	break;
    case Qt::Key_Right:
        /*
         * Key 'right' pressed.
         */
        change_translate(1*ctrl_factor);
	break;
    case Qt::Key_Up:
	/*
         * Key 'up' pressed.
         */
        change_translate_y(-1*ctrl_factor);
	break;
    case Qt::Key_Down:
	/*
         * Key 'down' pressed.
         */
        change_translate_y(1*ctrl_factor);
	break;
    case Qt::Key_PageUp:
	/*
         * Key 'Page Up' pressed.
         */
        if (true == Info::Render::_key_alt)
            change_translate_y(-_render_height);
        else
            change_translate(_render_width);
        break;
    case Qt::Key_PageDown:
	/*
         * Key 'Page Down' pressed.
         */
        if (true == Info::Render::_key_alt)
            change_translate_y(_render_height);
        else
            change_translate(-_render_width);
        break;
	
    default:
	/*
	 * Unknow key pressed.
	 */
	break;
    }
    
    _key_scrolling = true;
    event->accept();/* accept the event */
}


void Render_area::keyReleaseEvent(QKeyEvent * event){
    if (event->key() == Qt::Key_Alt) {
        Info::Render::_key_alt = false;
    }
    
    if (event->key() == Qt::Key_Control) {
	Info::Render::_key_ctrl = false;
    }
}



void  Render_area::change_scale(Element_pos scale_coeff){
    Element_pos new_scale;

    new_scale = _x_state_scale*(1+scale_coeff*0.05);/* 5% scale */
 
    if (new_scale<1.0)
        replace_scale(1.0);
    else
        replace_scale(new_scale);
}

void  Render_area::change_scale_y(Element_pos scale_coeff){

    Element_pos new_scale;
    
    new_scale = _y_state_scale*(1+scale_coeff*0.05);/* 5% scale */

    if (new_scale<1.0)
        _y_state_scale = 1.0;
    else{
        _y_state_translate += ( _y_state_translate - _ruler_height + (Element_pos)_render_height/2.0) * ((new_scale)/(_y_state_scale) - 1);
        _y_state_scale = new_scale;
        
	        

        _render_instance->updateGL();
	refresh_scroll_bars(true);
    }
}


void  Render_area::replace_scale(Element_pos new_scale){
    int buf;

    if(new_scale > 0.0){
        /* 
         * Reajust the entity translation to recenter the same point previously under the mouse pointer 
         *
         * Work out the distance between the point in the middle Widget and the state origin.
         * Then multiply it by the scale coefficient ( (new_scale / old_scale) ) less 1. 
         * 1 corresponding to the original translation of the point from the state origin.
         * Finally, translate the result.
         *
         */
        _x_state_translate += ( _x_state_translate - _default_entity_x_translate + (Element_pos)_render_width/2.0) * ((new_scale)/(_x_state_scale) - 1);
        _x_state_scale = new_scale;

        buf = (int)100*_x_state_scale;
	
        _core->launch_action(_core->_STATE_ZOOM_BOX_VALUE, &buf);
	
        _render_instance->updateGL();
	refresh_scroll_bars(true);
    }
}


void Render_area::change_translate(int translate){
    replace_translate(_x_state_translate + translate);
}


void Render_area::change_translate_y(int translate){
    replace_translate_y(_y_state_translate + translate);
}


void Render_area::replace_translate(Element_pos new_translate){
    if ( new_translate < 0 ){
	new_translate = 0;
    }else if ( new_translate > (100*_x_state_scale) ){
	new_translate = 100*_x_state_scale;
    }
    
    _x_state_translate = new_translate;
     
    if (_key_scrolling){
	refresh_scroll_bars();
	_key_scrolling = false;
    }
    _render_instance->updateGL();
}


void Render_area::replace_translate_y(Element_pos new_translate){
    if ( new_translate < 0 ){
	new_translate = 0;
    }else if ( new_translate > (100*_y_state_scale) ){
	new_translate = 100*_y_state_scale;
    }
    
    _y_state_translate = new_translate;
     
    if (_key_scrolling){
	refresh_scroll_bars();
	_key_scrolling = false;
    }
    _render_instance->updateGL();
}


void Render_area::registered_translate(int id){
   
    switch (id){

    case Info::Render::X_TRACE_BEGINNING:/* show the beginning entities */
        _x_state_translate =  _default_entity_x_translate - (Element_pos)_render_width/2.0;
        break;
    case Info::Render::Y_TRACE_BEGINNING:/* show the beginning entities */
        _y_state_translate = (Element_pos)_render_height/2.0 - _ruler_height;
        break;
    case Info::Render::X_TRACE_ENDING:/* show the ending entities */
        _x_state_translate =  (_default_entity_x_translate - (Element_pos)_render_width/2.0 + _state_x_max*_x_state_scale
                             *((_render_width-_default_entity_x_translate)/(_state_x_max-_state_x_min)));
        break;
    case Info::Render::Y_TRACE_ENDING:/* show the ending entities */
        _y_state_translate =  ( -_ruler_height + (Element_pos)_render_height/2.0 - _state_y_max*_y_state_scale
                                *((_render_height-_ruler_height)/(_state_y_max-_state_y_min)));
        break;
    case Info::Render::X_TRACE_ENTIRE:/* show the entire entities */
        _x_state_translate = 0;
        _x_state_scale = 1;    
        break;
    case Info::Render::Y_TRACE_ENTIRE:/* show the entire entities */
        _y_state_translate = 0;
        _y_state_scale = 1;  
        break;
    default:
        message << "Unknown registered translate" << Message::endw;
    }
    
    refresh_scroll_bars();
    _render_instance->updateGL();
}


void Render_area::refresh_scroll_bars(bool LENGTH_CHANGED){
    if (LENGTH_CHANGED){
	Element_pos scroll_bar_length[2] = {_x_state_scale*100, _y_state_scale*100};
	_core->launch_action(_core->_STATE_AJUST_SCROLL_BARS, scroll_bar_length);
    }

    _x_scroll_pos = _x_state_translate;
    _y_scroll_pos = _y_state_translate;
    Element_pos buf[2] = {_x_scroll_pos, _y_scroll_pos};
    _core->launch_action(_core->_STATE_REFRESH_SCROLL_BARS, buf);
}


void Render_area::change_scale_container_state(int view_size){
    
    _x_scale_container_state = 0.01 * view_size;
    _render_instance->updateGL();
}

void Render_area::update_visible_interval_value(){
    Info::Render::_x_min_visible = max(_state_x_min, (_x_state_translate - _default_entity_x_translate)/(_x_state_scale*((_render_width-_default_entity_x_translate)/(_state_x_max-_state_x_min)) ));
    Info::Render::_x_max_visible = min(_state_x_max, (_x_state_translate - _default_entity_x_translate + _render_width)/(_x_state_scale*((_render_width-_default_entity_x_translate)/(_state_x_max-_state_x_min)) ));
}




 /***********************************
  *
  *
  *
  * Coordinate convert functions.
  *
  *
  *
  **********************************/


Element_pos Render_area::screen_to_render_x(Element_pos e){

    return e * (Element_pos)_render_width / (Element_pos)_screen_width;
}

Element_pos Render_area::screen_to_render_y(Element_pos e){

    return e *  (Element_pos)_render_height / (Element_pos)_screen_height;
}

Element_pos Render_area::render_to_trace_x(Element_pos e){
    
    return (e - _default_entity_x_translate + _x_state_translate) *
        (_state_x_max - _state_x_min) / ((_render_width - _default_entity_x_translate) * _x_state_scale);
}

Element_pos Render_area::render_to_trace_y(Element_pos e){

    return  (e-_ruler_y-_ruler_height + _y_state_translate) *
        (_container_y_max - _container_y_min) / ((_render_height - _ruler_height) * _y_state_scale);
}

Element_pos Render_area::render_to_screen_x(Element_pos e){

    return e * (Element_pos)_screen_width / (Element_pos)_render_width;
}

Element_pos Render_area::render_to_screen_y(Element_pos e){

    return e *  (Element_pos)_screen_height / (Element_pos)_render_height;
}

Element_pos Render_area::trace_to_render_x(Element_pos e){

    return e *  ((_render_width-_default_entity_x_translate)*_x_state_scale) / (_state_x_max - _state_x_min) +
            _default_entity_x_translate - _x_state_translate;
}

Element_pos Render_area::trace_to_render_y(Element_pos e){

    return  e * ((_render_height-_ruler_height)*_y_state_scale) / (_container_y_max-_container_y_min) +
             _ruler_y + _ruler_height - _y_state_translate;
}




/***********************************
 *
 *
 *
 * Building functions.
 *
 *
 *
 **********************************/



bool Render_area::build(){
    
    _state = DRAWING_STATE_DRAWING;/* change the drawing state */

    /* disable some OpenGL features to enhance the rendering */
    glDisable(GL_TEXTURE_2D);
    glDisable(GL_BLEND);

    replace_scale(1);/* for states scaling */
    _x_state_translate = 0;/* for states translation */
    _y_state_scale = 1;/* for states scaling */
    _y_state_translate = 0;/* for states translation */
    _x_scroll_pos = 0;/* horizontal bar placed on 0 */
    _y_scroll_pos = 0;/* vertical bar placed on 0 */
 
    if (NULL == _render_instance)
        return true;

    return _render_instance->display_build();
}



bool Render_area::unbuild(){
     
    /**********************
     *
     * Init OpenGL features
     *
     **********************/

    /* enable some OpenGL features*/
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   
    glEnable(GL_TEXTURE_2D);

    glEnable(GL_BLEND);/* enable blending for the alpha color */ 
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

    glColor3d(1.0, 1.0, 1.0);/* init color to white */


    /*****************************
     *
     * Init render area attributes
     *
     *****************************/

    _state = DRAWING_STATE_WAITING;/* change the drawing state */

    _wait_angle=0.0f;/* begin with 0 rad angle */
    _wait_angle_y=0.0f;/* begin with 0 rad angle */

    /* init measurement attributes */    
    _container_x_max = 0;
    _container_y_max = 0;
    _container_x_min = 0;
    _container_y_min = 0;

    _state_x_max = 0;
    _state_y_max = 0;
    _state_x_min = 0;
    _state_y_min = 0;
    
    _line_already_begun = false;

    /* empty the selection stack */
    while(false == _previous_selection.empty())
        _previous_selection.pop();

    if (NULL == _render_instance)
        return true;

    return _render_instance->display_unbuild();
}





/***********************************
 *
 *
 *
 * Drawing function for the wait screen.
 *
 *
 *
 **********************************/



GLuint Render_area::draw_wait() {
    GLuint object;
    GLuint texture;
   
    object = glGenLists(1);/* create the list */

    if (object == 0)
        message << "Error when creating list" << Message::endw;


    glGenTextures(1, &texture);/* create the texture and link it with the list previously created */   
    
    QFile texture_file(QString(":/img/img/logo") +  QDate::currentDate().toString("MMdd") + QString(".png"));
  
    if (true == texture_file.exists())/* The texture exists */
        texture = bindTexture(QPixmap(texture_file.fileName()), GL_TEXTURE_2D);
    else/* use the default picture */
        texture = bindTexture(QPixmap(":/img/img/logo.png"), GL_TEXTURE_2D);

    glNewList(object, GL_COMPILE);/* open the list */
    {
        glBindTexture(GL_TEXTURE_2D, texture);/* load texture for drawing */
     
        glBegin(GL_QUADS);/* draw a square */
        {
            glTexCoord2d(0,0); glVertex2f(-1, -1);
            glTexCoord2d(1,0); glVertex2f(1, -1);
            glTexCoord2d(1,1); glVertex2f(1, 1);
            glTexCoord2d(0,1); glVertex2f(-1, 1);
        }
        glEnd();
        
     
    }    
    glEndList();/* close the list */
    
    /* apply some parameters on the texture */
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
   
    glEnable(GL_TEXTURE_2D);

    glEnable(GL_BLEND);/* enable blending for the alpha color */ 
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  
    /* Now, timer is set */
    if (_wait_timer == NULL){
        _wait_angle = 0.0f;
  
        _wait_timer = new QTimer(this);
        connect(_wait_timer, SIGNAL(timeout()), _render_instance, SLOT(updateGL()));
        _wait_timer->start(_wait_spf);
    }

    return object;
}





GLuint Render_area::draw_ruler() {
    GLuint object;
    unsigned int nb_graduation;
    Element_pos graduation_distance_per_5;
    double ruler_bg_r, ruler_bg_g, ruler_bg_b;/* the ruler background color (in RGB mode)*/

    /* light blue color */
    {
        ruler_bg_r = 0.5;
        ruler_bg_g = 0.5;
        ruler_bg_b = 1.0;
    }


    /*
     * NOTE:
     * _ruler_distance is the desire distance between 2 graduations.
     * There are ceil(_render_width/_ruler_distance) graduations.
     * Between 2 consecutive graduations, there are 4 small graduations.
     */


    /*
     * Graduation number.
     */
    nb_graduation = ceil(_render_width/_ruler_distance);

    /*
     * Distance between 2 small consecutive graduations.
     */
    graduation_distance_per_5 = _ruler_distance/5.0;

    /* 
     * create the list 
     */
    object = glGenLists(1);

    if (object == 0)
        message << "Error when creating list" << Message::endw;


    /* 
     * open the list 
     */
    glNewList(object, GL_COMPILE);
    {

        for (unsigned int i = 0 ; i<=nb_graduation ; i++){

            /* 
             * draw small graduations 
             */
            for (char k = 1 ; k < 5 ; k ++){
                glBegin(GL_LINES);
                {
                    glColor3d(1.0, 1.0, 1.0);glVertex2d(i*_ruler_distance+k*graduation_distance_per_5, _ruler_y);
                    glColor3d(0.6, 0.6, 1.0);glVertex2d(i*_ruler_distance+k*graduation_distance_per_5, _ruler_y+_ruler_height/3.0);
                }
                glEnd();
            }/* end for( k ) */ 


            /* 
             * Draw the graduation 
             */
            glBegin(GL_LINES);
            {
                glColor3d(1.0, 1.0, 1.0);glVertex2d(i*_ruler_distance, _ruler_y);
                glColor3d(0.6, 0.6, 1.0);glVertex2d(i*_ruler_distance, _ruler_y+_ruler_height);
            }
            glEnd(); 

        }/* end for ( i ) */

        /* Draw the background */
        glBegin(GL_QUADS);
        {
            glColor4d(ruler_bg_r, ruler_bg_g, ruler_bg_b, 0.8);glVertex2d(0,  _ruler_y+_ruler_height);
            glColor4d(ruler_bg_r, ruler_bg_g, ruler_bg_b, 0.5);glVertex2d(0, _ruler_y);
            glColor4d(ruler_bg_r, ruler_bg_g, ruler_bg_b, 0.5);glVertex2d(0 + nb_graduation*_ruler_distance, _ruler_y);
            glColor4d(ruler_bg_r, ruler_bg_g, ruler_bg_b, 0.8);glVertex2d(0 + nb_graduation*_ruler_distance, _ruler_y+_ruler_height);
        }
        glEnd();
    


    }    
    glEndList();/* close the list */
    
    return object;
}





void Render_area::call_ruler(){

    Element_pos nb_graduation;
    Element_pos graduation_distance_per_5;
    Element_pos buf_time;
    Element_pos _info_degree_x;
    Element_pos _info_degree_width;
    ostringstream buf_txt;

 
    Element_pos entire_part_d;
    Element_pos degree_d;
    Element_pos pow_10_degree_d;
    Element_pos buf_min;
    Element_pos buf_max;
    Element_pos buf_d;
    double ruler_bg_r, ruler_bg_g, ruler_bg_b;/* the ruler background color (in RGB mode)*/

    /* light blue color */
    {
        ruler_bg_r = 0.5;
        ruler_bg_g = 0.5;
        ruler_bg_b = 1.0;
    }

    /* Contains the degree scale to get a number from 0.01 to 100 */
    const Element_pos mul_scale_d = 4;

    /* Contains the scale to get a number from 0.01 to 100 (equals to 10^mul_scale_d)  */
    const Element_pos mul_scale = 10000;

    /* The ruler accurate */
    const Element_pos limit_d = 0.00000000001;


    /* Graduation number. */
    nb_graduation = ceil(_render_width/_ruler_distance);
    
    /* Distance between 2 small consecutive graduations. */
    graduation_distance_per_5 = _ruler_distance/5.0;

    /* Use to temporary store the graduation time value. */
    buf_time = 0;

    /* The info degree x beginning position. */
    _info_degree_x = 0;

    /* The info degree width: dynamically computed to fit its value. Default size: 1. */
    _info_degree_width = 0;

    /* Use to temporary store the graduation time value text. */
    buf_txt.str("");


    /*
     * The following variables are used to work out each graduation value
     * in a correct format.
     * Need to work out the degree of ruler graduation time value.
     *
     * Try to distinct the common number part of each graduation and the distinct part
     * from the left.
     * For example, between 3.67834 and 3.67854, the common part is: 3.678 and 
     * the distinct part is 0.000xx (xx = {34,54})
     */
     
    /* Update Info::Render::_x_min_visible and Info::Render::_x_max_visible values */
    update_visible_interval_value();
        
    /* Contains the common left part of each graduate values */
    entire_part_d = 0;

    /* Contains the degree used to know how many we need to multiply to get 
       the uncommon right part of each graduate values more than 0.01 */
    degree_d = 0;

    /* Contains 10^degree_d */
    pow_10_degree_d = 1;

    /* Just a buffer for intermediate computations */
    buf_d = 0;
  



    /* 
     * Draw ruler 
     */
    if (glIsList(_list_ruler) == GL_FALSE) {
        message << "ERROR LIST not exist for the ruler." << Message::ende;
    }
    else{
        glPushMatrix();

        /*
         * Just translate for ruler follows the entity translate.
         */
        glTranslated( - (int)_x_state_translate % (int)_ruler_distance, 0, _z_ruler);
        glCallList(_list_ruler);

        glPopMatrix();
    }
    
    
    /* 
     * Work out the graduation time value degree.
     */

    buf_max = Info::Render::_x_max_visible;
    buf_min = Info::Render::_x_min_visible;

    if (buf_max>limit_d){
        while (buf_max>1){
            buf_max/=10.0;
            buf_min/=10.0;
            degree_d--;
            pow_10_degree_d /= 10.0;
        }
    }

    buf_d = 0;/* to secure */

    while ( (floor(buf_max) == floor(buf_min)) && (buf_d<10) ){/* buf<10 means that buf_max if greater than limit_d ! */
        buf_max *= 10;
        buf_min *= 10;
        pow_10_degree_d *= 10;
        degree_d++;
        buf_d++;
    }

    /* ignore the last figure */
    //  if (degree_d >= 0){
    pow_10_degree_d /= 10;
    degree_d--;  
    //}
    
    entire_part_d = floor( Info::Render::_x_max_visible * pow_10_degree_d );
    
    /* 
     * Draw the current ruler degree 
     */
    
    /* Draw the text */
    glColor3d(1.0, 1.0, 1.0);/* text color */
    glRasterPos2f( 0, _ruler_y+_ruler_height);
    
    buf_txt << entire_part_d << "Xe-" << degree_d+mul_scale_d;
      
    for(unsigned int j = 0 ; j < buf_txt.str().length() ; j ++){
        glutBitmapCharacter(GLUT_BITMAP_HELVETICA_10, buf_txt.str().c_str()[j]);
        _info_degree_width += glutBitmapWidth(GLUT_BITMAP_HELVETICA_10,  buf_txt.str().c_str()[j]);
    }

    _info_degree_width = screen_to_render_x(_info_degree_width);

    buf_txt.str("");/* flush the buffer */

    /* Draw the foreground */
    glBegin(GL_QUADS);
    {
        glColor4d(ruler_bg_r, ruler_bg_g, ruler_bg_b, 0.8);glVertex3d(_info_degree_x,  _ruler_y+_ruler_height, _z_ruler_over);
        glColor4d(ruler_bg_r, ruler_bg_g, ruler_bg_b, 0.8);glVertex3d(_info_degree_x, _ruler_y, _z_ruler_over);
        glColor4d(ruler_bg_r, ruler_bg_g, ruler_bg_b, 0.8);glVertex3d(_info_degree_x + _info_degree_width, _ruler_y, _z_ruler_over);
        glColor4d(ruler_bg_r, ruler_bg_g, ruler_bg_b, 0.8);glVertex3d(_info_degree_x + _info_degree_width, _ruler_y+_ruler_height, _z_ruler_over);
    }
    glEnd();
    

    
    /* 
     * Draw coordinates 
     */
    glColor3d(1.0, 1.0, 1.0);/* text color */

    for (unsigned int i = 0 ; i<=nb_graduation ; i++){

        /* 
         * Work out the graduation time value.
         */

        buf_time = render_to_trace_x(i*_ruler_distance);

        if (buf_time < 0) continue;/* do not display negative time values */
      

        /* Now, convert (XXX,XXXXXXXXX) to (YYY * 10^ZZ and a common part WWWWWW * 10^TT) */

        /* trunc the common part of each number */
        buf_d =  buf_time*pow_10_degree_d;
        buf_d = buf_d - floor( buf_d );

        /* Mul buf_time to belongs to ]1000;0[ */
        buf_d *= mul_scale;

        /* Keep the entire part only */
        buf_d = floor(buf_d);

        /* Finally, fill the text buffer to display the graduate value */
        buf_txt << buf_d;
 
        /* 
         * Display the graduation time value.
         */
     
        glRasterPos2f( - (int)_x_state_translate % (int)_ruler_distance + i*_ruler_distance, _ruler_y+_ruler_height);/* set position */
     
        if (  _info_degree_width <  ( i*_ruler_distance - ((int)_x_state_translate % (int)_ruler_distance))  ){    
            for(unsigned int j = 0 ; j < buf_txt.str().length() ; j ++){
                glutBitmapCharacter(GLUT_BITMAP_HELVETICA_10, buf_txt.str().c_str()[j]);
            }
        }/* end for ( i ) */

        buf_txt.str("");/* flush the buffer */


    }/* end for ( i ) */

}    

