/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/

#include <fstream>
#include <string>
#include <sstream>
#include <queue>
/* -- */
#include <QObject>
/* -- */
#ifndef VITE_DEBUG
#  include "core/Core.hpp"
#else // TODO Include the debug one
#  include "core/Core.hpp"
#endif
/* -- */
#include "common/Message.hpp"
#include "common/Errors.hpp"
/* -- */
using namespace std;

queue<string> Error::_errors;
queue<string> Error::_warnings;


const int Error::_EVERYTHING = 0;
const int Error::_WARNING    = 1;
const int Error::_ERROR      = 2;

string Error::_content = QObject::tr("").toStdString();


const string Error::_PARSE             = QObject::tr("expected \" before end of file").toStdString();
const string Error::_MMAP              = QObject::tr("mmap error").toStdString();
const string Error::_EMPTY_FILE        = QObject::tr("empty file").toStdString();
const string Error::_FSTAT             = QObject::tr("status file error").toStdString();
const string Error::_OPEN              = QObject::tr("open file error").toStdString();
const string Error::_MUNMAP            = QObject::tr("munmap error").toStdString();
const string Error::_EXPECT_END_DEF    = QObject::tr("expected %EndEventDef").toStdString();
const string Error::_EXPECT_EVENT_DEF  = QObject::tr("expected %EventDef").toStdString();
const string Error::_EXPECT_NAME_DEF   = QObject::tr("the definition is not named").toStdString();
const string Error::_EXPECT_ID_DEF     = QObject::tr("the definition is not identified").toStdString();
const string Error::_UNKNOWN_ID_DEF    = QObject::tr("there is no definition with the identity: ").toStdString();
const string Error::_EXTRA_TOKEN       = QObject::tr("extra token(s) ignored").toStdString();
const string Error::_UNKNOWN_EVENT_DEF = QObject::tr("the following event doesn't match with any event known: ").toStdString();
const string Error::_FIELD_TYPE_MISSING = QObject::tr("a field type is missing ").toStdString();
const string Error::_FIELD_TYPE_UNKNOWN = QObject::tr("the following field type is unknown: ").toStdString();
const string Error::_EMPTY_DEF          = QObject::tr("a definition line is empty").toStdString();
const string Error::_INCOMPATIBLE_VALUE_IN_EVENT = QObject::tr("incompatible value: ").toStdString();
const string Error::_BAD_FILE_EXTENSION   = QObject::tr("the extension of the file is not .trace").toStdString();
const string Error::_LINE_TOO_SHORT_EVENT = QObject::tr("missing field value(s) in an event").toStdString();

const string Error::_UNKNOWN_CONTAINER_TYPE = QObject::tr("Unknown container type: ").toStdString();
const string Error::_UNKNOWN_CONTAINER      = QObject::tr("Unknown container: ").toStdString();
const string Error::_UNKNOWN_EVENT_TYPE     = QObject::tr("Unknown event type: ").toStdString();
const string Error::_UNKNOWN_STATE_TYPE     = QObject::tr("Unknown state type: ").toStdString();
const string Error::_UNKNOWN_VARIABLE_TYPE  = QObject::tr("Unknown variable type: ").toStdString();
const string Error::_UNKNOWN_LINK_TYPE      = QObject::tr("Unknown link type: ").toStdString();
const string Error::_UNKNOWN_ENTITY_TYPE    = QObject::tr("Unknown entity type: ").toStdString();
const string Error::_UNKNOWN_ENTITY_VALUE   = QObject::tr("Unknown entity value: ").toStdString();


void Error::set(const string kind_of_error, const int priority){
    Error::_content = kind_of_error;
    switch(priority){
    case _WARNING:
        Error::_warnings.push(Error::_content);
        break;
    default: // Include the _ERROR
        Error::_errors.push(Error::_content);
        break;
    }
}

void Error::set(const string kind_of_error, const unsigned int line_number, const int priority){
    char line[10];
    sprintf(line, "%d", line_number);
    set(kind_of_error + " on line " + line, priority);
}

void Error::set_and_print(const string kind_of_error, const int priority){
    set(kind_of_error, priority);
    print(priority);
}

void Error::set_and_print(const string kind_of_error, const unsigned int line_number, const int priority){
    char line[10];
    sprintf(line, "%d", line_number);
    set(kind_of_error + " on line " + line, priority);
    print(priority);
}


bool Error::set_if(bool condition, const string kind_of_error, const unsigned int line_number, const int priority){
    if(condition){
        char line[10];
        sprintf(line, "%d", line_number);
        set(kind_of_error + " on line " + line, priority);
        return true;
    }
    return false;
}


void Error::print(const int priority) {
    *Message::get_instance() << _content;
    switch(priority){
    case _WARNING:
        *Message::get_instance() << Message::endw;
        break;
    default: // Include the _ERROR
        *Message::get_instance() << Message::ende;
        break;
    }
}

void Error::print(const string content, const int priority) {
    *Message::get_instance() << content;
    switch(priority){
    case _WARNING:
        *Message::get_instance() << Message::endw;
        break;
    default: // Include the _ERROR
        *Message::get_instance() << Message::ende;
        break;
    }
}

void Error::print_numbers(){
    *Message::get_instance() << Error::_errors.size() << " errors and " << Error::_warnings.size() << " warnings were found during parsing.";
    if(Error::_warnings.size() == 0 && Error::_errors.size() == 0){
        *Message::get_instance() << Message::endi;
    }
    else if(Error::_errors.size() == 0){
        *Message::get_instance() << Message::endw;
    }
    else{
        *Message::get_instance() << Message::ende;
    }
}

void Error::flush(const string &filename){

    if(_errors.empty() && _warnings.empty()) {
        return;
    }
    else {

        print("Errors and warnings can be found in "+filename, _ERROR);

        ofstream outfile(filename.c_str(), ios::out | ios::trunc);
        
        const int number_of_errors = Error::_errors.size();
        const int number_of_warnings = Error::_warnings.size();

        if(!outfile.is_open()){
            cerr << "unable to open " << filename << " to print the errors encountered in the file opening" << endl;
            return;
        }
        else {
            outfile << "File " << Message::get_interface()->get_filename() << endl << endl;

            if(!_errors.empty()) {
                outfile << "Errors :" << endl;
            }

            // Print the errors
            while(!_errors.empty()) {
                outfile << _errors.front() << endl;
                //print(_errors.front(), _ERROR);
                _errors.pop();
            }

            // Print the warnings
            if(!_warnings.empty()) {
                outfile << endl << "Warnings :" << endl;
            }
            while(!_warnings.empty()) {
                outfile << _warnings.front() << endl;
                //print(_warnings.front(), _WARNING);
                _warnings.pop();
            }

            outfile << endl << "Your trace has " << number_of_errors << " errors and " << number_of_warnings << " warnings." << endl;

            outfile.close();
        }
    }
}

