/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/

#include <iostream>
#include <string>
/* -- */
#include <QStringList>
#include <QColor>
#include <QFileDialog>
#include <QColorDialog>
#include <QInputDialog>
#include <QMessageBox>
/* -- */
#include <qtcolorpicker.h>
/* -- */
#include "trace/values/Values.hpp"
#include "trace/EntityValue.hpp"
#include "trace/EntityTypes.hpp"
#include "trace/Entitys.hpp"
#include "trace/Trace.hpp"
/* -- */
#include "render/Palette.hpp"
/* -- */
#include "common/Session.hpp"
#include "interface/Interface.hpp"
#include "core/Core.hpp"
#include "interface/Settings_window.hpp"

using namespace std;

Settings_window::Settings_window(Core *c, QWidget *parent) : QWidget(parent), _c(c) {
    setupUi(this);
    plugin_load_dirs_name();
    reload_general_tab();
}

Settings_window::~Settings_window() {
}

void Settings_window::on_tabWidget_currentChanged(int index) {
    const string &tabName = tabWidget->tabText(index).toStdString();
    if(tabName == "States") {
        // Need to load the current palette.
        palette_list->clear();
        QStringList palettes;
        Session::get_palettes_name(palettes);
        palette_list->addItems(palettes);
        const string current = Session::get_current_palette();
        fill_table(current);
        palette_list->setCurrentIndex(palette_list->findText(QString::fromStdString(current))); // Set the combobox at the good index.
    }
    else if(tabName == "Minimap") {
        reload_minimap_tab();
    }
    else { // Need other initialisations for other tabs?
    }
}

void Settings_window::show() {
    // The plugin tab
    dir_list->clear();
    plugin_load_dirs_name();
    reload_minimap_tab();
    QWidget::show();
}

/************************/
/***** General tab. *****/
/************************/
void Settings_window::reload_general_tab() {
    use_palette_cb->setChecked(Session::get_use_palette());
}

/************************/
/***** Minimap tab. *****/
/************************/
void Settings_window::reload_minimap_tab() {
    minimap_hide_cb->setChecked(Session::Minimap::is_closed);
    minimap_x_sb->setValue(Session::Minimap::x);
    minimap_y_sb->setValue(Session::Minimap::y);
    minimap_w_sb->setValue(Session::Minimap::width);
    minimap_h_sb->setValue(Session::Minimap::height);
    // \todo Square border size? kesako?
    // \todo maybe colors?
}

/*************************/
/****** Plugin tab. ******/
/*************************/

void Settings_window::on_rm_dir_btn_clicked() {
    dir_list->takeItem(dir_list->currentRow());
}

void Settings_window::on_add_dir_btn_clicked() {
    QFileDialog dialog(this, tr("Add a directory"));

    dialog.setFileMode(QFileDialog::DirectoryOnly);
    dialog.exec();

    if(dialog.result() == QDialog::Accepted) {
        QStringList list = dialog.selectedFiles();
        dir_list->addItems(list);
    }
}

void Settings_window::plugin_load_dirs_name() {
    QStringList dirs;
    Session::load_plugin_directories(dirs);
    dir_list->addItems(dirs);
}

/************************/
/****** State tab. ******/
/************************/

void Settings_window::fill_table(const string &palette_name) {

    while(states_table->rowCount() > 0) { // Clear the current table
        states_table->removeRow(0);
    }

    map<string, Color *> colors;
    int row = 0;

    if(palette_name == "default") { // Loads the colors of the current trace
        if(_c->get_trace()) { // Need a trace loaded
            _c->get_trace()->get_states_colors(colors);
        }
        else {
            cerr << "no trace loaded" << endl;
            return;
        }
    }
    else { // Look for the state/color pairs in Session.
        colors = Session::get_palette(palette_name)->get_map();
    }
    
    for(map<string, Color *>::iterator it = colors.begin() ; 
        it != colors.end() ; ++ it) {
        add_state_line(row, (*it).first.c_str(), (*it).second, true);
    }
}

void Settings_window::add_state_line(int &row, std::string st_name, Color *c, bool checked) {
    states_table->insertRow(row);
    QTableWidgetItem *name = new QTableWidgetItem(st_name.c_str());
    name->setFlags(Qt::ItemIsEnabled);
    states_table->setItem(row, 0, name);
    QtColorPicker *color_widget = new QtColorPicker();
    color_widget->setStandardColors();
    if(c) {
        color_widget->setCurrentColor(QColor(c->get_red()*255, c->get_green()*255, c->get_blue()*255));
    }
    else { // \todo check the default color value -> put this in Palette as static member
        color_widget->setCurrentColor(QColor(0, 244, 222));
    }
    connect(color_widget, SIGNAL(colorChanged(const QColor &)), this, SLOT(state_color_changed(const QColor &)));

    states_table->setCellWidget(row, 1, color_widget);
    
    QCheckBox *cb = new QCheckBox();
    cb->setChecked(checked);
    states_table->setCellWidget(row, 2, cb);
    row ++;
}

void Settings_window::on_palette_list_currentIndexChanged(const QString &text) {
    fill_table(text.toStdString());
}

void Settings_window::on_add_palette_btn_clicked() {
    bool ok;
    QString text = QInputDialog::getText(this, tr("ViTE"),
                                         tr("Palette name:"), QLineEdit::Normal,
                                         "", &ok);
    if (ok && !text.isEmpty()) { // Palette name is set
        palette_list->addItem(text);
        Session::create_palette(text.toStdString());
    }
}

void Settings_window::on_copy_palette_btn_clicked() {
    bool ok;
    QString text = QInputDialog::getText(this, tr("ViTE"),
                                         tr("new palette name:"), QLineEdit::Normal,
                                         "", &ok);
    string current = palette_list->currentText().toStdString();
    if (ok && !text.isEmpty()) { // Palette name is set
        if (palette_list->findText(text) != -1) { // Already set
            if(QMessageBox::warning(this, tr("ViTE"), tr("Palette already exists.\nAre you sure you want to erase it?"), QMessageBox::Ok | QMessageBox::Cancel) != QMessageBox::Ok) {
                return;
            }
            
            // Save
            // We remove the existing one
            Session::remove_palette(text.toStdString());
        }
        else { // Adding if not existing in the list
            palette_list->addItem(text);    
        } 
        Session::copy_palette(current, text.toStdString());
    }
}

void Settings_window::on_rm_palette_btn_clicked() {
    QString text = palette_list->currentText();
    palette_list->removeItem (palette_list->currentIndex());
    Session::remove_palette(text.toStdString());
}

void Settings_window::on_add_state_btn_clicked() {
    bool ok;
    QString text = QInputDialog::getText(this, tr("ViTE"),
                                         tr("State name:"), QLineEdit::Normal,
                                         "", &ok);
    if (ok && !text.isEmpty()) {
        QColor qc = QColorDialog::getColor();
        if(!qc.isValid()) {
            cerr << "Color not chosen\n" << endl;
        }
        else { // State name is set and color is valid.
            Color color(qc.red()/255., qc.green()/255., qc.blue()/255.);
            // Add a row to the table
            const int new_row = states_table->rowCount();
            states_table->insertRow(new_row);
            // First item : the state name
            QTableWidgetItem *name = new QTableWidgetItem(text);
            name->setFlags(Qt::ItemIsEnabled);
            states_table->setItem(new_row, 0, name);
            // Second item : the color
            QtColorPicker *color_widget = new QtColorPicker();
            color_widget->setStandardColors();
            color_widget->setCurrentColor(qc);
            states_table->setCellWidget(new_row, 1, color_widget);
            // Third item : if we show the state or no (not implemented here and in the trace)
            QCheckBox *cb = new QCheckBox();
            cb->setChecked(true);
            states_table->setCellWidget(new_row, 2, cb);

            Session::add_state_to_palette(palette_list->currentText().toStdString(), text.toStdString(), color);
        }
    }
}

void Settings_window::on_rm_state_btn_clicked() {
    QString text = states_table->item(states_table->currentRow(), 0)->text();
    Session::remove_state_to_palette(palette_list->currentText().toStdString(), text.toStdString());
    states_table->removeRow(states_table->currentRow());
}

void Settings_window::state_color_changed(const QColor &qc) {
    if(!states_table || !palette_list)
        return;
    Color color(qc.red()/255., qc.green()/255., qc.blue()/255.);
    
    int row = 0;
    while(states_table->cellWidget(row, 1) && states_table->cellWidget(row, 1) != sender()) { // Find the row of the colorPicker
        row ++;
    }

    if(!states_table->cellWidget(row, 1))
        return; // epic fail...

    std::string palette_name = palette_list->currentText().toStdString();
    std::string state_name = states_table->item(row, 0)->text().toStdString();
    Palette *p = Session::get_palette(palette_name);

    Session::add_state_to_palette(palette_name, state_name, color);
    p->add_state(state_name, color);
}

/*************************/
/* Global buttons click. */
/*************************/
void Settings_window::on_apply_clicked() {
    // The plugin tab
    QStringList dir_names;
    for(int i = 0 ; i < dir_list->count() ; i ++) {
        dir_names << dir_list->item(i)->text();
    }
    Session::save_plugin_directories(dir_names);

    // The states tab
    Session::set_current_palette(palette_list->currentText().toStdString());

    // The general tab
    Session::set_use_palette(use_palette_cb->isChecked());

    // The minimap tab
    minimap_hide_cb->setChecked(Session::Minimap::is_closed);
    minimap_x_sb->setValue(Session::Minimap::x);
    minimap_y_sb->setValue(Session::Minimap::y);
    minimap_w_sb->setValue(Session::Minimap::width);
    minimap_h_sb->setValue(Session::Minimap::height);
    
    /*cout <<minimap_x_sb->value() << " " << minimap_y_sb->value() <<" " << 
        minimap_w_sb->value() <<" " <<  minimap_h_sb->value() <<" " << 
        minimap_hide_cb->isChecked() << endl;*/

    Session::save_minimap(minimap_x_sb->value(), minimap_y_sb->value(),
                          minimap_w_sb->value(), minimap_h_sb->value(),
                          minimap_hide_cb->isChecked());
    // emit changes. For now it is a global emit. Maybe only emits signals for what has changed using booleans?
    emit settings_changed(); // Catch by at least the interface_graphic which will dispatch it to the classes which needs
    hide();
}

void Settings_window::on_cancel_clicked() {
    hide();
}
