# Copyright 2018 - Nokia
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

from oslo_config import cfg

from vitrage.common.constants import DatasourceOpts as DSOpts
from vitrage.common.constants import DatasourceProperties as DSProps
from vitrage.common.constants import UpdateMethod
from vitrage.common.constants import VertexProperties as VProps
from vitrage.datasources.nova.host import NOVA_HOST_DATASOURCE
from vitrage.datasources.nova.host.transformer import HostTransformer
from vitrage.datasources.prometheus import PROMETHEUS_DATASOURCE
from vitrage.datasources.prometheus.properties import get_label
from vitrage.datasources.prometheus.properties import PrometheusAlertStatus \
    as PAlertStatus
from vitrage.datasources.prometheus.properties import PrometheusLabels \
    as PLabels
from vitrage.datasources.prometheus.properties import PrometheusProperties \
    as PProps
from vitrage.datasources.prometheus.transformer import PrometheusTransformer
from vitrage.datasources.transformer_base import TransformerBase
from vitrage.tests.mocks import mock_transformer
from vitrage.tests.unit.datasources.test_alarm_transformer_base import \
    BaseAlarmTransformerTest


# noinspection PyProtectedMember
class PrometheusTransformerTest(BaseAlarmTransformerTest):

    OPTS = [
        cfg.StrOpt(DSOpts.UPDATE_METHOD,
                   default=UpdateMethod.PUSH),
    ]

    # noinspection PyAttributeOutsideInit,PyPep8Naming
    @classmethod
    def setUpClass(cls):
        cls.transformers = {}
        cls.conf = cfg.ConfigOpts()
        cls.conf.register_opts(cls.OPTS, group=PROMETHEUS_DATASOURCE)
        cls.transformers[NOVA_HOST_DATASOURCE] = \
            HostTransformer(cls.transformers, cls.conf)
        cls.transformers[PROMETHEUS_DATASOURCE] = \
            PrometheusTransformer(cls.transformers, cls.conf)

    def test_create_update_entity_vertex(self):
        # Test setup
        host1 = 'host1'
        event = self._generate_event(host1)
        self.assertIsNotNone(event)

        # Test action
        transformer = self.transformers[PROMETHEUS_DATASOURCE]
        wrapper = transformer.transform(event)

        # Test assertions
        self._validate_vertex_props(wrapper.vertex, event)

        # Validate the neighbors: only one valid host neighbor
        entity_key1 = transformer._create_entity_key(event)
        entity_uuid1 = transformer.uuid_from_deprecated_vitrage_id(entity_key1)

        self._validate_host_neighbor(wrapper, entity_uuid1, host1)

        # Validate the expected action on the graph - update or delete
        self._validate_graph_action(wrapper)

    def _validate_vertex_props(self, vertex, event):
        self._validate_alarm_vertex_props(
            vertex, get_label(event, PLabels.ALERT_NAME),
            PROMETHEUS_DATASOURCE, event[DSProps.SAMPLE_DATE])

    @staticmethod
    def _generate_event(hostname):
        # fake query result to be used by the transformer for determining
        # the neighbor
        query_result = [{VProps.VITRAGE_TYPE: NOVA_HOST_DATASOURCE,
                         VProps.ID: hostname}]
        labels = {PLabels.SEVERITY: 'critical',
                  PLabels.INSTANCE: hostname}

        update_vals = {TransformerBase.QUERY_RESULT: query_result,
                       PProps.LABELS: labels}
        generators = mock_transformer.simple_prometheus_alarm_generators(
            update_vals=update_vals)

        return mock_transformer.generate_random_events_list(generators)[0]

    def _is_erroneous(self, vertex):
        return vertex[PProps.STATUS] == PAlertStatus.FIRING
