
// vbrename.cpp
// rename dicom files that have appropriate fields, etc.
// Copyright (c) 2003-2005 by The VoxBo Development Team

// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
//
// original version written by Dan Kimberg

using namespace std;

#include <stdio.h>
#include <string.h>
#include <sstream>
#include "vbutil.h"
#include "vbio.h"

class VBRename {
public:
  int forceflag;
  int Go(tokenlist &args);
  int Rename(string filename);
};

extern "C" {

#include "dicom.h"
void vbrename_help();
int renamefile(string fname);

int
main(int argc,char *argv[])
{
  tokenlist args;
  args.Transfer(argc-1,argv+1);
  if (args.size() <1) {
    vbrename_help();
    exit(0);
  }
  VBRename vbr;
  int err=vbr.Go(args);
  exit(err);
}
  
int
VBRename::Go(tokenlist &args)
{
  int i,j;
  int bad=0,good=0;
  struct stat st;
  tokenlist filelist,targetlist;
  stringstream tmps;

  forceflag=0;
  for (i=0; i<args.size(); i++) {
    if (args[i]=="-f") {
      forceflag=1;
      continue;
    }
    targetlist.Add(args[i]);
  }

  for (i=0; i<targetlist.size(); i++) {
    filelist.clear();
    // no such file or directory
    if (stat(targetlist(i),&st)) {
      bad++;
      continue;
    }
    // it's a directory, do all the files
    if (S_ISDIR(st.st_mode)) {
      vglob vg(targetlist[i]+"/*");
      for (size_t m=0; m<vg.size(); m++)
        filelist.Add(vg[m]);
    }
    // it's a file or something, try to do it
    else
      filelist.Add(args[i]);
    for (j=0; j<filelist.size(); j++) {
      if (Rename(filelist[j])) {
        printErrorMsg(VB_ERROR,(string)"vbrename: couldn't rename file " + filelist[j]);
        bad++;
        break;
      }
      else
        good++;
    }
  }
  tmps.str("");
  tmps << "vbrename: " << good << " files renamed, " << bad << " not renamed.";
  printErrorMsg(VB_INFO,tmps.str());
  return 0;
}

int
VBRename::Rename(string fname)
{
  dicominfo dci;
  stringstream tmps;
  struct stat st;
  string newdirname,newfilename;
  int i;
  if (read_dicom_header(fname,dci))
    return 101;
  tmps.str("");
  tmps << setfill('0')
       << setw(4) << dci.series
       << "_"
       << xstripwhitespace(dci.protocol)
    ;
  newdirname=tmps.str();
  tmps.str("");
  tmps << setfill('0')
       << setw(4) << dci.instance
       << "_"
       << xstripwhitespace(dci.date)
       << "_"
       << xstripwhitespace(dci.time)
    ;
  newfilename=tmps.str();
  for (i=0; i<(int)newdirname.size(); i++)
    if (newdirname[i]=='/') newdirname[i]='_';
  for (i=0; i<(int)newfilename.size(); i++)
    if (newfilename[i]=='/') newfilename[i]='_';
  if (newdirname.size()<1 || newdirname.size()<1)
    return 102;

  string dirname=xdirname(fname)+(string)"/"+newdirname;
  if (stat(dirname.c_str(),&st)) {
    if (mkdir(dirname.c_str(),S_IRWXU|S_IRWXG|S_IRWXO)) {
      if (errno!=EEXIST)
        return 140;
    }
    if (stat(dirname.c_str(),&st))
      return 160;
  }
  if (!(S_ISDIR(st.st_mode)))
    return 150;

  int err=rename(fname.c_str(),(dirname+(string)"/"+newfilename).c_str());
  if (err==-1) {
    stringstream tmps;
    tmps << "vbrename: couldn't rename " << fname;
    printErrorMsg(VB_ERROR,tmps.str());
    return err;
  }
  return 0;
}

void
vbrename_help()
{
  printf("\nVoxBo vbrename (v%s)\n",vbversion.c_str());
  printf("summary:\n");
  printf("  sorts DICOM files into separate directories\n");
  printf("usage:\n");
  printf("  vbrename [flags] <file/dir> [<file/dir> ...]\n");
  // printf("flags:\n");
  // printf("      -f      change permissions to 775 if needed (not yet working)\n");
  printf("notes:\n");
  printf("  vbrename will take a heap of mixed DICOM files and sort them into\n");
  printf("  directories based on the series and protocol, and rename the files\n");
  printf("  according to \"instance,\" date, and time\n");
  printf("\n");
}

} // extern "C"
