
// filecheck.cpp
// 
// Copyright (c) 1998-2002 by The VoxBo Development Team

// This file is part of VoxBo
// 
// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
// 
// original version written by Kosh Banerjee

/*********************************************************************
* Using the std namespace.                                           *
*********************************************************************/
using namespace std;

/*********************************************************************
* Required include file.                                             *
*********************************************************************/
#include "filecheck.h"

/*********************************************************************
* Constructor that takes a C-style string.                           *
*********************************************************************/
FileCheck::FileCheck(const char *fileName)
{

/*********************************************************************
* Setting this->fileName.                                            *
*********************************************************************/
  this->fileName = string(fileName);

/*********************************************************************
* Calling this->init() to populate the remaining data members.       *
*********************************************************************/
  this->init();

} // FileCheck::FileCheck(const char *fileName)

/*********************************************************************
* Constructor that takes a string object.                            *
*********************************************************************/
FileCheck::FileCheck(const string& fileName)
{

/*********************************************************************
* Setting this->fileName.                                            *
*********************************************************************/
  this->fileName = fileName;

/*********************************************************************
* Calling this->init() to populate the remaining data members.       *
*********************************************************************/
  this->init();

} // FileCheck::FileCheck(const string& fileName)

/*********************************************************************
* Constructor that takes an open file descriptor.                    *
*********************************************************************/
FileCheck::FileCheck(const int fd)
{

/*********************************************************************
* Calling this->init(int fd) to populate the data members.           *
*********************************************************************/
  this->init(fd);

} // FileCheck::FileCheck(const int fd)

/*********************************************************************
* Copy constructor.                                                  *
*********************************************************************/
FileCheck::FileCheck(const FileCheck &file)
{

/*********************************************************************
* Setting all the data members except this->fileInfo.                *
*********************************************************************/
  this->fileName = file.fileName;
  this->fd = file.fd;
  this->valid = file.valid;
  this->fileErrno = file.fileErrno;
  this->fileStrError = file.fileStrError;
  this->baseName = file.baseName;
  this->dirName = file.dirName;
  //  this->realPath = file.realPath;

/*********************************************************************
* Calling copyFileInfo() to populate this->fileInfo.                 *
*********************************************************************/
  copyFileInfo(file.fileInfo);

} // FileCheck::FileCheck(const FileCheck &file)

/*********************************************************************
* This function sets this->fd and calls stat() to fill in            *
* this->fileInfo.                                                    *
*********************************************************************/
void FileCheck::init()
{

/*********************************************************************
* Now assigning this->baseName and this->dirName.                    *
*********************************************************************/
  this->baseName = xfilename(this->fileName);
  this->dirName = xdirname(this->fileName);

/*********************************************************************
* Now initializing this->valid, this->fileErrno, and                 *
* this->fileStrError.                                                *
*********************************************************************/
  this->valid = true;
  this->fileErrno = 0;
  this->fileStrError = "";

/*********************************************************************
* Since one of the constructors that takes a file name as its        *
* argument was called (and not an open file descriptor), this->fd is *
* set to -1.                                                         *
*********************************************************************/
  this->fd = -1;

/*********************************************************************
* errno is set to 0 and then stat() is called.                       *
*********************************************************************/
  errno = 0;
  stat(this->fileName.c_str(), &this->fileInfo);

/*********************************************************************
* If errno is not zero, then this->valid is set to false,            *
* this->fileErrno is set to errno, and this->fileStrError is set to  *
* strerror(this->fileErrno).                                         *
*********************************************************************/
  if (errno != 0)
  {
    this->valid = false;
    this->fileErrno = errno;
    this->fileStrError = strerror(this->fileErrno);
  } // if
  else
  {

/*********************************************************************
* Now determining the real path to the file.                         *
*********************************************************************/
    char *resolvedPath = new char[PATH_MAX + 1];
    memset(resolvedPath, 0, PATH_MAX + 1);
    // this->realPath = realpath(this->fileName.c_str(), resolvedPath);
    deleteArrMem(resolvedPath);

  } // else

} // void FileCheck::init()

/*********************************************************************
* This function populates the data members for the consturctor that  *
* takes an open file descriptor as its single argument.              *
*********************************************************************/
void FileCheck::init(const int fd)
{

/*********************************************************************
* Since we don not know the file name associated with the open file  *
* descriptor (and it may not be appropriate for the file associated  *
* with the open file descriptor to even have a name, i.e., a socket),*
* this->fileName, this->baseName, this->dirName, and this->realPath  *
* are set to the empty string.                                       *
*********************************************************************/
  this->fileName = "";
  this->baseName = "";
  this->dirName = "";
  //  this->realPath = "";

/*********************************************************************
* Now initializing this->valid, this->fileErrno, and                 *
* this->fileStrError.                                                *
*********************************************************************/
  this->valid = true;
  this->fileErrno = 0;
  this->fileStrError = "";

/*********************************************************************
* Setting this->fd.                                                  *
*********************************************************************/
  this->fd = fd;

/*********************************************************************
* errno is set to 0 and then fstat() is called.                      *
*********************************************************************/
  errno = 0;
  fstat(fd, &this->fileInfo);

/*********************************************************************
* If errno is not zero, then this->valid is set to false,            *
* this->fileErrno is set to errno, and this->fileStrError is set to  *
* strerror(this->fileErrno).                                         *
*********************************************************************/
  if (errno != 0)
  {
    this->valid = false;
    this->fileErrno = errno;
    this->fileStrError = strerror(this->fileErrno);
  } // if

} // void FileCheck::init(const int fd)

/*********************************************************************
* This method returns the file's name.                               *
*********************************************************************/
string FileCheck::getFileName() const
{
  return this->fileName;
} // string FileCheck::getFileName() const

/*********************************************************************
* This method returns the file's file descriptor.                    *
*********************************************************************/
int FileCheck::getFD() const
{
  return this->fd;
} // int FileCheck::getFD() const

/*********************************************************************
* This method sets the file name and takes a C-style string as its   *
* single argument.                                                   *
*********************************************************************/
void FileCheck::setFileName(const char *fileName)
{
  this->fileName = string(fileName);

/*********************************************************************
* Now calling this->init() to stat the new file name.                *
*********************************************************************/
  this->init();

} // void FileCheck::setFileName(const char *fileName)

/*********************************************************************
* This method sets the file name and takes a string object as its    *
* single argument.                                                   *
*********************************************************************/
void FileCheck::setFileName(const string& fileName)
{
  this->fileName = fileName;

/*********************************************************************
* Now calling this->init() to stat the new file name.                *
*********************************************************************/
  this->init();

} // void FileCheck::setFileName(const string& fileName)

/*********************************************************************
* If the file is a regular valid file, then this method returns true.*
* Otherwise, false is returned.                                      *
*********************************************************************/
bool FileCheck::isRegularFile() const
{
  return (this->valid && S_ISREG(this->fileInfo.st_mode));
} // bool FileCheck::isRegularFile() const

/*********************************************************************
* If the file is a valid directory, then this method returns true.   *
* Otherwise, false is returned.                                      *
*********************************************************************/
bool FileCheck::isDirectory() const
{
  return (this->valid && S_ISDIR(this->fileInfo.st_mode));
} // bool FileCheck::isDirectory() const const

/*********************************************************************
* If the file is a valid character special device file, then this    *
* method returns true. Otherwise, false is returned.                 *
*********************************************************************/
bool FileCheck::isCharacterSpecialDevice() const
{
  return ( (this->checkValidAndWarn()) && S_ISCHR(this->fileInfo.st_mode) );
} // bool FileCheck::isCharacterSpecialDevice() const

/*********************************************************************
* If the file is a valid block special device file, then this method *
* returns true. Otherwise, false is returned.                        *
*********************************************************************/
bool FileCheck::isBlockSpecialDevice() const
{
  return ( (this->checkValidAndWarn()) && S_ISBLK(this->fileInfo.st_mode) );
} // bool FileCheck::isBlockSpecialDevice() const

/*********************************************************************
* If the file is a valid symbolic link, then this method returns     *
* true. Otherwise, false is returned.                                *
*********************************************************************/
bool FileCheck::isLink() const
{

/*********************************************************************
* If this->fileName is not the empty string, then we are dealing     *
* with a file and not a file descriptor.                             *
*********************************************************************/
  if (this->fileName.size() > 0)
  {

/*********************************************************************
* errno is set to 0 and linkInfo will hold the information gathered  *
* from the call to lstat().                                          *
*********************************************************************/
    errno = 0;
    struct stat linkInfo;
    lstat(this->fileName.c_str(), &linkInfo);

/*********************************************************************
* If errno is not zero, an approrpiate error message is assembled    *
* and printed out. Then false is returned.                           *
*********************************************************************/
    if (errno != 0)
    {
      ostringstream errorMsg;
      errorMsg << "lstat() failed for [" << this->fileName << "] due to [" << strerror(errno) << "].";
      printErrorMsg(VB_WARNING, errorMsg.str());
      return false;
    } // if

/*********************************************************************
* If we have a valid file and it's a link, true will be returned.    *
* otherwise, false will be returned.                                 *
*********************************************************************/
    return ( (this->checkValidAndWarn()) && S_ISLNK(linkInfo.st_mode) );
  } // if

/*********************************************************************
* If program flow ends up here, false is returned since we are       *
* dealing with a file descriptor and not a file.                     *
*********************************************************************/
  return false;

} // bool FileCheck::isLink() const

/*********************************************************************
* If the file is a valid FIFO file, then this method returns true.   *
* Otherwise, false is returned.                                      *
*********************************************************************/
bool FileCheck::isFIFO() const
{
  return ( (this->checkValidAndWarn()) && S_ISFIFO(this->fileInfo.st_mode) );
} // bool FileCheck::isFIFO() const

/*********************************************************************
* If the file is a valid UNIX-domain socket, then this method returns*
* true. Otherwise, false is returned.                                *
*********************************************************************/
bool FileCheck::isSocket() const
{
  return ( (this->checkValidAndWarn()) && S_ISSOCK(this->fileInfo.st_mode) );
} // bool FileCheck::isSocket() const

/*********************************************************************
* This method returns the last time the file was accessed for        *
* reading, or in the case of an executable file, the last time the   *
* file was executed.                                                 *
*********************************************************************/
const char * FileCheck::getLastAccess() const
{

/*********************************************************************
* If we have a valid file, then the time of last read or execution   *
* is returned, as appropriate.                                       *
*********************************************************************/
  if (this->valid)
  {
    string s = string(ctime((const time_t *) &this->fileInfo.st_atime));
    s.at(NEW_LINE_LOC) = 0;
    return s.c_str();
  } // if

/*********************************************************************
* If program flow ends up here, then we do not have a valid file.    *
*********************************************************************/
  return "NOT A VALID FILE.";

} // const char * FileCheck::getLastAccess() const

/*********************************************************************
* This method returns the last time the file was written to, if it's *
* a valid file.                                                      *
*********************************************************************/
const char * FileCheck::getLastModification() const
{

/*********************************************************************
* If we have a valid file, then the time of last write is returned.  *
*********************************************************************/
  if (this->valid)
  {
    string s = string(ctime((const time_t *) &this->fileInfo.st_mtime));
    s.at(NEW_LINE_LOC) = 0;
    return s.c_str();
  } // if

/*********************************************************************
* If program flow ends up here, then we do not have a valid file.    *
*********************************************************************/
  return "NOT A VALID FILE.";

} // const char * FileCheck::getLastModification() const

/*********************************************************************
* This method returns the last time the file's i-node was updated.   *
* for a valid file.                                                  *
* A file's i-node is updated when:                                   *
*                                                                    *
* The file permissions get changed.                                  *
* The file gets written to.                                          *
* (But not when the file's access time is changed.)                  *
*                                                                    *
* NOTE: This is not the file's creation time. A file's creation time *
* is not preserved.                                                  *
*********************************************************************/
const char * FileCheck::getLastINodeChange() const
{

/*********************************************************************
* If we have a valid file, then the time of the fiel's last i-node   *
* update is returned.                                                *
*********************************************************************/
  if (this->valid)
  {
    string s = string(ctime((const time_t *) &this->fileInfo.st_ctime));
    s.at(NEW_LINE_LOC) = 0;
    return s.c_str();
  } // if

/*********************************************************************
* If program flow ends up here, then we do not have a valid file.    *
*********************************************************************/
  return "NOT A VALID FILE.";

} // const char * FileCheck::getLastINodeChange() const

/*********************************************************************
* This method returns the file's i-node number.                      *
*********************************************************************/
// ino_t FileCheck::getINodeNumber() const
// {

// /*********************************************************************
// * If the file is not valid, a warning message will be printed and 0  *
// * will be returned.                                                  *
// *********************************************************************/
//   this->checkValidAndWarn();
//   return this->fileInfo.st_ino;
// } // ino_t FileCheck::getINodeNumber() const

/*********************************************************************
* This method returns the file owner's user id.                      *
*********************************************************************/
// uid_t FileCheck::getOwnerUserID() const
// {

// /*********************************************************************
// * If the file is not valid, a warning message will be printed and 0  *
// * will be returned.                                                  *
// *********************************************************************/
//   this->checkValidAndWarn();
//   return this->fileInfo.st_uid;
// } // uid_t FileCheck::getOwnerUserID() const

/*********************************************************************
* This method returns the file owner's group id.                     *
*********************************************************************/
// gid_t FileCheck::getOwnerGroupID() const
// {

// /*********************************************************************
// * If the file is not valid, a warning message will be printed and 0  *
// * will be returned.                                                  *
// *********************************************************************/
//   this->checkValidAndWarn();
//   return this->fileInfo.st_gid;
// } // gid_t FileCheck::getOwnerGroupID() const

/*********************************************************************
* This method returns the file size, in bytes.                       *
*********************************************************************/
off_t FileCheck::getFileSize() const
{

/*********************************************************************
* If the file is not valid, a warning message will be printed and 0  *
* will be returned.                                                  *
*********************************************************************/
  this->checkValidAndWarn();
  return this->fileInfo.st_size;
} // off_t FileCheck::getFileSize() const

/*********************************************************************
* This method copies the input struct stat to this->fileInfo.        *
*********************************************************************/
void FileCheck::copyFileInfo(const struct stat *fInfo)
{

  this->fileInfo.st_dev = fInfo->st_dev;
  this->fileInfo.st_ino = fInfo->st_ino;
  this->fileInfo.st_mode = fInfo->st_mode;
  this->fileInfo.st_nlink = fInfo->st_nlink;
  this->fileInfo.st_uid = fInfo->st_uid;
  this->fileInfo.st_gid = fInfo->st_gid;
  this->fileInfo.st_rdev = fInfo->st_rdev;
  this->fileInfo.st_size = fInfo->st_size;
  this->fileInfo.st_atime = fInfo->st_atime;
  this->fileInfo.st_mtime = fInfo->st_mtime;
  this->fileInfo.st_ctime = fInfo->st_ctime;
  // DYK: probably not needed, definitely not portable
  //  this->fileInfo.st_blksize = fInfo->st_blksize;  
  //  this->fileInfo.st_blocks = fInfo->st_blocks;

} // void FileCheck::copyFileInfo(const struct stat *fInfo)

/*********************************************************************
* This method copies the input struct stat to this->fileInfo.        *
*********************************************************************/
void FileCheck::copyFileInfo(const struct stat &fInfo)
{

  this->fileInfo.st_dev = fInfo.st_dev;
  this->fileInfo.st_ino = fInfo.st_ino;
  this->fileInfo.st_mode = fInfo.st_mode;
  this->fileInfo.st_nlink = fInfo.st_nlink;
  this->fileInfo.st_uid = fInfo.st_uid;
  this->fileInfo.st_gid = fInfo.st_gid;
  this->fileInfo.st_rdev = fInfo.st_rdev;
  this->fileInfo.st_size = fInfo.st_size;
  this->fileInfo.st_atime = fInfo.st_atime;
  this->fileInfo.st_mtime = fInfo.st_mtime;
  this->fileInfo.st_ctime = fInfo.st_ctime;
  // this->fileInfo.st_blksize = fInfo.st_blksize;
  // this->fileInfo.st_blocks = fInfo.st_blocks;

} // void FileCheck::copyFileInfo(const struct stat &fInfo)

/*********************************************************************
* This method checks this->valid. If this->valid is false, then a    *
* warning message is printed and false is returned. Otherwise, true  *
* is returned.                                                       *
*********************************************************************/
bool FileCheck::checkValidAndWarn() const
{

/*********************************************************************
* We now check to see if this->valid is false.                       *
*********************************************************************/
  if (!this->valid)
  {

/*********************************************************************
* At this poiont, this->valid is false and so a warning message is   *
* assembled in errorMsg.                                             *
*********************************************************************/
    ostringstream errorMsg;

/*********************************************************************
* We need to distinguish between a "file" and a file descriptor.     *
*********************************************************************/
    if (this->fileName.size() > 0)
    {
      errorMsg << "The file [" << this->fileName << "]";
    } // if
    else
    {
      errorMsg << "The file descriptor [" << this->fd << "]";
    } // else

/*********************************************************************
* Now we finish assembling the warning message, print it, and return *
* false.                                                             *
*********************************************************************/
    errorMsg << " is not valid due to [" << this->fileStrError << "].";
    printErrorMsg(VB_WARNING, errorMsg.str());
    return false;
  } // if

/*********************************************************************
* At this poiont, we know we are dealing with a valid file so true   *
* is returned.                                                       *
*********************************************************************/
  return true;

} // bool FileCheck::checkValidAndWarn() const

/*********************************************************************
* This method simply prints out the fields of the input stat struct. *
*********************************************************************/
void FileCheck::statToString(const struct stat *fInfo) const
{
  cout << "stat.st_dev     = [" << fInfo->st_dev << "]" << endl;
  cout << "stat.st_ino     = [" << fInfo->st_ino << "]" << endl;
  cout << "stat.st_mode    = [" << fInfo->st_mode << "]" << endl;
  cout << "stat.st_nlink   = [" << fInfo->st_nlink << "]" << endl;
  cout << "stat.st_uid     = [" << fInfo->st_uid << "]" << endl;
  cout << "stat.st_gid     = [" << fInfo->st_gid << "]" << endl;
  cout << "stat.st_rdev    = [" << fInfo->st_rdev << "]" << endl;
  cout << "stat.st_size    = [" << fInfo->st_size << "]" << endl;
  cout << "stat.st_atime   = [" << fInfo->st_atime << "]" << endl;
  cout << "stat.st_mtime   = [" << fInfo->st_mtime << "]" << endl;
  cout << "stat.st_ctime   = [" << fInfo->st_ctime << "]" << endl;
  // cout << "stat.st_blksize = [" << fInfo->st_blksize << "]" << endl;
  // cout << "stat.st_blocks  = [" << fInfo->st_blocks << "]" << endl;
} // void FileCheck::statToString(const struct stat *fInfo) const
