
// stringtokenizer.h
// header information for i/o functions
// Copyright (c) 1998-2002 by The VoxBo Development Team

// This file is part of VoxBo
// 
// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
// 
// original version written by Kosh Banerjee

/*********************************************************************
* This is the required header file for stringtokenizer.cpp.          *
*********************************************************************/

/*********************************************************************
* Include guard.                                                     *
*********************************************************************/
#ifndef STRINGTOKENIZER_H
#define STRINGTOKENIZER_H

/*********************************************************************
* Required include files.                                            *
*********************************************************************/
#include <string>
#include <vector>
#include <algorithm>
#include <iostream>
#include <iterator>

/*********************************************************************
* Using the standard namespace.                                      *
*********************************************************************/
using namespace std;

class StringTokenizer
{

/*********************************************************************
* DATA MEMBERS:                                                      *
*                                                                    *
* NAME              TYPE           DESCRIPTION                       *
* ----              ----           -----------                       *
* theString         string         Holds the string to be tokenized. *
*                                                                    *
* theDelimiters     string         Holds the set of characters that  *
*                                  act as field delimiters for       *
*                                  theString.                        *
*                                                                    *
* theTokens         vector<string> A vector container used to hold   *
*                                  the tokens.                       *
*                                                                    *
* theTokenStarts vector<unsigned int> A vector container used to     *
*                                     hold the beginning index of    *
*                                     each token.                    *
*                                                                    *
* currentToken      unsigned int   Holds the index of the current    *
*                                  token.                            *
*                                                                    *
* empty             bool           Will be set to true if there are  *
*                                  no tokens to be found in          *
*                                  theString. Otherwise, will be set *
*                                  to false.                         *
*                                                                    *
* success           bool           Is set to true when it is possible*
*                                  for a public method to returns a  *
*                                  non-empty token. Otherwise, will  *
*                                  be set to false. This data member *
*                                  makes it possible for a client of *
*                                  this class to gauge the success   *
*                                  of a public method (by calling the*
*                                  public method getSuccess()).      *
*********************************************************************/
  private:
    string theString;
    string theDelimiters;
    vector<string> theTokens;
    vector<unsigned int> theTokenStarts;
    unsigned long currentToken;
    bool empty;
    bool success;

/*********************************************************************
* Called by the constructors to initialize the data members.         *
*********************************************************************/
    void init(const string& myString, const string& myDelims);

/*********************************************************************
* PRIVATE METHODS:                                                   *
* isDelimiter() - returns true if this->theString.at(i) is a         *
*                 delimiter character, false otherwise.              *
*                                                                    *
* tokenize()    - tokenizes this->theString and stores the tokens in *
*                 this->theTokens.                                   *
*********************************************************************/
    bool isDelimiter(const unsigned long i) const;
    void tokenize();

  public:

/*********************************************************************
* Constructors.                                                      *
*********************************************************************/
    StringTokenizer();
    StringTokenizer(const char *myString);
    StringTokenizer(const string& myString);
    StringTokenizer(const char *myString, const char *myDelims);
    StringTokenizer(const string& myString, const string& myDelims);
    StringTokenizer(const string& myString, const char *myDelims);
    StringTokenizer(const char *myString, const string& myDelims);
    StringTokenizer(const StringTokenizer& theStringTokens); // Copy constructor.

/*********************************************************************
* Destructor.                                                        *
*********************************************************************/
    ~StringTokenizer();

/*********************************************************************
* Public methods.                                                    *
*********************************************************************/
    string getCurrentToken();
    string getSameToken();
    string getToken(const unsigned int i);
    void reset();
    void toString() const;
    void setString(const string& s);
    void setString(const char *s);
    int getCurrentTokenStart() const;
    int getTokenStart(const unsigned int i) const;

/*********************************************************************
* Accessor method to return this->theString.                         *
*********************************************************************/
    inline string getTheString() const
    {
      return this->theString;
    } // inline string getTheString() const

/*********************************************************************
* Accessor method to return this->theDelimiters.                     *
*********************************************************************/
    inline string getTheDelimiters() const
    {
      return this->theDelimiters;
    } // inline string getTheDelimiters() const

/*********************************************************************
* Method to get the length of the specified token.                   *
*********************************************************************/
    unsigned long getTokenLength(const unsigned long i);

/*********************************************************************
* Method to get the length of this->theTokens[i].                    *
*********************************************************************/
    unsigned long getCurrentTokenLength();

/*********************************************************************
* Accessor method to return this->currentToken.                      *
*********************************************************************/
    inline unsigned long getTokenIndex() const
    {
      return this->currentToken;
    } // inline unsigned long getTokenIndex() const

/*********************************************************************
* Accessor method to return this->theTokens.size().                  *
*********************************************************************/
    inline unsigned long getNumTokens() const
    {
      return this->theTokens.size();
    } // inline unsigned long getNumTokens() const

/*********************************************************************
* Accessor method to return this->empty.                             *
*********************************************************************/
    inline bool getEmpty() const
    {
      return this->empty;
    } // inline bool getEmpty() const

/*********************************************************************
* Accessor method to return this->success.                           *
*********************************************************************/
    inline bool getSuccess() const
    {
      return this->success;
    } // inline bool getSuccess() const

/*********************************************************************
* This method returns the length of the current token.               *
*********************************************************************/
    unsigned long getCurrentTokenLength() const;

/*********************************************************************
* These methods returns the specified range of tokens.               *
*********************************************************************/
    string getTokenRange(const unsigned long begin, 
		unsigned long end);
    string getTokenRange(const unsigned long begin, 
		unsigned long end, const string& delims);

}; // class StringTokenizer

#endif // STRINGTOKENIZER_H
