
// processvbx.cpp
// process vbx (message) files in the service of either scheduler or vbsingle
// Copyright (c) 1998-2010 by The VoxBo Development Team

// This file is part of VoxBo
// 
// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
//
// original version written by Dan Kimberg

using namespace std;

#include <sys/signal.h>
#include "vbutil.h"
#include "vbprefs.h"
#include "vbjobspec.h"

extern vector<VBHost> hostlist;

void process_vbx();
int process_jobrunning(int snum,int jnum,pid_t pid,pid_t childpid);
int process_setseqinfo(int ss,string newstatusinfo);
int process_setjobinfo(int ss,int jj,string newstatusinfo);
int process_email(string recipient,string fname);
int process_adminemail(string fname);
int process_hostupdate(string hh);
int process_missing(int snum,int jnum,pid_t pid);
void setjobinfo_update_hostlist(VBJobSpec newjs);

void
process_vbx()
{
  tokenlist line;
  vglob vg("*.vbx");
  for (size_t i=0; i<vg.size(); i++) {
    line.ParseFirstLine(vg[i]);
    if (line[0]=="setjobinfo")
      process_setjobinfo(strtol(line[1]),strtol(line[2]),line.Tail(3));
    else if (line[0]=="jobrunning")
      process_jobrunning(strtol(line[1]),strtol(line[2]),strtol(line[3]),strtol(line[4]));
    else if (line[0]=="setseqinfo")
      process_setseqinfo(strtol(line[1]),line.Tail(2));
    else if (line[0]=="email")
      process_email(line[1],gb.gl_pathv[i]);
    else if (line[0]=="adminemail")
      process_adminemail(gb.gl_pathv[i]);
    else if (line[0]=="missing")
      process_missing(strtol(line[1]),strtol(line[2]),strtol(line[3]));
    else if (line[0]=="hostupdate")
      process_hostupdate(line.Tail());
    else { 
      printf("[E] %s invalid vbx file %s\n",timedate().c_str(),vg[i].c_str());
    }
    unlink(vg[i].c_str());
    //rename(gb.gl_pathv[i],((string)"tmp/"+gb.gl_pathv[i]).c_str());
  }
}

int
process_missing(int snum,int jnum,pid_t pid)
{
  VBJobSpec *js=findjob(snum,jnum);
  if (!js)
    return 101;
  js->missingcount++;
  if (js->missingcount>2) {
    printf("[E] %s job %08d-%05d has gone missing (count %d) from host %s\n",
           timedate().c_str(),snum,jnum,js->missingcount,js->hostname.c_str());
    process_setjobinfo(snum,jnum,"status W");
  }
  return 0;
}

int
process_jobrunning(int snum,int jnum,pid_t pid,pid_t childpid)
{
  // set it in the main list
  process_setjobinfo(snum,jnum,(string)"pid "+strnum(pid));
  // set it in the running list and return
  for (HI h=hostlist.begin(); h!=hostlist.end(); h++) {
    for (JI j=h->speclist.begin(); j!=h->speclist.end(); j++) {
      if (snum==j->snum && jnum==j->jnum) {
        j->pid=pid;
        j->childpid=childpid;
        return 0;
      }
    }
  }
  // shouldn't happen unless a host disappears
  printf("[E] %s bad jobrunning message: %08d-%08d, %d %d\n",
         timedate().c_str(),snum,jnum,pid,childpid);
  return 1;
}

int
process_setjobinfo(int ss,int jj,string newstatusinfo)
{
  // update the file
  char jobfile[STRINGLEN];
  int err;
  sprintf(jobfile,"%08d/%05d.job",ss,jj);
  if ((err=appendline(jobfile,newstatusinfo))) {
    printf("[E] %s couldn't update job file %s (%d,%d,%s) [%d]\n",
           timedate().c_str(),jobfile,
           ss,jj,newstatusinfo.c_str(),err);
    return 101;
  }
  VBSequence seq(ss,jj);
  VBJobSpec *oldjob;
  // read the updated file
  if (seq.speclist.size()!=1) {
    printf("[E] %s couldn't read updated job file (%d,%d,%s)\n",timedate().c_str(),
           ss,jj,newstatusinfo.c_str());
    return 102;
  }
  // update it in hostlist (if running) and main list (regardless)
  if ((oldjob=findjob(ss,jj))) {
    if (oldjob->status=='R')
      setjobinfo_update_hostlist(seq.speclist[0]);
    *oldjob=seq.speclist[0];
  }
  else
    printf("[E] %s couldn't find job record (%d,%d,%s)\n",timedate().c_str(),
           ss,jj,newstatusinfo.c_str());
  return 0;
}

void
setjobinfo_update_hostlist(VBJobSpec newjs)
{
  // update or remove on hostlist
  for (HI h=hostlist.begin(); h!=hostlist.end(); h++) {
    for (JI j=h->speclist.begin(); j!=h->speclist.end(); j++) {
      if (newjs.snum==j->snum && newjs.jnum==j->jnum) {
        if (newjs.status!='R')
          h->speclist.erase(j);
        else
          *j=newjs;
        return;
      }
    }
  }
  printf("[E] %s couldn't update %08d-%05d on any host, even %s\n",
         timedate().c_str(),newjs.snum,newjs.jnum,newjs.hostname.c_str());
}

int
process_setseqinfo(int ss,string newstatusinfo)
{
  // update the file
  char seqfile[STRINGLEN];
  sprintf(seqfile,"%08d/%d.seq",ss,ss);
  if (appendline(seqfile,newstatusinfo)) {
    printf("[E] %s couldn't update sequence file\n",timedate().c_str());
    return 101;
  }
  VBSequence js,*oldseq;
  js.LoadSequence(xdirname(seqfile));
  // find a home for it
  if ((oldseq=findsequence(ss)))
    *oldseq=js;
  return 0;
}

int
process_hostupdate(string hh)
{
  string hostinfo,field;

  VBHost *tmph=NULL;
  tokenlist args,argx;
  args.SetQuoteChars("[<(\"'");
  argx.SetQuoteChars("[<(\"'");
  args.ParseLine(hh);
  // first argument better be full hostname
  argx.ParseLine(args[0]);
  if (argx[0]!="hostname") {
    printf("[E] %s malformed host update\n",timedate().c_str());
    return 101;
  }
  string hostname=argx[1];
  for (int i=0; i<(int)hostlist.size(); i++) {
    if (hostlist[i].hostname==hostname) {
      tmph=&(hostlist[i]);
    }
  }
  if (tmph==NULL) {
    printf("[E] %s invalid host update from %s\n",
           timedate().c_str(),hostname.c_str());
    return 101;
  }

  tmph->resources.clear();
  tmph->provides.clear();
  for (int i=1; i<args.size(); i++) {
    argx.ParseLine(args[i]);
    field=argx[0];
    if (field=="hostname")
      tmph->hostname=argx[1];
    else if (field=="shortname")
      tmph->shortname=argx[1];
    else if (field=="pri")
      tmph->currentpri=strtol(argx[1]);
    else if (field=="load")
      tmph->loadaverage=strtod(argx[1]);
    else if (field=="resource") {
      VBResource rr;
      rr.name=argx[1];
      rr.command=argx[2];
      rr.cnt=strtol(argx[3]);
      tmph->resources.push_back(rr);
    }
    else if (field=="provides") {
      VBResource rr;
      rr.type=argx[1];
      rr.name=argx[2];
      rr.cnt=strtol(argx[3]);
      tmph->provides.push_back(rr);
    }
    else if (field=="total_cpus")
      tmph->total_cpus=strtol(argx[1]);
    else if (field=="taken_cpus")
      tmph->taken_cpus=strtol(argx[1]);
    else if (field=="avail_cpus")
      tmph->avail_cpus=strtol(argx[1]);
    else if (field=="state")
      tmph->status=argx[1];
  }
  tmph->Update();
  tmph->lastresponse=time(NULL);
  tmph->lastupdate=time(NULL);
  return 0;
}

