
// fileview.cpp
// Copyright (c) 1998-2010 by The VoxBo Development Team

// This file is part of VoxBo
// 
// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
// 
// original version written by Tom King and Dan Kimberg

#include "fileview.h"
#include <qapplication.h>
#include <qvariant.h>
#include <q3buttongroup.h>
#include <q3header.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <q3listview.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <q3whatsthis.h>
#include <qaction.h>
#include <qmenubar.h>
#include <q3popupmenu.h>
#include <q3toolbar.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qmessagebox.h>
#include <qdialog.h>
#include <q3vbox.h>
#include <q3filedialog.h>
//Added by qt3to4:
#include <Q3VBoxLayout>

enum {COL_NAME=0,COL_SIZE=1,COL_INFO=2};

tokenlist fileview::returnedFiles;

fileview::fileview( QWidget* parent, const char* name, bool modal , Qt::WFlags fl )
  : QDialog( parent, name, modal, fl )
{
  QPushButton* button;
  if (!name) setName( "fileview" );
  // setMinimumSize(QSize(500,350));

  Q3VBoxLayout* layout = new Q3VBoxLayout(this);
  layout->setAutoAdd(true);
  Q3HBox *hb;

  hb=new Q3HBox(this);
  
  button=new QPushButton("Home",hb);
  connect(button, SIGNAL(clicked()), this, SLOT(HandleHome()));

  button=new QPushButton("Up",hb);
  connect(button, SIGNAL(clicked()), this, SLOT(HandleUp()));

  button=new QPushButton("/",hb);
  connect(button, SIGNAL(clicked()), this, SLOT(HandleRoot()));

  hb=new Q3HBox(this);
  
  button=new QPushButton("Directory: ",hb);
  connect(button,SIGNAL(clicked()),this,SLOT(HandleNewWD()));

  leDirectory = new QLineEdit(hb,"leDirectory" );

  hb=new Q3HBox(this);
  new QLabel("Filename pattern: ",hb);

  leFileNamePattern = new QLineEdit(hb,"leFileNamePattern" );

  grpFileBorder = new Q3HBox( this, "grpFileBorder" );
  grpFileBorder->setLineWidth( 2 );

  // lblNumFilesSelected = new QLabel(" 0/0 files",grpFileBorder);

  lvFiles = new Q3ListView( grpFileBorder, "lvFiles" );
  // lvFiles->setItemMargin(3);
  lvFiles->addColumn("Filename");
  lvFiles->addColumn("Size");
  lvFiles->addColumn("Information");
  lvFiles->setColumnAlignment(COL_SIZE,Qt::AlignRight);
  lvFiles->setColumnWidth(COL_INFO,0);
  lvFiles->setMinimumHeight(380);
  lvFiles->setSelectionMode(Q3ListView::Extended); 

  hb=new Q3HBox(this);

  button = new QPushButton("Okay",hb);
  connect(button, SIGNAL(clicked()), this, SLOT(Handler()));

  button=new QPushButton("Cancel",hb);
  connect(button,SIGNAL(clicked()), this, SLOT(Cancel()));

  button = new QPushButton("XXX",hb);
  button->resize(QSize(0,0));
  button->hide();
  button->setDefault(1);
  connect(button, SIGNAL(clicked()), this, SLOT(NoHandler()));

  okayed = false;
  showdirs = false;
  setCaption("Select Files");
  char tmps[STRINGLEN];
  getcwd(tmps,STRINGLEN-1);
  leDirectory->setText(tmps);
  leFileNamePattern->clear();
  leFileNamePattern->setText("*"); 

  connect(leDirectory,SIGNAL(textChanged(const QString &)),this,SLOT(grayDir()));
  connect(leFileNamePattern, SIGNAL(textChanged(const QString &)),this,SLOT(populateListBox()));
//   leDirectory->setFocusPolicy(QWidget::StrongFocus);
//   leFileNamePattern->setFocusPolicy(QWidget::StrongFocus);
  connect(leDirectory,SIGNAL(returnPressed()),this,SLOT(populateListBox()));
  connect(leFileNamePattern,SIGNAL(returnPressed()),this,SLOT(populateListBox()));
  connect(leDirectory,SIGNAL(lostFocus()),this,SLOT(populateListBox()));
  connect(leFileNamePattern,SIGNAL(lostFocus()),this,SLOT(populateListBox()));
  connect(lvFiles,SIGNAL(doubleClicked(Q3ListViewItem*)),this,SLOT(Selected(Q3ListViewItem*)));
}

fileview::~fileview()
{
}

void
fileview::SetDirectory(string dir) {
  return;
  leDirectory->clear();
  leDirectory->setText(dir.c_str());
  return;
}

void
fileview::ShowImageInformation(bool x)
{
  lvFiles->setColumnWidth(COL_INFO,200);
  return;
  if (x) {
    lvFiles->addColumn( tr( "Image Information" ) );
    lvFiles->header()->setLabel( 2, tr( "Image Information" ) );
  }
  return;
}   

void fileview::Go() {
  populateListBox();
  this->exec();
  return;
}

void
fileview::ShowDirectoriesAlso(bool x)
{
  showdirs = x;
  return;
}

// FIXME dirs first

void
fileview::populateListBox()
{
  lvFiles->clear();
  struct stat st;
  char size[STRINGLEN];
  if (chdir(leDirectory->text().ascii()))
    return;
  QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
  tokenlist list=vglob((string)leFileNamePattern->text().ascii());
  for (int i = 0; i<(int)list.size(); i++) {
    if (stat(list[i].c_str(),&st))
      continue;
    if (showdirs==false) {
      if (!(S_ISREG(st.st_mode))) {
        continue;
      }
    }
    string protocol;
    if (lvFiles->columnWidth(COL_INFO)) {
      string imageinfo;
      Cube cb;
      Tes ts;
      if (!(cb.ReadHeader(list[i]))) {
        imageinfo=" 3D volume: "+strnum(cb.dimx)+"x"+strnum(cb.dimy)+"x"+strnum(cb.dimz);
        protocol=cb.GetHeader("Protocol:");
      }
      else if(!(ts.ReadHeader(list[i]))) {
        imageinfo=" 4D volume: "+strnum(ts.dimx)+"x"+strnum(ts.dimy)+"x"+strnum(ts.dimz)
          +", "+strnum(ts.dimt)+"volumes";
        protocol=cb.GetHeader("Protocol:");
      }
      else
        imageinfo="";
      if (protocol.size())
        imageinfo+=" ("+protocol+")";
      if (S_ISDIR(st.st_mode))
        strcpy(size, "DIR");
      else
        sprintf(size,"%ldK",((long)st.st_size/1024));
      new Q3ListViewItem( lvFiles, list[i].c_str(),
                         (S_ISDIR(st.st_mode)?"DIR":prettysize(st.st_size).c_str()),
                          imageinfo.c_str() );
    }
    else {
      if (S_ISDIR(st.st_mode))
        strcpy(size, "DIR");
      else
        sprintf(size,"%ldK",((long)st.st_size/1024));
      new Q3ListViewItem( lvFiles, xfilename(list[i]).c_str(), size);
    }
  }
  QApplication::restoreOverrideCursor();
  leFileNamePattern->setPaletteBackgroundColor(qRgb(255,255,255));
  leDirectory->setPaletteBackgroundColor(qRgb(255,255,255));
  return;
}

void
fileview::Selected(Q3ListViewItem *item)
{
  if ((string)item->text(1).ascii()!="DIR") {
    okayed=1;
    returnedFiles=returnSelectedFiles();
    this->close();
    return;
  }
  string newdir=(string)leDirectory->text().latin1()+"/"+item->text(0).latin1();
  leDirectory->setText(newdir.c_str());
  populateListBox();
}

int fileview::numSiblings() {
  int count = 1;
  if (lvFiles->childCount()==0)return 0;
  Q3ListViewItem* item = lvFiles->firstChild();
  if (item) {
    while (item->nextSibling()) {
      count++;
      item = item->nextSibling();
    }
  }
  return count;
}

void fileview::Cancel() {
  this->close(); 
  return;
} 

int fileview::numSelected() {
  int number = 0;
  Q3ListViewItem * item = lvFiles->firstChild();
  while (item) {
    if (item->isSelected())  
      number++;
    if (item->nextSibling())
      item = item->nextSibling();
    else
      break;
  }
  return number;
}

tokenlist fileview::returnSelectedFiles() {
  tokenlist fileList;
  Q3ListViewItem * item = lvFiles->firstChild();
  while (item) {
    if (item->isSelected())
      fileList.Add(item->text(0).ascii());
    if (item->nextSibling())
      item = item->nextSibling();
    else
      break;
  }
  return fileList;
}

void
fileview::Handler() {
  okayed=true;
  returnedFiles = returnSelectedFiles();
  this->close();
  return;
}

void
fileview::NoHandler()
{
  populateListBox();
}

void
fileview::HandleHome()
{
  leDirectory->setText(getenv("HOME"));
  populateListBox();
}
 
void
fileview::HandleUp()
{
  leDirectory->setText(xdirname(leDirectory->text().ascii()).c_str());
  populateListBox();
}
 
void
fileview::HandleRoot()
{
  leDirectory->setText("/");
  populateListBox();
}

void
fileview::grayDir()
{
  leDirectory->setPaletteBackgroundColor(qRgb(220,160,160));
}
 
void
fileview::HandleNewWD()
{
  QString s=Q3FileDialog::getExistingDirectory(leDirectory->text(),
                                              this,
                                              "xxx",
                                              "new dir?");
  leDirectory->setText(s.ascii());
  populateListBox();
}
 
tokenlist fileview::ReturnFiles() {
  return returnedFiles;
}
  
bool fileview::Okayed() {
  return okayed;
}


