//<copyright>
//
// Copyright (c) 1996,97
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        editdlg.C
//
// Purpose:     implementation of edit dialogs
//
// Created:      4 Mar 96   Alexander Nussbaumer
//
// Changed:     21 Feb 97   Alexander Nussbaumer
//
// Changed:     23 Apr 97   Michael Pichler
//
// $Id: editdlg.C,v 1.8 1997/04/24 10:32:21 mpichler Exp $
//
//</file>


#include "editdlg.h"

#include "scenewin.h"
#include "vrmlscene.h"
#include "themenus.h"
#include "camera.h"
#include "stranslate.h"

#include <hyperg/widgets/adjvalue.h>
#include <hyperg/widgets/coldlg.h>
#include <hyperg/widgets/cursors.h>
#include <hyperg/widgets/fchooser.h>
#include <hyperg/widgets/fbrowser.h>
#include <hyperg/widgets/field.h>
#include <hyperg/widgets/fslider.h>
#include <hyperg/widgets/glyphutil.h>
#include <hyperg/widgets/menus.h>
#include <hyperg/widgets/msgbox.h>
#include <hyperg/widgets/rimage.h>
#include <hyperg/widgets/gifimage.h>
#include <hyperg/widgets/table.h>
#include <hyperg/widgets/textb.h>
#include <hyperg/widgets/hgraster.h>

#include <ge3d/ge3d.h>

#include <hyperg/utils/str.h>
#include <hyperg/utils/verbose.h>
#include <hyperg/hyperg/message.h>

#include <InterViews/bitmap.h>
#include <InterViews/background.h>
#include <InterViews/font.h>
#include <InterViews/image.h>
#include <InterViews/label.h>
#include <InterViews/layout.h>
#include <InterViews/observe.h>
#include <InterViews/patch.h>
#include <InterViews/raster.h>
#include <InterViews/style.h>
#include <InterViews/target.h>
#include <InterViews/window.h>
#include <IV-look/kit.h>
#include <IV-look/menu.h>
#include <IV-X11/xwindow.h>

#include <math.h>
#include <string.h>
#include <stdlib.h>

#include <vrml/QvNode.h>

/*** Filetable ***/

FileTable::FileTable ()
{
  listsize_ = 0;

  path_ = "";

} // FileTable


FileTable::~FileTable ()
{
  clearTable ();

} // ~FileTable


void FileTable::clearTable ()
{
  int i = listsize_;
  while (i--)
  { delete objectname_[i];
    delete filename_[i];
  }
  
  listsize_ = 0;

} // clear


int FileTable::load (const RString& path)
{
  clearTable ();

  if (!path.length ())
    return 0;

  path_ = path;
  if (path_[path_.length () - 1] != '/')
    path_ += "/";

  RString pathfilename;
  pathfilename = path_;
  pathfilename += "filetable";
  FILE* ftable = fopen (pathfilename.string (), "r");
  if (!ftable)
  { HgMessage::error ("Filetable does not exist.");
    return 0;
  }

  // read object filetable
  FILE* test;
  char s[64];
  RString rstr;
  char* charptr;
  int i = 0;
  int itemflag = 0,  // set when item is filename
      truncate = 0;

  while (fgets (s, 63, ftable))
  {
    if ((*s == '#') || (*s == '\n'))  // comment line or empty line are ignored
      continue;

    // ignore rest of truncated line (if line is larger than 64 char)
    charptr = strchr (s, '\n');  // search new line character
    if (truncate)
    { if (charptr)
        truncate = 0;
      continue;
    }
    else if (!charptr)
      truncate = 1;

    if (charptr)
      *charptr = 0;  // remove newline character

    rstr = s;
    if (!itemflag)  // title
    { objectname_[i] = new RString;
      filename_[i] = new RString;
      *objectname_[i] = rstr;
      itemflag = 1;
    }
    else  // filename
    { pathfilename = path_;
      pathfilename += rstr;
      test = fopen (pathfilename.string (), "r");
      if (test)  // file exists
      { fclose (test);
        *filename_[i] = rstr;
        i++;
      }
      else
      { delete objectname_[i];
        delete filename_[i];
        HgMessage::error ("File does not exist.");
      }
      itemflag = 0;
    }

    if (i >= tablesize_)
    { HgMessage::error ("Objecttable too long, truncated");
      break;
    }
  }

  if (itemflag)
  { delete objectname_[i];
    delete filename_[i];
  }

  fclose (ftable);

  listsize_ = i;
  if (!listsize_)
    HgMessage::error ("Objecttable is empty");

  return listsize_;

} // load


int FileTable::getObjectName (int index, RString& name)
{
  if ((index < 0) && (index >= listsize_))
  { name = "";
    return 0;
  }

  name = *objectname_[index];
  return 1;

} // getObjectName


int FileTable::getFileName (int index, RString& name)
{
  if ((index < 0) && (index >= listsize_))
  { name = "";
    return 0;
  }

  name = path_;
  name += *filename_[index];
  return 1;

} // getFileName


declareActionCallback (InsertObjectDialog)
implementActionCallback (InsertObjectDialog)

declareFieldEditorCallback31 (InsertObjectDialog)
implementFieldEditorCallback31 (InsertObjectDialog)

InsertObjectDialog::InsertObjectDialog (
  WidgetKit* kit, const LayoutKit* layout, SceneWindow* scene, SceneMenus* themenus
)
{ 
  scene_ = scene;
  vrmlscene_ = (VRMLScene*)scene->editdata ();
  themenus_ = themenus;
  kit_ = kit;
  layout_ = layout;
  win_ = 0;

  primitive_[0] = QvNodeType::QvCone;
  primitive_[1] = QvNodeType::QvCube;
  primitive_[2] = QvNodeType::QvCylinder;
  primitive_[3] = QvNodeType::QvSphere;
  vrmlprimitive_ = -1;

  setbrowser_ = 0;
  objectbrowser_ = 0;
  filechooser_ = 0;
  buildWindow ();

} // InsertObjectDialog


InsertObjectDialog::~InsertObjectDialog ()
{
  delete[] setpath_;
  delete[] setname_;
  unmapWindow ();
  if (win_)  
    delete win_;  // destroy window

} // ~InsertObjectDialog


void InsertObjectDialog::initSetBrowser ()
{
  // number of sets
  long value = 0;
  kit_->style ()->find_attribute ("ObjectSets", value);
  int setnumber = (int)value + 1;

  setname_ = new RString[setnumber];
  setpath_ = new RString[setnumber];

  setname_[0] = "VRML Primitives";
  setbrowser_->append_item (kit_->label (setname_[0].string ()));

  // read x-resource 
  int i = 0;
  int number = 1;
  char buf[16];
  while (++i < setnumber) 
  {
    sprintf (buf, "ObjectSetName%d", i);
    kit_->style ()->find_attribute (buf, setname_[number]);
    sprintf (buf, "ObjectSetPath%d", i);
    kit_->style ()->find_attribute (buf, setpath_[number]);

    if (!(setname_[number] == "") && !(setpath_[number] == ""))
    { setbrowser_->append_item (kit_->label (setname_[number].string ()));
      number++;
    }
  }
/*
   if (!number)
     cerr << "no object sets found.\n";
*/
} // initSetBrowser


void InsertObjectDialog::loadSet ()
{
  int whichset = setbrowser_->selected ();
  if (whichset == -1)
    return;

  // clear object browser
  int i = objectbrowser_->selectable_count ();
  while (i--)
    objectbrowser_->remove_item ((GlyphIndex)i);

  // update object browser
  if (whichset)
  {
   RString name;
    int n = filetable_.load (setpath_[whichset].string ());
    i = -1;
    while (++i < n)
    { filetable_.getObjectName (i, name);
      objectbrowser_->append_item (kit_->label (name.string ()));
    }
  }
  else // vrml node primitives
    for (i = 0; i < primitivenumber_; i++)
      objectbrowser_->append_item (kit_->label (qvNodeName[primitive_[i]]));

  // title patch
  titlepatch_->body(
    layout_->hbox (
      kit_->label (setname_[whichset].string ()), 
      layout_->hglue ()
    )
  );
  titlepatch_->reallocate ();
  titlepatch_->redraw ();

  // deselect fieldbrowser
  fieldbrowser_->field ("");
  vrmlprimitive_ = -1;

} // loadSet


void InsertObjectDialog::toggleWindow ()
{
  if (!win_)  
    return;

  if (win_->is_mapped ())
    win_->unmap ();
  else
    win_->map ();

} // toggleWindow


void InsertObjectDialog::unmapWindow ()
{
  if (win_ && win_->is_mapped ())
  { 
    win_->unmap ();
    themenus_->editdialogClosed (SceneMenus::edlg_insertobj);
  }

} // unmapWindow


// inserts an object specified in the fieldbrowser into the vrmlscene

void InsertObjectDialog::insertObject ()
{
  if (vrmlprimitive_ > 0)
  {
    if (vrmlscene_->insertVRMLPrimitive (vrmlprimitive_))
      scene_->redraw ();
  }
  else
  {
    const char* name = fieldbrowser_->field ();
    if (!name || !*name)
      return;
    
    scene_->setInsertFlag (1);  // set flag, 
    if (!scene_->readSceneFile (name))  // and insert it into vrml scene
      scene_->redraw ();
    scene_->setInsertFlag (0);  // set flag, 
  }

} // insertObject


// calls the filecooser and writes the chosen filename into the fieldbrowser

void InsertObjectDialog::fileChooser ()
{
  if (!filechooser_)
  { kit_->begin_style ("Insert");
    Style* style = kit_->style ();

    style->attribute ("name", "Insert File"/* STranslate::str (STranslate::DialogTitleOPENFILE)*/);
    style->attribute ("open", STranslate::str (STranslate::DialogButtonOPEN));
    style->attribute ("cancel", WTranslate::str (WTranslate::CANCEL, slanguage));
    style->attribute ("subcaption", STranslate::str (STranslate::DialogStringFILENAME));

    filechooser_ = new  WFileChooser ("", kit_, style, 0, "*.wrl");
    Resource::ref (filechooser_);

    kit_->end_style ();
  }

  if (filechooser_->post_for_aligned (win_, 0.5, 0.5))  // "OK"
  { const char* filename = filechooser_->selected ();
    fieldbrowser_->field (filename);
    
    objectbrowser_->select (-1); // deselect object
    vrmlprimitive_ = -1;
  }

} // fileChooser


// writes the filename of the selected object in the fieldbrowser

void InsertObjectDialog::selectObject ()
{
  int selected = (int)objectbrowser_->selected ();
  if (selected == -1)
    return;

  int whichset = setbrowser_->selected ();
  if (whichset == -1)
    return;

  RString name;
  if (whichset)
  { filetable_.getFileName (selected, name);
    vrmlprimitive_ = -1;
  }
  else  // vrml primitive node
  { name += qvNodeName[primitive_[selected]];
    vrmlprimitive_ = primitive_[selected];
  }

  fieldbrowser_->field (name.string ());
  fieldbrowser_->scrollToEnd ();
    
} // selectObject


// inserts a www inline node with the specified url into the vrml scene

void InsertObjectDialog::makeWWWInline ()
{
  const char* filename = urleditor_->field ();
  if (vrmlscene_->insertWWWInline (filename))
  { 
    scene_->redraw ();
    urleditor_->clearModification ();
  }

} // makeWWWInline


void InsertObjectDialog::buildWindow ()
{
  if (win_)
    return;

  kit_->begin_style ("InsertObject", "Dialog");
  Style* style = kit_->style ();
  const char* title = STranslate::str (STranslate::EditdialogINSERTOBJECT);
  style->attribute ("name", title);
  style->attribute ("iconName", title);

  Action* makeinline = new ActionCallback(InsertObjectDialog) (this, &InsertObjectDialog::makeWWWInline);
  Action* close = new ActionCallback(InsertObjectDialog) (this, &InsertObjectDialog::unmapWindow);
  FieldEditorAction31* fe_action = new FieldEditorCallback31(InsertObjectDialog) (this,
    &InsertObjectDialog::accept, 0, 0);

  // object set and file insertion
  Action* selectobject = new ActionCallback(InsertObjectDialog) (this, &InsertObjectDialog::selectObject);
  Action* insertobject = new ActionCallback(InsertObjectDialog) (this, &InsertObjectDialog::insertObject);
  Action* filechooser = new ActionCallback(InsertObjectDialog) (this, &InsertObjectDialog::fileChooser);
  Action* loadset = new ActionCallback(InsertObjectDialog) (this, &InsertObjectDialog::loadSet);

  setbrowser_ = new WFileBrowser (kit_, loadset, close, loadset);
  setbrowser_->ref();
  initSetBrowser ();
  Glyph* setbrowser_box = layout_->hbox(
      kit_->inset_frame (layout_->vfixed (setbrowser_, 70)),
      kit_->vscroll_bar (setbrowser_->adjustable())
  );

  objectbrowser_ = new WFileBrowser (kit_, insertobject, close, selectobject);
  objectbrowser_->ref();
  Glyph* objectbrowser_box = layout_->hbox(
      kit_->inset_frame (layout_->natural (objectbrowser_, 100, 150)),
      kit_->vscroll_bar (objectbrowser_->adjustable())
  );

  Glyph* titleglyph = layout_->hbox (
    kit_->label (""),
    layout_->hglue ()
  );
  titlepatch_ = new Patch (titleglyph);

  fieldbrowser_ = new FieldBrowser (kit_, style, "");

  Glyph* insertobjectglyph = layout_->vbox (
    layout_->hbox (
      kit_->label ("Sets:"),
      layout_->hglue ()
    ),
    layout_->vspace (5),
    setbrowser_box,
    layout_->vspace (10),
    layout_->hbox (
      kit_->label ("Objectlist:"),
      layout_->hspace (10),
      titlepatch_
    ),
    layout_->vspace (5),
    objectbrowser_box,
    layout_->vspace (5),
    layout_->vbox (
      layout_->hbox (
        kit_->label ("File: "),
        fieldbrowser_,
        layout_->hspace (10),
        kit_->push_button ("File...", filechooser)
      ),
      layout_->vspace (5),
      layout_->hbox (
        layout_->hglue (),
        kit_->push_button ("Insert", insertobject),
        layout_->hglue ()
      )
    )
  );

  // url editor for www inline
  urleditor_ = new FieldEditor31 (kit_, style, 0, fe_action);
  urleditor_->ref();
  urleditor_->showModifications (1);

  Glyph* wwwinlineglyph = layout_->vbox (
    layout_->hbox (
      kit_->label ("URL: "),
      urleditor_
    ),
    layout_->vspace (5),
    layout_->hbox (
      layout_->hglue (),
      kit_->push_button ("Reference", makeinline),
      layout_->hglue ()
    )
  );
  
  // window
  Glyph* windowglyph =
    new Background (
      layout_->margin (
        layout_->vbox (
          kit_->inset_frame (
            layout_->margin (insertobjectglyph, 10)
          ),
          layout_->vspace (10),
          kit_->inset_frame (
            layout_->margin (wwwinlineglyph, 10)
          ),
          layout_->vspace (10),
          layout_->hbox (
            layout_->hglue (120, 100, 70),
            kit_->push_button ("Close", close),
            layout_->hglue (120, 100, 70)
          )
        ),
        10),
      kit_->background ()
    );

  win_ = new ApplicationWindow (windowglyph);
  win_->style (style);

  kit_->end_style (); 

} // buildWindow


void SceneMenus::toggleInsertObjectDialog ()
{
  if (!insertobjectdialog_)
    insertobjectdialog_ = new InsertObjectDialog (kit_, layout_, scene_, this);
  
  insertobjectdialog_->toggleWindow ();

} // SceneMenus::toggleInsertObjectDialog


/**** WWWAnchor *****/

declareActionCallback (WWWAnchorDialog)
implementActionCallback (WWWAnchorDialog)

declareFieldEditorCallback31(WWWAnchorDialog)
implementFieldEditorCallback31(WWWAnchorDialog)


WWWAnchorDialog::WWWAnchorDialog (
  WidgetKit* kit, const LayoutKit* layout, SceneWindow* scene, SceneMenus* themenus
)
{ 
  scene_ = scene;
  vrmlscene_ = (VRMLScene*)scene->editdata ();
  themenus_ = themenus;
  kit_ = kit;
  layout_ = layout;
  win_ = 0;

  urleditor_ = 0;
  descriptioneditor_ = 0;

  buildWindow ();

} // WWWAnchorDialog


WWWAnchorDialog::~WWWAnchorDialog ()
{
  unmapWindow ();
  vrmlscene_->setWWWAnchorDialog (0);
  if (win_)
    delete win_;

} // ~WWWAnchorDialog


void WWWAnchorDialog::update (const char* url, const char* description)
{
  org_url_ = url;
  org_description_ = description;

  urleditor_->field (org_url_.string ());
  descriptioneditor_->field (org_description_.string ());
  urleditor_->clearModification ();
  descriptioneditor_->clearModification ();

} // update
 

void WWWAnchorDialog::applyWWWAnchor ()
{
//   if (!*urleditor_->field ())  // description alone legal too
//     return;

  if (!vrmlscene_->applyWWWAnchor (urleditor_->field  (), descriptioneditor_->field  ()))
    return;

  urleditor_->clearModification ();
  descriptioneditor_->clearModification ();

} // applywwwanchor


void WWWAnchorDialog::deleteWWWAnchor ()
{
  vrmlscene_->deleteWWWAnchor ();

  urleditor_->field ("");
  descriptioneditor_->field ("");
  urleditor_->clearModification ();
  descriptioneditor_->clearModification ();

} // deleteWWWAnchor


void WWWAnchorDialog::reset ()
{
  urleditor_->field (org_url_.string ());
  descriptioneditor_->field (org_description_.string ());
  urleditor_->clearModification ();
  descriptioneditor_->clearModification ();

  if (*org_url_.string ())
    applyWWWAnchor ();
  else
    deleteWWWAnchor ();

} // reset


void WWWAnchorDialog::unmapWindow ()
{
  if (win_ && win_->is_mapped ())
  {
    win_->unmap ();
    vrmlscene_->setWWWAnchorDialog (0);
    themenus_->editdialogClosed (SceneMenus::edlg_anchor);
   }

} // unmapWindow


void WWWAnchorDialog::toggleWindow ()
{
  if (!win_)
    return;

  if (win_->is_mapped ())
  { win_->unmap ();
    vrmlscene_->setWWWAnchorDialog (0);
   }
  else
  { win_->map ();
    vrmlscene_->setWWWAnchorDialog (this);
    vrmlscene_->updateWWWAnchorDialog ();
  }

} // toggleWindow


// construct window

void WWWAnchorDialog::buildWindow ()
{
  if (win_)
    return;

  kit_->begin_style ("WWWAnchorDialog", "Dialog");
  Style* style = kit_->style ();
  const char* title = STranslate::str (STranslate::EditdialogANCHOREDITOR);
  style->attribute ("name", title);
  style->attribute ("iconName", title);

  Action* accept = new ActionCallback(WWWAnchorDialog) (this, &WWWAnchorDialog::applyWWWAnchor);
  Action* remove = new ActionCallback(WWWAnchorDialog) (this, &WWWAnchorDialog::deleteWWWAnchor);
  Action* reset = new ActionCallback(WWWAnchorDialog) (this, &WWWAnchorDialog::reset);
  Action* close = new ActionCallback(WWWAnchorDialog) (this, &WWWAnchorDialog::unmapWindow);
  FieldEditorAction31* fe_action = new FieldEditorCallback31(WWWAnchorDialog) (this, &WWWAnchorDialog::accept, 0, 0);

  // url box
  urleditor_ = new FieldEditor31 (kit_, style, org_url_, fe_action);
  urleditor_->ref();
  urleditor_->showModifications (1);
  descriptioneditor_ = new FieldEditor31 (kit_, style, org_description_, fe_action);
  descriptioneditor_->ref();
  descriptioneditor_->showModifications (1);

  TableGlyph* table = new TableGlyph (2, 2, 10, 0, 0);
  table->appendRow (
    layout_->hbox (
      kit_->label ("URL:"),
      layout_->hnatural (urleditor_, 200)
    )
  );
  table->appendRow (layout_->hbox (layout_->vspace (15)));
  table->appendRow (
    layout_->hbox (
      kit_->label ("Description:"),
      layout_->hnatural (descriptioneditor_, 200)
    )
  );
  Glyph* box = kit_->inset_frame (
    layout_->margin (
      table,
      10
    )
  );

  // buttons
  Glyph* buttons = layout_->hbox (
    layout_->hglue (0, 100, 0),
    kit_->push_button ("Apply", accept),
    layout_->hglue (20, 100, 30),
    kit_->push_button ("Delete", remove),
    layout_->hglue (20, 100, 30),
    kit_->push_button ("Reset", reset),
    layout_->hglue (20, 100, 30),
    kit_->push_button ("Close", close),
    layout_->hglue (0, 100, 0)
  );

  Glyph* editor_window = 
    new Background (
      layout_->margin (
        layout_->vbox (
          box,
          layout_->vspace (15),
          buttons
          ), 
        10),
      kit_->background ()
  );

  win_ = new ApplicationWindow (editor_window);
  win_->style (style);

  kit_->end_style ();

} // buildWindow


void SceneMenus::toggleWWWAnchorDialog ()
{
  if (!wwwanchordialog_)
    wwwanchordialog_ = new WWWAnchorDialog (kit_, layout_, scene_, this);

  wwwanchordialog_->toggleWindow ();

} // toggleWWWAnchorDialog


/* TextureDialog */


declareActionCallback (TextureDialog)
implementActionCallback (TextureDialog)

declareFieldEditorCallback31(TextureDialog)
implementFieldEditorCallback31(TextureDialog)


TextureDialog::TextureDialog (
  WidgetKit* kit, const LayoutKit* layout, SceneWindow* scene, SceneMenus* themenus
)
{ 
  scene_ = scene;
  vrmlscene_ = (VRMLScene*)scene->editdata ();
  themenus_ = themenus;
  kit_ = kit;
  layout_ = layout;
  win_ = 0;
  patch_ = 0;

  browser_ = 0;
  filechooser_ = 0;
  buildWindow ();

  clear ();

} // TextureDialog


TextureDialog::~TextureDialog ()
{ 
  unmapWindow ();
  vrmlscene_->setTextureDialog (0);
  if (win_)  
    delete win_;  // destroy window

} // ~TextureDialog 


// update dialog when new object was selected

void TextureDialog::update (
  const char* path, int wraph, int wrapv, 
  float transx, float transy, float rot, float scalex, float scaley
)
{
  org_values[0] = transx;
  org_values[1] = transy;
  org_values[2] = rot;
  org_values[3] = scalex;
  org_values[4] = scaley;

  org_wraph_ = wraph;
  org_wrapv_ = wrapv;

  org_url_ = path;

  applybutton_->state ()->set (TelltaleState::is_enabled, 1);
  urleditor_->enable ();
  reset ();

} // update


void TextureDialog::clear ()
{
  applybutton_->state ()->set (TelltaleState::is_enabled, 0);

  org_values[0] = 0;
  org_values[1] = 0;
  org_values[2] = 0;
  org_values[3] = 1;
  org_values[4] = 1;

  org_wraph_ = 0; // repeat
  org_wrapv_ = 0;

  org_url_ = "";
  urleditor_->disable ();

  reset ();
  
} // clear


// set origianal values

void TextureDialog::reset ()
{
  int i = 5;
  while (i--)
    slider_[i]->set_value (org_values[i]);

  wraph_button_->state ()->set (TelltaleState::is_chosen, !org_wraph_);
  wrapv_button_->state ()->set (TelltaleState::is_chosen, !org_wrapv_);

  urleditor_->field (org_url_.string ());
  urleditor_->scrollToEnd ();
  urleditor_->clearModification ();

} // reset


void TextureDialog::selectItem ()
{
  int selected = (int)browser_->selected ();
  RString name;
  filetable_.getFileName (selected, name);

  urleditor_->field (name.string ());
  urleditor_->scrollToEnd ();
  urleditor_->clearModification ();

  patch_->body (getImage (name.string ()));
  patch_->reallocate ();
  patch_->redraw ();

} // selectItem



// apply texture file and coordinates to selected object

void TextureDialog::applyTexture ()
{
  int wraph = !wraph_button_->state ()->test (TelltaleState::is_chosen);
  int wrapv = !wrapv_button_->state ()->test (TelltaleState::is_chosen);

  int redraw = vrmlscene_->applyTexture (
    urleditor_->field (), wraph, wrapv, 
    slider_[0]->get_value (), slider_[1]->get_value (), slider_[2]->get_value (), 
    slider_[3]->get_value (), slider_[4]->get_value ()
  );

  if (redraw)
    scene_->redraw ();

  urleditor_->clearModification ();

} // applyTexture


void TextureDialog::reloadFiletable ()
{
  // clear browser
  int i = filetable_.size ();
  while (i--)
    browser_->remove_item ((GlyphIndex)i);

  RString name;
  kit_->style ()->find_attribute ("TexturePath", name);
  int n = filetable_.load (name);

  i = -1;
  while (++i < n)
  { filetable_.getObjectName (i, name);
    browser_->append_item (kit_->label (name.string ()));
  }

  // clear image
  patch_->body (getImage (0));
  patch_->reallocate ();
  patch_->redraw ();

} // reloadFiletable


// open filechooser to select a texture file

void TextureDialog::fileChooser ()
{
  if (!filechooser_)
  { kit_->begin_style ("Insert");
    Style* style = kit_->style ();

    style->attribute ("name", "Choose Texture"/* STranslate::str (STranslate::DialogTitleOPENFILE)*/);
    style->attribute ("open", STranslate::str (STranslate::DialogButtonOPEN));
    style->attribute ("cancel", WTranslate::str (WTranslate::CANCEL, slanguage));
    style->attribute ("subcaption", STranslate::str (STranslate::DialogStringFILENAME));

    filechooser_ = new  WFileChooser ("", kit_, style);
    Resource::ref (filechooser_);

    kit_->end_style ();
  }

  if (filechooser_->post_for_aligned (win_, 0.5, 0.5))  // "OK"
  { const char* filename = filechooser_->selected ();
    org_url_ = filename;
    browser_->select (-1);
    reset ();

    patch_->body (getImage (filename));
    patch_->reallocate ();
    patch_->redraw ();
  }

} // fileChooser


void TextureDialog::unmapWindow ()
{
  if (win_ && win_->is_mapped ())
  {
    win_->unmap ();
    vrmlscene_->setTextureDialog (0);
    themenus_->editdialogClosed (SceneMenus::edlg_texture);
  }


} // unmapWindow


void TextureDialog::toggleWindow ()
{
  if (!win_)  
    return;

  if (win_->is_mapped ())
  { win_->unmap ();
    vrmlscene_->setTextureDialog (0);
  }
  else
  { win_->map ();
    vrmlscene_->setTextureDialog (this);
    vrmlscene_->updateTextureDialog ();
  }

} // toggleWindow


// reads an image file from harddisk and transforms it to a glyph

Image* TextureDialog::getImage (const char* path)
{
  Raster* raster = 0;
  Bitmap* bitmap = 0;
  HgRaster rasterimage;
  if (path && *path)
  { rasterimage.load (path, raster, bitmap);
    if (raster)
    { raster->setSize (100, 100);
      return new Image (raster);
    }
  }

  raster = new Raster (100, 100);
  int y, x = raster->pwidth ();
  while (x--)
  { y = raster->pheight ();
    while (y--)
      raster->pokeChr (x, y, 100, 100, 100);
  }

  return new Image (raster);

} // getImage



void TextureDialog::accept (FieldEditor31*, char)
{
  applyTexture ();
}


void TextureDialog::resetTexture ()
{
  reset ();
  applyTexture ();
}


void TextureDialog::buildWindow ()
{
  if (win_)
    return;

  kit_->begin_style ("TextureDialog", "Dialog");
  Style* style = kit_->style ();
  const char* title = STranslate::str (STranslate::EditdialogTEXTUREEDITOR);
  style->attribute ("name", title);
  style->attribute ("iconName", title);

  Action* accept = new ActionCallback(TextureDialog) (this, &TextureDialog::applyTexture);
  Action* reset = new ActionCallback(TextureDialog) (this, &TextureDialog::resetTexture);
  Action* reload = new ActionCallback(TextureDialog) (this, &TextureDialog::reloadFiletable);
  Action* close = new ActionCallback(TextureDialog) (this, &TextureDialog::unmapWindow);
  Action* select = new ActionCallback(TextureDialog) (this, &TextureDialog::selectItem);
  Action* filechooser = new ActionCallback(TextureDialog) (this, &TextureDialog::fileChooser);
  FieldEditorAction31* fe_action = new FieldEditorCallback31(TextureDialog) (this, &TextureDialog::accept, 0, 0);

  // image
  patch_ = new Patch (getImage (""));

  // filebrowser
  browser_ = new WFileBrowser (kit_, accept, close, select);  // accept: double click
  browser_->ref ();

  reloadFiletable ();

  // browser box and file button
  Glyph* browser_box = layout_->hbox (
    kit_->inset_frame (layout_->vnatural (browser_, 100)),
    kit_->vscroll_bar (browser_->adjustable ())
  );
  Glyph* texture_box = layout_->hbox (
    browser_box,
    layout_->hspace (10),
    layout_->vbox (
      layout_->vglue (),
      kit_->outset_frame (patch_),
      layout_->vglue ()
    )
  );



  // filename field browser
  urleditor_ = new FieldEditor31 (kit_, style, "", fe_action);
  urleditor_->ref();
  urleditor_->showModifications (1);

  Glyph* filenamebox = layout_->hbox (
    kit_->label ("URL:"),
    layout_->hspace (5),
    urleditor_,
    layout_->hspace (5),
    kit_->push_button ("File...", filechooser)
  );

  // sliders
  Glyph* label[num_sliders];
  label[0] = kit_->label ("Translation X");
  label[1] = kit_->label ("Translation Y");
  label[2] = kit_->label ("Rotation");
  label[3] = kit_->label ("Scalefactor X");
  label[4] = kit_->label ("Scalefactor Y");

  AdjValue* adjval[num_sliders];
  adjval[0] = new AdjValue (-10, 10, 0, 0.1, 1);  // translation x
  adjval[1] = new AdjValue (-10, 10, 0, 0.1, 1);  // translation y
  adjval[2] = new AdjValue (0, 6.28, 0, 0.1, 1);  // rotation
  adjval[3] = new AdjValue (-10, 10, 1, 0.1, 1);  // scalefactor x
  adjval[4] = new AdjValue (-10, 10, 1, 0.1, 1);  // scalefactor y

  TableGlyph* sliders = new TableGlyph (2, 2, 10, 0, 0);
  for (int i = 0; i < num_sliders;  i++)
  {
    slider_[i] = new FieldSlider (adjval [i], 2, "-10.00");
    sliders->appendRow (
      layout_->hbox (
        label[i],
        layout_->vmargin (layout_->hnatural (slider_[i], 100), 3)
      )
    );
  }

  wraph_button_ = kit_->check_box ("Repeat Horizontal", 0); // repeath);
  wrapv_button_ = kit_->check_box ("Repeat Vertical", 0); //repeatv);
  Glyph* checkbox = layout_->hbox (
    layout_->hglue (20.0, 140.0, 10.0),
    wraph_button_,
    layout_->hglue (20.0, 140.0, 10.0),
    wrapv_button_,
    layout_->hglue (20.0, 140.0, 10.0)
  );

  Glyph* sliderbox = kit_->inset_frame (
    layout_->margin (
      layout_->vbox (
        sliders,
        layout_->vspace (10),
        checkbox
      ),
      10
    )
  );

  // buttons
  applybutton_ = kit_->push_button ("Apply", accept);
  Glyph* buttons = layout_->hbox (
    layout_->hglue (0.0, 40.0, 0.0),
    applybutton_,
    layout_->hglue (20.0, 90.0, 15.0),
    kit_->push_button ("Reset",reset),
    layout_->hglue (20.0, 90.0, 15.0),
    kit_->push_button ("Reload", reload),
    layout_->hglue (20.0, 90.0, 15.0),
    kit_->push_button ("Close",close),
    layout_->hglue (0.0, 40.0, 0.0)
  );

  // window
  Glyph* browser_window = 
    new Background (
      layout_->margin (
        layout_->vbox (
          texture_box,
          layout_->vspace (10),
          filenamebox,
          layout_->vspace (10),
          sliderbox,
          layout_->vspace (10),
          buttons
          ), 
        10),
      kit_->background ()
  );


  win_ = new ApplicationWindow (browser_window);
  win_->style (style);

  kit_->end_style ();

} // buildWindow


void SceneMenus::toggleTextureDialog ()
{
  if (!texturedialog_)
    texturedialog_ = new TextureDialog (kit_, layout_, scene_, this);

  texturedialog_->toggleWindow ();

} // SceneMenus::toggleTextureDialog



/***** material editor *****/

declareActionCallback (MaterialDialog)
implementActionCallback (MaterialDialog)

declareColourDlgCallback(MaterialDialog)
implementColourDlgCallback(MaterialDialog)


MaterialDialog::MaterialDialog (SceneWindow* scene, SceneMenus* themenus)
{
  scene_ = scene;
  vrmlscene_ = (VRMLScene*)scene->editdata ();
  themenus_ = themenus;

  buildWindow ();

} // MaterialDialog


MaterialDialog::~MaterialDialog ()
{
  unmapWindow ();
  vrmlscene_->setMaterialDialog (0);
  delete colourdialog_;
}


void MaterialDialog::unmapWindow ()
{ 

  if (colourdialog_ && colourdialog_->ismapped ())
  {
    colourdialog_->unmap ();
    vrmlscene_->setMaterialDialog (0);
    themenus_->editdialogClosed (SceneMenus::edlg_material);
  }    

} // unmapWindow


void MaterialDialog::closedColourDialog (ColourDialog* /*coldlg*/)
{
  vrmlscene_->setMaterialDialog (0);
  themenus_->editdialogClosed (SceneMenus::edlg_material);
  colourdialog_->unmap (); // colour dialog does not unmap itself in this case

} // closedColourDialog


void MaterialDialog::toggleWindow ()
{
  if (colourdialog_->ismapped ())
  { colourdialog_->unmap ();
    vrmlscene_->setMaterialDialog (0);
  }
  else
  { colourdialog_->map ();
    vrmlscene_->setMaterialDialog (this);
    vrmlscene_->updateMaterialDialog ();
  }

} // toggleWindow


void MaterialDialog::update (
  float* ambientColor, float* diffuseColor, float* specularColor, float* emissiveColor, 
  float* shininess, float* transparency
)
{
  colourdialog_->setRGB (diffuseColor[0], diffuseColor[1], diffuseColor[2], 0);
  colourdialog_->setRGB (ambientColor[0], ambientColor[1], ambientColor[2], 1);
  colourdialog_->setRGB (specularColor[0], specularColor[1], specularColor[2], 2);
  colourdialog_->setRGB (emissiveColor[0], emissiveColor[1], emissiveColor[2], 3);
  colourdialog_->setRGB (*shininess, *shininess, *shininess, 4);
  colourdialog_->setRGB (*transparency, *transparency, *transparency, 5);

} // update


void MaterialDialog::applyMaterial (ColourDialog* /*coldlg*/)
{

  float ambientcolor[3], diffusecolor[3], specularcolor[3], emissivecolor[3];
  float r, g, b, shininess, transparency;
//  float shininess = 0, transparency = 0;  // TODO
  colourdialog_->getRGB (diffusecolor[0], diffusecolor[1], diffusecolor[2], 0);
  colourdialog_->getRGB (ambientcolor[0], ambientcolor[1], ambientcolor[2], 1);
  colourdialog_->getRGB (specularcolor[0], specularcolor[1], specularcolor[2], 2);
  colourdialog_->getRGB (emissivecolor[0], emissivecolor[1], emissivecolor[2], 3);
  colourdialog_->getRGB (r, g, b, 4);  shininess = (r + g + b) / 3.0;
  colourdialog_->getRGB (r, g, b, 5);  transparency = (r + g + b) / 3.0;

  if (vrmlscene_->applyMaterial (ambientcolor, diffusecolor, specularcolor, emissivecolor, shininess, transparency))
    scene_->redraw ();

} // applyMaterial


int MaterialDialog::isMapped ()
{ 
  return colourdialog_ && colourdialog_->ismapped (); 

} // isMapped


void MaterialDialog::buildWindow ()
{
  colourdialog_ = new ColourDialog (
    slanguage,
    STranslate::str (STranslate::EditdialogMATERIALEDITOR),
    new ColourDlgCallback(MaterialDialog) (this, &MaterialDialog::applyMaterial, &MaterialDialog::closedColourDialog)
  );

  colourdialog_->appendColourEntry ("Diffuse Color", 0.4, 0.4, 0.4);  // 0
  colourdialog_->appendColourEntry ("Ambient Color", 0.4, 0.4, 0.4);  // 1
  colourdialog_->appendColourEntry ("Specular Color", 0.4, 0.4, 0.4);  // 2
  colourdialog_->appendColourEntry ("Emissive Color", 0.4, 0.4, 0.4);  // 3
  colourdialog_->appendColourEntry ("Shininess Value", 0.4, 0.4, 0.4);  // 4
  colourdialog_->appendColourEntry ("Transparency Value", 0.4, 0.4, 0.4);  // 5

} // buildWindow


void SceneMenus::toggleMaterialDialog ()
{
  // editdata is a vrmlscene
  if (!materialdialog_)
    materialdialog_ = new MaterialDialog (scene_, this);

  materialdialog_->toggleWindow ();

} // SceneMenus::toggleMaterialDialog


/* structure viewer */

declareActionCallback (StructureViewerDialog)
implementActionCallback (StructureViewerDialog)

declareFieldEditorCallback31(StructureViewerDialog)
implementFieldEditorCallback31(StructureViewerDialog)


StructureViewerDialog::StructureViewerDialog (
  WidgetKit* kit, const LayoutKit* layout, SceneWindow* scene, SceneMenus* themenus
)
{ 
  scene_ = scene;
  vrmlscene_ = (VRMLScene*)scene->editdata ();
  themenus_ = themenus;
  kit_ = kit;
  layout_ = layout;
  win_ = 0;

  browser_ = 0;
  listsize_ = 0;

  upper_ = -1;

  buildWindow ();

} // StructureViewerDialog


StructureViewerDialog::~StructureViewerDialog ()
{ 
  unmapWindow ();
  if (win_)  
    delete win_;  // destroy window

} // ~StructureViewerDialog


void StructureViewerDialog::toggleWindow ()
{
  if (!win_)  
    return;

  if (win_->is_mapped ())
  { win_->unmap ();
    vrmlscene_->setStructureViewerDialog (0);
  }
  else
  { win_->map ();
    vrmlscene_->setStructureViewerDialog (this);
    vrmlscene_->updateStructureViewerDialog ();
  }

} // toggleWindow


void StructureViewerDialog::unmapWindow ()
{
  if (win_ && win_->is_mapped ())
  {
    win_->unmap ();
    vrmlscene_->setStructureViewerDialog (0);
    themenus_->editdialogClosed (SceneMenus::edlg_structure);
  }

} // toggleWindow


// clear filebrowser

void StructureViewerDialog::clear ()
{
  int i = listsize_;
  while (i--)
    browser_->remove_item ((GlyphIndex)i);
  listsize_ = 0;

}  // clear


// append item into filebrowser

void StructureViewerDialog::append (const char* node, const char* name, int level, int selection)
{

  Glyph* item = layout_->hbox (
    kit_->label (node),
    kit_->label ("  "),
    kit_->label (name)
  );

  if (selection)
    item = new Background (
      item,
      new Color ((ColorIntensity) 0.863, 0.729, 0.0196)  // dcba05
    );

//   if (selection)
//     kit_->outset_frame (item);

  browser_->append_item (item, 15 * level);
  listsize_++;

} // append


// select specified or last item

void StructureViewerDialog::selectItem (int item)
{
  if (item < 0)
    item = listsize_ - 1;
  browser_->select (item);

  RString name;
  vrmlscene_->getNameOfSelectedNode (name);
  setNodeNameField (name);

} // selectItem


void StructureViewerDialog::setPasteName (RString& name)
{
  pastename_->field (name.string ());

} // selectPasteName


void StructureViewerDialog::setNodeNameField (RString& name)
{
  nodename_->field (name.string ());
  nodename_->clearModification ();

} // selectPasteName

// action, if an item was chosen 

void StructureViewerDialog::selectNode ()
{
  int selected = browser_->selected ();

  vrmlscene_->applyNodeSelection (selected);
  scene_->redraw ();

  RString name;
  vrmlscene_->getNameOfSelectedNode (name);
  nodename_->field (name.string ());

} // selectNode


void StructureViewerDialog::applyName (FieldEditor31*, char)
{
  RString name (nodename_->field ());
  int selected = browser_->selected ();
  vrmlscene_->applyNodeName (selected, name);
  nodename_->clearModification ();

} // applyName


void StructureViewerDialog::viewSelectedNode ()
{ if (vrmlscene_->viewSelectedNode ())
  scene_->redraw ();
}
void StructureViewerDialog::deleteSelectedNode ()
{ if (vrmlscene_->deleteSelectedNode ())
  scene_->redraw ();
}
void StructureViewerDialog::cutSelectedNode ()
{ if (vrmlscene_->cutSelectedNode ())
  scene_->redraw ();
}
void StructureViewerDialog::copySelectedNode ()
{ if (vrmlscene_->copySelectedNode ())
  scene_->redraw ();
}
void StructureViewerDialog::pasteSelectedNode ()
{ if (vrmlscene_->pasteSelectedNode ())
  scene_->redraw ();
}
void StructureViewerDialog::selectParent ()
{ if (vrmlscene_->selectParent ())
  scene_->redraw ();
}
void StructureViewerDialog::selectChild ()
{ if (vrmlscene_->selectChild ())
  scene_->redraw ();
}
void StructureViewerDialog::selectNext ()
{ if (vrmlscene_->selectNext (1))
  scene_->redraw ();
}
void StructureViewerDialog::selectPrevious ()
{ if (vrmlscene_->selectNext (-1))
  scene_->redraw ();
}
void StructureViewerDialog::removeSelection ()
{ if (scene_->selectNode (0))
  { vrmlscene_->updateEditContext ();
    scene_->redraw ();
  }
  
}


void StructureViewerDialog::storeAdjustable ()
{
  if (!browser_)
    return;

  upper_ = browser_->adjustable ()->cur_upper (Dimension_Y);
}


void StructureViewerDialog::useStoredAjustable ()
{ 
  if (!browser_)
    return;

  float upper = browser_->adjustable ()->cur_upper (Dimension_Y);
  float length = browser_->adjustable ()->cur_length (Dimension_Y);
  int selected = browser_->selected ();
  float newlower;
  if (upper_ > 0)
    newlower = upper_ - length + 1;
  else
    newlower = upper - length + 1;

  // cerr << "selected: " << selected << " of " << listsize_ << ", upper: " << upper_
  // << ", newlower: " << newlower << ", length: " << length << endl;

  // topmost vis. item: topvis = size-upper-1 = size-lower-length  // upper = lower+length-1
  // bottommost vis. item: botvis = size-lower-1
  // selected visible when topvis <= selected <= botvis

  if (selected < listsize_ - newlower - length)
    newlower = listsize_ - selected - length;
  else if (selected > listsize_ - newlower - 1)
    newlower = listsize_ - selected - 1;
  // use length/2 if you prefer centered item after scrolling

  browser_->adjustable ()->scroll_to (Dimension_Y, newlower);
}


// build window with filebrowser

void StructureViewerDialog::buildWindow ()
{
  if (win_)
    return;

  kit_->begin_style ("StructureViewer", "Dialog");
  Style* style = kit_->style ();
  const char* title = STranslate::str (STranslate::EditdialogSTRUCTUREVIEWER);
  style->attribute ("name", title);
  style->attribute ("iconName", title);

  // actions
  Action* view = new ActionCallback(StructureViewerDialog) (this, &StructureViewerDialog::viewSelectedNode);
  Action* del = new ActionCallback(StructureViewerDialog) (this, &StructureViewerDialog::deleteSelectedNode);
  Action* cut = new ActionCallback(StructureViewerDialog) (this, &StructureViewerDialog::cutSelectedNode);
  Action* copy = new ActionCallback(StructureViewerDialog) (this, &StructureViewerDialog::copySelectedNode);
  Action* paste = new ActionCallback(StructureViewerDialog) (this, &StructureViewerDialog::pasteSelectedNode);
  Action* parent = new ActionCallback(StructureViewerDialog) (this, &StructureViewerDialog::selectParent);
  Action* child = new ActionCallback(StructureViewerDialog) (this, &StructureViewerDialog::selectChild);
  Action* next = new ActionCallback(StructureViewerDialog) (this, &StructureViewerDialog::selectNext);
  Action* previous = new ActionCallback(StructureViewerDialog) (this, &StructureViewerDialog::selectPrevious);
  Action* deselect = new ActionCallback(StructureViewerDialog) (this, &StructureViewerDialog::removeSelection);
  Action* accept = new ActionCallback(StructureViewerDialog) (this, &StructureViewerDialog::selectNode);
  Action* select = new ActionCallback(StructureViewerDialog) (this, &StructureViewerDialog::selectNode);
  Action* close  = new ActionCallback(StructureViewerDialog) (this, &StructureViewerDialog::unmapWindow);

  FieldEditorAction31* applyname = 
    new FieldEditorCallback31(StructureViewerDialog) (this, &StructureViewerDialog::applyName, 0, 0);
  nodename_ = new FieldEditor31 (kit_, style, "", applyname);
  nodename_->showModifications (1);

  browser_ = new WFileBrowser (kit_, accept, close, select);  // accept: double click
  browser_->ref();

  // build Window
  Glyph* titlename = layout_->hbox (
    kit_->label ("VRML Structure:"),
    layout_->hglue (100, 400, 0)
  );
  Glyph* browser_box = layout_->hbox(
      kit_->inset_frame(layout_->natural (browser_, 100, 300)),
      kit_->vscroll_bar(browser_->adjustable())
  );
  Glyph* structure = layout_->vbox (
    titlename,
    layout_->vspace (5),
    browser_box,
    layout_->vspace (10),
    layout_->hbox (
      kit_->label ("Name: "),
      nodename_,
      layout_->hspace (15)
    ),
    layout_->vspace (5)
  );

  // button bar
  pastename_ = new FieldBrowser (kit_, style, "");
  Glyph* buttons = layout_->vbox (
    layout_->vspace (20),
    kit_->push_button ("View", view),
    layout_->vspace (10),
    kit_->push_button ("Parent", parent),
    kit_->push_button ("Child", child),
    kit_->push_button ("Previous", previous),
    kit_->push_button ("Next", next),
    kit_->push_button ("Deselect", deselect),
    layout_->vspace (10)
  );
  buttons = layout_->vbox (
    buttons, // max. 10 arguments in vbox
    kit_->push_button ("Delete", del),
    kit_->push_button ("Cut", cut),
    kit_->push_button ("Copy", copy)
  );
  buttons->append (kit_->push_button ("Paste", paste));
  buttons->append (layout_->vspace (10));
  buttons->append (kit_->label ("Paste:"));
  buttons->append (pastename_);
  buttons->append (layout_->vglue (10, 400, 0));
  buttons->append (kit_->push_button ("Close", close));

  // window
  Glyph* browser_window = 
    new Background (
      layout_->margin (
        layout_->hbox (
          structure,
          layout_->hspace (15),
          buttons
        ),
        10
      ),
      kit_->background ()
  );

  win_ = new ApplicationWindow (browser_window);
  win_->style (style);

  kit_->end_style ();

} // buildWindow


// toggle StructureViewerDialog window

void SceneMenus::toggleStructureViewerDialog ()
{
  // editdata is a vrmlscene
  if (!structureviewerdialog_)
    structureviewerdialog_ = new StructureViewerDialog (kit_, layout_, scene_, this);

  structureviewerdialog_->toggleWindow ();

} // SceneMenus::toggleStructureViewerDialog


/* grid dialog */

declareActionCallback (GridDialog)
implementActionCallback (GridDialog)


GridDialog::GridDialog (
  WidgetKit* kit, const LayoutKit* layout, SceneWindow* scene, SceneMenus* themenus
)
{
  scene_ = scene;
  vrmlscene_ = (VRMLScene*)scene->editdata ();
  themenus_ = themenus;
  kit_ = kit;
  layout_ = layout;
  win_ = 0;

  absolutemetric_ = 0;
  buildWindow ();

} // GridDialog


GridDialog::~GridDialog ()
{
  unmapWindow ();
  if (win_)
    delete win_;

} // ~GridDialog


// toggle application window

void GridDialog::toggleWindow ()
{
  if (!win_)  
    return;

  if (win_->is_mapped ())
  { win_->unmap ();
    vrmlscene_->setGridDialog (0);
  }
  else
  { win_->map ();
    vrmlscene_->setGridDialog (this);
    vrmlscene_->updateGridDialog ();
  }

} // toggleWindow


// unmap application window

void GridDialog::unmapWindow ()
{
  if (win_ && win_->is_mapped ())
  {
    win_->unmap ();
    vrmlscene_->setGridDialog (0);
    themenus_->editdialogClosed (SceneMenus::edlg_grid);
  }

} // unmapWindow


// apply current values in the dialog to the scene

void GridDialog::setGrid ()
{
  // get values from fieldsliders
  vector3D axis = { 
    checkbox_[0]->state ()->test (TelltaleState::is_chosen),
    checkbox_[1]->state ()->test (TelltaleState::is_chosen),
    checkbox_[2]->state ()->test (TelltaleState::is_chosen)
  };
  vector3D position = { 
    slider_[0]->get_value (), 
    slider_[1]->get_value (), 
    slider_[2]->get_value () 
  };
  float extent = slider_[3]->get_value ();
  float distance = slider_[4]->get_value ();

  if (absolutemetric_)
  { // convert to relative metric system
    float l[5],u[5];
    int i = 5;
    while (i--)
    { l[i] = adjvalue_[i]->lower (Dimension_X);
      u[i] = adjvalue_[i]->upper (Dimension_X);
    }
    position.x = ((slider_[0]->get_value () - l[0]) / (u[0] - l[0])) - 0.5;
    position.y = ((slider_[1]->get_value () - l[1]) / (u[1] - l[1])) - 0.5;
    position.z = ((slider_[2]->get_value () - l[2]) / (u[2] - l[2])) - 0.5;
    extent     = ((slider_[3]->get_value () - l[3]) / (u[3] - l[3])) * 4 + 1;
    distance   = ((slider_[4]->get_value () - l[4]) / (u[4] - l[4])) * 0.49 + 0.01;
  }

  // update vrml scene
  vrmlscene_->setGrid (axis, position, extent, distance);
  scene_->redraw ();

} // setGrid


// reset values in dialog

void GridDialog::resetValues ()
{
  int i = 4;
  while (i--)
    slider_[i]->set_value (adjvalue_[i]->lower (Dimension_X));

  if (absolutemetric_)
  { float l = adjvalue_[4]->lower (Dimension_X);
    float u = adjvalue_[4]->upper (Dimension_X);
    slider_[4]->set_value ((u - l) / 49 * 9 + l);
  }
  else 
    slider_[4]->set_value (0.1);

  setGrid ();

} // resetValues


// changes boundareie of sliders and calculates relative values

void GridDialog::changeBoundaries (float* lower, float* upper)
{
  float value[5], l, u;

  // calculate relative values
  int i = 5;
  while (i--)
  {
    l = adjvalue_[i]->lower (Dimension_X);
    u = adjvalue_[i]->upper (Dimension_X);
    value[i] = ((slider_[i]->get_value () - l) / (u - l));
  }

  // set values into new context
  i = 5;
  while (i--)
  {
    adjvalue_[i]->setLower (lower[i]);
    adjvalue_[i]->setUpper (upper[i]);
    slider_[i]->set_value (lower[i] + (upper[i] - lower[i]) * value[i]);
  }

} // changeBoundaries


void GridDialog::setToRelativeMetric ()
{
  absolutemetric_ = 0;

  float lower[5] = { -0.5,  -0.5,  -0.5, 1, 0.01 };
  float upper[5] = { 0.5,  0.5, 0.5, 5, 0.5 };
  changeBoundaries (lower, upper);

  setGrid ();

} // setToRelativeMetric


void GridDialog::setToAbsoluteMetric ()
{
  absolutemetric_ = 1;

  vector3D min, max;
  vrmlscene_->getSceneExtent (min, max);

  // maximum extent of scene
  // vector3D se = max;
  // dec3D (se, min);
  float extent = scene_->size ();  // sqrtf (se.x * se.x + se.y * se.y + se.z * se.z);
  
  float lower[5] = { min.x, min.y, min.z, extent, extent / 100 };
  float upper[5] = { max.x, max.y, max.z, 5 * extent, extent / 2};
  changeBoundaries (lower, upper);

  setGrid ();

} // setToAbsoluteMetric


// updates boundaries for absolute metrics, if the scene extent has changed

void GridDialog::update ()
{
  if (absolutemetric_)
    setToAbsoluteMetric ();
    
} // update


// build the window

void GridDialog::buildWindow ()
{
  if (win_)
    return;

  kit_->begin_style ("GridDialog", "Dialog");
  Style* style = kit_->style ();
  const char* title = STranslate::str (STranslate::EditdialogGRIDSETTINGS);
  style->attribute ("name", title);
  style->attribute ("iconName", title);

  // actions
  Action* close = new ActionCallback(GridDialog) (this, &GridDialog::unmapWindow);
  Action* apply = new ActionCallback(GridDialog) (this, &GridDialog::applyValues);
  Action* reset = new ActionCallback(GridDialog) (this, &GridDialog::resetValues);
  Action* check0 = new ActionCallback(GridDialog) (this, &GridDialog::xaxis);
  Action* check1 = new ActionCallback(GridDialog) (this, &GridDialog::yaxis);
  Action* check2 = new ActionCallback(GridDialog) (this, &GridDialog::zaxis);
  Action* radio0 = new ActionCallback(GridDialog) (this, &GridDialog::setToRelativeMetric);
  Action* radio1 = new ActionCallback(GridDialog) (this, &GridDialog::setToAbsoluteMetric);

  // sliders
  // axis and position of axis
  checkbox_[0] = kit_->check_box ("Axis X", check0);
  checkbox_[1] = kit_->check_box ("Axis Y", check1);
  checkbox_[2] = kit_->check_box ("Axis Z", check2);

  TableGlyph* table1 = new TableGlyph (2, 5, 10, 0, 0);
  for (int i = 0; i < 3;  i++)
  { 
    adjvalue_[i] = new AdjValue (-0.5, 0.5, -0.5, 0.1, 0.5);
    slider_[i] = new FieldSlider (adjvalue_[i], 2, "-0.001");
    table1->appendRow (
      layout_->hbox (
        checkbox_[i],
        layout_->vmargin (
          layout_->hfixed (slider_[i], 250),
          5
        )
      )
    );
  }

  // extent
  adjvalue_[3] = new AdjValue (1, 5, 1, 0.5, 2);
  slider_[3] = new FieldSlider (adjvalue_[3], 2, "-0.001");
  TableGlyph* table2 = new TableGlyph (2, 2, 10, 0, 0);
  table2->appendRow (
    layout_->hbox (
      kit_->label ("Extent"),
      layout_->vmargin (
        layout_->hfixed (slider_[3], 250),
        5
      )
    )
  );
  // distance
  adjvalue_[4] = new AdjValue (0.01, 0.5, 0.1, 0.05, 0.5);
  slider_[4] = new FieldSlider (adjvalue_[4], 2, "-0.001");
  table2->appendRow (
    layout_->hbox (
      kit_->label ("Distance"),
      layout_->vmargin (
        layout_->hfixed (slider_[4], 250),
        5
      )
    )
  );

  Glyph* sliders = layout_->vbox (
    table1,
    layout_->vspace (10),
    layout_->hbox (
      layout_->hglue (),
      table2,
      layout_->hglue ()
    )
  );

  // relative/absolute radiobuttons
  TelltaleGroup* ttgroup = new TelltaleGroup ();
  Button* radiobutton0 = kit_->radio_button (ttgroup, "relative to scene", radio0);
  Button* radiobutton1 = kit_->radio_button (ttgroup, "absolute values", radio1);
  radiobutton0->state ()->set (TelltaleState::is_chosen, 1);

  Glyph* radiobuttons = layout_->hbox (
    layout_->hglue (),
    layout_->vbox (
      kit_->label ("Metric System:"),
      layout_->vglue ()
    ),
    layout_->hspace (10),
    layout_->vbox (
      radiobutton0,
      layout_->vspace (10),
      radiobutton1
    ),
    layout_->hglue ()
  );

  // push buttons
  Glyph* buttons = layout_->hbox (
    layout_->hglue (),
    kit_->push_button ("Apply", apply),
    layout_->hglue (),
    kit_->push_button ("Reset", reset),
    layout_->hglue (),
    kit_->push_button ("Close", close),
    layout_->hglue ()
  );

  // window
  Glyph* window = new Background (
    layout_->margin (
      layout_->vbox (
        kit_->inset_frame (
          layout_->margin (
            layout_->vbox (
              sliders,
              layout_->vspace (20),
              radiobuttons
            ),
            10
          )
        ),
        layout_->vspace (10),
        buttons
      ),
      10),
    kit_->background ()
  );

  win_ = new ApplicationWindow (window);
  win_->style (style);

  kit_->end_style ();

  resetValues ();

} // buildWindow


void SceneMenus::toggleGridDialog ()
{
  if (!griddialog_)
    griddialog_ = new GridDialog (kit_, layout_, scene_, this);

  griddialog_->toggleWindow ();

} // SceneMenus::toggleGridDialog


/* transform dialog */

declareActionCallback (TransformDialog)
implementActionCallback (TransformDialog)

declareFieldEditorCallback31(TransformDialog)
implementFieldEditorCallback31(TransformDialog)

TransformDialog::TransformDialog (
  WidgetKit* kit, const LayoutKit* layout, SceneWindow* scene, SceneMenus* themenus
)
{
  scene_ = scene;
  vrmlscene_ = (VRMLScene*)scene->editdata ();
  themenus_ = themenus;
  kit_ = kit;
  layout_ = layout;
  win_ = 0;

  coordinatesystem_ = model;

  buildWindow ();

} // TransformDialog


// destructor

TransformDialog::~TransformDialog ()
{
  unmapWindow ();
  if (win_)
    delete win_;

} // ~TransformDialog


// toggle application window

void TransformDialog::toggleWindow ()
{
  if (!win_)  
    return;

  if (win_->is_mapped ())
  { win_->unmap ();
    vrmlscene_->setTransformDialog (0);
  }
  else
  { win_->map ();
    vrmlscene_->setTransformDialog (this);
    vrmlscene_->updateTransformDialog ();
  }
} // toggleWindow


// unmap application window

void TransformDialog::unmapWindow ()
{
  if (win_ && win_->is_mapped ())
  {
    win_->unmap ();
    vrmlscene_->setTransformDialog (0);
    themenus_->editdialogClosed (SceneMenus::edlg_transform);
  }

} // unmapWindow


void TransformDialog::toggleCoordSystem ()
{
  if (coordinatesystem_ == world)
    coordinatesystem_ = model;
  else
    coordinatesystem_ = world;
  vrmlscene_->updateTransformDialog ();
}


// use current values of selected trasform

void TransformDialog::resetValues ()
{
  vrmlscene_->updateTransformDialog ();

} // resetValues


// clears all fields

void TransformDialog::clear ()
{
  int i = 3;
  while (i--)
  {
    translate_[i]->field ("");
    rotate_[i]->field ("");
    scale_[i]->field ("");
    translate_[i]->clearModification ();
    rotate_[i]->clearModification ();
    scale_[i]->clearModification ();
  }
  angle_->field ("");

} // clear


// clears rotation field (for world coordinates)

void TransformDialog::clearRotation ()
{
  int i = 3;
  while (i--)
  {
    rotate_[i]->field ("");
    rotate_[i]->clearModification ();
  }
  angle_->field ("");
  angle_->clearModification ();

} // clearRotation


void TransformDialog::applyValues ()
{
  if (coordinatesystem_ == world)
  { cerr << "world coordinates can not be applied\n";
    resetValues ();
    return;
  }

  vector3D translate = { 
    atof (translate_[0]->field ()), atof (translate_[1]->field ()), atof (translate_[2]->field ()) 
  };
  vector3D rotate = { 
    atof (rotate_[0]->field ()), atof (rotate_[1]->field ()), atof (rotate_[2]->field ()) 
  };
  vector3D scale = { 
    atof (scale_[0]->field ()), atof (scale_[1]->field ()), atof (scale_[2]->field ()) 
  };
  float angle = atof (angle_->field ());

  if (vrmlscene_->applyTransform (&translate, &rotate, &angle, &scale))
    scene_->redraw ();

  vrmlscene_->updateTransformDialog (); // corrects input errors, i.e. letters

} // applyValues


void TransformDialog::updateTranslation (const vector3D& translate)
{
  char buf[16];
  const float* s = &translate.x;

  int i = 3;
  while (i--)
  { sprintf (buf, "%4.2f", s[i]);
    translate_[i]->field (buf);
    translate_[i]->clearModification ();
  }

} // updateTranslation


void TransformDialog::updateRotation (const vector3D& rotate, float angle)
{
  char buf[16];
  const float* s = &rotate.x;

  int i = 3;
  while (i--)
  { sprintf (buf, "%4.2f", s[i]);
    rotate_[i]->field (buf);
    rotate_[i]->clearModification ();
  }
  sprintf (buf, "%4.2f", angle);
  angle_->field (buf);
  angle_->clearModification ();

} // updateRotation


void TransformDialog::updateScaling (const vector3D& scale)
{
  char buf[16];
  const float* s = &scale.x;

  int i = 3;
  while (i--)
  { sprintf (buf, "%4.2f", s[i]);
    scale_[i]->field (buf);
    scale_[i]->clearModification ();
  }

} // updateScaling


void TransformDialog::buildWindow ()
{
  if (win_)
    return;

  kit_->begin_style ("TransformDialog", "Dialog");
  Style* style = kit_->style ();
  const char* title = STranslate::str (STranslate::EditdialogTRANSFORMEDITOR);
  style->attribute ("name", title);
  style->attribute ("iconName", title);

  // actions
  Action* close = new ActionCallback(TransformDialog) (this, &TransformDialog::unmapWindow);
  Action* apply = new ActionCallback(TransformDialog) (this, &TransformDialog::applyValues);
  Action* reset = new ActionCallback(TransformDialog) (this, &TransformDialog::resetValues);
  FieldEditorAction31* fe_action = 
    new FieldEditorCallback31(TransformDialog) (this, &TransformDialog::accept, 0, 0);
  Action* toggle_cs = new ActionCallback(TransformDialog) (this, &TransformDialog::toggleCoordSystem);

  for (int i = 0; i < 3; i++)
  {
    translate_[i] = new FieldEditor31 (kit_, style, "0", fe_action);
    rotate_[i] = new FieldEditor31 (kit_, style, "0", fe_action);
    scale_[i] = new FieldEditor31 (kit_, style, "0", fe_action);
    translate_[i]->showModifications (1);
    rotate_[i]->showModifications (1);
    scale_[i]->showModifications (1);
  }
  angle_ = new FieldEditor31 (kit_, style, "0", fe_action);
  angle_->showModifications (1);
  float fieldsize = kit_->font ()->width ("-10.00", 6) + 15;

  TableGlyph* table = new TableGlyph (5, 7, 10, 20, 5);
/*
  table->appendRow (
    layout_->hbox (
      layout_->hglue (),
      layout_->hcenter (kit_->label ("X")), 
      layout_->hcenter (kit_->label ("Y")),
      layout_->hcenter (kit_->label ("Z")),
      layout_->hcenter (kit_->label ("Angle"))
    )
  );
*/
  table->appendRow (layout_->hbox (layout_->vspace (5)));
  table->appendRow (
    layout_->hbox (
      kit_->label ("Translate"),
      layout_->hfixed (translate_[0], fieldsize),
      layout_->hfixed (translate_[1], fieldsize),
      layout_->hfixed (translate_[2], fieldsize),
      layout_->hglue ()
    )
  );
  table->appendRow (layout_->hbox (layout_->vspace (5)));
  table->appendRow (
    layout_->hbox (
      kit_->label ("Rotate"),
      layout_->hfixed (rotate_[0], fieldsize),
      layout_->hfixed (rotate_[1], fieldsize),
      layout_->hfixed (rotate_[2], fieldsize),
      layout_->hfixed (angle_, fieldsize)
    )
  );
  table->appendRow (layout_->hbox (layout_->vspace (5)));
  table->appendRow (
    layout_->hbox (
      kit_->label ("Scalefactor"),
      layout_->hfixed (scale_[0], fieldsize),
      layout_->hfixed (scale_[1], fieldsize),
      layout_->hfixed (scale_[2], fieldsize),
      layout_->hglue ()

    )
  );
/*
  // radiobuttons model/world coordinates
  TelltaleGroup* ttgroup = new TelltaleGroup ();
  Glyph* radiobuttons = layout_->vbox (
    kit_->radio_button (ttgroup, "Model Coordinates", model),
    layout_->vspace (5),
    kit_->radio_button (ttgroup, "World Coortiantes", world)
  );
*/
  Glyph* checkbox = layout_->hbox (
    layout_->hglue (),
    kit_->check_box ("World Coordinate System", toggle_cs),
    layout_->hglue ()
  );

  // buttons
  Glyph* buttons = layout_->hbox (
    layout_->hglue (20.0, 50.0, 20.0),
    kit_->push_button ("Apply", apply),
    layout_->hglue (20.0, 50.0, 20.0),
    kit_->push_button ("Reset", reset),
    layout_->hglue (20.0, 50.0, 20.0),
    kit_->push_button ("Close", close),
    layout_->hglue (20.0, 50.0, 20.0)
  );

  // window
  Glyph* window = 
    new Background (
      layout_->margin (
        layout_->vbox (
          kit_->inset_frame (
            layout_->margin (
              layout_->vbox (
                table,
                layout_->vspace (10),
                checkbox
              ),
              10
            )
          ),
          layout_->vspace (10),
          buttons
        ),
        10
      ),
      kit_->background ()
    );

  win_ = new ApplicationWindow (window);
  win_->style (style);

  kit_->end_style ();

} // buildWindow


void SceneMenus::toggleTransformDialog ()
{
  if (!transformdialog_)
    transformdialog_ = new TransformDialog (kit_, layout_, scene_, this);

  transformdialog_->toggleWindow ();

} // toggleTransformDialog


/**** viewpoint dialog ****/

declareActionCallback (ViewpointDialog)
implementActionCallback (ViewpointDialog)

declareFieldEditorCallback31(ViewpointDialog)
implementFieldEditorCallback31(ViewpointDialog)

ViewpointDialog::ViewpointDialog (
  WidgetKit* kit, const LayoutKit* layout, SceneWindow* scene, SceneMenus* themenus
)
{
  scene_ = scene;
  vrmlscene_ = (VRMLScene*)scene->editdata ();
  themenus_ = themenus;
  kit_ = kit;
  layout_ = layout;
  win_ = 0;

  browser_ = 0;
  listsize_ = 0;

  buildWindow ();

} // ViewpointDialog


// destructor

ViewpointDialog::~ViewpointDialog ()
{
  unmapWindow ();
  vrmlscene_->setViewpointDialog (0);
  if (win_)
    delete win_;

} // ~ViewpointDialog


// toggle application window

void ViewpointDialog::toggleWindow ()
{
  if (!win_)  
    return;

  if (win_->is_mapped ())
  { win_->unmap ();
    vrmlscene_->setViewpointDialog (0);
  }
  else
  { win_->map ();
    vrmlscene_->setViewpointDialog (this);
    vrmlscene_->updateViewpointDialog ();
  }
} // toggleWindow


// unmap application window

void ViewpointDialog::unmapWindow ()
{
  if (win_ && win_->is_mapped ())
  {
    win_->unmap ();
    vrmlscene_->setViewpointDialog (0);
    themenus_->editdialogClosed (SceneMenus::edlg_viewpoint);
  }

} // unmapWindow



// clear filebrowser

void ViewpointDialog::clear ()
{
  int i = listsize_;
  while (i--)
    browser_->remove_item ((GlyphIndex)i);
  listsize_ = 0;

}  // clear


// append item into filebrowser

void ViewpointDialog::append (const char* number, const char* name)
{
  Glyph* item = layout_->hbox (
    kit_->label (number),
    layout_->hspace (5),
    kit_->label (name)
  );

  browser_->append_item (item);
  listsize_++;

} // append


// 

void ViewpointDialog::select (int item)
{
  if (item >= listsize_)
    item = listsize_ - 1;
  
  browser_->select (item);
  vrmlscene_->updateViewpointValues (item);

} // select


// action, if an item was selected

void ViewpointDialog::selectItem ()
{
/*
  int selected = browser_->selected ();
  vrmlscene_->updateViewpointValues (selected);
*/
  acceptItem ();

} // selectItem


void ViewpointDialog::acceptItem ()
{
  int selected = browser_->selected ();
  
  vrmlscene_->switchtoViewpoint (selected);
  scene_->redraw ();

  vrmlscene_->updateViewpointValues (selected);

} // acceptItem



void ViewpointDialog::applyViewpoint ()
{
  int selected = browser_->selected ();
  if (selected == -1)
    return; // no viewpoint is selected

  point3D position = { 
    atof (position_[0]->field ()), atof (position_[1]->field ()), atof (position_[2]->field ()) 
  };
  vector3D orientation = { 
    atof (orientation_[0]->field ()), atof (orientation_[1]->field ()), atof (orientation_[2]->field ()) 
  };
  float angle = atof (angle_->field ());
  float viewangle = atof (viewangle_->field ());

  // set values to scene graph
  vrmlscene_->applyViewpoint (selected, position, orientation, angle, viewangle, name_->field ());

  vrmlscene_->switchtoViewpoint (selected); // activate viewpoint
  scene_->redraw ();

  vrmlscene_->updateViewpointDialog ();
//   // update dialog
//   if (name_->modified ())
//     browser_->replace ...

//   int i = 3;
//   while (i--)
//   { position_[i]->clearModification ();
//     orientation_[i]->clearModification ();
//   }
//   angle_->clearModification ();
//   viewangle_->clearModification ();
//   name_->clearModification ();
  
} // applyViewpoint


void ViewpointDialog::clearValues ()
{
  int i = 3;
  while (i--)
  {
    position_[i]->field ("");
    orientation_[i]->field ("");
    position_[i]->clearModification ();
    orientation_[i]->clearModification ();
  }

  angle_->field ("");
  viewangle_->field ("");
  name_->field ("");
  camertype_->field ("");

  angle_->clearModification ();
  viewangle_->clearModification ();
  name_->clearModification ();

} // clearValues


void ViewpointDialog::updateValues (
  const char* name, const float* pos, const float* axis, float angle, float viewangle, const char* camtype
)
{
  char buf[16];
  for (int i = 0; i < 3; i++)
  { 
    sprintf (buf, "%4.2f", pos[i]);
    position_[i]->field (buf);
    sprintf (buf, "%4.2f", axis[i]);
    orientation_[i]->field (buf);

    position_[i]->clearModification ();
    orientation_[i]->clearModification ();
  }
  sprintf (buf, "%4.2f", angle);
  angle_->field (buf);
  angle_->clearModification ();

  sprintf (buf, "%4.2f", viewangle);
  viewangle_->field (buf);
  viewangle_->clearModification ();

  name_->field (name);
  name_->clearModification ();

  camertype_->field (camtype);

} // updateValues


void ViewpointDialog::resetValues ()
{
  int selected = browser_->selected ();
  vrmlscene_->updateViewpointValues (selected);

} // resetValues


void ViewpointDialog::setViewpoint ()
{
  int selected = browser_->selected ();
  if (selected < 0)
    return;

  vrmlscene_->setViewpoint (selected);
  vrmlscene_->updateViewpointValues (selected);
  vrmlscene_->getCamera ()->reset ();
  scene_->redraw ();

} // setViewpoint



void ViewpointDialog::addVP ()
{
  vrmlscene_->addViewpoint ();
  scene_->redraw ();
  // VRMLScene updates the dialog

} // addVP


void ViewpointDialog::deleteVP ()
{
  int selected = browser_->selected ();
  vrmlscene_->deleteViewpoint (selected);
  scene_->redraw ();
  // VRMLScene updates the dialog

} // deleteVP


// build window with filebrowser

void ViewpointDialog::buildWindow ()
{
  if (win_)
    return;

  kit_->begin_style ("ViewpointsDialog", "Dialog");
  Style* style = kit_->style ();
  const char* title = STranslate::str (STranslate::EditdialogVIEWPOINTSETTINGS);
  style->attribute ("name", title);
  style->attribute ("iconName", title);

  FieldEditorAction31* fe_action = 
    new FieldEditorCallback31(ViewpointDialog) (this, &ViewpointDialog::accept, 0, 0);
  Action* reset = new ActionCallback(ViewpointDialog) (this, &ViewpointDialog::resetValues);
  //Action* apply = new ActionCallback(ViewpointDialog) (this, &ViewpointDialog::applyViewpoint);
  Action* accept = new ActionCallback(ViewpointDialog) (this, &ViewpointDialog::acceptItem);
  Action* select = new ActionCallback(ViewpointDialog) (this, &ViewpointDialog::selectItem);
  Action* set    = new ActionCallback(ViewpointDialog) (this, &ViewpointDialog::setViewpoint);
  Action* addvp  = new ActionCallback(ViewpointDialog) (this, &ViewpointDialog::addVP);
  Action* deletevp = new ActionCallback(ViewpointDialog) (this, &ViewpointDialog::deleteVP);
  Action* close  = new ActionCallback(ViewpointDialog) (this, &ViewpointDialog::unmapWindow);

  // values
  for (int i = 0; i < 3; i++)
  {
    position_[i] = new FieldEditor31 (kit_, style, "0", fe_action);
    orientation_[i] = new FieldEditor31 (kit_, style, "0", fe_action);
    position_[i]->showModifications (1);
    orientation_[i]->showModifications (1);

  }
  angle_ = new FieldEditor31 (kit_, style, "0", fe_action);
  viewangle_ = new FieldEditor31 (kit_, style, "0", fe_action);
  name_ = new FieldEditor31 (kit_, style, "0", fe_action);
  angle_->showModifications (1);
  viewangle_->showModifications (1);
  name_->showModifications (1);

  float fieldsize = kit_->font ()->width ("-10.00", 6) + 5;

  TableGlyph* table = new TableGlyph (5, 5, 10, 5, 5);
/*
  table->appendRow (
    layout_->hbox (
      layout_->hglue (),
      layout_->hcenter (kit_->label ("X")), // hcenter does not work!
      layout_->hcenter (kit_->label ("Y")),
      layout_->hcenter (kit_->label ("Z")),
      layout_->hcenter (kit_->label ("Angle"))
    )
  );
  table->appendRow (layout_->hbox (layout_->vspace (5)));
*/
  table->appendRow (
    layout_->hbox (
      kit_->label ("Position:"),
      layout_->hfixed (position_[0], fieldsize),
      layout_->hfixed (position_[1], fieldsize),
      layout_->hfixed (position_[2], fieldsize)
    )
  );
  table->appendRow (layout_->hbox (layout_->vspace (5)));
  table->appendRow (
    layout_->hbox (
      kit_->label ("Orientation:"),
      layout_->hfixed (orientation_[0], fieldsize),
      layout_->hfixed (orientation_[1], fieldsize),
      layout_->hfixed (orientation_[2], fieldsize),
      layout_->hfixed (angle_, fieldsize)
    )
  );

  Glyph* name = layout_->hbox (
    kit_->label ("Name:"),
    layout_->hspace (5),
    layout_->hnatural (name_, 100),
    layout_->hspace (5),
    kit_->label ("Zoom:"),
    layout_->hspace (5),
    layout_->hfixed (viewangle_, fieldsize)
  );

  camertype_ = new FieldBrowser (kit_, style, "");
  Glyph* dummy = layout_->hbox (
    kit_->label ("Type:"),
    layout_->hspace (10),
    camertype_,
    layout_->hglue (10, 0, 5),
    kit_->push_button ("Reset",reset)
    //layout_->hglue (5, 0, 5),
    //kit_->push_button ("Apply",apply)
  );


  Glyph* values = kit_->inset_frame (
    layout_->margin (
      layout_->vbox (
        table,
        layout_->vspace (10),
        name,
        layout_->vspace (10),
        dummy
      ),
      5
    )
  );

  // filebrowser
  browser_ = new WFileBrowser (kit_, accept, close, select);  // accept: double click
  browser_->ref();

  Glyph* browser_box = layout_->hbox(
      kit_->inset_frame(layout_->natural (browser_, 100, 100)),
      kit_->vscroll_bar(browser_->adjustable())
  );

  Glyph* buttons = layout_->hbox (
    layout_->hglue (0.0),
    kit_->push_button ("Set",set),
    layout_->hglue (5.0),
    kit_->push_button ("Add", addvp),
    layout_->hglue (5.0),
    kit_->push_button ("Delete", deletevp),
    layout_->hglue (5.0),
    kit_->push_button ("Close",close),
    layout_->hglue (0.0)
  );

  Glyph* browser_window = 
    new Background (
      layout_->margin (
        layout_->vbox (
          layout_->hbox (
            kit_->label ("Values:"),
            layout_->hglue ()
          ),
          values,
          layout_->vspace (10.0),
          layout_->hbox (
            kit_->label ("Viewpoints:"),
            layout_->hglue ()
          ),
          browser_box,
          layout_->vspace (10.0),
          buttons
        ),
        10),
      kit_->background ()
  );


  win_ = new ApplicationWindow (browser_window);
  win_->style (style);

  kit_->end_style ();

} // buildWindow


void SceneMenus::toggleViewpointDialog ()
{
  if (!viewpointdialog_)
    viewpointdialog_ = new ViewpointDialog (kit_, layout_, scene_, this);

  viewpointdialog_->toggleWindow ();

} // toggleGridDialog


/*** handle all edit dialogs from scenemenus ***/


// destroy all edit dialogs

void SceneMenus::destroyEditDialogs ()
{
//  unmapEditDialogs ();

  if (wwwanchordialog_)
  { delete wwwanchordialog_;
    wwwanchordialog_ = 0;
  }

  if (materialdialog_)
  { delete materialdialog_;
    materialdialog_ = 0;
  }

  if (texturedialog_)
  { delete texturedialog_;
    texturedialog_ = 0;
  }

  if (insertobjectdialog_)
  { delete insertobjectdialog_;
    insertobjectdialog_ = 0;
  }

  if (structureviewerdialog_)
  { delete structureviewerdialog_;
    structureviewerdialog_ = 0;
  }

  if (griddialog_)
  { delete griddialog_;
    griddialog_ = 0;
  }

  if (transformdialog_)
  { delete transformdialog_;
    transformdialog_ = 0;
  }

  if (viewpointdialog_)
  { delete viewpointdialog_;
    viewpointdialog_ = 0;
  }

} // destroyEditDialogs
