/*=========================================================================

  Program: DICOM for VTK

  Copyright (c) 2012-2014 David Gobbi
  All rights reserved.
  See Copyright.txt or http://dgobbi.github.io/bsd3.txt for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#ifndef __vtkDICOMUtilities_h
#define __vtkDICOMUtilities_h

#include <vtkSystemIncludes.h>
#include <vtkObject.h>
#include "vtkDICOMModule.h"
#include "vtkDICOMTag.h"

#include <string>

class vtkStringArray;

//! Utility functions for use with DICOM classes.
class VTK_DICOM_EXPORT vtkDICOMUtilities : public vtkObject
{
public:
  //! VTK dynamic type information macro.
  vtkTypeMacro(vtkDICOMUtilities, vtkObject);

  //! Set a UID prefix to use when generating UIDs.
  /*!
   *  This is optional, if you do not own a UID prefix then the publicly
   *  available 2.25 prefix will be used.
   */
  static void SetUIDPrefix(const char *prefix);
  static const char *GetUIDPrefix();

  //! Generate a UID for the provided tag.
  static std::string GenerateUID(vtkDICOMTag tag);

  //! Generate a series of UIDs, sorted from low to high.
  /*!
   *  Before passing the string array, call SetNumberOfValues() on
   *  the array to specify the number of UIDs that you want to be
   *  stored in it.  The stored UIDs will be sorted, low to high.
   *  Generating a batch of UIDs is more efficient than calling
   *  GenerateUID() repeatedly.
   */
  static void GenerateUIDs(vtkDICOMTag tag, vtkStringArray *uids);

  //! Numerically compare two UIDs, returns -1, 0, or +1.
  static int CompareUIDs(const char *u1, const char *u2);

  //! Generate a DICOM date time string in the given timezone.
  /*!
   *  The time zone is to be given in the DICOM format of
   *  +HHMM or -HHMM where HH is the hour offset and MM is the
   *  minute offset.  If no time zone is given, then local time
   *  is used (according to the computer's time zone preferences).
   */
  static std::string GenerateDateTime(const char *zone);

  //! Generate a DICOM date time string from long integer.
  /*!
   *  Given an integer number of microseconds since the UNIX epoch
   *  (00:00:00 UTC on 1 Jan 1970), return a DICOM date time string
   *  for the given time zone.  The time zone should be given in the
   *  format +HHMM or -HHMM.  If no time zone is given, then the local
   *  zone is used (according to the computer's time zone preferences).
   */
  static std::string GenerateDateTime(long long microsecs, const char *zone);

  //! Convert a DICOM data time string into a long integer.
  /*!
   *  Given a DICOM date time string, generate a long integer that counts
   *  microseconds since the UNIX epoch (00:00:00 UTC on 1 Jan 1970).
   *  If the datetime string contains a timezone offset, then that timezone
   *  is used to convert the time to UTC.  Otherwise, the local timezone
   *  (according to the computer's timezone preferences) is used to
   *  convert the time to UTC.  The return value is always intended to be
   *  interpreted as a UTC time.
   */
  static long long ConvertDateTime(const char *datetime);

  //! Get the current UTC time in microseconds and an offset to localtime.
  /*!
   *  The time is in microseconds since the UNIX epoch (00:00:00 UTC on
   *  1 Jan 1970).  The offset can be added to the returned UTC time in
   *  order to get the local time.  If you do not need the offset, you
   *  can pass a null pointer.
   */
  static long long GetUTC(long long *offset);

  //! Check if the specified file is a DICOM file.
  /*!
   *  This will return true if the file exists, is readable, and
   *  if it has the DICM magic number or if the first few bytes of
   *  the file look like DICOM data elements.
   */
  static bool IsDICOMFile(const char *filename);

  //! Get the UID for this DICOM implementation.
  static const char *GetImplementationClassUID();

  //! Set the UID for this DICOM implementation.
  /*!
   *  The supplied UID will be copied into a static storage area,
   *  and used for all future calls to GetImplementationClassUID.
   *  The ImplementationClassUID appears in the DICOM meta header.
   */
  static void SetImplementationClassUID(const char *uid);

  //! Get the versioned name for this DICOM implementation.
  static const char *GetImplementationVersionName();

  //! Set the versioned name for this DICOM implementation.
  /*!
   *  The supplied name will be copied into a static storage area,
   *  and used for all future calls to GetImplementationVersionName.
   *  The ImplementationVersionName appears in the DICOM meta header.
   */
  static void SetImplementationVersionName(const char *name);

  //! Unpack one little-endian int from a stream of bytes.
  /*!
   *  This is a very common operation that is required in many different
   *  classes, so the code is centralized here.
   */
  static unsigned int UnpackUnsignedInt(const unsigned char *cp) {
    return cp[0] + (cp[1] << 8) + (cp[2] << 16) + (cp[3] << 24); }

  //! Pack one little-endian int into a stream of bytes.
  /*!
   *  This is a very common operation that is required in many different
   *  classes, so the code is centralized here.
   */
  static void PackUnsignedInt(unsigned int i, unsigned char *cp) {
    cp[0] = static_cast<unsigned char>(i);
    cp[1] = static_cast<unsigned char>(i >> 8);
    cp[2] = static_cast<unsigned char>(i >> 16);
    cp[3] = static_cast<unsigned char>(i >> 24); }

protected:
  vtkDICOMUtilities();
  ~vtkDICOMUtilities();

  static char UIDPrefix[64];
  static char ImplementationClassUID[65];
  static char ImplementationVersionName[17];

private:
  vtkDICOMUtilities(const vtkDICOMUtilities&);  // Not implemented.
  void operator=(const vtkDICOMUtilities&);  // Not implemented.
};

#endif /* __vtkDICOMUtilities_h */
