/*-----------------------------------------------------------------------------
 *  FILE: LibNotify.cc
 *
 *      Copyright(c) 2006 Gareth Foster.
 *
 *      Applet class which controls the program.
 *
 *-----------------------------------------------------------------------------
 */

#include "LibNotify.hh"
#include "Applet.hh"

//------------------------------------------------------------------------
/**
<fn_NotifyCallbackReject>
@brief Callback handler for the reject new wallpaper event.

@param _pNotification The Notification daemon calling us back.
@param _sz The string associated with the callback.
@param _pData A pointer to the LibNotify class to call back on.
@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void fn_NotifyCallbackReject
(
	NotifyNotification *	_pNotification,
	gchar *					_sz,
	gpointer				_pData
)
{
	reinterpret_cast<LibNotify *>(_pData)->OnNotificationChangeReject();
}// end fn_NotifyCallback

//------------------------------------------------------------------------
/**
<fn_NotifyCallbackDelete>
@brief Callback handler for the delete new wallpaper event.

@param _pNotification The Notification daemon calling us back.
@param _sz The string associated with the callback.
@param _pData A pointer to the LibNotify class to call back on.
@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void fn_NotifyCallbackDelete
(
	NotifyNotification *	_pNotification,
	gchar *					_sz,
	gpointer				_pData
)
{
	reinterpret_cast<LibNotify *>(_pData)->OnNotificationChangeDelete();
}// end fn_NotifyCallback

//------------------------------------------------------------------------
/**
<LibNotify::LibNotify>
@brief Construct a new LibNotify object.

@param _pApplet A pointer to the Applet class.
@param _pWallpapers A pointer to the Wallpapers class which owns this.
@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
LibNotify::LibNotify
(
	Applet *				_pApplet,
	Wallpapers *			_pWallpapers
	
) :
	m_pApplet(_pApplet),
	m_pWallpapers(_pWallpapers)
{
	notify_init("Wallpaper Tray");
}// end LibNotify::LibNotify

//------------------------------------------------------------------------
/**
<LibNotify::LibNotify>
@brief Destroy a LibNotify object.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
LibNotify::~LibNotify
(
)
{
	notify_uninit();
}// end LibNotify::LibNotify

//------------------------------------------------------------------------
/**
<LibNotify::WallpaperChanged>
@brief Destroy a LibNotify object.

@param _szNewWallpaper To be called when the wallpaper changes.
@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void LibNotify::WallpaperChanged
(
	Glib::ustring			_szNewWallpaper
)
{
	Glib::ustring sz_MsgText =
			_("Do you want to accept this new wallpaper?\n") +
			_szNewWallpaper;

	NotifyNotification * p_Notification = notify_notification_new	(
			_("Wallpaper Tray is about to change your wallpaper"),
			sz_MsgText.c_str(),
			NULL,
			NULL//GTK_WIDGET(m_pApplet->gobj())
		);

	m_SigcConnection = Glib::signal_timeout().connect	(
			sigc::mem_fun(this, &LibNotify::OnNotificationChangeAccept), 1000 * 7
		);

	if(m_pApplet->GetThumbnail())
	{
		notify_notification_set_icon_from_pixbuf(p_Notification, m_pApplet->GetThumbnail()->gobj());
	}// end if

	// allow the user to block the change
	notify_notification_add_action	(
			p_Notification,
			_("Don't change"),
			_("Don't change"),
			fn_NotifyCallbackReject,
			this,
			NULL
		);

	// allow the user to delete the wp
	notify_notification_add_action	(
			p_Notification,
			_("Delete it"),
			_("delete it"),
			fn_NotifyCallbackDelete,
			this,
			NULL
		);

	notify_notification_show(p_Notification, NULL);
}// end LibNotify::WallpaperChanged

//------------------------------------------------------------------------
/**
<LibNotify::OnNotificationChangeAccept>
@brief Called when a user does nothing with the notify popup.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
bool LibNotify::OnNotificationChangeAccept
(
)
{
	m_pWallpapers->AcceptChange();

	return false;	// dump this callback
}// end LibNotify::OnNotificationChangeAccept

//------------------------------------------------------------------------
/**
<LibNotify::OnNotificationChangeReject>
@brief Called when a user presses reject on the notify popup.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void LibNotify::OnNotificationChangeReject
(
)
{
	m_SigcConnection.disconnect();

	m_pWallpapers->RejectChange();
}// end LibNotify::OnNotificationChangeReject

//------------------------------------------------------------------------
/**
<LibNotify::OnNotificationChangeDelete>
@brief Called when a user presses delete on the notify popup.

@date 12-11-2006 GAF Written
**/
//------------------------------------------------------------------------
void LibNotify::OnNotificationChangeDelete
(
)
{
	m_SigcConnection.disconnect();

	m_pWallpapers->DeleteChange();
}// end OnNotificationChangeDelete
