// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
// 
// $Id: main.cpp
// 
//---------------------------------------------------------------------------

#include <clutils/StringUtilities.h>
#include <clutils/tokens.h>
#include <clutils/ConfigurationParserHandle.h>
#include <clutils/ConfigurationScope.h>
#include "SimulationConfiguration.h"
#include <fstream>
#include <iostream>
#include <cstdlib>
#include <cstdio>

class SimulationConfiguration::Implementation {
public:
  const vector<string> &getArguments(){ return myArguments; }

  bool communicationManagerIs( const string &testValue ) const;
  const string getCommunicationManagerType() const;

  bool eventListTypeIs( const string &testValue ) const;
  bool eventListOrganizationIs( const string &testValue ) const;

  const string getEventListType() const;
  const string getEventListOrganization() const;

  bool gvtManagerTypeIs( const string &testValue ) const;
  const string getGVTManagerType() const;
  bool getGVTPeriod( unsigned int &period );

  bool simulationTypeIs( const string &testValue ) const;
  const string getSimulationType() const;

  bool physicalLayerIs( const string &testValue ) const;
  const string getPhysicalLayerType() const;

  bool outputManagerIs( const string &testValue ) const;
  const string getOutputManagerType() const;

  bool spinKeySet( const string &testValue ) const;

  bool schedulerTypeIs( const string &testValue ) const;
  const string getSchedulerType() const;

  bool getNumberOfSimulationManagers( unsigned int &number ) const;

  bool stateManagerTypeIs( const string &testValue ) const;
  const string getStateManagerType() const;

  bool getStatePeriod( unsigned int &period ) const;

  vector<string> getNodes() const;
  const string getBinaryName() const;

  Implementation() :
    myOuterScope( 0 ){}

  Implementation( const ConfigurationScope *initScope,
		  const vector<string> &arguments ) :
    myOuterScope( initScope ),
    myArguments( arguments ){}

  ~Implementation();

private:
  /**
     Returns the outer ConfigurationScope - i.e. global level options, and access
     to all "first level" scopes.
  */
  const ConfigurationScope &getOuterScope() const;

  /**
     Finds the global level configuration choice "choiceName".
     @param choiceName The choice name to look for.
     @return The choice, if it exists in the outerscope, or NULL if it doesn't.
  */
  const ConfigurationChoice *findChoice( const string &choiceName ) const;

  /**
     Finds the global level configuration scope "scopeName".
     @param scopeName The scope name to look for.
     @return The scope, if it exists in the outerscope, or NULL if it doesn't.
  */
  const ConfigurationScope *findScope( const string &scopeName ) const;

  /**
     Returns the scope for TimeWarp::CommunicationManager or null if not
     found.
  */
  const ConfigurationScope *getTimeWarpScope() const;

  /**
     Returns the scope for TimeWarp::CommunicationManager or null if not
     found.
  */
  const ConfigurationScope *getCommunicationManagerScope() const;

  const ConfigurationScope *getGVTManagerScope() const;

  /**
     Returns the ConfigurationScope for TimeWarp::EventList
  */
  const ConfigurationScope *getEventListScope() const;

  /**
     Returns the ConfigurationChoice for TimeWarp::EventList::Organization
  */
  const ConfigurationChoice *getEventListOrganizationChoice() const; 
  /**
     Returns ConfigurationChoice for TimeWarp::EventList::Type
  */
  const ConfigurationChoice *getEventListTypeChoice() const;

  const ConfigurationScope *getOutputManagerScope() const;  
  const ConfigurationScope *getSchedulerScope() const;

  /**
     Returns ConfigurationScope for TimeWarp::StateManager
  */
  const ConfigurationScope *getStateManagerScope() const;  
  /**
     Returns ConfigurationChoice for TimeWarp::StateManager::Type
  */
  const ConfigurationChoice *getStateManagerTypeChoice() const; 
  /**
     Returns ConfigurationChoice for TimeWarp::StateManager::Period
  */
  const ConfigurationChoice *getStateManagerPeriodChoice() const; 

  const ConfigurationValue *getNodesValue() const;

  static const string &getCommManagerScopeName();
  static const string &getEventListScopeName();
  static const string &getEventListOrganizationName();
  static const string &getPeriodName();
  static const string &getStateManagerScopeName();
  static const string &getTypeName();
  static const string &getTimeWarpScopeName();
  
  const ConfigurationScope *myOuterScope;
  const vector<string> myArguments;
};

SimulationConfiguration::SimulationConfiguration( const ConfigurationScope *outerScope,
						  const vector<string> &argumentVector ) :
  _impl( new SimulationConfiguration::Implementation( outerScope,
						      argumentVector )){}

SimulationConfiguration::~SimulationConfiguration(){
}	

SimulationConfiguration *
SimulationConfiguration::parseConfiguration( const string &configFileName,
					     const vector<string> &argumentVector ){
  SimulationConfiguration *configuration = 0;
  const ConfigurationScope *outerScope = ConfigurationParserHandle::parseFile( configFileName );
  if( outerScope != 0 ){
    configuration = new SimulationConfiguration( outerScope, argumentVector );
  }

  return configuration;
}

const vector<string> &
SimulationConfiguration::getArguments(){
  return _impl->getArguments();
}

bool 
SimulationConfiguration::communicationManagerIs( const string &testValue ) const {
  return _impl->communicationManagerIs( testValue );
}

const string 
SimulationConfiguration::getCommunicationManagerType() const {
  return _impl->getCommunicationManagerType();
}

bool
SimulationConfiguration::eventListTypeIs( const string &testValue ) const {
  return _impl->eventListTypeIs( testValue );
}

bool
SimulationConfiguration::eventListOrganizationIs( const string &testValue ) const {
  return _impl->eventListOrganizationIs( testValue );
}

const string 
SimulationConfiguration::getEventListType() const {
  return _impl->getEventListType();
}

const string
SimulationConfiguration::getEventListOrganization() const {
  return _impl->getEventListOrganization();
}

bool
SimulationConfiguration::gvtManagerTypeIs( const string &testValue ) const {
  return _impl->gvtManagerTypeIs( testValue );
}

const string
SimulationConfiguration::getGVTManagerType() const {
  return _impl->getGVTManagerType();
}
  

bool
SimulationConfiguration::simulationTypeIs( const string &testValue ) const {
  return _impl->simulationTypeIs( testValue );
}

const string
SimulationConfiguration::getSimulationType() const {
  return _impl->getSimulationType();
}

bool
SimulationConfiguration::getGVTPeriod( unsigned int &period ){
  return _impl->getGVTPeriod( period );
}

bool
SimulationConfiguration::physicalLayerIs( const string &testValue ) const {
  return _impl->physicalLayerIs( testValue );
}

const string
SimulationConfiguration::getPhysicalLayerType() const {
  return _impl->getPhysicalLayerType();
}

bool
SimulationConfiguration::outputManagerIs( const string &testValue ) const {
  return _impl->outputManagerIs( testValue );
}

const string
SimulationConfiguration::getOutputManagerType() const {
  return _impl->getOutputManagerType();
}

bool
SimulationConfiguration::spinKeySet( const string &testValue ) const {
  return _impl->spinKeySet( testValue );
}

bool
SimulationConfiguration::schedulerTypeIs( const string &testValue ) const {
  return _impl->schedulerTypeIs( testValue );
}

const string
SimulationConfiguration::getSchedulerType() const {
  return _impl->getSchedulerType();
}

bool
SimulationConfiguration::getNumberOfSimulationManagers( unsigned int &number ){
  return _impl->getNumberOfSimulationManagers( number );
}

bool
SimulationConfiguration::stateManagerTypeIs( const string &testValue ) const {
  return _impl->stateManagerTypeIs( testValue );
}

const string
SimulationConfiguration::getStateManagerType() const {
  return _impl->getStateManagerType();
}

bool
SimulationConfiguration::getStatePeriod( unsigned int &period ) const {
  return _impl->getStatePeriod( period );
}

vector<string>
SimulationConfiguration::getNodes() const {
  return _impl->getNodes();
}

const string
SimulationConfiguration::getBinaryName() const {
  return _impl->getBinaryName();
}


const ConfigurationChoice *
SimulationConfiguration::Implementation::findChoice( const string &choiceName ) const {
  return myOuterScope->findChoice( choiceName );
}

const ConfigurationScope *
SimulationConfiguration::Implementation::findScope( const string &scopeName ) const {
  return myOuterScope->findScope( scopeName );
}

const ConfigurationScope &
SimulationConfiguration::Implementation::getOuterScope() const {
  return *myOuterScope;
}

const ConfigurationScope *
SimulationConfiguration::Implementation::getTimeWarpScope() const {
  return findScope( getTimeWarpScopeName() );
}

const ConfigurationScope *
SimulationConfiguration::Implementation::getCommunicationManagerScope() const {
  const ConfigurationScope *twScope = getTimeWarpScope();
  return twScope->findScope( getCommManagerScopeName() );
}

const ConfigurationScope *
SimulationConfiguration::Implementation::getEventListScope() const {
  const ConfigurationScope *retval = 0;

  if( stringCaseCompare( getSimulationType(), "sequential" ) ){
    retval = findScope( getEventListScopeName() );
  }
  else if( stringCaseCompare( getSimulationType(), "timewarp" ) ){
    const ConfigurationScope *twScope = getTimeWarpScope();
    if( twScope != 0 ){
      retval = twScope->findScope( getEventListScopeName() );
    }
  }
  
  return retval;
}

const ConfigurationChoice *
SimulationConfiguration::Implementation::getEventListOrganizationChoice() const {
  const ConfigurationChoice *retval = 0;
  if( getEventListScope() != 0 ){
    retval = getEventListScope()->findChoice( getEventListOrganizationName() );
  }
  return retval;
}

const ConfigurationChoice *
SimulationConfiguration::Implementation::getEventListTypeChoice() const {
  const ConfigurationChoice *retval = 0;
  if( getEventListScope() != 0 ){
    retval = getEventListScope()->findChoice( getTypeName() );
  }
  return retval;
}

const ConfigurationScope *
SimulationConfiguration::Implementation::getStateManagerScope() const {
  const ConfigurationScope *retval = 0;
  if( getTimeWarpScope() != 0 ){
    retval = getTimeWarpScope()->findScope( getStateManagerScopeName() );
  }
  return retval;
}

const ConfigurationChoice *
SimulationConfiguration::Implementation::getStateManagerTypeChoice() const {
  const ConfigurationChoice *retval = 0;
  if( getStateManagerScope() != 0 ){
    retval = getStateManagerScope()->findChoice( getTypeName() );
  }
  return retval;
}

const ConfigurationChoice *
SimulationConfiguration::Implementation::getStateManagerPeriodChoice() const {
  const ConfigurationChoice *retval = 0;
  if( getStateManagerScope() != 0 ){
    retval = getStateManagerScope()->findChoice( getPeriodName() );
  }
  return retval;
}

const ConfigurationScope *
SimulationConfiguration::Implementation::getGVTManagerScope() const {
  const ConfigurationScope *retval = 0;
  if( getTimeWarpScope() != 0 ){
    retval = getTimeWarpScope()->findScope( "gvtmanager" );
  }
  return retval;
}

const ConfigurationScope *
SimulationConfiguration::Implementation::getOutputManagerScope() const {
  const ConfigurationScope *retval = 0;
  if( getTimeWarpScope() != 0 ){
    retval = getTimeWarpScope()->findScope( "outputmanager" );
  }
  return retval;
}

const ConfigurationScope *
SimulationConfiguration::Implementation::getSchedulerScope() const {
  const ConfigurationScope *retval = 0;
  if( getTimeWarpScope() != 0 ){
    retval = getTimeWarpScope()->findScope( "scheduler" );
  }
  return retval;
}

const string &
SimulationConfiguration::Implementation::getCommManagerScopeName(){
  static const string commMgrScope = "CommunicationManager";
  return commMgrScope;
}

const string &
SimulationConfiguration::Implementation::getEventListScopeName(){
  static const string eventListScopeName = "EventList";
  return eventListScopeName;
}

const string &
SimulationConfiguration::Implementation::getEventListOrganizationName(){
  static const string eventListOrgName = "Organization";
  return eventListOrgName;
}

const string &
SimulationConfiguration::Implementation::getPeriodName(){
  static const string periodName = "Period";
  return periodName;
}

const string &
SimulationConfiguration::Implementation::getStateManagerScopeName(){
  static const string stateManagerScopeName = "StateManager";
  return stateManagerScopeName;
}

const string &
SimulationConfiguration::Implementation::getTypeName(){
  static const string typeName = "type";
  return typeName;
}

const string &
SimulationConfiguration::Implementation::getTimeWarpScopeName(){
  static const string timeWarpScope = "TimeWarp";
  return timeWarpScope;
}

const string
SimulationConfiguration::Implementation::getCommunicationManagerType() const {
  string retval = "(none)";
  if( getCommunicationManagerScope() != 0 ){
    const ConfigurationChoice *type = getCommunicationManagerScope()->findChoice( "type" );
    if( type != 0 ){
      retval = type->getStringValue();
    }
  }
  return retval;
}

bool
SimulationConfiguration::Implementation::communicationManagerIs( const string &testValue ) const {
  return stringCaseCompare( testValue, getCommunicationManagerType() );
}

const string
SimulationConfiguration::Implementation::getEventListType() const {
  string retval = "(none)";
  if( getEventListScope() != 0 ){
    const ConfigurationChoice *type = getEventListScope()->findChoice( "type" );
    if( type != 0 ){
      retval = type->getStringValue();
    }
  }
  return retval;
}

bool
SimulationConfiguration::Implementation::eventListTypeIs( const string &testValue ) const {
  return stringCaseCompare( testValue, getEventListType() );
}

const string
SimulationConfiguration::Implementation::getEventListOrganization() const {
  string retval = "(none)";
  if( getEventListScope() != 0 ){
    const ConfigurationChoice *Organization = getEventListScope()->findChoice( "Organization" );
    if( Organization != 0 ){
      retval = Organization->getStringValue();
    }
  }
  return retval;
}

bool
SimulationConfiguration::Implementation::eventListOrganizationIs( const string &testValue ) const {
  return stringCaseCompare( testValue, getEventListOrganization() );
}

const string
SimulationConfiguration::Implementation::getGVTManagerType() const {
  string retval = "(none)";
  if( getGVTManagerScope() != 0 ){
    const ConfigurationChoice *type = getGVTManagerScope()->findChoice( "Type" );
    if( type != 0 ){
      retval = type->getStringValue();
    }
  }
  return retval;
}

bool
SimulationConfiguration::Implementation::gvtManagerTypeIs( const string &testValue ) const {
  return stringCaseCompare( testValue, getGVTManagerType() );
}

const string
SimulationConfiguration::Implementation::getSimulationType() const {
  string retval = "(none)";
  if( findChoice( "Simulation" ) != 0 ){
    const ConfigurationChoice *type = findChoice( "Simulation" );
    if( type != 0 ){
      retval = type->getStringValue();
    }
  }
  return retval;
}

bool
SimulationConfiguration::Implementation::simulationTypeIs( const string &testValue ) const {
  return stringCaseCompare( testValue, getSimulationType() );
}

bool
SimulationConfiguration::Implementation::getGVTPeriod( unsigned int &period ){
  bool retval = false;
  if( getGVTManagerScope() != 0 ){
    if( getGVTManagerScope()->findChoice( "Period" ) != 0 ){
      period = getGVTManagerScope()->findChoice( "Period")->getIntValue();
      retval = true;
    }
  }
  return retval;
}

bool
SimulationConfiguration::Implementation::physicalLayerIs( const string &testValue ) const {
  return stringCaseCompare( testValue, getPhysicalLayerType() );
}

const string
SimulationConfiguration::Implementation::getPhysicalLayerType() const {
  string retval = "(none)";
  if( getCommunicationManagerScope() != 0 ){
    const ConfigurationChoice *type = getCommunicationManagerScope()->findChoice("PhysicalLayer");
    if( type != 0 ){
      retval = type->getStringValue();
    }
  }
  return retval;
}

const string
SimulationConfiguration::Implementation::getOutputManagerType() const {
  string retval = "(none)";
  if( getOutputManagerScope() != 0 ){
    const ConfigurationChoice *type = getOutputManagerScope()->findChoice( "Type" );
    if( type != 0 ){
      retval = type->getStringValue();
    }
  }
  return retval;
}

bool
SimulationConfiguration::Implementation::outputManagerIs( const string &testValue ) const {
  return stringCaseCompare( testValue, getOutputManagerType() );
}

const string
SimulationConfiguration::Implementation::getSchedulerType() const {
  string retval = "(none)";
  if( getSchedulerScope() != 0 ){
    const ConfigurationChoice *type = getSchedulerScope()->findChoice( "Type" );
    if( type != 0 ){
      retval = type->getStringValue();
    }
  }
  return retval;
}

bool
SimulationConfiguration::Implementation::schedulerTypeIs( const string &testValue ) const {
  return stringCaseCompare( testValue, getSchedulerType() );
}

bool
SimulationConfiguration::Implementation::spinKeySet( const string &testValue ) const {
  bool retval = false;
  const ConfigurationScope *debugScope = findScope( "ParallelDebug" );
  if( debugScope != 0 ){
    const ConfigurationChoice *choice = debugScope->findChoice( testValue );
    if( choice != 0 ){
      retval = stringCaseCompare( choice->getStringValue(), "true" );
    }
  }
  return retval;
}

const string
SimulationConfiguration::Implementation::getStateManagerType() const {
  string retval = "(none)";
  if( getStateManagerScope() != 0 ){
    const ConfigurationChoice *type = getStateManagerScope()->findChoice( "Type" );
    if( type != 0 ){
      retval = type->getStringValue();
    }
  }
  return retval;
}

bool
SimulationConfiguration::Implementation::stateManagerTypeIs( const string &testValue ) const {
  return stringCaseCompare( testValue, getStateManagerType() );
}

bool
SimulationConfiguration::Implementation::getStatePeriod( unsigned int &period ) const {
  bool retval = false;
  const ConfigurationScope *scope = getStateManagerScope();
  if( scope != 0 ){
    if( scope->getIntValue( "Period" ) != -1 ){
      period = scope->getIntValue( "Period" );
      retval = true;
    }
  }
  return retval;
}

const ConfigurationValue *
SimulationConfiguration::Implementation::getNodesValue() const {
  const ConfigurationValue *retval = 0;

 const ConfigurationScope *scope = getCommunicationManagerScope();
  if( scope != 0 ){
    const ConfigurationChoice *nodesList = scope->findChoice( "nodes" );
    if( nodesList != 0 ){
      retval = nodesList->getConfigurationValue();
    }
  }

  return retval;
}

vector<string>
SimulationConfiguration::Implementation::getNodes( ) const {
  vector<string> retval;

  const ConfigurationValue *nodesValue = getNodesValue();
  if( nodesValue &&
      dynamic_cast<const VectorConfigurationValue *>(nodesValue) ){
    const vector<const ConfigurationValue *> *configVector = nodesValue->getVectorValue();
    if( configVector != 0 ){
      for( vector<const ConfigurationValue *>::const_iterator i = configVector->begin();
	   i < configVector->end();
	   i++ ){
	if( dynamic_cast<const StringConfigurationValue *>( *i )){
	  retval.push_back( (*i)->getStringValue() );
	}
      }
    }
  }
  else if( dynamic_cast<const StringConfigurationValue *>(nodesValue) ){
    retval.push_back( nodesValue->getStringValue() );
  }
  return retval;
}

const string
SimulationConfiguration::Implementation::getBinaryName() const {
  string retval = "";
  if( !myArguments.empty() ){
    retval = myArguments[0];
  }

  return retval;
}

bool
SimulationConfiguration::Implementation::getNumberOfSimulationManagers( unsigned int &numManagers ) const {
  bool retval = false;
  const ConfigurationScope *scope = getCommunicationManagerScope();
  if( scope != 0 ){
    const ConfigurationChoice *nodesList = scope->findChoice( "nodes" );
    if( nodesList != 0 ){
      if( dynamic_cast<const VectorConfigurationValue *>(nodesList->getConfigurationValue()) ){
	numManagers = 
	  dynamic_cast<const VectorConfigurationValue *>(nodesList->getConfigurationValue())->getVectorValue()->size() + 1; // Have to add ourself in...
	retval = true;
      }
      else if( dynamic_cast<const StringConfigurationValue *>(nodesList->getConfigurationValue()) ){
	numManagers = 1 + 1; // Have to add ourself in...
	retval = true;	
      }
    }
  }
  return retval;
}
