#ifndef TIMEWARP_SIMULATION_MANAGER_H
#define TIMEWARP_SIMULATION_MANAGER_H

// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

#include "warped.h"
#include "SimulationStream.h"
#include "CommunicatingEntity.h"
#include "TimeWarpConfigurationManager.h"
#include "SimulationManagerImplementationBase.h"
#include "EventId.h"
#include <clutils/StringHashMap.h>

class Application;
class CommunicationManager;
class GVTManager;
class OutputManager;
class SchedulingData;
class SchedulingManager;
class SimulationObjectProxy;
class StateManager;
class TerminationManager;
class TimeWarpSimulationStream;
class TimeWarpEventSet;

/** The TimeWarpSimulationManager class.

    The TimeWarpSimulationManager class implements a simulation
    manager that performs a TimeWarp synchronized parallel
    discrete-event simulation. Since simulation managers need to
    communicate, the TimeWarpSimulationManager is also derived from
    the CommunicatingEntity abstract base class.

*/
class TimeWarpSimulationManager : public SimulationManagerImplementationBase,
                                  public CommunicatingEntity {
public:

  /**@type friend class declarations */
  //@{

  /** Builder class */
  friend class TimeWarpEventSetFactory;

  //@} // End of friend class declarations  

  /**@name Public Class Methods of TimeWarpSimulationManager. */
  //@{

  /** Constructor.

  @param numProcessors Number of processors requested
  @param initApplication The application that we're going to start up.

  */
  TimeWarpSimulationManager( unsigned int numProcessors,
			     Application *initApplication );
   
  /// Destructor.
  ~TimeWarpSimulationManager();

  /// initialize the simulation objects before starting the simulation.
  void initialize();

  /** Run the simulation.
       
  @param simulateUntil Time upto which to simulate.
  */
  void simulate ( const VTime &simulateUntil );

  /// Clean up after the simulation has finished.
  void finalize();
   
  /** Registers a set of simulation objects with this simulation manager.
      Some tasks this function is responsible for:
      a. creating and storing the map of simulation object pointers
      b. assigning unique ids to each simulation object
      c. setting the simulation manager handle in each object
      d. configuring the state by calling allocateState on each object */
  void registerSimulationObjects ();
   
  /** Construct array of local object names to send out to other Sim-Mgrs.

  @return The array of local object names.
  */
  virtual vector<string> *getSimulationObjectNames();

  /** Add these simulation object proxies to the total array of sim objects.

  The proxies are used for simulation objects that are handled
  by another simulation manager.

  @param arrayOfObjectProxies The set of sim-object proxies.
  @param sourceSimulationManagerID The owning sim-manager.
  @param destSimulationManagerID The sim-manager who has the proxy.
  */
  virtual void registerSimulationObjectProxies(const vector<string> *arrayOfObjectProxies,
					       unsigned int sourceSimulationManagerID,
					       unsigned int destSimulationManagerID);

  /** Remove the next event in the event set and return it.

  @param object The simulation object whose event set we get event from.
  @return A reference to the removed event.
  */
  const Event *getEvent(SimulationObject *object);

  /** Return a reference to the next event in the event set.

  @param object The simulation object whose event set we get event from.
  @return A reference to the next event in the event set.
  */
  const Event *peekEvent(SimulationObject *object);

  /** Receive an event.

  @param event A pointer to the received event.
  */
  void handleEvent( const Event *event );
  
  /**
     The output managers call this method to cancel events when they are
     rolled back.
  */
  virtual void cancelEvents( const vector<const Event *> &eventsToCancel );
   
  /** Return a handle to the communication manager.

  @return A handle to the communication manager.
  */
  virtual CommunicationManager *getCommunicationManager() const {
    return myCommunicationManager;
  }

  /** Return a handle to the scheduling manager.

  @return A handle to the scheduling manager.
  */
  virtual SchedulingManager *getSchedulingManager(){
    return mySchedulingManager;
  }

  /** Return a handle to the gvt manager.

  @return A handle to the gvt manager.
  */
  virtual GVTManager *getGVTManager(){
    return myGVTManager;
  }

  /** Return a handle to the state manager.

  @return A handle to the state manager.
  */
  virtual StateManager *getStateManager(){
    return myStateManager;
  }

  /** Return a handle to the event set factory.

  @return A handle to the event set factory.
  */
  virtual TimeWarpEventSet *getEventSetManager(){
    return myEventSet;
  }

  /** Return a handle to the output manager.

  @return A handle to the output manager.
  */
  virtual OutputManager *getOutputManager(){
    return myOutputManager;
  }

  /** Return a handle to the Scheduling Data

  @return A handle to SchedulingData in Decentralized EventSet.
  */
  SchedulingData* getSchedulingData(){
    return mySchedulingData;
  }

  /** Returns a simulation object pointer.

  @param object The string representation of the object.
  @return A pointer to the object corresponding to the string rep.
  */
  virtual SimulationObject *getObjectHandle( const string &object ) const {
    return globalArrayOfSimObjPtrs.find( object );
  }

  virtual OBJECT_ID &getObjectId( const string &objectName ){
    return *getObjectHandle( objectName )->getObjectID();
  }

  /*
    Returns true if this simulation manager contains this object, false if
    it's non-local.
  */
  virtual bool contains( const string &object ) const;

  /** Returns a simulation object pointer.

  @param objectID The object Id of the simulation object.
  @return A pointer to the object with the given object Id.
  */
  virtual SimulationObject *getObjectHandle( const OBJECT_ID &objectID ) const {
    return globalArrayOfSimObjIDs[objectID.getSimulationManagerID()][objectID.getSimulationObjectID()];
  }

  /// Register a particular message type with the communication manager.
  virtual void registerWithCommunicationManager();

  /** Method the communication manager calls to deliver a kernel message.

  @param msg The message to receive.
  */
  virtual void receiveKernelMessage(KernelMessage *msg);

  /** Return the simulation manager's id.

  @return The Id of the simulation manager.
  */
  virtual unsigned int getSimulationManagerID() const {
    return mySimulationManagerID;
  }

  /** Get the current simulation time.
  @return The current simulation time.
  */
  virtual const VTime &getSimulationTime() const;
   
  virtual const VTime &getCommittedTime();

  virtual const VTime &getNextEventTime();

  /** Check if this simulation manager has work to do or not.

  @return Idle status (true/false).
  */
  virtual bool checkIdleStatus();

  /// check if message suppression is enabled ...
  virtual bool checkSuppressMessageFlag();

  /// set the message suppression flag
  virtual void setSuppressMessageFlag(MsgSuppression flag);

  /// get the message suppression flag
  virtual MsgSuppression getSuppressMessageFlag();
   
  /// check if message aggregation is enabled ...
  virtual bool checkMessageAggregationFlag();

  /// set the message aggregation flag
  virtual void setMessageAggregationFlag(bool flag);

  /// get a handle to a simulation input stream
  virtual SimulationStream *getIFStream(const string &filename,
					SimulationObject *object);
   
  /// get a handle to a simulation output stream
  virtual SimulationStream *getOFStream(const string &filename,
					SimulationObject *object,
					ios::openmode mode);
   
  /// get a handle to a simulation input-output stream
  virtual SimulationStream *getIOFStream(const string &filename,
					 SimulationObject *object);

  void configure( SimulationConfiguration &configuration );

  const VTime &getPositiveInfinity() const;
  const VTime &getZero() const;

  void shutdown( const string &errorMessage );
   
  //@} // End of Public Class Methods of TimeWarpSimulationManager.

protected:
  /**@name Protected Class Methods of TimeWarpSimulationManager. */
  //@{

  /** Create a map of simulation objects.

  @return An STL hash-map of the simulation objects.
  */
  StringHashMap<SimulationObject *, true> *createMapOfObjects();

  /** call garbage collect on the state, output and input queue.

  @param garbageCollectTime time upto which garbage collect is performed.
  */
  void garbageCollect(const VTime &garbageCollectTime);

  /** initiate rollback recovery.

  @param object handle to the simulation object
  @param rollbackTime The time it needs to rollback to.
  */
  virtual void rollback( SimulationObject *object, const VTime &rollbackTime );
   
  /// call coastforward if an infrequent state saving strategy is used
  void coastForward(const VTime &coastForwardFromTime,
		    const VTime &rollbackToTime,
		    SimulationObject *object);
   
  /** Display the global simulation object map.

  @param out The output stream.
  */
  void displayGlobalObjectMap(ostream &out);

  void calculateSimulationTime();

  bool simulationComplete(){ return simulationCompleteFlag; }

  void setSimulationTime( const VTime &newSimulationTime ){
    delete simulationTime;
    simulationTime = newSimulationTime.clone();
  }

  MsgSuppression getMessageSuppression(){
    return suppressMessage;
  }

  const VTime &getCoastForwardTime() const;

  //@} // End of Protected Class Methods of TimeWarpSimulationManager.

private:
  void setCoastForwardTime( const VTime &newCoastForwardTime ){
    delete coastForwardTime;
    coastForwardTime = newCoastForwardTime.clone();
  }


  /**@name Private Class Attributes of TimeWarpSimulationManager. */
  //@{
   
  /// This simulation manager's id.
  unsigned int mySimulationManagerID;

  /// Status of the simulation manager.
  bool simulationCompleteFlag;

  /// Flag to determine when to suppress a message
  MsgSuppression suppressMessage;

  /// Time up to which coast forwarding should be done.
  const VTime *coastForwardTime;

  /// Flag to determine if message aggregation is enabled or not
  bool messageAggregation;
   
  /// the simulation time of this simulation manager
  const VTime *simulationTime;

  /// Mapping between simulation object names, object pointers & ids.
  StringHashMap<SimulationObject *, true> globalArrayOfSimObjPtrs; 
   
  /// Mapping between simulation object ids to names.
  vector< vector<SimulationObject*> > globalArrayOfSimObjIDs;
   
  /// handle to the StateManager Factory
  StateManager *myStateManager;
   
  // handle to the GVTManager Factory
  GVTManager *myGVTManager;
   
  /// Handle to the communication manager Factory
  CommunicationManager *myCommunicationManager;

  /// Handle to the scheduler Factory
  SchedulingManager *mySchedulingManager;

  /// Handle to the output manager Factory
  OutputManager *myOutputManager;

  ///This is the handle to the set of events
  TimeWarpEventSet *myEventSet;

  ///This is the handle to head of event sets in decentralized eventset
  SchedulingData *mySchedulingData;

  /// This is a handle to a termination manager
  TerminationManager *myTerminationManager;

  /// map of objects where each object can have several input file queues
  hash_map<const OBJECT_ID*, vector<TimeWarpSimulationStream*>,
	   hashID, EqualID> mapOfInFileQueues;

  /// map of objects where each object can have several output file queues
  hash_map<const OBJECT_ID*, vector<TimeWarpSimulationStream*>,
	   hashID, EqualID> mapOfOutFileQueues;
   
  //@} // End of Protected Class Attributes of TimeWarpSimulationManager.

private:
  /**
     Used to enqueue new local events.
  */
  void acceptNewEvent( const Event *event );
  
  /**
     Used to cancel local events.
  */
  void cancelLocalEvents( const vector<const Event *> &eventsToCancel );
  /**
     Used to cancel remote events.
  */
  void cancelRemoteEvents( const vector<const Event *> &eventsToCancel );

  /**
     Used to route local events.
  */
  void handleLocalEvent( const Event *event );

  /**
     Used to route remote events.
  */
  void handleRemoteEvent( const Event *event );

  /**
     Used to deal with negative events that we've been informed about.
  */
  void handleNegativeEvents( const vector<EventId> &negativeEvents );

  /**
     Returns true if the simulation is complete, false otherwise.
  */
  bool simulationComplete( const VTime &simulateUntil );

  /**
     Talks to the physical layer to get messages.
  */
  void getMessages();

  /**
     Executes the local objects.  Returns true if there was an event to
     execute, false otherwise.
  */
  bool executeObjects();

  Application *myApplication;
};

#endif

