--  2001- 2005 Peter Thiemann
-- |Definition of the monad underlying the CGI library.
module CGIMonad
where

import Maybe

import CGITypes
import HTMLMonad hiding (span, map, div, head)
import RawCGIInternal hiding (contentType)
import qualified SHA1

newtype CGI a = CGI { unCGI :: CGIAction a }
type CGIAction a = CGIState -> IO (a, CGIState)
data CGIState
  = CGIState { inparm   :: [PARAMETER]
	     , outparm  :: [PARAMETER]
	     , stateID  :: String
	     , mcount	:: Int
	     , cgiInfo  :: CGIInfo
	     , pageInfo :: PageInfo
	     , encoder  :: String -> String
	     , cookieMap     :: [(String, (Maybe String, Maybe String))]
	     , cookiesToSend :: [String]
	     }

data PageInfo =
     PageInfo { count      :: Int
     	      , nextaction :: Element -> CGIAction ()
	      , actionTable :: [(String, Element -> CGIAction ())]
	      , bindings   :: Maybe CGIParameters
	      , enctype    :: String
	      , inFrame    :: Int
	      , allFields  :: [(String, Bool)]
	      , faultyfields :: [(String, String)]
	      }

data CGIFieldName =
     CGIFieldName { fnMcount :: Int, fnCount :: Int }

instance Show CGIFieldName where
  showsPrec i cfn = showChar 'f' .
                    shows (fnMcount cfn) .
		    showChar 'x' .
		    shows (fnCount cfn)

args = cgiArgs . cgiInfo
url = unURL . cgiUrl . cgiInfo
contentType = cgiContentType . cgiInfo

fromCGIstate select =
  CGI $ \cgistate ->
  return (select cgistate, cgistate)

getCGIArgs = fromCGIstate args
getUrl = fromCGIstate url
getParm = fromCGIstate outparm
getStateID = fromCGIstate stateID
getInfo = fromCGIstate pageInfo
getEncoder = fromCGIstate encoder
getScriptName = fromCGIstate (cgiScriptName . cgiInfo)
getPathInfo = fromCGIstate (cgiPathInfo . cgiInfo)
getHandle = fromCGIstate (cgiHandle . cgiInfo)
getFields = fromCGIstate (reverse . allFields . pageInfo)
getMcount = fromCGIstate mcount

-- | wrapper to transform IO computation to CGIAction
wrapIO :: IO a -> CGIAction a
wrapIO ioa = \ cgistate -> ioa >>= \ a -> return (a, cgistate)

-- | lift IO monad to CGI monad
lift :: IO a -> CGI a
lift = CGI . wrapIO

inc = 
  CGI $ \cgistate ->
  let info = pageInfo cgistate in
  return (info 
         ,cgistate { pageInfo = info { count = count info + 1}})

setAction :: (Element -> CGI ()) -> CGI ()
setAction actionFun =
  CGI $ \cgistate ->
  return (()
         ,cgistate { pageInfo = (pageInfo cgistate) { nextaction = unCGI . actionFun }})

registerAction :: String -> (Element -> CGI ()) -> CGI ()
registerAction submitter actionFun =
  CGI $ \cgistate ->
  let pi = pageInfo cgistate
      pi' = pi { actionTable = (submitter, unCGI . actionFun) : actionTable pi }
  in return ((), cgistate { pageInfo = pi'})

incFrame :: CGI Int
incFrame =
  CGI $ \cgistate ->
  let info = pageInfo cgistate
      lastFrame = inFrame info
      nextFrame = lastFrame + 1
  in
  return (nextFrame
         ,cgistate { pageInfo = info { inFrame = nextFrame }})

resetFrame :: CGI ()
resetFrame =
  CGI $ \cgistate ->
  let info = pageInfo cgistate
  in
  return (()
         ,cgistate { pageInfo = info { inFrame = 0 }})

setEnctype :: String -> CGI ()
setEnctype contentType =
  CGI $ \cgistate ->
  return (()
         ,cgistate { pageInfo = (pageInfo cgistate) { enctype = contentType } })

setFaulty :: [(String, String)] -> CGI ()
setFaulty ss =
  CGI $ \cgistate ->
  return (()
         ,cgistate { pageInfo = (pageInfo cgistate) { faultyfields = ss } })

data PARAMETER 
	= PAR_RESULT String		-- ^ result of an IO operation shown as a string
	| PAR_VALUES CGIParameters	-- ^ record of a form input
	| PAR_MARK   String		-- ^ recorded stateID before the mark
	| PAR_IGNORED

instance Show PARAMETER where
  showsPrec i par = case par of
    PAR_RESULT str -> showString ":R" . shows str
    PAR_VALUES cps -> showString ":V" . shows cps
    PAR_MARK   str -> showString ":M" . shows str
    PAR_IGNORED    -> showString ":I"

instance Read PARAMETER where
  readsPrec i str = case str of
    ':' : rest -> g rest
    _  -> []
    where g ('R' : str) = [(PAR_RESULT v, rest) | (v, rest) <- reads str]
	  g ('V' : str) = [(PAR_VALUES v, rest) | (v, rest) <- reads str]
	  g ('M' : str) = [(PAR_MARK   v, rest) | (v, rest) <- reads str]
	  g ('I' : str) = [(PAR_IGNORED, str)]
	  g _           = []

instance Monad CGI where
  return a = 
	CGI ( \cgistate -> return (a, cgistate))
  CGI cgi >>= f = 
	CGI ( \cgistate -> 
	    cgi cgistate >>= \ (x, cgistate') ->
	    unCGI (f x) cgistate')

{-- 
fixCGI :: (a -> CGI a) -> CGI a
fixCGI f = CGI (\cgistate -> fixIO (\as' -> unCGI (f (fst as')) cgistate))
--}

nextName :: CGI CGIFieldName
nextName =
  do mc <- getMcount
     pageInfo <- inc
     return CGIFieldName { fnMcount = mc, fnCount = count pageInfo }

addField :: String -> Bool -> CGI ()
addField s f =
  CGI ( \cgistate ->
  	let info = pageInfo cgistate in
	return ((), cgistate { pageInfo = info { allFields = (s,f): allFields info }}))

-- 
initialPageInfo cgistate =
  let bnds = listToMaybe [ parms | PAR_VALUES parms <- inparm cgistate ]
  in  PageInfo
  	{ count = 0
	, nextaction = \ _ st -> return ((), st)
	, actionTable = []
	, bindings = bnds
	, enctype = contentType cgistate
	, inFrame = inFrame (pageInfo cgistate)
	, allFields = []
	, faultyfields = []
	}

dropFirstPARVALUES parms =
  let f rps [] = error "dropFirstPARVALUES: no PAR_VALUES found"
      f rps (p : ps) = 
	case p of
	  PAR_VALUES _ ->
	    (p, reverse rps ++ ps)
	  _ ->
	    f (p : rps) ps
  in  f [] parms

nextCGIState cgistate = cgistate'
  where 
    (newparm, inparm') = dropFirstPARVALUES (inparm cgistate)
    cgistate' = cgistate { inparm = inparm'
			 , stateID = nextstid (stateID cgistate) newparm
			 , mcount = mcount cgistate + 1
			 , pageInfo = (initialPageInfo cgistate')
			     	{ inFrame = inFrame (pageInfo cgistate) }
			 }
-- 
initialStateID = "00000000000000000000"

nextstid oldstid parm =
  SHA1.sha1 (oldstid ++ show parm)

