/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef COMMAND_H
#define COMMAND_H

#include <QObject>
#include <QList>
#include <QVariant>

#include "reply.h"

namespace WebIssues
{

/**
* Class storing information about a command to execute.
*
* This class stores the command keyword and arguments, optional attachment
* and rules for validating the response. This information is used by CommandManager
* to send a request to the WebIssues server.
*
* This class also provides signals for updating progress and for processing
* the server's reply to the command (or binary response when requested).
*
* In order to provide commands to be executed by the CommandManager, create
* a subclass of the AbstractBatch and implement the fetchNext() method which
* creates the Command objects.
*/
class Command : public QObject
{
    Q_OBJECT
public:
    /**
    * Default constructor.
    */
    Command();

    /**
    * Destructor.
    */
    ~Command();

public:
    /**
    * Set the command keyword.
    *
    * Refer to the WebIssues protocol documentation for a list of recognized commands.
    */
    void setKeyword( const QString& keyword ) { m_keyword = keyword; }

    /**
    * Return the command keyword.
    */
    const QString& keyword() const { return m_keyword; }

    /**
    * Set a list of command arguments.
    *
    * The arguments can be integers or strings. Refer to the WebIssues protocol
    * documentation for a list of arguments for each command.
    */
    void setArgs( const QList<QVariant>& args ) { m_args = args; }

    /**
    * Return the list of command arguments.
    */
    const QList<QVariant>& args() const { return m_args; }

    /**
    * Add an integer value to the list of command arguments.
    */
    void addArg( int number );

    /**
    * Add a string value to the list of command arguments.
    */
    void addArg( const QString& string );

    /**
    * Set the attachment data for the command.
    *
    * The attachment is sent to the server together with the command.
    */
    void setAttachment( const QByteArray& data ) { m_attachment = data; }

    /**
    * Return the attachment data for the command.
    */
    const QByteArray& attachment() const { return m_attachment; }

    /**
    * Set a flag indicating if binary response is accepted.
    *
    * If @c true, the binaryBlock() signal is emitted for each chunk of received data.
    * By default this flag is set to @c false.
    */
    void setAcceptBinaryResponse( bool accept ) { m_acceptBinaryResponse = accept; }

    /**
    * Return @c true if binary response is accepted.
    */
    bool acceptBinaryResponse() const { return m_acceptBinaryResponse; }

    /**
    * Set a flag indicating if a @c NULL reply is accepted.
    *
    * If the server returns a @c NULL reply, the validation rules are not checked.
    * The commandReply() signal is still emitted with an empty reply.
    */
    void setAcceptNullReply( bool accept ) { m_acceptNullReply = accept; }

    /**
    * Return @c true if a @c NULL reply is accepted.
    */
    bool acceptNullReply() const { return m_acceptNullReply; }

    /**
    * Set a list of rules for validating the reply.
    *
    * The server's reply is validated against this list. Each line must match the
    * corresponding rule in the list, in the same order as they are defined. Otherwise
    * an error is reported and processing the command is aborted.
    */
    void setRules( const QList<ReplyRule>& rules ) { m_rules = rules; }

    /**
    * Get the list of rules for validating the reply.
    */
    const QList<ReplyRule>& rules() const { return m_rules; }

    /**
    * Add a rule to the list of rules for validating the reply.
    *
    * @param signature The signature of the reply line.
    * @param multiplicity A flag indicating how many times the line can occur.
    */
    void addRule( const QString& signature, ReplyRule::Multiplicity multiplicity );

    /**
    * Emit the sendProgress() signal.
    *
    * This method is called by the CommandManager while sending the request.
    */
    void setSendProgress( int done, int total );

    /**
    * Emit the readProgress() signal.
    *
    * This method is called by the CommandManager while receiving the response.
    */
    void setReadProgress( int done, int total );

    /**
    * Emit the commandReply() signal.
    *
    * This method is called by the CommandManager after receiving and validating
    * the server's reply.
    */
    void setCommandReply( const Reply& reply );

    /**
    * Emit the binaryBlock() signal.
    *
    * This method is called by the CommandManager while receiving the response.
    */
    void setBinaryBlock( const char* data, int length );

signals:
    /**
    * Emitted when the progress of sending the request changes.
    *
    * This signal is emitted while sending the request to the server.
    */
    void sendProgress( int done, int total );

    /**
    * Emitted when the progress of reading the response changes.
    *
    * This signal is emitted while receiving the response from the server.
    */
    void readProgress( int done, int total );

    /**
    * Emitted when the reply is received and validated.
    *
    * This signal is emitted after receiving and validating the server's reply.
    * The reply is guaranteed to be valid according to the validation rules.
    */
    void commandReply( const Reply& reply );

    /**
    * Emitted when a block of binary response is received.
    *
    * This signal is emitted for each block of received binary response
    * if the acceptBinaryResponse flag is set to @c true.
    */
    void binaryBlock( const char* data, int length );

private:
    QString m_keyword;
    QList<QVariant> m_args;
    QByteArray m_attachment;

    bool m_acceptBinaryResponse;
    bool m_acceptNullReply;
    QList<ReplyRule> m_rules;
};

}

#endif
