/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef ABSTRACTTABLEMODEL_H
#define ABSTRACTTABLEMODEL_H

#include <QObject>

class QPixmap;

namespace WebIssues
{

class UpdateEvent;

/**
* Column type.
*/
enum Column
{
    Column_Name,
    Column_ID,
    Column_CreatedDate,
    Column_CreatedBy,
    Column_ModifiedDate,
    Column_ModifiedBy,
    Column_Size,
    Column_Description,
    Column_Login,
    Column_Access,
    Column_Required,
    Column_Details,
    Column_DefaultValue,
    Column_OldValue,
    Column_NewValue,
    Column_Type,

    /**
    * The value added to attribute identifier to create user defined columns.
    */
    Column_UserDefined = 1000
};

/**
* Abstract model providing information about rows of a table.
*
* One or more table models can be added to a MultiTableModel
* in order to populate a ListView.
*
* Inherited classes must implement at least the text() method.
* They should use the signals to notify the MultiTableModel model
* about the changes.
*
* The model automatically registers itself to receive UpdateEvent
* events when some data is updated.
*
* @see MultiTableModel
*/
class AbstractTableModel : public QObject
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param parent The parent object.
    */
    AbstractTableModel( QObject* parent );

    /**
    * Destructor.
    */
    ~AbstractTableModel();

public:
    /**
    * Return the text of a given column of a row.
    *
    * This method must be implemented in the inherited class.
    */
    virtual QString text( int id, Column column ) const = 0;

    /**
    * Return the icon of a given row.
    *
    * The default implementation returns an empty pixmap.
    */
    virtual QPixmap icon( int id ) const;

    /**
    * Return the tool tip of a given row.
    *
    * The default implementation returns an empty string.
    */
    virtual QString toolTip( int id ) const;

    /**
    * Compare two rows using given column.
    *
    * The default implementation compares the texts using QString::localeAwareCompare().
    */
    virtual int compare( int id1, int id2, Column column ) const;

protected:
    /**
    * Handle the event indicating that some data was updated.
    */
    virtual void updateEvent( UpdateEvent* e );

protected: // overrides
    void customEvent( QEvent* e );

signals:
    /**
    * Emitted when the entire table needs to be re-indexed.
    */
    void tableChanged();

    /**
    * Emitted when all rows need to be updated.
    */
    void rowsChanged();

    /**
    * Emitted when a single row needs to be updated.
    */
    void rowChanged( int id );
};

}

#endif
