/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef FORMDATAMESSAGE_H
#define FORMDATAMESSAGE_H

#include <QString>
#include <QBuffer>

/**
* Class encapsulating a form data message.
*
* This class creates a <tt>multipart/form-data</tt> message as defined in [RFC 2388].
*
* The message may consist of a number of form fields and attachments. It can be used as
* a body of a <tt>POST</tt> request in the <tt>HTTP</tt> protocol.
*/
class FormDataMessage
{
public:
    /**
    * Default constructor.
    *
    * Create an empty message.
    */
    FormDataMessage();

    /**
    * Destructor.
    */
    ~FormDataMessage();

public:
    /**
    * Add a form field to the message.
    *
    * @param name Name of the form field.
    *
    * @param value Value of the form field.
    */
    void addField( const QString& name, const QByteArray& value );

    /**
    * Add an attached file to the message.
    *
    * @param name Name of the form field.
    *
    * @param fileName Name of the attached file.
    *
    * @param data The content of the attached file.
    */
    void addAttachment( const QString& name, const QString& fileName, const QByteArray& data );

    /**
    * Finish the message.
    *
    * This method must be called after adding the last field to properly terminate the message.
    */
    void finish();

    /**
    * Return the <tt>Content-Type</tt> header of the message.
    *
    * @return The <tt>multipart/form-data</tt> content type with appropriate boundary.
    */
    QString contentType();

    /**
    * Return the body of the message.
    */
    QByteArray body() { return m_body; }

private:
    QString randomString( int length );

    void writeLine();
    void writeLine( const QString& line );

    void writeRawBytes( const QByteArray& bytes );

private:
    QString m_boundary;

    QByteArray m_body;
    QBuffer m_buffer;
};

#endif
