/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef CONNECTIONMANAGER_H
#define CONNECTIONMANAGER_H

#include <QObject>

class QSettings;

class ConnectionInfo;
class AttachmentsCache;

/**
* Class handling the connection to the server.
*
* This class creates and owns the command manager, data manager
* and view manager. It also stores the information about the connection.
*
* The connection manager also provides a mechanism for accessing
* configuration and cache files unique to each instance of the server.
*
* The instance of this class is available using the connectionManager
* global variable. It is created and owned by the Application object.
*/
class ConnectionManager : public QObject
{
    Q_OBJECT
public:
    /**
    * Constructor.
    */
    ConnectionManager();

    /**
    * Destructor.
    */
    ~ConnectionManager();

public:
    /**
    * Create the command manager.
    *
    * Call this method before executing the login batch.
    */
    void createCommandManager();

    /**
    * Set up the connection.
    *
    * This methods creates the data manager and view manager
    * and loads user's configuration.
    *
    * @param info The connection information object created by
    * the login batch
    */
    void openConnection( ConnectionInfo* info );

    /**
    * Post closing the connection.
    */
    void delayedCloseConnection();

    /**
    * Return @c true if the connection is opened.
    */
    bool isConnected() const { return m_info != NULL; }

    /**
    * Return the information about the connection.
    */
    ConnectionInfo* connectionInfo() const { return m_info; }

    /**
    * Return the server settings.
    *
    * The settings file is unique to each server.
    */
    QSettings* serverSettings() const { return m_settings; }

    /**
    * Return the attachments cache.
    */
    AttachmentsCache* attachmentsCache() const { return m_attachmentsCache; }

    /**
    * Return the path of a data file.
    *
    * The data file if unique to each server.
    *
    * @param name The name of the data file.
    */
    QString locateDataFile( const QString& name );

    /**
    * Return the path of a cache file.
    *
    * The cache file if unique to each server.
    *
    * @param name The name of the cache file.
    */
    QString locateCacheFile( const QString& name );

    /**
    * Export all server settings to the given settings file.
    */
    void exportSettings( QSettings* settings );

    /**
    * Check if the given settings file can be imported.
    */
    bool canImportSettings( const QSettings* settings );

    /**
    * Import all server settings to the given settings file.
    */
    void importSettings( const QSettings* settings );

public slots:
    /**
    * Close the connection.
    *
    * This method destroys the view manager, data manager
    * and comand manager and frees all data.
    */
    void closeConnection();

private:
    bool checkAccess( const QString& path );

    void copyAllKeys( QSettings* dest, const QSettings* src );

private:
    QString m_dataPath;
    QString m_cachePath;

    ConnectionInfo* m_info;

    QSettings* m_settings;

    AttachmentsCache* m_attachmentsCache;
};

/**
* Global pointer used to access the ConnectionManager.
*/
extern ConnectionManager* connectionManager;

#endif
