/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "tablemodels.h"

#include <QPixmap>

#include "attributes/definitioninfo.h"
#include "attributes/definitionparser.h"
#include "attributes/definitionhelper.h"
#include "attributes/datetimehelper.h"
#include "data/datamanager.h"
#include "data/datarows.h"
#include "data/updateevent.h"
#include "data/connectioninfo.h"
#include "rdb/utilities.h"
#include "connectionmanager.h"
#include "tablemodelshelper.h"
#include "issueviewsettings.h"
#include "linklocator.h"
#include "iconloader.h"

BaseTableModel::BaseTableModel( QObject* parent ) : RDB::AbstractTableModel( parent )
{
    dataManager->addObserver( this );
}

BaseTableModel::~BaseTableModel()
{
    if ( dataManager )
        dataManager->removeObserver( this );
}

QString BaseTableModel::columnName( int column ) const
{
    return TableModelsHelper::columnName( column );
}

void BaseTableModel::updateEvent( UpdateEvent* /*e*/ )
{
}

void BaseTableModel::customEvent( QEvent* e )
{
    if ( e->type() == UpdateEvent::Type )
        updateEvent( (UpdateEvent*)e );
}

UsersTableModel::UsersTableModel( QObject* parent ) : BaseTableModel( parent )
{
}

UsersTableModel::~UsersTableModel()
{
}

QString UsersTableModel::text( int id, int column ) const
{
    const UserRow* row = dataManager->users()->find( id );
    if ( !row )
        return QString();

    switch ( column ) {
        case Column_Name:
            return row->name();
        case Column_Login:
            return row->login();
        case Column_Access:
            if ( row->access() == NormalAccess )
                return tr( "Regular User" );
            if ( row->access() == AdminAccess )
                return tr( "System Administrator" );
            return tr( "Disabled User" );
        default:
            return QString();
    }
}

QPixmap UsersTableModel::icon( int id, int column ) const
{
    if ( column != Column_Name )
        return QPixmap();

    const UserRow* row = dataManager->users()->find( id );
    if ( row ) {
        if ( row->access() == AdminAccess )
            return IconLoader::overlayedPixmap( "user", "overlay-admin" );
        if ( row->access() == NoAccess )
            return IconLoader::pixmap( "user-disabled" );
    }
    return IconLoader::pixmap( "user" );
}

void UsersTableModel::updateEvent( UpdateEvent* e )
{
    if ( e->unit() == UpdateEvent::Users )
        emit dataChanged();
}

MembersTableModel::MembersTableModel( int projectId, QObject* parent ) : BaseTableModel( parent ),
    m_projectId( projectId )
{
}

MembersTableModel::~MembersTableModel()
{
}

QString MembersTableModel::text( int id, int column ) const
{
    const MemberRow* row = dataManager->members()->find( id, m_projectId );
    if ( !row )
        return QString();

    switch ( column ) {
        case Column_Name:
            return TableModelsHelper::userName( row->userId() );
        case Column_Access:
            if ( row->access() == NormalAccess )
                return tr( "Regular Member" );
            if ( row->access() == AdminAccess )
                return tr( "Project Administrator" );
            return QString();
        default:
            return QString();
    }
}

QPixmap MembersTableModel::icon( int id, int column ) const
{
    if ( column != Column_Name )
        return QPixmap();

    const MemberRow* row = dataManager->members()->find( id, m_projectId );
    if ( row && row->access() == AdminAccess )
        return IconLoader::overlayedPixmap( "user", "overlay-admin" );

    return IconLoader::pixmap( "user" );
}

void MembersTableModel::updateEvent( UpdateEvent* e )
{
    if ( e->unit() == UpdateEvent::Users )
        emit dataChanged();
}

TypesTableModel::TypesTableModel( QObject* parent ) : BaseTableModel( parent )
{
}

TypesTableModel::~TypesTableModel()
{
}

QString TypesTableModel::text( int id, int column ) const
{
    const TypeRow* row = dataManager->types()->find( id );
    if ( !row )
        return QString();

    switch ( column ) {
        case Column_Name:
            return row->name();
        default:
            return QString();
    }
}

QPixmap TypesTableModel::icon( int /*id*/, int column ) const
{
    if ( column != Column_Name )
        return QPixmap();
    return IconLoader::pixmap( "type" );
}

void TypesTableModel::updateEvent( UpdateEvent* e )
{
    if ( e->unit() == UpdateEvent::Types )
        emit dataChanged();
}

AttributesTableModel::AttributesTableModel( QObject* parent ) : BaseTableModel( parent )
{
}

AttributesTableModel::~AttributesTableModel()
{
}

QString AttributesTableModel::text( int id, int column ) const
{
    const AttributeRow* row = dataManager->attributes()->find( id );
    if ( !row )
        return QString();

    if ( column == Column_Name )
        return row->name();

    DefinitionParser parser;
    DefinitionInfo info = parser.parse( row->definition() );

    switch ( column ) {
        case Column_Type:
            return DefinitionHelper::typeName( info.type() );
        case Column_DefaultValue:
            return info.defaultValue();
        case Column_Required:
            return info.required() ? tr( "Yes" ) : tr( "No" );
        case Column_Details:
            return DefinitionHelper::metadataDetails( info );
        default:
            return QString();
    }
}

QPixmap AttributesTableModel::icon( int /*id*/, int column ) const
{
    if ( column != Column_Name )
        return QPixmap();
    return IconLoader::pixmap( "attribute" );
}

void AttributesTableModel::updateEvent( UpdateEvent* /*e*/ )
{
}

ProjectsTableModel::ProjectsTableModel( QObject* parent ) : BaseTableModel( parent )
{
}

ProjectsTableModel::~ProjectsTableModel()
{
}

QString ProjectsTableModel::text( int id, int column ) const
{
    const ProjectRow* row = dataManager->projects()->find( id );
    if ( !row )
        return QString();

    switch ( column ) {
        case Column_Name:
            return row->name();
        default:
            return QString();
    }
}

QPixmap ProjectsTableModel::icon( int id, int column ) const
{
    if ( column != Column_Name )
        return QPixmap();

    if ( connectionManager->connectionInfo()->access() == AdminAccess )
        return IconLoader::overlayedPixmap( "project", "overlay-admin" );

    int userId = connectionManager->connectionInfo()->userId();
    const MemberRow* member = dataManager->members()->find( userId, id );
    if ( member && member->access() == AdminAccess )
        return IconLoader::overlayedPixmap( "project", "overlay-admin" );

    return IconLoader::pixmap( "project" );
}

void ProjectsTableModel::updateEvent( UpdateEvent* e )
{
    if ( e->unit() == UpdateEvent::Projects )
        emit dataChanged();
    if ( e->unit() == UpdateEvent::Types )
        emit dataChanged();
}

FoldersTableModel::FoldersTableModel( QObject* parent ) : BaseTableModel( parent )
{
}

FoldersTableModel::~FoldersTableModel()
{
}

QString FoldersTableModel::text( int id, int column ) const
{
    const FolderRow* row = dataManager->folders()->find( id );
    if ( !row )
        return QString();

    switch ( column ) {
        case Column_Name:
            return row->name();
        case Column_Type:
            return TableModelsHelper::typeName( row->typeId() );
        default:
            return QString();
    }
}

QPixmap FoldersTableModel::icon( int id, int column ) const
{
    if ( column != Column_Name )
        return QPixmap();

    WatchState state = dataManager->folderWatchState( id );
    switch ( state ) {
        case WatchNormal:
            return IconLoader::overlayedPixmap( "folder", "overlay-gray" );
        case WatchUpdated:
            return IconLoader::overlayedPixmap( "folder", "overlay-green" );
        case WatchNew:
            return IconLoader::overlayedPixmap( "folder", "overlay-yellow" );
        default:
            return IconLoader::pixmap( "folder" );
    }
}

void FoldersTableModel::updateEvent( UpdateEvent* e )
{
    if ( e->unit() == UpdateEvent::Folder )
        emit dataChanged();
}

IssuesTableModel::IssuesTableModel( int folderId, QObject* parent ) : BaseTableModel( parent ),
    m_folderId( folderId )
{
}

IssuesTableModel::~IssuesTableModel()
{
}

QString IssuesTableModel::text( int id, int column ) const
{
    const IssueRow* row = dataManager->issues()->find( id );
    if ( !row )
        return QString();

    if ( column > Column_UserDefined ) {
        int attributeId = column - Column_UserDefined;
        const ValueRow* value = dataManager->values()->find( attributeId, id );
        return value ? value->value() : QString();
    }

    switch ( column ) {
        case Column_Name:
            return row->name();
        case Column_ID:
            return TableModelsHelper::formatId( id );
        case Column_CreatedDate:
            return DateTimeHelper::formatDateTime( row->createdDate() );
        case Column_CreatedBy:
            return TableModelsHelper::userName( row->createdUser() );
        case Column_ModifiedDate:
            return DateTimeHelper::formatDateTime( row->modifiedDate() );
        case Column_ModifiedBy:
            return TableModelsHelper::userName( row->modifiedUser() );
        default:
            return QString();
    }
}

QPixmap IssuesTableModel::icon( int id, int column ) const
{
    if ( column != Column_Name )
        return QPixmap();

    WatchState state = dataManager->issueWatchState( id );
    switch ( state ) {
        case WatchUpdated:
            return IconLoader::overlayedPixmap( "issue", "overlay-green" );
        case WatchNew:
            return IconLoader::overlayedPixmap( "issue", "overlay-yellow" );
        default:
            return IconLoader::pixmap( "issue" );
    }
}

QString IssuesTableModel::toolTip( int id, int /*column*/ ) const
{
    const IssueRow* issue = dataManager->issues()->find( id );
    const FolderRow* folder = issue ? dataManager->folders()->find( issue->folderId() ) : NULL;
    if ( !folder )
        return QString();

    IssueViewSettings settings;
    settings.openIssueView( folder->typeId() );

    QList<int> attributes = settings.loadAttributes();

    QString tip = "<table cellspacing=\"3\" cellpadding=\"0\" width=\"400\">";

    tip += QLatin1String( "<tr><td><b>" );
    tip += QString( "%1: %2" ).arg( TableModelsHelper::formatId( id ), LinkLocator::convertToHtml( issue->name(), 0 ) );
    tip += QLatin1String( "</b></td></tr>" );

    tip += QLatin1String( "<tr><td>" );
    tip += tr( "Created by %1 on %2" ).arg( TableModelsHelper::userName( issue->createdUser() ), DateTimeHelper::formatDateTime( issue->createdDate() ) );
    tip += QLatin1String( "</td></tr>" );

    if ( issue->stamp() > id ) {
        tip += QLatin1String( "<tr><td>" );
        tip += tr( "Modified by %1 on %2" ).arg( TableModelsHelper::userName( issue->modifiedUser() ), DateTimeHelper::formatDateTime( issue->modifiedDate() ) );
        tip += QLatin1String( "</td></tr>" );
    }

    tip += QLatin1String( "</table><table cellspacing=\"3\" cellpadding=\"0\">" );

    for ( int i = 0; i < attributes.count(); i++ ) {
        int attributeId = attributes.at( i );
        const ValueRow* value = dataManager->values()->find( attributeId, id );
        if ( value ) {
            tip += QLatin1String( "<tr><td>" );
            tip += LinkLocator::escape( TableModelsHelper::attributeName( attributeId ) );
            tip += QLatin1String( ":</td><td>" );
            tip += LinkLocator::convertToHtml( value->value(), 0 );
            tip += QLatin1String( "</td></tr>" );
        }
    }

    tip += QLatin1String( "</table>" );

    return tip;
}

int IssuesTableModel::compare( int id1, int id2, int column ) const
{
    if ( column == Column_ID )
        return id1 - id2;

    if ( column == Column_CreatedDate || column == Column_ModifiedDate ) {
        const IssueRow* row1 = dataManager->issues()->find( id1 );
        const IssueRow* row2 = dataManager->issues()->find( id2 );

        if ( !row1 || !row2 )
            return 0;

        if ( column == Column_CreatedDate )
            return DateTimeHelper::compareDateTime( row1->createdDate(), row2->createdDate() );
        if ( column == Column_ModifiedDate )
            return DateTimeHelper::compareDateTime( row1->modifiedDate(), row2->modifiedDate() );
    }

    return AbstractTableModel::compare( id1, id2, column );
}

void IssuesTableModel::updateEvent( UpdateEvent* e )
{
    if ( e->unit() == UpdateEvent::Folder && e->id() == m_folderId )
        emit dataChanged();

    if ( e->unit() == UpdateEvent::Issue ) {
        const IssueRow* row = dataManager->issues()->find( e->id() );
        if ( row && row->folderId() == m_folderId )
            emit dataChanged();
    }

    if ( e->unit() == UpdateEvent::Users )
        emit dataChanged();
}

AttachmentsTableModel::AttachmentsTableModel( int issueId, QObject* parent ) : BaseTableModel( parent ),
    m_issueId( issueId )
{
}

AttachmentsTableModel::~AttachmentsTableModel()
{
}

QString AttachmentsTableModel::text( int id, int column ) const
{
    const AttachmentRow* row = dataManager->attachments()->find( id );
    if ( !row )
        return QString();

    switch ( column ) {
        case Column_Name:
            return row->name();
        case Column_ID:
            return TableModelsHelper::formatId( id );
        case Column_CreatedDate:
            return DateTimeHelper::formatDateTime( row->createdDate() );
        case Column_CreatedBy:
            return TableModelsHelper::userName( row->createdUser() );
        case Column_Size:
            return TableModelsHelper::formatSize( row->size() );
        case Column_Description:
            return row->description();
        default:
            return QString();
    }
}

int AttachmentsTableModel::compare( int id1, int id2, int column ) const
{
    if ( column == Column_ID )
        return id1 - id2;

    if ( column == Column_Size || column == Column_ModifiedDate ) {
        const AttachmentRow* row1 = dataManager->attachments()->find( id1 );
        const AttachmentRow* row2 = dataManager->attachments()->find( id2 );

        if ( !row1 || !row2 )
            return 0;

        if ( column == Column_Size )
            return row1->size() - row2->size();
        if ( column == Column_CreatedDate )
            return DateTimeHelper::compareDateTime( row1->createdDate(), row2->createdDate() );
    }

    return AbstractTableModel::compare( id1, id2, column );
}

QPixmap AttachmentsTableModel::icon( int /*id*/, int column ) const
{
    if ( column != Column_Name )
        return QPixmap();

    return IconLoader::pixmap( "file" );
}

void AttachmentsTableModel::updateEvent( UpdateEvent* e )
{
    if ( e->unit() == UpdateEvent::Issue && e->id() == m_issueId )
        emit dataChanged();

    if ( e->unit() == UpdateEvent::Users )
        emit dataChanged();
}
