/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "viewerwindow.h"

#include <QAction>
#include <QCloseEvent>
#include <QApplication>
#include <QSettings>
#include <QToolBar>

#include "widgets/statusbar.h"
#include "dialogs/settingsdialog.h"
#include "views/view.h"
#include "xmlui/builder.h"
#include "configdata.h"
#include "iconloader.h"

ViewerWindow::ViewerWindow() :
    m_view( NULL )
{
    setAttribute( Qt::WA_DeleteOnClose, true );

    QAction* action;

    action = new QAction( IconLoader::icon( "file-close" ), tr( "&Close" ), this );
    action->setShortcut( tr( "Ctrl+W" ) );
    connect( action, SIGNAL( triggered() ), this, SLOT( close() ) );
    setAction( "close", action );

    action = new QAction( IconLoader::icon( "configure" ), tr( "&Configure WebIssues..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( configure() ) );
    setAction( "configure", action );

    action = new QAction( IconLoader::icon( "help-contents" ), tr( "WebIssues &Manual" ), this );
    action->setShortcut( QKeySequence::HelpContents );
    connect( action, SIGNAL( triggered() ), qApp, SLOT( manual() ) );
    setAction( "manual", action );

    action = new QAction( IconLoader::icon( "webissues" ), tr( "&About WebIssues" ), this );
    connect( action, SIGNAL( triggered() ), qApp, SLOT( about() ) );
    setAction( "about", action );

    action = new QAction( IconLoader::icon( "about-qt" ), tr( "About &Qt" ), this );
    connect( action, SIGNAL( triggered() ), qApp, SLOT( aboutQt() ) );
    setAction( "aboutQt", action );

    setTitle( "menuMain", tr( "&View" ) );
    setTitle( "menuTools", tr( "&Tools" ) );
    setTitle( "menuHelp", tr( "&Help" ) );

    loadXmlUiFile( ":/resources/viewerwindow.xml" );

    m_builder = new XmlUi::Builder( this );
    connect( m_builder, SIGNAL( toolBarCreated( QToolBar* ) ), this, SLOT( toolBarCreated( QToolBar* ) ) );

    m_builder->addClient( this );

    setStatusBar( new ::StatusBar( this ) );
}

ViewerWindow::~ViewerWindow()
{
    if ( !m_view )
        return;

    QSettings* settings = configData->settings();
    settings->beginGroup( QString( "ViewerWindow/WebIssues::%1" ).arg( m_view->metaObject()->className() ) );
    settings->setValue( "Size", size() );
    settings->endGroup();
}

void ViewerWindow::setView( View* view )
{
    m_view = view;

    connect( view, SIGNAL( captionChanged( const QString& ) ), this, SLOT( captionChanged( const QString& ) ) );

    connect( view, SIGNAL( enabledChanged( bool ) ), this, SLOT( enabledChanged( bool ) ) );

    connect( view, SIGNAL( showInfo( const QString& ) ), statusBar(), SLOT( showInfo( const QString& ) ) );
    connect( view, SIGNAL( showWarning( const QString& ) ), statusBar(), SLOT( showWarning( const QString& ) ) );
    connect( view, SIGNAL( showBusy( const QString& ) ), statusBar(), SLOT( showBusy( const QString& ) ) );

    connect( view, SIGNAL( showSummary( const QPixmap&, const QString& ) ), statusBar(), SLOT( showSummary( const QPixmap&, const QString& ) ) );

    setCentralWidget( view->mainWidget() );

    enabledChanged( view->isEnabled() );

    QSettings* settings = configData->settings();
    settings->beginGroup( QString( "ViewerWindow/WebIssues::%1" ).arg( m_view->metaObject()->className() ) );
    resize( settings->value( "Size", view->viewerSizeHint() ).toSize() );
    settings->endGroup();

    captionChanged( view->caption() );
}

void ViewerWindow::enabledChanged( bool enabled )
{
    m_view->mainWidget()->setVisible( enabled );

    if ( enabled )
        m_builder->addClient( m_view );
    else
        m_builder->removeClient( m_view );
}

void ViewerWindow::closeEvent( QCloseEvent* e )
{
    if ( !m_view || m_view->queryClose() )
        e->accept();
    else
        e->ignore();
}

QMenu* ViewerWindow::createPopupMenu()
{
    return NULL;
}

void ViewerWindow::configure()
{
    SettingsDialog dialog( this );
    dialog.exec();
}

void ViewerWindow::captionChanged( const QString& caption )
{
    setWindowTitle( tr( "%1 - WebIssues" ).arg( caption ) );
}

void ViewerWindow::toolBarCreated( QToolBar* toolBar )
{
    toolBar->setIconSize( QSize( 22, 22 ) );
}
