/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "treeviewsettings.h"

#include <QSettings>
#include <QStringList>

#include "attributes/definitioninfo.h"
#include "data/datamanager.h"
#include "data/datarows.h"
#include "rdb/utilities.h"
#include "tablemodels.h"
#include "connectionmanager.h"

TreeViewSettings::TreeViewSettings()
{
    clear();
}

TreeViewSettings::~TreeViewSettings()
{
}

void TreeViewSettings::openUsersList()
{
    clear();

    m_group = "UsersList";

    m_columns.append( Column_Name );
    m_columns.append( Column_Login );
    m_columns.append( Column_Access );
}

void TreeViewSettings::openMembersList()
{
    clear();

    m_group = "MembersList";

    m_columns.append( Column_Name );
    m_columns.append( Column_Access );
}

void TreeViewSettings::openTypesTree()
{
    clear();

    m_group = "TypesTree";

    m_columns.append( Column_Name );
    m_columns.append( Column_Type );
    m_columns.append( Column_DefaultValue );
    m_columns.append( Column_Required );
    m_columns.append( Column_Details );
}

void TreeViewSettings::openProjectsTree()
{
    clear();

    m_group = "ProjectsTree";

    m_columns.append( Column_Name );
    m_columns.append( Column_Type );
}

void TreeViewSettings::openIssuesList( int typeId )
{
    clear();

    m_group = "IssuesList";
    m_typeId = typeId;

    m_columns.append( Column_ID );
    m_columns.append( Column_Name );
    m_columns.append( Column_ModifiedDate );
    m_columns.append( Column_ModifiedBy );

    m_sortColumn = Column_ID;
    m_sortOrder = Qt::AscendingOrder;

    m_longName = true;
}

void TreeViewSettings::openAttachmentsList()
{
    clear();

    m_group = "AttachmentsList";

    m_columns.append( Column_ID );
    m_columns.append( Column_Name );
    m_columns.append( Column_Size );
    m_columns.append( Column_CreatedDate );
    m_columns.append( Column_CreatedBy );
    m_columns.append( Column_Description );

    m_sortColumn = Column_ID;
    m_sortOrder = Qt::AscendingOrder;
}

void TreeViewSettings::clear()
{
    m_group.clear();
    m_typeId = 0;

    m_columns.clear();

    m_sortColumn = Column_Name;
    m_sortOrder = Qt::AscendingOrder;

    m_longName = false;
}

QList<int> TreeViewSettings::loadColumns()
{
    if ( m_typeId != 0 ) {
        QSettings* settings = connectionManager->serverSettings();

        settings->beginGroup( QString( "%1/Columns" ).arg( m_group ) );

        QStringList list = settings->value( QString( "Type%1" ).arg( m_typeId ) ).toStringList();

        settings->endGroup();

        if ( !list.isEmpty() )
            return deserializeColumns( list );
    }

    return m_columns;
}

void TreeViewSettings::saveColumns( const QList<int>& columns )
{
    if ( m_typeId != 0 ) {
        QSettings* settings = connectionManager->serverSettings();

        settings->beginGroup( QString( "%1/Columns" ).arg( m_group ) );

        settings->setValue( QString( "Type%1" ).arg( m_typeId ), serializeColumns( columns ) );

        settings->endGroup();
    }
}

QList<int> TreeViewSettings::availableColumns()
{
    QList<int> columns = m_columns;

    if ( m_group == QLatin1String( "IssuesList" ) ) {
        columns.append( Column_CreatedDate );
        columns.append( Column_CreatedBy );
    }

    if ( m_typeId != 0 ) {
        RDB::ForeignConstIterator<AttributeRow> it( dataManager->attributes()->parentIndex(), m_typeId );
        QList<const AttributeRow*> list = RDB::localeAwareSortRows( it, &AttributeRow::name );
        for ( int i = 0; i < list.count(); i++ )
            columns.append( Column_UserDefined + list.at( i )->attributeId() );
    }

    return columns;
}

QPair<int, Qt::SortOrder> TreeViewSettings::loadSortOrder()
{
    QPair<int, Qt::SortOrder> order( m_sortColumn, m_sortOrder );

    if ( m_typeId != 0 ) {
        QSettings* settings = connectionManager->serverSettings();

        settings->beginGroup( QString( "%1/SortOrder" ).arg( m_group ) );

        QStringList list = settings->value( QString( "Type%1" ).arg( m_typeId ) ).toStringList();

        if ( list.count() == 2 ) {
            order.first = list.at( 0 ).toInt();
            order.second = (Qt::SortOrder)list.at( 1 ).toInt();
        }

        settings->endGroup();
    }

    return order;
}

void TreeViewSettings::saveSortOrder( const QPair<int, Qt::SortOrder>& order )
{
    if ( m_typeId != 0 ) {
        QSettings* settings = connectionManager->serverSettings();

        settings->beginGroup( QString( "%1/SortOrder" ).arg( m_group ) );

        QStringList list;
        list.append( QString::number( order.first ) );
        list.append( QString::number( (int)order.second ) );
        settings->setValue( QString( "Type%1" ).arg( m_typeId ), list );

        settings->endGroup();
    }
}

QPair<int, Qt::SortOrder> TreeViewSettings::defaultSortOrder()
{
    return qMakePair( m_sortColumn, m_sortOrder );
}

QList<int> TreeViewSettings::fixedColumns()
{
    QList<int> columns;

    columns.append( Column_ID );
    columns.append( Column_Name );

    return columns;
}

QMap<int, int> TreeViewSettings::loadColumnWidths()
{
    QList<int> usedColumns = loadColumns();
    QMap<int, int> widths;

    QSettings* settings = connectionManager->serverSettings();

    settings->beginGroup( QString( "%1/Widths" ).arg( m_group ) );

    for ( int i = 0; i < usedColumns.count(); i++ ) {
        int column = usedColumns.at( i );
        int width = settings->value( QString( "Column%1" ).arg( (int)column ), defaultColumnWidth( column ) ).toInt();
        widths.insert( column, width );
    }

    settings->endGroup();

    return widths;
}

void TreeViewSettings::saveColumnWidths( const QMap<int, int>& widths )
{
    QList<int> usedColumns = widths.keys();

    QSettings* settings = connectionManager->serverSettings();

    settings->beginGroup( QString( "%1/Widths" ).arg( m_group ) );

    for ( int i = 0; i < usedColumns.count(); i++ ) {
        int column = usedColumns.at( i );
        int width = widths.value( column );
        settings->setValue( QString( "Column%1" ).arg( column ), width );
    }

    settings->endGroup();
}

QList<int> TreeViewSettings::loadExpandedNodes()
{
    QList<int> nodes;

    QSettings* settings = connectionManager->serverSettings();

    settings->beginGroup( QString( "%1/Nodes" ).arg( m_group ) );

    QStringList list = settings->value( "Expanded" ).toStringList();

    for ( int i = 0; i < list.count(); i++ )
        nodes.append( list.at( i ).toInt() );

    settings->endGroup();

    return nodes;
}

void TreeViewSettings::saveExpandedNodes( const QList<int>& nodes )
{
    QSettings* settings = connectionManager->serverSettings();

    settings->beginGroup( QString( "%1/Nodes" ).arg( m_group ) );

    QStringList list;
    for ( int i = 0; i < nodes.count(); i++ )
        list.append( QString::number( nodes.at( i ) ) );

    settings->setValue( "Expanded", list );

    settings->endGroup();
}

QStringList TreeViewSettings::serializeColumns( const QList<int>& columns )
{
    QStringList list;

    for ( int i = 0; i < columns.count(); i++ )
        list.append( QString::number( columns.at( i ) ) );

    return list;
}

QList<int> TreeViewSettings::deserializeColumns( const QStringList& list )
{
    QList<int> columns;

    for ( int i = 0; i < list.count(); i++ ) {
        int column = list.at( i ).toInt();
        if ( column > Column_UserDefined ) {
            int attributeId = column - Column_UserDefined;
            if ( !dataManager->attributes()->find( attributeId ) )
                continue;
        }
        columns.append( column );
    }

    if ( columns.count() == 0 )
        columns.append( Column_Name );

    return columns;
}

int TreeViewSettings::defaultColumnWidth( int column )
{
    switch ( column ) {
        case Column_Name:
            return m_longName ? 300 : 150;
        case Column_ID:
            return 50;
        case Column_CreatedDate:
        case Column_ModifiedDate:
            return 110;
        case Column_CreatedBy:
        case Column_ModifiedBy:
            return 150;
        case Column_Login:
            return 100;
        case Column_Size:
            return 70;
        case Column_Description:
            return 300;
        case Column_Access:
            return 150;
        case Column_Required:
            return 60;
        case Column_Details:
            return 300;
        case Column_DefaultValue:
        case Column_OldValue:
        case Column_NewValue:
            return 150;
        case Column_Type:
            return 150;
        default:
            if ( column > Column_UserDefined ) {
                int attributeId = column - Column_UserDefined;
                const AttributeRow* attribute = dataManager->attributes()->find( attributeId );
                if ( attribute ) {
                    DefinitionInfo info = DefinitionInfo::fromString( attribute->definition() );
                    return defaultColumnWidth( info );
                }
            }
            return 150;
    }
}

int TreeViewSettings::defaultColumnWidth( const DefinitionInfo& info )
{
    switch ( info.type() ) {
        case DefinitionInfo::Text:
            return 150;
        case DefinitionInfo::Enum:
            return 150;
        case DefinitionInfo::Numeric:
            return 100;
        case DefinitionInfo::DateTime:
            return 110;
        case DefinitionInfo::User:
            return 150;
        default:
            return 150;
    }
}
