#
# Copyright (C) 2005, 2006, 2007 Nikolas Zimmermann <zimmermann@kde.org>
# Copyright (C) 2006 Anders Carlsson <andersca@mac.com>
# Copyright (C) 2006, 2007 Samuel Weinig <sam@webkit.org>
# Copyright (C) 2006 Alexey Proskuryakov <ap@webkit.org>
# Copyright (C) 2006, 2007 Apple Inc. All rights reserved.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public License
# aint with this library; see the file COPYING.LIB.  If not, write to
# the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
# Boston, MA 02110-1301, USA.

package CodeGeneratorJS;

use File::stat;

my $module = "";
my $outputDir = "";

my @headerContent = ();
my @implContentHeader = ();
my @implContent = ();
my %implIncludes = ();

# Default .h template
my $headerTemplate = << "EOF";
/*
    This file is part of the WebKit open source project.
    This file has been generated by generate-bindings.pl. DO NOT MODIFY!

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/
EOF

# Default constructor
sub new
{
    my $object = shift;
    my $reference = { };

    $codeGenerator = shift;
    $outputDir = shift;

    bless($reference, $object);
    return $reference;
}

sub finish
{
    my $object = shift;

    # Commit changes!
    $object->WriteData();
}

sub leftShift($$) {
    my ($value, $distance) = @_;
    return (($value << $distance) & 0xFFFFFFFF);
}

# Uppercase the first letter, while respecting WebKit style guidelines. 
# E.g., xmlEncoding becomes XMLEncoding, but xmlllang becomes Xmllang.
sub WK_ucfirst
{
    my $param = shift;
    my $ret = ucfirst($param);
    $ret =~ s/Xml/XML/ if $ret =~ /^Xml[^a-z]/;
    return $ret;
}

# Params: 'domClass' struct
sub GenerateInterface
{
    my $object = shift;
    my $dataNode = shift;
    my $defines = shift;

    # Start actual generation
    $object->GenerateHeader($dataNode);
    $object->GenerateImplementation($dataNode);

    my $name = $dataNode->name;

    # Open files for writing
    my $headerFileName = "$outputDir/JS$name.h";
    my $implFileName = "$outputDir/JS$name.cpp";

    open($IMPL, ">$implFileName") || die "Couldn't open file $implFileName";
    open($HEADER, ">$headerFileName") || die "Couldn't open file $headerFileName";
}

# Params: 'idlDocument' struct
sub GenerateModule
{
    my $object = shift;
    my $dataNode = shift;

    $module = $dataNode->module;
}

sub GetParentClassName
{
    my $dataNode = shift;

    return $dataNode->extendedAttributes->{"LegacyParent"} if $dataNode->extendedAttributes->{"LegacyParent"};
    return "KJS::DOMObject" if @{$dataNode->parents} eq 0;
    return "JS" . $codeGenerator->StripModule($dataNode->parents(0));
}

sub GetLegacyHeaderIncludes
{
    my $legacyParent = shift;

    return "#include \"JSHTMLInputElementBase.h\"\n\n" if $legacyParent eq "JSHTMLInputElementBase";
    return "#include \"kjs_window.h\"\n\n" if $legacyParent eq "KJS::Window";
    return "#include \"kjs_events.h\"\n\n" if $module eq "events";
    return "#include \"kjs_css.h\"\n\n" if $module eq "css";
    return "#include \"kjs_html.h\"\n\n" if $module eq "html";

    die "Don't know what headers to include for module $module";
}

sub AvoidInclusionOfType
{
    my $type = shift;

    # Special case: SVGRect.h / SVGPoint.h / SVGNumber.h / SVGMatrix.h do not exist.
    return 1 if $type eq "SVGRect" or $type eq "SVGPoint" or $type eq "SVGNumber" or $type eq "SVGMatrix";
    return 0;
}

sub UsesManualToJSImplementation
{
    my $type = shift;

    return 1 if $type eq "Node" or $type eq "Document" or $type eq "HTMLCollection" or $type eq "SVGPathSeg" or $type eq "StyleSheet" or $type eq "CSSRule" or $type eq "CSSValue" or $type eq "Event";
    return 0;
}

sub IndexGetterReturnsStrings
{
    my $type = shift;

    return 1 if $type eq "CSSStyleDeclaration" or $type eq "MediaList";
    return 0;
}

sub CreateSVGContextInterfaceName
{
    my $type = shift;

    return $type if $codeGenerator->IsSVGAnimatedType($type);
    return "SVGPathSeg" if $type =~ /^SVGPathSeg/ and $type ne "SVGPathSegList";

    return "";
}

sub AddIncludesForType
{
    my $type = $codeGenerator->StripModule(shift);

    # When we're finished with the one-file-per-class
    # reorganization, we won't need these special cases.
    if ($codeGenerator->IsPrimitiveType($type) or AvoidInclusionOfType($type)
        or $type eq "DOMString" or $type eq "DOMObject" or $type eq "RGBColor") {
    } elsif ($type =~ /SVGPathSeg/) {
        $joinedName = $type;
        $joinedName =~ s/Abs|Rel//;
        $implIncludes{"${joinedName}.h"} = 1;
    } elsif ($type eq "XPathNSResolver") {
        $implIncludes{"JSXPathNSResolver.h"} = 1;
        $implIncludes{"JSCustomXPathNSResolver.h"} = 1;
    } else {
        # default, include the same named file
        $implIncludes{"${type}.h"} = 1;
    }

    # additional includes (things needed to compile the bindings but not the header)

    if ($type eq "CanvasRenderingContext2D") {
        $implIncludes{"CanvasGradient.h"} = 1;
        $implIncludes{"CanvasPattern.h"} = 1;
        $implIncludes{"CanvasStyle.h"} = 1;
    }

    if ($type eq "CanvasGradient" or $type eq "XPathNSResolver") {
        $implIncludes{"PlatformString.h"} = 1;
    }
}

sub AddIncludesForSVGAnimatedType
{
    my $type = shift;
    $type =~ s/SVGAnimated//;

    if ($type eq "Point" or $type eq "Rect") {
        $implIncludes{"Float$type.h"} = 1;
    } elsif ($type eq "String") {
        $implIncludes{"PlatformString.h"} = 1;
    }
}

sub AddClassForwardIfNeeded
{
    my $implClassName = shift;

    # SVGAnimatedLength/Number/etc.. are typedefs to SVGAnimtatedTemplate, so don't use class forwards for them!
    push(@headerContent, "class $implClassName;\n\n") unless $codeGenerator->IsSVGAnimatedType($implClassName);
}

sub IsSVGTypeNeedingContextParameter
{
    my $implClassName = shift;

    if ($implClassName =~ /SVG/ and not $implClassName =~ /Element/) {
        return 1 unless $implClassName =~ /SVGPaint/ or $implClassName =~ /SVGColor/ or $implClassName =~ /SVGDocument/;
    }

    return 0;
}

sub HashValueForClassAndName
{
    my $class = shift;
    my $name = shift;

    # SVG Filter enums live in WebCore namespace (platform/graphics/)
    if ($class =~ /^SVGFE*/ or $class =~ /^SVGComponentTransferFunctionElement$/) {
        return "WebCore::$name";
    }

    return "${class}::$name";
}

sub GenerateHeader
{
    my $object = shift;
    my $dataNode = shift;

    my $interfaceName = $dataNode->name;
    my $className = "JS$interfaceName";
    my $implClassName = $interfaceName;

    # We only support multiple parents with SVG (for now).
    if (@{$dataNode->parents} > 1) {
        die "A class can't have more than one parent" unless $interfaceName =~ /SVG/;
        $codeGenerator->AddMethodsConstantsAndAttributesFromParentClasses($dataNode);
    }

    my $hasLegacyParent = $dataNode->extendedAttributes->{"LegacyParent"};
    my $hasRealParent = @{$dataNode->parents} > 0;
    my $hasParent = $hasLegacyParent || $hasRealParent;
    my $parentClassName = GetParentClassName($dataNode);
    my $conditional = $dataNode->extendedAttributes->{"Conditional"};

    # - Add default header template
    @headerContent = split("\r", $headerTemplate);

    # - Add header protection
    push(@headerContent, "\n#ifndef $className" . "_H");
    push(@headerContent, "\n#define $className" . "_H\n\n");

    my $conditionalString;
    if ($conditional) {
        $conditionalString = "ENABLE(" . join(") && ENABLE(", split(/&/, $conditional)) . ")";
        push(@headerContent, "\n#if ${conditionalString}\n\n");
    }

    if (exists $dataNode->extendedAttributes->{"LegacyParent"}) {
        push(@headerContent, GetLegacyHeaderIncludes($dataNode->extendedAttributes->{"LegacyParent"}));
    } else {
        if ($hasParent) {
            push(@headerContent, "#include \"$parentClassName.h\"\n");
        } else {
            push(@headerContent, "#include \"kjs_binding.h\"\n");
        }
    }

    # Get correct pass/store types respecting PODType flag
    my $podType = $dataNode->extendedAttributes->{"PODType"};
    my $passType = $podType ? "JSSVGPODTypeWrapper<$podType>*" : "$implClassName*";
    push(@headerContent, "#include \"$podType.h\"\n") if $podType and $podType ne "float";

    push(@headerContent, "#include \"JSSVGPODTypeWrapper.h\"\n") if $podType;

    my $numConstants = @{$dataNode->constants};
    my $numAttributes = @{$dataNode->attributes};
    my $numFunctions = @{$dataNode->functions};

    push(@headerContent, "\nnamespace WebCore {\n\n");

    # Implementation class forward declaration
    AddClassForwardIfNeeded($implClassName) unless $podType;

    # Class declaration
    push(@headerContent, "class $className : public $parentClassName {\n");
    push(@headerContent, "public:\n");

    # Constructor
    if ($dataNode->extendedAttributes->{"DoNotCache"}) {
        push(@headerContent, "    $className($passType);\n");
    } else {
        if (IsSVGTypeNeedingContextParameter($implClassName)) {
            push(@headerContent, "    $className(KJS::ExecState*, $passType, SVGElement* context);\n");
        } else {
            push(@headerContent, "    $className(KJS::ExecState*, $passType);\n");
        }
    }

    # Destructor
    if (!$hasParent or $interfaceName eq "Document") {
        push(@headerContent, "    virtual ~$className();\n");
    }

    # Getters
    if ($numAttributes > 0 || $dataNode->extendedAttributes->{"GenerateConstructor"}) {
        push(@headerContent, "    virtual bool getOwnPropertySlot(KJS::ExecState*, const KJS::Identifier&, KJS::PropertySlot&);\n");
        push(@headerContent, "    KJS::JSValue* getValueProperty(KJS::ExecState*, int token) const;\n");
        if ($dataNode->extendedAttributes->{"CustomGetOwnPropertySlot"}) {
            push(@headerContent, "    bool customGetOwnPropertySlot(KJS::ExecState*, const KJS::Identifier&, KJS::PropertySlot&);\n");
        }
    }

    # Check if we have any writable properties
    my $hasReadWriteProperties = 0;
    foreach (@{$dataNode->attributes}) {
        if ($_->type !~ /^readonly\ attribute$/) {
            $hasReadWriteProperties = 1;
        }
    }

    if ($hasReadWriteProperties) {
        push(@headerContent, "    virtual void put(KJS::ExecState*, const KJS::Identifier&, KJS::JSValue*, int attr = KJS::None);\n");
        push(@headerContent, "    void putValueProperty(KJS::ExecState*, int, KJS::JSValue*, int attr);\n");
        if ($dataNode->extendedAttributes->{"CustomPutFunction"}) {
            push(@headerContent, "    bool customPut(KJS::ExecState*, const KJS::Identifier&, KJS::JSValue*, int attr);\n");
        }
    }

    # Class info
    push(@headerContent, "    virtual const KJS::ClassInfo* classInfo() const { return &info; }\n");
    push(@headerContent, "    static const KJS::ClassInfo info;\n\n");

    # Custom mark function
    if ($dataNode->extendedAttributes->{"CustomMarkFunction"}) {
        push(@headerContent, "    virtual void mark();\n\n");
    }

    # Custom pushEventHandlerScope function
    if ($dataNode->extendedAttributes->{"CustomPushEventHandlerScope"}) {
        push(@headerContent, "    virtual void pushEventHandlerScope(KJS::ExecState*, KJS::ScopeChain&) const;\n\n");
    }

    # Custom call functions
    if ($dataNode->extendedAttributes->{"CustomCall"}) {
        push(@headerContent, "    virtual KJS::JSValue* callAsFunction(KJS::ExecState*, KJS::JSObject*, const KJS::List&);\n");
        push(@headerContent, "    virtual bool implementsCall() const;\n\n");
    }

    # Constructor object getter
    if ($dataNode->extendedAttributes->{"GenerateConstructor"}) {
        push(@headerContent, "    static KJS::JSValue* getConstructor(KJS::ExecState*);\n");
    }

    my $numCustomFunctions = 0;
    my $numCustomAttributes = 0;

    # Attribute and function enums
    my $hasAttrEnum = ($numAttributes > 0) || $dataNode->extendedAttributes->{"GenerateConstructor"};
    push(@headerContent, "    enum {\n") if ($hasAttrEnum);

    if ($numAttributes > 0) {
        push(@headerContent, "        // Attributes\n        ");

        my $i = -1;
        foreach (@{$dataNode->attributes}) {
            my $attribute = $_;

            $numCustomAttributes++ if $attribute->signature->extendedAttributes->{"Custom"};
            $numCustomAttributes++ if $attribute->signature->extendedAttributes->{"CustomGetter"};
            $numCustomAttributes++ if $attribute->signature->extendedAttributes->{"CustomSetter"};

            $i++;
            if ((($i % 4) eq 0) and ($i ne 0)) {
                push(@headerContent, "\n        ");
            }

            my $value = $attribute->signature->type =~ /Constructor$/
                      ? $attribute->signature->name . "ConstructorAttrNum"
                      : WK_ucfirst($attribute->signature->name) . "AttrNum";
            $value .= ", " if (($i < $numAttributes - 1) or (($i eq $numAttributes - 1) and (($numFunctions ne 0) or $dataNode->extendedAttributes->{"GenerateConstructor"})));
            push(@headerContent, $value);
        }
    }

    if ($dataNode->extendedAttributes->{"GenerateConstructor"}) {
        push(@headerContent, "\n\n") if $numAttributes > 0;
        push(@headerContent, "        // The Constructor Attribute\n");
        push(@headerContent, "        ConstructorAttrNum");
    }

    push(@headerContent, "\n    };\n") if ($hasAttrEnum);

    if ($numCustomAttributes > 0) {
        push(@headerContent, "\n    // Custom attributes\n");

        foreach my $attribute (@{$dataNode->attributes}) {
            if ($attribute->signature->extendedAttributes->{"Custom"}) {
                push(@headerContent, "    KJS::JSValue* " . $attribute->signature->name . "(KJS::ExecState*) const;\n");
                if ($attribute->type !~ /^readonly/) {
                    push(@headerContent, "    void set" . WK_ucfirst($attribute->signature->name) . "(KJS::ExecState*, KJS::JSValue*);\n");
                }
            } elsif ($attribute->signature->extendedAttributes->{"CustomGetter"}) {
                push(@headerContent, "    KJS::JSValue* " . $attribute->signature->name . "(KJS::ExecState*) const;\n");
            } elsif ($attribute->signature->extendedAttributes->{"CustomSetter"}) {
                if ($attribute->type !~ /^readonly/) {
                    push(@headerContent, "    void set" . WK_ucfirst($attribute->signature->name) . "(KJS::ExecState*, KJS::JSValue*);\n");
                }
            }
        }
    }

    foreach my $function (@{$dataNode->functions}) {
        $numCustomFunctions++ if $function->signature->extendedAttributes->{"Custom"};
    }

    if ($numCustomFunctions > 0) {
        push(@headerContent, "\n    // Custom functions\n");
        foreach my $function (@{$dataNode->functions}) {
            if ($function->signature->extendedAttributes->{"Custom"}) {
                push(@headerContent, "    KJS::JSValue* " . $function->signature->name . "(KJS::ExecState*, const KJS::List&);\n");
            }
        }
    }

    # Index setter
    if ($dataNode->extendedAttributes->{"HasCustomIndexSetter"}) {
        push(@headerContent, "    void indexSetter(KJS::ExecState*, unsigned index, KJS::JSValue*, int attr);\n");
    }

    if (!$hasParent) {
        if ($podType) {
            push(@headerContent, "    JSSVGPODTypeWrapper<$podType>* impl() const { return m_impl.get(); }\n");
            push(@headerContent, "    SVGElement* context() const { return m_context.get(); }\n\n");
            push(@headerContent, "private:\n");
            push(@headerContent, "    RefPtr<SVGElement> m_context;\n");
            push(@headerContent, "    RefPtr<JSSVGPODTypeWrapper<$podType> > m_impl;\n");
        } elsif (IsSVGTypeNeedingContextParameter($implClassName)) {
            push(@headerContent, "    $implClassName* impl() const { return m_impl.get(); }\n");
            push(@headerContent, "    SVGElement* context() const { return m_context.get(); }\n\n");
            push(@headerContent, "private:\n");
            push(@headerContent, "    RefPtr<SVGElement> m_context;\n");
            push(@headerContent, "    RefPtr<$implClassName > m_impl;\n");
        } else {
            push(@headerContent, "    $implClassName* impl() const { return m_impl.get(); }\n\n");
            push(@headerContent, "private:\n");
            push(@headerContent, "    RefPtr<$implClassName> m_impl;\n");
        }
    } elsif ($dataNode->extendedAttributes->{"GenerateNativeConverter"}) {
        push(@headerContent, "    $implClassName* impl() const;\n");
    }

    # Index getter
    if ($dataNode->extendedAttributes->{"HasIndexGetter"}) {
        push(@headerContent, "private:\n");
        push(@headerContent, "    static KJS::JSValue* indexGetter(KJS::ExecState*, KJS::JSObject*, const KJS::Identifier&, const KJS::PropertySlot&);\n");
    }
    # Name getter
    if ($dataNode->extendedAttributes->{"HasNameGetter"} || $dataNode->extendedAttributes->{"HasOverridingNameGetter"}) {
        push(@headerContent, "private:\n");
        push(@headerContent, "    static bool canGetItemsForName(KJS::ExecState*, $implClassName*, const KJS::Identifier&);\n");
        push(@headerContent, "    static KJS::JSValue* nameGetter(KJS::ExecState*, KJS::JSObject*, const KJS::Identifier&, const KJS::PropertySlot&);\n");
    }

    push(@headerContent, "};\n\n");

    if (!$hasParent || $dataNode->extendedAttributes->{"GenerateToJS"}) {
        if ($podType) {
            push(@headerContent, "KJS::JSValue* toJS(KJS::ExecState*, JSSVGPODTypeWrapper<$podType>*, SVGElement* context);\n");
        } elsif (IsSVGTypeNeedingContextParameter($implClassName)) {
            push(@headerContent, "KJS::JSValue* toJS(KJS::ExecState*, $passType, SVGElement* context);\n");
        } elsif ($interfaceName eq "Node") {
            push(@headerContent, "KJS::JSValue* toJS(KJS::ExecState*, PassRefPtr<Node>);\n");
        } else {
            push(@headerContent, "KJS::JSValue* toJS(KJS::ExecState*, $passType);\n");
        }
    }
    if (!$hasParent || $dataNode->extendedAttributes->{"GenerateNativeConverter"}) {
        if ($podType) {
            push(@headerContent, "$podType to${interfaceName}(KJS::JSValue*);\n");
        } else {
            push(@headerContent, "$implClassName* to${interfaceName}(KJS::JSValue*);\n");
        }
    }
    push(@headerContent, "\n");

    # Add prototype declaration -- code adopted from the KJS_DEFINE_PROTOTYPE and KJS_DEFINE_PROTOTYPE_WITH_PROTOTYPE macros
    push(@headerContent, "class ${className}Prototype : public KJS::JSObject {\n");
    push(@headerContent, "public:\n");
    if ($dataNode->extendedAttributes->{"DoNotCache"}) {
        push(@headerContent, "    static KJS::JSObject* self();\n");
    } else {
        push(@headerContent, "    static KJS::JSObject* self(KJS::ExecState* exec);\n");
    }
    push(@headerContent, "    virtual const KJS::ClassInfo* classInfo() const { return &info; }\n");
    push(@headerContent, "    static const KJS::ClassInfo info;\n");
    if ($numFunctions > 0 || $numConstants > 0) {
        push(@headerContent, "    bool getOwnPropertySlot(KJS::ExecState*, const KJS::Identifier&, KJS::PropertySlot&);\n");
    }
    if ($numConstants ne 0) {
        push(@headerContent, "    KJS::JSValue* getValueProperty(KJS::ExecState*, int token) const;\n");
    }
    if ($dataNode->extendedAttributes->{"DoNotCache"}) {
        push(@headerContent, "    ${className}Prototype() { }\n");
    } else {
        push(@headerContent, "    ${className}Prototype(KJS::ExecState* exec)\n");
        if ($hasParent && $parentClassName ne "KJS::DOMNodeFilter") {
            push(@headerContent, "        : KJS::JSObject(${parentClassName}Prototype::self(exec)) { }\n");
        } else {
            push(@headerContent, "        : KJS::JSObject(exec->lexicalInterpreter()->builtinObjectPrototype()) { }\n");
        }
    }

    push(@headerContent, "};\n\n");

    if ($numFunctions > 0) {
        push(@headerContent,"// Functions\n\n");
        foreach my $function (@{$dataNode->functions}) {
            push(@headerContent, prototypeFunctionFor($className, WK_ucfirst($function->signature->name)));
        }        
    }

    push(@headerContent, "} // namespace WebCore\n\n");
    push(@headerContent, "#endif // ${conditionalString}\n\n") if $conditional;
    push(@headerContent, "#endif\n");
}

sub GenerateImplementation
{
    my ($object, $dataNode) = @_;

    my $interfaceName = $dataNode->name;
    my $className = "JS$interfaceName";
    my $implClassName = $interfaceName;

    my $hasLegacyParent = $dataNode->extendedAttributes->{"LegacyParent"};
    my $hasRealParent = @{$dataNode->parents} > 0;
    my $hasParent = $hasLegacyParent || $hasRealParent;
    my $parentClassName = GetParentClassName($dataNode);
    my $conditional = $dataNode->extendedAttributes->{"Conditional"};

    # - Add default header template
    @implContentHeader = split("\r", $headerTemplate);
    push(@implContentHeader, "\n#include \"config.h\"\n\n");
    my $conditionalString;
    if ($conditional) {
        $conditionalString = "ENABLE(" . join(") && ENABLE(", split(/&/, $conditional)) . ")";
        push(@implContentHeader, "\n#if ${conditionalString}\n\n");
    }

    if ($className =~ /^JSSVG/) {
        push(@implContentHeader, "#include \"Document.h\"\n");
        push(@implContentHeader, "#include \"Frame.h\"\n");
        push(@implContentHeader, "#include \"SVGDocumentExtensions.h\"\n");
        push(@implContentHeader, "#include \"SVGElement.h\"\n");
        push(@implContentHeader, "#include \"SVGAnimatedTemplate.h\"\n");

        if ($className =~ /^JSSVGAnimated/) {
            AddIncludesForSVGAnimatedType($interfaceName);
        }
    }

    push(@implContentHeader, "#include \"$className.h\"\n\n");
    push(@implContentHeader, "#include <wtf/GetPtr.h>\n\n");

    AddIncludesForType($interfaceName);

    @implContent = ();

    push(@implContent, "\nusing namespace KJS;\n\n");
    push(@implContent, "namespace WebCore {\n\n");

    # - Add all attributes in a hashtable definition
    my $numAttributes = @{$dataNode->attributes};
    $numAttributes++ if $dataNode->extendedAttributes->{"GenerateConstructor"};

    if ($numAttributes > 0) {
        my $hashSize = $numAttributes;
        my $hashName = $className . "Table";

        my @hashKeys = ();      # ie. 'insertBefore'
        my @hashValues = ();    # ie. 'JSNode::InsertBefore'
        my @hashSpecials = ();    # ie. 'DontDelete|Function'
        my @hashParameters = ();  # ie. '2'

        foreach my $attribute (@{$dataNode->attributes}) {
            my $name = $attribute->signature->name;
            push(@hashKeys, $name);

            my $value = $className . "::" . ($attribute->signature->type =~ /Constructor$/
                                       ? $attribute->signature->name . "ConstructorAttrNum"
                                       : WK_ucfirst($attribute->signature->name) . "AttrNum");
            push(@hashValues, $value);

            my @specials = ();
            push(@specials, "DontDelete") unless $attribute->signature->extendedAttributes->{"Deletable"};
            push(@specials, "DontEnum") if $attribute->signature->extendedAttributes->{"DontEnum"};
            push(@specials, "ReadOnly") if $attribute->type =~ /readonly/;
            my $special = (@specials > 0) ? join("|", @specials) : "0";
            push(@hashSpecials, $special);

            my $numParameters = "0";
            push(@hashParameters, $numParameters);
        }

        if ($dataNode->extendedAttributes->{"GenerateConstructor"}) {
            push(@hashKeys, "constructor");
            push(@hashValues, $className . "::ConstructorAttrNum");
            push(@hashSpecials, "DontDelete|DontEnum|ReadOnly");
            push(@hashParameters, "0");
        }

        $object->GenerateHashTable($hashName, $hashSize,
                                   \@hashKeys, \@hashValues,
                                   \@hashSpecials, \@hashParameters);
    }

    my $numConstants = @{$dataNode->constants};
    my $numFunctions = @{$dataNode->functions};

    # - Add all constants
    if ($dataNode->extendedAttributes->{"GenerateConstructor"}) {
        $hashSize = $numConstants;
        $hashName = $className . "ConstructorTable";

        @hashKeys = ();
        @hashValues = ();
        @hashSpecials = ();
        @hashParameters = ();

        foreach my $constant (@{$dataNode->constants}) {
            my $name = $constant->name;
            push(@hashKeys, $name);

            my $value = HashValueForClassAndName($implClassName, $name);
            push(@hashValues, $value);

            my $special = "DontDelete|ReadOnly";
            push(@hashSpecials, $special);

            my $numParameters = 0;
            push(@hashParameters, $numParameters);
        }

        $object->GenerateHashTable($hashName, $hashSize,
                                   \@hashKeys, \@hashValues,
                                   \@hashSpecials, \@hashParameters);

        my $protoClassName;
        $protoClassName = "${className}Prototype";

        push(@implContent, constructorFor($className, $protoClassName, $interfaceName, $dataNode->extendedAttributes->{"CanBeConstructed"}));
    }

    # - Add functions and constants to a hashtable definition
    $hashSize = $numFunctions + $numConstants;
    $hashName = $className . "PrototypeTable";

    @hashKeys = ();
    @hashValues = ();
    @hashSpecials = ();
    @hashParameters = ();

    foreach my $constant (@{$dataNode->constants}) {
        my $name = $constant->name;
        push(@hashKeys, $name);

        my $value = HashValueForClassAndName($implClassName, $name);
        push(@hashValues, $value);

        my $special = "DontDelete|ReadOnly";
        push(@hashSpecials, $special);

        my $numParameters = 0;
        push(@hashParameters, $numParameters);
    }

    foreach my $function (@{$dataNode->functions}) {
        my $name = $function->signature->name;
        push(@hashKeys, $name);

        my $value = "&" . $className . "PrototypeFunction" . WK_ucfirst($name) . "::create";
        push(@hashValues, $value);

        my @specials = ();
        push(@specials, "DontDelete") unless $function->signature->extendedAttributes->{"Deletable"};
        push(@specials, "DontEnum") if $function->signature->extendedAttributes->{"DontEnum"};
        push(@specials, "Function");        
        my $special = (@specials > 0) ? join("|", @specials) : "0";
        push(@hashSpecials, $special);

        my $numParameters = @{$function->parameters};
        push(@hashParameters, $numParameters);
    }

    $object->GenerateHashTable($hashName, $hashSize,
                               \@hashKeys, \@hashValues,
                               \@hashSpecials, \@hashParameters);

    push(@implContent, "const ClassInfo ${className}Prototype::info = { \"${interfaceName}Prototype\", 0, &${className}PrototypeTable };\n\n");
    if ($dataNode->extendedAttributes->{"DoNotCache"}) {
        push(@implContent, "JSObject* ${className}Prototype::self()\n");
        push(@implContent, "{\n");
        push(@implContent, "    return new ${className}Prototype();\n");
        push(@implContent, "}\n\n");
    } else {
        push(@implContent, "JSObject* ${className}Prototype::self(ExecState* exec)\n");
        push(@implContent, "{\n");
        push(@implContent, "    return KJS::cacheGlobalObject<${className}Prototype>(exec, \"[[${className}.prototype]]\");\n");
        push(@implContent, "}\n\n");
    }
    if ($numConstants > 0 || $numFunctions > 0) {
        push(@implContent, "bool ${className}Prototype::getOwnPropertySlot(ExecState* exec, const Identifier& propertyName, PropertySlot& slot)\n");
        push(@implContent, "{\n");
        if ($numConstants eq 0) {
            push(@implContent, "    return getStaticFunctionSlot<JSObject>(exec, &${className}PrototypeTable, this, propertyName, slot);\n");
        } elsif ($numFunctions eq 0) {
            push(@implContent, "    return getStaticValueSlot<${className}Prototype, JSObject>(exec, &${className}PrototypeTable, this, propertyName, slot);\n");
        } else {
            push(@implContent, "    return getStaticPropertySlot<${className}Prototype, JSObject>(exec, &${className}PrototypeTable, this, propertyName, slot);\n");
        }
        push(@implContent, "}\n\n");
    }
    if ($numConstants ne 0) {
        push(@implContent, "JSValue* ${className}Prototype::getValueProperty(ExecState*, int token) const\n{\n");
        push(@implContent, "    // The token is the numeric value of its associated constant\n");
        push(@implContent, "    return jsNumber(token);\n}\n\n");
    }

    # - Initialize static ClassInfo object
    push(@implContent, "const ClassInfo $className" . "::info = { \"$interfaceName\", ");
    if ($hasParent) {
        push(@implContent, "&" .$parentClassName . "::info, ");
    } else {
        push(@implContent, "0, ");
    }

    if ($numAttributes > 0) {
        push(@implContent, "&${className}Table ");
    } else {
        push(@implContent, "0 ")
    }
    push(@implContent, "};\n\n");

    # Get correct pass/store types respecting PODType flag
    my $podType = $dataNode->extendedAttributes->{"PODType"};
    my $passType = $podType ? "JSSVGPODTypeWrapper<$podType>*" : "$implClassName*";

    # Constructor
    if ($dataNode->extendedAttributes->{"DoNotCache"}) {
        push(@implContent, "${className}::$className($passType impl)\n");
        push(@implContent, "    : $parentClassName(impl)\n");
    } else {
        my $needsSVGContext = IsSVGTypeNeedingContextParameter($implClassName);
        if ($needsSVGContext) {
            push(@implContent, "${className}::$className(ExecState* exec, $passType impl, SVGElement* context)\n");
        } else {
            push(@implContent, "${className}::$className(ExecState* exec, $passType impl)\n");
        }

        if ($hasParent) {
            if ($needsSVGContext and $parentClassName =~ /SVG/) {
                push(@implContent, "    : $parentClassName(exec, impl, context)\n");
            } else {
                push(@implContent, "    : $parentClassName(exec, impl)\n");
            }
        } else {
            if ($needsSVGContext) {
                push(@implContent, "    : m_context(context)\n");
                push(@implContent, "    , m_impl(impl)\n");
            } else {
                push(@implContent, "    : m_impl(impl)\n");
            }            
        }
    }

    if ($dataNode->extendedAttributes->{"DoNotCache"}) {
        push(@implContent, "{\n    setPrototype(${className}Prototype::self());\n}\n\n");
    } else {
        push(@implContent, "{\n    setPrototype(${className}Prototype::self(exec));\n}\n\n");
    }

    # Destructor
    if (!$hasParent) {
        push(@implContent, "${className}::~$className()\n");
        push(@implContent, "{\n");

        if ($interfaceName eq "Node") {
            push(@implContent, "    ScriptInterpreter::forgetDOMNodeForDocument(m_impl->document(), m_impl.get());\n");
        } else {
            if ($podType) {
                my $animatedType = $implClassName;
                $animatedType =~ s/SVG/SVGAnimated/;

                # Special case for JSSVGNumber
                if ($codeGenerator->IsSVGAnimatedType($animatedType) and $podType ne "float") {
                    push(@implContent, "    JSSVGPODTypeWrapperCache<$podType, $animatedType>::forgetWrapper(m_impl.get());\n");
                }
            }
            push(@implContent, "    ScriptInterpreter::forgetDOMObject(m_impl.get());\n");
        }

        push(@implContent, "\n}\n\n");
    }

    # Document needs a special destructor because it's a special case for caching. It needs
    # its own special handling rather than relying on the caching that Node normally does.
    if ($interfaceName eq "Document") {
        push(@implContent, "${className}::~$className()\n");
        push(@implContent, "{\n    ScriptInterpreter::forgetDOMObject(static_cast<${implClassName}*>(impl()));\n}\n\n");
    }

    # Attributes
    if ($numAttributes ne 0) {
        push(@implContent, "bool ${className}::getOwnPropertySlot(ExecState* exec, const Identifier& propertyName, PropertySlot& slot)\n");
        push(@implContent, "{\n");

        if ($interfaceName eq "NamedNodeMap" or $interfaceName eq "HTMLCollection") {
            push(@implContent, "    JSValue* proto = prototype();\n");
            push(@implContent, "    if (proto->isObject() && static_cast<JSObject*>(proto)->hasProperty(exec, propertyName))\n");
            push(@implContent, "        return false;\n\n");
        }

        my $hasNameGetterGeneration = sub {
            push(@implContent, "    if (canGetItemsForName(exec, static_cast<$implClassName*>(impl()), propertyName)) {\n");
            push(@implContent, "        slot.setCustom(this, nameGetter);\n");
            push(@implContent, "        return true;\n");
            push(@implContent, "    }\n");
            $implIncludes{"AtomicString.h"} = 1;
        };

        if ($dataNode->extendedAttributes->{"HasOverridingNameGetter"}) {
            &$hasNameGetterGeneration();
        }

        my $requiresManualLookup = $dataNode->extendedAttributes->{"HasIndexGetter"} || $dataNode->extendedAttributes->{"HasNameGetter"};
        if ($requiresManualLookup) {
            push(@implContent, "    const HashEntry* entry = Lookup::findEntry(&${className}Table, propertyName);\n");
            push(@implContent, "    if (entry) {\n");
            push(@implContent, "        slot.setStaticEntry(this, entry, staticValueGetter<$className>);\n");
            push(@implContent, "        return true;\n");
            push(@implContent, "    }\n");
        }

        if ($dataNode->extendedAttributes->{"HasIndexGetter"}) {
            push(@implContent, "    bool ok;\n");
            push(@implContent, "    unsigned index = propertyName.toUInt32(&ok, false);\n");
            push(@implContent, "    if (ok && index < static_cast<$implClassName*>(impl())->length()) {\n");
            push(@implContent, "        slot.setCustomIndex(this, index, indexGetter);\n");
            push(@implContent, "        return true;\n");
            push(@implContent, "    }\n");
        }

        if ($dataNode->extendedAttributes->{"HasNameGetter"}) {
            &$hasNameGetterGeneration();
        }

        if ($dataNode->extendedAttributes->{"CustomGetOwnPropertySlot"}) {
                push(@implContent, "    if (customGetOwnPropertySlot(exec, propertyName, slot))\n");
                push(@implContent, "        return true;\n");
        }

        if ($requiresManualLookup) {
            push(@implContent, "    return ${parentClassName}::getOwnPropertySlot(exec, propertyName, slot);\n");
        } else {
            push(@implContent, "    return getStaticValueSlot<$className, $parentClassName>(exec, &${className}Table, this, propertyName, slot);\n");
        }
        push(@implContent, "}\n\n");

        push(@implContent, "JSValue* ${className}::getValueProperty(ExecState* exec, int token) const\n{\n");

        push(@implContent, "    switch (token) {\n");

        foreach my $attribute (@{$dataNode->attributes}) {
            my $name = $attribute->signature->name;

            my $implClassNameForValueConversion = "";
            if (!$podType and ($codeGenerator->IsSVGAnimatedType($implClassName) or $attribute->type !~ /^readonly/)) {
                $implClassNameForValueConversion = $implClassName;
            }

            if ($attribute->signature->type =~ /Constructor$/) {
                push(@implContent, "    case " . $name . "ConstructorAttrNum: {\n");
            } else {
                push(@implContent, "    case " . WK_ucfirst($name) . "AttrNum: {\n");
            }

            if ($dataNode->extendedAttributes->{"CheckDomainSecurity"} && !$attribute->signature->extendedAttributes->{"DoNotCheckDomainSecurity"}) {
                push(@implContent, "        if (!isSafeScript(exec))\n");
                push(@implContent, "            return jsUndefined();\n");
            }

            if ($attribute->signature->extendedAttributes->{"Custom"} || $attribute->signature->extendedAttributes->{"CustomGetter"}) {
                push(@implContent, "        return $name(exec);\n");
            } elsif ($attribute->signature->extendedAttributes->{"CheckNodeSecurity"}) {
                $implIncludes{"kjs_dom.h"} = 1;
                push(@implContent, "        $implClassName* imp = static_cast<$implClassName*>(impl());\n\n");
                push(@implContent, "        return checkNodeSecurity(exec, imp->$name()) ? " . NativeToJSValue($attribute->signature, 0, $implClassName, $implClassNameForValueConversion, "imp->$name()") . " : jsUndefined();\n");
            } elsif ($attribute->signature->extendedAttributes->{"CheckFrameSecurity"}) {
                $implIncludes{"Document.h"} = 1;
                $implIncludes{"kjs_dom.h"} = 1;
                push(@implContent, "        $implClassName* imp = static_cast<$implClassName*>(impl());\n\n");
                push(@implContent, "        return checkNodeSecurity(exec, imp->contentDocument()) ? " . NativeToJSValue($attribute->signature,  0, $implClassName, $implClassNameForValueConversion, "imp->$name()") . " : jsUndefined();\n");
            } elsif ($attribute->signature->type =~ /Constructor$/) {
                my $constructorType = $codeGenerator->StripModule($attribute->signature->type);
                $constructorType =~ s/Constructor$//;
                push(@implContent, "        return JS" . $constructorType . "::getConstructor(exec);\n");
            } elsif (!@{$attribute->getterExceptions}) {
                if ($podType) {
                    push(@implContent, "        $podType imp(*impl());\n\n");
                    if ($podType eq "float") { # Special case for JSSVGNumber
                        push(@implContent, "        return " . NativeToJSValue($attribute->signature, 0, $implClassName, "", "imp") . ";\n");
                    } else {
                        push(@implContent, "        return " . NativeToJSValue($attribute->signature, 0, $implClassName, "", "imp.$name()") . ";\n");
                    }
                } else {
                    push(@implContent, "        $implClassName* imp = static_cast<$implClassName*>(impl());\n\n");
                    my $type = $codeGenerator->StripModule($attribute->signature->type);
                    my $jsType = NativeToJSValue($attribute->signature, 0, $implClassName, $implClassNameForValueConversion, "imp->$name()");

                    if ($codeGenerator->IsSVGAnimatedType($type)) {
                        push(@implContent, "        RefPtr<$type> obj = $jsType;\n");
                        push(@implContent, "        return toJS(exec, obj.get(), imp);\n");
                    } else {
                        push(@implContent, "        return $jsType;\n");
                    }
                }
            } else {
                push(@implContent, "        ExceptionCode ec = 0;\n");

                if ($podType) {
                    push(@implContent, "        $podType imp(*impl());\n\n");
                    push(@implContent, "        KJS::JSValue* result = " . NativeToJSValue($attribute->signature, 0, $implClassName, "", "imp.$name(ec)") . ";\n");
                } else {
                    push(@implContent, "        $implClassName* imp = static_cast<$implClassName*>(impl());\n\n");
                    push(@implContent, "        KJS::JSValue* result = " . NativeToJSValue($attribute->signature, 0, $implClassName, $implClassNameForValueConversion, "imp->$name(ec)") . ";\n");
                }

                push(@implContent, "        setDOMException(exec, ec);\n");
                push(@implContent, "        return result;\n");
            }
            push(@implContent, "    }\n");
        }

        if ($dataNode->extendedAttributes->{"GenerateConstructor"}) {
            push(@implContent, "    case ConstructorAttrNum:\n");
            push(@implContent, "        return getConstructor(exec);\n");
        }

        push(@implContent, "    }\n");
        push(@implContent, "    return 0;\n}\n\n");

        # Check if we have any writable attributes
        my $hasReadWriteProperties = 0;
        foreach my $attribute (@{$dataNode->attributes}) {
            $hasReadWriteProperties = 1 if $attribute->type !~ /^readonly/;
        }
        if ($hasReadWriteProperties) {
            push(@implContent, "void ${className}::put(ExecState* exec, const Identifier& propertyName, JSValue* value, int attr)\n");
            push(@implContent, "{\n");
            if ($dataNode->extendedAttributes->{"HasCustomIndexSetter"}) {
                push(@implContent, "    bool ok;\n");
                push(@implContent, "    unsigned index = propertyName.toUInt32(&ok, false);\n");
                push(@implContent, "    if (ok) {\n");
                push(@implContent, "        indexSetter(exec, index, value, attr);\n");
                push(@implContent, "        return;\n");
                push(@implContent, "    }\n");
            }
            if ($dataNode->extendedAttributes->{"CustomPutFunction"}) {
                push(@implContent, "    if (customPut(exec, propertyName, value, attr))\n");
                push(@implContent, "        return;\n");
            }

            push(@implContent, "    lookupPut<$className, $parentClassName>(exec, propertyName, value, attr, &${className}Table, this);\n");
            push(@implContent, "}\n\n");

            push(@implContent, "void ${className}::putValueProperty(ExecState* exec, int token, JSValue* value, int /*attr*/)\n");
            push(@implContent, "{\n");

            push(@implContent, "    switch (token) {\n");

            foreach my $attribute (@{$dataNode->attributes}) {
                if ($attribute->type !~ /^readonly/) {
                    my $name = $attribute->signature->name;

                    if ($attribute->signature->type =~ /Constructor$/) {
                        push(@implContent, "    case " . $name ."ConstructorAttrNum: {\n");
                    } else {
                        push(@implContent, "    case " . WK_ucfirst($name) . "AttrNum: {\n");
                    }

                    if ($dataNode->extendedAttributes->{"CheckDomainSecurity"} && !$attribute->signature->extendedAttributes->{"DoNotCheckDomainSecurity"}) {
                        push(@implContent, "        if (!isSafeScript(exec))\n");
                        push(@implContent, "            return;\n");
                    }

                    if ($attribute->signature->extendedAttributes->{"Custom"} || $attribute->signature->extendedAttributes->{"CustomSetter"}) {
                        push(@implContent, "        set" . WK_ucfirst($name) . "(exec, value);\n");
                    } elsif ($attribute->signature->type =~ /Constructor$/) {
                        my $constructorType = $attribute->signature->type;
                        $constructorType =~ s/Constructor$//;
                        $implIncludes{"JS" . $constructorType . ".h"} = 1;
                        push(@implContent, "        // Shadowing a built-in constructor\n");
                        push(@implContent, "        JSObject::put(exec, \"$name\", value);\n");
                    } elsif ($attribute->signature->extendedAttributes->{"Replaceable"}) {
                        push(@implContent, "        JSObject::put(exec, \"$name\", value);\n");
                    } else {
                        if ($podType) {
                            push(@implContent, "        $podType imp(*impl());\n\n");
                            if ($podType eq "float") { # Special case for JSSVGNumber
                                push(@implContent, "        imp = " . JSValueToNative($attribute->signature, "value") . ";\n");
                            } else {
                                push(@implContent, "        imp.set" . WK_ucfirst($name) . "(" . JSValueToNative($attribute->signature, "value") . ");\n");
                            }
                            push(@implContent, "        m_impl->commitChange(exec, imp);\n");
                        } else {
                            push(@implContent, "        $implClassName* imp = static_cast<$implClassName*>(impl());\n\n");
                            push(@implContent, "        ExceptionCode ec = 0;\n") if @{$attribute->setterExceptions};
                            push(@implContent, "        imp->set" . WK_ucfirst($name) . "(" . JSValueToNative($attribute->signature, "value"));
                            push(@implContent, ", ec") if @{$attribute->setterExceptions};
                            push(@implContent, ");\n");
                            push(@implContent, "        setDOMException(exec, ec);\n") if @{$attribute->setterExceptions};
                        }
                    }
                    push(@implContent, "        break;\n");
                    push(@implContent, "    }\n");
                }
            }
            push(@implContent, "    }\n"); # end switch

            if (IsSVGTypeNeedingContextParameter($implClassName)) {
                push(@implContent, "    if (context())\n");
                push(@implContent, "        context()->notifyAttributeChange();\n");
            }

            push(@implContent, "}\n\n"); # end function
        }
    }

    if ($dataNode->extendedAttributes->{"GenerateConstructor"}) {
        push(@implContent, "JSValue* ${className}::getConstructor(ExecState* exec)\n{\n");
        push(@implContent, "    return KJS::cacheGlobalObject<${className}Constructor>(exec, \"[[${interfaceName}.constructor]]\");\n");
        push(@implContent, "}\n");
    }

    # Functions
    if ($numFunctions > 0) {
        foreach my $function (@{$dataNode->functions}) {
            my $functionClassName = $className . "PrototypeFunction" . WK_ucfirst($function->signature->name);
            push(@implContent, "JSValue* ${functionClassName}::callAsFunction(ExecState* exec, JSObject* thisObj, const List& args)\n");
            push(@implContent, "{\n");
            push(@implContent, "    if (!thisObj->inherits(&${className}::info))\n");
            push(@implContent, "      return throwError(exec, TypeError);\n\n");

            AddIncludesForType($function->signature->type);

            push(@implContent, "    $className* castedThisObj = static_cast<$className*>(thisObj);\n");

            if ($function->signature->extendedAttributes->{"Custom"}) {
                push(@implContent, "        return castedThisObj->" . $function->signature->name . "(exec, args);\n");
            } else {
                if ($podType) {
                    push(@implContent, "    JSSVGPODTypeWrapper<$podType>* wrapper = castedThisObj->impl();\n");
                    push(@implContent, "    $podType imp(*wrapper);\n\n");
                } else {
                    push(@implContent, "    $implClassName* imp = static_cast<$implClassName*>(castedThisObj->impl());\n\n");
                }


                if (@{$function->raisesExceptions}) {
                    push(@implContent, "    ExceptionCode ec = 0;\n");
                }

                if ($function->signature->extendedAttributes->{"SVGCheckSecurityDocument"}) {
                    push(@implContent, "    if (!checkNodeSecurity(exec, imp->getSVGDocument(" . (@{$function->raisesExceptions} ? "ec" : "") .")))\n");
                    push(@implContent, "        return jsUndefined();\n");
                    $implIncludes{"kjs_dom.h"} = 1;
                }

                my $paramIndex = 0;
                my $functionString = "imp" . ($podType ? "." : "->") . $function->signature->name . "(";

                my $numParameters = @{$function->parameters};
                my $hasOptionalArguments = 0;

                foreach my $parameter (@{$function->parameters}) {
                    if (!$hasOptionalArguments && $parameter->extendedAttributes->{"Optional"}) {
                        push(@implContent, "\n    int argsCount = args.size();\n");
                        $hasOptionalArguments = 1;
                    }

                    if ($hasOptionalArguments) {
                        push(@implContent, "    if (argsCount < " . ($paramIndex + 1) . ") {\n");
                        GenerateImplementationFunctionCall($function, $functionString, $paramIndex, "    " x 2, $podType, $implClassName);
                        push(@implContent, "    }\n\n");
                    }

                    my $name = $parameter->name;
                    
                    if ($parameter->type eq "XPathNSResolver") {
                        push(@implContent, "    RefPtr<XPathNSResolver> customResolver;\n");
                        push(@implContent, "    XPathNSResolver* resolver = toXPathNSResolver(args[$paramIndex]);\n");
                        push(@implContent, "    if (!resolver) {\n");
                        push(@implContent, "        customResolver = JSCustomXPathNSResolver::create(exec, args[$paramIndex]);\n");
                        push(@implContent, "        if (exec->hadException())\n");
                        push(@implContent, "            return jsUndefined();\n");
                        push(@implContent, "        resolver = customResolver.get();\n");
                        push(@implContent, "    }\n");
                    } else {
                        push(@implContent, "    bool ${name}Ok;\n") if TypeCanFailConversion($parameter);
                        push(@implContent, "    " . GetNativeTypeFromSignature($parameter) . " $name = " . JSValueToNative($parameter, "args[$paramIndex]", TypeCanFailConversion($parameter) ? "${name}Ok" : undef) . ";\n");
                        if (TypeCanFailConversion($parameter)) {
                            push(@implContent, "    if (!${name}Ok) {\n");
                            push(@implContent, "        setDOMException(exec, TYPE_MISMATCH_ERR);\n");
                            push(@implContent, "        return jsUndefined();\n        }\n");
                        }

                        # If a parameter is "an index", it should throw an INDEX_SIZE_ERR
                        # exception
                        if ($parameter->extendedAttributes->{"IsIndex"}) {
                            $implIncludes{"ExceptionCode.h"} = 1;
                            push(@implContent, "    if ($name < 0) {\n");
                            push(@implContent, "        setDOMException(exec, INDEX_SIZE_ERR);\n");
                            push(@implContent, "        return jsUndefined();\n");
                            push(@implContent, "    }\n");
                        }
                    }

                    $functionString .= ", " if $paramIndex;
                    $functionString .= $name;

                    $paramIndex++;
                }

                push(@implContent, "\n");
                GenerateImplementationFunctionCall($function, $functionString, $paramIndex, "    ", $podType, $implClassName);
            }
            push(@implContent, "}\n\n");
        }
    }

    if ($dataNode->extendedAttributes->{"HasIndexGetter"}) {
        push(@implContent, "\nJSValue* ${className}::indexGetter(ExecState* exec, JSObject* originalObject, const Identifier& propertyName, const PropertySlot& slot)\n");
        push(@implContent, "{\n");
        push(@implContent, "    ${className}* thisObj = static_cast<$className*>(slot.slotBase());\n");
        if (IndexGetterReturnsStrings($implClassName)) {
            $implIncludes{"PlatformString.h"} = 1;
            push(@implContent, "    return jsStringOrNull(thisObj->impl()->item(slot.index()));\n");
        } else {
            push(@implContent, "    return toJS(exec, static_cast<$implClassName*>(thisObj->impl())->item(slot.index()));\n");
        }
        push(@implContent, "}\n");
        if ($interfaceName eq "HTMLCollection") {
            $implIncludes{"JSNode.h"} = 1;
            $implIncludes{"Node.h"} = 1;
        }
    }

    if ((!$hasParent or $dataNode->extendedAttributes->{"GenerateToJS"}) and !UsesManualToJSImplementation($implClassName)) {
        if ($podType) {
            push(@implContent, "KJS::JSValue* toJS(KJS::ExecState* exec, JSSVGPODTypeWrapper<$podType>* obj, SVGElement* context)\n");
        } elsif (IsSVGTypeNeedingContextParameter($implClassName)) {
             push(@implContent, "KJS::JSValue* toJS(KJS::ExecState* exec, $passType obj, SVGElement* context)\n");
        } else {
            push(@implContent, "KJS::JSValue* toJS(KJS::ExecState* exec, $passType obj)\n");
        }

        push(@implContent, "{\n");
        if ($podType) {
            push(@implContent, "    return KJS::cacheSVGDOMObject<JSSVGPODTypeWrapper<$podType>, $className>(exec, obj, context);\n");
        } elsif (IsSVGTypeNeedingContextParameter($implClassName)) {
            push(@implContent, "    return KJS::cacheSVGDOMObject<$implClassName, $className>(exec, obj, context);\n");
        } else {
            push(@implContent, "    return KJS::cacheDOMObject<$implClassName, $className>(exec, obj);\n");
        }
        push(@implContent, "}\n");
    }

    if ((!$hasParent or $dataNode->extendedAttributes->{"GenerateNativeConverter"}) and !$dataNode->extendedAttributes->{"CustomNativeConverter"}) {
        if ($podType) {
            push(@implContent, "$podType to${interfaceName}(KJS::JSValue* val)\n");
        } else {
            push(@implContent, "$implClassName* to${interfaceName}(KJS::JSValue* val)\n");
        }

        push(@implContent, "{\n");

        push(@implContent, "    return val->isObject(&${className}::info) ? " . ($podType ? "($podType) *" : "") . "static_cast<$className*>(val)->impl() : ");
        if ($podType and $podType ne "float") {
            push(@implContent, "$podType();\n}\n");
        } else {
            push(@implContent, "0;\n}\n");
        }
    }

    if ($dataNode->extendedAttributes->{"GenerateNativeConverter"} && $hasParent) {
        push(@implContent, "\n$implClassName* ${className}::impl() const\n");
        push(@implContent, "{\n");
        push(@implContent, "    return static_cast<$implClassName*>(${parentClassName}::impl());\n");
        push(@implContent, "}\n");
    }

    push(@implContent, "\n}\n");

    push(@implContent, "\n#endif // ${conditionalString}\n") if $conditional;
}

sub GenerateImplementationFunctionCall()
{
    my $function = shift;
    my $functionString = shift;
    my $paramIndex = shift;
    my $indent = shift;
    my $podType = shift;
    my $implClassName = shift;

    if (@{$function->raisesExceptions}) {
        $functionString .= ", " if $paramIndex;
        $functionString .= "ec";
    }
    $functionString .= ")";

    if ($function->signature->type eq "void") {
        push(@implContent, $indent . "$functionString;\n");
        push(@implContent, $indent . "setDOMException(exec, ec);\n") if @{$function->raisesExceptions};

        if ($podType) {
            push(@implContent, $indent . "wrapper->commitChange(exec, imp);\n\n");
            push(@implContent, $indent . "if (castedThisObj->context())\n");
            push(@implContent, $indent . "    castedThisObj->context()->notifyAttributeChange();\n");
        }

        push(@implContent, $indent . "return jsUndefined();\n");
    } else {
        push(@implContent, "\n" . $indent . "KJS::JSValue* result = " . NativeToJSValue($function->signature, 1, $implClassName, "", $functionString) . ";\n");
        push(@implContent, $indent . "setDOMException(exec, ec);\n") if @{$function->raisesExceptions};

        if ($podType) {
            push(@implContent, $indent . "wrapper->commitChange(exec, imp);\n\n");
            push(@implContent, $indent . "if (castedThisObj->context())\n");
            push(@implContent, $indent . "    castedThisObj->context()->notifyAttributeChange();\n");
        }

        push(@implContent, $indent . "return result;\n");
    }
}

sub GetNativeTypeFromSignature
{
    my $signature = shift;
    my $type = $codeGenerator->StripModule($signature->type);

    if ($type eq "unsigned long" and $signature->extendedAttributes->{"IsIndex"}) {
        # Special-case index arguments because we need to check that they aren't < 0.
        return "int";
    }

    return GetNativeType($type);
}

sub GetNativeType
{
    my $type = shift;

    return "unsigned" if $type eq "unsigned long";
    return $type if $type eq "unsigned short" or $type eq "float" or $type eq "double" or $type eq "AtomicString";
    return "bool" if $type eq "boolean";
    return "int" if $type eq "long";
    return "String" if $type eq "DOMString";
    return "Range::CompareHow" if $type eq "CompareHow";
    return "EventTargetNode*" if $type eq "EventTarget";
    return "FloatRect" if $type eq "SVGRect";
    return "FloatPoint" if $type eq "SVGPoint";
    return "AffineTransform" if $type eq "SVGMatrix";
    return "SVGTransform" if $type eq "SVGTransform";
    return "SVGLength" if $type eq "SVGLength";
    return "float" if $type eq "SVGNumber";
    return "SVGPaint::SVGPaintType" if $type eq "SVGPaintType";

    # Default, assume native type is a pointer with same type name as idl type
    return "${type}*";
}

sub TypeCanFailConversion
{
    my $signature = shift;

    my $type = $codeGenerator->StripModule($signature->type);

    # FIXME: convert to use a hash

    return 0 if $type eq "boolean" or
                $type eq "float" or
                $type eq "double" or
                $type eq "AtomicString" or
                $type eq "DOMString" or
                $type eq "Node" or
                $type eq "Element" or
                $type eq "DocumentType" or
                $type eq "Event" or
                $type eq "EventListener" or
                $type eq "EventTarget" or
                $type eq "Range" or
                $type eq "NodeFilter" or
                $type eq "DOMWindow" or
                $type eq "SQLResultSet" or
                $type eq "XPathEvaluator" or
                $type eq "XPathNSResolver" or
                $type eq "XPathResult" or
                $type eq "SVGAngle" or
                $type eq "SVGLength" or
                $type eq "SVGNumber" or
                $type eq "SVGPoint" or
                $type eq "SVGTransform" or
                $type eq "SVGPathSeg" or
                $type eq "SVGMatrix" or
                $type eq "SVGRect" or
                $type eq "SVGElement" or
                $type eq "HTMLElement" or
                $type eq "HTMLOptionElement" or
                $type eq "unsigned short" or # or can it?
                $type eq "CompareHow" or # or can it?
                $type eq "SVGPaintType" or # or can it?
                $type eq "VoidCallback";

    if ($type eq "unsigned long" or $type eq "long" or $type eq "Attr") {
        $implIncludes{"ExceptionCode.h"} = 1;
        return 1;
    }

    die "Don't know whether a JS value can fail conversion to type $type."
}

sub JSValueToNative
{
    my $signature = shift;
    my $value = shift;
    my $okParam = shift;
    my $maybeOkParam = $okParam ? ", ${okParam}" : "";

    my $type = $codeGenerator->StripModule($signature->type);

    return "$value->toBoolean(exec)" if $type eq "boolean";
    return "$value->toNumber(exec)" if $type eq "double";
    return "$value->toFloat(exec)" if $type eq "float" or $type eq "SVGNumber";
    return "$value->toInt32(exec${maybeOkParam})" if $type eq "unsigned long" or $type eq "long" or $type eq "unsigned short";

    return "static_cast<Range::CompareHow>($value->toInt32(exec))" if $type eq "CompareHow";
    return "static_cast<SVGPaint::SVGPaintType>($value->toInt32(exec))" if $type eq "SVGPaintType";

    return "$value->toString(exec)" if $type eq "AtomicString";
    if ($type eq "DOMString") {
        return "valueToStringWithNullCheck(exec, $value)" if $signature->extendedAttributes->{"ConvertNullToNullString"};
        return "valueToStringWithUndefinedOrNullCheck(exec, $value)" if $signature->extendedAttributes->{"ConvertUndefinedOrNullToNullString"};
        return "$value->toString(exec)";
    }

    if ($type eq "EventTarget") {
        $implIncludes{"JSEventTargetNode.h"} = 1;
        return "toEventTargetNode($value)";
    }

    if ($type eq "Attr") {
        $implIncludes{"kjs_dom.h"} = 1;
        return "toAttr($value${maybeOkParam})";
    }

    if ($type eq "SVGRect") {
        $implIncludes{"FloatRect.h"} = 1;
    }

    if ($type eq "SVGPoint") {
        $implIncludes{"FloatPoint.h"} = 1;
    }

    if ($type eq "VoidCallback") {
        $implIncludes{"VoidCallback.h"} = 1;
        return "toVoidCallback($value)";
    }

    # Default, assume autogenerated type conversion routines
    $implIncludes{"JS$type.h"} = 1;
    return "to$type($value)";
}

sub NativeToJSValue
{
    my $signature = shift;
    my $inFunctionCall = shift;
    my $implClassName = shift;
    my $implClassNameForValueConversion = shift;
    my $value = shift;

    my $type = $codeGenerator->StripModule($signature->type);

    return "jsBoolean($value)" if $type eq "boolean";
    return "jsNumber($value)" if $codeGenerator->IsPrimitiveType($type) or $type eq "SVGPaintType" or $type eq "DOMTimeStamp";

    if ($codeGenerator->IsStringType($type)) {
        $implIncludes{"PlatformString.h"} = 1;
        my $conv = $signature->extendedAttributes->{"ConvertNullStringTo"};
        if (defined $conv) {
            return "jsStringOrNull($value)" if $conv eq "Null";
            return "jsStringOrUndefined($value)" if $conv eq "Undefined";
            return "jsStringOrFalse($value)" if $conv eq "False";

            die "Unknown value for ConvertNullStringTo extended attribute";
        }
        return "jsString($value)";
    }

    if ($type eq "RGBColor") {
        $implIncludes{"kjs_css.h"} = 1;
        return "getJSRGBColor(exec, $value)";
    }

    if ($codeGenerator->IsPodType($type)) {
        $implIncludes{"JS$type.h"} = 1;

        my $nativeType = GetNativeType($type);

        my $getter = $value;
        $getter =~ s/imp->//;
        $getter =~ s/\(\)//;

        my $setter = "set" . WK_ucfirst($getter);

        if ($implClassNameForValueConversion eq "") {
            if (IsSVGTypeNeedingContextParameter($implClassName)) {
                return "toJS(exec, new JSSVGPODTypeWrapperCreatorReadOnly<$nativeType>($value), castedThisObj->context())" if $inFunctionCall eq 1;

                # Special case: SVGZoomEvent - it doesn't have a context, but it's no problem, as there are no readwrite props
                return "toJS(exec, new JSSVGPODTypeWrapperCreatorReadOnly<$nativeType>($value), 0)" if $implClassName eq "SVGZoomEvent";
                return "toJS(exec, new JSSVGPODTypeWrapperCreatorReadOnly<$nativeType>($value), context())";
            } else {
                return "toJS(exec, new JSSVGPODTypeWrapperCreatorReadOnly<$nativeType>($value), imp)";
            }
        } else { # These classes, always have a m_context pointer!
            return "toJS(exec, JSSVGPODTypeWrapperCache<$nativeType, $implClassNameForValueConversion>::lookupOrCreateWrapper(imp, &${implClassNameForValueConversion}::$getter, &${implClassNameForValueConversion}::$setter), context())";
        }
    }

    if ($codeGenerator->IsSVGAnimatedType($type)) {
        $value =~ s/\(\)//;
        $value .= "Animated()";
    }

    if ($type eq "CSSStyleDeclaration") {
        $implIncludes{"CSSMutableStyleDeclaration.h"} = 1;
    }

    if ($type eq "NamedNodeMap") {
        $implIncludes{"NamedAttrMap.h"} = 1;
    }

    if ($type eq "NodeList") {
        $implIncludes{"NameNodeList.h"} = 1;
    }

    if ($type eq "EventTarget") {
        $implIncludes{"EventTargetNode.h"} = 1;
        $implIncludes{"JSEventTargetNode.h"} = 1;
        $implIncludes{"kjs_dom.h"} = 1;
    } elsif ($type eq "DOMWindow") {
        $implIncludes{"kjs_window.h"} = 1;
    } elsif ($type eq "DOMObject") {
        $implIncludes{"JSCanvasRenderingContext2D.h"} = 1;
    } elsif ($type eq "Clipboard") {
        $implIncludes{"kjs_events.h"} = 1;
        $implIncludes{"Clipboard.h"} = 1;
    } elsif ($type =~ /SVGPathSeg/) {
        $implIncludes{"JS$type.h"} = 1;
        $joinedName = $type;
        $joinedName =~ s/Abs|Rel//;
        $implIncludes{"$joinedName.h"} = 1;
    } else {
        # Default, include header with same name.
        $implIncludes{"JS$type.h"} = 1;
        $implIncludes{"$type.h"} = 1;
    }

    return $value if $codeGenerator->IsSVGAnimatedType($type);

    if (IsSVGTypeNeedingContextParameter($type)) {
        if (IsSVGTypeNeedingContextParameter($implClassName)) {
            if ($inFunctionCall eq 1) {
                return "toJS(exec, WTF::getPtr($value), castedThisObj->context())";
            } else {
                return "toJS(exec, WTF::getPtr($value), context())";
            }
        } else {
            return "toJS(exec, WTF::getPtr($value), imp)";
        }
    }

    return "toJS(exec, WTF::getPtr($value))";
}

sub ceilingToPowerOf2
{
    my ($size) = @_;

    my $powerOf2 = 1;
    while ($size > $powerOf2) {
        $powerOf2 <<= 1;
    }

    return $powerOf2;
}

# Internal Helper
sub GenerateHashTable
{
    my $object = shift;

    my $name = shift;
    my $size = shift;
    my $keys = shift;
    my $values = shift;
    my $specials = shift;
    my $parameters = shift;

    # Helpers
    my @table = ();
    my @links = ();

    $size = ceilingToPowerOf2($size * 2);

    my $maxDepth = 0;
    my $collisions = 0;
    my $numEntries = $size;

    # Collect hashtable information
    my $i = 0;
    foreach (@{$keys}) {
        my $depth = 0;
        my $h = $object->GenerateHashValue($_) % $numEntries;

        while (defined($table[$h])) {
            if (defined($links[$h])) {
                $h = $links[$h];
                $depth++;
            } else {
                $collisions++;
                $links[$h] = $size;
                $h = $size;
                $size++;
            }
        }

        $table[$h] = $i;

        $i++;
        $maxDepth = $depth if ($depth > $maxDepth);
    }

    # Ensure table is big enough (in case of undef entries at the end)
    if ($#table + 1 < $size) {
        $#table = $size - 1;
    }

    # Start outputing the hashtables
    my $nameEntries = "${name}Entries";
    $nameEntries =~ s/:/_/g;

    if (($name =~ /Prototype/) or ($name =~ /Constructor/)) {
        my $type = $name;
        my $implClass;

        if ($name =~ /Prototype/) {
            $type =~ s/Prototype.*//;
            $implClass = $type; $implClass =~ s/Wrapper$//;
            push(@implContent, "/* Hash table for prototype */\n");
        } else {
            $type =~ s/Constructor.*//;
            $implClass = $type; $implClass =~ s/Constructor$//;
            push(@implContent, "/* Hash table for constructor */\n");
        }
    } else {
        push(@implContent, "/* Hash table */\n");
    }

    # Dump the hash table
    push(@implContent, "\nstatic const HashEntry $nameEntries\[\] =\n\{\n");

    $i = 0;
    foreach $entry (@table) {
        if (defined($entry)) {
            my $key = @$keys[$entry];

            push(@implContent, "    \{ \"" . $key . "\"");
            push(@implContent, ", \{ (intptr_t)" . @$values[$entry] . " \}");
            push(@implContent, ", " . @$specials[$entry]);
            push(@implContent, ", " . @$parameters[$entry]);
            push(@implContent, ", ");

            if (defined($links[$i])) {
                push(@implContent, "&" . $nameEntries . "[$links[$i]]" . " \}");
            } else {
                push(@implContent, "0 \}");
            }
        } else {
            push(@implContent, "    { 0, { 0 }, 0, 0, 0 }");
        }

        push(@implContent, ",") unless($i eq $size - 1);
        push(@implContent, "\n");

        $i++;
    }

    my $sizeMask = $numEntries - 1;

    push(@implContent, "};\n\n");
    push(@implContent, "static const HashTable $name = \n");
    push(@implContent, "{\n    3, $size, $nameEntries, $sizeMask\n};\n\n");
}

# Internal helper
sub GenerateHashValue
{
    my $object = shift;

    @chars = split(/ */, $_[0]);

    # This hash is designed to work on 16-bit chunks at a time. But since the normal case
    # (above) is to hash UTF-16 characters, we just treat the 8-bit chars as if they
    # were 16-bit chunks, which should give matching results

    my $EXP2_32 = 4294967296;

    my $hash = 0x9e3779b9;
    my $l    = scalar @chars; #I wish this was in Ruby --- Maks
    my $rem  = $l & 1;
    $l = $l >> 1;

    my $s = 0;

    # Main loop
    for (; $l > 0; $l--) {
        $hash   += ord($chars[$s]);
        my $tmp = leftShift(ord($chars[$s+1]), 11) ^ $hash;
        $hash   = (leftShift($hash, 16)% $EXP2_32) ^ $tmp;
        $s += 2;
        $hash += $hash >> 11;
        $hash %= $EXP2_32;
    }

    # Handle end case
    if ($rem != 0) {
        $hash += ord($chars[$s]);
        $hash ^= (leftShift($hash, 11)% $EXP2_32);
        $hash += $hash >> 17;
    }

    # Force "avalanching" of final 127 bits
    $hash ^= leftShift($hash, 3);
    $hash += ($hash >> 5);
    $hash = ($hash% $EXP2_32);
    $hash ^= (leftShift($hash, 2)% $EXP2_32);
    $hash += ($hash >> 15);
    $hash = $hash% $EXP2_32;
    $hash ^= (leftShift($hash, 10)% $EXP2_32);

    # this avoids ever returning a hash code of 0, since that is used to
    # signal "hash not computed yet", using a value that is likely to be
    # effectively the same as 0 when the low bits are masked
    $hash = 0x80000000 if ($hash == 0);

    return $hash;
}

# Internal helper
sub WriteData
{
    if (defined($IMPL)) {
        # Write content to file.
        print $IMPL @implContentHeader;

        foreach my $implInclude (sort keys(%implIncludes)) {
            my $checkType = $implInclude;
            $checkType =~ s/\.h//;

            print $IMPL "#include \"$implInclude\"\n" unless $codeGenerator->IsSVGAnimatedType($checkType);
        }

        print $IMPL @implContent;
        close($IMPL);
        undef($IMPL);

        @implHeaderContent = ();
        @implContent = ();
        %implIncludes = ();
    }

    if (defined($HEADER)) {
        # Write content to file.
        print $HEADER @headerContent;
        close($HEADER);
        undef($HEADER);

        @headerContent = ();
    }
}

sub constructorFor
{
    my $className = shift;
    my $protoClassName = shift;
    my $interfaceName = shift;
    my $canConstruct = shift;

my $implContent = << "EOF";
class ${className}Constructor : public DOMObject {
public:
    ${className}Constructor(ExecState* exec)
    {
        setPrototype(exec->lexicalInterpreter()->builtinObjectPrototype());
        putDirect(exec->propertyNames().prototype, ${protoClassName}::self(exec), None);
    }
    virtual bool getOwnPropertySlot(ExecState*, const Identifier&, PropertySlot&);
    JSValue* getValueProperty(ExecState*, int token) const;
    virtual const ClassInfo* classInfo() const { return &info; }
    static const ClassInfo info;

    virtual bool implementsHasInstance() const { return true; }
EOF

    if ($canConstruct) {
$implContent .= << "EOF";
    virtual bool implementsConstruct() const { return true; }
    virtual JSObject* construct(ExecState* exec, const List& args) { return static_cast<JSObject*>(toJS(exec, new $interfaceName)); }
EOF
    }

$implContent .= << "EOF";
};

const ClassInfo ${className}Constructor::info = { "${interfaceName}Constructor", 0, &${className}ConstructorTable };

bool ${className}Constructor::getOwnPropertySlot(ExecState* exec, const Identifier& propertyName, PropertySlot& slot)
{
    return getStaticValueSlot<${className}Constructor, DOMObject>(exec, &${className}ConstructorTable, this, propertyName, slot);
}

JSValue* ${className}Constructor::getValueProperty(ExecState*, int token) const
{
    // The token is the numeric value of its associated constant
    return jsNumber(token);
}

EOF

    return $implContent;
}

sub prototypeFunctionFor
{
    my ($className, $functionName) = @_;
    my $name = $className . "PrototypeFunction" . $functionName;

my $implContent = << "EOF";
class ${name} : public KJS::InternalFunctionImp {
public: \
    static KJS::InternalFunctionImp* create(KJS::ExecState* exec, int len, const KJS::Identifier& name)
    {
        return new ${name}(exec, len, name);
    }

    ${name}(KJS::ExecState* exec, int len, const KJS::Identifier& name)
        : KJS::InternalFunctionImp(static_cast<KJS::FunctionPrototype*>(exec->lexicalInterpreter()->builtinFunctionPrototype()), name)
    {
        put(exec, exec->propertyNames().length, KJS::jsNumber(len), KJS::DontDelete | KJS::ReadOnly | KJS::DontEnum);
    }

    virtual KJS::JSValue* callAsFunction(KJS::ExecState*, KJS::JSObject*, const KJS::List&);
};

EOF

    return $implContent;
}

1;
