/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#ifndef _FILTER_H
#define _FILTER_H

#include <string>
#include <iostream>

#include "wflogentry.h"
#include "wfipaddr.h"
#include "wfnetwork.h"
#include "wfmacaddr.h"

enum vars_num {
  VARS_UNKNOWN = -1,

  VARS_FORMAT,
  VARS_COUNT,
  VARS_START_TIME,
  VARS_END_TIME,
  VARS_HOSTNAME,
  VARS_CHAINLABEL,
  VARS_BRANCHNAME,
  VARS_INPUT_IFACE,
  VARS_OUTPUT_IFACE,
  VARS_PROTOCOL,
  VARS_DATALEN,
  VARS_SIPADDR,
  VARS_SPORT,
  VARS_SMACADDR,
  VARS_DIPADDR,
  VARS_DPORT,
  VARS_DMACADDR,
  VARS_TCPFLAGS
};

enum filter_op_type { FOP_TYPE_UNARY, FOP_TYPE_BINARY, FOP_TYPE_VAR };

/* Order must match the one of ops_unary array. */
enum filter_op_unary { FOP_NEG, FOP_NOT, FOP_POS };

/* Order must match the one of ops_binary array. */
enum filter_op_binary { FOP_LOR, FOP_LAND, FOP_EQ, FOP_NE, FOP_REG,
			FOP_LT, FOP_GT, FOP_LE, FOP_GE,
			FOP_XOR, FOP_OR, FOP_AND, FOP_PLUS, FOP_MINUS };

enum filter_op_var_type { FOP_VAR_TYPE_ID, FOP_VAR_TYPE_INT,
			  FOP_VAR_TYPE_STR, FOP_VAR_TYPE_REGEXP,
			  FOP_VAR_TYPE_IPNET, FOP_VAR_TYPE_MACADDR };

/**
 * 
 */
class exprnode {
 public:
  virtual enum filter_op_type node_type() const = 0;
  virtual enum filter_op_var_type ret_type() const = 0;
  virtual bool check() const = 0;
  virtual ostream& print(ostream& os) const = 0;

  virtual int eval() const = 0;
};

class exprnode_unary : public exprnode {
 public:
  exprnode_unary(enum filter_op_unary op, exprnode* node);
  virtual ~exprnode_unary();
  exprnode_unary(const exprnode_unary& unary);

  virtual enum filter_op_type node_type() const { return FOP_TYPE_UNARY; }
  virtual enum filter_op_var_type ret_type() const { return FOP_VAR_TYPE_INT; }
  virtual bool check() const;
  virtual ostream& print(ostream& os) const;
  virtual int eval() const;

 private:
  enum filter_op_unary _op;
  exprnode* _child;
};

class exprnode_binary : public exprnode {
 public:
  exprnode_binary(enum filter_op_binary op, exprnode* lft, exprnode* rt);
  virtual ~exprnode_binary();
  exprnode_binary(const exprnode_binary& binary);

  virtual enum filter_op_type node_type() const { return FOP_TYPE_BINARY; }
  virtual enum filter_op_var_type ret_type() const { return FOP_VAR_TYPE_INT; }
  virtual bool check() const;
  virtual ostream& print(ostream& os) const;
  virtual int eval() const;

 private:
  enum filter_op_binary _op;
  exprnode* _left;  // left subtree
  exprnode* _right; // right subtree
};

class exprnode_var : public exprnode {
 public:
  virtual enum filter_op_type node_type() const { return FOP_TYPE_VAR; }
  virtual enum filter_op_var_type ret_type() const = 0;
  virtual bool check() const = 0;

  virtual int eval() const; /* dummy method */
  virtual string str() const; /* dummy method */
  virtual wf_ipaddr ipaddr() const; /* dummy method */
  virtual wf_macaddr macaddr() const; /* dummy method */
};


class exprnode_var_id : public exprnode_var {
 public:
  exprnode_var_id(const string& name);
  virtual ~exprnode_var_id();

  virtual enum filter_op_var_type ret_type() const;
  virtual bool check() const;
  virtual ostream& print(ostream& os) const;

  enum vars_num num() const;

  virtual int eval() const = 0; /* called if type is INT */
  virtual string str() const = 0; /* called if type is STR */
  virtual wf_ipaddr ipaddr() const = 0; /* called if type is IPNUM */
  virtual wf_macaddr macaddr() const = 0; /* called if type is MACADDR */

 protected:
  enum vars_num _num;
};

/* We need to inherit from exprnode_var_id in order to define adequate
   evaluation methods for each member of the wflogs class. */
class exprnode_var_id_wflogs : public exprnode_var_id {
 public:
  exprnode_var_id_wflogs(const string& name);
  // virtual ~exprnode_var_id_wflogs();

  virtual int eval() const; /* called if type is INT */
  virtual string str() const; /* called if type is STR */
  virtual wf_ipaddr ipaddr() const; /* called if type is IPNUM */
  virtual wf_macaddr macaddr() const; /* called if type is MACADDR */
};

/* We need to inherit from exprnode in order to store a pointer to the
   wf_logentry object which contains the values. */
class exprnode_wflogs : public exprnode {
 public:
  //  exprnode_wflogs(const string& name);
  // virtual ~exprnode_wflogs();
  exprnode_wflogs(const exprnode_wflogs& wflogs);

 private:
  wf_logentry* _entry;
};

class exprnode_var_int : public exprnode_var {
 public:
  exprnode_var_int(int val);
  virtual ~exprnode_var_int();

  virtual enum filter_op_var_type ret_type() const { return FOP_VAR_TYPE_INT; }
  virtual bool check() const { return true; }
  virtual ostream& print(ostream& os) const;

  virtual int eval() const;

  void value(int val);

 private:
  int _value;
};

class exprnode_var_str : public exprnode_var {
 public:
  exprnode_var_str(const string& str);
  virtual ~exprnode_var_str();

  virtual enum filter_op_var_type ret_type() const { return FOP_VAR_TYPE_STR; }
  virtual bool check() const { return true; }
  virtual ostream& print(ostream& os) const;

  virtual string str() const;

 private:
  string _str;
};

#include <regex.h>

class exprnode_var_regexp : public exprnode_var {
 public:
  exprnode_var_regexp();
  virtual ~exprnode_var_regexp();

  virtual enum filter_op_var_type ret_type() const { return FOP_VAR_TYPE_REGEXP; }
  virtual bool check() const { return true; }
  virtual ostream& print(ostream& os) const;

  bool set(const string& str, int flags = 0);
  bool regmatch(const string& str) const;
  
 private:
  regex_t _regexp;
  string _str; /* for displaying purpose */
};

class exprnode_var_ipnet : public exprnode_var {
 public:
  exprnode_var_ipnet(const wf_ipaddr& ipaddr);
  exprnode_var_ipnet(const wf_network& ipnet);
  virtual ~exprnode_var_ipnet() {}

  virtual enum filter_op_var_type ret_type() const { return FOP_VAR_TYPE_IPNET; }
  virtual bool check() const { return true; }
  virtual ostream& print(ostream& os) const;

  bool belong(const wf_ipaddr& ipaddr) const;

 public:  // RV@@9 should be private
  wf_network _network;
};

class exprnode_var_macaddr : public exprnode_var {
 public:
  exprnode_var_macaddr(const wf_macaddr& macaddr);
  virtual ~exprnode_var_macaddr() {}

  virtual enum filter_op_var_type ret_type() const { return FOP_VAR_TYPE_MACADDR; }
  virtual bool check() const { return true; }
  virtual ostream& print(ostream& os) const;

 public:  // RV@@9 should be private
  wf_macaddr _macaddr;
};

class exprnode_var_date : public exprnode_var {
 public:
  exprnode_var_date();
  virtual ~exprnode_var_date();

  virtual enum filter_op_var_type ret_type() const { return FOP_VAR_TYPE_INT; }
  virtual bool check() const { return true; }
  virtual int eval() const;
  bool set(const string& str);
  virtual ostream& print(ostream& os) const;

 private:
  time_t _time;
};

#endif
