/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001-2002 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sstream>

#include "wflogentry.h"
#include "defs.h"


struct tcpflags {
  int flag;
  string str;
  char letter;
};

static struct tcpflags mytcpflags[] = {
  { TCP_SYN, "SYN", 's' },
  { TCP_ACK, "ACK", 'a' },
  { TCP_FIN, "FIN", 'f' },
  { TCP_RST, "RST", 'r' },
  { TCP_PSH, "PSH", 'p' },
  { TCP_URG, "URG", 'u' },
  { TCP_ECE, "ECE", 'e' },
  { TCP_CWR, "CWR", 'c' }
};

wf_logentry::wf_logentry() :
  format(),
  count(0),
  start_time(0),
  end_time(0),
  hostname(),
  chainlabel(),
  branchname(),
  input_iface(),
  output_iface(),
  protocol(-1),
  datalen(0),
  sipaddr(),
  sport(0),
  smacaddr(),
  dipaddr(),
  dport(0),
  dmacaddr(),
  tcpflags(0)
{}


bool
wf_logentry::check() const {
  if (count == 0 || start_time == 0 || protocol == -1 ||
      sipaddr.isdefined() == false || dipaddr.isdefined() == false)
    return false; /* uninitialised value */

  if (end_time != 0) {
    if (count == 1)
      return false; /* end_time should not be defined for only one packet */

    if (start_time > end_time)
      return false; /* wrong time order */
  }

  /* this should be part of each module.
     We should make a check() method for each input module RV@@9 */
  if (format != "cisco_pix" && format != "cisco_ios" && format != "snort" &&
      input_iface.empty() && output_iface.empty())
    return false; /* no interface information at all */


#if 0
  /* Forged packets can sometimes have at least a port num set to 0 ALL@@7.
     We should find another way to identify that ports have been parsed, but
     are null anyway. */
  if (protocol == IPPROTO_UDP || protocol == IPPROTO_TCP) {
    if (sport == 0 || dport == 0)
      return false; /* port missing */
  }
#endif

  if (protocol != IPPROTO_TCP && tcpflags != 0)
    return false; /* bizarre */

  return true; /* OK */
}

string
wf_logentry::tcpflags_tostr() const {
  if (protocol != IPPROTO_TCP)
    return "-";

  ostringstream os;
  unsigned int i;

  for (i = 0; i < sizeof(mytcpflags) / sizeof(struct tcpflags); i++) {
    if (tcpflags & mytcpflags[i].flag)
      os << ' ' << mytcpflags[i].str;
  }

  return os.str();
}

string
wf_logentry::tcpflags_tostr_mini() const {
  if (protocol != IPPROTO_TCP)
    return "-";
  
  ostringstream os;
  unsigned int i;

  if (tcpflags) {
    if (tcpflags != TCP_SYN) {
      for (i = 0; i < sizeof(mytcpflags) / sizeof(struct tcpflags); i++) {
	if (tcpflags & mytcpflags[i].flag)
	  os << mytcpflags[i].letter;
	else
	  os << '-';
      }
    }
    else
      os << "SYN";
  }
  else
    os << '-';

  return os.str();
}

ostream&
wf_logentry::debugprint(ostream& os) const {
  os << "Log entry:" << endl;
  os << "format: " << format << endl;
  os << "count: " << count << endl;

  os << "start_time: " << ctime(&start_time);
  if (end_time != 0)
    os << "end_time: " << ctime(&end_time);
  os << "hostname: " << hostname << endl;


  os << "chainlabel: " << chainlabel << endl;
  os << "branchname: " << branchname << endl;

  os << "input interface: " << input_iface << endl;
  os << "output interface: " << output_iface << endl;

  os << "protocol: " << protocol << endl;
  os << "datalen: " << datalen << endl;

  os << "src host: " << sipaddr << endl;
  os << "src port: " << sport << endl;
  os << "src MAC address: " << smacaddr << endl;
  os << "dst host: " << dipaddr << endl;
  os << "dst port: " << dport << endl;
  os << "dst MAC address: " << dmacaddr << endl;

  os << "flags:" << tcpflags_tostr() << endl;
  
  return os;
}
