/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
// written by Karsten Laux, June 1999  

#ifndef _SURFACE_H_
#define _SURFACE_H_

#include <SDL/SDL.h>

#include <wftk/color.h>
#include <wftk/point.h>
#include <wftk/pixelformat.h>
#include <wftk/rect.h>
#include <wftk/resources.h>


namespace wftk {

class Region;

/** A texture to use with Surface::shiftGamma()
 *  The "texture" always has to have a size of 256*256 values !
 */
class GammaFunction
{
 public:
  /// Construct a GammaFunction
  GammaFunction();
  /// Construct a GammaFunction from xpm data
  GammaFunction(char** xpm) : shift_x(0), shift_y(0) {read(xpm);}
  /// Copy a GammaFunction
  GammaFunction(const GammaFunction&);

  /// Copy another GammaFunction into this one
  GammaFunction& operator=(const GammaFunction&);

  /// Read xpm data into a GammaFunction
  void read(char** xpm);

  /// Set the offset of the GammaFunction origin from the target Surface origin
  void setOffset(int x, int y) {offset_x = x & 0xff; offset_y = y & 0xff;}
  /// same thing
  void setOffset(const wftk::Point& p) {setOffset(p.x, p.y);}
  /// Shift the offset of the GammaFunction origin
  void shiftOffset(int x, int y) {offset_x += x & 0xff; offset_y += y & 0xff;}
  /// same thing
  void shiftOffset(const wftk::Point& p) {shiftOffset(p.x, p.y);}

  /**
   * Allows textures to act on areas larger than 256x256, by making
   * each element in the GammaFunction act on more than one pixel.
   * With a shift of 1, each element acts twice, giving an effective
   * width or height of 512. With a shift of 2, each acts 4 times,
   * for an effective width or height of 1024, etc., etc. Uclient
   * uses (or will use, for compatibility with libuta)
   * setBitShift(2, 1).
   **/
  void setBitShift(char x, char y) {shift_x = x; shift_y = y;}

  /// access to GammaFunction internal data
  char operator()(int x, int y) const {return values[(x & 0xff) + ((y & 0xff) << 8)];}
  /// access to GammaFunction internal data
  char& operator()(int x, int y) {return values[(x & 0xff) + ((y & 0xff) << 8)];}

 private:
  friend class Surface;
  // An wftk::GammaFunction acts like a texture which
  // shades the target surface differently at different points.
  // It's a 256x256 tile.
  char getPoint(int x, int y) const {
    // Bitwise & with 0xff gives (mod 256). It's also
    // safe for negative values of the offset.
    x = (shift_x >= 0) ? (x >> shift_x) : (x << -shift_x);
    y = (shift_y >= 0) ? (y >> shift_y) : (y << -shift_y);
    return values[((offset_x + x) & 0xff)
      + (((offset_y + y) & 0xff) << 8)];
  }

  char values[256*256];    // array of 256*256 values 
  unsigned char offset_x;  // offset to use
  unsigned char offset_y;
  char shift_x, shift_y;
};

/// Surface represents a bitmap (texture)
class Surface
{
 public:

  /// surface drawing helper class
  friend class Painter;

  ///create an empty surface
  Surface();
  /// Create a surface of the given size and pixelformat
  Surface(unsigned w, unsigned h, const Pixelformat& pixelformat
	= Pixelformat::ABGR8888);
  /// Copy a surface
  Surface(const Surface& surf);
  /// Create a child surface.
  /**
   * The child surface shares the parent's pixel data.
   * The parent reference is _not_ const.
   **/
  Surface(Surface& parent, const Rect& rect);

  /// set this surface to be a child of the given surface
  void setAsChild(Surface& parent, const Rect& rect);

  /// set the surface (uninitialzed) to a particular width, height, and pixelformat
  void setSurface(unsigned w, unsigned h, const Pixelformat& pixelformat
	= Pixelformat::ABGR8888);

  ~Surface();
  /** Read from included XPM.
   *  Use #include "image.xpm" to compile images into your binaries.
   *  You may then construct your surface by surface.readFromXPM(image_xpm).
   */  
  bool readFromXPM(char** data);
  /** Read from included gimp header file.
   *  This is the alternative to readFromXPM; it uses header files
   *  generated by GIMP.
  */
  bool readFromHeader(unsigned char* header_data, unsigned int w, unsigned int h);
  /** Load surface from a PNG or BMP file.
   *
   * \param filename File to load
   * \return true if surface loaded successfully
   */
  bool readFromFile(const std::string& filename);
  ///Write surface to a png formatted file \a filename.
  bool writeToFile(const std::string& filename);

  /// Copy another surface into this one
  Surface& operator = (const Surface& surf);
  /** Change palette of this surface.
   *  This does not change the pixel data !
  */
  void setPalette(const SDL_Palette* pal);

  /** check for existing pixeldata.
   *  An empty surface results from an unsuccessful file read for example.
  */
  bool empty() const { return sdlSurface_ == 0; }
  /// return surface width
  unsigned width() const { return ((sdlSurface_ == 0) ? 0 : sdlSurface_->w); }
  /// return surface height
  unsigned height() const { return ((sdlSurface_ == 0) ? 0 : sdlSurface_->h); }
  /// return a rectangle the size of the surface
  Rect rect() const {return Rect(0, 0, width(), height());}
  ///return surface pitch 
  Uint16 pitch() const { return ((sdlSurface_ == 0) ? 0 : sdlSurface_->pitch);}

  /// Return the pixels in the region which are at least as opaque as the cutoff
  /**
   * For surfaces with alpha channel, all pixels lying opaque-ward of the
   * cutoff, not including pixels whose alpha value equals the cutoff,
   * are included.
   *
   * For surfaces with a color key, the cutoff is ignored, and all
   * non-color-key pixels are returned.
   *
   * For surfaces without any form of transparency, returns a rectangle
   * equal to the size of the surface.
   **/
  Region opaqueRegion(Uint8 alpha_cutoff = (SDL_ALPHA_OPAQUE ? 127 : 128)) const;

  ///return pixelformat of this surface
  Pixelformat pixelformat() const {return Pixelformat(sdlSurface_);}

  /**Convert surface into given pixelformat.
   * Since version 0.3.37 a instance of Pixelformat is passed; by this
   * the palette of the target format can be set !
   * Per default floyd-steinberg dithering is used.
   */
  bool convert(const Pixelformat& pixelformat, bool dither = true);
  
  ///mirror this surface horizontally
  bool mirror();
  ///scale this surface to the new size
  bool scale(unsigned new_width, unsigned new_height);
  ///scale this surface by given factor
  bool scale(float n);

  /// lighten (g > 0) or darken (g < 0) the surface
  void gammaShift(char g);
  /// lighten or darken the surface using a texture
  void gammaShift(const GammaFunction &gamma);

  /// set colorkey for non-alpha surfaces
  void setColorKey(const Color& color) {doSetColorKey(&color);}
  /// disable colorkey for non-alpha surfaces
  void clearColorKey() {doSetColorKey(0);}
  /// get the Color value of the colorkey
  Color transparentColor() const;

  ///set alpha, in the range defined by SDL_ALPHA_OPAQUE and SDL_ALPHA_TRANSPARENT
  void setAlpha(unsigned char alpha);
  ///return true if the surface uses a color key
  bool usesColorKey() const
    {return sdlSurface_ && (sdlSurface_->flags & SDL_SRCCOLORKEY);}
  ///check if surface contains an alpha channel
  bool hasAlphaChannel() const
    {return sdlSurface_ && sdlSurface_->format->Amask;}
  ///return true if the surface is RLE accelerated
  void useRLEAcceleration(bool flag);
  
  /** Blit (copy) pixels.  Blit all of this surface to the target Surface \a s
   */
  Rect blit(Surface& s) const;
  /** Blit pixels with clipping rectangle.
   *  Blit all of this surface within the Rect \a dest of the target Surface \a s
   */ 
  Rect blit(Surface& s, const Rect& dest) const;
  /** Blit pixels, clipping both source and destination.
   *  Blit contents within Rect \a src of this surface to within the 
   *  Rect \a dest of Surface \a s
   */
  Rect blit(Surface& s, const Rect& dest, const Rect& src) const;
  /** Blit pixels from this surface to target Region of given Surface
   *
   * \param s destination Surface for blit 
   * \param dest Point (x,y) offset for blitting to target 
   * \param destMask Region describing destination mask
   */
  void blit(Surface& s, const Point& dest, const Region& destMask) const;
  /// scaled blit: zoom all of this Surface to fit into target Surface \a s
  Rect scaledBlit(Surface& s, bool smooth = false) const;
  /// Scaled blit with destination clipping
  Rect scaledBlit(Surface&, const Rect& dest, bool smooth = false) const;
  /// Scaled blit with destination and source clipping
  Rect scaledBlit(Surface&, const Rect& dest, const Rect& src, bool smooth =
			false) const;
  /** Textured blit, into destination surface \a dst.
   *
   * Source is this entire Surface, destination polygon described
   * by the four vertices (points) \a p1, \a p2, \a p3, \a p4.
   */
  Rect textureBlit(Surface& dst, const Point& p1, const Point& p2,
                   const Point& p3, const Point& p4) const;
  /** Clipped &amp; textured blit, into destination Surface \a dst.
   *
   * Source rectangle described by the Rect \a src, destination
   * polygon described by the four points \a p1, \a p2, \a p3, \a p4.
   */
  Rect textureBlit(Surface& dst, const Point& p1, const Point& p2,
                   const Point& p3, const Point& p4, const Rect& src) const;

  ///clear surface to background color 
  void clear();
  ///clear given rect
  void clear(const Rect&);
  ///fill whole surface with the given color
  void fill(const Color&);
  ///fill the given Rect (rectangle) with the given Color
  void fill(const Rect&, const Color&);
  ///fill the given Region with the given Color
  void fill(const Region&, const Color&);
  /// like fill, but with alpha blending
  void blend(const Color& c) {blend(rect(), c);}
  /// blend the given Rect (rectangle) with the given Color
  void blend(const Rect&, const Color&);
  /// blend the given Region with the given Color
  void blend(const Region&, const Color&);

  /// set window manager icon - for when the application is running
  /// By default WFTK supplies a simple 'tools on stained glass' icon.
  void setWMIcon() const {if(sdlSurface_) SDL_WM_SetIcon(sdlSurface_, 0);}

  /**this allows pixelaccess.... you should know what you're doing.
   * direct pixel access should be enclosed in lock()/unlock() pairs
  */
  void* pixels() 
    {  return ((sdlSurface_ == NULL) ? 0 : sdlSurface_->pixels); };
  /**this allows pixelaccess.... you should know what you do here.
   * pixels accesses should be enclosed in lock()/unlock() pairs
  */
  const void* pixels() const 
    {  return ((sdlSurface_ == NULL) ? 0 : sdlSurface_->pixels); };

  /**lock pixel data for direct access.
   * This is also need for pure read access, so it seems 
   * acceptable to make lock() a const method.
   */
  void lock() const ;
  /**unlock pixel data.
   * Never forget to call this after a lock(), otherwise the application 
   * might run into a deadlock.
   */
  void unlock() const;

  // you need to do your own locking for calls to readPixel()/writePixel()

  ///pixel address in bytes
  void writePixel(Uint32 pixeladdr, Uint32 pixel);
  ///pixel address in bytes
  Uint32 readPixel(Uint32 pixeladdr) const;

  // the next two do the locking for you, as well as color conversion

  /// get pixel color
  Color getPixel(const Point&) const;
  /// set pixel color
  void setPixel(const Point&, const Color&);

  struct ResLoad {
    std::pair<Surface*,bool> operator()(const std::string&);
  };
  /** Load bitmapped surface from file
   *
   * Use the \b load() function to load a surface from a file on disk and 
   * register it with wftk's Resources engine, e.g.:
   * \code
   *   Surface::registry::load("button_clicked", "button.png");
   * \endcode
   * Use the \b find() function to retrieve a pointer to the surface.  If the
   * named surface has not been registered, find() will return 0.
   * \code
   *   Surface::registry.find("button_clicked");
   *   myWidget->setBackground(Surface::registry.find("back_image"));
   * \endcode
   */
  static ResourceRegistry<Surface*,ResLoad> registry;
  /// refcounted resource type
  typedef Resource<Surface*> Resource;

  void makeGLTexture();

 protected:
  /// the underlying SDL_Surface
  SDL_Surface* sdlSurface_;
	///
	SDL_Surface* glSurface_;
	unsigned int glTexture_;
	float glTexMaxX_;
	float glTexMaxY_;

 private:
  /// for child surfaces
  SDL_Surface* parent_;

  /// setColorKey()/clearColorKey() implementation
  void doSetColorKey(const Color*);

  /// Used internally
  void drawGL(SDL_Rect src, SDL_Rect dst) const;
};

}

#endif
