# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2012-2014 Canonical, Ltd.
# Author: Thomi Richards
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.

"""Base classes for application type implementations."""

import logging


logger = logging.getLogger(__name__)


class ApplicationTypePluginBase(object):

    """A base class for all application type plugins."""

    _window_default = {
        'Title': "Default Window Title",
        'Maximized': False,
        'Minimized': False,                    # Start application minimized
        'MinimizeImmediatelyAfterShow': False, # After starting, minimize immediately
        'Menu': [],
        'Contents': None
    }

    def __init__(self, spec_dictionary):
        self.create_application()

        if not spec_dictionary:
            logger.warning("Specification contains no window specification. Creating a default window.")

        if isinstance(spec_dictionary, dict):
            window_spec = self._create_window_spec(spec_dictionary)
            self.create_window(window_spec)
        elif isinstance(spec_dictionary, list):
            for window in spec_dictionary:
                window_spec = self._create_window_spec(window)
                self.create_window(window_spec)

    def _create_window_spec(self, window_params):
        """Return a dictionary that contains window_params with all the defaults
        added.

        """
        defaults = self._window_default.copy()
        defaults.update(window_params)
        return defaults

    def create_application(self):
        """Create an application."""
        raise NotImplementedError(
            "This method must be implemented by child classes.")

    def create_window(self, window_spec):
        """Create a window, according to 'window_spec'."""
        raise NotImplementedError(
            "This method must be implemented by child classes.")

    def run(self):
        """Enter the event loop - run the application & windows."""
        raise NotImplementedError(
            "This method must be implemented by child classes.")
