/*
   Copyright (C) Andrzej Hajda 2006
   Contact: andrzej.hajda@wp.pl
   License: GNU General Public License version 2
*/

#include "includes.h"
#include "lib/cmdline/popt_common.h"
#include "librpc/rpc/dcerpc.h"
#include "librpc/gen_ndr/ndr_svcctl_c.h"
#include "librpc/gen_ndr/ndr_security.h"
#include "libcli/libcli.h"
#include "lib/events/events.h"

#include "winexe.h"
#include "winexesvc/shared.h"

#include <sys/fcntl.h>
#include <sys/unistd.h>
#include <sys/termios.h>

struct program_args {
	char *hostname;
	char *cmd;
	struct cli_credentials *credentials;
	int reinstall;
	int uninstall;
	int system;
	char *runas;
};

void parse_args(int argc, char *argv[], struct program_args *pmyargs)
{
	poptContext pc;
	int opt, i;

	int argc_new;
	char **argv_new;

	struct poptOption long_options[] = {
		POPT_AUTOHELP
		POPT_COMMON_SAMBA
		POPT_COMMON_CONNECTION
		POPT_COMMON_CREDENTIALS
		POPT_COMMON_VERSION
		{"uninstall", 0, POPT_ARG_NONE, &pmyargs->uninstall, 0,
		 "Uninstall winexe service after remote execution", NULL},
		{"reinstall", 0, POPT_ARG_NONE, &pmyargs->reinstall, 0,
		 "Reinstall winexe service before remote execution", NULL},
		{"system", 0, POPT_ARG_NONE, &pmyargs->system, 0,
		 "Use SYSTEM account" , NULL},
		{"runas", 0, POPT_ARG_STRING, &pmyargs->runas, 0,
		 "Run as user (BEWARE: password is sent in cleartext over net)" , "[DOMAIN\\]USERNAME%PASSWORD"},
		POPT_TABLEEND
	};

	pc = poptGetContext("winexe", argc, (const char **) argv,
			    long_options, POPT_CONTEXT_KEEP_FIRST);

	poptSetOtherOptionHelp(pc, "//host command");

	while ((opt = poptGetNextOpt(pc)) != -1) {
		poptPrintUsage(pc, stdout, 0);
		exit(1);
	}

	argv_new = discard_const_p(char *, poptGetArgs(pc));

	argc_new = argc;
	for (i = 0; i < argc; i++) {
		if (argv_new[i] == NULL) {
			argc_new = i;
			break;
		}
	}

	if (argc_new != 3 || argv_new[1][0] != '/'
	    || argv_new[1][1] != '/') {
		poptPrintUsage(pc, stdout, 0);
		exit(1);
	}

	pmyargs->hostname = argv_new[1] + 2;
	pmyargs->cmd = argv_new[2];
}

struct winexe_context {
	struct program_args *args;
	struct smbcli_tree *tree;
	struct async_context *ac_ctrl;
	struct async_context *ac_io;
	struct async_context *ac_err;
	int return_code;
};

void exit_program(struct winexe_context *c);

void on_ctrl_pipe_error(struct winexe_context *c, int func, NTSTATUS status)
{
	DEBUG(1, ("ERROR: on_ctrl_pipe_error - %s\n", nt_errstr(status)));
	static int activated = 0;
	if (!activated
	    && NT_STATUS_EQUAL(status, NT_STATUS_OBJECT_NAME_NOT_FOUND)) {
		status =
		    svc_install(c->args->hostname, c->args->credentials);
		if (!NT_STATUS_IS_OK(status)) {
			DEBUG(0,
			      ("ERROR: Failed to install service winexesvc - %s\n",
			       nt_errstr(status)));
			c->return_code = 1;
			exit_program(c);
		}
		activated = 1;
		async_open(c->ac_ctrl, "\\pipe\\" PIPE_NAME, OPENX_MODE_ACCESS_RDWR);
	} else if (func == ASYNC_OPEN_RECV) {
		DEBUG(0,
		      ("ERROR: Cannot open control pipe - %s\n",
		       nt_errstr(status)));
		c->return_code = 1;
		exit_program(c);
	} else
		exit_program(c);
}

void on_io_pipe_open(struct winexe_context *c);
void on_io_pipe_read(struct winexe_context *c, const char *data, int len);
void on_io_pipe_error(struct winexe_context *c, int func, NTSTATUS status);
void on_err_pipe_read(struct winexe_context *c, const char *data, int len);
void on_err_pipe_error(struct winexe_context *c, int func, NTSTATUS status);

const char *cmd_check(const char *data, const char *cmd, int len)
{
	int lcmd = strlen(cmd);
	if (lcmd >= len)
		return 0;
	if (!strncmp(data, cmd, lcmd)
	    && (data[lcmd] == ' ' || data[lcmd] == '\n')) {
		return data + lcmd + 1;
	}
	return 0;
}

void on_ctrl_pipe_open(struct winexe_context *c)
{
	char *str;
	if (c->args->runas)
		str = talloc_asprintf(c, "set runas %s\nrun %s\n", c->args->runas, c->args->cmd);
	else
		str = talloc_asprintf(c, "%srun %s\n", c->args->system ? "set system 1\n" : "" , c->args->cmd);
	DEBUG(1, ("CTRL: Sending command: %s", str));
	async_write(c->ac_ctrl, str, strlen(str));
	talloc_free(str);
}

void on_ctrl_pipe_read(struct winexe_context *c, const char *data, int len)
{
	const char *p;
	if ((p = cmd_check(data, CMD_STD_IO_ERR, len))) {
		DEBUG(1, ("CTRL: Recieved command: %.*s", len, data));
		unsigned int npipe = strtoul(p, 0, 16);
		c->ac_io = talloc_zero(c, struct async_context);
		c->ac_io->tree = c->tree;
		c->ac_io->cb_ctx = c;
		c->ac_io->cb_open = (async_cb_open) on_io_pipe_open;
		c->ac_io->cb_read = (async_cb_read) on_io_pipe_read;
		c->ac_io->cb_error = (async_cb_error) on_io_pipe_error;
		char *fn = talloc_asprintf(c->ac_io, "\\pipe\\" PIPE_NAME_IO, npipe);
		async_open(c->ac_io, fn, OPENX_MODE_ACCESS_RDWR);
		c->ac_err = talloc_zero(c, struct async_context);
		c->ac_err->tree = c->tree;
		c->ac_err->cb_ctx = c;
		c->ac_err->cb_read = (async_cb_read) on_err_pipe_read;
		c->ac_err->cb_error = (async_cb_error) on_err_pipe_error;
		fn = talloc_asprintf(c->ac_err, "\\pipe\\" PIPE_NAME_ERR, npipe);
		async_open(c->ac_err, fn, OPENX_MODE_ACCESS_RDWR);
	} else if ((p = cmd_check(data, CMD_RETURN_CODE, len))) {
		c->return_code = strtoul(p, 0, 16);
	} else {
		DEBUG(0, ("CTRL: Unknown command: %.*s", len, data));
	}
}

static void on_stdin_read_event(struct event_context *event_ctx,
			     struct fd_event *fde, uint16_t flags,
			     struct winexe_context *c)
{
	char buf[256];
	int len;
	if ((len = read(0, &buf, sizeof(buf))) > 0) {
		async_write(c->ac_io, buf, len);
	}
}

void on_io_pipe_open(struct winexe_context *c)
{
	event_add_fd(c->tree->session->transport->socket->event.ctx,
		     c->tree, 0, EVENT_FD_READ,
		     (event_fd_handler_t) on_stdin_read_event, c);
	struct termios term;
	tcgetattr(0, &term);
	term.c_lflag &= ~ICANON;
	tcsetattr(0, TCSANOW, &term);
	setbuf(stdin, NULL);
}

void on_io_pipe_read(struct winexe_context *c, const char *data, int len)
{
	write(1, data, len);
}

void on_io_pipe_error(struct winexe_context *c, int func, NTSTATUS status)
{
	async_close(c->ac_io);
}

void on_err_pipe_read(struct winexe_context *c, const char *data, int len)
{
	write(2, data, len);
}

void on_err_pipe_error(struct winexe_context *c, int func, NTSTATUS status)
{
	async_close(c->ac_err);
}

void exit_program(struct winexe_context *c)
{
	if (c->args->uninstall)
		svc_uninstall(c->args->hostname, c->args->credentials);
	exit(c->return_code);
}

int main(int argc, char *argv[])
{
	NTSTATUS status;
	struct smbcli_state *cli;
	struct program_args myargs = {.reinstall = 0,.uninstall = 0, .system = 0 };

	parse_args(argc, argv, &myargs);

	dcerpc_init();

	if (myargs.reinstall)
		svc_uninstall(myargs.hostname, cmdline_credentials);
	status =
	    smbcli_full_connection(NULL, &cli, myargs.hostname, "IPC$",
				   NULL, cmdline_credentials, NULL);
	if (!NT_STATUS_IS_OK(status)) {
		DEBUG(0,
		      ("ERROR: Failed to open connection - %s\n",
		       nt_errstr(status)));
		return 1;
	}

	struct winexe_context *c =
	    talloc_zero(cli->tree, struct winexe_context);
	if (c == NULL) {
		DEBUG(0,
		      ("ERROR: Failed to allocate struct winexe_context\n"));
		return 1;
	}

	c->tree = cli->tree;
	c->ac_ctrl = talloc_zero(cli->tree, struct async_context);
	c->ac_ctrl->tree = cli->tree;
	c->ac_ctrl->cb_ctx = c;
	c->ac_ctrl->cb_open = (async_cb_open) on_ctrl_pipe_open;
	c->ac_ctrl->cb_read = (async_cb_read) on_ctrl_pipe_read;
	c->ac_ctrl->cb_error = (async_cb_error) on_ctrl_pipe_error;
	c->args = &myargs;
	c->args->credentials = cmdline_credentials;
	c->return_code = 99;

	async_open(c->ac_ctrl, "\\pipe\\" PIPE_NAME, OPENX_MODE_ACCESS_RDWR);

	event_loop_wait(cli->tree->session->transport->socket->event.ctx);
	return 0;
}
