/*
 * (C)opyright MMIV-MMV Anselm R. Garbe <garbeam at gmail dot com>
 * See LICENSE file for license details.
 */

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "wmii.h"

static pid_t mypid;
static char *mysockfile;

/* convenience stuff ----------------------------------------------- */

File *
wmii_create_ixpfile(IXPServer * s, char *key, char *val)
{
    File *f = ixp_create(s, key);
    if(f && !is_directory(f)) {
        size_t l = val ? strlen(val) : 0;
        f->content = l ? strdup(val) : 0;
        f->size = l;
        return f;
    }
    /* forbidden, file is directory */
    return 0;
}

void
wmii_get_ixppath(File * f, char *path, size_t size)
{
    char buf[512];

    buf[0] = '\0';
    if(path)
        STRLCPY(buf, path, sizeof(buf));
    snprintf(path, size, "%s/", f->name);
    if(buf[0] != '\0')
        STRLCAT(path, buf, size);
    if(f->parent)
        wmii_get_ixppath(f->parent, path, size);
}

void
wmii_move_ixpfile(File * f, File * to_parent)
{
    File *p = f->parent;
    File *fil = p->content;

    /* detach */
    if(p->content == f)
        p->content = fil->next;
    else {
        while(fil->next != f)
            fil = fil->next;
        fil->next = f->next;
    }
    f->next = 0;


    /* attach */
    if(!to_parent->content)
        to_parent->content = f;
    else {
        for(fil = to_parent->content; fil->next; fil = fil->next);
        fil->next = f;
    }
    f->parent = to_parent;
}

static void
setup_dirs()
{
    char buf[128];
    char *home = getenv("HOME");
    char *user = getenv("USER");

    if(!home) {
        fprintf(stderr, "%s",
                "libwmii: error: $HOME environment variable is not set\n");
        exit(1);
    }
    if(!user) {
        fprintf(stderr, "%s",
                "libwmii: error: $USER environment variable is not set\n");
        exit(1);
    }

    snprintf(buf, sizeof(buf), "%s/.wmii", home);
    if(access(buf, F_OK)) {
        mkdir(buf, S_IRWXU);
        snprintf(buf, sizeof(buf), "%s/.wmii/.firstrun", home);
        close(open(buf, O_WRONLY | O_CREAT, S_IRWXU));
    }
    snprintf(buf, sizeof(buf), "/tmp/.ixp-%s", user);
    if(access(buf, F_OK)) {
        mkdir(buf, S_IRWXU);
    }
}

static void
exit_cleanup()
{
    if(mypid == getpid())
        unlink(mysockfile);
}

IXPServer *
wmii_setup_server(char *sockfile, size_t socklen, char *binname)
{
    IXPServer *s;

    setup_dirs();
    if(sockfile[0] == '\0') {
        if(!getenv("USER")) {
            fprintf(stderr, "%s",
                    "libwmii: error: $USER environment variable is not set\n");
            exit(1);

        }
        if(!getenv("WMII_IDENT")) {
            fprintf(stderr, "%s",
                    "libwmii: error: $WMII_IDENT environment variable is not set\n");
            exit(1);
        }
        snprintf(sockfile, socklen, "/tmp/.ixp-%s/%s-%s",
                 getenv("USER"), binname, getenv("WMII_IDENT"));
    }
    mysockfile = sockfile;
    mypid = getpid();
    s = init_server(sockfile, exit_cleanup);
    if(!s) {
        perror("libwmii: cannot initialize IXP server");
        exit(1);
    }

    return s;
}
