#include "config.h"

/*  Copyright (C) 2002  Brad Jorsch <anomie@users.sourceforge.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <unistd.h>
#include <limits.h>
#include <fcntl.h>
#include <errno.h>
#include <WWWInit.h>

#include "die.h"
#include "download.h"

struct SockInfo {
    SOCKET s;
    HTEvent *events[HTEvent_TYPES];
    HTTimer *timeouts[HTEvent_TYPES];
    struct SockInfo *next;
};

static struct SockInfo *list;
static fd_set fdsets[HTEvent_TYPES];
static SOCKET maxfd=0;

static struct SockInfo *get_sock_info(SOCKET s, int create){
    struct SockInfo *i;
    int j;

    for(i=list; i!=NULL; i=i->next){
        if(i->s==s) return i;
    }
    if(!create) return NULL;
    if((i=malloc(sizeof(*i)))==NULL) return NULL;
    i->s=s;
    for(j=0; j<HTEvent_TYPES; j++){
        i->events[j]=NULL;
        i->timeouts[j]=NULL;
    }
    i->next=list;
    list=i;
    return i;
}

static int del_sock_info(SOCKET s){
    struct SockInfo *i, *j;
    
    if(list->s==s){
        j=list;
        list=list->next;
        free(j);
        return 0;
    }

    for(i=list; i->next!=NULL; i=i->next){
        if(i->next->s==s){
            j=i->next;
            i->next=i->next->next;
            free(j);
            return 0;
        }
    }
    return 1;
}

static int timer_handler(HTTimer *timer, void *param, HTEventType type){
    struct SockInfo *info=(struct SockInfo *)param;
    HTEvent *event=NULL;
    int i;

    for(i=0; i<HTEvent_TYPES; i++){
        if(info->timeouts[i]==timer){
            event=info->events[i];
            return (*event->cbf)(info->s, event->param, HTEvent_TIMEOUT);
        }
    }
    return HT_ERROR;
}

static int reg(SOCKET s, HTEventType type, HTEvent *event){
    struct SockInfo *info;

    if(s==INVSOC || HTEvent_INDEX(type)>=HTEvent_TYPES) return HT_ERROR;

    info=get_sock_info(s, 1);
    if(info==NULL) return HT_ERROR;
    info->events[HTEvent_INDEX(type)]=event;
    if(event->millis>=0){
        info->timeouts[HTEvent_INDEX(type)]=HTTimer_new(info->timeouts[HTEvent_INDEX(type)], timer_handler, info, event->millis, YES, YES);
    }

    FD_SET(s, fdsets+HTEvent_INDEX(type));
    if(s>maxfd) maxfd=s;

    return HT_OK;
}


static int unreg(SOCKET s, HTEventType type){
    struct SockInfo *info;
    int i;

    if(s==INVSOC || HTEvent_INDEX(type)>=HTEvent_TYPES) return HT_OK;

    info=get_sock_info(s, 0);
    if(info==NULL) return HT_OK;
    info->events[HTEvent_INDEX(type)]=NULL;
    if(info->timeouts[HTEvent_INDEX(type)]){
        HTTimer_delete(info->timeouts[HTEvent_INDEX(type)]);
        info->timeouts[HTEvent_INDEX(type)]=NULL;
    }
    FD_CLR(s, fdsets+HTEvent_INDEX(type));
    for(i=0; i<HTEvent_TYPES; i++){
        if(info->events[i]!=NULL) return HT_OK;
    }
    del_sock_info(s);
    return HT_OK;
}

static HTList *converters=NULL;
static HTList *transfer_encodings=NULL;
static HTList *content_encodings=NULL;

static BOOL error_printer(HTRequest *request, HTAlertOpcode op, int msgnum, const char *dfault, void *input, HTAlertPar *reply){
    char *msg=HTDialog_errorMessage(request, op, msgnum, dfault, input);
    int i;
    
    if(msg){
        for(i=0; msg[i]; i++){
            if(msg[i]=='\r' || msg[i]=='\n') msg[i]=' ';
        }
        warn("%s", msg);
        HT_FREE(msg);
    }
    return YES;
}

void DownloadInit(char *email){
    if(!HTLib_isInitialized()){
        HTLibInit("download.c", "2.1");
        HTEvent_setRegisterCallback(reg);
        HTEvent_setUnregisterCallback(unreg);
        if(email!=NULL) HTUserProfile_setEmail(HTLib_userProfile(), email);

        if(!converters) converters=HTList_new();
        if(!transfer_encodings) transfer_encodings=HTList_new();
        if(!content_encodings) content_encodings=HTList_new();

        HTTransportInit();
        HTProtocolInit();
        HTNet_setMaxSocket(64);
        HTNet_addBefore(HTRuleFilter, NULL, NULL, HT_FILTER_MIDDLE); 
        HTNet_addBefore(HTProxyFilter, NULL, NULL, HT_FILTER_MIDDLE); 
        HTNet_addAfter(HTRedirectFilter, NULL, NULL, HT_ALL, HT_FILTER_MIDDLE);
        HTNet_addAfter(HTInfoFilter, NULL, NULL, HT_ALL, HT_FILTER_LATE);
        HTProxy_getEnvVar();
        HTConverterInit(converters);
        HTFormat_setConversion(converters);
        HTTransferEncoderInit(transfer_encodings);
        HTFormat_setTransferCoding(transfer_encodings);
        HTContentEncoderInit(content_encodings);
        if(HTList_count(content_encodings) > 0)
            HTFormat_setContentCoding(content_encodings);
        else {
            HTList_delete(content_encodings);
            content_encodings = NULL;
        }
        HTMIMEInit();
        HTAlert_add(error_printer, HT_A_MESSAGE);
        HTAlert_setInteractive(NO);

        HTProtocolInit();
    }
}

static int handle_io(SOCKET s, HTEventType type, ms_t now){
    struct SockInfo *info;
    HTEvent *event;

    info=get_sock_info(s, 0);
    if(info==NULL || (event=info->events[HTEvent_INDEX(type)]) == NULL) return 1;
    if(info->timeouts[HTEvent_INDEX(type)]) HTTimer_refresh(info->timeouts[HTEvent_INDEX(type)], now);
    return (*event->cbf)(s, event->param, type);
}


/* Damnit! One of the FDs is bad, find out which one and unreg it */
static void handle_EBADF(void){
    int s, i;
    ms_t now;

    now=HTGetTimeInMillis();
    for(s=0; s<=maxfd; s++){
        if(!FD_ISSET(s, fdsets+HTEvent_INDEX(HTEvent_READ)) &&
           !FD_ISSET(s, fdsets+HTEvent_INDEX(HTEvent_WRITE)) &&
           !FD_ISSET(s, fdsets+HTEvent_INDEX(HTEvent_OOB))){
            /* This can't be it, we're not checking this one... */
            continue;
        }

        /* fcntl(F_GETFL) should either succeed or EBADF */
        if(fcntl(s, F_GETFL)!=-1) continue;
        if(errno!=EBADF) warn("WTF? fcntl(%d, F_GETFL) set errno=%d", s, errno);
        for(i=0; i<HTEvent_TYPES; i++) unreg(s, i);
    }
}

/* WTF? EINVAL should never happen... */
void handle_EINVAL(void){
    struct SockInfo *i;
    int j;

    for(j=0; j<HTEvent_TYPES; j++){
        FD_ZERO(fdsets+j);
    }
    maxfd=0;
    for(i=list; i!=NULL; i=i->next){
        if(i->s>maxfd) maxfd=i->s;
        for(j=0; j<HTEvent_TYPES; j++){
            if(i->events[j]) FD_SET(i->s, fdsets+j);
        }
    }
}

void ProcessDownloads(unsigned long sleeptime){
    fd_set rd, wr, er;
    struct timeval tv;
    int n;
    SOCKET s;
    ms_t now;
    static unsigned long error_counter=0;

    HTTimer_next(NULL);

    rd=fdsets[HTEvent_INDEX(HTEvent_READ)];
    wr=fdsets[HTEvent_INDEX(HTEvent_WRITE)];
    er=fdsets[HTEvent_INDEX(HTEvent_OOB)];
    tv.tv_sec=0;
    tv.tv_usec=sleeptime;
    if(sleeptime>=1000000){
        tv.tv_sec=sleeptime/1000000;
        tv.tv_usec=sleeptime%1000000;
    }
    
    n=select(maxfd+1, &rd, &wr, &er, &tv);
    if(n>=0) error_counter=0;
    if(n==0) return;
    if(n<0){
        switch(errno){
          case EINTR:
          case ENOMEM:
            /* transient errors, hope it's good next time */
            break;

          case EINVAL:
            warn("WTF? select returned EINVAL (maxfd=%d)", maxfd);
            handle_EINVAL();
            break;

          case EBADF:
            if(error_counter>10*1000000){ /* 10 seconds of error */
                warn("Hmmm, an fd went bad...");
                handle_EBADF();
                break;
            }

          default:
            warn("WTF? select errno=%d", errno);
            handle_EINVAL();
            handle_EBADF();
            break;
        }
        if(ULONG_MAX-error_counter<sleeptime) error_counter=ULONG_MAX;
        else error_counter+=sleeptime;
        usleep(sleeptime);
        return;
    }

    now=HTGetTimeInMillis();
    for(s=0; s<=maxfd; s++){
        if(FD_ISSET(s, &er)) if(handle_io(s, HTEvent_OOB, now)) return;
        if(FD_ISSET(s, &wr)) if(handle_io(s, HTEvent_WRITE, now)) return;
        if(FD_ISSET(s, &rd)) if(handle_io(s, HTEvent_READ, now)) return;
    }
}

struct download_info {
    void (*callback)(char *filename, void *data);
    char *filename;
    void *data;
    FILE *fp;
    HTParentAnchor *src;
    int flags;
};

static int terminate_handler(HTRequest *request, HTResponse *response, void *param, int status){
    struct download_info *info=param;
    struct stat statbuf;
    char *msg;
    int i;
    
    if(status!=200 && status!=204){
        msg=HTDialog_errorMessage(request, HT_A_MESSAGE, HT_MSG_NULL, NULL, HTRequest_error(request));
        if(msg){
            if(info->flags&DOWNLOAD_NO_404 && 
               (!strncmp(msg, "Fatal Error: 404 Not Found", 26))){
                /* ignore 404 */
            } else {
                for(i=0; msg[i]; i++){
                    if(msg[i]=='\r' || msg[i]=='\n') msg[i]=' ';
                }
                errno=0;
                warn("download of %s failed: %s", info->filename, msg);
            }
            HT_FREE(msg);
        } else {
            warn("download of %s failed: unknown libwww error", info->filename);
        }
        unlink(info->filename);
    } else if(stat(info->filename, &statbuf)<0){
        warn("download of %s failed: stat failed", info->filename);
        unlink(info->filename);
    } else if(!S_ISREG(statbuf.st_mode)){
        warn("download of %s failed: not a regular file", info->filename);
        unlink(info->filename);
    } else {
        (*info->callback)(info->filename, info->data);
    }
    HTRequest_delete(request);
    if(info->src) HTAnchor_delete(info->src);
    free(info->filename);
    free(info);

    return HT_ERROR;
}

int download_file(char *filename, char *from_addr, char *postdata, int flags, void (*callback)(char *filename, void *data), void *data){
    struct download_info *info=NULL;
    HTRequest *request=NULL;
    BOOL status=NO;

    if(callback==NULL || filename==NULL || from_addr==NULL) return 1;

    if((info=malloc(sizeof(*info)))==NULL){
        warn("Malloc error in download_file");
        goto fail;
    }

    if((info->filename=strdup(filename))==NULL) goto fail;
    if((info->fp=fopen(info->filename, "wb"))==NULL){
        warn("Error opening %s for output", info->filename);
        goto fail;
    }

    info->callback=callback;
    info->data=data;
    info->src=NULL;
    info->flags=flags;

    if(flags&DOWNLOAD_KILL_OTHER_REQUESTS){
        request=HTRequest_new();
        HTLoadAbsolute(from_addr, request);
        HTHost_killPipe(HTNet_host(HTRequest_net(request)));
        HTRequest_delete(request);
    }

    request=HTRequest_new();
    if(request==NULL){
        warn("Error creating a HTRequest");
        goto fail;
    }

    HTRequest_setOutputFormat(request, WWW_SOURCE);
    HTRequest_setOutputStream(request, HTFWriter_new(request, info->fp, NO));
    HTRequest_addAfter(request, terminate_handler, NULL, info, HT_ALL, HT_FILTER_LAST, YES);

    if(postdata!=NULL){
        info->src=HTTmpAnchor(NULL);
        HTAnchor_setDocument(info->src, postdata);
        HTAnchor_setFormat(info->src, WWW_FORM);
        HTAnchor_setLength(info->src, strlen(postdata));
        status=HTPostAbsolute(info->src, from_addr, request);
    } else {
        status=HTLoadAbsolute(from_addr, request);
    }


    return 0;

fail:
    if(request) HTRequest_delete(request);
    if(info){
        if(info->fp){
            fclose(info->fp);
            unlink(info->filename);
        }
        if(info->filename) free(info->filename);
        free(info);
    }
    return 1;
}
