/* worker.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2001-2009 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "worker.h"
#include "lister.h"
#include "worker_locale.h"
#include "normalmode.h"
#include "wconfig.h"
#include "wcdoubleshortkey.hh"
#include "wcpath.hh"
#include "filereq.h"
#include "grouphash.h"
#include "wdefines.h"
#include "showimagemode.h"
#include "informationmode.h"
#include <string>
#include <list>
#include "wpucontext.h"
#include "cgpl.h"

#ifdef HAVE_SIGACTION
#include <signal.h>
#endif

#include "execlass.h"
#include "configtokens.h"
#include "configheader.h"
#include "configparser.tab.h"
#include "worker_shm.h"
#include "parentactionop.h"
#include <aguix/textstorage.h>
#include "wchotkey.hh"
#include "wcbutton.hh"
#include <aguix/utf8.hh>
#include "workerversion.h"
#include "nwc_path.hh"
#include <iostream>
#include <aguix/popupmenu.hh>
#include "magic_db.hh"
#include "bookmarkdb.hh"
#include "bookmarkdbproxy.hh"
#include <algorithm>

Requester *Worker::req_static=NULL;
int Worker::argc = 0;
char **Worker::argv = NULL;
std::string Worker::startcwd = "/";
AGUIX *Worker::aguix = NULL;
int Worker::workerInstances = 0;

std::auto_ptr<BookmarkDBProxy> Worker::m_bookmarkproxy;

short usr1signal = 0;

#ifdef HAVE_SIGACTION
static void worker_sighandler( int s )
{
  if ( s == SIGUSR1 ) {
    usr1signal = 1;
  }
}
#endif

Worker::Worker(int targc,char **targv)
{
  char* actdir;
  int size;

  workerInstances++;
  if ( aguix == NULL ) {
    aguix = new AGUIX( targc, targv, "Worker" );
    aguix->initX();
  }
  mainwin=NULL;
  aboutb=NULL;
  configureb=NULL;
  clockbar=NULL;
  statebar=NULL;
  pathbs=NULL;
  buttons=NULL;
  lister[0]=NULL;
  lister[1]=NULL;
  listerwin[0] = NULL;
  listerwin[1] = NULL;
  maincont = NULL;
  m_worker_menu = NULL;
  curRows = 0;
  curColumns = 0;
  w=600;
  h=400;
  pbanknr=0;
  bbanknr=0;
  runningmode=WORKER_NORMAL;
  this->argc=targc;
  this->argv=targv;

  size = 1024;
  for(;;) {
    actdir = (char*)_allocsafe( size );
    if ( getcwd( actdir, size ) != NULL ) break;
    else if ( errno != ERANGE ) {
      strcpy( actdir, "/" );  // use the root if getcwd doesn't report the actual
      break;
    }
    _freesafe( actdir );
    size *= 2;
  }
  startcwd = actdir;
  _freesafe( actdir );
  
  dm=new DNDMsg(this);
  lasttimep=0;
  
  if(req_static==NULL) {
    // only if not initialized
    req_static=new Requester(aguix);
  }
  freesp = new PartSpace();

  if ( geteuid() == 0 )
    isRoot = true;
  else
    isRoot = false;
  
  lastkey = 0;
  lastmod = 0;
  lastkey_time = 0;

  keyhashtable = NULL;
  keyhashtable_size = 1000;
  waitcursorcount = 0;
  restart = false;
  configstarted = false;

  m_interpret_recursion_depth = 0;
}

Worker::~Worker()
{
  // free tokenlist
  // it doesn't hurt if we need it again, it will
  // be recreated but that's the easiest way to release
  // the memory
  cleanUpTokens();
  lexer_cleanup();

  if(req_static!=NULL) {
    delete req_static;
    req_static=NULL;
  }

  workerInstances--;
  if ( ( workerInstances < 1 ) && ( aguix != NULL ) ) {
    delete aguix;
    aguix = NULL;
    WorkerSHM::freeSHM();
  }
  delete dm;
  delete freesp;
}

void Worker::run()
{
  int i,j;
  time_t lastConfigCheck;
  bool found;
  int waitcount=0;
  int erg;
  time_t lastConfig = 0;
  std::string str1;

  restart = false;
  configstarted = false;
  wconfig->loadsize(&w,&h);
  if ( w < 300 ) w = 300;
  if ( h < 200 ) h = 200;
  aguix->setFont(wconfig->getFont(0));
  createMainWin();
  setupMainWin();
  lister[0]=new Lister(this);
  lister[1]=new Lister(this);
  initListerState();
  AGMessage *msg;
  bool handleSub;
  int rows,columns;
  rows=wconfig->getRows();
  columns=wconfig->getColumns();
  int pathact=-1;

  buildtable();

  lastConfigCheck = time( NULL );

  while(runningmode!=WORKER_QUIT) {
    msg=aguix->GetMessage(NULL);
    handleSub=true;
    found=false;
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
            if ( msg->size.window == mainwin->getWindow() ) {
                quit(1);
                handleSub=false;
            }
            break;
        case AG_SIZECHANGED:
          if(msg->size.window==mainwin->getWindow()) {
            w=msg->size.neww;
            h=msg->size.newh;
            if ( ( w < 300 ) || ( h < 200 ) ) {
              if ( w < 300 ) w = 300;
              if ( h < 200 ) h = 200;
              // user resized the window to too little values
              // (perhaps the WM ignores minsize)
              // at the moment I won't resize the win again to valid values
              // because it's the users problem when the window is too small
              // mainwin->resize( w, h );
            }
            maincont->resize( w, h );
            maincont->rearrange();
          } else handleSub=false;
          break;
        case AG_BUTTONPRESSED:
          if(msg->button.state==2) {
            for(i=0;i<rows;i++) {
              if(pathbs[i]==msg->button.button) {
                for(j=0;j<i;j++) pathbs[j]->setState(2);
                for(j=i+1;j<rows;j++) pathbs[j]->setState(2);
                pathact=i;
                handleSub=false;
                break;
              }
            }
          }
          break;
        case AG_BUTTONRELEASED:
          if(msg->button.button==aboutb) {
            about();
            handleSub=false;
          } else if(msg->button.button==configureb) {
            configure();
            handleSub=false;
          } else if(msg->button.button==clockbar) {
            if( msg->button.state == 1 ) {
              shuffleButton( 1 );
            } else if ( msg->button.state == 2 ) {
              shuffleButton( -1 );
            } else if ( msg->button.state == 3 ) { // mouse wheel up
              shuffleButton( -1, true );
            } else if ( msg->button.state == 4 ) { // mouse wheel down
              shuffleButton( 1, true );              
            }
            handleSub=false;
          } else if ( msg->button.button == statebar ) {
              if ( msg->button.state == 2 ) {
                  openWorkerMenu();
              }
              handleSub = false;
          } else {
            for(i=0;i<rows;i++) {
              if(pathbs[i]==msg->button.button) {
                if(msg->button.state==2) {
                  shufflePath();
                } else {
                  // Path setzen
                  setPath(i+pbanknr*rows);
                }
                found=true;
                break;
              }
            }
            if(found==false) {
              for(i=0;i<rows;i++) {
                for(j=0;j<columns;j++) {
                  if(msg->button.button==buttons[i*columns+j]) {
                    activateButton(i,j,msg->button.state);
                    found=true;
                  }
                }
              }
            } else handleSub=false;
          }
          break;
        case AG_MOUSERELEASED:
          if(pathact>=0) {
            for(j=0;j<pathact;j++) pathbs[j]->setState(0);
            for(j=pathact+1;j<rows;j++) pathbs[j]->setState(0);
            pathact=-1;
            handleSub=false;
          }
          break;
        case AG_DND_START:
          // Start of DND; only safe the message
          dm->setStart(&(msg->dnd));
          break;
        case AG_DND_END:
          // End of DND; safe the message and find the mode
          dm->setEnd(&(msg->dnd));
          if(dm->getSourceLister()!=NULL) dm->getSourceLister()->startdnd(dm);
          break;
        case AG_KEYPRESSED:
          if ( activateShortkey(msg) != 0 ) {
            handleSub = false;
	  }
          break;
        case AG_POPUPMENU_CLICKED:
            if ( msg->popupmenu.menu == m_worker_menu ) {
                startMenuAction( msg );
                handleSub = false;
            } else {
                handlePopUps( msg );
            }
            break;
        default:
          break;
      }
      if(handleSub==true) {
        lister[0]->messageHandler(msg);
        lister[1]->messageHandler(msg);
      }
      aguix->ReplyMessage(msg);
    } else {
      updateTime();
      lister[0]->cyclicfunc(CYCLICFUNC_MODE_NORMAL);
      lister[1]->cyclicfunc(CYCLICFUNC_MODE_NORMAL);
      waitcount++;
      if ( difftime( time( NULL ), lastConfigCheck ) > 60 ) {
        if ( wconfig->checkForNewerConfig() == true ) {
          // config is newer then used config
          // now check if this config is even newer then the
          // last decided
          if ( difftime( wconfig->getFileConfigMod(), lastConfig ) > 0 ) {
            // config changed after last check
            str1 = catalog.getLocale( 511 );
            str1 += "|";
            str1 += catalog.getLocale( 512 );
            erg = req_static->request( catalog.getLocale( 123 ),
                                       catalog.getLocale( 510 ),
                                       str1.c_str() );
            if ( erg == 0 ) {
              wconfig->load();
              wconfig->applyColorList(wconfig->getColors());
              wconfig->applyLanguage();
              configstarted = true;
              restart = true;
            }
            lastConfig = wconfig->getFileConfigMod();
          }
        }
        lastConfigCheck = time( NULL );
      }
      waittime(20);
    }
    if(configstarted==true) {
      if(restart==false) {
        lister[0]->cyclicfunc(CYCLICFUNC_MODE_RESTART);
        lister[1]->cyclicfunc(CYCLICFUNC_MODE_RESTART);
      }
      configstarted=false;
      buildtable();
    }
    if(restart==true) {
      //TODO
      aguix->setFont(wconfig->getFont(0));
      setupMainWin();
      rows=wconfig->getRows();
      columns=wconfig->getColumns();
      lister[0]->reconfig();
      lister[1]->reconfig();
      lister[0]->cyclicfunc(CYCLICFUNC_MODE_RESTART);
      lister[1]->cyclicfunc(CYCLICFUNC_MODE_RESTART);
      restart=false;
      lasttimep=0;
    }
  }

  if ( wconfig->getSaveWorkerStateOnExit() == true ) {
      saveWorkerState();
  }
  delete lister[0];
  delete lister[1];

  closeMainWin();
  deletetable();
}

AGUIX *Worker::getAGUIX()
{
  return aguix;
}

int main(int argc,char **argv)
{
  bool showversion,showhelp;
  int i;
  char *setlang;

#ifdef HAVE_SIGACTION
  struct sigaction sig_ac;
  
  memset( &sig_ac, 0, sizeof( sig_ac ) );
  sig_ac.sa_handler = worker_sighandler;

  sigaction( SIGUSR1, &sig_ac, NULL );
#endif

  showversion=false;
  showhelp=false;
  for(i=1;i<argc;i++) {
    if((strcmp(argv[i],"-V")==0)||(strcmp(argv[i],"--version")==0)) {
      showversion=true;
    } else if((strcmp(argv[i],"-h")==0)||(strcmp(argv[i],"--help")==0)) {
      showhelp=true;
    }
  }
  if(showversion==true) {
    printf("Worker Version: %d.%d.%d%s\n",WORKER_MAJOR,WORKER_MINOR,WORKER_PATCH,WORKER_COMMENT);
    printf("  Author: Ralf Hoffmann <ralf@boomerangsworld.de>\n");
    printf("  Homepage: http://www.boomerangsworld.de/worker\n\n");
    exit(0);
  }
  if(showhelp==true) {
    printf("Worker Filemanager\n");
    printf("  Version: %d.%d.%d%s\n",WORKER_MAJOR,WORKER_MINOR,WORKER_PATCH,WORKER_COMMENT);
    printf("  Author: Ralf Hoffmann <ralf@boomerangsworld.de>\n");
    printf("  Homepage: http://www.boomerangsworld.de/worker\n\n");
    printf("  Options: -V, --version\tPrint the program version and exit\n");
    printf("           -h, --help\tPrint this help and exit\n\n");
    printf("           <dir>\t\tShows the dir on the left side\n");
    printf("           <dir1> <dir2>\tShows dir1 on the left side, dir2 on the right side\n\n");
    exit(0);
  }
 
  if ( worker_tmpdir() == NULL ) {
    exit( 0 );
  }

  ugdb=new GroupHash();
  Worker *worker=new Worker(argc,argv);
  if(worker->getAGUIX()->checkX()==True) {
    wconfig=new WConfig(worker);
    wconfig->applyColorList(wconfig->getColors());
    
    // now check for existence of ".worker" and if not try to create it and copy an example config
    setlang = NULL;
    worker->checkfirststart( &setlang );
    
    wconfig->load();
    
    if ( setlang != NULL ) {
      wconfig->setLang( setlang );
    }

    wconfig->applyColorList(wconfig->getColors());
    wconfig->applyLanguage();

    if ( setlang != NULL ) {
      wconfig->save();
      _freesafe( setlang );
    }
    
    MagicDB::getInstance().init();

    worker->run();
    
    MagicDB::getInstance().fini();
    
    delete wconfig;
  } else {
    printf("Worker: Can't connect to X!\n");
  }

  delete worker;
  delete ugdb;

  catalog.freeLanguage();
  return 0;
}

AWindow *Worker::getMainWin()
{
  return mainwin;
}

void Worker::setupMainWin()
{
  int rows, columns;
  int th;
  unsigned int i;
  int i2;
  AGUIXFont *font;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;
  const int conlyh = AContainer::ACONT_MINH +
                     AContainer::ACONT_MAXH;
  bool bv, lvv, statebarfound;
  int e1, pos;
  std::list<WConfig::layoutID_t> orders;
  std::list<WConfig::layoutID_t>::iterator it1;
  AContainer *sbc = NULL;
  AContainer *clbc = NULL;
  AContainer *bc = NULL;
  AContainer *lc = NULL;
  AContainer *lc2 = NULL;
  int listviewsCount = 0;
  int x,y;
  int statebarCount, clockbarCount, buttonsCount, bllCount, blCount;
  int l;
  
  rows=wconfig->getRows();
  columns=wconfig->getColumns();

  if ( maincont != NULL ) delete maincont;
  
  bv = wconfig->getLayoutButtonVert();
  lvv = wconfig->getLayoutListviewVert();
  orders = wconfig->getLayoutOrders();
  
  if ( orders.size() < 1 ) {
    // default values for empty orders
    bv = lvv = false;
    orders.push_back( WConfig::LO_STATEBAR );
    orders.push_back( WConfig::LO_LISTVIEWS );
    orders.push_back( WConfig::LO_BUTTONS );
    orders.push_back( WConfig::LO_CLOCKBAR );
  }
  
  if ( ( bv == true ) && ( lvv == true ) ) e1 = 3;
  else if ( ( bv == false ) && ( lvv == true ) ) e1 = 5;
  else if ( ( bv == true ) && ( lvv == false ) ) e1 = 3;
  else e1 = 4;
  
  if ( orders.size() < (unsigned int)e1 ) e1 = orders.size();
  if ( e1 < 1 ) e1 = 1;
  
  statebarfound = false;
  for ( it1 = orders.begin(); it1 != orders.end(); it1++ ) {
    if ( *it1 == WConfig::LO_STATEBAR ) {
      statebarfound = true;
      break;
    }
  }
  if ( statebarfound == false ) e1++;

  maincont = mainwin->setContainer( new AContainer( mainwin, 0, e1 ) );
  maincont->setMaxSpace( 0 );
  maincont->setMinSpace( 0 );
  maincont->setBorderWidth( 0 );
  
  statebarCount = 0;
  clockbarCount = 0;
  buttonsCount = 0;
  bllCount = 0;
  blCount = 0;
  for ( pos = 0, it1 = orders.begin(); ( pos < e1 ) && ( it1 != orders.end() ); pos++ ) {
    if ( ( statebarfound == false ) ||
         ( *it1 == WConfig::LO_STATEBAR ) ) {
      if ( statebarCount < 1 ) {
        sbc = maincont->add( new AContainer( mainwin, 3, 1 ), 0, pos );
        sbc->setMaxSpace( 0 );
        sbc->setMinSpace( 0 );
        sbc->setBorderWidth( 0 );
        sbc->add( aboutb, 0, 0, cfix );
        sbc->add( configureb, 1, 0, cfix );
        sbc->add( statebar, 2, 0, conlyh );
        statebarCount++;
        
        if ( statebarfound == false ) {
          statebarfound = true;
          continue;
        }
      }
    } else if ( *it1 == WConfig::LO_CLOCKBAR ) {
      if ( clockbarCount < 1 ) {
        clbc = maincont->add( new AContainer( mainwin, 1, 1 ), 0, pos );
        clbc->setMaxSpace( 0 );
        clbc->setMinSpace( 0 );
        clbc->setBorderWidth( 0 );
        clbc->add( clockbar, 0, 0, conlyh );
        clockbarCount++;
      }
    } else if ( *it1 == WConfig::LO_BUTTONS ) {
      if ( ( bv == false ) && ( buttonsCount < 1 ) ) {
        bc = maincont->add( new AContainer( mainwin, columns + 1, rows ), 0, pos );
        bc->setMaxSpace( 0 );
        bc->setMinSpace( 0 );
        bc->setBorderWidth( 0 );
        buttonsCount++;
      }
    } else if ( *it1 == WConfig::LO_LISTVIEWS ) {
      if ( bv == false ) {
        if ( ( listviewsCount < 1 ) && ( lvv == false ) ) {
          lc = maincont->add( new AContainer( mainwin, 2, 1 ), 0, pos );
          lc->setMaxSpace( 0 );
          lc->setMinSpace( 0 );
          lc->setBorderWidth( 0 );
          lc->add( listerwin[0], 0, 0 );
          lc->add( listerwin[1], 1, 0 );
          listviewsCount++;
        } else if ( ( listviewsCount < 2 ) && ( lvv == true ) ) {
          lc = maincont->add( new AContainer( mainwin, 1, 1 ), 0, pos );
          lc->setMaxSpace( 0 );
          lc->setMinSpace( 0 );
          lc->setBorderWidth( 0 );
          lc->add( listerwin[listviewsCount], 0, 0 );
          listviewsCount++;
        }
      }
    } else if ( *it1 == WConfig::LO_BLL ) {
      if ( ( bv == true ) && ( lvv == false ) && ( bllCount < 1 ) ) {
        lc = maincont->add( new AContainer( mainwin, 3, 1 ), 0, pos );
        lc->setMaxSpace( 0 );
        lc->setMinSpace( 0 );
        lc->setBorderWidth( 0 );
        lc->add( listerwin[0], 1, 0 );
        lc->add( listerwin[1], 2, 0 );
        
        bc = lc->add( new AContainer( mainwin, 1, ( columns + 1 ) * rows ), 0, 0 );
        bc->setMaxSpace( 0 );
        bc->setMinSpace( 0 );
        bc->setBorderWidth( 0 );
        bllCount++;
        
        lc->setWWeight( 2, 1, 0 );
        lc->setWWeight( 2, 2, 0 );

        l = getMaxButtonsWidth() + 4;
        lc->setMaxWidth( l, 0, 0 );
      }
    } else if ( *it1 == WConfig::LO_LBL ) {
      if ( ( bv == true ) && ( lvv == false ) && ( bllCount < 1 ) ) {
        lc = maincont->add( new AContainer( mainwin, 3, 1 ), 0, pos );
        lc->setMaxSpace( 0 );
        lc->setMinSpace( 0 );
        lc->setBorderWidth( 0 );
        lc->add( listerwin[0], 0, 0 );
        lc->add( listerwin[1], 2, 0 );
        
        bc = lc->add( new AContainer( mainwin, 1, ( columns + 1 ) * rows ), 1, 0 );
        bc->setMaxSpace( 0 );
        bc->setMinSpace( 0 );
        bc->setBorderWidth( 0 );
        bllCount++;
        
        lc->setWWeight( 2, 0, 0 );
        lc->setWWeight( 2, 2, 0 );

        l = getMaxButtonsWidth() + 4;
        lc->setMaxWidth( l, 1, 0 );
      }
    } else if ( *it1 == WConfig::LO_LLB ) {
      if ( ( bv == true ) && ( lvv == false ) && ( bllCount < 1 ) ) {
        lc = maincont->add( new AContainer( mainwin, 3, 1 ), 0, pos );
        lc->setMaxSpace( 0 );
        lc->setMinSpace( 0 );
        lc->setBorderWidth( 0 );
        lc->add( listerwin[0], 0, 0 );
        lc->add( listerwin[1], 1, 0 );
        
        bc = lc->add( new AContainer( mainwin, 1, ( columns + 1 ) * rows ), 2, 0 );
        bc->setMaxSpace( 0 );
        bc->setMinSpace( 0 );
        bc->setBorderWidth( 0 );
        bllCount++;
        
        lc->setWWeight( 2, 0, 0 );
        lc->setWWeight( 2, 1, 0 );

        l = getMaxButtonsWidth() + 4;
        lc->setMaxWidth( l, 2, 0 );
      }
    } else if ( *it1 == WConfig::LO_BL ) {
      if ( ( bv == true ) && ( lvv == true ) && ( blCount < 1 ) ) {
        lc = maincont->add( new AContainer( mainwin, 2, 1 ), 0, pos );
        lc->setMaxSpace( 0 );
        lc->setMinSpace( 0 );
        lc->setBorderWidth( 0 );
        lc2 = lc->add( new AContainer( mainwin, 1, 2 ), 1, 0 );
        lc2->setMaxSpace( 0 );
        lc2->setMinSpace( 0 );
        lc2->setBorderWidth( 0 );
        lc2->add( listerwin[0], 0, 0 );
        lc2->add( listerwin[1], 0, 1 );
        
        bc = lc->add( new AContainer( mainwin, 1 , ( columns + 1 ) * rows ), 0, 0 );
        bc->setMaxSpace( 0 );
        bc->setMinSpace( 0 );
        bc->setBorderWidth( 0 );
        blCount++;
        
        lc->setWWeight( 4, 1, 0 );
        l = getMaxButtonsWidth() + 4;
        lc->setMaxWidth( l, 0, 0 );
      }
    } else if ( *it1 == WConfig::LO_LB ) {
      if ( ( bv == true ) && ( lvv == true ) && ( blCount < 1 ) ) {
        lc = maincont->add( new AContainer( mainwin, 2, 1 ), 0, pos );
        lc->setMaxSpace( 0 );
        lc->setMinSpace( 0 );
        lc->setBorderWidth( 0 );
        lc2 = lc->add( new AContainer( mainwin, 1, 2 ), 0, 0 );
        lc2->setMaxSpace( 0 );
        lc2->setMinSpace( 0 );
        lc2->setBorderWidth( 0 );
        lc2->add( listerwin[0], 0, 0 );
        lc2->add( listerwin[1], 0, 1 );
        
        bc = lc->add( new AContainer( mainwin, 1, ( columns + 1 ) * rows ), 1, 0 );
        bc->setMaxSpace( 0 );
        bc->setMinSpace( 0 );
        bc->setBorderWidth( 0 );
        blCount++;
        
        lc->setWWeight( 4, 0, 0 );

        l = getMaxButtonsWidth() + 4;
        lc->setMaxWidth( l, 1, 0 );
      }
    }
    it1++;
  }
  
  if ( ( curRows != 0 ) && ( curColumns != 0 ) ) {
    for ( i = 0; i < curRows; i++ ) {
      for( unsigned int j = 0; j < curColumns; j++ ) {
        mainwin->remove(buttons[ i * curColumns + j ]);
        delete buttons[ i * curColumns + j ];
        buttons[ i * curColumns + j ] = NULL;
      }
    }
    delete [] buttons;
    for(i=0;i<curRows;i++) {
      mainwin->remove(pathbs[i]);
      delete pathbs[ i ];
      pathbs[ i ] = NULL;
    }
    delete [] pathbs;
  }

  font = aguix->getFont( wconfig->getFont(1) );

  th = aguix->getCharHeight() + 4;
  aboutb->resize( th, th );
  configureb->resize( th, th );
  statebar->resize( statebar->getWidth(), th );
  if ( sbc != NULL ) sbc->readLimits();
  
  clockbar->resize( clockbar->getWidth(), th );
  if ( clbc != NULL ) clbc->readLimits();
  
  if ( font != NULL ) th = font->getCharHeight() + 4;
  buttons = new Button*[ columns * rows ];
  for ( i2 = 0; i2 < rows; i2++ ) {
    for ( int j = 0; j < columns; j++ ) {
      if ( bv == true ) {
        x = 0;
        y = ( j + 1 ) * rows + i2;
      } else {
        x = j + 1;
        y = i2;
      }
    
      buttons[ i2 * columns + j ] = new Button( aguix, 0, 0, th, th, "", 1, 0, 0 );
      buttons[ i2 * columns + j ]->setAcceptFocus( false );
      if ( font != NULL ) buttons[ i2 * columns + j ]->setFont( wconfig->getFont(1) );

      if ( bc != NULL ) bc->add( buttons[ i2 * columns + j ], x, y, ( bv == false ) ? conlyh : 0 );
    }
  }
  pathbs = new Button*[ rows ];
  for ( i2 = 0; i2 < rows; i2++ ) {
    pathbs[i2] = new Button( aguix, 0, 0, th, "", "", 1, 1, 0, 0, 0 );
    pathbs[i2]->setAcceptFocus( false );
    pathbs[i2]->resize( th, th );
    if ( font != NULL ) pathbs[i2]->setFont( wconfig->getFont(1) );
    pathbs[i2]->setShowDual(false);

    if ( bc != NULL ) bc->add( pathbs[i2], 0, i2, ( bv == false ) ? conlyh : 0 );
  }
  
  if ( clockbarCount > 0 ) {
    clockbar->show();
  } else {
    clockbar->hide();
  }
  if ( ( listviewsCount < 1 ) && ( bllCount < 1 ) && ( blCount < 1 ) ) {
    listerwin[0]->hide();
    listerwin[1]->hide();
  } else if ( ( listviewsCount == 1 ) && ( lvv == true ) ) {
    listerwin[0]->show();
    listerwin[1]->hide();
  } else {
    listerwin[0]->show();
    listerwin[1]->show();
  }

  curRows=rows;
  curColumns=columns;
  showPathBank();
  showButtonBank();
  if ( isRoot == false ) {
    aboutb->setFG( 0, wconfig->getStatusbar(0) );
    aboutb->setBG( 0, wconfig->getStatusbar(1) );
    configureb->setFG( 0, wconfig->getStatusbar(0) );
    configureb->setBG( 0, wconfig->getStatusbar(1) );
    statebar->setFG( 0, wconfig->getStatusbar(0) );
    statebar->setBG( 0, wconfig->getStatusbar(1) );
    statebar->setFG( 1, wconfig->getStatusbar(0) );
    statebar->setBG( 1, wconfig->getStatusbar(1) );
  } else {
    aboutb->setFG( 0, 2 );
    aboutb->setBG( 0, 4 );
    configureb->setFG( 0, 2 );
    configureb->setBG( 0, 4 );
    statebar->setFG( 0, 2 );
    statebar->setBG( 0, 4 );
    statebar->setFG( 1, 2 );
    statebar->setBG( 1, 4 );
  }
  clockbar->setFG( 0, wconfig->getClockbar(0) );
  clockbar->setBG( 0, wconfig->getClockbar(1) );
  clockbar->setFG( 1, wconfig->getClockbar(0) );
  clockbar->setBG( 1, wconfig->getClockbar(1) );
  maincont->resize( w, h );
  maincont->rearrange();

  AGMessage *msg=new AGMessage;
  msg->type = AG_SIZECHANGED;
  msg->size.window = mainwin->getWindow();
  msg->size.neww = w;
  msg->size.newh = h;
  if ( lister[0] != NULL ) lister[0]->messageHandler( msg );
  if ( lister[1] != NULL ) lister[1]->messageHandler( msg );
  delete msg;

  FieldListView::changeButtonFunction( FieldListView::SELECT_BUTTON, wconfig->getMouseSelectButton() );
  FieldListView::changeButtonFunction( FieldListView::ACTIVATE_BUTTON,
                                       wconfig->getMouseActivateButton(),
                                       wconfig->getMouseActivateMod() );
  FieldListView::changeButtonFunction( FieldListView::SCROLL_BUTTON,
                                       wconfig->getMouseScrollButton(),
                                       wconfig->getMouseScrollMod() );
  FieldListView::changeButtonFunction( FieldListView::ENTRY_PRESSED_MSG_BUTTON,
                                       wconfig->getMouseContextButton(),
                                       wconfig->getMouseContextMod() );
  FieldListView::changeSelectMode( ( wconfig->getMouseSelectMethod() == WConfig::MOUSECONF_ALT_MODE ) ? FieldListView::MOUSE_SELECT_ALT : FieldListView::MOUSE_SELECT_NORMAL );

  mainwin->show();
}

void Worker::setStatebarText(const char *str)
{
  if ( statebar != NULL ) {
    statebar->setText( 0, str );
    last_statebar_text[0] = last_statebar_text[1];
    last_statebar_text[1] = str;
  }
}

void Worker::resetStatebarText()
{
  std::string str1;

  str1 = last_statebar_text[0];
  setStatebarText( str1.c_str() );
}

void Worker::shufflePath()
{
  pbanknr++;
  showPathBank();
}

void Worker::shuffleButton( int dir )
{
  shuffleButton( dir, false );
}

void Worker::shuffleButton( int dir, bool nocycle )
{
  int rows, columns, maxbank, oldbank;

  rows = wconfig->getRows();
  columns = wconfig->getColumns();
  List *buttonlist = wconfig->getButtons();
  maxbank = buttonlist->size() / ( rows * columns * 2 );

  oldbank = bbanknr;
  bbanknr += dir;
  if ( bbanknr < 0 ) {
    bbanknr = ( ( nocycle == false ) ? ( maxbank - 1 ) : 0 );
  } else if ( bbanknr >= maxbank ) {
    bbanknr = ( ( nocycle == false ) ? 0 : ( maxbank - 1 ) );
  }
  if ( oldbank != bbanknr ) showButtonBank();
}

void Worker::showPathBank()
{
  int rows,id,maxbank;
  List *paths=wconfig->getPaths();
  rows=wconfig->getRows();
  maxbank=paths->size()/rows;
  if(pbanknr>=maxbank) pbanknr=0;
  id=paths->initEnum();
  WCPath *p1=(WCPath*)paths->getElementAt(id,pbanknr*rows);
  for(int i=0;i<rows;i++) {
    if( p1->getCheck() == true ) {
      pathbs[i]->setText(0,p1->getName());
      pathbs[i]->setFG(0,p1->getFG());
      pathbs[i]->setBG(0,p1->getBG());
      pathbs[i]->setText(1,p1->getName());
      pathbs[i]->setFG(1,p1->getFG());
      pathbs[i]->setBG(1,p1->getBG());
      pathbs[i]->activate(0);
    } else {
      pathbs[i]->setText(0,"");
      pathbs[i]->setFG(0,0);
      pathbs[i]->setBG(0,0);
      pathbs[i]->setText(1,"");
      pathbs[i]->setFG(1,0);
      pathbs[i]->setBG(1,0);
      pathbs[i]->deactivate(0);
    }
    p1=(WCPath*)paths->getNextElement(id);
  }
  paths->closeEnum(id);
}

void Worker::showButtonBank()
{
  int rows,columns,id,maxbank;
  rows=wconfig->getRows();
  columns=wconfig->getColumns();
  List *buttonlist=wconfig->getButtons();
  maxbank=buttonlist->size()/(rows*columns*2);
  if(bbanknr>=maxbank) bbanknr=0;
  id=buttonlist->initEnum();
  WCButton *b1=(WCButton*)buttonlist->getElementAt(id,bbanknr*rows*columns*2);
  for(int i=0;i<rows;i++) {
    for(int j=0;j<columns;j++) {
      if( b1->getCheck() == true ) {
        buttons[i*columns+j]->setText(0,b1->getText());
        buttons[i*columns+j]->setFG(0,b1->getFG());
        buttons[i*columns+j]->setBG(0,b1->getBG());
        // only activate if this button has some commands
        if ( b1->getComs()->size() > 0 )
          buttons[i*columns+j]->activate(0);
        else
          buttons[i*columns+j]->deactivate(0);
      } else {
        buttons[i*columns+j]->setText(0,"");
        buttons[i*columns+j]->setFG(0,0);
        buttons[i*columns+j]->setBG(0,0);
        buttons[i*columns+j]->deactivate(0);
      }
      b1=(WCButton*)buttonlist->getNextElement(id);
      if(b1->getCheck()==true) {
        buttons[i*columns+j]->setDualState(true);
        buttons[i*columns+j]->setShowDual(true);
        buttons[i*columns+j]->setText(1,b1->getText());
        buttons[i*columns+j]->setFG(1,b1->getFG());
        buttons[i*columns+j]->setBG(1,b1->getBG());
        buttons[i*columns+j]->activate(1);
      } else {
        buttons[i*columns+j]->setDualState(false);
        buttons[i*columns+j]->setShowDual(false);
        buttons[i*columns+j]->deactivate(1);
      }
      b1=(WCButton*)buttonlist->getNextElement(id);
    }
  }
  buttonlist->closeEnum(id);
}

void Worker::about()
{
  char *tstr;
  std::string str1, str2;
  int erg;
  
  tstr = (char*)_allocsafe( 1024 + 3 * A_BYTESFORNUMBER( int ) + strlen( WORKER_COMMENT ) );
  sprintf(tstr,"Worker %d.%d.%d%s|Copyright (C) 1998 - 2009 Ralf Hoffmann",WORKER_MAJOR,WORKER_MINOR,WORKER_PATCH,WORKER_COMMENT);
  str1 = tstr;
  _freesafe( tstr );
  
  str1 += "||Regular expressions for filetype-pattern: ";
#ifdef HAVE_REGEX
  str1 += "yes";
#else
  str1 += "no";
#endif

  str1 += "|Large file support: ";
#ifdef WORKER_LFS_SUPPORTED
  str1 += "yes";
#else
  str1 += "no";
#endif

  str1 += "|Virtual file system (AVFS) usage: ";
#ifdef HAVE_AVFS
  str1 += "yes";
#else
  str1 += "no";
#endif

  str1 += "|UTF8 support: ";
#ifdef DISABLE_UTF8_SUPPORT
  str1 += "no";
#else
  str1 += "yes";
  str1 += "|  current encoding: ";
  switch ( UTF8::checkSupportedEncodings() ) {
    case UTF8::ENCODING_UTF8:
      str1 += "utf8";
      break;
    default:
      str1 += "8 bit";
      break;
  }
#endif
  str1 += "|File magic support: ";
#ifdef HAVE_LIBMAGIC
  str1 += "yes";
#else
  str1 += "no";
#endif


#ifdef USE_MEM_SYSTEM
  str1 += "|This program is compiled with SLOW memorysystem";
#endif
#ifdef DEVELOPER
  str1 += "|This program is compiled as developer version|  If you aren't a developer, please notice the author about it!";
#endif
#ifdef DEBUG
  str1 += "|This program is compiled as debugging version|  If you aren't a developer, please notice the author about it!";
#endif
  str1 += "||Homepage: http://www.boomerangsworld.de/worker|EMail: Ralf Hoffmann <ralf@boomerangsworld.de>";
  str1 += "||Released under the GNU General Public License:||";
  str1 += "This program is free software; you can redistribute it and/or modify|";
  str1 += "it under the terms of the GNU General Public License as published by|";
  str1 += "the Free Software Foundation; either version 2 of the License, or|";
  str1 += "(at your option) any later version.||";
  str1 += "This program is distributed in the hope that it will be useful,|";
  str1 += "but WITHOUT ANY WARRANTY; without even the implied warranty of|";
  str1 += "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the|";
  str1 += "GNU General Public License for more details.";
  
  str2 = catalog.getLocale( 508 );
  str2 += "|";
  str2 += catalog.getLocale( 11 );
  do {
    erg = req_static->request( catalog.getLocale( 208 ), str1.c_str() , str2.c_str() );
    if ( erg == 0 ) showLicense();
  } while ( erg != 1 );
}

bool Worker::configure()
{
  std::string oldsbt;
  bool erg;

  oldsbt = statebar->getText( 0 );
  setStatebarText( catalog.getLocale( 595 ) );
  setWaitCursor();
  erg = wconfig->configure();

  setStatebarText( oldsbt.c_str() );
  unsetWaitCursor();

  restart = erg;
  configstarted = true;

  return erg;
}

Lister *Worker::getOtherLister(Lister *l1)
{
  if(l1==lister[0]) return lister[1];
  if(l1==lister[1]) return lister[0];
  return NULL;
}

int Worker::getSide(Lister *l1)
{
  if(l1==lister[0]) return 0;
  if(l1==lister[1]) return 1;
  return -1;
}

void Worker::setPath(int nr)
{
  List *paths=wconfig->getPaths();
  WCPath *p1=(WCPath*)paths->getElementAt(nr);
  if(p1!=NULL) {
    setPath( p1 );
  } else {
    printf("Unexpected error in Worker::setPath 1\n");
  }
}

void Worker::setPath( WCPath *p )
{
  char *tstr;

  if ( p != NULL ) {
    if(p->getCheck()==true) {
      Lister *l1=lister[0];
      if(lister[1]->isActive()==true) l1=lister[1];
      l1->switch2Mode(0);
      NormalMode *nm=(NormalMode*)l1->getActiveMode();
      if(nm!=NULL) {
        tstr = Worker_replaceEnv( p->getPath() );
        nm->enterDir( tstr );
        _freesafe( tstr );
      } else {
        printf("Unexpected error in Worker::setPath 2\n");
      }
    }
  }
}

int Worker::interpret(List* coms,ActionMessage *msg)
{
#if 0
  FunctionProto *fp;
  int id=coms->initEnum();
  fp=(FunctionProto*)coms->getFirstElement(id);
  while(fp!=NULL) {
    fp->run(msg);
    fp=(FunctionProto*)coms->getNextElement(id);
  }
  coms->closeEnum(id);
#else
  WPUContext *wpu;
  List *parentActionList = NULL;

  if ( ( coms == NULL ) || ( msg == NULL ) ) return 1;

  if ( m_interpret_recursion_depth > m_interpret_max_recursion ) {
      if ( getRequester() != NULL ) {
          getRequester()->request( catalog.getLocale( 125 ), catalog.getLocale( 821 ), catalog.getLocale( 11 ) );
      } else {
          fprintf( stderr, "%s", catalog.getLocale( 821 ) );
      }
      return 1;
  }
  m_interpret_recursion_depth++;
  
  mainwin->forbidUserInput();
  usr1signal = 0;

  if ( ( coms->size() < 1 ) && ( msg->filetype != NULL ) ) {
    // no action and filetype, so use list with parentaction
    parentActionList = new List();
    parentActionList->addElement( new ParentActionOp() );
    coms = parentActionList;
  }

  wpu = new WPUContext( this, coms, msg );
  while( wpu->next( msg ) == 0 ) {
    if ( usr1signal != 0 ) {
      //Command execution stopped by USR1 signal
      if ( getRequester() != NULL ) {
        getRequester()->request( catalog.getLocale( 124 ), catalog.getLocale( 503 ), catalog.getLocale( 11 ) );
      }
      usr1signal = 0;
      break;
    }
  }
  delete wpu;

  if ( parentActionList != NULL ) {
    int id;
    FunctionProto *fp1;

    id = parentActionList->initEnum();
    fp1 = (FunctionProto*)parentActionList->getFirstElement( id );
    while ( fp1 != NULL ) {
      delete fp1;
      fp1 = (FunctionProto*)parentActionList->getNextElement( id );
    }
    parentActionList->closeEnum( id );
    delete parentActionList;
  }

  mainwin->permitUserInput();
  m_interpret_recursion_depth--;
#endif
  return 0;
}

static bool isDoubleKeyTime( Time t2, Time t1 )
{
    if ( abs( (int)( t2 - t1 ) ) < ( 60 * 1000 ) ) return true;
    return false;
}

int Worker::activateShortkey(AGMessage *msg)
{
  int keystate;
  WCHotkey *hk1;
  WCButton *b1;
  WCPath *p1;
  ActionMessage amsg( this );
  bool found;
  if((msg->type==AG_KEYPRESSED)||(msg->type==AG_KEYRELEASED)) {

      amsg.setKeyAction( true );

    if ( msg->type == AG_KEYPRESSED ) {
      // I will ignore here any modifier key because
      // it makes no sense (for me) to allow a single modifier press
      // release is still supported
      // reason for this is the double shortkey: e.g. ctrl-x
      // will send first "ctrl press" and then "ctrl-x press"
      // and this would lead to wrong results
      // other way could be to just dont save this modifier presses in lastkey
      if ( AGUIX::isModifier( msg->key.key ) == true )
        return 0;
    }

    keystate=KEYSTATEMASK(msg->key.keystate);
    found=false;

    /* procedure:
     * a previous key present?
     *   no: Check for normal key
     *     found: execute
     *     not found: store key as previous
     *   yes: Check for double key
     *     found: execute and clear previous
     *     not found: Check for normal key
     *       found: execute and clear previous
     *       not found: store key as previous
     */

    if ( lastkey != 0 && isDoubleKeyTime( msg->key.time, lastkey_time ) == true ) {
        if ( findkey( lastkey, lastmod, msg->key.key, keystate, &b1, &hk1, &p1 ) > 0 ) {
            if ( b1 != NULL ) {
                amsg.mode = amsg.AM_MODE_NORMAL;
                interpret( b1->getComs(), &amsg );
                found = true;
            } else if ( hk1 != NULL ) {
                amsg.mode = amsg.AM_MODE_NORMAL;
                interpret( hk1->getComs(), &amsg );
                found = true;
            } else if ( p1 != NULL ) {
                setPath( p1 );
                found = true;
            }
            lastkey = 0;
            lastmod = 0;
        }
    }
    if ( found == false ) {
        if ( findkey( msg->key.key, keystate, &b1, &hk1, &p1 ) > 0 ) {
            if ( b1 != NULL ) {
                amsg.mode = amsg.AM_MODE_NORMAL;
                interpret( b1->getComs(), &amsg );
                found = true;
            } else if ( hk1 != NULL ) {
                amsg.mode = amsg.AM_MODE_NORMAL;
                interpret( hk1->getComs(), &amsg );
                found = true;
            } else if ( p1 != NULL ) {
                setPath( p1 );
                found = true;
            }
            lastkey = 0;
            lastmod = 0;
        } else {
            lastkey = msg->key.key;
            lastmod = keystate;
            lastkey_time = msg->key.time;
        }
    }
    return ( found == true ) ? 1 : 0;
  }
  return 0;
}

Lister *Worker::getLister(int side)
{
  if((side<0)||(side>1)) return NULL;
  return lister[side];
}

Lister *Worker::getActiveLister()
{
  if(lister[0]->isActive()==true) return lister[0];
  if(lister[1]->isActive()==true) return lister[1];
  return NULL;
}

void Worker::activateButton(int i,int j,int m)
{
  List *l1;
  WCButton *b1;
  int id,pos;
  int rows,columns;
  ActionMessage amsg( this );
  rows=wconfig->getRows();
  columns=wconfig->getColumns();
  l1=wconfig->getButtons();
  id=l1->initEnum();
  pos=2*(bbanknr*rows*columns+i*columns+j)+m-1;
  b1=(WCButton*)l1->getElementAt(id,pos);
  if(b1!=NULL) {
    if(b1->getCheck()==true) {
      amsg.mode=amsg.AM_MODE_NORMAL;
      interpret(b1->getComs(),&amsg);
    }
  }
  l1->closeEnum(id);
}

int Worker::saveListerState()
{
  // Datei oeffnen ($HOME/.worker/listerstates)
  // int-Wert mit Seite (0/1)
  // Vielleicht: Die ListerModes nicht durchnumeriert laden sondern
  //             die ID speichern und dann dem entsprechenden Mode
  //             das laden ueberlassen, nach Seitennummer die Anzahl der
  //             der abgespeicherten Modis
  // Also der Lister-lade-funktion den FH geben und die macht dann den Rest

  const char *home = wconfig->getHome();
  int len = strlen( home );
  std::string str1;
  Datei *fh;
  
  if ( len < 1 ) return false;
  
  str1 = home;
  if ( len > 1 ) if ( home[ len - 1 ] != '/' ) str1 += "/";
#ifdef USEOWNCONFIGFILES
  str1 += "lvconf-dev";
#else
  str1 += "lvconf";
#endif

  fh = new Datei();
  if ( fh->open( str1.c_str(), "w" ) == 0 ) {
    if ( lister[1]->getFocus() == true ) {
      fh->configPutPair( "activeside", "right" );
    } else {
      fh->configPutPair( "activeside", "left" );
    }
    fh->configOpenSection( "left" );
    lister[0]->saveState( fh );
    fh->configCloseSection(); //left
    fh->configOpenSection( "right" );
    lister[1]->saveState( fh );
    fh->configCloseSection(); //right

    if ( fh->errors() != 0 ) {
      //TODO: Requester zeigen?
      fprintf( stderr, "Worker:error while writing listview configuration (lvconf)\n" );
    }
    fh->close();
  }
  delete fh;
  return 0;
}

int Worker::initListerState()
{
  int pos,s;
  char *startd[2], *tstr;
  const char *home = wconfig->getHome();
  int len = strlen( home );
  std::string str1;
  Datei *fh;
  FILE *fp;
  int rside;
  char *actdir;
  int size = 4096, i, activeside;
  int found_error = 0;
  
  if ( len < 1 ) return 1;

  str1 = home;
  if ( len > 1 ) if ( home[ len - 1 ] != '/' ) str1 += "/";
#ifdef USEOWNCONFIGFILES
  str1 += "lvconf-dev";
#else
  str1 += "lvconf";
#endif

  activeside = -1;
  fp = worker_fopen( str1.c_str(), "r" );
  if ( fp != NULL ) {
    yyrestart( fp );
    readtoken();
    for (;;) {
      if ( worker_token == ACTIVESIDE_WCP ) {
        readtoken();

        if ( worker_token != '=' ) {
          found_error = 1;
          break;
        }
        readtoken();

        if ( worker_token == LEFT_WCP ) {
          activeside = 0;
        } else if ( worker_token == RIGHT_WCP ) {
          activeside = 1;
        } else {
          found_error = 1;
          break;
        }
        readtoken();

        if ( worker_token != ';' ) {
          found_error = 1;
          break;
        }
        readtoken();
      } else if ( worker_token == LEFT_WCP ) {
        readtoken();

        if ( worker_token != LEFTBRACE_WCP ) {
          found_error = 1;
          break;
        }
        readtoken();
        
        if ( lister[0]->loadState() != 0 ) {
          found_error = 1;
          break;
        }
        
        if ( worker_token != RIGHTBRACE_WCP ) {
          found_error = 1;
          break;
        }
        readtoken();
      } else if ( worker_token == RIGHT_WCP ) {
        readtoken();

        if ( worker_token != LEFTBRACE_WCP ) {
          found_error = 1;
          break;
        }
        readtoken();
        
        if ( lister[1]->loadState() != 0 ) {
          found_error = 1;
          break;
        }
        
        if ( worker_token != RIGHTBRACE_WCP ) {
          found_error = 1;
          break;
        }
        readtoken();
      } else if ( worker_token != 0 ) {
        // parse error
        found_error = 1;
        break;
      } else break;  // end
    }
    if ( found_error != 0 ) {
      //TODO: Requester zeigen?
      //      Eigentlich kann der User eh nicht viel machen
      //      denn beim Beenden wird neue Datei geschrieben
      fprintf( stderr, "Worker:error in listview configuration (lvconf)\n" );
    }
    worker_fclose( fp );
  } else {
    fh = new Datei();
    str1 = home;
    if ( len > 1 ) if ( home[ len - 1 ] != '/' ) str1 += "/";
    str1 += "listerstates";

    if ( fh->open( str1.c_str(), "r" ) == 0 ) {
      for ( i = 0; i < 2; i++ ) {
        rside = fh->getInt();
        if ( ( rside >= 0 ) && ( rside < 2 ) ) {
          lister[rside]->loadBinState( fh );
        }
      }
      fh->close();
    }
    delete fh;
  }

  if ( ( activeside == 0 ) || ( activeside == 1 ) ) {
    lister[activeside]->makeActive();
  }
  // TODO: Vielleicht nur diesen check, wenn Datei nicht gefunden oder beide auf -1
  int lmode=-1,rmode=-1;
  if(lister[0]->getActiveMode()!=NULL) {
    lmode=getID4Mode(lister[0]->getActiveMode());
  }
  if(lister[1]->getActiveMode()!=NULL) {
    rmode=getID4Mode(lister[1]->getActiveMode());
  }
  if((lmode!=0)&&(rmode!=0)) {
    lister[0]->switch2Mode(0);
    lister[1]->switch2Mode(0);
  }
  //now check for startdirs
  startd[0]=NULL;
  startd[1]=NULL;
  pos=1;
  for(s=0;s<2;s++) {
    for(;pos<argc;pos++) {
      if(strncmp(argv[pos],"-",1)!=0) {
        if(strncmp(argv[pos],"--",2)!=0) {
          startd[s]=dupstring(argv[pos]);
          break;
        }
      }
    }
    pos++;
  }

  for(;;) {
    actdir=(char*)_allocsafe(size);
    if(getcwd(actdir,size)!=NULL) break;
    else if(errno!=ERANGE) {
      strcpy(actdir,"/");  // use the root if getcwd doesn't report the actual
      break;
    }
    _freesafe(actdir);
    size*=2;
  }

  for(s=0;s<2;s++) {
    if(startd[s]==NULL) {
      tstr=wconfig->getStartDir(s);
      if(tstr!=NULL) startd[s]=dupstring(tstr);
    }
    if(startd[s]!=NULL) {
      if(strlen(startd[s])>0) {
        lister[s]->switch2Mode(0);
        NormalMode *nm=(NormalMode*)lister[s]->getActiveMode();
        if(nm!=NULL) {
	  tstr = HandlePathExt( startd[s] );

          if ( tstr[0] != '/' ) {
	    str1 = actdir;
	    str1 += "/";
	    str1 += tstr;
          } else {
	    str1 = tstr;
          }
          nm->enterDir( str1.c_str() );
          _freesafe(tstr);
        }
      }
      _freesafe(startd[s]);
    }
  }
  _freesafe(actdir);
  
  /* if no side is active make 0 active */
  if(getActiveLister()==NULL) lister[0]->makeActive();
  return 0;
}

int Worker::quit(int mode)
{
  int returnvalue=0;
  if(mode==1) {
    runningmode=WORKER_QUIT;
    returnvalue=1;
  } else {
    char *str;
    str=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+strlen(catalog.getLocale(8))+1);
    sprintf(str,"%s|%s",catalog.getLocale(11),catalog.getLocale(8));
    int res=req_static->request(catalog.getLocale(114),catalog.getLocale(115),str);
    _freesafe(str);
    if(res==0) {
      runningmode=WORKER_QUIT;
      returnvalue=1;
    }
  }
  return returnvalue;
}

void Worker::updateTime(void)
{
  struct tm *timeptr;
#if defined HAVE_SYS_SYSINFO_H && defined HAVE_SYSINFO
  struct sysinfo info;
  unsigned long freeram,freeswap;
#endif
  time_t timep;
  char *str,*buffer;
  const char *clbc;
  int dt=wconfig->getClockbarUpdatetime();
  WConfig::clockbar_mode_t cbm=wconfig->getClockbarMode();
  int tlen;
  
  time(&timep);
  if ( ( lasttimep == 0 ) ||
       ( ( ( lasttimep + dt ) <= timep ) &&
         ( cbm != WConfig::CLOCKBAR_MODE_VERSION )
       )
     ) {
    if(cbm==WConfig::CLOCKBAR_MODE_VERSION) {
      buffer = (char*)_allocsafe( strlen( "Worker %d.%d.%d%s - (C) Ralf Hoffmann" ) +
                                  strlen( WORKER_COMMENT ) + 3 * A_BYTESFORNUMBER( int ) );
      sprintf(buffer,"Worker %d.%d.%d%s - (C) Ralf Hoffmann",WORKER_MAJOR,WORKER_MINOR,WORKER_PATCH,WORKER_COMMENT);
    } else if(cbm==WConfig::CLOCKBAR_MODE_EXTERN) {
      clbc=wconfig->getClockbarCommand();
      if(strlen(clbc)>0) {
        // start the program
        ExeClass *myexe = new ExeClass( ExeClass::EXECUTE_NORMAL );
        myexe->addCommand( "%s", clbc );
        buffer = myexe->getOutput();
        if ( buffer != NULL ) {
	  tlen = aguix->getStrlen4Width( buffer, clockbar->getWidth() - 10, NULL );
          if ( (int)strlen( buffer ) > tlen ) {
            buffer[tlen] = '\0';
          }
        }
        delete myexe;
      } else buffer=dupstring("");
    } else { // _TIME or _TIMERAM
      timeptr=localtime(&timep);
      str=asctime(timeptr);
      str[strlen(str)-1]=0;
#if defined HAVE_SYS_SYSINFO_H && defined HAVE_SYSINFO
      if(cbm==WConfig::CLOCKBAR_MODE_TIMERAM) {
        sysinfo(&info);
        freeram = info.freeram / 1024;
        freeram *= info.mem_unit;
        freeswap = info.freeswap / 1024;
        freeswap *= info.mem_unit;
        
        buffer=(char*)_allocsafe(strlen(str)+
                                 strlen("  --  %s: %lu KB  -  %s: %lu KB")+
                                 strlen(catalog.getLocale(105))+strlen(catalog.getLocale(106))+
                                 2 * ( A_BYTESFORNUMBER( unsigned long ) ) + 1);
        
        strcpy(buffer,str);
        sprintf( buffer + strlen( str ),
                 "  --  %s: %lu KB  -  %s: %lu KB",
                 catalog.getLocale( 105 ), freeram,
                 catalog.getLocale( 106 ), freeswap );
      } else {
        buffer=(char*)_allocsafe(strlen(str)+1);
        strcpy(buffer,str);
      }
#else
      buffer=(char*)_allocsafe(strlen(str)+1);
      strcpy(buffer,str);
#endif
    }
    clockbar->setText(0,buffer);
    clockbar->setText(1,buffer);
    _freesafe(buffer);
    lasttimep=timep;
  }
}

void Worker::checkfirststart( char **setlang )
{
  char *homestr,*filestr;
  bool copyconfig=false;
  const char *textstr,*buttonstr;
  int erg;
  bool firststart = false;

  homestr=getenv("HOME");
  if(homestr==NULL) {
    textstr="There is no HOME-variable. Please set it to your home and retry.";
    buttonstr="Ok";
    req_static->request( "Worker warning", textstr, buttonstr );
  } else {
    filestr=catstring(homestr,"/.worker/config");
    if(Datei::fileExists(filestr)==false) {
      firststart = true;
    
      textstr="It seems that Worker is started the first time!|I will try to install an example configuration in $HOME/.worker|but I will not overwrite any existing file!||If you choose \"Skip\", Worker will use a simple builtin config\
|but then not all functions are accessable (without reconfiguration)!";
      buttonstr="Ok|Skip";
      erg = req_static->request( "Worker message", textstr, buttonstr );
      if(erg==0) copyconfig=true;
    }
    if(copyconfig==true) {
      std::string catdir, str2, str3;
      std::string chosen_lang;

      chosen_lang = queryLanguage();

      catdir = homestr;
      catdir += "/.worker";
      if(Datei::fileExists( catdir.c_str() )==false) {
        worker_mkdir( catdir.c_str(), 0700 );
      }

      catdir = homestr;
      catdir += "/.worker/catalogs";
      if(Datei::fileExists( catdir.c_str() )==false) {
        worker_mkdir( catdir.c_str(), 0700 );
      }

      if ( Worker::getDataDir().length() > 0 ) {
        str2 = Worker::getDataDir();
        str2 += "/config-";
        str2 += chosen_lang;
        if ( UTF8::getUTF8Mode() != UTF8::UTF8_DISABLED && UTF8::getCurrentEncoding() == UTF8::ENCODING_UTF8 ) {
          str2 += ".utf8";
        }
      } else str2 = "";
      
      if ( Datei::fileExists( str2.c_str() ) == false ) {
        // choosed language has no default config so take the english
        // and set the language later
        if ( Worker::getDataDir().length() > 0 ) {
          str2 = Worker::getDataDir();
          str2 += "/config-english";
        } else str2 = "";
        
        if ( setlang != NULL ) {
          // if not english remember to set the language later
          if ( chosen_lang != "english" )
            *setlang = dupstring( chosen_lang.c_str() );
        }
      }
      
      if(Datei::fileExists( str2.c_str() )==true) {
        char *tstr1, *tstr2, *tstr3;
        tstr1 = AGUIX_catTrustedAndUnTrusted( "cp ", str2.c_str() );
        tstr2 = catstring( tstr1, " " );
        tstr3 = AGUIX_catTrustedAndUnTrusted( tstr2, filestr );
        system( tstr3 );
        _freesafe( tstr1 );
        _freesafe( tstr2 );
        _freesafe( tstr3 );
        if(Datei::fileExists(filestr)==false) {
          textstr="The configfile couldn't be copyied!";
          buttonstr="Ok";
          req_static->request( "Worker error", textstr, buttonstr );
        } else {
          textstr="Installation finished.";
          buttonstr="Ok";
          req_static->request( "Worker message", textstr, buttonstr );
        }
      } else {
        str3 = "There is no example configuration at " + str2 + "!|Perhaps worker is not completely installed!";
        buttonstr="Ok";
        req_static->request( "Worker error", str3.c_str(), buttonstr );
      }
    }
    _freesafe(filestr);
  }
  if ( firststart == true ) about();
}

void Worker::setTitle(const char *add_infos)
{
  if(add_infos==NULL) {
    if ( isRoot == false )
      mainwin->setTitle("Worker");
    else
      mainwin->setTitle( "rootWorker" );
  } else {
    char *tstr;
    
    if ( isRoot == false )
      tstr=catstring("Worker - ",add_infos);
    else
      tstr=catstring("rootWorker - ",add_infos);
    mainwin->setTitle(tstr);
    _freesafe(tstr);
  }
}

Requester *Worker::getRequester()
{
  if ( req_static == NULL ) {
    req_static = new Requester( aguix );
  }
  return req_static;
}

void Worker::activateShortkeyFromList()
{
  // 1.alle moeglichen Shortkeys finden und in eine Liste packen
  // 2.Diese Liste sortieren
  //   Hotkey<Button<Pfad   (Hotkey zuerst, weil man die am schnellsten vergisst)
  //   Unter gleichem Typ einfach nach Name case-ins sortieren
  // 3.Aus dieser Liste wird nun das LV erstellt:
  //   <Typ>:<Name>   <Shortkey>
  //   Dabei <Typ>:<Name> maximieren, damit Shortkey ganz rechts steht
  //   Achtung: Vielleicht fuer das LV den Font des linken LVs nehmen, falls der User doch mal
  //            einen nichtpro. Font einstellt, dann klappt das auch hier
  // 4.lv Horizontal maximieren
  // 5.lv->takeFocus(), damit man das ganze auch einfach mit der Tastatur benutzen kann
  // 6.der User waehlt nun aus
  // 7.Bei Abbruch nichts machen
  // 8.Bei Okay/Return/Doppelklick dann diesen Shortkey aktivieren

  List *sks;
  struct shortkeylisttype *sk1;
  int id;
  List *l1;
  WCHotkey *hk1;
  WCButton *b1;
  WCPath *p1;
  ActionMessage amsg( this );
  AWindow *win;
  Text *ttext;
  FieldListView *lv;
  int tx,ty,lw,tw,l,t1,t2,lh;
  const char *name1, *name2;
  char *tstr1;
  AGMessage *msg;
  int ende=0;
  Button *okb,*cb;
  int lastrow, row;
  struct timeval lastclick;
  List *dkeys;
  int id_k;
  WCDoubleShortkey *dk;
  int used_cols, cur_col;
  std::string str1;

  sks=new List();
  l1=wconfig->getHotkeys();
  id=l1->initEnum();
  hk1=(WCHotkey*)l1->getFirstElement(id);
  while(hk1!=NULL) {
    if ( hk1->getDoubleKeys()->size() > 0 ) {
      sk1=(struct shortkeylisttype*)_allocsafe(sizeof(struct shortkeylisttype));
      sk1->h=hk1;
      sk1->b=NULL;
      sk1->p=NULL;
      sks->addElement(sk1);
    }
    hk1=(WCHotkey*)l1->getNextElement(id);
  }
  l1->closeEnum(id);

  l1=wconfig->getButtons();
  id=l1->initEnum();
  b1=(WCButton*)l1->getFirstElement(id);
  while(b1!=NULL) {
    if ( b1->getDoubleKeys()->size() > 0 ) {
      sk1=(struct shortkeylisttype*)_allocsafe(sizeof(struct shortkeylisttype));
      sk1->h=NULL;
      sk1->b=b1;
      sk1->p=NULL;
      sks->addElement(sk1);
    }
    b1=(WCButton*)l1->getNextElement(id);
  }
  l1->closeEnum(id);

  l1=wconfig->getPaths();
  id=l1->initEnum();
  p1=(WCPath*)l1->getFirstElement(id);
  while(p1!=NULL) {
    if ( p1->getDoubleKeys()->size() > 0 ) {
      sk1=(struct shortkeylisttype*)_allocsafe(sizeof(struct shortkeylisttype));
      sk1->h=NULL;
      sk1->b=NULL;
      sk1->p=p1;
      sks->addElement(sk1);
    }
    p1=(WCPath*)l1->getNextElement(id);
  }
  l1->closeEnum(id);
  
  // sorting
  sks->sort(Worker::shortkeysort);
  
  // create window
  lw=lh=20;
  win = new AWindow( aguix, 10, 10, lw, lh, 0, catalog.getLocale( 387 ) );
  win->create();
  
  tx=ty=5;
  ttext=(Text*)win->add(new Text(aguix,tx,ty,catalog.getLocale(388),1));
  tw=ttext->getWidth();
  if(tw>lw) lw=tw;
  ty+=ttext->getHeight()+5;
  lv = (FieldListView*)win->add( new FieldListView( aguix,
                                                    tx,
                                                    ty,
                                                    50,
                                                    400,
                                                    0 ) );
  lv->setNrOfFields( 4 );
  lv->setFieldWidth( 1, 1 );
  lv->setFieldWidth( 3, 1 );
  used_cols = 4;
  
  // add the shortkeys
  id=sks->initEnum();
  sk1=(struct shortkeylisttype*)sks->getFirstElement(id);
  while(sk1!=NULL) {
    dkeys = NULL;
    name2 = NULL;
    name1 = NULL;
    if(sk1->b!=NULL) {
      dkeys = sk1->b->getDoubleKeys();
      name1=catalog.getLocale(389);
      name2=sk1->b->getText();
    } else if(sk1->h!=NULL) {
      dkeys = sk1->h->getDoubleKeys();
      name1=catalog.getLocale(391);
      name2=sk1->h->getName();
    } else if(sk1->p!=NULL) {
      dkeys = sk1->p->getDoubleKeys();
      name1=catalog.getLocale(390);
      name2=sk1->p->getName();
    }
    if ( ( dkeys != NULL ) && ( name2 != NULL ) ) {
      row = lv->addRow();
      lv->setText( row, 0, name1 );
      lv->setText( row, 2, name2 );
      id_k = dkeys->initEnum();
      dk = (WCDoubleShortkey*)dkeys->getFirstElement( id_k );
      cur_col = 4;
      while ( dk != NULL ) {
        if ( cur_col >= used_cols ) {
          lv->setNrOfFields( cur_col + 1 );
          used_cols = cur_col + 1;
        }
        if ( cur_col > 4 ) {
          // not the first key
          lv->setText( row, cur_col - 1, ", " );
        }

        tstr1 = aguix->getNameOfKey( dk->getKeySym( 0 ), dk->getMod( 0 ) );
        if ( tstr1 != NULL )
          str1 = tstr1;
        else
          str1 = "???";
        _freesafe(tstr1);

        if ( dk->getType() == WCDoubleShortkey::WCDS_DOUBLE ) {
          str1 += " ";
          tstr1 = aguix->getNameOfKey( dk->getKeySym( 1 ), dk->getMod( 1 ) );
          if ( tstr1 != NULL )
            str1 += tstr1;
          else
            str1 += "???";
          _freesafe(tstr1);
        }
        
        lv->setText( row, cur_col, str1 );

        dk = (WCDoubleShortkey*)dkeys->getNextElement( id_k );
        cur_col += 2;
      }
      dkeys->closeEnum( id_k );
      lv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
    }

    sk1=(struct shortkeylisttype*)sks->getNextElement(id);
  }
  sks->closeEnum(id);
  // lv ready
  
  lv->setHBarState(2);
  lv->setVBarState(2);
  lv->maximizeX();
  tw=lv->getWidth();
  if ( tw > ( aguix->getRootWindowWidth() * 90 / 100 ) ) {
    lv->resize( aguix->getRootWindowWidth() * 90 / 100, lv->getHeight() );
    tw =lv->getWidth();
  }
  if(tw>lw) lw=tw;
  ty+=lv->getHeight()+5;
  
  lw+=10;

  t1 = aguix->getTextWidth( catalog.getLocale( 392 ) ) + 10;
  t2 = aguix->getTextWidth( catalog.getLocale( 8 ) ) + 10;
  tw=5+t1+5+t2+5;
  if(tw>lw) lw=tw;
  okb=(Button*)win->add(new Button(aguix,
                                   5,
                                   ty,
                                   t1,
                                   catalog.getLocale(392),
                                   1,
                                   0,
                                   0));
  cb=(Button*)win->add(new Button(aguix,
                                  lw-5-t2,
                                  ty,
                                  t2,
                                  catalog.getLocale(8),
                                  1,
                                  0,
                                  0));
  ty+=okb->getHeight()+5;
  
  okb->takeFocus();
  win->setDoTabCycling( true );
  win->resize(lw,ty);
  win->setMinSize(lw,ty);
  win->show();

  lv->takeFocus();

  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  lastrow = -1;
  while(ende==0) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      if(msg->type==AG_CLOSEWINDOW) {
        if(msg->closewindow.window==win->getWindow()) ende=-1;
      } else if(msg->type==AG_BUTTONCLICKED) {
        if(msg->button.button==okb) ende=1;
        else if(msg->button.button==cb) ende=-1;
      } else if(msg->type==AG_SIZECHANGED) {
        if(msg->size.window==win->getWindow()) {
          lw=msg->size.neww;
          lh=msg->size.newh;
          okb->move(okb->getX(),lh-5-okb->getHeight());
          cb->move(lw-5-cb->getWidth(),lh-5-cb->getHeight());
          lv->resize(lw-10,okb->getY()-lv->getY()-5);
        }
      } else if(msg->type==AG_KEYPRESSED) {
        if ( win->isParent( msg->key.window, false ) == true ) {
          if((msg->key.key==XK_Return)||
             (msg->key.key==XK_KP_Enter)) {
            if ( cb->getHasFocus() == false ) {
              ende=1;
            }
          } else if(msg->key.key==XK_Escape) {
            ende=-1;
          }
        }
      } else if(msg->type==AG_FIELDLV_ONESELECT) {
        if(msg->fieldlv.lv==lv) {
          if(lastrow == -1) {
            lastrow=msg->fieldlv.row;
            gettimeofday(&lastclick,NULL);
          } else {
            if(msg->fieldlv.row == lastrow ) {
              struct timeval acttime;
              gettimeofday(&acttime,NULL);
              if(aguix->isDoubleClick(&acttime,&lastclick)==true) {
                ende=1;
              } else {
                lastrow = msg->fieldlv.row;
                gettimeofday(&lastclick,NULL);
              }
            } else {
              lastrow=msg->fieldlv.row;
              gettimeofday(&lastclick,NULL);
            }
          }
        }
      }
      aguix->ReplyMessage(msg);
    }
  }
  l=-1;
  if(ende==1) {
    l = lv->getActiveRow();
  }
  delete win;

  if(l>=0) {
    sk1=(struct shortkeylisttype*)sks->getElementAt(l);
    if(sk1!=NULL) {
      amsg.mode=amsg.AM_MODE_NORMAL;
      if(sk1->h!=NULL) {
        interpret(sk1->h->getComs(),&amsg);
      } else if(sk1->b!=NULL) {
        interpret(sk1->b->getComs(),&amsg);
      } else if(sk1->p!=NULL) {
        setPath(wconfig->getPaths()->getIndex(sk1->p));
      }
    }
  }
  id=sks->initEnum();
  sk1=(struct shortkeylisttype*)sks->getFirstElement(id);
  while(sk1!=NULL) {
    _freesafe(sk1);
    sks->removeFirstElement();
    sk1=(struct shortkeylisttype*)sks->getFirstElement(id);    
  }
  sks->closeEnum(id);
  delete sks;
}

int Worker::shortkeysort(void*p1,void*p2)
{
  struct shortkeylisttype *sk1=(struct shortkeylisttype*)p1;
  struct shortkeylisttype *sk2=(struct shortkeylisttype*)p2;
  const char *str1,*str2;

  // Hotkey<Button<Pfad
  // Unter gleichem Typ einfach nach Name case-ins sortieren

  if((sk1->h==NULL)&&(sk1->b==NULL)&&(sk1->p==NULL)) return 0;
  if((sk2->h==NULL)&&(sk2->b==NULL)&&(sk2->p==NULL)) return 0;

  // Hotkey ist kleiner als Button & Pfad
  if((sk1->h!=NULL)&&(sk2->h==NULL)) return -1;
  // Button is kleiner Pfad, aber groesser als Hotkey
  if(sk1->b!=NULL) {
    if(sk2->h!=NULL) return 1;
    if(sk2->p!=NULL) return -1;
  }
  // Pfad ist groesser als Hotkey & Button
  if((sk1->p!=NULL)&&(sk2->p==NULL)) return 1;
  
  // Ab hier sind beide vom gleichen Typ
  str1=str2=NULL;
  if(sk1->h!=NULL) {
    str1=sk1->h->getName();
    str2=sk2->h->getName();
  } else if(sk1->b!=NULL) {
    str1=sk1->b->getText();
    str2=sk2->b->getText();
  } else if(sk1->p!=NULL) {
    str1=sk1->p->getName();
    str2=sk2->p->getName();  
  }
  if((str1!=NULL)&&(str2!=NULL)) {
    return strcasecmp(str1,str2);
  }
  return 0;
}

int Worker::PS_readSpace( const char*name )
{
  return freesp->readSpace( name );
}

loff_t Worker::PS_getBlocksize()
{
  return freesp->getBlocksize();
}

loff_t Worker::PS_getFreeSpace()
{
  return freesp->getFreeSpace();
}

loff_t Worker::PS_getSpace()
{
  return freesp->getSpace();
}

std::string Worker::PS_getFreeSpaceH() const
{
  return freesp->getFreeSpaceH();
}

std::string Worker::PS_getSpaceH() const
{
  return freesp->getSpaceH();
}

void Worker::buildtable()
{
  int i, id, id_dk;
  List *l1;
  List *dkeys;
  WCHotkey *hk1;
  WCButton *b1;
  WCPath *p1;
  WCDoubleShortkey *dk;

  if ( keyhashtable != NULL )
    deletetable();
  keyhashtable = new keyhash*[ keyhashtable_size ];
  
  for ( i = 0; i < keyhashtable_size; i++ ) {
    keyhashtable[i] = NULL;
  }

  l1 = wconfig->getHotkeys();
  id = l1->initEnum();
  hk1 = (WCHotkey*)l1->getFirstElement( id );
  while ( hk1 != NULL ) {
    dkeys = hk1->getDoubleKeys();
    id_dk = dkeys->initEnum();
    dk = (WCDoubleShortkey*)dkeys->getFirstElement( id_dk );
    while ( dk != NULL ) {
      insertkey( dk->getKeySym( 0 ), dk->getMod( 0 ), NULL, hk1, NULL );
      dk = (WCDoubleShortkey*)dkeys->getNextElement( id_dk );
    }
    dkeys->closeEnum( id_dk );
    hk1 = (WCHotkey*)l1->getNextElement( id );
  }
  l1->closeEnum( id );

  l1 = wconfig->getButtons();
  id = l1->initEnum();
  b1 = (WCButton*)l1->getFirstElement( id );
  while ( b1 != NULL ) {
    dkeys = b1->getDoubleKeys();
    id_dk = dkeys->initEnum();
    dk = (WCDoubleShortkey*)dkeys->getFirstElement( id_dk );
    while ( dk != NULL ) {
      insertkey( dk->getKeySym( 0 ), dk->getMod( 0 ), b1, NULL, NULL );
      dk = (WCDoubleShortkey*)dkeys->getNextElement( id_dk );
    }
    dkeys->closeEnum( id_dk );
    b1 = (WCButton*)l1->getNextElement( id );
  }
  l1->closeEnum( id );

  l1 = wconfig->getPaths();
  id = l1->initEnum();
  p1 = (WCPath*)l1->getFirstElement( id );
  while ( p1 != NULL ) {
    dkeys = p1->getDoubleKeys();
    id_dk = dkeys->initEnum();
    dk = (WCDoubleShortkey*)dkeys->getFirstElement( id_dk );
    while ( dk != NULL ) {
      insertkey( dk->getKeySym( 0 ), dk->getMod( 0 ), NULL, NULL, p1 );
      dk = (WCDoubleShortkey*)dkeys->getNextElement( id_dk );
    }
    dkeys->closeEnum( id_dk );
    p1 = (WCPath*)l1->getNextElement( id );
  }
  l1->closeEnum( id );
}

void Worker::deletetable()
{
  int i, j;
  keyhash *kh, *kh2;
  
  if ( keyhashtable != NULL ) {
    for ( i = 0; i < keyhashtable_size; i++ ) {
      kh = keyhashtable[i];
      j = 0;
      while ( kh != NULL ) {
        kh2 = kh->next;
        delete kh;
        kh = kh2;
        j++;
      }
#ifdef WANT_KEYHASHSTAT
      if ( j > 0 ) printf( "keyhashtable[%d] used %d entries\n", i, j );
#endif
    }
  
    delete [] keyhashtable;
    keyhashtable = NULL;
  }
}

unsigned int Worker::keyhashfunc( KeySym k, unsigned int m )
{
  unsigned long hashv;
  
  hashv = k;
  hashv += m << 16;
  return ( hashv % keyhashtable_size );
}

void Worker::insertkey( KeySym k, unsigned int m, WCButton *b, WCHotkey *hk, WCPath *p )
{
  unsigned int hashvalue;
  keyhash *kh, *kh2;

  if ( ( b == NULL ) && ( hk == NULL ) && ( p == NULL ) ) return;
  if ( k == 0 ) return;
  if ( keyhashtable == NULL ) return;
  
  kh2 = new keyhash;
  kh2->next = NULL;
  kh2->b = b;
  kh2->h = hk;
  kh2->p = p;

  hashvalue = keyhashfunc( k, m );
  kh = keyhashtable[ hashvalue ];
  if ( kh == NULL ) {
    keyhashtable[ hashvalue ] = kh2;
  } else {
    while ( kh->next != NULL ) kh = kh->next;
    kh->next = kh2;
  }
}

int Worker::findkey( KeySym k, unsigned int m, WCButton **b, WCHotkey **hk, WCPath **p )
{
  unsigned int hashvalue;
  keyhash *kh;
  int returnvalue = 0;

  if ( k == 0 ) return returnvalue;
  if ( keyhashtable == NULL ) return returnvalue;
  
  hashvalue = keyhashfunc( k, m );
  kh = keyhashtable[ hashvalue ];

  if ( b != NULL ) *b = NULL;
  if ( hk != NULL ) *hk = NULL;
  if ( p != NULL ) *p = NULL;

  if ( kh != NULL ) {
    while ( kh != NULL ) {
      if ( kh->b != NULL ) {
        if ( kh->b->hasKey( k, m ) == true ) {
          if ( b != NULL ) *b = kh->b;
          returnvalue = 1;
          break;
        }
      } else if ( kh->h != NULL ) {
        if ( kh->h->hasKey( k, m ) == true ) {
          if ( hk != NULL ) *hk = kh->h;
          returnvalue = 1;
          break;
        }
      } else if ( kh->p != NULL ) {
        if ( kh->p->hasKey( k, m ) == true ) {
          if ( p != NULL ) *p = kh->p;
          returnvalue = 1;
          break;
        }
      }
      kh = kh->next;
    }
  }

  return returnvalue;
}

int Worker::findkey( KeySym k1, unsigned int m1, KeySym k2, unsigned int m2, WCButton **b, WCHotkey **hk, WCPath **p )
{
  unsigned int hashvalue;
  keyhash *kh;
  int returnvalue = 0;

  if ( ( k1 == 0 ) || ( k2 == 0 ) ) return returnvalue;
  if ( keyhashtable == NULL ) return returnvalue;
  
  hashvalue = keyhashfunc( k1, m1 );
  kh = keyhashtable[ hashvalue ];

  if ( b != NULL ) *b = NULL;
  if ( hk != NULL ) *hk = NULL;
  if ( p != NULL ) *p = NULL;

  if ( kh != NULL ) {
    while ( kh != NULL ) {
      if ( kh->b != NULL ) {
        if ( kh->b->hasKey( k1, m1, k2, m2 ) == true ) {
          if ( b != NULL ) *b = kh->b;
          returnvalue = 1;
          break;
        }
      } else if ( kh->h != NULL ) {
        if ( kh->h->hasKey( k1, m1, k2, m2 ) == true ) {
          if ( hk != NULL ) *hk = kh->h;
          returnvalue = 1;
          break;
        }
      } else if ( kh->p != NULL ) {
        if ( kh->p->hasKey( k1, m1, k2, m2 ) == true ) {
          if ( p != NULL ) *p = kh->p;
          returnvalue = 1;
          break;
        }
      }
      kh = kh->next;
    }
  }

  return returnvalue;
}

void Worker::closeMainWin()
{
  unsigned int i, j;

  if ( m_worker_menu != NULL ) {
      delete m_worker_menu;
      m_worker_menu = NULL;
  }

  delete req_static;
  req_static = new Requester( aguix );

  delete listerwin[0];
  listerwin[0] = NULL;
  delete listerwin[1];
  listerwin[1] = NULL;

  if ( buttons != NULL ) {
    if ( ( curRows != 0 ) && ( curColumns != 0 ) ) {
      for ( i = 0; i < curRows; i++ ) {
        for( j = 0; j < curColumns; j++ ) {
          mainwin->remove(buttons[ i * curColumns + j ]);
          delete buttons[ i * curColumns + j ];
          buttons[ i * curColumns + j ] = NULL;
        }
      }
    }
    delete [] buttons;
    buttons = NULL;
  }
  if ( pathbs != NULL ) {
    if ( ( curRows != 0 ) && ( curColumns != 0 ) ) {
      for ( i = 0; i < curRows; i++ ) {
        mainwin->remove( pathbs[i] );
        delete pathbs[ i ];
        pathbs[ i ] = NULL;
      }
    }
    delete [] pathbs;
    pathbs = NULL;
  }
  delete mainwin;
  mainwin = NULL;
}

void Worker::showLicense()
{
    RefCount<AFontWidth> lencalc( new AFontWidth( aguix, NULL ) );
    TextStorageString ts( cgpl, lencalc );
    req_static->request( catalog.getLocale( 509 ), ts, catalog.getLocale( 11 ) );
}

void Worker::setWaitCursor()
{
  if ( waitcursorcount == 0 ) {
    getMainWin()->setCursor( AGUIX::WAIT_CURSOR );
    aguix->Flush();
  }
  waitcursorcount++;
}

void Worker::unsetWaitCursor()
{
  if ( waitcursorcount > 0 ) {
    waitcursorcount--;
    if ( waitcursorcount == 0 ) getMainWin()->unsetCursor();
  }
}

void Worker::PS_setLifetime( double t )
{
  freesp->setLifetime( t );
}

/*
 * static getDatatDir()
 *
 * will return the basedir of the worker share files
 * or empty string if not found
 *
 * search order is:
 * 1.PREFIX/share/worker
 * 2.`dirname $0`/../share/worker
 *   (if worker is in a bin-directory
 * 3./usr/share/worker
 * 4./usr/local/share/worker
 * 5.$HOME/share/worker
 *
 */
const std::string Worker::getDataDir()
{
  static std::string datadir="";
  std::string str1;
  const char *s1;
  char *s2, *s3;
  std::list<std::string> searchlist;
  std::list<std::string>::iterator it1;
  static bool warned = false;

#ifdef PREFIX
  if ( datadir.length() < 1 ) {
    // first try PREFIX
    if ( strlen( PREFIX ) > 0 ) {
      str1 = PREFIX;
      str1 += "/share/worker";
      if ( Datei::fileExistsExt( str1.c_str() ) == Datei::D_FE_DIR ) {
        // dir found
        datadir = str1;
      }
      searchlist.push_back( str1 );
    }
  }
#endif
  if ( datadir.length() < 1 ) {
    // now try to get the datadir from argv[0]
    if ( ( argc > 0 ) && ( argv != NULL ) ) {
      if ( argv[0][0] != '/' ) {
        str1 = startcwd;
        str1 += "/";
        str1 += argv[0];
      } else {
        str1 = argv[0];
      }
      s2 = HandlePath( str1.c_str() );  // remove . and .. parts
      if ( s2 != NULL ) {
        s3 = ParentDir( s2, NULL );  // get dirname from exename
        _freesafe( s2 );
        if ( s3 != NULL ) {
          s2 = ParentDir( s3, NULL );  // now the parent of the dirname
          _freesafe( s3 );
          if ( s2 != NULL ) {
            str1 = s2;
            str1 += "/share/worker";
            if ( Datei::fileExistsExt( str1.c_str() ) == Datei::D_FE_DIR ) {
              // dir found
              datadir = str1;
            }
            searchlist.push_back( str1 );
            _freesafe( s2 );
          }
        }
      }
    }
  }
  if ( datadir.length() < 1 ) {
    // try /usr
    str1 = "/usr/share/worker";
    if ( Datei::fileExistsExt( str1.c_str() ) == Datei::D_FE_DIR ) {
      // dir found
      datadir = str1;
    }
    searchlist.push_back( str1 );
  }
  if ( datadir.length() < 1 ) {
    // try /usr/local
    str1 = "/usr/local/share/worker";
    if ( Datei::fileExistsExt( str1.c_str() ) == Datei::D_FE_DIR ) {
      // dir found
      datadir = str1;
    }
    searchlist.push_back( str1 );
  }
  if ( datadir.length() < 1 ) {
    // finally try $HOME
    s1 = getenv( "HOME" );
    if ( s1 != NULL ) {
      str1 = s1; 
      str1 += "/share/worker";
      if ( Datei::fileExistsExt( str1.c_str() ) == Datei::D_FE_DIR ) {
        // dir found
        datadir = str1;
      }
      searchlist.push_back( str1 );
    }
  }
  if ( datadir.length() < 1 ) {
    if ( Worker::getRequester() != NULL ) {
      if ( warned == false ) {
        str1 = "Worker couldn't find the data dir! I was looking in:|";
        for ( it1 = searchlist.begin(); it1 != searchlist.end(); it1++ ) {
          str1 += *it1;
          str1 += "|";
        }
        str1 += "Check for correct worker installation!";
        Worker::getRequester()->request( "Worker Warning", str1.c_str(), "Okay" );
        warned = true;
      }
    }
  }
  return datadir;
}

void Worker::createMainWin()
{
  int rows, columns;
  int tw, th;
  
  if ( mainwin != NULL ) return;
  
  rows=wconfig->getRows();
  columns=wconfig->getColumns();
  mainwin = new AWindow( aguix, 10, 10, w, h, 0, "Worker" );
  mainwin->create();
  mainwin->setMinSize(300,200);

  aboutb=new Button(aguix,0,0,"A",1,0,1);
  aboutb->setAcceptFocus( false );
  aboutb->getSize(&tw,&th);
  aboutb->resize(th,th);
  configureb=new Button(aguix,0,0,"C",1,0,2);
  configureb->setAcceptFocus( false );
  statebar = new Button( aguix, 0, 0, th, "", "", 1, 1, 0, 0, 4 );
  statebar->deactivate( 0 );
  statebar->setShowDual( false );
  statebar->setAcceptFocus( false );
  statebar->setText( 1, catalog.getLocale( 786 ) );
  clockbar=new Button(aguix,0,0,th,"","",1,1,0,0,3);
  clockbar->setAcceptFocus( false );
  clockbar->setShowDual(false);
  clockbar->setAllowWheel( true );
  
  listerwin[0] = new AWindow( aguix, 0, 0, 200, 200, 0, "" );
  mainwin->add( listerwin[0] );
  listerwin[0]->create();
  listerwin[0]->show();
  listerwin[1] = new AWindow( aguix, 0, 0, 200, 200, 0, "" );
  mainwin->add( listerwin[1] );
  listerwin[1]->create();
  listerwin[1]->show();
  
  mainwin->show();

  delete req_static;
  req_static = new Requester( aguix, mainwin );
  aguix->setTransientWindow( mainwin );

  createWorkerMenu();
}

AWindow *Worker::getListerWin( Lister *l )
{
  if ( lister[0] == NULL ) return listerwin[0];
  else if ( lister[0] == l ) return listerwin[0];
  else if ( lister[1] == NULL ) return listerwin[1];
  else if ( lister[1] == l ) return listerwin[1];
  return NULL;
}

int Worker::getMaxButtonsWidth()
{
  int bw;
  int id;
  List *buttonlist;
  WCButton *b1;
  int t;
  const char *tstr;

  AGUIXFont *font = aguix->getFont( wconfig->getFont( 1 ) );
  
  if ( font == NULL ) return 0;

  buttonlist = wconfig->getButtons();
  id = buttonlist->initEnum();
  b1 = (WCButton*)buttonlist->getFirstElement( id );
  bw = 0;
  while ( b1 != NULL ) {
    if( b1->getCheck() == true ) {
      tstr = b1->getText();
      if ( tstr != NULL ) {
        t = aguix->getTextWidth( tstr, font );
        if ( t > bw ) bw = t;
      }
    }
    b1 = (WCButton*)buttonlist->getNextElement( id );
  }
  buttonlist->closeEnum( id );
  
  return bw;
}

/*
 * this method will execute a given string
 * Worker will fork and wait for this child while the child also forks
 * and wait for his child
 *
 * This way Worker can sent disconnect signal and
 * the actual program can run even when Worker quits
 *
 * for detach I use 2 pipes with the following protocol:
 * 1.child will always wait for ack from Worker before
 *   he quits (to avoid broken pipes!)
 * 2.Worker will not access any pipes when he sent ack single
 *   (some reason)
 * 3.Worker will always wait for his child to quit
 *   (for the child the pipes are opened until he exits)
 * 4.Child will send Q when his child quits
 * 5.Worker can send D when user want to detach
 * 6.After D signal or receiving Q signal Worker
 *   send A
 *
 * it's complicated but avoids any problems with broken pipes (child
 * exits same time user want to detach->Worker would write to closed pipe)
 *
 * an other idea is to use shared memory segmentbut that's not so portable
 * although it's much nicer
 */
int Worker::runCommand( const char*exestr,
			const char *tmpname,
			const char *tmpoutput,
			bool inbackground )
{
  pid_t child, ret, p2;
  int status, retval;
  
  setWaitCursor();
  WorkerSHM::initSHM();
  if ( WorkerSHM::WSHMInit == true ) {
    *( WorkerSHM::WSHMPtr ) = ' ';
  }
  child = fork();
  if ( child != 0 ) {
    // parent
    if ( child == -1 ) {
      unsetWaitCursor();
      return -2;
    }
    retval = 0;
    // calling these functions will clear old values
    aguix->getLastKeyRelease();
    aguix->getLastMouseRelease();
    if ( inbackground == false ) {
      setStatebarText( catalog.getLocale( 547 ) );
    }
    // after setting this flag Worker won't read/write from the pipe
    // because the child is expected to exit right after this
    for(;;) {
      // first answer all messages to be able to redraw
      aguix->doXMsgs( NULL, false );
      // check for events to disconnect from child
      if ( ( aguix->getLastKeyRelease() == XK_Escape ) ||
           ( aguix->getLastMouseRelease() == Button2 ) ) {
        if ( WorkerSHM::WSHMInit == true ) {
          *(WorkerSHM::WSHMPtr ) = 'Q';
        }
      }
      // now check for existence of the child
      ret = waitpid( child, &status, WNOHANG );
      if ( ret == child ) {
        if ( WIFEXITED( status ) ) {
          // normal exit
#ifdef DEBUG
          printf( "child exited normally!\n" );
#endif
          retval = WEXITSTATUS( status );
          break;
        } else if ( WIFSIGNALED( status ) ) {
#ifdef DEBUG
          printf("child exited by a signal!\n");
#endif
          retval = -1;
          break;
        } else {
#ifdef DEBUG
          printf( "child exited by an unhandled way!\n" );
#endif
          retval = -1;
          break;
        }
      } else if ( ret == -1 ) {
          if ( errno == ECHILD ) {
              // child does not exists?
              retval = -1;
              break;
          }
      }
      // wait some time and the retry
      waittime( 10 );
    }
    unsetWaitCursor();
    if ( inbackground == false ) {
      resetStatebarText();
    }
    return retval;
  }
  // child
  // from now on I will use _exit instead of exit to avoid calling
  // exit handlers
  // avfs currently has the problem that a exit handler removes
  // the temp. directory rendering the vfs for the main process
  // useless
  //TODO:If this changes in avfs, I will use exit again
  // wait for second child exit until there is something in the pipe

  // disconnect from parent
  setsid();
  worker_chdir( "/" );

  p2 = fork();
  if ( p2 != 0 ) {
    // parent
    if ( p2 < 0 ) {
      // error
      _exit( EXIT_FAILURE );
    } else {
      // exit immediately when command should run in background
      bool doexit = inbackground;

      while ( doexit == false ) {
        ret = waitpid( p2, &status, WNOHANG );
        if ( ret == p2 ) {
          doexit = true;
          continue;
        } else if ( ret == -1 ) {
            doexit = true;
            continue;
        }
        if ( WorkerSHM::WSHMInit == true ) {
          if ( *(WorkerSHM::WSHMPtr) == 'Q' ) {
            doexit = true;
            continue;
          }
        }
        // wait some time and retry
        waittime( 10 );
      }
      _exit( EXIT_SUCCESS );
    }
  }
  setsid();
  // execute command
  // we could use exec* to call this
  //   but then we have to change exestr to give a char**argv
  //   (which is not so hard because it contains only the commands to execute
  //    tmpname)
  //   we also have to make another fork because exec* doesn't return
  //     and so we need an other process to remove the tmp-files after command finished
  system( exestr );
  
  // remove temp files
  worker_unlink( tmpname );
  worker_unlink( tmpoutput );
  
  // and exit
  _exit( EXIT_SUCCESS );
  // just for some compiler to compile without warning
  return 0;
}

std::string Worker::queryLanguage()
{
    std::vector<std::string> langs;
    DIR *dir;
    std::string catdir, buttonstr2, str2, str3;
    worker_struct_dirent *namelist;
    std::string selected_lang;

    selected_lang = "english";
    
    langs.push_back( selected_lang );
    if ( Worker::getDataDir().length() > 0 ) {
        catdir = Worker::getDataDir();
        catdir += "/catalogs";
        dir = worker_opendir( catdir.c_str() );
        if ( dir != NULL ) {
            std::string ending = ".catalog";
            if ( UTF8::getUTF8Mode() != UTF8::UTF8_DISABLED &&
                 UTF8::getCurrentEncoding() == UTF8::ENCODING_UTF8 ) {
                ending += ".utf8";
            }
            
            while ( ( namelist = worker_readdir( dir ) ) != NULL ) {
                if ( strlen( namelist->d_name ) > ending.length() ) {
                    if ( strcmp( ending.c_str(),
                                 ( namelist->d_name ) + strlen( namelist->d_name ) - ending.length() ) == 0 ) {
                        // looks a catalog file
                        std::string str1( namelist->d_name,
                                          strlen( namelist->d_name ) - ending.length() );
                        langs.push_back( str1 );
                    }
                }
            }
            worker_closedir( dir );
        }
    }
    // now all available languages are in the list
    // build a requester

    AWindow *win;
    AGMessage *msg;
    int endmode=-1;
    
    win = new AWindow( aguix, 10, 10, 10, 10, 0, "Language selection" );
    win->create();
    
    AContainer *ac1 = win->setContainer( new AContainer( win, 1, 3 ), true );
    ac1->setMinSpace( 5 );
    ac1->setMaxSpace( 5 );
    
    ac1->add( new Text( aguix, 0, 0, "Choose your language (this can be changed later):", 1 ),
              0, 0, AContainer::CINCWNR );

    FieldListView *lv1 = (FieldListView*)ac1->add( new FieldListView( aguix,
                                                                      0, 0,
                                                                      200, 200, 0 ),
                                                   0, 1, AContainer::CMIN );
    lv1->setHBarState( 2 );
    lv1->setVBarState( 2 );
    lv1->setAcceptFocus( true );
    lv1->setDisplayFocus( true );

    for ( std::vector<std::string>::iterator it1 = langs.begin();
          it1 != langs.end();
          it1++ ) {
        int row = lv1->addRow();
        lv1->setText( row, 0, *it1 );
        lv1->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
    }

    AContainer *ac1_1 = ac1->add( new AContainer( win, 3, 1 ), 0, 2 );
    ac1_1->setMinSpace( 5 );
    ac1_1->setMaxSpace( -1 );
    ac1_1->setBorderWidth( 0 );
    Button *contb =(Button*)ac1_1->add( new Button( aguix,
                                                    0,
                                                    0,
                                                    "Continue",
                                                    1,
                                                    0,
                                                    0 ), 1, 0, AContainer::CFIX );
    win->setDoTabCycling( true );
    win->contMaximize( true );
    win->show();
    
    for ( ;endmode == -1; ) {
        msg = aguix->WaitMessage( win );
        if ( msg != NULL ) {
            switch ( msg->type ) {
              case AG_CLOSEWINDOW:
                  if ( msg->closewindow.window == win->getWindow() ) endmode = 1;
                  break;
              case AG_BUTTONCLICKED:
                  if ( msg->button.button == contb ) endmode = 0;
                  break;
            }
            aguix->ReplyMessage( msg );
        }
    }
    
    if ( endmode == 0 ) {
        int row = lv1->getActiveRow();
        if ( lv1->isValidRow( row ) == true ) {
            selected_lang = langs[row];
        }
    }
    
    delete win;
    return selected_lang;
}

std::string Worker::getWorkerConfigDir()
{
    const char *homestr;
    std::string d;
    
    homestr = getenv( "HOME" );
    if ( homestr == NULL ) {
        std::cerr << "Worker: No $HOME found, aborting!" << std::endl;
        exit( EXIT_FAILURE );
    }

    d = NWC::Path::join( homestr, ".worker" );

    //TODO the result could be cached
    if ( Datei::fileExists( d.c_str() ) == false ) {
        if ( worker_mkdir( d.c_str(), 0700 ) != 0 ) {
            std::cerr << "Worker: Couldn't create worker configuration directory " << d << std::endl;
            exit( EXIT_FAILURE );
        }
    }
    return d;
}

void Worker::createWorkerMenu()
{
    if ( m_worker_menu != NULL ) {
        delete m_worker_menu;
        m_worker_menu = NULL;
    }

    std::list<PopUpMenu::PopUpEntry> m1;
    PopUpMenu::PopUpEntry e1;
    e1.name = catalog.getLocale( 787 );
    e1.type = PopUpMenu::HEADER;
    m1.push_back( e1 );
    e1.type = PopUpMenu::HLINE;
    m1.push_back( e1 );

    // about
    e1.type = PopUpMenu::NORMAL;
    e1.name = catalog.getLocale( 208 );
    e1.id = 1;
    m1.push_back( e1 );
    
    // configure
    e1.type = PopUpMenu::NORMAL;
    e1.name = catalog.getLocale( 0 );
    e1.id = 2;
    m1.push_back( e1 );
    
    e1.type = PopUpMenu::HLINE;
    m1.push_back( e1 );

    // save worker state
    e1.type = PopUpMenu::NORMAL;
    e1.name = catalog.getLocale( 788 );
    e1.id = 3;
    m1.push_back( e1 );
    
    e1.type = PopUpMenu::HLINE;
    m1.push_back( e1 );

    // quit
    e1.type = PopUpMenu::NORMAL;
    e1.name = catalog.getLocale( 789 );
    e1.id = 4;
    m1.push_back( e1 );
    
    m_worker_menu = new PopUpMenu( aguix, m1 );
    m_worker_menu->create();
}

void Worker::openWorkerMenu()
{
    if ( m_worker_menu != NULL ) {
        m_worker_menu->show();
    }
}

void Worker::startMenuAction( AGMessage *msg )
{
    if ( msg->popupmenu.menu != m_worker_menu ) return;

    switch ( msg->popupmenu.clicked_entry_id ) {
        case 1: // about
            about();
            break;
        case 2: // configure
            configure();
            break;
        case 3: // save worker state
            saveWorkerState();
            break;
        case 4: // quit worker
            quit( 1 );
            break;
        default:
            break;
    }
}

void Worker::saveWorkerState()
{
    saveListerState();
    wconfig->savesize( w, h);
}

int Worker::getStatebarWidth()
{
    return statebar->getWidth() - 2 * 2;
}

BookmarkDBProxy &Worker::getBookmarkDBInstance()
{
    if ( m_bookmarkproxy.get() != NULL ) return *m_bookmarkproxy;

    std::string cfgfile = Worker::getWorkerConfigDir();
#ifdef DEBUG
    cfgfile = NWC::Path::join( cfgfile, "bookmarks2" );
#else
    cfgfile = NWC::Path::join( cfgfile, "bookmarks" );
#endif
    m_bookmarkproxy.reset( new BookmarkDBProxy( std::auto_ptr<BookmarkDB>( new BookmarkDB( cfgfile ) ) ) );
    m_bookmarkproxy->read();
    return *m_bookmarkproxy;
}

struct cb_cmp : public std::unary_function<const RefCount<GenericCallbackArg<void, AGMessage*> >&, bool> {
    cb_cmp( const RefCount<GenericCallbackArg<void, AGMessage*> > &entry ) : m_entry( entry )
    {}
    bool operator()( const RefCount<GenericCallbackArg<void, AGMessage*> > &other )
    {
        if ( other.getVal() == NULL || m_entry.getVal() == NULL ) return false;
        if ( other.getVal() == m_entry.getVal() ) return true;
        return false;
    }
    const RefCount<GenericCallbackArg<void, AGMessage*> > &m_entry;
};

void Worker::registerPopUpCallback( RefCount<GenericCallbackArg<void, AGMessage*> > cb,
                                    const PopUpMenu *menu )
{
    std::map<const PopUpMenu *, std::list<RefCount<GenericCallbackArg<void, AGMessage*> > > >::iterator it1;

    it1 = m_popup_callbacks.find( menu );

    if ( it1 != m_popup_callbacks.end() ) {
        if ( std::find_if( m_popup_callbacks[menu].begin(),
                           m_popup_callbacks[menu].end(),
                           cb_cmp( cb ) ) == m_popup_callbacks[menu].end() ) {
            m_popup_callbacks[menu].push_back( cb );
        }
    } else {
        m_popup_callbacks[menu].push_back( cb );
    }
}

void Worker::unregisterPopUpCallback( RefCount<GenericCallbackArg<void, AGMessage*> > cb )
{
    std::map<const PopUpMenu *, std::list<RefCount<GenericCallbackArg<void, AGMessage*> > > >::iterator it1;
    std::list<RefCount<GenericCallbackArg<void, AGMessage*> > >::iterator it2;

    for ( it1 = m_popup_callbacks.begin();
          it1 != m_popup_callbacks.end();
          ++it1 ) {
        it2 = std::find_if( it1->second.begin(),
                            it1->second.end(),
                            cb_cmp( cb ) );
        if ( it2 != it1->second.end() ) {
            it1->second.erase( it2 );
        }
    }
}

void Worker::handlePopUps( AGMessage *msg )
{
    if ( msg == NULL ) return;

    if ( msg->type != AG_POPUPMENU_CLICKED ) return;

    std::map<const PopUpMenu *, std::list<RefCount<GenericCallbackArg<void, AGMessage*> > > >::iterator it1;
    std::list<RefCount<GenericCallbackArg<void, AGMessage*> > > temp_list;
    std::list<RefCount<GenericCallbackArg<void, AGMessage*> > >::iterator it2;

    it1 = m_popup_callbacks.find( msg->popupmenu.menu );

    if ( it1 != m_popup_callbacks.end() ) {
        // copy list because a callback might remove it from the list
        // thus invalidating the iterator
        temp_list = it1->second;
        for ( it2 = temp_list.begin();
              it2 != temp_list.end();
              ++it2 ) {
            (*it2)->callback( msg );
        }
    }

    it1 = m_popup_callbacks.find( NULL );

    if ( it1 != m_popup_callbacks.end() ) {
        temp_list = it1->second;
        for ( it2 = temp_list.begin();
              it2 != temp_list.end();
              ++it2 ) {
            (*it2)->callback( msg );
        }
    }
}
