/* TextRunStyle: Stores (and writes) paragraph/span-style-based information 
 * (e.g.: a paragraph might be bold) that is needed at the head of an OO 
 * document.
 *
 * Copyright (C) 2002-2003 William Lachance (william.lachance@sympatico.ca)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * For further information visit http://libwpd.sourceforge.net
 *
 */

/* "This product is not manufactured, approved, or supported by 
 * Corel Corporation or Corel Corporation Limited."
 */
#include "FilterInternal.h"
#include "TextRunStyle.h"
#include "WriterProperties.h"
#include "DocumentElement.h"

TextRunStyle::TextRunStyle(const guint32 iTextAttributeBits, const gchar *pFontName, const float fFontSize, const gchar *psName) : 
	Style(psName),
	miTextAttributeBits(iTextAttributeBits),
	msFontName(pFontName),
	mfFontSize(fFontSize)
{
}

TextRunStyle::~TextRunStyle()
{
}

void TextRunStyle::_addTextProperties(TagOpenElement *pStylePropertiesOpenElement) const
{
 	if (miTextAttributeBits & WPX_SUPERSCRIPT_BIT) {
		UTF8String sSuperScript;
		sSuperScript.sprintf("super %s", IMP_DEFAULT_SUPER_SUB_SCRIPT);
		pStylePropertiesOpenElement->addAttribute("style:text-position", sSuperScript.getUTF8());
	}
 	if (miTextAttributeBits & WPX_SUBSCRIPT_BIT) {
		UTF8String sSubScript;
		sSubScript.sprintf("sub %s", IMP_DEFAULT_SUPER_SUB_SCRIPT);
		pStylePropertiesOpenElement->addAttribute("style:text-position", sSubScript.getUTF8());
	}
	if (miTextAttributeBits & WPX_ITALICS_BIT) {
		pStylePropertiesOpenElement->addAttribute("fo:font-style", "italic");
	}
	if (miTextAttributeBits & WPX_BOLD_BIT) {
		pStylePropertiesOpenElement->addAttribute("fo:font-weight", "bold");
	}
	if (miTextAttributeBits & WPX_STRIKEOUT_BIT) {
		pStylePropertiesOpenElement->addAttribute("style:text-crossing-out", "single-line");
	}
 	if (miTextAttributeBits & WPX_UNDERLINE_BIT) {
		pStylePropertiesOpenElement->addAttribute("style:text-underline", "single");
	}
	if (miTextAttributeBits & WPX_DOUBLE_UNDERLINE_BIT) {
		pStylePropertiesOpenElement->addAttribute("style:text-underline", "double");
	}
	if (miTextAttributeBits & WPX_OUTLINE_BIT) {
		pStylePropertiesOpenElement->addAttribute("style:text-outline", "true");
	}
	// FIXME: when we add actual colour properties capabilities to our text spans, we
	// will have to figure out some other way of doing this. doesn't OOo have real redlining
	// capabilities?
	if (miTextAttributeBits & WPX_REDLINE_BIT) {
		pStylePropertiesOpenElement->addAttribute("fo:color", "#ff3333"); // #ff3333 = a nice bright red
	}
	if (miTextAttributeBits & WPX_SMALL_CAPS_BIT) {
		pStylePropertiesOpenElement->addAttribute("fo:font-variant", "small-caps");
	}
	if (miTextAttributeBits & WPX_BLINK_BIT) {
		pStylePropertiesOpenElement->addAttribute("style:text-blinking", "true");
	}
	if (miTextAttributeBits & WPX_SHADOW_BIT) {
		pStylePropertiesOpenElement->addAttribute("fo:text-shadow", "1pt 1pt");
	}


	pStylePropertiesOpenElement->addAttribute("style:font-name", msFontName.getUTF8());
	UTF8String sFontSize;
	sFontSize.sprintf("%ipt", (int)mfFontSize);
	pStylePropertiesOpenElement->addAttribute("fo:font-size", sFontSize.getUTF8());
}

ParagraphStyle::ParagraphStyle(const guint8 iParagraphJustification, const guint32 iTextAttributeBits,
			       const float fMarginLeft, const float fMarginRight, 		       
			       const gchar *pFontName, const float fFontSize,
			       const float fLineSpacing,
			       const bool bColumnBreak, const bool bPageBreak, 
			       const gchar *psName, const gchar *psParentName) :
	TextRunStyle(iTextAttributeBits, pFontName, fFontSize, psName), 
	mpsParentName(g_string_new(psParentName)),
	mpsListStyleName(NULL),
	mfMarginLeft(fMarginLeft),
	mfMarginRight(fMarginRight),
	mfLineSpacing(fLineSpacing),
	miParagraphJustification(iParagraphJustification),
	mbColumnBreak(bColumnBreak),
	mbPageBreak(bPageBreak)
{
}

ParagraphStyle::~ParagraphStyle()
{
	g_string_free(mpsParentName, TRUE);
	if (mpsListStyleName)
		delete mpsListStyleName;
}

void ParagraphStyle::write(GsfOutput *pOutput) const
{ 
	WRITER_DEBUG_MSG(("Writing a paragraph style..\n"));
	TagOpenElement styleOpen("style:style");
	styleOpen.addAttribute("style:name", getName()->str);
	styleOpen.addAttribute("style:family", "paragraph");
	styleOpen.addAttribute("style:parent-style-name", mpsParentName->str);
	if (getMasterPageName())
		styleOpen.addAttribute("style:master-page-name", getMasterPageName()->getUTF8());
	if (mpsListStyleName)
		styleOpen.addAttribute("style:list-style-name", mpsListStyleName->getUTF8());
	styleOpen.write(pOutput);

	TagOpenElement stylePropertiesOpen("style:properties");
	// margin properties
	if (mfMarginLeft != 0.0f || mfMarginRight != 0.0f)
	{
		UTF8String sMarginLeft;
		sMarginLeft.sprintf("%finch", mfMarginLeft);
		UTF8String sMarginRight;
		sMarginRight.sprintf("%finch", mfMarginRight);
		stylePropertiesOpen.addAttribute("fo:margin-left", sMarginLeft.getUTF8());
		stylePropertiesOpen.addAttribute("fo:margin-right", sMarginRight.getUTF8());
	}
	// line spacing
	if (mfLineSpacing != 1.0f) {
		UTF8String sLineSpacing;
		sLineSpacing.sprintf("%.2f%%", mfLineSpacing*100.0f);
		stylePropertiesOpen.addAttribute("fo:line-height", sLineSpacing.getUTF8());		
	}

 	_addTextProperties(&stylePropertiesOpen);

	// column break
	if (mbColumnBreak) {
		stylePropertiesOpen.addAttribute("fo:break-before", "column");
	}

	if (mbPageBreak) {
		stylePropertiesOpen.addAttribute("fo:break-before", "page");
	}	

	WRITER_DEBUG_MSG(("WriterWordPerfect: Adding justification style props: %i\n", miParagraphJustification));
	switch (miParagraphJustification) 
		{
		case WPX_PARAGRAPH_JUSTIFICATION_FULL:
		case WPX_PARAGRAPH_JUSTIFICATION_LEFT:
			// doesn't require a paragraph prop - it is the default, but, like, whatever
			stylePropertiesOpen.addAttribute("fo:text-align", "left");
			break;
		case WPX_PARAGRAPH_JUSTIFICATION_CENTER:
			stylePropertiesOpen.addAttribute("fo:text-align", "center");
			break;
		case WPX_PARAGRAPH_JUSTIFICATION_RIGHT:
			stylePropertiesOpen.addAttribute("fo:text-align", "end");
			break;
		case WPX_PARAGRAPH_JUSTIFICATION_FULL_ALL_LINES:
			stylePropertiesOpen.addAttribute("fo:text-align", "justify");
			break;
	}
	stylePropertiesOpen.addAttribute("style:justify-single-word", "false");
	stylePropertiesOpen.write(pOutput);

	gsf_output_printf (pOutput, "</style:properties>\n");
	gsf_output_printf (pOutput, "</style:style>\n");
}

void SpanStyle::write(GsfOutput *pOutput) const
{
	WRITER_DEBUG_MSG(("Writing a span style..\n"));
	TagOpenElement styleOpen("style:style");
	styleOpen.addAttribute("style:name", getName()->str);
	styleOpen.addAttribute("style:family", "text");
	styleOpen.write(pOutput);

	TagOpenElement stylePropertiesOpen("style:properties");
 	_addTextProperties(&stylePropertiesOpen);
	stylePropertiesOpen.write(pOutput);

	gsf_output_printf (pOutput, "</style:properties>\n");
	gsf_output_printf (pOutput, "</style:style>\n");
}
