/* Parsing FTP `ls' output.
   Copyright (C) 1995, 1996, 1997, 2000, 2001
   Free Software Foundation, Inc. 

This file is part of GNU Wget.

GNU Wget is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

GNU Wget is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Wget; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* TODO IMP is there some licensing stuff i have to care about? */

/* this code is adopted from Wget and modified to work within
 * wputs environtment.
 * the main difference is, that Wget assumes the listing to be
 * stored in a file, whereas wput has it in its "memory" */
 
/* TODO INT64-support */
#include "../config.h"

#define TRUE  1
#define FALSE 0

#define ISDIGIT(x) ((x) >= '0' && (x) <= '9')

#include "wput.h"
#include "ftp.h"
#include "utils.h"

/* standalone-linking: gcc ftp-ls.c -o ftp-ls.o utils.o socket.o -DSTANDALONE */
/* for debugging purposes */
#ifdef STANDALONE
    #include <stdio.h>
    #include <stdlib.h>
    #ifdef HAVE_STRING_H
        #include <string.h>
    #else
        #include <strings.h>
    #endif
    #ifdef HAVE_UNISTD_H
        #include <unistd.h>
    #endif
    #include <sys/types.h>
    #include <errno.h>
    #include <time.h>
    #define printout(x, y ...) printf(y)
#endif

/* Cleans a line of text so that it can be consistently parsed. Destroys
   <CR> and <LF> in case that thay occur at the end of the line and
   replaces all <TAB> character with <SPACE>. Returns the length of the
   modified line. */
/* Moves the pointer nextline to the end of the line */
static int
clean_line(char *line, char ** nextline)
{
  char * p = strchr(line, '\n');
  int len;
  if (!p) return 0;
  *nextline = p+1;
  *p-- = 0;
  if (*p == '\r') *p-- = 0;
  len = p - line;
  for ( ; *line ; line++ ) if (*line == '\t') *line = ' '; 
  return len;
}

/* Convert the Un*x-ish style directory listing stored in FILE to a
   linked list of fileinfo (system-independent) entries.  The contents
   of FILE are considered to be produced by the standard Unix `ls -la'
   output (whatever that might be).  BSD (no group) and SYSV (with
   group) listings are handled.

   The time stamps are stored in a separate variable, time_t
   compatible (I hope).  The timezones are ignored.  */
static struct fileinfo *
ftp_parse_unix_ls (char *listing)
{
  static const char *months[] = {
    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
  };
  int next, len, i, error, ignore;
  int year, month, day;		/* for time analysis */
  int hour, min, sec;
  struct tm timestruct, *tnow;
  time_t timenow;

  char *nextline = listing;
  char *line, *tok;		/* tokenizer */
  struct fileinfo *dir, *l, cur; /* list creation */

  dir = l = NULL;

  /* Line loop to end of file: */
  while (*(line = nextline))
    {
      len = clean_line (line, &nextline);
      /* Skip if total...  */
      if (!strncasecmp (line, "total", 5)) continue;
      
      /* Get the first token (permissions).  */
      tok = strtok (line, " ");
      if (!tok) continue;

      cur.name = NULL;
      //cur.linkto = NULL;

      /* Decide whether we deal with a file or a directory.  */
      switch (*tok)
    	{
    	case '-':
    	  cur.type = FT_PLAINFILE;
    	  printout(vDEBUG, "PLAINFILE; ");
    	  break;
    	case 'd':
    	  cur.type = FT_DIRECTORY;
    	  printout(vDEBUG, "DIRECTORY; ");
    	  break;
    	case 'l':
    	  cur.type = FT_SYMLINK;
    	  printout(vDEBUG, "SYMLINK; ");
    	  break;
    	default:
    	  cur.type = FT_UNKNOWN;
    	  printout(vDEBUG, "UNKNOWN; ");
    	  break;
    	}
      /* always ignore permissions ... */
      
      error = ignore = 0;       /* Erroneous and ignoring entries are
                                   treated equally for now.  */
      year = hour = min = sec = 0; /* Silence the compiler.  */
      month = day = 0;
      next = -1;
      /* While there are tokens on the line, parse them.  Next is the
         number of tokens left until the filename.
         
         Use the month-name token as the "anchor" (the place where the
         position wrt the file name is "known").  When a month name is
         encountered, `next' is set to 5.  Also, the preceding
         characters are parsed to get the file size.
         
         This tactic is quite dubious when it comes to
         internationalization issues (non-English month names), but it
	       works for now.  */
      while ((tok = strtok (NULL, " ")))
    	{
    	  --next;
    	  if (next < 0)		/* a month name was not encountered */
    	    {
    	      for (i = 0; i < 12; i++)
    		      if (!strcmp (tok, months[i]))
    		        break;
    	      /* If we got a month, it means the token before it is the
    		       size, and the filename is three tokens away.  */
        	  if (i != 12)
        		{
        		  char *t = tok - 2;
        		  long mul = 1;
        		  for (cur.size = 0; t > line && ISDIGIT (*t); mul *= 10, t--)
        		    cur.size += mul * (*t - '0');
        		  if (t == line)
        		    {
        		      /* Something is seriously wrong.  */
        		      error = 1;
        		      break;
        		    }
        		  month = i;
        		  next = 5;
        		  printout(vDEBUG, "month: %s; ", months[month]);
        		}
    	  }
    	  else if (next == 4)	/* days */
    	    {
    	      if (tok[1])	/* two-digit... */
    		      day = 10 * (*tok - '0') + tok[1] - '0';
    	      else		/* ...or one-digit */
    		      day = *tok - '0';
    	      printout(vDEBUG, "day: %d; ", day);
    	    }
    	  else if (next == 3)
    	    {
    	      /* This ought to be either the time, or the year.  Let's
          		 be flexible!
          
          		 If we have a number x, it's a year.  If we have x:y,
          		 it's hours and minutes.  If we have x:y:z, z are
          		 seconds.  */
    	      year = 0;
    	      min = hour = sec = 0;
    	      /* We must deal with digits.  */
    	      if (ISDIGIT (*tok))
        		{
        		  /* Suppose it's year.  */
        		  for (; ISDIGIT (*tok); tok++)
        		    year = (*tok - '0') + 10 * year;
        		  if (*tok == ':')
        		    {
        		      /* This means these were hours!  */
        		      hour = year;
        		      year = 0;
        		      ++tok;
        		      /* Get the minutes...  */
        		      for (; ISDIGIT (*tok); tok++)
        			      min = (*tok - '0') + 10 * min;
        		      if (*tok == ':')
            			{
            			  /* ...and the seconds.  */
            			  ++tok;
            			  for (; ISDIGIT (*tok); tok++)
            			    sec = (*tok - '0') + 10 * sec;
            			}
        		    }
    		      }
	            if (year)
		            printout(vDEBUG, "year: %d (no tm); ", year);
	            else
		            printout(vDEBUG, "time: %d:%d:%d (no yr); ", hour, min, sec);
	          }
	        else if (next == 2)    /* The file name */
    	    {
    	      int fnlen;
    	      char *p;
    
    	      /* Since the file name may contain a SPC, it is possible
    		       for strtok to handle it wrong.  */
    	      fnlen = strlen (tok);
    	      if (fnlen < len - (tok - line))
        		{
        		  /* So we have a SPC in the file name.  Restore the
        		     original.  */
        		  tok[fnlen] = ' ';
        		  /* If the file is a symbolic link, it should have a
        		     ` -> ' somewhere.  */
        		  if (cur.type == FT_SYMLINK)
        		    {
        		      p = strstr (tok, " -> ");
        		      if (!p)
            			{
            			  error = 1;
            			  break;
            			}
        		      //cur.linkto = cpy(p + 4);
        		      //printout(vDEBUG, "link to: %s\n", cur.linkto);
        		      /* And separate it from the file name.  */
        		      *p = 0;
        		    }
		        }
    	      /* If we have the filename, add it to the list of files or
    		       directories.  */
    	      /* "." and ".." are an exception!  */
    	      if (!strcmp (tok, ".") || !strcmp (tok, ".."))
        		{
        		  printout(vDEBUG, "\nIgnoring `.' and `..'; ");
        		  ignore = 1;
        		  break;
        		}
    	      /* Some FTP sites choose to have ls -F as their default
        		 LIST output, which marks the symlinks with a trailing
        		 `@', directory names with a trailing `/' and
        		 executables with a trailing `*'.  This is no problem
        		 unless encountering a symbolic link ending with `@',
        		 or an executable ending with `*' on a server without
        		 default -F output.  I believe these cases are very
        		 rare.  */
    	      fnlen = strlen (tok); /* re-calculate `fnlen' */
    	      cur.name = (char *) malloc (fnlen + 1);
    	      memcpy (cur.name, tok, fnlen + 1);
    	      if (fnlen)
        		{
        		  if (cur.type == FT_DIRECTORY && cur.name[fnlen - 1] == '/')
        		    {
        		      cur.name[fnlen - 1] = '\0';
        		      printout(vDEBUG, "trailing `/' on dir.\n");
        		    }
        		  else if (cur.type == FT_SYMLINK && cur.name[fnlen - 1] == '@')
        		    {
        		      cur.name[fnlen - 1] = '\0';
        		      printout(vDEBUG, "trailing `@' on link.\n");
        		    }
        		  else if (cur.type == FT_PLAINFILE
        			   //&& (cur.perms & 0111) /* we don't know the permissions. but an asterix in the filename might be fatal */
        			   && cur.name[fnlen - 1] == '*')
        		    {
        		      cur.name[fnlen - 1] = '\0';
        		      printout(vDEBUG, "trailing `*' on exec.\n");
        		    }
        		} /* if (fnlen) */
    	      else
    		      error = 1;
    	      break;
    	    }
    	  else
    	    abort ();
    	} /* while */

      if (!cur.name) // || (cur.type == FT_SYMLINK && !cur.linkto))
	      error = 1;

      printout(vDEBUG, "\n");

      if (error || ignore)
    	{
    	  printout(vDEBUG, "Skipping.\n");
    	  if(cur.name) free(cur.name);
    	  //if(cur.linkto) free(cur.linkto);
    	  continue;
    	}

      if (!dir)
    	{
    	  l = dir = (struct fileinfo *) malloc (sizeof (struct fileinfo));
    	  memcpy (l, &cur, sizeof (cur));
    	  l->next = NULL;
    	}
      else
    	{
    	  //cur.prev = l;
    	  l->next = (struct fileinfo *) malloc (sizeof (struct fileinfo));
    	  l = l->next;
    	  memcpy (l, &cur, sizeof (cur));
    	  l->next = NULL;
    	}
      /* Get the current time.  */
      timenow = time (NULL);
      tnow = localtime (&timenow);
      /* Build the time-stamp (the idea by zaga@fly.cc.fer.hr).  */
      timestruct.tm_sec   = sec;
      timestruct.tm_min   = min;
      timestruct.tm_hour  = hour;
      timestruct.tm_mday  = day;
      timestruct.tm_mon   = month;
      if (year == 0)
    	{
    	  /* Some listings will not specify the year if it is "obvious"
    	     that the file was from the previous year.  E.g. if today
    	     is 97-01-12, and you see a file of Dec 15th, its year is
    	     1996, not 1997.  Thanks to Vladimir Volovich for
    	     mentioning this!  */
    	  if (month > tnow->tm_mon)
    	    timestruct.tm_year = tnow->tm_year - 1;
    	  else
    	    timestruct.tm_year = tnow->tm_year;
    	}
      else
	      timestruct.tm_year = year;
      if (timestruct.tm_year >= 1900)
	      timestruct.tm_year -= 1900;
      timestruct.tm_wday  = 0;
      timestruct.tm_yday  = 0;
      timestruct.tm_isdst = -1;
      l->tstamp = mktime (&timestruct); /* store the time-stamp */
    }

  return dir;
}

static struct fileinfo *
ftp_parse_winnt_ls (char *listbuf)
{
  int len;
  int year, month, day;         /* for time analysis */
  int hour, min;
  struct tm timestruct;

  char *nextline = listbuf;
  char *line, *tok;             /* tokenizer */
  struct fileinfo *dir, *l, cur; /* list creation */
  
  dir = l = NULL;

  /* Line loop to end of file: */
  while (*(line = nextline))
    {
      len = clean_line (line, &nextline);

      /* Extracting name is a bit of black magic and we have to do it
         before `strtok' inserted extra \0 characters in the line
         string. For the moment let us just suppose that the name starts at
         column 39 of the listing. This way we could also recognize
         filenames that begin with a series of space characters (but who
         really wants to use such filenames anyway?). */
      if (len < 40) continue;
      tok = line + 39;
      cur.name = cpy(tok);
      printout(vDEBUG, "Name: '%s'\n", cur.name);

      /* First column: mm-dd-yy. Should atoi() on the month fail, january
	     * will be assumed.  */
      tok = strtok(line, "-");
      month = atoi(tok) - 1;
      if (month < 0) month = 0;
      tok = strtok(NULL, "-");
      day = atoi(tok);
      tok = strtok(NULL, " ");
      year = atoi(tok);
      /* Assuming the epoch starting at 1.1.1970 */
      if (year <= 70) year += 100;

      /* Second column: hh:mm[AP]M, listing does not contain value for
         seconds */
      tok = strtok(NULL,  ":");
      hour = atoi(tok);
      tok = strtok(NULL,  "M");
      min = atoi(tok);
      /* Adjust hour from AM/PM. Just for the record, the sequence goes
         11:00AM, 12:00PM, 01:00PM ... 11:00PM, 12:00AM, 01:00AM . */
      tok+=2;
      if (hour == 12)  hour  = 0;
      if (*tok == 'P') hour += 12;

      printout(vDEBUG, "YYYY/MM/DD HH:MM - %d/%d/%d %d:%d\n", 
              year+1900, month, day, hour, min);
      
      /* Build the time-stamp (copy & paste from above) */
      timestruct.tm_sec   = 0;
      timestruct.tm_min   = min;
      timestruct.tm_hour  = hour;
      timestruct.tm_mday  = day;
      timestruct.tm_mon   = month;
      timestruct.tm_year  = year;
      timestruct.tm_wday  = 0;
      timestruct.tm_yday  = 0;
      timestruct.tm_isdst = -1;
      cur.tstamp = mktime (&timestruct); /* store the time-stamp */

      printout(vDEBUG, "Timestamp: %ld\n", cur.tstamp);

      /* Third column: Either file length, or <DIR>. We also set the
         permissions (guessed as 0644 for plain files and 0755 for
         directories as the listing does not give us a clue) and filetype
         here. */
      tok = strtok(NULL, " ");
      while (*tok == '\0')  tok = strtok(NULL, " ");
      if (*tok == '<')
    	{
    	  cur.type  = FT_DIRECTORY;
    	  cur.size  = 0;
    	  //cur.perms = 0755;
    	  printout(vDEBUG, "Directory\n");
    	}
      else
    	{
    	  cur.type  = FT_PLAINFILE;
    	  cur.size  = atoi(tok);
    	  //cur.perms = 0644;
    	  printout(vDEBUG, "File, size %d bytes\n", cur.size);
    	}

      //cur.linkto = NULL;

      /* And put everything into the linked list */
      if (!dir)
    	{
    	  l = dir = (struct fileinfo *) malloc (sizeof (struct fileinfo));
    	  memcpy (l, &cur, sizeof (cur));
    	  l->next = NULL;
    	}
      else
    	{
    	  //cur.prev = l;
    	  l->next = (struct fileinfo *) malloc (sizeof (struct fileinfo));
    	  l = l->next;
    	  memcpy (l, &cur, sizeof (cur));
    	  l->next = NULL;
    	}
    }

  return dir;
}


static struct fileinfo *
ftp_parse_vms_ls (char * listing)
{
  /* #### A third copy of more-or-less the same array ? */
  static const char *months[] = {
    "JAN", "FEB", "MAR", "APR", "MAY", "JUN",
    "JUL", "AUG", "SEP", "OCT", "NOV", "DEC"
  };
  int i;
  int year, month, day;          /* for time analysis */
  int hour, min, sec;
  struct tm timestruct;

  char *line, *tok;		 /* tokenizer */
  struct fileinfo *dir, *l, cur; /* list creation */
  char *nextline;

  dir = l = NULL;
  line = listing;

  /* Skip empty line. */
  /* TODO has VMS \r\ns? maybe this OS does not work with wput anyway... */
  line = strchr(line, '\n');
  line++;
    
  /* Skip "Directory PUB$DEVICE[PUB]" */
  line = strchr(line, '\n');
  line++;

  /* Skip empty line. */
  line = strchr(line, '\n');
  nextline = ++line;
  
  /* Line loop to end of buffer: */
  while (*(line = nextline))
    {
      char *p;
      i = clean_line(line, &nextline);
      if(!i) break;

      /* First column: Name. A bit of black magic again. The name my be
         either ABCD.EXT or ABCD.EXT;NUM and it might be on a separate
         line. Therefore we will first try to get the complete name
         until the first space character; if it fails, we assume that the name
         occupies the whole line. After that we search for the version
         separator ";", we remove it and check the extension of the file;
         extension .DIR denotes directory. */

      tok = strtok(line, " ");
      if (tok == NULL) tok = line;
      printout(vDEBUG, "file name: '%s'\n", tok);
      
      for (p = tok ; *p && *p != ';' ; p++);
      if (*p == ';') *p = '\0';
      p   = tok + strlen(tok) - 4;
      if (!strcmp(p, ".DIR")) *p = '\0';
      cur.name = cpy(tok);
      printout(vDEBUG, "Name: '%s'\n", cur.name);

      /* If the name ends on .DIR or .DIR;#, it's a directory. We also set
         the file size to zero as the listing does tell us only the size in
         filesystem blocks - for an integrity check (when mirroring, for
         example) we would need the size in bytes. */
      
      if (! *p)
        {
          cur.type  = FT_DIRECTORY;
          cur.size  = 0;
          printout(vDEBUG, "Directory\n");
        }
      else
        {
          cur.type  = FT_PLAINFILE;
          printout(vDEBUG, "File\n");
        }

      cur.size  = 0;

      /* Second column, if exists, or the first column of the next line
         contain file size in blocks. We will skip it. */

      tok = strtok(NULL, " ");
      if (tok == NULL) 
      {
        printout(vDEBUG, "Getting additional line\n");
        line = nextline;
        i = clean_line (line, &nextline);
        if (!i) 
        {
          printout(vDEBUG, "confusing VMS listing item, leaving listing parser\n");
	        break;
        }
        tok = strtok(line, " ");
      }
      printout(vDEBUG, "second token: '%s'\n", tok);

      /* Third/Second column: Date DD-MMM-YYYY. */

      tok = strtok(NULL, "-");
      printout(vDEBUG, "day: '%s'\n",tok);
      day = atoi(tok);
      tok = strtok(NULL, "-");
      if (!tok)
      {
        /* If the server produces garbage like
           'EA95_0PS.GZ;1      No privilege for attempted operation'
           the first strtok(NULL, "-") will return everything until the end
           of the line and only the next strtok() call will return NULL. */
        printout(vDEBUG, "nonsense in VMS listing, skipping this line\n");
	      break;
	    }
      for (i=0; i<12; i++) if (!strcmp(tok,months[i])) break;
      /* Uknown months are mapped to January */
      month = i % 12 ; 
      tok = strtok (NULL, " ");
      year = atoi (tok) - 1900;
      printout(vDEBUG, "date parsed\n");

      /* Fourth/Third column: Time hh:mm[:ss] */
      tok = strtok (NULL, " ");
      hour = min = sec = 0;
      p = tok;
      hour = atoi (p);
      for (; *p && *p != ':'; ++p);
      if (*p)
	min = atoi (++p);
      for (; *p && *p != ':'; ++p);
      if (*p)
	sec = atoi (++p);

      printout(vDEBUG, "YYYY/MM/DD HH:MM:SS - %d/%d/%d %d:%d:%d\n", 
              year+1900, month, day, hour, min, sec);
      
      /* Build the time-stamp (copy & paste from above) */
      timestruct.tm_sec   = sec;
      timestruct.tm_min   = min;
      timestruct.tm_hour  = hour;
      timestruct.tm_mday  = day;
      timestruct.tm_mon   = month;
      timestruct.tm_year  = year;
      timestruct.tm_wday  = 0;
      timestruct.tm_yday  = 0;
      timestruct.tm_isdst = -1;
      cur.tstamp = mktime (&timestruct); /* store the time-stamp */

      /* TODO tstamp is time_t (is this int?) if not we get probably a crash here */
      printout(vDEBUG, "Timestamp: %d\n", cur.tstamp);

      /* Skip the fifth column */

      tok = strtok(NULL, " ");

      /* Sixth column: Permissions */

      tok = strtok(NULL, ","); /* Skip the VMS-specific SYSTEM permissons */
      tok = strtok(NULL, ")");
      if (tok == NULL)
        {
          printout(vDEBUG, "confusing VMS permissions, skipping line\n");
          continue;
        }
      /* Permissons have the format "RWED,RWED,RE" */
      //cur.perms = vmsperms(tok);
      //printout(vDEBUG, "permissions: %s -> 0%o\n", tok, cur.perms);

      //cur.linkto = NULL;

      /* And put everything into the linked list */
      if (!dir)
        {
          l = dir = (struct fileinfo *) malloc (sizeof (struct fileinfo));
          memcpy (l, &cur, sizeof (cur));
          //l->prev = l->next = NULL;
        }
      else
        {
          //cur.prev = l;
          l->next = (struct fileinfo *) malloc (sizeof (struct fileinfo));
          l = l->next;
          memcpy (l, &cur, sizeof (cur));
          l->next = NULL;
        }
    }

  return dir;
}


/* This function switches between the correct parsing routine depending on
   the SYSTEM_TYPE. The system type should be based on the result of the
   "SYST" response of the FTP server. According to this repsonse we will
   use on of the three different listing parsers that cover the most of FTP
   servers used nowadays.  */

struct fileinfo *
ftp_parse_ls (char * listing, const enum stype system_type)
{
  switch (system_type)
    {
    case ST_UNIX:
      return ftp_parse_unix_ls (listing);
    case ST_WINNT:
  	  /* Detect whether the listing is simulating the UNIX format */
  	  /* If the first character of the file is '0'-'9', it's WINNT
  	   * format. */
  	  if (*listing >= '0' && *listing <='9')
  	    return ftp_parse_winnt_ls (listing);
      else
        return ftp_parse_unix_ls (listing);
    case ST_VMS:
      return ftp_parse_vms_ls  (listing);
    case ST_MACOS:
      return ftp_parse_unix_ls (listing);
    default:
      printout(vNORMAL, "Unsupported listing type, trying Unix listing parser.\n");
      return ftp_parse_unix_ls (listing);
    }
}
directory_list * add_directory(directory_list * A, struct fileinfo * K) {
    if(A == NULL) {
        A = (directory_list *) malloc(sizeof(directory_list));
        A->list = K;
        A->name = cpy(cc.current_directory);
        A->next = NULL;
    } else
        A->next = add_directory(A->next, K);
    return A;
}
void fileinfo_free(void) {
    directory_list  * K = cc.directorylist;
    directory_list  * L;
    struct fileinfo * M;
    struct fileinfo * N;
    while(K != NULL) {
        L = K->next;
        free(K->name);
        M = K->list;
        while(M != NULL) {
            N = M->next;
            if(M->name) free(M->name);
            free(M);
            M = N;
        }
        free(K);
        K = L;
    }
}
    
struct fileinfo * fileinfo_find_file(struct fileinfo * F, char * name) {
    while(F != NULL) {
        if( !strcmp(F->name, name) ) return F;
        F = F->next;
    }
    return NULL;
}
struct fileinfo * find_directory(_fsession * F) {
    directory_list * K = cc.directorylist;
    while(K != NULL) {
        if( !strcmp(K->name, F->target_dname) ) return K->list;
        K = K->next;
    }
    return NULL;
}
#ifdef STANDALONE
char * email_address = "none";
int main(int argc, char ** argv) {
  FILE * f = fopen(argv[1], "r");
  char * line = read_line(f);
  char * l2;
  struct fileinfo *dir;
  opt.verbose = vDEBUG;
  while( (l2 = read_line(f)) ) {
    line = realloc(line, strlen(line) + strlen(l2) + 1);
    strcat(line, l2);
    free(l2);
  }
  printout(vDEBUG, "Loaded file successfully:\n%s\n-----\n", line);
  dir = ftp_parse_ls(line, ST_UNIX);
  free(line);
  while(dir != NULL) {
    printout(vDEBUG, "Type: %d; `%s' [%d] %d\n", dir->type, dir->name, (int) dir->size, dir->tstamp);
    dir = dir->next;
  }
}
#endif
