#!/usr/bin/env python
# -*- coding: utf-8 -*-

#This file is part of SVGUIEditor, a library implementing an editor for SVGUI
#interfaces
#
#Copyright (C) 2007: Edouard TISSERANT, Laurent BESSARD and Jonathan HURTREL
#
#See COPYING file for copyrights details.
#
#This library is free software; you can redistribute it and/or
#modify it under the terms of the GNU General Public
#License as published by the Free Software Foundation; either
#version 2.1 of the License, or (at your option) any later version.
#
#This library is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#General Public License for more details.
#
#You should have received a copy of the GNU General Public
#License along with this library; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os, sys, platform, time, traceback, getopt, locale

import wx, wx.grid, wx.wxsvg

if wx.VERSION >= (2, 8, 0):
    import wx.aui

from SVGUIControler import *

__version__ = "$Revision: 1.12 $"

CWD = os.path.split(os.path.realpath(__file__))[0]

HANDLE_SIZE = 6

def ComputeBBox(bbox):
    return wx.Rect(bbox.GetX(), bbox.GetY(), bbox.GetWidth(), bbox.GetHeight())

def AppendMenu(parent, help, id, kind, text):
    if wx.VERSION >= (2, 6, 0):
        parent.Append(help=help, id=id, kind=kind, text=text)
    else:
        parent.Append(helpString=help, id=id, kind=kind, item=text)


#-------------------------------------------------------------------------------
#                         Helper for AttributesGrid values
#-------------------------------------------------------------------------------


class AttributesTable(wx.grid.PyGridTableBase):
    
    """
    A custom wxGrid Table using user supplied data
    """
    def __init__(self, parent, data, colnames):
        # The base class must be initialized *first*
        wx.grid.PyGridTableBase.__init__(self)
        self.data = data
        self.colnames = colnames
        self.Parent = parent
        # XXX
        # we need to store the row length and collength to
        # see if the table has changed size
        self._rows = self.GetNumberRows()
        self._cols = self.GetNumberCols()
    
    def GetNumberCols(self):
        return len(self.colnames)
        
    def GetNumberRows(self):
        return len(self.data)

    def GetColLabelValue(self, col):
        if col < len(self.colnames):
            return self.colnames[col]

    def GetRowLabelValues(self, row):
        return row

    def GetValue(self, row, col):
        if row < self.GetNumberRows():
            name = str(self.data[row].get(self.GetColLabelValue(col), ""))
            return name
    
    def GetValueByName(self, row, colname):
        return self.data[row].get(colname)

    def SetValue(self, row, col, value):
        if col < len(self.colnames):
            self.data[row][self.GetColLabelValue(col)] = value
        
    def ResetView(self, grid):
        """
        (wxGrid) -> Reset the grid view.   Call this to
        update the grid if rows and columns have been added or deleted
        """
        grid.BeginBatch()
        for current, new, delmsg, addmsg in [
            (self._rows, self.GetNumberRows(), wx.grid.GRIDTABLE_NOTIFY_ROWS_DELETED, wx.grid.GRIDTABLE_NOTIFY_ROWS_APPENDED),
            (self._cols, self.GetNumberCols(), wx.grid.GRIDTABLE_NOTIFY_COLS_DELETED, wx.grid.GRIDTABLE_NOTIFY_COLS_APPENDED),
        ]:
            if new < current:
                msg = wx.grid.GridTableMessage(self,delmsg,new,current-new)
                grid.ProcessTableMessage(msg)
            elif new > current:
                msg = wx.grid.GridTableMessage(self,addmsg,new-current)
                grid.ProcessTableMessage(msg)
                self.UpdateValues(grid)
        grid.EndBatch()

        self._rows = self.GetNumberRows()
        self._cols = self.GetNumberCols()
        # update the column rendering scheme
        self._updateColAttrs(grid)

        # update the scrollbars and the displayed part of the grid
        grid.AdjustScrollbars()
        grid.ForceRefresh()

    def UpdateValues(self, grid):
        """Update all displayed values"""
        # This sends an event to the grid table to update all of the values
        msg = wx.grid.GridTableMessage(self, wx.grid.GRIDTABLE_REQUEST_VIEW_GET_VALUES)
        grid.ProcessTableMessage(msg)

    def _updateColAttrs(self, grid):
        """
        wxGrid -> update the column attributes to add the
        appropriate renderer given the column name.

        Otherwise default to the default renderer.
        """
        
        grid.SetColSize(0, 120)
        grid.SetColSize(1, 100)
        
        typelist = None
        accesslist = None
        for row in range(self.GetNumberRows()):
            for col in range(self.GetNumberCols()):
                editor = None
                renderer = None
                colname = self.GetColLabelValue(col)
                grid.SetReadOnly(row, col, False)
                
                if colname == "Value":                
                    value_type = self.GetValueByName(row, "type")
                    if isinstance(value_type, ListType):
                        editor = wx.grid.GridCellChoiceEditor()
                        editor.SetParameters(",".join(value_type))
                    elif value_type == "decimal":
                        editor = wx.grid.GridCellFloatEditor()
                        grid.SetCellAlignment(row, col, wx.ALIGN_RIGHT, wx.ALIGN_CENTER)
                    elif value_type == "integer":
                        editor = wx.grid.GridCellNumberEditor()
                        grid.SetCellAlignment(row, col, wx.ALIGN_RIGHT, wx.ALIGN_CENTER)
                    elif value_type == "string":
                        editor = wx.grid.GridCellTextEditor()
                    elif value_type == "boolean":
                        editor = wx.grid.GridCellChoiceEditor()
                        editor.SetParameters(",Yes,No")
                else:
                    grid.SetReadOnly(row, col, True)
                
                grid.SetCellEditor(row, col, editor)
                grid.SetCellRenderer(row, col, renderer)
                
                grid.SetCellBackgroundColour(row, col, wx.WHITE)
    
    def SetData(self, data):
        self.data = data
    
    def GetData(self):
        return self.data
    
    def GetCurrentIndex(self):
        return self.CurrentIndex
    
    def SetCurrentIndex(self, index):
        self.CurrentIndex = index
    
    def AppendRow(self, row_content):
        self.data.append(row_content)

    def RemoveRow(self, row_index):
        self.data.pop(row_index)

    def GetRow(self, row_index):
        return self.data[row_index]

    def Empty(self):
        self.data = []
        self.editors = []


#-------------------------------------------------------------------------------
#                          SVGUIEditor Main Frame Class
#-------------------------------------------------------------------------------


[ID_SVGUIEDITOR, ID_SVGUIEDITORMAINSPLITTER, 
 ID_SVGUIEDITORSECONDSPLITTER, ID_SVGUIEDITORINTERFACETREE, 
 ID_SVGUIEDITORSVGCTRL, ID_SVGUIEDITOREDITORPANEL, 
 ID_SVGUIEDITORATTRIBUTESGRID, 
] = [wx.NewId() for _init_ctrls in range(7)]

[ID_SVGUIEDITORFILEMENUGENERATE, 
] = [wx.NewId() for _init_ctrls in range(1)]

class SVGUIEditor(wx.Frame):
    
    if wx.VERSION < (2, 6, 0):
        def Bind(self, event, function, id = None):
            if id is not None:
                event(self, id, function)
            else:
                event(self, function)
    
    def _init_coll_FileMenu_Items(self, parent):
        AppendMenu(parent, help='Create a new DEF File', id=wx.ID_NEW,
              kind=wx.ITEM_NORMAL, text=u'&New\tCTRL+N')
        AppendMenu(parent, help='Open a DEF File', id=wx.ID_OPEN,
              kind=wx.ITEM_NORMAL, text=u'&Open\tCTRL+O')
        AppendMenu(parent, help='Close this DEF File', id=wx.ID_CLOSE,
              kind=wx.ITEM_NORMAL, text=u'&Close\tCTRL+F')
        parent.AppendSeparator()
        AppendMenu(parent, help='Save this DEF File', id=wx.ID_SAVE,
              kind=wx.ITEM_NORMAL, text=u'&Save\tCTRL+S')
        AppendMenu(parent, help='Save this DEF File with a new name', id=wx.ID_SAVEAS,
              kind=wx.ITEM_NORMAL, text=u'Save &As...\tCTRL+SHIFT+S')
        AppendMenu(parent, help='Generate Program', id=ID_SVGUIEDITORFILEMENUGENERATE,
              kind=wx.ITEM_NORMAL, text=u'&Generate Program\tCTRL+G')
        parent.AppendSeparator()
        AppendMenu(parent, help='Quit Program', id=wx.ID_EXIT,
              kind=wx.ITEM_NORMAL, text=u'&Quit\tCTRL+Q')
        self.Bind(wx.EVT_MENU, self.OnNewMenu, id=wx.ID_NEW)
        self.Bind(wx.EVT_MENU, self.OnOpenMenu, id=wx.ID_OPEN)
        self.Bind(wx.EVT_MENU, self.OnCloseMenu, id=wx.ID_CLOSE)
        self.Bind(wx.EVT_MENU, self.OnSaveMenu, id=wx.ID_SAVE)
        self.Bind(wx.EVT_MENU, self.OnSaveAsMenu, id=wx.ID_SAVEAS)
        self.Bind(wx.EVT_MENU, self.OnGenerateMenu, 
              id=ID_SVGUIEDITORFILEMENUGENERATE)
        self.Bind(wx.EVT_MENU, self.OnQuitMenu, id=wx.ID_EXIT)

    def _init_coll_EditMenu_Items(self, parent):
        AppendMenu(parent, help='', id=wx.ID_REFRESH,
              kind=wx.ITEM_NORMAL, text=u'Refresh\tCTRL+R')
        AppendMenu(parent, help='', id=wx.ID_UNDO,
              kind=wx.ITEM_NORMAL, text=u'Undo\tCTRL+Z')
        AppendMenu(parent, help='', id=wx.ID_REDO,
              kind=wx.ITEM_NORMAL, text=u'Redo\tCTRL+Y')
        parent.AppendSeparator()
        AppendMenu(parent, help='', id=wx.ID_ADD,
              kind=wx.ITEM_NORMAL, text=u'&Add Element\tCTRL+A')
        AppendMenu(parent, help='', id=wx.ID_DELETE,
              kind=wx.ITEM_NORMAL, text=u'&Delete Element\tCTRL+D')
        AppendMenu(parent, help='', id=wx.ID_CUT,
              kind=wx.ITEM_NORMAL, text=u'&Cut Element\tCTRL+X')
        AppendMenu(parent, help='', id=wx.ID_PASTE,
              kind=wx.ITEM_NORMAL, text=u'&Paste Element\tCTRL+V')
        self.Bind(wx.EVT_MENU, self.OnRefreshMenu, id=wx.ID_REFRESH)
        self.Bind(wx.EVT_MENU, self.OnUndoMenu, id=wx.ID_UNDO)
        self.Bind(wx.EVT_MENU, self.OnRedoMenu, id=wx.ID_REDO)
        self.Bind(wx.EVT_MENU, self.OnAddMenu, id=wx.ID_ADD)
        self.Bind(wx.EVT_MENU, self.OnDeleteMenu, id=wx.ID_DELETE)
        self.Bind(wx.EVT_MENU, self.OnCutMenu, id=wx.ID_CUT)
        self.Bind(wx.EVT_MENU, self.OnPasteMenu, id=wx.ID_PASTE)
    
##    def _init_coll_HelpMenu_Items(self, parent):
##        AppendMenu(parent, help='', id=ID_PLCOPENEDITORHELPMENUITEMS0, 
##            kind=wx.ITEM_NORMAL, text=u'PLCOpenEditor\tF1')
##        AppendMenu(parent, help='', id=ID_PLCOPENEDITORHELPMENUITEMS1,
##              kind=wx.ITEM_NORMAL, text=u'PLCOpen\tF2')
##        AppendMenu(parent, help='', id=ID_PLCOPENEDITORHELPMENUITEMS2,
##              kind=wx.ITEM_NORMAL, text=u'IEC 61131-3\tF3')
##        if self.ModeSolo:
##            AppendMenu(parent, help='', id=ID_PLCOPENEDITORHELPMENUITEMS3,
##                  kind=wx.ITEM_NORMAL, text=u'About')
##        self.Bind(wx.EVT_MENU, self.OnPLCOpenMenu,
##              id=ID_PLCOPENEDITORHELPMENUITEMS1)

    def _init_coll_MenuBar_Menus(self, parent):
        if self.ModeSolo:
            parent.Append(menu=self.FileMenu, title=u'&File')
        parent.Append(menu=self.EditMenu, title=u'&Edit')
        #parent.Append(menu=self.HelpMenu, title=u'Help')
    
    def _init_utils(self):
        self.MenuBar = wx.MenuBar()

        if self.ModeSolo:
            self.FileMenu = wx.Menu(title='')
        else:
            self.FileMenu = None
        self.EditMenu = wx.Menu(title='')
        #self.HelpMenu = wx.Menu(title='')
        
        self._init_coll_MenuBar_Menus(self.MenuBar)
        if self.ModeSolo:
            self._init_coll_FileMenu_Items(self.FileMenu)
        self._init_coll_EditMenu_Items(self.EditMenu)
        #self._init_coll_HelpMenu_Items(self.HelpMenu)

    def _init_coll_EditorGridSizer_Items(self, parent):
        # generated method, don't edit
        parent.AddWindow(self.AttributesGrid, 0, border=0, flag=wx.GROW)
    
    def _init_coll_EditorGridSizer_Growables(self, parent):
        # generated method, don't edit
        parent.AddGrowableCol(0)
        parent.AddGrowableCol(1)
        parent.AddGrowableRow(0)

    def _init_sizers(self):
        # generated method, don't edit
        self.EditorGridSizer = wx.FlexGridSizer(cols=2, hgap=0, rows=1, vgap=0)

        self._init_coll_EditorGridSizer_Items(self.EditorGridSizer)
        self._init_coll_EditorGridSizer_Growables(self.EditorGridSizer)
        
        self.EditorPanel.SetSizer(self.EditorGridSizer)
    
    def _init_ctrls(self, prnt, init_sizers=True):
        wx.Frame.__init__(self, id=ID_SVGUIEDITOR, name=u'SVGUIEditor',
                  parent=prnt, pos=wx.DefaultPosition, size=wx.Size(800, 650),
                  style=wx.DEFAULT_FRAME_STYLE, title=u'SVGUIEditor')
        self._init_utils()
        self.SetClientSize(wx.Size(1000, 600))
        self.SetMenuBar(self.MenuBar)
        self.Bind(wx.EVT_CLOSE, self.OnCloseFrame)
        if not self.ModeSolo:
            self.Bind(wx.EVT_MENU, self.OnSaveMenu, id=wx.ID_SAVE)
            accel = wx.AcceleratorTable([wx.AcceleratorEntry(wx.ACCEL_CTRL, 83, wx.ID_SAVE)])
            self.SetAcceleratorTable(accel)
        
        if wx.VERSION >= (2, 8, 0):
            self.AUIManager = wx.aui.AuiManager(self)
            self.AUIManager.SetDockSizeConstraint(0.5, 0.5)
        
        if wx.VERSION < (2, 8, 0):
            self.MainSplitter = wx.SplitterWindow(id=ID_SVGUIEDITORMAINSPLITTER, 
                  name='MainSplitter', parent=self, point=wx.Point(0, 0),
                  size=wx.Size(-1, -1), style=wx.SP_3D)
            self.MainSplitter.SetNeedUpdating(True)
            self.MainSplitter.SetMinimumPaneSize(1)
        
            self.InterfaceTree = wx.TreeCtrl(id=ID_SVGUIEDITORINTERFACETREE, 
                  name='InterfaceTree', parent=self.MainSplitter, pos=wx.Point(0, 0),
                  size=wx.Size(-1, -1), style=wx.TR_HAS_BUTTONS|wx.TR_SINGLE|wx.SUNKEN_BORDER)
        else:
            self.InterfaceTree = wx.TreeCtrl(id=ID_SVGUIEDITORINTERFACETREE, 
                  name='InterfaceTree', parent=self, pos=wx.Point(0, 0),
                  size=wx.Size(-1, -1), style=wx.TR_HAS_BUTTONS|wx.TR_SINGLE|wx.SUNKEN_BORDER)
            self.AUIManager.AddPane(self.InterfaceTree, wx.aui.AuiPaneInfo().Caption("Interface Tree").Left().Layer(1).BestSize(wx.Size(200, 500)).CloseButton(False))
        self.Bind(wx.EVT_TREE_SEL_CHANGING, self.OnInterfaceTreeItemChanging, 
              id=ID_SVGUIEDITORINTERFACETREE)
        self.Bind(wx.EVT_TREE_SEL_CHANGED, self.OnInterfaceTreeItemSelected, 
              id=ID_SVGUIEDITORINTERFACETREE)
        self.Bind(wx.EVT_TREE_BEGIN_DRAG, self.OnInterfaceTreeBeginDrag, 
              id=ID_SVGUIEDITORINTERFACETREE)
        self.Bind(wx.EVT_TREE_END_DRAG, self.OnInterfaceTreeEndDrag, 
              id=ID_SVGUIEDITORINTERFACETREE)
        
        if wx.VERSION < (2, 8, 0):
            self.SecondSplitter = wx.SplitterWindow(id=ID_SVGUIEDITORINTERFACETREE,
                  name='SecondSplitter', parent=self.MainSplitter, point=wx.Point(0, 0),
                  size=wx.Size(-1, -1), style=wx.SP_3D)
            self.SecondSplitter.SetNeedUpdating(True)
            self.SecondSplitter.SetMinimumPaneSize(1)
            
            self.MainSplitter.SplitVertically(self.InterfaceTree, self.SecondSplitter, 200)
            
            self.SVGCtrl = wx.wxsvg.SVGUIWindow(id=ID_SVGUIEDITORSVGCTRL,
                  name='SVGCtrl', parent=self.SecondSplitter, pos=wx.Point(0, 0),
                  size=wx.Size(-1, -1), style=0)
        else:
            self.SVGCtrl = wx.wxsvg.SVGUIWindow(id=ID_SVGUIEDITORSVGCTRL,
                  name='SVGCtrl', parent=self, pos=wx.Point(0, 0),
                  size=wx.Size(-1, -1), style=0)
            self.AUIManager.AddPane(self.SVGCtrl, wx.aui.AuiPaneInfo().CenterPane())
        self.SVGCtrl.Bind(wx.EVT_LEFT_DOWN, self.OnSVGCtrlClick)
        self.SVGCtrl.Bind(wx.EVT_PAINT, self.OnPaintSVGctrl)
        
        if wx.VERSION < (2, 8, 0):
            self.EditorPanel = wx.Panel(id=ID_SVGUIEDITOREDITORPANEL, 
                  name='Panel', parent=self.SecondSplitter, pos=wx.Point(0, 0),
                  size=wx.Size(-1, -1), style=wx.TAB_TRAVERSAL)
                
            self.SecondSplitter.SplitHorizontally(self.SVGCtrl, self.EditorPanel, -200)
        else:
            self.EditorPanel = wx.Panel(id=ID_SVGUIEDITOREDITORPANEL, 
                  name='Panel', parent=self, pos=wx.Point(0, 0),
                  size=wx.Size(-1, -1), style=wx.TAB_TRAVERSAL)
            self.AUIManager.AddPane(self.EditorPanel, wx.aui.AuiPaneInfo().Caption("Editor Panel").Bottom().Layer(0).BestSize(wx.Size(800, 200)).CloseButton(False))
        
        self.AttributesGrid = wx.grid.Grid(id=ID_SVGUIEDITORATTRIBUTESGRID, 
                  name='AttributesGrid', parent=self.EditorPanel, pos=wx.Point(0, 0),
                  size=wx.Size(0, 0), style=wx.VSCROLL)
        self.AttributesGrid.SetRowLabelSize(0)
        self.AttributesGrid.SetFont(wx.Font(12, 77, wx.NORMAL, wx.NORMAL, False, 'Sans'))
        self.AttributesGrid.SetLabelFont(wx.Font(10, 77, wx.NORMAL, wx.NORMAL, False, 'Sans'))
        self.AttributesGrid.Bind(wx.grid.EVT_GRID_CELL_CHANGE, self.OnAttributesGridCellChange)
        self.AttributesGrid.Bind(wx.grid.EVT_GRID_SELECT_CELL, self.OnAttributesGridSelectCell)
        
        self.StatusBar = wx.StatusBar( name='HelpBar',
              parent=self, style=wx.ST_SIZEGRIP)
        self.SetStatusBar(self.StatusBar)
        
        if init_sizers:
            self._init_sizers()
        if wx.VERSION >= (2, 8, 0):
            self.AUIManager.Update()
    
    def __init__(self, parent, controler = None, fileOpen = None):
        self.ModeSolo = controler == None
        if self.ModeSolo:
            self.Controler = SVGUIControler()
            if fileOpen is not None:
                self.Controler.OpenXMLFile(fileOpen)
        else:
            self.Controler = controler
        
        self._init_ctrls(parent)
        
        locale.setlocale(locale.LC_ALL, 'C')
        
        self.Table = AttributesTable(self, [], ["Attribute", "Value"])
        self.AttributesGrid.SetTable(self.Table)
        
        self.SelectedSVGElement = None
        self.DraggingObject = None
        
        self.RefreshSVGCtrl()
        self.RefreshView()

    def RefreshTitle(self):
        if self.Controler.HasOpenedInterface() > 0:
            self.SetTitle("SVGUIEditor - %s"%self.Controler.GetFilename())
        else:
            self.SetTitle("SVGUIEditor")

    def RefreshView(self, select_id = None):
        self.RefreshTitle()
        self.RefreshInterfaceTree()
        if select_id is not None:
            self.SelectInterfaceTreeItem(self.InterfaceTree.GetRootItem(), select_id)
        else:
            self.RefreshAttributesGrid()
            self.RefreshFileMenu()
            self.RefreshEditMenu()


#-------------------------------------------------------------------------------
#                       SVGUIEditor menus management functions
#-------------------------------------------------------------------------------


    def RefreshFileMenu(self):
        if self.ModeSolo:
            if self.Controler.HasOpenedInterface() and len(self.GetGridRequiredNotFiled()) == 0:
                self.FileMenu.Enable(wx.ID_CLOSE, True)
                self.FileMenu.Enable(wx.ID_SAVE, True)
                self.FileMenu.Enable(wx.ID_SAVEAS, True)
                self.FileMenu.Enable(ID_SVGUIEDITORFILEMENUGENERATE, True)
            else:
                self.FileMenu.Enable(wx.ID_CLOSE, False)
                self.FileMenu.Enable(wx.ID_SAVE, False)
                self.FileMenu.Enable(wx.ID_SAVEAS, False)
                self.FileMenu.Enable(ID_SVGUIEDITORFILEMENUGENERATE, False)
                
    def RefreshEditMenu(self):
        if self.Controler.HasOpenedInterface():
            if self.ModeSolo:
                self.MenuBar.EnableTop(1, True)
            else:
                self.MenuBar.EnableTop(0, True)
            selected = self.GetSelected()
            undo, redo = self.Controler.GetBufferState()
            self.EditMenu.Enable(wx.ID_UNDO, undo)
            self.EditMenu.Enable(wx.ID_REDO, redo)
            if selected is None or len(self.GetGridRequiredNotFiled()) > 0:
                self.EditMenu.Enable(wx.ID_ADD, False)
                self.EditMenu.Enable(wx.ID_DELETE, False)
                self.EditMenu.Enable(wx.ID_CUT, False)
                self.EditMenu.Enable(wx.ID_PASTE, False)
            else:
                if self.Controler.IsElementContainer(selected) or self.Controler.IsElementNoteBook(selected):
                    self.EditMenu.Enable(wx.ID_ADD, True)
                    if self.Controler.IsCutBufferEmpty():
                        self.EditMenu.Enable(wx.ID_PASTE, False)
                    else:
                        self.EditMenu.Enable(wx.ID_PASTE, True)
                else:
                    self.EditMenu.Enable(wx.ID_ADD, False)
                    self.EditMenu.Enable(wx.ID_PASTE, False)
                if self.Controler.IsElementInterface(selected):
                    self.EditMenu.Enable(wx.ID_DELETE, False)
                    self.EditMenu.Enable(wx.ID_CUT, False)
                else:
                    self.EditMenu.Enable(wx.ID_DELETE, True)
                    self.EditMenu.Enable(wx.ID_CUT, True)
        elif self.ModeSolo:
            self.MenuBar.EnableTop(1, False)
        else:
            self.MenuBar.EnableTop(0, False)


#-------------------------------------------------------------------------------
#                        SVGUIEditor menus callback functions
#-------------------------------------------------------------------------------


    def SaveRequest(self):
        if not self.Controler.InterfaceIsSaved():
            dialog = wx.MessageDialog(self, "There are changes, do you want to save?", "Close Application", wx.YES_NO|wx.CANCEL|wx.ICON_QUESTION)
            answer = dialog.ShowModal()
            dialog.Destroy()
            if answer == wx.ID_YES:
                self.SaveInterface()
                return True
            elif answer == wx.ID_CANCEL:
                return False
        return True

    def OnCloseFrame(self, event):
        if not self.ModeSolo and getattr(self, "_onclose", None) is not None:
            self.AUIManager.UnInit()
            self._onclose()
            event.Skip()
        elif not self.Controler.InterfaceIsSaved():
            dialog = wx.MessageDialog(self, "There are changes, do you want to save?", "Close Application", wx.YES_NO|wx.CANCEL|wx.ICON_QUESTION)
            answer = dialog.ShowModal()
            dialog.Destroy()
            if answer == wx.ID_YES:
                self.SaveInterface()
                self.AUIManager.UnInit()
                event.Skip()
            elif answer == wx.ID_NO:
                self.AUIManager.UnInit()
                event.Skip()
            else:
                event.Veto()
        else:
            event.Skip()

    def OnNewMenu(self, event):
        if self.SaveRequest():
            dialog = wx.TextEntryDialog(self, "New interface name", "Please enter an interface name", "", wx.OK|wx.CANCEL|wx.CENTRE)
            if dialog.ShowModal():
                self.Controler.CreateNewInterface(dialog.GetValue())
                self.RefreshView()
            dialog.Destroy()
        event.Skip()
    
    def OnOpenMenu(self, event):
        if self.SaveRequest():
            filepath = self.Controler.GetFilePath()
            if filepath != "":
                directory = os.path.dirname(filepath)
            else:
                directory = os.getcwd()
            dialog = wx.FileDialog(self, "Choose a file", directory, "",  "SVGUI files (*.xml)|*.xml|All files|*.*", wx.OPEN)
            if dialog.ShowModal() == wx.ID_OK:
                filepath = dialog.GetPath()
                if os.path.isfile(filepath):
                    self.Controler.OpenXMLFile(filepath)
                    if self.RefreshSVGCtrl():
                        self.StatusBar.SetStatusText("SVGUI file succesfully open with his SVG file")
                    else:
                        self.StatusBar.SetStatusText("No SVG File found with your SVGUI File")
                    self.RefreshView()
            dialog.Destroy()
        event.Skip()
            
    def OnCloseMenu(self, event):
        if self.SaveRequest():
            self.Controler.Reset()
            self.InterfaceTree.DeleteAllItems()
            wx.CallAfter(self.RefreshSVGCtrl)
            wx.CallAfter(self.RefreshView)
        event.Skip()
        
    def OnSaveMenu(self, event):
        self.SaveInterface()
        event.Skip()
        
    def OnSaveAsMenu(self, event):
        self.SaveInterfaceAs()
        event.Skip()
    
    def SaveInterfaceAs(self):
        filepath = self.Controler.GetFilePath()
        directory, filename = os.path.split(filepath)
        dialog = wx.FileDialog(self, "Choose a file", directory, filename,  "SVGUI files (*.xml)|*.xml|All files|*.*", wx.SAVE|wx.OVERWRITE_PROMPT)
        if dialog.ShowModal() == wx.ID_OK:
            filepath = dialog.GetPath()
            self.Controler.filepath = filepath
            if os.path.isdir(os.path.dirname(filepath)):
                result = self.Controler.SaveXMLFile(filepath)
                if not result:
                    message = wx.MessageDialog(self, "Can't save project to file %s!"%filepath, "Error", wx.OK|wx.ICON_ERROR)
                    message.ShowModal()
                    message.Destroy()
                else:
                    wx.CallAfter(self.RefreshSVGCtrl)
                    wx.CallAfter(self.RefreshView)
            else:
                message = wx.MessageDialog(self, "%s is not a valid folder!"%os.path.dirname(filepath), "Error", wx.OK|wx.ICON_ERROR)
                message.ShowModal()
                message.Destroy()
        dialog.Destroy()
        
    def SaveInterface(self):
        result = self.Controler.SaveXMLFile()
        if not result:
            self.SaveInterfaceAs()
        else:
            self.RefreshSVGCtrl()
            self.RefreshView()
    
    def OnGenerateMenu(self, event):
        if self.Controler.HasOpenedInterface():
            self.SaveInterface()
            dialog = wx.SingleChoiceDialog(self, "Select a backend", "Generate Program", ["C"])#, "Haxe"])
            if dialog.ShowModal() == wx.ID_OK:
                svgui_root_element = self.SVGCtrl.GetSVGUIRootElement()
                
                if svgui_root_element is not None:
                    root_bbox = ComputeBBox(svgui_root_element.GetBBox())
                    size = (root_bbox.width, root_bbox.height)
                else :
                    size = (250, 350)
                
                self.Controler.GenerateProgram(dialog.GetStringSelection(), size)
            dialog.Destroy()
        event.Skip()
    
    def OnQuitMenu(self, event):
        self.Close()
        event.Skip()
    
    def OnRefreshMenu(self, event):
        self.RefreshView()
        event.Skip()
    
    def OnUndoMenu(self, event):
        self.Controler.LoadPrevious()
        self.RefreshView()
        event.Skip()
    
    def OnRedoMenu(self, event):
        self.Controler.LoadNext()
        self.RefreshView()
        event.Skip()
    
    def OnAddMenu(self, event):
        selected = self.GetSelected()
        if self.Controler.IsElementContainer(selected):
            dialog = AddControlDialog(self, self.Controler.GetControlTypes())
        elif self.Controler.IsElementNoteBook(selected):
            dialog = AddControlDialog(self, ["NoteBookContent"])
        dialog.SetElementsNames(self.Controler.GetElementsNames())
        if dialog.ShowModal() == wx.ID_OK:
            values = dialog.GetValues()
            new_id = self.Controler.AddElement(selected, values["type"], values["name"])
            if new_id is not None:
                self.RefreshInterfaceTree()
                self.RefreshView(new_id)
        dialog.Destroy()
        event.Skip()
    
    def OnDeleteMenu(self, event):
        selected = self.GetSelected()
        selected_typename = GetElementName(self.Controler.GetElementById(selected))
        if self.Controler.IsElementInterface(selected):
            message = wx.MessageDialog(self, "You can't delete the Interface!", "Error", wx.OK|wx.ICON_ERROR)
        elif self.Controler.IsElementContainer(selected):
            message = wx.MessageDialog(self, "If you delete this %s, his children will be deleted too.\nDo you want to continue?"%selected_typename, "Confirmation", wx.YES_NO|wx.ICON_QUESTION)
        elif self.Controler.IsElementNoteBook(selected):
            message = wx.MessageDialog(self, "If you delete this %s, his contents will be deleted too.\nDo you want to continue?"%selected_typename, "Confirmation", wx.YES_NO|wx.ICON_QUESTION)
        else :
            message = wx.MessageDialog(self, "Are You sure you want to delete this %s?"%selected_typename, "Confirmation", wx.YES_NO|wx.ICON_QUESTION)
        if message.ShowModal() == wx.ID_YES:
            self.Controler.DeleteElement(selected)
            self.RefreshView(self.InterfaceTree.GetPyData(self.InterfaceTree.GetRootItem()))
        message.Destroy()
        event.Skip()
        
    def OnCutMenu(self,event):
        self.Controler.CutElement(self.GetSelected())
        self.RefreshView(self.InterfaceTree.GetPyData(self.InterfaceTree.GetRootItem()))
        event.Skip()
        
    def OnPasteMenu(self,event):
        id = self.Controler.PasteElement(self.GetSelected())
        if id is not None:
            self.RefreshView(id)
        else:
            message = wx.MessageDialog(self, "You can't paste the buffered element into selected element!", "Error", wx.OK|wx.ICON_ERROR)
            message.ShowModal()
            message.Destroy()
        event.Skip()


#-------------------------------------------------------------------------------
#                        InterfaceTree management functions
#-------------------------------------------------------------------------------


    def GetSelected(self):
        item = self.InterfaceTree.GetSelection()
        if item.IsOk():
           return self.InterfaceTree.GetPyData(item)
        return None

    def RefreshInterfaceTree(self):
        infos = self.Controler.GetInterfaceInfos()
        if infos != None:
            root = self.InterfaceTree.GetRootItem()
            self.GenerateTreeBranch(root, infos)
            self.InterfaceTree.Expand(self.InterfaceTree.GetRootItem())
    
    def GenerateTreeBranch(self, root, infos):
        to_delete = []
        if root.IsOk():
            self.InterfaceTree.SetItemText(root, infos["name"])
        else:
            root = self.InterfaceTree.AddRoot(infos["name"])
        self.InterfaceTree.SetPyData(root, infos["data"])
        item, root_cookie = self.InterfaceTree.GetFirstChild(root)
        for child in infos["children"]:
            if not item.IsOk():
                item = self.InterfaceTree.AppendItem(root, "")
                if wx.Platform != '__WXMSW__':
                    item, root_cookie = self.InterfaceTree.GetNextChild(root, root_cookie)
            self.GenerateTreeBranch(item, child)
            item, root_cookie = self.InterfaceTree.GetNextChild(root, root_cookie)
        while item.IsOk():
            to_delete.append(item)
            item, root_cookie = self.InterfaceTree.GetNextChild(root, root_cookie)
        for item in to_delete:
            self.InterfaceTree.Delete(item)
    
    def SelectInterfaceTreeItem(self, root, id):
        if self.InterfaceTree.GetPyData(root) == id:
            self.InterfaceTree.SelectItem(root)
            return True
        else:
            item, root_cookie = self.InterfaceTree.GetFirstChild(root)
            while item.IsOk():
                result = self.SelectInterfaceTreeItem(item, id)
                if result:
                    return True
                item, root_cookie = self.InterfaceTree.GetNextChild(root, root_cookie)
        return False

    def OnInterfaceTreeItemChanging(self, event):
        attrs = self.GetGridRequiredNotFiled()
        if len(attrs) > 0:
            message = wx.MessageDialog(self, "You have to complete theses required cells :\n%s"%"\n".join(attrs), "Error", wx.OK|wx.ICON_ERROR)
            message.ShowModal()
            message.Destroy()
            event.Veto()
        else:
            event.Skip()
            
    def OnInterfaceTreeItemSelected(self, event):
        self.AttributesGrid.SetGridCursor(0, 0)
        self.RefreshAttributesGrid()
        self.RefreshEditMenu()

    def OnInterfaceTreeBeginDrag(self, event):
        selected = self.InterfaceTree.GetPyData(event.GetItem())
        if not self.Controler.IsElementInterface(selected) and len(self.GetGridRequiredNotFiled()) == 0:
            self.DraggingObject = selected
            event.Allow()
        else:
            event.Skip()
        
    def OnInterfaceTreeEndDrag(self, event):
        if self.DraggingObject is not None:
            target = self.InterfaceTree.GetPyData(event.GetItem())
            if self.Controler.IsElementNoteBookContent(self.DraggingObject):
                if not self.Controler.IsElementNotebook(target):
                    message = wx.MessageDialog(self, "You can only drop an notebook content in a notebook!", "Error", wx.OK|wx.ICON_ERROR)
                    message.ShowModal()
                    message.Destroy()
                elif not self.Controler.MoveElement(self.DraggingObject, target):
                    message = wx.MessageDialog(self, "You can't drop a notebook content into one of its children!", "Error", wx.OK|wx.ICON_ERROR)
                    message.ShowModal()
                    message.Destroy()
            elif not self.Controler.IsElementContainer(target):
                message = wx.MessageDialog(self, "You can only drop an element in a container!", "Error", wx.OK|wx.ICON_ERROR)
                message.ShowModal()
                message.Destroy()
            elif not self.Controler.MoveElement(self.DraggingObject, target):
                message = wx.MessageDialog(self, "You can't drop a container into one of its children!", "Error", wx.OK|wx.ICON_ERROR)
                message.ShowModal()
                message.Destroy()
            self.RefreshView(self.DraggingObject)
            self.DraggingObject = None
        event.Skip()


#-------------------------------------------------------------------------------
#                         SVGCtrl management functions
#-------------------------------------------------------------------------------


    def RefreshSVGCtrl(self):
        svgfile = self.Controler.GetSVGFilePath()
        if os.path.isfile(svgfile):
            self.SVGCtrl.LoadFiles(svgfile, self.Controler.GetFilePath())
            self.InitSVGScrollbars()
            return True
        else:
            self.SVGCtrl.Clear()
            self.SVGCtrl.Refresh()
            self.SVGCtrl.SetRoot(None)
            return False
    
    def SVGFileLoaded(self):
        return self.SVGCtrl.GetSVG() is not None
    
    def IsSVGElementIdValue(self, row, col):
        if row >= 0 and col >= 0:
            return self.Table.GetColLabelValue(col) == "Value" and \
                   self.Table.GetValueByName(row, "type") == "string" and \
                   self.Table.GetValueByName(row, "Attribute").endswith("_id")
        else:
            return False
    
    def SelectSVGElementById(self, id):
        if id:
            svgfile = self.SVGCtrl.GetSVG()
            if svgfile is not None:
                self.SelectedSVGElement = svgfile.GetElementById(id)
            else:
                self.SelectedSVGElement = None
        else:
            self.SelectedSVGElement = None
        self.SVGCtrl.Refresh()

    def InitSVGScrollbars(self):
        for scrollbar in self.Controler.GetElementsByType(ITEM_SCROLLBAR):
            svgui_element = self.SVGCtrl.GetScrollBarById(str(scrollbar.getid()))
            if svgui_element is not None:
                svgui_element.Init_ScrollBar(0, 3, 10)

    def OnSVGCtrlClick(self,event):
        row, col = self.AttributesGrid.GetGridCursorRow(), self.AttributesGrid.GetGridCursorCol()
        if self.IsSVGElementIdValue(row, col) and self.SVGFileLoaded():
            scale = self.SVGCtrl.GetScale()
            rect = wx.wxsvg.SVGRect(event.GetX() * scale, event.GetY() * scale, 1, 1)            
            root_element = self.SVGCtrl.GetSVG().GetRootElement()
            nodelist = wx.wxsvg.NodeListCls(root_element.GetIntersectionList(rect, root_element))
            if nodelist.GetSize() > 0:
                options = []
                for i in xrange(nodelist.GetSize()):
                    element = nodelist.GetElement(i)
                    options.append(element.GetAttribute("id"))
                dialog = SVGElementChoiceDialog(self, options)
                if dialog.ShowModal() == wx.ID_OK:
                    value = dialog.GetValue()
                    self.Controler.SetElementAttribute(self.GetSelected(), self.Table.GetValueByName(row, "Attribute"), value)
                    self.RefreshAttributesGrid()
                    self.SelectSVGElementById(value)
                    self.RefreshTitle()
                    self.RefreshFileMenu()
                    self.RefreshEditMenu()
                dialog.Destroy()
        event.Skip()
 
    def OnPaintSVGctrl(self,event):
        wx.CallAfter(self.DrawBoundingBox)
        event.Skip()
 
    def DrawBoundingBox(self):
        if self.SelectedSVGElement is not None:
            svg_file = self.SVGCtrl.GetSVG()
            svg_root_element = svg_file.GetRootElement()
            root_bbox = ComputeBBox(self.SelectedSVGElement.GetElementBBox(svg_root_element))
            element_bbox = ComputeBBox(self.SelectedSVGElement.GetElementBBox(self.SelectedSVGElement))
            
            dc = wx.ClientDC(self.SVGCtrl)
            dc.SetPen(wx.BLACK_PEN)
            dc.SetBrush(wx.BLACK_BRUSH)
            
            if root_bbox.ContainsRect(element_bbox):
                scale = 1 / self.SVGCtrl.GetScale()
                x1, y1 = element_bbox.x * scale, element_bbox.y * scale
                width , height = element_bbox.width * scale, element_bbox.height * scale
                x2, y2 = x1 + width, y1 + height

                dc.DrawRectangle(x1 - HANDLE_SIZE - 2, y1 - HANDLE_SIZE - 2, HANDLE_SIZE, HANDLE_SIZE)
                dc.DrawRectangle(x1 + (width - HANDLE_SIZE) / 2, y1 - HANDLE_SIZE - 2, HANDLE_SIZE, HANDLE_SIZE)
                dc.DrawRectangle(x2 + 2, y1 - HANDLE_SIZE - 2, HANDLE_SIZE, HANDLE_SIZE)
                dc.DrawRectangle(x2 + 2, y1 + (height - HANDLE_SIZE) / 2, HANDLE_SIZE, HANDLE_SIZE)
                dc.DrawRectangle(x2 + 2, y2 + 2, HANDLE_SIZE, HANDLE_SIZE)
                dc.DrawRectangle(x1 + (width - HANDLE_SIZE) / 2, y2 + 2, HANDLE_SIZE, HANDLE_SIZE)
                dc.DrawRectangle(x1 - HANDLE_SIZE - 2, y2 + 2, HANDLE_SIZE, HANDLE_SIZE)
                dc.DrawRectangle(x1 - HANDLE_SIZE - 2, y1 + (height - HANDLE_SIZE) / 2, HANDLE_SIZE, HANDLE_SIZE)
                dc.SetBrush(wx.WHITE_BRUSH)


#-------------------------------------------------------------------------------
#                         AttributesGrid management functions
#-------------------------------------------------------------------------------


    def RefreshAttributesGrid(self):
        selected = self.GetSelected()
        if selected is not None:
            data = []
            attrs = self.Controler.GetElementAttributes(selected)
            for attr in attrs:
                row = {"Attribute" : attr.pop("name")}
                value = attr.pop("value")
                if value == None:
                    value = ""
                if attr["type"] == "boolean":
                    row["Value"] = {True : "Yes", False : "No", "" : ""}[value]
                else:
                    row["Value"] = value
                row.update(attr)
                data.append(row)
            self.Table.SetData(data)
        else:
            self.Table.Empty()
        self.Table.ResetView(self.AttributesGrid)
    
    def GetGridRequiredNotFiled(self):
        attrs = []
        for row in self.Table.GetData():
            if row["use"] == "required" and row["Value"] == "":
                attrs.append(row["Attribute"])
        return attrs
    
    def OnAttributesGridSelectCell(self, event):
        row, col = event.GetRow(), event.GetCol() 
        if self.IsSVGElementIdValue(row, col):
            self.SelectSVGElementById(self.Table.GetValue(row, col))
            if self.SVGFileLoaded():
                self.StatusBar.SetStatusText("Pick up an element in the SVG View to set the value")
            else:
                self.StatusBar.SetStatusText("")
        else:
            self.SelectSVGElementById("")
            self.StatusBar.SetStatusText("")
        event.Skip()
    
    def OnAttributesGridCellChange(self, event):
        row, col = event.GetRow(), event.GetCol()
        value = self.Table.GetValue(row, col)
        attr = self.Table.GetValueByName(row, "Attribute")
        use = self.Table.GetValueByName(row, "use")
        type = self.Table.GetValueByName(row, "type")
        
        if attr == "name" and self.Controler.IsExistingName(value):
            message = wx.MessageDialog(self, "This name is already used : %s!"%value, "Error", wx.OK|wx.ICON_ERROR)
            message.ShowModal()
            message.Destroy()
        elif attr == "id" and self.Controler.IsExistingId(value):
            message = wx.MessageDialog(self, "This id is already used : %s!"%value, "Error", wx.OK|wx.ICON_ERROR)
            message.ShowModal()
            message.Destroy()
        elif use == "required" and value == "":
            message = wx.MessageDialog(self, "This value is required : %s!"%attr, "Error", wx.OK|wx.ICON_ERROR)
            message.ShowModal()
            message.Destroy()
        elif type == "boolean":
            self.Controler.SetElementAttribute(self.GetSelected(), attr, {"Yes" : True, "No" : False, "" : None}[value])
        else:
            if value == "":
                value = None
            elif type == "boolean":
                value = {"Yes" : True, "No" : False}[value]
            self.Controler.SetElementAttribute(self.GetSelected(), attr, value)
            self.SelectSVGElementById(value)
            if attr in ["name", "id"]:
                self.RefreshInterfaceTree()
        wx.CallAfter(self.RefreshAttributesGrid)
        self.RefreshTitle()
        self.RefreshFileMenu()
        self.RefreshEditMenu() 
        event.Skip()


#-------------------------------------------------------------------------------
#                          SVG Element Choice Dialog
#-------------------------------------------------------------------------------


[ID_SVGELEMENTCHOICEDIALOG, ID_SVGELEMENTCHOICEDIALOGELEMENTS, 
 ID_SVGELEMENTCHOICEDIALOGSTATICTEXT1, 
] = [wx.NewId() for _init_ctrls in range(3)]

class SVGElementChoiceDialog(wx.Dialog):
    
    def _init_coll_MainSizer_Items(self, parent):
        # generated method, don't edit
        parent.AddSizer(self.ControlSizer, 0, border=20, flag=wx.GROW|wx.TOP|wx.LEFT|wx.RIGHT)
        parent.AddSizer(self.ButtonSizer, 0, border=20, flag=wx.GROW|wx.BOTTOM|wx.LEFT|wx.RIGHT)

    def _init_coll_MainSizer_Growables(self, parent):
        parent.AddGrowableCol(0)
        parent.AddGrowableRow(0)

    def _init_coll_ControlSizer_Items(self, parent):
        # generated method, don't edit
        parent.AddWindow(self.staticText1, 1, border=10, flag=wx.GROW|wx.RIGHT)
        parent.AddWindow(self.Elements, 1, border=0, flag=wx.GROW)
    
    def _init_coll_ControlSizer_Growables(self, parent):
        parent.AddGrowableCol(1)
        
    def _init_sizers(self):
        # generated method, don't edit
        self.MainSizer = wx.FlexGridSizer(cols=1, hgap=0, rows=2, vgap=10)
        self.ControlSizer = wx.FlexGridSizer(cols=2, hgap=5, rows=1, vgap=0)
        
        self._init_coll_MainSizer_Items(self.MainSizer)
        self._init_coll_MainSizer_Growables(self.MainSizer)
        self._init_coll_ControlSizer_Items(self.ControlSizer)
        self._init_coll_ControlSizer_Growables(self.ControlSizer)
        
        self.SetSizer(self.MainSizer)

    def _init_ctrls(self, prnt):
        # generated method, don't edit
        wx.Dialog.__init__(self, id=ID_SVGELEMENTCHOICEDIALOG,
              name='ProjectDialog', parent=prnt, pos=wx.Point(376, 223),
              size=wx.Size(300, 100), style=wx.DEFAULT_DIALOG_STYLE,
              title='Choose a SVG Element')
        self.SetClientSize(wx.Size(300, 100))
        
        self.staticText1 = wx.StaticText(id=ID_SVGELEMENTCHOICEDIALOGSTATICTEXT1,
              label='Elements:', name='staticText1', parent=self,
              pos=wx.Point(0, 0), size=wx.Size(80, 17), style=0)

        self.Elements = wx.Choice(id=ID_SVGELEMENTCHOICEDIALOGELEMENTS,
              name='Elements', parent=self, pos=wx.Point(0, 0),
              size=wx.Size(0, 24), style=0)
        
        self.ButtonSizer = self.CreateButtonSizer(wx.OK|wx.CANCEL|wx.CENTRE)
        self.Bind(wx.EVT_BUTTON, self.OnOK, id=self.ButtonSizer.GetAffirmativeButton().GetId())
        
        self._init_sizers()

    def __init__(self, parent, svgelements):
        self._init_ctrls(parent)
        
        for option in svgelements:
            self.Elements.Append(option)

    def OnOK(self,event):
        element = self.Elements.GetStringSelection()
        error = False
        if element == "":
            message = wx.MessageDialog(self, "You must choose an element !", "Error", wx.OK|wx.ICON_ERROR)
            message.ShowModal()
            message.Destroy()
            error = True
        if not error:
            self.EndModal(wx.ID_OK)
            
    def GetValue(self):
        return self.Elements.GetStringSelection()


#-------------------------------------------------------------------------------
#                          Add Control Element Dialog
#-------------------------------------------------------------------------------


[ID_ADDCONTROLDIALOG, ID_ADDCONTROLDIALOGCONTROLTYPE, 
 ID_ADDCONTROLDIALOGCONTROLNAME, ID_ADDCONTROLDIALOGSTATICTEXT1, 
 ID_ADDCONTROLDIALOGSTATICTEXT2, 
] = [wx.NewId() for _init_ctrls in range(5)]

class AddControlDialog(wx.Dialog):
    
    def _init_coll_MainSizer_Items(self, parent):
        # generated method, don't edit
        parent.AddSizer(self.ControlSizer, 0, border=20, flag=wx.GROW|wx.TOP|wx.LEFT|wx.RIGHT)
        parent.AddSizer(self.ButtonSizer, 0, border=20, flag=wx.GROW|wx.BOTTOM|wx.LEFT|wx.RIGHT)

    def _init_coll_MainSizer_Growables(self, parent):
        parent.AddGrowableCol(0)
        parent.AddGrowableRow(0)

    def _init_coll_ControlSizer_Items(self, parent):
        # generated method, don't edit
        parent.AddWindow(self.staticText1, 1, border=10, flag=wx.GROW|wx.RIGHT)
        parent.AddWindow(self.ControlType, 1, border=0, flag=wx.GROW)
        parent.AddWindow(self.staticText2, 1, border=10, flag=wx.GROW|wx.RIGHT)
        parent.AddWindow(self.ControlName, 1, border=0, flag=wx.GROW)
    
    def _init_coll_ControlSizer_Growables(self, parent):
        parent.AddGrowableCol(1)
        
    def _init_sizers(self):
        # generated method, don't edit
        self.MainSizer = wx.FlexGridSizer(cols=1, hgap=0, rows=2, vgap=10)
        self.ControlSizer = wx.FlexGridSizer(cols=2, hgap=5, rows=2, vgap=10)
        
        self._init_coll_MainSizer_Items(self.MainSizer)
        self._init_coll_MainSizer_Growables(self.MainSizer)
        self._init_coll_ControlSizer_Items(self.ControlSizer)
        self._init_coll_ControlSizer_Growables(self.ControlSizer)
        
        self.SetSizer(self.MainSizer)

    def _init_ctrls(self, prnt):
        # generated method, don't edit
        wx.Dialog.__init__(self, id=ID_ADDCONTROLDIALOG,
              name='AddControlDialog', parent=prnt, pos=wx.Point(376, 223),
              size=wx.Size(300, 150), style=wx.DEFAULT_DIALOG_STYLE,
              title='Add a new control')
        self.SetClientSize(wx.Size(300, 150))

        self.staticText1 = wx.StaticText(id=ID_ADDCONTROLDIALOGSTATICTEXT1,
              label='Control type:', name='staticText1', parent=self,
              pos=wx.Point(0, 0), size=wx.Size(80, 17), style=0)

        self.ControlType = wx.Choice(id=ID_ADDCONTROLDIALOGCONTROLTYPE,
              name='Control', parent=self, pos=wx.Point(0, 0),
              size=wx.Size(0, 24), style=0)
        
        self.staticText2 = wx.StaticText(id=ID_ADDCONTROLDIALOGSTATICTEXT2,
              label='Name:', name='staticText2', parent=self,
              pos=wx.Point(24, 68), size=wx.Size(95, 17), style=0)

        self.ControlName = wx.TextCtrl(id=ID_ADDCONTROLDIALOGCONTROLNAME,
              name='Name', parent=self, pos=wx.Point(0, 0), 
              size=wx.Size(0, 24), style=0)

        self.ButtonSizer = self.CreateButtonSizer(wx.OK|wx.CANCEL|wx.CENTRE)
        self.Bind(wx.EVT_BUTTON, self.OnOK, id=self.ButtonSizer.GetAffirmativeButton().GetId())
        
        self._init_sizers()

    def __init__(self, parent, element_types):
        self._init_ctrls(parent)
        
        for option in element_types:
            self.ControlType.Append(option)

    def OnOK(self, event):
        control_type = self.ControlType.GetStringSelection()
        control_name = self.ControlName.GetValue()
        error = False
        if control_type == "":
            message = wx.MessageDialog(self, "You must choose a control type!", "Error", wx.OK|wx.ICON_ERROR)
            message.ShowModal()
            message.Destroy()
            error = True
        elif control_name == "":
            message = wx.MessageDialog(self, "You must enter a name for this new control!", "Error", wx.OK|wx.ICON_ERROR)
            message.ShowModal()
            message.Destroy()
            error = True
        elif control_name.upper() in self.ElementsNames:
            message = wx.MessageDialog(self, "This name is already used!", "Error", wx.OK|wx.ICON_ERROR)
            message.ShowModal()
            message.Destroy()
            error = True
        if not error:
            self.EndModal(wx.ID_OK)
        
    def GetValues(self):
        return {"type" : self.ControlType.GetStringSelection(), "name" : self.ControlName.GetValue()}
    
    def SetElementsNames(self, elements_names):
        self.ElementsNames = [name.upper() for name in elements_names]


#-------------------------------------------------------------------------------
#                               Exception Handler
#-------------------------------------------------------------------------------


Max_Traceback_List_Size = 20

def Display_Exception_Dialog(e_type,e_value,e_tb):
    trcbck_lst = []
    for i,line in enumerate(traceback.extract_tb(e_tb)):
        trcbck = " " + str(i+1) + ". "
        if line[0].find(os.getcwd()) == -1:
            trcbck += "file : " + str(line[0]) + ",   "
        else:
            trcbck += "file : " + str(line[0][len(os.getcwd()):]) + ",   "
        trcbck += "line : " + str(line[1]) + ",   " + "function : " + str(line[2])
        trcbck_lst.append(trcbck)
        
    # Allow clicking....
    cap = wx.Window_GetCapture()
    if cap:
        cap.ReleaseMouse()

    dlg = wx.SingleChoiceDialog(None, 
        """
An error happens.

Click on OK for saving an error report.

Please contact LOLITech at:
+33 (0)3 29 52 95 67
bugs_DefEditor@lolitech.fr


Error:
""" +
        str(e_type) + " : " + str(e_value), 
        "Error",
        trcbck_lst)
    try:
        res = (dlg.ShowModal() == wx.ID_OK)
    finally:
        dlg.Destroy()

    return res

def Display_Error_Dialog(e_value):
    message = wx.MessageDialog(None, str(e_value), "Error", wx.OK|wx.ICON_ERROR)
    message.ShowModal()
    message.Destroy()

def get_last_traceback(tb):
    while tb.tb_next:
        tb = tb.tb_next
    return tb


def format_namespace(d, indent='    '):
    return '\n'.join(['%s%s: %s' % (indent, k, repr(v)[:10000]) for k, v in d.iteritems()])


ignored_exceptions = [] # a problem with a line in a module is only reported once per session

def wxAddExceptHook(path, app_version='[No version]'):#, ignored_exceptions=[]):
    
    def handle_exception(e_type, e_value, e_traceback):
        traceback.print_exception(e_type, e_value, e_traceback) # this is very helpful when there's an exception in the rest of this func
        last_tb = get_last_traceback(e_traceback)
        ex = (last_tb.tb_frame.f_code.co_filename, last_tb.tb_frame.f_lineno)
        if str(e_value).startswith("!!!"):
            Display_Error_Dialog(e_value)
        elif ex not in ignored_exceptions:
            result = Display_Exception_Dialog(e_type,e_value,e_traceback)
            if result:
                ignored_exceptions.append(ex)
                info = {
                    'app-title' : wx.GetApp().GetAppName(), # app_title
                    'app-version' : app_version,
                    'wx-version' : wx.VERSION_STRING,
                    'wx-platform' : wx.Platform,
                    'python-version' : platform.python_version(), #sys.version.split()[0],
                    'platform' : platform.platform(),
                    'e-type' : e_type,
                    'e-value' : e_value,
                    'date' : time.ctime(),
                    'cwd' : os.getcwd(),
                    }
                if e_traceback:
                    info['traceback'] = ''.join(traceback.format_tb(e_traceback)) + '%s: %s' % (e_type, e_value)
                    last_tb = get_last_traceback(e_traceback)
                    exception_locals = last_tb.tb_frame.f_locals # the locals at the level of the stack trace where the exception actually occurred
                    info['locals'] = format_namespace(exception_locals)
                    if 'self' in exception_locals:
                        info['self'] = format_namespace(exception_locals['self'].__dict__)
                
                output = open(path+os.sep+"bug_report_"+info['date'].replace(':','-').replace(' ','_')+".txt",'w')
                lst = info.keys()
                lst.sort()
                for a in lst:
                    output.write(a+":\n"+str(info[a])+"\n\n")

    #sys.excepthook = lambda *args: wx.CallAfter(handle_exception, *args)
    sys.excepthook = handle_exception
 
if __name__ == '__main__':
    def usage():
        print "\nUsage of SVGUIEditor.py :"
        print "\n   %s [Filepath]\n"%sys.argv[0]

    try:
        opts, args = getopt.getopt(sys.argv[1:], "h", ["help"])
    except getopt.GetoptError:
        # print help information and exit:
        usage()
        sys.exit(2)

    for o, a in opts:
        if o in ("-h", "--help"):
            usage()
            sys.exit()

    fileOpen = None
    if len(args) > 1:
        usage()
        sys.exit()
    elif len(args) == 1:
        fileOpen = args[0]
    
    app = wx.PySimpleApp()
    wx.InitAllImageHandlers()
    
    # Install a exception handle for bug reports
    wxAddExceptHook(os.getcwd(),__version__)
    
    frame = SVGUIEditor(None, fileOpen=fileOpen)

    frame.Show()
    app.MainLoop()

