/*
 *  xdemorse: An application to decode Morse signals to text
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 3 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifndef COMMON_H
#define COMMON_H	1

#include <gtk/gtk.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

/* Definitions of flags for various actions */
#define ADAPT_SPEED		 0x000001 /* Enable speed tracking */
#define MAN_THRESHOLD	 0x000002 /* Enable manual detector threshold */
#define WORD_WRAP		 0x000004 /* Enable word wrapping */
#define DISPLAY_SIGNAL	 0x000008 /* Display output of signal detector */
#define DISPLAY_RATIO	 0x000010 /* Display situation of Ratio threshold */
#define DISPLAY_LEVEL	 0x000020 /* Display situation of Level threshold */
#define SELECT_LEVEL	 0x000040 /* Select Level or Ratio situation display */
#define ENABLE_CAT		 0x000080 /* Enable CAT for transceiver */
#define CAT_SETUP		 0x000100 /* CAT is set up */
#define ENABLE_RECEIVE	 0x000200 /* Enable reception of Morse code */
#define RCCONFIG_OK		 0x000400 /* xdemorserc file read ok */
#define MARK_TONE		 0x000800 /* Input signal is mark tone (key down) */
#define SPACE_TONE		 0x001000 /* Input signal is space tone (key up)  */
#define CENTERLINE		 0x002000 /* Enable center line in waterfall */

/* Return values */
#define ERROR			1	 /* Error condition */
#define SUCCESS			0	 /* No error condition */

/*-------------------------------------------------------------------*/

/* Runtine configuration data */
typedef struct
{
  char
	snd_card[32], /* Sound card name */
	cap_src[32],  /* Capture source  */
	cap_vol[32];  /* Capture volume  */

  int
	channel,	/* ALSA "channel" for use in mixer setup */
	use_chnl,	/* Channel in use (frontleft=0, frontright=1 etc) */
	cap_level,	/* Recording/Capture level */
	num_chnl,	/* Number of audio channels (2=stereo, 1=mono) */
	dsp_rate;	/* DSP rate (speed) samples/sec */

  /* Transceiver serial device */
  char cat_serial[32];

  /* Transceiver type 1=FT847 2=FT857 3=K2 4=K3 0=NONE */
  int tcvr_type;

  int
	tone_freq,	 /* Receiver BFO Tone freq */
	center_line, /* Waterfall BFO Tone freq marker line */
	fft_stride,  /* Stride of fft over input samples   */
	speed_wpm,		/* Current Morse speed words/min   */
	unit_elem,		/* Morse unit element (dot) length */
	min_unit,		/* Minimum length of unit element  */
	max_unit,		/* Maximum length of unit element  */
	max_unit_x2,	/* Maximum length of unit element * 2 */
	det_squelch,	/* Squelch level of signal detector */
	det_threshold;	/* Current threshold value of mark/space detector */

  double
	det_ratio;		/* Ratio of lead/trail edge for mark/space detection */

} rc_data_t;

typedef struct
{
  /* Signal/dsp samples buffer */
  short *buffer;

  int
	buffer_idx,	 /* Index to signal samples buffer */
	buffer_size; /* Buffer size according to stereo/mono mode */

} samples_buffer_t;

/*-------------------------------------------------------------------*/

/* C99 problem */
#ifndef M_PI
#define M_PI           3.14159265358979323846
#endif

/*-------------------------------------------------------------------*/

/* Function prototypes produced by cproto */

/* callbacks.c */
void Error_Dialog(char *message);
void on_main_window_destroy(GtkObject *object, gpointer user_data);
gboolean on_main_window_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
gboolean on_scope_expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer user_data);
gboolean on_scope_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
void on_speed_changed(GtkEditable *editable, gpointer user_data);
void on_squelch_changed(GtkEditable *editable, gpointer user_data);
void on_err_quit_clicked(GtkButton *button, gpointer user_data);
void on_clear_clicked(GtkButton *button, gpointer user_data);
gboolean on_waterfall_expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer user_data);
gboolean on_waterfall_button_press_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
void on_err_ok_button_clicked(GtkButton *button, gpointer user_data);
void on_wpm_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_error_dialog_destroy(GtkObject *object, gpointer user_data);
gboolean on_error_dialog_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data);
void on_waterfall_size_allocate(GtkWidget *widget, GdkRectangle *allocation, gpointer user_data);
void on_rx_togglebutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
void on_scope_size_allocate(GtkWidget *widget, GdkRectangle *allocation, gpointer user_data);
void on_centerline_checkbutton_toggled(GtkToggleButton *togglebutton, gpointer user_data);
/* cat.c */
gboolean Open_Tcvr_Serial(void);
void Close_Tcvr_Serial(void);
gboolean Read_Tcvr_Serial(char *data, size_t len);
gboolean Write_Tcvr_Serial(char *data, size_t len);
gboolean Read_Rx_Freq(int *freq);
gboolean Write_Rx_Freq(int freq);
gboolean Tune_Tcvr(double x);
/* decode.c */
gboolean Get_Character(char *chr);
void Adapt_Decoder(void);
char Hex_to_Ascii(int *hex_code);
/* detect.c */
gboolean Get_Fragment(void);
gboolean Initialize_Detector(void);
/* display.c */
void Display_Detector(int plot);
void Display_Signal(int plot);
void Display_Waterfall(void);
int Carrier_Amplitude(short x, short y);
/* fft.c */
void Ifft_Init(int fft_input_size, int fft_bin_size);
void Ifft(int fft_input_size, int fft_bin_size);
/* interface.c */
GtkWidget *create_main_window(void);
GtkWidget *create_error_dialog(void);
/* main.c */
int main(int argc, char *argv[]);
gboolean Load_Config(gpointer data);
int Load_Line(char *buff, FILE *pfile, char *messg);
gboolean Print_Character(gpointer data);
void Usage(void);
/* shared.c */
/* sound.c */
gboolean Setup_Sound_Card(char *mesg, int *error);
void Close_Capture_Handle(void);
void Close_Mixer_Handle(void);
gboolean Signal_Sample(short *sample);
gboolean Xrun_Recovery(int error);
/* support.c */
GtkWidget *lookup_widget(GtkWidget *widget, const gchar *widget_name);
void add_pixmap_directory(const gchar *directory);
GtkWidget *create_pixmap(GtkWidget *widget, const gchar *filename);
GdkPixbuf *create_pixbuf(const gchar *filename);
void glade_set_atk_action_description(AtkAction *action, const gchar *action_name, const gchar *description);
/* utils.c */
int isFlagSet(int flag);
int isFlagClear(int flag);
void Set_Flag(int flag);
void Clear_Flag(int flag);
void Toggle_Flag(int flag);
void Cleanup(void);
gboolean mem_alloc(void **ptr, size_t req);
gboolean mem_realloc(void **ptr, size_t req);
void free_ptr(void **ptr);
void Strlcpy(char *dest, const char *src, size_t n);
void Strlcat(char *dest, const char *src, size_t n);
#endif

