/*
  3Com ADSL Modem USB ioctl call
  Copyright (C) 2002 Josep Comas

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

  Author     : Josep Comas <jcomas@gna.es>
  Creation   : 25/8/2002

  Description: This program does ioctl calls to 3Com driver.

  Log:

  5/6/2003 Josep Comas
  Updated to work with libusb-0.1.7

  7/6/2003 Josep Comas
  Added release version for kernel module

*/

#include <stdio.h>
#include <string.h>
#include <usb.h>
#include "usbi.h"
#include <libintl.h>
#include <locale.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

/* modem usb identifications */
#define ID_VENDOR    0x0506  /* Vendor = 3Com */
#define ID_PRODUCT1  0xF002  /* Product = ADSL modem without EZ-USB firmware loaded */
#define ID_PRODUCT1b 0xF100  /* Product = ADSL modem without EZ-USB firmware loaded */
#define ID_PRODUCT2  0xF003  /* Product = ADSL modem with EZ-USB firmware loaded */

/* command received */
#define ADSL_Operational_Mode_Ack  0x12
#define ADSL_Open_Rejected         0x21
#define ADSL_Connection_Opened     0x24
#define ADSL_Connection_Closed     0x25
#define ADSL_Connection_Lost       0x41
#define ADSL_In_Training           0x5f

/* modem status */
#define MODEM_DOWN  0x00
#define MODEM_WAIT  0x08
#define MODEM_INIT  0x10
#define MODEM_UP    0x20

/* open mode */
#define ADSL_Open            0x03
#define ADSL_Open_ANSI       0x05
#define ADSL_Open_GDMT       0x06
#define ADSL_Open_GLITE      0x07
#define ADSL_Open_MULTIMODE  0x08

/* valid ADSL operational mode */
#define ADSL_OPERATIONAL_MODE_ANSI     0x01
#define ADSL_OPERATIONAL_MODE_GDMT     0x02
#define ADSL_OPERATIONAL_MODE_GLITE    0x08

/* data path */
#define FastPath         1
#define InterleavedPath  2

/* translation files */
#define TF_CODE "3cioctl"

/* struct to call to 3Com kernel driver (module) */
struct usbdevfs_ioctl {
  int ifno;
  int ioctl_code;
  void *data;
};

/* USBDEVFS IOCTL to call to 3Com kernel driver (module) */
#define USBDEVFS_IOCTL _IOWR('U', 18, struct usbdevfs_ioctl)

/* struct to fill with info provided by 3Com kernel driver (module) */
struct udsl_modem_info {
  unsigned char lastcmd;
  unsigned char laststatus;
  unsigned char modminor;
  unsigned char modmajor;
  unsigned char modrelease;
  unsigned char modspeed;
  unsigned char firmminor;
  unsigned char firmmajor;
  unsigned char hardversion;
  unsigned char harddata[4];
  unsigned char mac[6];
  unsigned char serial[12];
  char alcfirmminor;
  char alcfirmmajor;
  short vpi;
  int vci;
  unsigned int atmcelltx;
  unsigned int atmcellrx;
  unsigned int oamloopback;
  unsigned int downbitrate;
  unsigned int upbitrate;
  unsigned char openmode;
  unsigned char operationalmode;
  unsigned char linestatus;
  unsigned char datapath;
  unsigned int linklost;
  unsigned char extendedinfo;
  unsigned short downcorrectederrors;
  unsigned short downuncorrectederrors;
  unsigned short downhecerrors;
  unsigned short upcorrectederrors;
  unsigned short upuncorrectederrors;
  unsigned short uphecerrors;
  unsigned char downattenuation;
  unsigned char upattenuation;
  signed char downsignalnoiseratio;
  signed char upsignalnoiseratio;
  signed char downoutputpower;
  signed char upoutputpower;
} udsl_modem_info_t;

/* IOCTL codes when call to 3Com kernel driver (module) */
#define IOCTL_START                1
#define IOCTL_STOP                 2
#define IOCTL_INFO _IOWR('U', 3, struct udsl_modem_info)
#define IOCTL_RECONNECT            4
#define IOCTL_UPDATE_OPS_MODE      5
#define IOCTL_START_EXTENDED_INFO  6
#define IOCTL_STOP_EXTENDED_INFO   7
#define IOCTL_CLEAR_COUNTERS       8

/* check if a file exists */
int file_exists(const char *filename)
{
  struct stat info_file;

  return stat(filename, &info_file) == 0;
}

/* show printable char */
void print_char(unsigned char c)
{
  if (c >= ' ' && c < 0x7f)
    printf("%c", c);
  else
    printf(".");
}

int main(int argc, char *argv[])
{
  /* struct to call to 3Com kernel driver (module) */
  struct usbdevfs_ioctl u_ioctl;
  /* struct where put info provided by 3Com kernel driver (module) */
  struct udsl_modem_info modem_info;

  /* bus structures variables */
  struct usb_bus * bus;
  struct usb_device * dev;
  struct usb_device * adsl_dev = NULL;
  usb_dev_handle * adsl_handle;

  /* boolean value */
  int goon;

  /* result code */
  int r;

  /* counter */
  int i;

  /* init locale */
  setlocale(LC_ALL, "");
  //if (file_exists("./locale"))
  //  bindtextdomain(TF_CODE, "./locale");  /* set directory for a domain (source code messages) */
  //else
    bindtextdomain(TF_CODE, "/usr/share/locale");  /* set directory for a domain (source code messages) */
  textdomain(TF_CODE);  /* set domain */

  /* show program information */
  printf(gettext("3Com ADSL Modem USB ioctl call\n"));
  printf("Josep Comas <jcomas@gna.es>. 7/6/2003\n\n");

  /* check parameters */
  if (argc < 2)
  {
    printf(gettext("Usage: %s ioctl_command\n"), argv[0]);
    printf(gettext("\nIOCTL commands:\n"));
    printf(gettext("  1 (start, enable receive data in driver)\n"));
    printf(gettext("  2 (stop, disable receive data in driver)\n"));
    printf(gettext("  3 (info about modem)\n"));
    printf(gettext("  4 (reconnect ADSL line when using max speed mode in driver)\n"));
    printf(gettext("  5 (update ops mode when using max speed mode in driver)\n"));
    printf(gettext("  6 (start getting additional info when using normal speed mode in driver)\n"));
    printf(gettext("  7 (stop getting additional info when using normal speed mode in driver)\n"));
    printf(gettext("  8 (clear counters)\n"));
    printf(gettext("\nExamples:\n"));
    printf(gettext("   Enable receive data in driver: %s 1\n"), argv[0]);
    printf(gettext("   Disable receive data in driver: %s 2\n"), argv[0]);
    printf(gettext("   Show info about modem provided by driver: %s 3\n"), argv[0]);
    return -1;
  }
  if (strcmp(argv[1], "1")  && strcmp(argv[1], "2") && strcmp(argv[1], "3") && strcmp(argv[1], "4") &&
      strcmp(argv[1], "5")  && strcmp(argv[1], "6") && strcmp(argv[1], "7") && strcmp(argv[1], "8"))
  {
    printf(gettext("Error: invalid command: %s\n"), argv[1]);
    return -1;
  }

  /* init USB bus and find devices */
  usb_init();
  if (usb_find_busses() < 0)
  {
    printf(gettext("Error: I can't find busses\n"));
    return -1;
  }
  if (usb_find_devices() < 0)
  {
    printf(gettext("Error: I can't find devices\n"));
    return -1;
  }

  /* search first 3Com ADSL modem */
  bus = usb_busses;
  goon = 1;
  while (bus && goon)
  {
    dev = bus->devices;
    while (dev && goon)
    {
      if (dev->descriptor.idVendor == ID_VENDOR)
      {
        if (dev->descriptor.idProduct == ID_PRODUCT1 || dev->descriptor.idProduct == ID_PRODUCT1b)
        {
          printf(gettext("Error: you need load EZ-USB firmware\n"));
          return -1;
        }
        if (dev->descriptor.idProduct == ID_PRODUCT2)
        {
          goon = 0;
          adsl_dev = dev;
        }
      }
      if (goon)
        dev = dev->next;
    }
    bus = bus->next;
  }
  if (adsl_dev == NULL)
  {
    printf(gettext("Error: I didn't find ADSL modem\n"));
    return -1;
  }
  printf(gettext("I found ADSL modem with VendorID = %04x & ProductID = %04x\n"),
         adsl_dev->descriptor.idVendor, adsl_dev->descriptor.idProduct);

  /* connect to ADSL modem */
  adsl_handle = usb_open(adsl_dev);
  if (adsl_handle == NULL)
  {
    printf(gettext("Error: Couldn't get device handle for ADSL modem\n"));
    return -1;
  }
  if (usb_set_configuration(adsl_handle, 1) < 0)
  {
    printf("Error: usb_set_configuration: %s\n", usb_strerror());
    return -1;
  }

  u_ioctl.ifno = 0;
  if (argv[1][0] == '1') { 

    u_ioctl.ioctl_code = IOCTL_START;
    u_ioctl.data  = NULL;

  } else if (argv[1][0] == '2') { 

    u_ioctl.ioctl_code = IOCTL_STOP;
    u_ioctl.data  = NULL;

  } else if (argv[1][0] == '3') { 

    modem_info.lastcmd = 0xff;
    modem_info.laststatus = 0xff;
    u_ioctl.ioctl_code = IOCTL_INFO;
    u_ioctl.data = &modem_info;

  } else if (argv[1][0] == '4') { 

    u_ioctl.ioctl_code = IOCTL_RECONNECT;
    u_ioctl.data  = NULL;

  } else if (argv[1][0] == '5') {

    u_ioctl.ioctl_code = IOCTL_UPDATE_OPS_MODE;
    u_ioctl.data  = NULL;

  } else if (argv[1][0] == '6') { 

    u_ioctl.ioctl_code = IOCTL_START_EXTENDED_INFO;
    u_ioctl.data  = NULL;

  } else if (argv[1][0] == '7') { 

    u_ioctl.ioctl_code = IOCTL_STOP_EXTENDED_INFO;
    u_ioctl.data  = NULL;

  } else if (argv[1][0] == '8') { 

    u_ioctl.ioctl_code = IOCTL_CLEAR_COUNTERS;
    u_ioctl.data  = NULL;
  }

  r = ioctl(adsl_handle->fd, USBDEVFS_IOCTL, &u_ioctl);
  if (r < 0) {
    printf(gettext("Error in ioctl call, status = %d\n"), r);
    usb_close(adsl_handle);
    return -1;
  }
  else if (argv[1][0] == '3') {
    printf(gettext("Last command = 0x%02x ("), modem_info.lastcmd);
    switch (modem_info.lastcmd) {
      case ADSL_Operational_Mode_Ack: printf("ack"); break;
      case ADSL_Open_Rejected: printf(gettext("open rejected")); break;
      case ADSL_Connection_Opened: printf(gettext("connection opened")); break;
      case ADSL_Connection_Closed: printf(gettext("connection closed")); break;
      case ADSL_Connection_Lost: printf(gettext("connection lost")); break;
      default: printf(gettext("unknown")); break;
    };
    printf(")\n");
    printf(gettext("Modem status = 0x%02x ("), modem_info.laststatus);
    switch (modem_info.laststatus) {
      case MODEM_DOWN: printf(gettext("down")); break;
      case MODEM_WAIT: printf(gettext("wait")); break;
      case MODEM_INIT: printf(gettext("init")); break;
      case MODEM_UP: printf(gettext("up")); break;
      default: printf(gettext("unknown")); break;
    };
    printf(")\n");
    printf(gettext("Open mode = "));
    switch (modem_info.openmode) {
      case ADSL_Open_ANSI: printf("ANSI T1.413"); break;
      case ADSL_Open_GDMT: printf("ITU-T G.992.1 (G.DMT)"); break;
      case ADSL_Open_GLITE: printf("ITU-T G.992.2 (G.LITE)"); break;
      case ADSL_Open_MULTIMODE: printf(gettext("multimode")); break;
      case ADSL_Open: printf(gettext("other")); break;
      case 0xff: printf(gettext("not available")); break;
      default: printf(gettext("unknown")); break;
    }
    printf("\n");
    printf(gettext("Operational mode = "));
    switch (modem_info.operationalmode) {
      case ADSL_OPERATIONAL_MODE_ANSI: printf("ANSI T1.413"); break;
      case ADSL_OPERATIONAL_MODE_GDMT: printf("ITU-T G.992.1 (G.DMT)"); break;
      case ADSL_OPERATIONAL_MODE_GLITE: printf("ITU-T G.992.2 (G.LITE)"); break;
      case 0xff: printf("not available"); break;
      default: printf("unknown"); break;
    }
    printf("\n");
    printf(gettext("Line status = 0x%02x ("), modem_info.linestatus);
    switch (modem_info.linestatus) {
      case ADSL_Connection_Closed: printf(gettext("connection closed")); break;
      case ADSL_Connection_Opened: printf(gettext("connection opened")); break;
      case ADSL_Connection_Lost: printf(gettext("connection lost")); break;
      case ADSL_In_Training: printf(gettext("in training")); break;
      default: printf(gettext("unknown")); break;
    };
    printf(")\n");
    printf(gettext("Downstream bitrate (bits/s) = %u\n"), modem_info.downbitrate);
    printf(gettext("Upstream bitrate (bits/s) = %u\n"), modem_info.upbitrate);
    printf(gettext("Data path = 0x%02x ("), modem_info.datapath);
    switch (modem_info.datapath) {
      case FastPath: printf("fastpath"); break;
      case InterleavedPath: printf("interleaved"); break;
      default: printf(gettext("unknown")); break;
    };
    printf(")\n");


    printf(gettext("Module version (driver) = %d.%d.%d (%s speed)\n"), modem_info.modmajor, modem_info.modminor, modem_info.modrelease, modem_info.modspeed==1?gettext("max"):gettext("normal"));
    printf(gettext("Hardware version = %d:%02x%02x%02x%02x:%d.%d\n"), modem_info.hardversion, modem_info.harddata[0], modem_info.harddata[1], modem_info.harddata[2], modem_info.harddata[3], modem_info.firmmajor, modem_info.firmminor);
    printf(gettext("Alcatel firmware version = %d.%d.%d\n"), (int) modem_info.alcfirmmajor >> 4, (int) modem_info.alcfirmmajor &0xf, (int) modem_info.alcfirmminor);
    printf(gettext("MAC address = "));
    for (i = 0; i < 6; i++) {
      printf("%02x", modem_info.mac[i]);
      if (i != 5)
        printf(".");
    }
    printf("\n");
    printf(gettext("Serial number = "));
    for (i = 0; i < 12; i++)
      print_char(modem_info.serial[i]);
    printf("\n");
    printf("VPI = %d\n", modem_info.vpi);
    printf("VCI = %d\n", modem_info.vci);
    printf(gettext("ATM cells transmited = %u\n"), modem_info.atmcelltx);
    printf(gettext("ATM cells received = %u\n"), modem_info.atmcellrx);
    printf(gettext("OAM loopback cells = %u\n"), modem_info.oamloopback);
    printf(gettext("Link lost = %u\n"), modem_info.linklost);
    if (modem_info.extendedinfo != 1)
      printf(gettext("Extended info not available\n"));
    else {
      printf(gettext("Extended info:\n"));
      printf(gettext("  down corrected errors = %d\n"), modem_info.downcorrectederrors);
      printf(gettext("  up corrected errors = %d\n"), modem_info.upcorrectederrors);
      printf(gettext("  down uncorrected errors = %d\n"), modem_info.downuncorrectederrors);
      printf(gettext("  up uncorrected errors = %d\n"), modem_info.upuncorrectederrors);
      printf(gettext("  down hec errors = %d\n"), modem_info.downhecerrors);
      printf(gettext("  up hec errors = %d\n"), modem_info.uphecerrors);
      printf(gettext("  down attenuation = %d\n"), modem_info.downattenuation);
      printf(gettext("  up attenuation = %d\n"), modem_info.upattenuation);
      printf(gettext("  down SNR margin = %d\n"), modem_info.downsignalnoiseratio);
      printf(gettext("  up SNR margin = %d\n"), modem_info.upsignalnoiseratio);
      printf(gettext("  down output power = %d\n"), modem_info.downoutputpower);
      printf(gettext("  up output power = %d\n"), modem_info.upoutputpower);
    }
  }
  usb_close(adsl_handle);

  return 0;
}

