/*
  Conexant AccessRunner init process
  Copyright (C) 2003 Josep Comas

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

  Author     : Josep Comas <jcomas@gna.es>
  Creation   : 24/1/2003

  Description: This program inits Conexant AccessRunner (USB ADSL Modem).

  Modems:

  User: Josep Comas <jcomas@gna.es>
  Manufacturer: Mac System (http://www.macsysco.com/)
  Model: MA08CU (ADSL 100U modem)
  Distribuited by: Vitelcom
  Model: EPS 5002 USB
  Internet provider: Telefonica
  Country: Spain
  Protocols: RFC1483/2684 routed, PPPoE

  User: Pascal Boucher <bcrp17@tiscali.fr>
  Manufacturer: Olitec (http://www.olitec.com/)
  Model: Modem USB ADSL V3
  Internet provider: Tiscali
  Country: France
  Protocol: PPPoATM

  User: Thomas Mikosch
  Manufacturer: Amigo Technology Co. (http://www.amigo.com.tw)
  Model: AMX-CA86U
  Distribuited by: Trust (http://www.trust.com) 
  Model: 235A Speedlink ADSL Web Modem (item no. 13141)
  Internet provider: HetNet
  Country: The Netherlands
  Protocol: PPPoATM

  User: Pawel Konieczny <konieczp@users.sourceforge.net>
  Manufacturer: Amigo Technology Co. (http://www.amigo.com.tw)
  Model: AMX-CA80U-4
  Distributed by (OEM): E-Tech (http://www.e-tech.nu/zon)
  Model: E-Tech ADSL USB Modem V2
  Internet provider: Zon (http://www.zonnet.nl)
  ADSL provider: Versatel (http://www.versatel.nl)
  Country: The Netherlands
  Protocol: RFC1483/2684 bridged/routed

  Log:

  28/4/2003 Josep Comas
  Added Olitec Modem USB ADSL V3 changes from Pascal Boucher
  Added support for Amigo AMX-CA86U

  30/4/2003 Josep Comas
  Added support for Olitec Modem USB ADSL V2

  1/5/2003 Josep Comas
  Added dispatch routine to retrieve info for commands sent
  Retrieve firmware version
  Retrieve MAC address
  Retrieve line and ADSL status
  Retrieve downbitrate and upbitrate

  2/5/2003 Josep Comas
  Better retrieving info (sleeps added)

  5/5/2003 Josep Comas
  Update to work with libusb-0.1.7

  3/6/2003 Josep Comas
  Added support for Amigo AMX-CA80U-2M
  Added some changes from Pawel Konieczny
  Added ADSL open mode
  Send commands in background and wait to finish

  4/6/2003 Josep Comas
  Fix usb_bulk_msg timeout
  Adjust for speed machines (2.4GHz processor)

  6/6/2003 Josep Comas
  Minor fixes for timeouts

  7/6/2003 Josep Comas
  Retries for send and wait commands

  13/6/2003 Josep Comas
  Support for all modems in same executable
  Return to old dispatch_info process

  16/6/2003 Josep Comas
  Fix getting info for Debian

  21/6/2003 Josep Comas
  Increased timeout for some usb controllers/modems when it is waiting answers

  24/6/2003 Josep Comas
  Remove thread to get answers and it uses a slow process

  28/6/2003 Josep Comas
  Increased sleep

  29/6/2003 Josep Comas
  Remove some commands and change others

  1/9/2003 Josep Comas
  Added support for Zoom 5510
  Added support for generic VID=0x0572, PID=0xCB01 and PID=0xCB06

  6/9/2003 Josep Comas
  Added support for Draytek Vigor 318
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <pthread.h>
#include <usb.h>
#include <usbi.h>
#include <libintl.h>
#include <locale.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <cxacru.h>


/* translation files */
#define TF_CODE "cxload"

/* patch boot rom */
char inifirm[] = {
  0x78, 0x20, 0x9f, 0xe5, 0x01, 0x10, 0xa0, 0xe3,
  0x00, 0x10, 0x82, 0xe5, 0x70, 0x20, 0x9f, 0xe5, 0x00, 0x10, 0xa0, 0xe3, 0x00, 0x10, 0x82, 0xe5,
  0x00, 0x10, 0xa0, 0xe3, 0x64, 0x20, 0x9f, 0xe5, 0x00, 0x10, 0x82, 0xe5, 0x60, 0x20, 0x9f, 0xe5,
  0x00, 0x10, 0x82, 0xe5, 0x01, 0x10, 0xa0, 0xe3, 0x58, 0x20, 0x9f, 0xe5, 0x00, 0x10, 0x82, 0xe5,

  0x00, 0x10, 0xa0, 0xe3, 0x50, 0x20, 0x9f, 0xe5,
  0x00, 0x10, 0x82, 0xe5, 0x4c, 0x20, 0x9f, 0xe5, 0x00, 0x10, 0x82, 0xe5, 0x48, 0x20, 0x9f, 0xe5,
  0x00, 0x10, 0x82, 0xe5, 0x44, 0x20, 0x9f, 0xe5, 0x00, 0x10, 0x82, 0xe5, 0x00, 0x10, 0xe0, 0xe3,
  0x3c, 0x20, 0x9f, 0xe5, 0x00, 0x10, 0x82, 0xe5, 0x38, 0x20, 0x9f, 0xe5, 0x00, 0x10, 0x82, 0xe5,

  0x34, 0x20, 0x9f, 0xe5, 0x00, 0x10, 0x82, 0xe5,
  0x30, 0x00, 0x9f, 0xe5, 0x00, 0xf0, 0xa0, 0xe1, 0x78, 0x00, 0x33, 0x00, 0x4c, 0x00, 0x35, 0x00,
  0x3c, 0x00, 0x33, 0x00, 0x44, 0x00, 0x33, 0x00, 0x2c, 0x00, 0x33, 0x00, 0x30, 0x00, 0x35, 0x00,
  0x34, 0x00, 0x35, 0x00, 0x38, 0x00, 0x35, 0x00, 0x3c, 0x00, 0x35, 0x00, 0x44, 0x00, 0x35, 0x00,

  0x38, 0x00, 0x33, 0x00, 0x40, 0x00, 0x33, 0x00,
  0x00, 0x10, 0x80, 0x00};

/* modem internal characteristics */
struct usb_modem_char {
  unsigned int vid;  /* VendorID */
  unsigned int pid;  /* ProductID */
  unsigned int arm_pll_f;  /* FCLK PLL Register value */
  unsigned int arm_pll_b;  /* BCLK PLL Register value */
  char patch_jump;  /* 1 = patch, 2 = jump */
  char *firmfile;  /* firmware file name */
  char param_first;  /* param first offset */
  char param_last;  /* param last offset */
  int params[0x50];  /* params */
};
struct usb_modem_char modem_char;

/* info about modem */
struct usb_modem_info {
  char firm_version[5];  /* firmware version */
  char mac[6];  /* MAC address */
  int down_bitrate;  /* download bitrate */
  int up_bitrate;  /* upload bitrate */
  int link_status;  /* link status */
  int line_status;  /* line status */
  int operational_mode;  /* operational mode */
};
struct usb_modem_info modem_info;

/* adsl modes */
const char *adsl_modes[] = {
 "ANSI",
 "G.DMT",
 "G.Lite"
};

/* modem_cmd_state = 0 (command not send/finished), 1 (wait 1 answer), 2 (wait 2 answers), ... */
unsigned char modem_cmd_state[32];

/* check if a file exists */
int file_exists(const char *filename)
{
  struct stat info_file;

  return stat(filename, &info_file) == 0;
}

/* show printable char */
void print_char(unsigned char c)
{
  if (c >= ' ' && c < 0x7f)
    printf("%c", c);
  else
    printf(".");
}

/* show buffer */
void dump(unsigned char *buf, int lenbuf, int lenline)
{
  int i, j;  /* counters */

  for (i = 0; i < lenbuf; i+= lenline)
  {
    for (j = i; j < lenbuf && j < i + lenline; j++)
      printf("%02x ", buf[j]);
    for (; j < i + lenline; j++)
      printf("   ");
    for (j = i; j < lenbuf && j < i + lenline; j++)
      print_char(buf[j]);
    printf("\n");
  }
}

/* transfer a control message to USB bus */
int transfer_ctrl_msg(usb_dev_handle *adsl_handle, int requesttype, int request, int value, int index, char *buf, int size)
{
  int j;  /* counter */
  int n;  /* bytes transfed or error code */
  int tmout = CTRL_TIMEOUT;  /* timeout value */

  n = 0;
  for (j = 0; j < CTRL_MSG_RETRIES; j++) {
#ifdef SIMULATE
    n = size;
#else
    n = usb_control_msg(adsl_handle, requesttype, request, value, index,  buf, size, tmout);
#endif
    if (n >= 0) {
#if DEBUG_TRANSFER
      printf(gettext("%d bytes transferred:\n"), n);
      dump(buf, n, 16);
#endif
      break;
    }
    else {
      printf(gettext("Error: usb_control_msg: %s\n"), usb_strerror());
      if (n == -EPIPE) {
        usb_clear_halt(adsl_handle, 0x00);
        usb_clear_halt(adsl_handle, 0x80);
      }
      else if (n == -ETIMEDOUT) {
	tmout += TIMEOUT_ADD;
      }
    }
  }
  if (n < 0) {
    printf(gettext("Error: usb_control_msg failed after %d retries\n"), CTRL_MSG_RETRIES);
    return -1;
  }
  return n;
}

/* receive a packet from USB bus by bulk transfer */
int read_bulk(usb_dev_handle *adsl_handle, int ep, char *buf, int size)
{
  int n;  /* bytes readed or error code */
  int i;  /* counter */
  int tmout = DATA_TIMEOUT;  /* timeout value */

  memset(buf, 0, sizeof(buf));
  n = 0;
  for (i = 0; i < READ_BULK_RETRIES; i++) {
#ifdef SIMULATE
    n = size;
#else
    n = usb_bulk_read(adsl_handle, ep, buf, size, tmout);
#endif
    if (n >= 0) {
#if DEBUG_TRANSFER
      printf(gettext("%d bytes downloaded:\n"), n);
      dump(buf, n, 16);
#endif
      break;
    }
    else {
      printf(gettext("Error: usb_bulk_read: %s\n"), usb_strerror());
      if (n == -EPIPE) {
        usb_clear_halt(adsl_handle, ep);
      }
      else if (n == -ETIMEDOUT) {
	tmout += TIMEOUT_ADD;
      }
    }
  }
  if (n < 0) {
    printf(gettext("Error: usb_bulk_read failed after %d retries\n"), READ_BULK_RETRIES);
    return -1;
  }
  return 0;
}

/* send one or more packets to USB bus by bulk transfer */
int send_bulk(usb_dev_handle *adsl_handle, int ep, char *buf, int nfil, int ncol)
{
  int i, j;  /* counters */
  int n;  /* bytes sent or error code */
  int tmout = DATA_TIMEOUT;  /* timeout value */

  n = 0;
  for (i = 0; i < nfil; i++)
  {
    for (j = 0; j < SEND_BULK_RETRIES; j++) {
#ifdef SIMULATE
      n = ncol;
#else
      n = usb_bulk_write(adsl_handle, ep, buf+(i*ncol), ncol, tmout);
#endif
      if (n >= 0) {
#if DEBUG_TRANSFER
        printf(gettext("%d bytes uploaded:\n"), n);
        dump(buf+(i*ncol), ncol, 16);
#endif
        break;
      }
      else {
        printf(gettext("Error: usb_bulk_write: %s\n"), usb_strerror());
	if (n == -EPIPE) {
	  usb_clear_halt(adsl_handle, ep);
	}
	else if (n == -ETIMEDOUT) {
	  tmout += TIMEOUT_ADD;
	}
      }
    }
    if (n < 0) {
      printf(gettext("Error: usb_bulk_write failed after %d retries\n"), SEND_BULK_RETRIES);
      return -1;
    }
  }
  return 0;
}

/* check if there are pending commands */
int check_pending_cmd(void)
{
  int i;  /* counter */

  for (i = 0; i < sizeof(modem_cmd_state); i++)
    if (modem_cmd_state[i] != 0)
      return -1;
  return 0;
}

/* return a value of an id */
int find_id_value(char id, char *buf, int *value)
{
  int i;  /* counter */
  int pos = -1;  /* where id was found */

  for (i = 0; i < buf[4]; i++)
  {
    if (buf[8+(i*8)] == id) {
      pos = i;
      *value = ((buf[15+(i*8)] & 0xff) << 24) | ((buf[14+(i*8)] & 0xff) << 16) | ((buf[13+(i*8)] & 0xff) << 8) | (buf[12+(i*8)] & 0xff);
      break;
    }
  }
  return pos;
}

/* dispatch received info */
int dispatch_info(usb_dev_handle *adsl_handle, unsigned int timeout)
{
  int n;  /* bytes received */
  char buf[MAX_MESSAGE_SIZE];  /* where store message received */
  int value;  /* value of a id */
  int cmd;  /* command */
  time_t first, last;  /* to wait */
  int pending;  /* if there are pending commands to answer */

  time(&first);
  do {
    if ((pending = check_pending_cmd()) != 0)
    {
      memset(buf, 0, MAX_MESSAGE_SIZE);
      n = usb_bulk_read(adsl_handle, USB_IN_INFO, buf, MAX_MESSAGE_SIZE, DATA_TIMEOUT);
      if (n == MAX_MESSAGE_SIZE) 
      {
        cmd = buf[0] & 0xff;
#if DEBUG_TRANSFER
  //      if (cmd != CMD_CARD_INFO_GET)
  //      {
        printf("Received message:\n");
        dump(buf, n, 16);
  //      }
#endif
        if ((buf[1] & 0xff) == REPLY_CMD) {
          switch (cmd) {
            case CMD_CARD_CONTROLLER_VERSION_GET:
              memcpy(modem_info.firm_version, buf+4, sizeof(modem_info.firm_version));
              break; 
            case CMD_CARD_INFO_GET:
              if (find_id_value(0x00, buf, &value) != -1)
                modem_info.down_bitrate = value;
              if (find_id_value(0x01, buf, &value) != -1)
                modem_info.up_bitrate = value;
              if (find_id_value(0x02, buf, &value) != -1)
                modem_info.link_status = value;
              if (find_id_value(0x03, buf, &value) != -1)
                modem_info.line_status = value;
              if (find_id_value(0x05, buf, &value) != -1) {
                memcpy(&modem_info.mac, &value, sizeof(value));
                if (find_id_value(0x04, buf, &value) != -1)
                  memcpy(&modem_info.mac[4], &value, 2);
              }
              if (find_id_value(0x15, buf, &value) != -1)
                modem_info.operational_mode = value;
              break;
          }
          if ((cmd >= FIRST_CMD) && (cmd < (FIRST_CMD+sizeof(modem_cmd_state))))
            if (modem_cmd_state[cmd-FIRST_CMD] > 0)
              modem_cmd_state[cmd-FIRST_CMD]--;
        } 
      }
      else
        if (n < 0) {
          printf(gettext("Error: usb_bulk_read: %s\n"), usb_strerror());
          if (n == -EPIPE)
            usb_clear_halt(adsl_handle, USB_IN_INFO);
          else if (n == -ETIMEDOUT) {
            usb_resetep(adsl_handle, USB_IN_INFO);
            sleep(1);
          }
        }
    }
    time(&last);
  }
  while ((pending == -1) && (difftime(last, first) < timeout));
  if (pending == 0)
    return 0;
  else {
    printf(gettext("Error: timeout waiting answer\n"));
    return -1;
  }
}

/* send a bulk transfer and wait to receive x transfers */
int send_cmd_wait_answers(usb_dev_handle *adsl_handle, char *buf, unsigned int wait, unsigned char answers, unsigned int timeout)
{
  int i;  /* counter */
  char id;

  if ((adsl_handle == NULL) || (buf == NULL))
    return -1;
  id = buf[0];
  for (i = 0; i < 4; i++) {
    modem_cmd_state[(id & 0xff)-FIRST_CMD] = 0;
    if (!send_bulk(adsl_handle, USB_OUT_INFO, buf, 1, MAX_MESSAGE_SIZE)) {
      modem_cmd_state[(id & 0xff)-FIRST_CMD] = answers;
      sleep(wait);
      if (dispatch_info(adsl_handle, timeout) == 0)
        return 0;
    }
    usb_resetep(adsl_handle, USB_IN_INFO);
    usb_resetep(adsl_handle, USB_OUT_INFO);
    wait += wait;
  }
  return -1;
}

/* format a message */
void format_message(int cmd, int ldata, int atype, int address, char *bufin)
{
  char buf[8];  /* initial bytes of a message */

  memset(buf, 0, sizeof(buf));
  buf[0] = cmd & 0xff;    /* usb command */
  buf[1] = ldata & 0xff;  /* data length */ 
  buf[2] = atype & 0xff;  /* access type */
  buf[3] = 0x00;          /* ack request (0 = FALSE) */
  /* address */
  buf[4] = address & 0xff;
  buf[5] = (address >> 8) & 0xff;
  buf[6] = (address >> 16) & 0xff;
  buf[7] = (address >> 24) & 0xff;
  memcpy(bufin, buf, sizeof(buf));
}

/* write a value into an address */
int write_value(usb_dev_handle *adsl_handle, int address, int value)
{
  char buf[MAX_MESSAGE_SIZE];  /* message to send */

  memset(buf, 0, sizeof(buf));
  buf[8] = value & 0xff;
  buf[9] = (value >> 8) & 0xff;
  buf[10] = (value >> 16) & 0xff;
  buf[11] = (value >> 24) & 0xff;
  format_message(CMD_USB_WRITE_MEM, 4, MA_DWORD, address, buf);
  if (send_bulk(adsl_handle, USB_OUT_INFO, buf, 1, MAX_MESSAGE_SIZE))
    return -1;

  return 0;
}

/* send a command */
int send_cmd(usb_dev_handle *adsl_handle, char id, unsigned char answers)
{
  char buf[MAX_MESSAGE_SIZE];  /* message to send */

  memset(buf, 0, MAX_MESSAGE_SIZE);
  buf[0] = id;
  modem_cmd_state[(id & 0xff)-FIRST_CMD] = 0;
  if (send_bulk(adsl_handle, USB_OUT_INFO, buf, 1, MAX_MESSAGE_SIZE))
    return -1;
  modem_cmd_state[(id & 0xff)-FIRST_CMD] = answers;
  return 0;
}

/* send a command */
int send_cmd_wait(usb_dev_handle *adsl_handle, char id, unsigned int wait, unsigned char answers, unsigned int timeout)
{
  char buf[MAX_MESSAGE_SIZE];  /* message to send */

  memset(buf, 0, MAX_MESSAGE_SIZE);
  buf[0] = id;
  return send_cmd_wait_answers(adsl_handle, buf, wait, answers, timeout);
}

/* send a command with a value */
int send_cmd_value(usb_dev_handle *adsl_handle, char id, unsigned char answers, int value)
{
  char buf[MAX_MESSAGE_SIZE];  /* message to send */

  memset(buf, 0, sizeof(buf));
  format_message(id, 0, 0, value, buf);
  modem_cmd_state[(id & 0xff)-FIRST_CMD] = 0;
  if (send_bulk(adsl_handle, USB_OUT_INFO, buf, 1, MAX_MESSAGE_SIZE))
    return -1;
  modem_cmd_state[(id & 0xff)-FIRST_CMD] = answers;
  return 0;
}

/* send a goto command */
int send_goto_cmd(usb_dev_handle *adsl_handle, int address)
{
  char buf[MAX_MESSAGE_SIZE];  /* message to send */

  memset(buf, 0, sizeof(buf));
  format_message(CMD_USB_GOTO_MEM, 0, MA_BYTE, address, buf);
  if (send_bulk(adsl_handle, USB_OUT_INFO, buf, 1, MAX_MESSAGE_SIZE))
    return -1;

  return 0;
}

/* write open_mode into a file */
int write_configuration(char *filename, int open_mode)
{
  FILE *params;   /* file handle */

  params = fopen(filename, "w");
  if (params == NULL)
  {
    printf(gettext("Error: I can't create file %s\n"), filename);
    return -1;
  }
  if (fprintf(params, "PARAM_0a=%i\n", open_mode) < 0)
  {
    printf(gettext("Error: I can't write into file %s\n"), filename);
    return -1;
  }
  fclose(params);
  return 0;
}

/* send configuration */
int send_configuration(usb_dev_handle *adsl_handle, char first, char last)
{
  char buf[MAX_MESSAGE_SIZE];  /* message to send */

  int len = last - first + 1;  /* number of variables */
  int npackets = len/7;  /* number of packets to send */
  int i, j;  /* counters */
  char place = first;  /* address */
  char nvp; /* number of variables in a packet */

  if (len % 7 != 0)
    npackets++;

  PDEBUG(gettext("Packets to send = %i\n"), npackets);
  for (i = 0; i < npackets; i++)
  {
    memset(buf, 0, MAX_MESSAGE_SIZE);
    buf[0] = CMD_CARD_DATA_SET;
    if ((i == npackets - 1) && (len % 7 != 0))
      nvp = len % 7;
    else
      nvp = 7;
    buf[4] = nvp;
    for (j = 0; j < nvp; j++)
    {
      buf[(8*j)+8] = place;
      buf[(8*j)+12] = modem_char.params[place & 0xff] & 0xff;
      buf[(8*j)+13] = (modem_char.params[place & 0xff] >> 8) & 0xff;
      if (place == 0x0a) /* ADSL open mode */
        if (write_configuration("/tmp/cxacru.params", modem_char.params[place & 0xff]))
          return -1;
      place++;
    }
    //if (send_cmd_wait_answers(adsl_handle, buf, 2, ONE_ANSWER, 10))
    if (send_bulk(adsl_handle, USB_OUT_INFO, buf, 1, MAX_MESSAGE_SIZE))
      return -1;
  }
  return 0;
}

/* clear endpoints that we use */
void clear_endpoints(usb_dev_handle *adsl_handle) {
  usb_resetep(adsl_handle, USB_IN_INFO);
  usb_resetep(adsl_handle, USB_OUT_INFO);
  usb_resetep(adsl_handle, USB_IN_DATA);
  usb_resetep(adsl_handle, USB_OUT_DATA);
/*
  usb_clear_halt(adsl_handle, USB_IN_INFO);
  usb_clear_halt(adsl_handle, USB_OUT_INFO);
  usb_clear_halt(adsl_handle, USB_IN_DATA);
  usb_clear_halt(adsl_handle, USB_OUT_DATA);
*/
}

/* send_block */
int send_block(usb_dev_handle *adsl_handle, int place, char *bufin, int len)
{
  int npackets = len/MAX_DATA_MESSAGE_SIZE;  /* number of packets to send */
  int i;  /* counter */
  int lmes;  /* message length */
  int lblock = 0;  /* block length */
  char mes[MAX_MESSAGE_SIZE];  /* message to send */
  char block[MAX_DATA_TRANSFER];  /* block of messages to send */

  if (len % MAX_DATA_MESSAGE_SIZE != 0)
    npackets++;

  PDEBUG(gettext("Packets to send = %i\n"), npackets);
  for (i = 0; i < npackets; i++)
  {
     if (i == (npackets - 1) && (len % MAX_DATA_MESSAGE_SIZE != 0))
       lmes = len % MAX_DATA_MESSAGE_SIZE;
     else
       lmes = MAX_DATA_MESSAGE_SIZE;
     memset(mes, 0, MAX_MESSAGE_SIZE);
     memcpy(mes+8, bufin+(i*MAX_DATA_MESSAGE_SIZE), lmes);

     /* Code for remaining bytes (last bytes) if it is not divisible by 4 (length calcul) */
     if (lmes % 4 != 0)  /* round up for MA_DWORD transfer */
       lmes = (lmes/4 + 1) * 4;

     format_message(CMD_USB_WRITE_MEM, lmes, MA_DWORD, place+(i*MAX_DATA_MESSAGE_SIZE), mes);
     memcpy(block+lblock, mes, MAX_MESSAGE_SIZE);
     lblock += MAX_MESSAGE_SIZE;
     if ((i == (npackets - 1)) || (lblock == MAX_DATA_TRANSFER)) {
       PDEBUG(gettext("Sending block...\n"));
       if (send_bulk(adsl_handle, USB_OUT_INFO, block, 1, lblock))
         return -1;
       lblock = 0;
     }
  }
  return 0;
}

/* load firmware */
int load_firmware(usb_dev_handle *adsl_handle, unsigned int tmodem)
{
  unsigned char buf[64*MAX_DATA_MESSAGE_SIZE];   /* max data that we can send */
  FILE *soft;   /* file handle */
  long len;     /* length */
  int place;    /* initial target address */
//#ifndef SIMULATE
//  int i;  /* counter */
//#endif
  time_t first, last, before;  /* to wait */

  /* clear endpoints */
  clear_endpoints(adsl_handle);

  /* load firmware */
  printf(gettext("Loading and sending %s...\n"), modem_char.firmfile); 
  soft = fopen(modem_char.firmfile, "rb");
  if (soft == NULL)
  {
    printf(gettext("Error: I can't open file %s\n"), modem_char.firmfile);
    return -1;
  }
  fseek(soft, 0L, SEEK_END);
  len = ftell(soft);
  PDEBUG(gettext("Length of file %s = %ld bytes\n"), modem_char.firmfile, len);
/*
  fseek(soft, 0L, SEEK_SET);
  len = fread(buf, 1, 5, soft);
  if (len <= 0)
  {
    printf(gettext("Error: No bytes to read from file %s\n"), modem_char.firmfile);
    return -1;
  }
  if (len != 5)
  {
    printf(gettext("Error: I can't read initial 5 bytes from file %s\n"), filename);
    return -1;
  }
*/
  /* check initial bytes */
/*
  PDEBUG(gettext("Initial bytes from file %s:\n"), filename);
#if DEBUG
  dump(buf, 5, 5);
#endif
  if (buf[0] != FIRMBYTE1 || buf[1] != FIRMBYTE2 || buf[2] != FIRMBYTE3 || buf[3] != FIRMBYTE4 || buf[4] != FIRMBYTE5) {
    printf(gettext("Error: Maybe file %s isn't Conexant firmware, contact with author of this program\n"), filename);
    return -1;
  }
*/

  /*******************************/
  /* initialize clock and memory */
  /*******************************/

  PDEBUG("PreInit...\n");
  if (write_value(adsl_handle, ARM_PLL_F, modem_char.arm_pll_f))
    return -1;
  if (write_value(adsl_handle, ARM_PLL_B, modem_char.arm_pll_b))
    return -1;
  if (write_value(adsl_handle, ARM_EMCR, SDRAM_ENABLE))
    return -1;


  /*****************/
  /* send firmware */
  /*****************/

  PDEBUG("Firmware...\n");
  fseek(soft, 0L, SEEK_SET);
  place = FW_ADDRESS;
  while ((len = fread(buf, 1, 64*MAX_DATA_MESSAGE_SIZE, soft)) > 0)
  {
    PDEBUG(gettext("%ld bytes readed from file %s\n"), len, modem_char.firmfile);
    if (send_block(adsl_handle, place, buf, len))
      return -1;
    place += len;
  } 
  fclose(soft);


  /*************/
  /* post load */
  /*************/

  PDEBUG("PostInit...\n");

  /* patch (boot rom) */
  if (modem_char.patch_jump == CXPATCH) {
    /* patch (boot rom) */
    if (send_block(adsl_handle, BR_ADDRESS, inifirm, sizeof(inifirm)))
      return -1;
  }

  /* write vendorID and productID (boot signature) */
  if (write_value(adsl_handle, WB_SIGN_ADDRESS, (modem_char.pid << 16) | modem_char.vid))
    return -1;

  /* jump to firmware address */
  if (modem_char.patch_jump == CXJUMP) {
    if (send_goto_cmd(adsl_handle, FW_ADDRESS))
      return -1;
  }

  /* patch return address (boot rom) */
  if (modem_char.patch_jump == CXPATCH) {
    if (write_value(adsl_handle, BR_STACK_ADDRESS, BR_ADDRESS))
      return -1;
  }

  printf(gettext("Firmware is sent!\n"));

  /* wait until firmware is ready */
  sleep(1);


  /**************************************************/
  /* set parameters, get info, initialize ADSL line */
  /**************************************************/

  /* clear endpoints */
  clear_endpoints(adsl_handle);

  /* send configuration */
  printf(gettext("Setting configuration...\n"));
  PDEBUG("Sending CMD_CARD_DATA_SET...\n");
  if (send_configuration(adsl_handle, modem_char.param_first, modem_char.param_last))
    return -1;
  sleep(4);

  /* get information about modem */
/* Remove this (not needed):
  printf(gettext("Getting info...\n"));
  memset(&modem_info, 0, sizeof(modem_info));
  PDEBUG("Sending CMD_CARD_CONTROLLER_VERSION_GET...\n");
  if (send_cmd_wait(adsl_handle, CMD_CARD_CONTROLLER_VERSION_GET, 2, ONE_ANSWER, 10))
    return -1;

  PDEBUG("Sending CMD_CARD_INFO_GET...\n");
  if (send_cmd_wait(adsl_handle, CMD_CARD_INFO_GET, 2, FOUR_ANSWERS, 10))
    return -1;

#ifndef SIMULATE
  printf("Firmware version = ");
  for (i = 0; i < sizeof(modem_info.firm_version); i++)
    printf("%c", modem_info.firm_version[i]);
  printf("\n");
  printf("MAC address = ");
  for (i = 0; i < sizeof(modem_info.mac)-1; i++)
    printf("%02x:", modem_info.mac[i] & 0xff);
  printf("%02x", modem_info.mac[sizeof(modem_info.mac)-1] & 0xff);
  printf("\n");
#endif
*/

  /* start line */
  PDEBUG("Sending CMD_CHIP_ADSL_LINE_START...\n");
  if (send_cmd_wait(adsl_handle, CMD_CHIP_ADSL_LINE_START, 1, ONE_ANSWER, 5))
    return -1;

#ifdef SIMULATE
  exit(0);
#endif

  /* waiting until line is up (a maximum time) */
  printf(gettext("Waiting ADSL line is up (until %d seconds)...\n"), MAX_WAIT_LINE_UP);
  time(&first); before = first;
  do {
    PDEBUG("Sending CMD_CARD_INFO_GET...\n");
    if (send_cmd_wait(adsl_handle, CMD_CARD_INFO_GET, 1, FOUR_ANSWERS, 5))
      return -1;
    PDEBUG(gettext("Link status = %02x\n"), modem_info.link_status);
    PDEBUG(gettext("Line status = %02x\n"), modem_info.line_status);
    if (difftime(time(&last), before) > 1) {
#ifndef DEBUG
      printf(".");
      fflush(stdout);
#endif
      before = last;
    }
  }
  while ((modem_info.line_status != LINE_STATUS_UP) && (difftime(last, first) < MAX_WAIT_LINE_UP));
  printf("\n");

  if (modem_info.line_status == LINE_STATUS_UP)
  {
    printf(gettext("ADSL line is up (Downstream %u Kbits/s, Upstream %u Kbits/s)\n"), modem_info.down_bitrate, modem_info.up_bitrate);
#ifdef DEBUG
    printf(gettext("ADSL mode = %i"), modem_info.operational_mode);
    if ((modem_info.operational_mode > 0) && (modem_info.operational_mode < 4))
      printf(" (%s)", adsl_modes[modem_info.operational_mode-1]);
    printf("\n");
#endif
  }
  else
    printf(gettext("ADSL line is down\n"));

#ifdef DEBUG
  printf(gettext("Waiting to receive first ATM cells...\n"));
  time(&first); before = first;
  do {
    len = usb_bulk_read(adsl_handle, USB_IN_DATA, buf, sizeof(buf), DATA_TIMEOUT);
    if (len > 0)  {
      printf("\n");
      printf(gettext("ATM cells received:\n"));
      dump(buf, len, 16);
      return 0;
    }
    if (difftime(time(&last), before) > 1) {
      printf(".");
      fflush(stdout);
      before = last;
    }
  } 
  while (difftime(last, first) < 60);
  printf("\n");
#endif

  return 0;
}

/* it inits modem according modem type */
void init_modem(unsigned int tmodem, struct usb_device *adsl_dev, int open_mode) {

  memset(&modem_char, 0, sizeof(modem_char));
  if (adsl_dev != NULL) {
    modem_char.vid = adsl_dev->descriptor.idVendor;
    modem_char.pid = adsl_dev->descriptor.idProduct;
  }

  switch (tmodem) {

    /* Conexant (Euphrates project) */
    case 1:
      modem_char.arm_pll_f = PLL_FAST_144MHZ;
      modem_char.arm_pll_b = PLL_FAST_100MHZ;
      modem_char.patch_jump = CXPATCH;
      modem_char.firmfile = "/usr/sbin/cxfirm1.bin";
      modem_char.param_first = 0x00;
      modem_char.param_last = 0x40;
      modem_char.params[0x00] = 0x30;
      modem_char.params[0x03] = 0x01;
      modem_char.params[0x04] = 0x01;
      modem_char.params[0x07] = 0x09;
      modem_char.params[0x08] = 0x10;
      modem_char.params[0x0a] = 0x03;
      modem_char.params[0x0c] = 0x01;
      modem_char.params[0x0d] = 0x02;
      modem_char.params[0x0f] = 0xc8;
      modem_char.params[0x10] = 0x01;
      modem_char.params[0x12] = 0x39;
      modem_char.params[0x14] = 0x01;
      modem_char.params[0x17] = 0x0e;
      modem_char.params[0x18] = 0x05;
      modem_char.params[0x1b] = 0x0a;
      modem_char.params[0x1c] = 0x01;
      modem_char.params[0x1d] = 0x03;
      modem_char.params[0x1e] = 0x0a;
      modem_char.params[0x2f] = 0x14;
      modem_char.params[0x30] = 0x0c;
      modem_char.params[0x31] = 0x02;
      modem_char.params[0x32] = 0x03;
      break;

    /* Olitec V2 */
    case 2:
      modem_char.arm_pll_f = PLL_FAST_144MHZ;
      modem_char.arm_pll_b = PLL_FAST_100MHZ;
      modem_char.patch_jump = CXPATCH;
      modem_char.firmfile = "/usr/sbin/cxfirm2.bin";
      modem_char.param_first = 0x00;
      modem_char.param_last = 0x41;
      modem_char.params[0x00] = 0x30;
      modem_char.params[0x03] = 0x01;
      modem_char.params[0x04] = 0x01;
      modem_char.params[0x07] = 0x09;
      modem_char.params[0x08] = 0x10;
      modem_char.params[0x0c] = 0x01;
      modem_char.params[0x0d] = 0x02;
      modem_char.params[0x0f] = 0xc8;
      modem_char.params[0x10] = 0x01;
      modem_char.params[0x12] = 0x22;
      modem_char.params[0x14] = 0x01;
      modem_char.params[0x17] = 0x0e;
      modem_char.params[0x18] = 0x05;
      modem_char.params[0x2f] = 0x14;
      modem_char.params[0x30] = 0x0c;
      modem_char.params[0x31] = 0x02;
      modem_char.params[0x32] = 0x03;
      break;

    /* Olitec V3 */
    case 3:
      modem_char.arm_pll_f = PLL_FAST_144MHZ_RM;
      modem_char.arm_pll_b = PLL_FAST_100MHZ_RM;
      modem_char.patch_jump = CXJUMP;
      modem_char.firmfile = "/usr/sbin/cxfirm3.bin";
      modem_char.param_first = 0x00;
      modem_char.param_last = 0x48;
      modem_char.params[0x03] = 0x01;
      modem_char.params[0x04] = 0x01;
      modem_char.params[0x07] = 0x09;
      modem_char.params[0x08] = 0x10;
      modem_char.params[0x0a] = 0x04;
      modem_char.params[0x0c] = 0x01;
      modem_char.params[0x0f] = 0xc8;
      modem_char.params[0x10] = 0x01;
      modem_char.params[0x14] = 0x01;
      modem_char.params[0x17] = 0x0e;
      modem_char.params[0x18] = 0x05;
      modem_char.params[0x2d] = 0x80;
      modem_char.params[0x2f] = 0x0a;
      modem_char.params[0x30] = 0x0c;
      modem_char.params[0x31] = 0x02;
      modem_char.params[0x32] = 0x03;
      modem_char.params[0x36] = 0x0180;
      break;

    /* Amigo AMX-CA86U */
    case 4:
      modem_char.arm_pll_f = PLL_FAST_144MHZ;
      modem_char.arm_pll_b = PLL_FAST_100MHZ;
      modem_char.patch_jump = CXPATCH;
      modem_char.firmfile = "/usr/sbin/cxfirm3.bin";
      modem_char.param_first = 0x00;
      modem_char.param_last = 0x48;
      modem_char.params[0x03] = 0x01;
      modem_char.params[0x04] = 0x01;
      modem_char.params[0x07] = 0x09;
      modem_char.params[0x08] = 0x10;
      modem_char.params[0x0d] = 0x02;
      modem_char.params[0x0f] = 0xc8;
      modem_char.params[0x10] = 0x01;
      modem_char.params[0x12] = 0x22;
      modem_char.params[0x14] = 0x01;
      modem_char.params[0x17] = 0x0e;
      modem_char.params[0x18] = 0x05;
      modem_char.params[0x2d] = 0x80;
      modem_char.params[0x2f] = 0x0a;
      modem_char.params[0x30] = 0x0c;
      modem_char.params[0x31] = 0x02;
      modem_char.params[0x32] = 0x03;
      modem_char.params[0x36] = 0x0180;
      break;

    /* Amigo AMX-CA80U-2M */
    case 5:
      modem_char.arm_pll_f = PLL_FAST_144MHZ_RM;
      modem_char.arm_pll_b = PLL_FAST_100MHZ_RM;
      modem_char.patch_jump = CXJUMP;
      modem_char.firmfile = "/usr/sbin/cxfirm4.bin";
      modem_char.param_first = 0x00;
      modem_char.param_last = 0x4a;
      modem_char.params[0x03] = 0x01;
      modem_char.params[0x04] = 0x01;
      modem_char.params[0x07] = 0x09;
      modem_char.params[0x08] = 0x10;
      modem_char.params[0x0c] = 0x01;
      modem_char.params[0x0d] = 0x02;
      modem_char.params[0x0f] = 0xc8;
      modem_char.params[0x10] = 0x01;
      modem_char.params[0x14] = 0x01;
      modem_char.params[0x17] = 0x0e;
      modem_char.params[0x18] = 0x05;
      modem_char.params[0x19] = 0x04;
      modem_char.params[0x2d] = 0x80;
      modem_char.params[0x2f] = 0x0a;
      modem_char.params[0x30] = 0x0c;
      modem_char.params[0x31] = 0x02;
      modem_char.params[0x32] = 0x03;
      modem_char.params[0x36] = 0x0180;
      modem_char.params[0x4a] = 0x1100;
      break;

    /* Zoom 5510 */
    case 6:
      modem_char.arm_pll_f = PLL_FAST_144MHZ_RM;
      modem_char.arm_pll_b = PLL_FAST_100MHZ_RM;
      modem_char.patch_jump = CXJUMP;
      modem_char.firmfile = "/usr/sbin/cxfirm5.bin";
      modem_char.param_first = 0x00;
      modem_char.param_last = 0x48;
      modem_char.params[0x03] = 0x01;
      modem_char.params[0x04] = 0x01;
      modem_char.params[0x07] = 0x09;
      modem_char.params[0x08] = 0x10;
      modem_char.params[0x0c] = 0x01;
      modem_char.params[0x0d] = 0x02;
      modem_char.params[0x0f] = 0xc8;
      modem_char.params[0x10] = 0x01;
      modem_char.params[0x14] = 0x01;
      modem_char.params[0x17] = 0x0e;
      modem_char.params[0x18] = 0x05;
      modem_char.params[0x2d] = 0x80;
      modem_char.params[0x2f] = 0x0a;
      modem_char.params[0x30] = 0x0c;
      modem_char.params[0x31] = 0x02;
      modem_char.params[0x32] = 0x03;
      modem_char.params[0x36] = 0x0180;
      break;
  }
  if (open_mode != -1)
    modem_char.params[0x0a] = open_mode & 0xff;
}

/* check if it is a known modem */
int check_modem(unsigned int vid, unsigned int pid)
{
  /* Vendor = Conexant, Product = ADSL modem (Euphrates project) */
  if (vid == 0x0572 && pid == 0xcafe)
    return 1;

  /* Vendor = Olitec, Product = ADSL modem version 2 */
  else if (vid == 0x08e3 && pid == 0x0100)
    return 2;

  /* Vendor = Olitec, Product = ADSL modem version 3 */
  else if (vid == 0x08e3 && pid == 0x0102)
    return 3;

  /* Vendor = Trust/Amigo Technology Co., Product = AMX-CA86U */
  else if (vid == 0x0eb0 && pid == 0x3457)
    return 4;

  /* Vendor = Conexant, Product = ADSL modem (Hasbani project) */
  /* Used for:
       Vendor = Amigo Technology Co.
       Product = AMX-CA80U-2M
       Vendor = E-Tech
       Product = USB ADSL Modem
  */
  else if (vid == 0x0572 && pid == 0xcb00)
    return 5;
  /* Used for:
       Vendor = 
       Product = 
  */
  else if (vid == 0x0572 && pid == 0xcb01)
    return 5;
  /* Used for:
       Vendor = Etec
       Product = USB ADSL Modem
       Vendor = Aethra
       Product = Starmodem USB
  */
  else if (vid == 0x0572 && pid == 0xcb06)
    return 5;

  /* Vendor = Draytek, Product = Vigor 318 */
  else if (vid == 0x0675 && pid == 0x0200)
    return 5;

  /* Vendor = Zoom, Product = 5510 */
  else if (vid == 0x1803 && pid == 0x5510)
    return 6;
 
  return -1;
}

int main(int argc, char *argv[])
{
  /* bus structures variables */
  struct usb_bus * bus;
  struct usb_device * dev;
  struct usb_device * adsl_dev = NULL;
  usb_dev_handle * adsl_handle;

  /* boolean value */
  int goon;

  /* result code */
  int r = 0; 

  /* open mode */
  int open_mode = -1;

  /* type of modem */
  int tmodem = -1;


  /* reset command queries */
  memset(modem_cmd_state, 0, sizeof(modem_cmd_state));

  /* init locale */
  setlocale(LC_ALL, "");
  //if (file_exists("./locale")) 
  //  bindtextdomain(TF_CODE, "./locale");  /* set directory for a domain (source code messages) */
  //else 
    bindtextdomain(TF_CODE, "/usr/share/locale");  /* set directory for a domain (source code messages) */
  textdomain(TF_CODE);  /* set domain */

  /* show program information */
  printf(gettext("Conexant AccessRunner microcode upload program."));
  printf(" 6/9/2003\n");
  printf("Josep Comas <jcomas@gna.es>\n");
  printf("See credits in documentation\n\n");

  /* check parameters */
  if (argc < 1)
  {
    printf(gettext("Usage:\n"));
    printf(gettext("   %s [open_mode]\n"), argv[0]);
    return -1;
  }
  if (argc > 1) {
    open_mode = atoi(argv[1]);
    if ((open_mode < 0) || (open_mode > 5)) {
      printf(gettext("Error: Incorrect open mode\n"));
      return -1;
    }
  }

  /* init USB bus and find devices */
  usb_init();
  if (usb_find_busses() < 0)
  {
    printf(gettext("Error: I can't find busses\n"));
    return -1;
  }
  if (usb_find_devices() < 0)
  {
    printf(gettext("Error: I can't find devices\n"));
    return -1;
  }

  /* search first Conexant AccessRunner */
  bus = usb_busses;
  goon = 1;
  while (bus && goon)
  {
    dev = bus->devices;
    while (dev && goon)
    {
      tmodem = check_modem(dev->descriptor.idVendor, dev->descriptor.idProduct);
      if (tmodem > 0)
      {
        goon = 0;
        adsl_dev = dev;
      }
      else
        dev = dev->next;
    }
    if (goon)
      bus = bus->next;
  }
  if (adsl_dev == NULL)
  {
    printf(gettext("Error: I didn't find ADSL modem\n"));
    return -1;
  }
  printf(gettext("I found ADSL modem with VendorID = %04x & ProductID = %04x\n"), adsl_dev->descriptor.idVendor, adsl_dev->descriptor.idProduct);

#if DEBUG
   printf(" bLength: 0x%02x\n", adsl_dev->config->bLength);
   printf(" bDescriptorType: 0x%02x\n", adsl_dev->config->bDescriptorType);
   printf(" wTotalLength: 0x%04x\n", adsl_dev->config->wTotalLength);
   printf(" bNumInterfaces: 0x%02x\n", adsl_dev->config->bNumInterfaces);
   printf(" bConfigurationValue: 0x%02x\n", adsl_dev->config->bConfigurationValue);
   printf(" iConfiguration: 0x%02x\n", adsl_dev->config->iConfiguration);
   printf(" bmAttributes: 0x%02x\n", adsl_dev->config->bmAttributes);
   printf(" MaxPower: 0x%02x\n", adsl_dev->config->MaxPower);
#endif

  /* connect to ADSL modem */
  adsl_handle = usb_open(adsl_dev);
  if (adsl_handle == NULL)
  {
    printf(gettext("Error: Couldn't get device handle for ADSL modem\n"));
    return -1;
  }
  /* set configuration */
  if (usb_set_configuration(adsl_handle, 1) < 0)
  {
    printf("Error: usb_set_configuration: %s\n", usb_strerror());
    return -1;
  }
  /* check if other program is using interface 0 */
  if (usb_claim_interface(adsl_handle, 0) < 0)
  {
    printf("Error: usb_claim_interface: %s\n", usb_strerror());
    return -1;
  }
  PDEBUG(gettext("Interface = %d\n"), adsl_handle->interface);

  init_modem(tmodem, adsl_dev, open_mode); 
  r = load_firmware(adsl_handle, tmodem); 

  PDEBUG(gettext("Releasing interface...\n"));
  usb_release_interface(adsl_handle, 0);
  PDEBUG(gettext("Releasing device...\n"));
  usb_close(adsl_handle);

  return r;
   
}

