/*************************************************************************** 
 * RT2400 SourceForge Project - http://rt2400.sourceforge.net              * 
 *                                                                         * 
 *   This program is free software; you can redistribute it and/or modify  * 
 *   it under the terms of the GNU General Public License as published by  * 
 *   the Free Software Foundation; either version 2 of the License, or     * 
 *   (at your option) any later version.                                   * 
 *                                                                         * 
 *   This program is distributed in the hope that it will be useful,       * 
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        * 
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         * 
 *   GNU General Public License for more details.                          * 
 *                                                                         * 
 *   You should have received a copy of the GNU General Public License     * 
 *   along with this program; if not, write to the                         * 
 *   Free Software Foundation, Inc.,                                       * 
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             * 
 *                                                                         * 
 *   Licensed under the GNU GPL                                            * 
 *   Original code supplied under license from RaLink Inc, 2003.           * 
 ***************************************************************************/ 

 /*************************************************************************** 
 *      Module Name: rtmp_data.c 
 *              
 *      Abstract: Data path subroutines 
 *              
 *      Revision History: 
 *      Who             When            What 
 *      --------        -----------     ----------------------------- 
 *      PaulL           2nd  Oct 02     Merge and Modify 
 *      MarkW           9th  Feb 04     Baseline of code
 *      PaulL           5th  May 04     Initial Promisc code changes (off)
 *      Ivo             27th Jun 04     Fixed spinlock to irqsave
 * 	MarkW		24th Sep 04	Changes to RX counters
 *	MarkW		26th Sep 04	Fixed MAC source-rewrite issue
 *      MarkW           8th  Dec 04     Cleanup of TX counters
 *      RobinC(RT2500)  21st Dec 04     RFMON Support
 *      MarkW(RT2500)   21st Dec 04     iwconfig mode monitor support
 ***************************************************************************/ 

 
#include "rt_config.h"

static	UCHAR	SNAP_802_1H[] = {0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00};
static	UCHAR	SNAP_BRIDGE_TUNNEL[] = {0xaa, 0xaa, 0x03, 0x00, 0x00, 0xf8};
static	UCHAR	EAPOL[] = {0x88, 0x8e};

static	UCHAR	IPX[] = {0x81, 0x37};
static	UCHAR	APPLE_TALK[] = {0x80, 0xf3};

/*
	========================================================================

	Routine	Description:
		Process	RxDone interrupt, running in DPC level

	Arguments:
		pAd		Pointer	to our adapter

	Return Value:
		None

	Note:
		This routine has to	maintain Rx	ring read pointer.
	========================================================================
*/
VOID	RTMPHandleRxDoneInterrupt(
	IN	struct net_device *net_dev)
{
	RTMP_ADAPTER	        *pAd = netdev_priv(net_dev);
	PRXD_STRUC		pRxD;
	PHEADER_802_11	        pHeader;
	PUCHAR			pData;
	PUCHAR			pEncap;
	PVOID			pManage;
	PUCHAR			pDestMac, pSrcMac;
	UCHAR			Header802_3[14];
	UCHAR			LLC_Len[2];
	UINT			PacketSize;
	ULONG			High32TSF, Low32TSF;
	UCHAR			Count;
	NDIS_STATUS		Status;
	struct sk_buff		*skb;
	unsigned long		irqflag;

	// Make sure Rx ring resource won't be used by other threads
	spin_lock_irqsave(&pAd->RxRingLock, irqflag);
	Count = 0;
	do
	{
		// Point to Rx indexed rx ring descriptor
		pRxD = (PRXD_STRUC) pAd->RxRing[pAd->CurRxIndex].va_addr;

		// In case of false alarm or processed at last instance
		if (pRxD->Owner	!= DESC_OWN_HOST)
		{
			break;
		}

		// Point to Rx ring buffer where stores the real data frame
		pData	= (PUCHAR) (pAd->RxRing[pAd->CurRxIndex].va_data_addr);
		// Cast to 802.11 header for flags checking
		pHeader	= (PHEADER_802_11) pData;

                if (pAd->PortCfg.BssType == BSS_MONITOR)
                {
                    struct sk_buff *skb;
                    if ((skb = __dev_alloc_skb(2048, GFP_DMA|GFP_ATOMIC)) != NULL)
                    {
                       skb->dev = pAd->net_dev;
                       memcpy(skb_put(skb, pRxD->DataByteCnt), pData, pRxD->DataByteCnt);
                       skb->mac.raw = skb->data;
                       skb->pkt_type = PACKET_OTHERHOST;
                       skb->protocol = htons(ETH_P_802_2); 
                       skb->ip_summed = CHECKSUM_NONE;
                       netif_rx(skb);
                    }
                }                   

		// Check for all RxD errors
		Status = RTMPCheckRxDescriptor(pRxD);

		if (Status == NDIS_STATUS_SUCCESS)
		{
			// Apply packet filtering rule based on microsoft requirements.
			Status = RTMPApplyPacketFilter(pAd, pRxD, pHeader);
		}	
		
		// Add receive counters
		if (Status != NDIS_STATUS_SUCCESS && pAd->PortCfg.BssType != BSS_MONITOR)
		{
			// Increase general counters
			pAd->Counters.RxErrors++;
		}
		
		// Check for retry bit, if this bit is on, search the cache with SA & sequence
		// as index, if matched, discard this frame, otherwise, update cache
		// This check only apply to unicast data & management frames
		if ((pRxD->U2M) && (Status == NDIS_STATUS_SUCCESS))
		{
			if (pHeader->Controlhead.Frame.Retry)
			{
				if (RTMPSearchTupleCache(pAd, pHeader) == TRUE)
				{
					// Found retry frame in tuple cache, Discard this frame / fragment
					// Increase 802.11 counters
					pAd->WlanCounters.FrameDuplicateCount++;
					Status = NDIS_STATUS_FAILURE;
				}
				else
				{
					RTMPUpdateTupleCache(pAd, pHeader);
				}
			}
			else	// Update Tuple Cache
			{
				RTMPUpdateTupleCache(pAd, pHeader);
			}

		}

		//
		// Do RxD release operation	for	all	failure	frames
		//
		if (Status == NDIS_STATUS_SUCCESS && pAd->PortCfg.BssType != BSS_MONITOR)
		{
			// Saved data pointer for management frame which will pass to MLME block
			pManage	= (PVOID) pData;

			// pData : Pointer skip	the	first 24 bytes,	802.11 HEADER
			pData += LENGTH_802_11;

			//
			// Start of	main loop to parse receiving frames.
			// The sequence	will be	Type first,	then subtype...
			//
			switch (pHeader->Controlhead.Frame.Type)
			{
				// Frame with data type
				case BTYPE_DATA:
					// Drop not my BSS frame
					if (INFRA_ON(pAd))
					{
						// Infrastructure mode, check address 2 for BSSID
						if (!RTMPEqualMemory(&pHeader->Controlhead.Addr2, &pAd->PortCfg.Bssid, 6))
						{
							// Receive frame not my BSSID
							break;
						}
					}
					else
					{
						// Ad-Hoc mode, check address 3 for BSSID
						if (!RTMPEqualMemory(&pHeader->Addr3, &pAd->PortCfg.Bssid, 6))
						{
							// Receive frame not my BSSID
							break;
						}
						
						// Drop frame from AP while we are in Ad-hoc mode
						if (pHeader->Controlhead.Frame.FrDs)
						{
							break;
						}
					}

					// Drop Null data frame, or CF with NULL data frame
					if ((pHeader->Controlhead.Frame.Subtype == SUBTYPE_NULL_FUNC) ||
						(pHeader->Controlhead.Frame.Subtype == SUBTYPE_CFACK)     ||
						(pHeader->Controlhead.Frame.Subtype == SUBTYPE_CFPOLL)    ||
						(pHeader->Controlhead.Frame.Subtype == SUBTYPE_CFACK_CFPOLL))
					{
						break;
					}

					// Process Multicast data frame
					if (pRxD->Mcast)
					{
						// Multicast 802.11 Counter
						pAd->WlanCounters.MulticastReceivedFrameCount++;
						DBGPRINT(RT_DEBUG_INFO,"Receiving multicast frame\n");
					}

					// Process Broadcast & Multicast data frame
					if (pRxD->Bcast || pRxD->Mcast)
					{
						// Drop Mcast / Bcast frame with fragment bit on
						if (pHeader->Controlhead.Frame.MoreFrag)
						{
							printk(KERN_WARNING DRV_NAME ": Receiving multicast frame with fragment bit on\n");
							Status = NDIS_STATUS_FAILURE;
							break;
						}	
							
						// Filter out Bcast frame which AP relayed for us
						if ((RTMPEqualMemory(&pHeader->Addr3, pAd->CurrentAddress, 6)) && pHeader->Controlhead.Frame.FrDs)
						{
							Status = NDIS_STATUS_FAILURE;
							break;
						}	
						
						// The total available payload should exclude 24-byte 802.11 Header
						PacketSize = pRxD->DataByteCnt - LENGTH_802_11;	

						// WEP encrypted frame
						if (pHeader->Controlhead.Frame.Wep)
						{
							// Check our WEP setting, if no WEP turning on, just drop this frame
							if (pAd->PortCfg.PrivacyInvoked)
							{
								if (RTMPDecryptData(pAd, pData, PacketSize) == FALSE)
								{
									Status = NDIS_STATUS_FAILURE;
#if WIRELESS_EXT >= 12
									pAd->iw_stats.discard.code++;
#endif
									break;
								}
								// Minus WEP size
								PacketSize -= 8;
							}
							else
							{
								// Add error counter
								Status = NDIS_STATUS_FAILURE;
#if WIRELESS_EXT >= 12
								pAd->iw_stats.discard.code++;
#endif						
								break;
							}
						}

						// DA is always	address	1
						pDestMac = (PUCHAR)	&(pHeader->Controlhead.Addr1);
						// Seclect SA by different mode
						if (INFRA_ON(pAd))
						{
							// For infrastructure, SA is address 3
							pSrcMac	= (PUCHAR) &(pHeader->Addr3);
						}
						else
						{
							// For IBSS	mode, SA is	address	2
							pSrcMac	= (PUCHAR) &(pHeader->Controlhead.Addr2);
						}

						// Save encapaturation starting pointer
						pEncap = pData;
						
						// Check for encapsulataion other than RFC1042 & Bridge tunnel
						if ((!RTMPEqualMemory(SNAP_802_1H, pEncap, 6)) && (!RTMPEqualMemory(SNAP_BRIDGE_TUNNEL, pEncap, 6)))
						{
							LLC_Len[0] = PacketSize / 256;
							LLC_Len[1] = PacketSize % 256;
							MAKE_802_3_HEADER(Header802_3, pDestMac, pSrcMac, ((PUCHAR) LLC_Len));
						}
						else
						{
							// Remove 802.11 H header & reconstruct 802.3 header
							pData += (LENGTH_802_1_H - LENGTH_802_3_TYPE);
							// Patch for WHQl only, which did not turn on Netbios but use IPX within its payload
							if ((RTMPEqualMemory(IPX, pData, 2) || RTMPEqualMemory(APPLE_TALK, pData, 2)) && RTMPEqualMemory(SNAP_802_1H, pEncap, 6))
							{
								LLC_Len[0] = PacketSize / 256;
								LLC_Len[1] = PacketSize % 256;
								pData = pData - LENGTH_802_1_H;
								MAKE_802_3_HEADER(Header802_3, pDestMac, pSrcMac, ((PUCHAR) LLC_Len));
							}
							else
							{
								MAKE_802_3_HEADER(Header802_3, pDestMac, pSrcMac, pData);
								// The total available payload should exclude 24-byte 802.11 Header
								// and 8-byte 802.2 LLC
								PacketSize -= LENGTH_802_1_H;
							}

							// Point to read 802.3 payload
							pData += LENGTH_802_3_TYPE;
						}
							
			            pAd->PortCfg.LedCntl.fRxActivity = TRUE; // for RX ACTIVITY LED

						PacketSize += LENGTH_802_3;
							
						// Acknolwdge upper	layer the received frame
						if ((skb = dev_alloc_skb(PacketSize + 2)) != NULL) {
							skb->dev = net_dev;
							skb_reserve(skb, 2);	// 16 byte align the IP header
							memcpy(skb_put(skb, LENGTH_802_3), Header802_3, LENGTH_802_3);
							memcpy(skb_put(skb, (PacketSize -LENGTH_802_3)), pData, (PacketSize -LENGTH_802_3));
							skb->protocol = eth_type_trans(skb, net_dev);
							netif_rx(skb);
							net_dev->last_rx = jiffies;
							pAd->RalinkCounters.ReceivedByteCount += PacketSize;
							pAd->WlanCounters.ReceivedFragmentCount++;
						}

						memset(Header802_3, 0, LENGTH_802_3);
						DBGPRINT(RT_DEBUG_INFO, "!!! Broadcast Ethenet rx Indicated !!!\n");
					}

					// Begin process unicast to	me frame
					else if	((pRxD->U2M) || (pAd->bAcceptPromiscuous == TRUE))
					{
						// Send PS-Poll for AP to send next data frame					
						if ((pHeader->Controlhead.Frame.MoreData) && INFRA_ON(pAd) && (pAd->PortCfg.Psm == PWR_SAVE))
						{
							//Send PS-Poll frame
							EnqueuePsPoll(pAd);
							DBGPRINT(RT_DEBUG_TRACE, "Sending PS-POLL\n");
						}
						
						//
						// Begin frame processing
						//
						// DA is always	address	1
						pDestMac = (PUCHAR)	&(pHeader->Controlhead.Addr1);
						// Seclect SA by different mode
						if (INFRA_ON(pAd))		// For infrastructure, SA is address 3
						{
							pSrcMac	= (PUCHAR) &(pHeader->Addr3);
						}
						else									// For IBSS	mode, SA is	address	2
						{
							pSrcMac	= (PUCHAR) &(pHeader->Controlhead.Addr2);
						}

						// The total available payload should exclude 24-byte 802.11 Header
						PacketSize = pRxD->DataByteCnt - LENGTH_802_11;

						// WEP encrypted frame
						if (pAd->PortCfg.PrivacyInvoked == TRUE)
						{
							if (pHeader->Controlhead.Frame.Wep)
							{
								if (RTMPDecryptData(pAd, pData, PacketSize) == FALSE)
								{
									Status = NDIS_STATUS_FAILURE;
#if WIRELESS_EXT >= 12
									pAd->iw_stats.discard.code++;
#endif
									break;
								}
								// Minus WEP size
								PacketSize -= 8;
							}
#if 0
							else if (pAd->PortCfg.ExcludeUnencrypt == FALSE)
							{
								// Add error counter
								Status = NDIS_STATUS_FAILURE;
#if WIRELESS_EXT >= 12
								pAd->iw_stats.discard.code++;
#endif
								break;
							}
							// Pass ExcludeUnencrypt == TRUE to next step
#endif
						}
						else if (pHeader->Controlhead.Frame.Wep)
						{
							// Drop WEP frame when PrivacyInvoked is FALSE
							Status = NDIS_STATUS_FAILURE;
#if WIRELESS_EXT >= 12
							pAd->iw_stats.discard.code++;
#endif
							break;
						}
						
						if (pHeader->Frag == 0)		// First or	Only fragment
						{
							// Save encapaturation starting pointer
							pEncap = pData;
						
							// Check for encapsulataion other than RFC1042 & Bridge tunnel
							if ((!RTMPEqualMemory(SNAP_802_1H, pEncap, 6)) && (!RTMPEqualMemory(SNAP_BRIDGE_TUNNEL, pEncap, 6)))
							{
								LLC_Len[0] = PacketSize / 256;
								LLC_Len[1] = PacketSize % 256;
								MAKE_802_3_HEADER(Header802_3, pDestMac, pSrcMac, ((PUCHAR) LLC_Len));
							}
							else
							{
							// Remove 802.11 H header &	reconstruct	802.3 header
							pData += (LENGTH_802_1_H - LENGTH_802_3_TYPE);
								// Patch for WHQl only, which did not turn on Netbios but use IPX within its payload
								if ((RTMPEqualMemory(IPX, pData, 2) || RTMPEqualMemory(APPLE_TALK, pData, 2)) && RTMPEqualMemory(SNAP_802_1H, pEncap, 6))
							{
								LLC_Len[0] = PacketSize / 256;
								LLC_Len[1] = PacketSize % 256;
								pData = pData - LENGTH_802_1_H;
									MAKE_802_3_HEADER(Header802_3, pDestMac, pSrcMac, ((PUCHAR) LLC_Len));
							}
							else
							{
									MAKE_802_3_HEADER(Header802_3, pDestMac, pSrcMac, pData);
								// The total available payload should exclude 24-byte 802.11 Header
								// and 8-byte 802.2	LLC
								PacketSize -= LENGTH_802_1_H;
							}	

							// Point to	read 802.3 payload
							pData += LENGTH_802_3_TYPE;
							}

							PacketSize += LENGTH_802_3;
							
							// One & The only fragment
							if (pHeader->Controlhead.Frame.MoreFrag	== FALSE)
							{
			                    pAd->PortCfg.LedCntl.fRxActivity = TRUE; // for RX ACTIVITY LED
								
								// Acknolwdge upper	layer the received frame
								if ((skb = dev_alloc_skb(PacketSize + 2)) != NULL) {
									skb->dev = net_dev;
									skb_reserve(skb, 2);	/* 16 byte align the IP header */
									memcpy(skb_put(skb, LENGTH_802_3), Header802_3, LENGTH_802_3);
									memcpy(skb_put(skb, (PacketSize - LENGTH_802_3)), pData, (PacketSize - LENGTH_802_3));
									skb->protocol = eth_type_trans(skb, net_dev);
									netif_rx(skb);
									net_dev->last_rx = jiffies;
									pAd->RalinkCounters.ReceivedByteCount += PacketSize;
									pAd->WlanCounters.ReceivedFragmentCount++; 	
								}								
								
								memset(Header802_3, 0, LENGTH_802_3);
								DBGPRINT(RT_DEBUG_INFO, "!!! Frame without Fragment Indicated !!!\n");
							}
							// First fragment of fragmented	frames
							else
							{
								memcpy(pAd->FragFrame.Buffer, pData, (PacketSize - LENGTH_802_3));
								memcpy(pAd->FragFrame.Header802_3, Header802_3, LENGTH_802_3);
								memset(Header802_3, 0, LENGTH_802_3);
								pAd->FragFrame.RxSize	= PacketSize;
								pAd->FragFrame.Sequence = pHeader->Sequence;
								pAd->FragFrame.LastFrag = pHeader->Frag;		// Should be 0
							}
						}
						// Middle &	End	of fragment	burst fragments
						else
						{
							// No LLC-SNAP header in except the first fragment frame
							
							if ((pHeader->Sequence != pAd->FragFrame.Sequence)	||
								(pHeader->Frag != (pAd->FragFrame.LastFrag	+ 1)))
							{
								// Fragment	is not the same	sequence or	out	of fragment	number order
								// Clear Fragment frame	contents
								memset(&pAd->FragFrame, 0, sizeof(FRAGMENT_FRAME));
								Status = NDIS_STATUS_FAILURE;
#if WIRELESS_EXT >= 12
								pAd->iw_stats.discard.fragment++;
#endif
								break;
							}
							else if ((pAd->FragFrame.RxSize + PacketSize) > MAX_FRAME_SIZE)
							{
								// Fragment	frame is too large,	it exeeds the maximum frame	size.
								// We have to drop it.
								// Clear Fragment frame	contents
								memset(&pAd->FragFrame, 0, sizeof(FRAGMENT_FRAME));
								Status = NDIS_STATUS_FAILURE;
#if WIRELESS_EXT >= 12
								pAd->iw_stats.discard.fragment++;
#endif
								break;
							}
							memcpy(&pAd->FragFrame.Buffer[pAd->FragFrame.RxSize - LENGTH_802_3], pData, PacketSize);
							pAd->FragFrame.RxSize  += PacketSize;
							pAd->FragFrame.LastFrag = pHeader->Frag;		// Update fragment number
							
							// Last	fragment
							if (pHeader->Controlhead.Frame.MoreFrag	== FALSE)
							{
			                    pAd->PortCfg.LedCntl.fRxActivity = TRUE; // for RX ACTIVITY LED

								// Acknolwdge upper	layer the received frame
								if ((skb = dev_alloc_skb(pAd->FragFrame.RxSize + 2)) != NULL) {
									skb->dev = net_dev;
									skb_reserve(skb, 2);	/* 16 byte align the IP header */
									memcpy(skb_put(skb, LENGTH_802_3), pAd->FragFrame.Header802_3, LENGTH_802_3);
									memcpy(skb_put(skb, (pAd->FragFrame.RxSize - LENGTH_802_3)), pAd->FragFrame.Buffer, (pAd->FragFrame.RxSize - LENGTH_802_3));
									skb->protocol = eth_type_trans(skb, net_dev);
									netif_rx(skb);
									net_dev->last_rx = jiffies;
									pAd->RalinkCounters.ReceivedByteCount += pAd->FragFrame.RxSize;
									pAd->WlanCounters.ReceivedFragmentCount++;
								}	
								
								// Clear Fragment frame	contents
								memset(&pAd->FragFrame, 0, sizeof(FRAGMENT_FRAME));
								DBGPRINT(RT_DEBUG_INFO, "!!! Frame with Fragment Indicated !!!\n");
							}
						}
					}
					break;

				case BTYPE_MGMT:
					// Read required regsiter for MLME engine
					RTMP_IO_READ32(pAd, CSR17,	&High32TSF);		// TSF value
					RTMP_IO_READ32(pAd, CSR16,	&Low32TSF);			// TSF vlaue
				
					// Enqueue this frame to MLME engine
					MlmeEnqueueForRecv(
						pAd,
						&pAd->Mlme.Queue,	
						High32TSF, 
						Low32TSF,
						(UCHAR)pRxD->BBR1, 
						pRxD->DataByteCnt, 
						pManage);
					
					break;

				case BTYPE_CNTL:
					// Ignore ???
					break;

				default	:
					break;
			}
		}
		pRxD->Owner = DESC_OWN_NIC;
		pAd->CurRxIndex++;
		if (pAd->CurRxIndex >= RX_RING_SIZE)
		{
			pAd->CurRxIndex = 0;
		}
		Count++;
		
	}	while (Count < MAX_RX_PROCESS);
	
	// Make sure to release Rx ring resource
	spin_unlock_irqrestore(&pAd->RxRingLock, irqflag);
}

/*
	========================================================================

	Routine	Description:
		Process	TxRing TxDone interrupt, running in	DPC	level

	Arguments:
		Adapter		Pointer	to our adapter

	Return Value:
		None

	Note:

	========================================================================
*/
VOID	RTMPHandleTxRingTxDoneInterrupt(
	IN	PRTMP_ADAPTER	pAd)
{
	PTXD_STRUC		pTxD;
	UCHAR			Count;
	unsigned long		irqflag;
	
	// Make sure Tx ring resource won't be used by other threads
	spin_lock_irqsave(&pAd->TxRingLock, irqflag);
	
	Count = 0;
	do
	{
		pTxD = (PTXD_STRUC)	(pAd->TxRing[pAd->NextTxDoneIndex].va_addr);
		// Check for the descriptor ownership
		if ((pTxD->Owner == DESC_OWN_NIC) || (pTxD->Valid == FALSE))
		{
			break;
		}
		
		RTMPHardTransmitDone(
			pAd, 
			pTxD,
			pAd->TxRing[pAd->NextTxDoneIndex].FrameType);
		
		// It might happend with no Ndis packet to indicate back to upper layer
		// Clear for NdisSendComplete request
		pTxD->Valid = FALSE;
		
		// Increase Total transmit byte counter after real data sent out
		pAd->RalinkCounters.TransmittedByteCount +=  pTxD->DataByteCnt;
		
		pAd->NextTxDoneIndex++;
		if (pAd->NextTxDoneIndex >= TX_RING_SIZE)
		{
			pAd->NextTxDoneIndex = 0;
		}
	}	while (++Count < MAX_TX_PROCESS);

	// Make sure to release Tx ring resource
	spin_unlock_irqrestore(&pAd->TxRingLock, irqflag);
	
	// Some Tx ring resource freed, check for pending send frame for hard transmit
	if ((!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_BSS_SCAN_IN_PROGRESS)) && 
		(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF)) &&
		(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS)))
	{
		RTMPDeQueuePacket(pAd);
	}
}

/*
	========================================================================

	Routine	Description:
		Process	Priority ring TxDone interrupt,	running	in DPC level

	Arguments:
		Adapter		Pointer	to our adapter

	Return Value:
		None

	Note:

	========================================================================
*/
VOID	RTMPHandlePrioRingTxDoneInterrupt(
	IN	PRTMP_ADAPTER	pAd)
{
	PTXD_STRUC		pTxD;
	UCHAR			Count;
	PMGMT_STRUC		pMgmt;
	unsigned long		irqflag;
	
	// Make sure Prio ring resource won't be used by other threads
	spin_lock_irqsave(&pAd->PrioRingLock, irqflag);	
	
	Count = 0;
	do
	{
		pTxD = (PTXD_STRUC)	(pAd->PrioRing[pAd->NextPrioDoneIndex].va_addr);
		// Check for the descriptor ownership
		if ((pTxD->Owner == DESC_OWN_NIC) || (pTxD->Valid == FALSE))
		{
			break;
		}
		
		// No need to put in reply for MLME
		RTMPHardTransmitDone(
			pAd, 
			pTxD,
			pAd->PrioRing[pAd->NextPrioDoneIndex].FrameType);
		
		// It might happend with no Ndis packet to indicate back to upper layer
		pTxD->Valid = FALSE;		
		
		// Increase Total transmit byte counter after real data sent out
		pAd->RalinkCounters.TransmittedByteCount +=  pTxD->DataByteCnt;

		pAd->NextPrioDoneIndex++;
		if (pAd->NextPrioDoneIndex >= PRIO_RING_SIZE)
		{
			pAd->NextPrioDoneIndex = 0;
		}
	}	while (++Count < MAX_TX_PROCESS);

	// Make sure to release Prio ring resource
	spin_unlock_irqrestore(&pAd->PrioRingLock, irqflag);	
	
	if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF))
		return;
	
	if (pAd->PushMgmtIndex != pAd->PopMgmtIndex)
	{
		if (RTMPFreeDescriptorRequest(pAd, PRIO_RING, 1) == NDIS_STATUS_SUCCESS)
		{
			pMgmt = (PMGMT_STRUC) &pAd->MgmtRing[pAd->PopMgmtIndex];
			if (pMgmt->Valid == TRUE)
			{
				MlmeHardTransmit(pAd, pMgmt->pBuffer, pMgmt->Length);
				MlmeConfirm((PVOID)pAd, pMgmt->pBuffer, 0);
				pMgmt->Valid = FALSE;
				pAd->PopMgmtIndex++;
				pAd->MgmtQueueSize--;
				if (pAd->PopMgmtIndex >= MGMT_RING_SIZE)
				{
					pAd->PopMgmtIndex = 0;
				}
			}
		}
	}
}

/*
	========================================================================

	Routine	Description:
		Process	Atim ring TxDone interrupt,	running	in DPC level

	Arguments:
		Adapter		Pointer	to our adapter

	Return Value:
		None

	Note:

	========================================================================
*/
VOID	RTMPHandleAtimRingTxDoneInterrupt(
	IN	PRTMP_ADAPTER	pAd)
{
	/* 
	unsigned long irqflag;
	// Make sure Atim ring resource won't be used by other threads
	spin_lock_irqsave(&pAd->AtimRingLock, irqflag);
	
	// Did not support ATIM, remove everything.
	
	// Make sure to release Atim ring resource
	spin_unlock_irqrestore(&pAd->AtimRingLock, irqflag);
	*/
}

/*
	========================================================================

	Routine	Description:
	Arguments:
		Adapter		Pointer	to our adapter
	========================================================================
*/
void    RTMPHandleTbcnInterrupt(IN PRTMP_ADAPTER pAd)
{
	if (ADHOC_ON(pAd))
    {
        MACHDR *pBcnHdr = (MACHDR *)pAd->BeaconRing.va_data_addr;

        // update BEACON frame's sequence number
        pAd->Sequence = ((pAd->Sequence) + 1) & (MAX_SEQ_NUMBER);
        pBcnHdr->Seq = pAd->Sequence;
    }
}

void    RTMPHandleTwakeupInterrupt(IN PRTMP_ADAPTER pAd)
{
    // DBGPRINT(RT_DEBUG_ERROR, ("Twakeup Expired... !!!\n"));
	pAd->PortCfg.Pss = PWR_ACTIVE;
}

/*
	========================================================================

	Routine	Description:
		Process	all transmit ring Tx Done interrupt, running	in DPC level

	Arguments:
		Adapter		Pointer	to our adapter

	Return Value:
		None

	Note:

	========================================================================
*/
VOID	RTMPHardTransmitDone(
	IN	PRTMP_ADAPTER	pAd,
	IN	PTXD_STRUC		pTxD,
	IN	UCHAR			FrameType)
{
	switch (pTxD->TxResult)
	{
		case SUCCESS_WITHOUT_RETRY:			// Success without any retry
			// Return send complete status
			DBGPRINT(RT_DEBUG_INFO, "One Packet Complete Success without retry<<<\n");
			if (pTxD->RTS)
			{
				// Increase 802.11 counters
				pAd->WlanCounters.RTSSuccessCount++;
				pTxD->RTS = 0;
			}
			
			// Increase general counters
			pAd->Counters.GoodTransmits++;
			break;
			  
		case SUCCESS_WITH_RETRY:			// Success with some retry
			// Return send complete status
			DBGPRINT(RT_DEBUG_ERROR, "One Packet Complete Success with retry<<<\n");
			
			// Increase 802.11 counters
			pAd->WlanCounters.RetryCount++;
			pAd->WlanCounters.ACKFailureCount++;
			
			// Increase general counters
			pAd->Counters.GoodTransmits++;
			
			if (pTxD->RetryCount > 1)
			{
				// Increase 802.11 counters
				pAd->WlanCounters.MultipleRetryCount++;
				
				// Increase general counters
				pAd->Counters.MoreCollisions++;
			}
			else
			{
				// Increase general counters
				pAd->Counters.OneCollision++;
			}
			
			if (pTxD->RTS)
			{
				pAd->WlanCounters.RTSSuccessCount++;
				pTxD->RTS = 0;
			}
			
			break;

		case FAIL_RETRY_LIMIT:				// Fail on hitting retry count limit
			DBGPRINT(RT_DEBUG_ERROR, "One Packet Complete Failed <<<\n");
			// Increase 802.11 counters
			pAd->WlanCounters.FailedCount++;
			pAd->WlanCounters.ACKFailureCount++;

#if WIRELESS_EXT >= 12
			pAd->iw_stats.discard.retries++;
#endif
			// Increase general counters
			pAd->Counters.TxErrors++;
			
			if (pTxD->RTS)
			{
				pAd->WlanCounters.RTSFailureCount++;
				pTxD->RTS = 0;
			}
			break;
			
		case FAIL_INVALID:
			DBGPRINT(RT_DEBUG_ERROR, "One Packet Complete Failed with invalid bit ???<<<\n");

#if WIRELESS_EXT >= 12
			pAd->iw_stats.discard.misc++;
#endif

			// Increase general counters
			pAd->Counters.TxErrors++;
			
			if (pTxD->RTS)
			{
				pAd->WlanCounters.RTSFailureCount++;
				pTxD->RTS = 0;
			}
			break;			
			
		case FAIL_OTHER:
		default:
			DBGPRINT(RT_DEBUG_ERROR, "One Packet Complete Failed other ???<<<\n");
			
			// Increase 802.11 counters
			pAd->WlanCounters.FailedCount++;
			pAd->WlanCounters.ACKFailureCount++;

#if WIRELESS_EXT >= 12
			pAd->iw_stats.discard.misc++;
#endif

			// Increase general counters
			pAd->Counters.TxErrors++;
			
			if (pTxD->RTS)
			{
				pAd->WlanCounters.RTSFailureCount++;
				pTxD->RTS = 0;
			}
			break;			
	}
}

/*
	========================================================================

	Routine	Description:
		API for MLME to transmit management frame to AP (BSS Mode)
	or station (IBSS Mode)
	
	Arguments:
		pAd			Pointer	to our adapter
		Buffer		Pointer to  memory of outgoing frame
		Length		Size of outgoing management frame
		
	Return Value:
		NDIS_STATUS_FAILURE
		NDIS_STATUS_PENDING
		NDIS_STATUS_SUCCESS

	Note:
	
	========================================================================
*/
NDIS_STATUS	MiniportMMRequest(
	IN	PRTMP_ADAPTER	pAd,
	IN	VOID			*pBuffer,
	IN	ULONG			Length)
{
	PMGMT_STRUC		pMgmt;
	NDIS_STATUS		Status = NDIS_STATUS_SUCCESS;    

	DBGPRINT(RT_DEBUG_INFO, "---> MiniportMMRequest\n");
	// Check management ring free avaliability
	pMgmt = (PMGMT_STRUC) &pAd->MgmtRing[pAd->PushMgmtIndex];
	
	// This management cell has been occupied
	if (pMgmt->Valid == TRUE)	
	{
		// No Management ring buffer avaliable
		if (pBuffer)
		{
			MlmeConfirm((PVOID)pAd, pBuffer, 0);
		}
		Status = NDIS_STATUS_FAILURE; 
		DBGPRINT(RT_DEBUG_WARN, "<--- MiniportMMRequest (error:: MgmtRing full)\n");
		pAd->RalinkCounters.MgmtRingFullCount++;
		return (Status);
	}
	
	// Insert this request into software managemnet ring
	if (pBuffer)
	{
		pMgmt->pBuffer = pBuffer;		
		pMgmt->Length  = Length;
		pMgmt->Valid   = TRUE;
		pAd->PushMgmtIndex++;
		pAd->MgmtQueueSize++;
		if (pAd->PushMgmtIndex >= MGMT_RING_SIZE)
		{
			pAd->PushMgmtIndex = 0;
		}
	}	
	else
	{
		// Null pBuffer, no need to free memory buffer.
		// This should not happen
		DBGPRINT(RT_DEBUG_WARN, "<--- MiniportMMRequest (error:: NULL msg)\n");
		Status = NDIS_STATUS_FAILURE; 
		return (Status);
	}
	
	if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF))
		return (Status);
	
	// Check Free priority queue
	if (RTMPFreeDescriptorRequest(pAd, PRIO_RING, 1) == NDIS_STATUS_SUCCESS)
	{
		pMgmt = (PMGMT_STRUC) &pAd->MgmtRing[pAd->PopMgmtIndex];
		if (pMgmt->Valid == TRUE)
		{
			MlmeHardTransmit(pAd, pMgmt->pBuffer, pMgmt->Length);
			MlmeConfirm((PVOID)pAd, pMgmt->pBuffer, 0);
			pMgmt->Valid = FALSE;
			pAd->PopMgmtIndex++;
			pAd->MgmtQueueSize--;
			if (pAd->PopMgmtIndex >= MGMT_RING_SIZE)
			{
				pAd->PopMgmtIndex = 0;
			}
		}
	}
	else
	{
		DBGPRINT(RT_DEBUG_INFO, "not enough space in PrioRing\n");
	}

	DBGPRINT(RT_DEBUG_INFO, "<--- MiniportMMRequest\n");
	return (Status);
}

/*
	========================================================================

	Routine	Description:
		Copy frame from waiting queue into relative ring buffer and set 
	appropriate ASIC register to kick hardware transmit function
	
	Arguments:
		pAd			Pointer	to our adapter
		pBuffer		Pointer to  memory of outgoing frame
		Length		Size of outgoing management frame
		
	Return Value:
		NDIS_STATUS_FAILURE
		NDIS_STATUS_PENDING
		NDIS_STATUS_SUCCESS

	Note:
	
	========================================================================
*/
VOID	MlmeHardTransmit(
	IN	PRTMP_ADAPTER	pAd,
	IN	VOID			*pBuffer,
	IN	ULONG			Length)
{
	PTXD_STRUC		pTxD;
	PUCHAR			pDest;	
	PHEADER_802_11		pHeader_802_11;
	BOOLEAN         	AckRequired, InsertTimestamp;
	unsigned long		irqflag;
	
	DBGPRINT(RT_DEBUG_TRACE, "MlmeHardTransmit\n");
    
	// Make sure Prio ring resource won't be used by other threads
	spin_lock_irqsave(&pAd->PrioRingLock, irqflag);
		
	pDest = (PUCHAR) pAd->PrioRing[pAd->CurPrioIndex].va_data_addr;              
	pTxD  = (PTXD_STRUC) pAd->PrioRing[pAd->CurPrioIndex].va_addr;
		
	if (pTxD->Owner == DESC_OWN_NIC)
	{
		// Descriptor owned by NIC. No descriptor avaliable
		// This should not happen since caller guaranteed.
		// Make sure to release Prio ring resource
		spin_unlock_irqrestore(&pAd->PrioRingLock, irqflag);
		return;
	}
	if (pTxD->Valid == TRUE)
	{
		// Ndis packet of last round did not cleared.
		// This should not happen since caller guaranteed.
		// Make sure to release Prio ring resource
		spin_unlock_irqrestore(&pAd->PrioRingLock, irqflag);
		return;
	}
	
    // outgoing frame always wakeup PHY to prevent frame lost 
    AsicForceWakeup(pAd);
    
    pHeader_802_11           = (PHEADER_802_11) pBuffer;
	//pHeader_802_11->Controlhead.Frame.PwrMgt = (pAd->PortCfg.Psm == PWR_SAVE);
	pHeader_802_11->Controlhead.Frame.PwrMgt = 0;
    InsertTimestamp = FALSE;
	if (pHeader_802_11->Controlhead.Frame.Type == BTYPE_CNTL) // must be PS-POLL
    {
        AckRequired = FALSE;
    	pAd->PrioRing[pAd->CurPrioIndex].FrameType = BTYPE_CNTL;
    }
	else // BTYPE_MGMT or BMGMT_DATA(must be NULL frame)
    {
    	pAd->PrioRing[pAd->CurPrioIndex].FrameType = BTYPE_MGMT;
        pAd->Sequence       = ((pAd->Sequence) + 1) & (MAX_SEQ_NUMBER);
        pHeader_802_11->Sequence = pAd->Sequence;

        if (pHeader_802_11->Controlhead.Addr1.Octet[0] & 0x01) // MULTICAST, BROADCAST
        {
            AckRequired = FALSE;
            pHeader_802_11->Controlhead.Duration = 0;
        }
        else
        {
            AckRequired = TRUE;
            pHeader_802_11->Controlhead.Duration = RTMPCalcDuration(pAd, RATE_2, 14);
            if (pHeader_802_11->Controlhead.Frame.Subtype == SUBTYPE_PROBE_RSP)
            {
                InsertTimestamp = TRUE;
            }
        }
    }
	
	memcpy(pDest, pBuffer, Length);
   
	// Initialize Priority Descriptor
	// For inter-frame gap, the number is for this frame and next frame
	// For MLME rate, we will fix as 2Mb to match other vendor's implement
	RTMPWriteTxDescriptor(pTxD, AckRequired, FALSE, InsertTimestamp, SHORT_RETRY, IFS_BACKOFF, RATE_2, 4, Length, pAd->PortCfg.TxPreamble);

	// Increase & maintain Tx Ring Index
	pAd->CurPrioIndex++;
	if (pAd->CurPrioIndex >= PRIO_RING_SIZE)
	{
		pAd->CurPrioIndex = 0;
	}
		
	// Kick priority ring transmit
	RTMP_IO_WRITE32(pAd,TXCSR0,0x4);
	
	// Make sure to release Prio ring resource
	spin_unlock_irqrestore(&pAd->PrioRingLock, irqflag);
}   
/*
	========================================================================

	Routine	Description:
		This routine is	used to	en-queue outgoing packets when
		there is no	enough shread memory
		
	Arguments:
		pAd			Pointer	to our adapter
		pPacket		Pointer to send packet
		
	Return Value:
		None

	Note:
	
	========================================================================
*/
NDIS_STATUS	RTMPSendPacket(
	IN	struct net_device *net_dev,
	IN	struct sk_buff *skb)
{
	PRTMP_ADAPTER	pAd = netdev_priv(net_dev);
	UINT			AllowFragSize;
	UCHAR			NumberOfFrag;
	UCHAR			RTSRequired;
	NDIS_STATUS		Status = NDIS_STATUS_FAILURE;
	UCHAR			PsMode;
	unsigned long		irqflag;

	DBGPRINT(RT_DEBUG_INFO, "<==== RTMPSendPacket\n");

	// Check for virtual address allocation, it might fail !!!
	if (skb->data == NULL)
	{
		// Resourece is low, system did not allocation virtual address
		// return NDIS_STATUS_FAILURE directly to upper layer
		return (Status);
	}

	//
	// Check for multicast or broadcast (First byte of DA)
	//
	if ((skb->data[0] & 0x01) != 0)
	{
		// For multicast & broadcast, there is no fragment allowed
		NumberOfFrag = 1;
	}
	else
	{
		// Check for payload allowed for each fragment
		AllowFragSize = (pAd->PortCfg.FragmentThreshold) - LENGTH_802_11 - LENGTH_CRC;

		// Calculate fragments required
		NumberOfFrag = ((skb->len - LENGTH_802_3 + LENGTH_802_1_H) / AllowFragSize) + 1;
		// Minus 1 if the size just match to allowable fragment size
		if (((skb->len - LENGTH_802_3 + LENGTH_802_1_H) % AllowFragSize) == 0)
		{
			NumberOfFrag--;
		}
	}

	// Check for requirement of RTS 
	if (NumberOfFrag > 1)
	{
		// If multiple fragment required, RTS is required only for the first fragment
		// if the fragment size large than RTS threshold
		RTSRequired = (pAd->PortCfg.FragmentThreshold > pAd->PortCfg.RtsThreshold) ? 1 : 0;
	}
	else
	{
		RTSRequired = (skb->len > pAd->PortCfg.RtsThreshold) ? 1 : 0;
	}
	DBGPRINT(RT_DEBUG_INFO, "Number of fragments include RTS :%d\n", NumberOfFrag + RTSRequired);

	// Save framnet number to Ndis packet reserved field
	RTMP_SET_PACKET_FRAGMENTS(skb, NumberOfFrag);

	// Save RTS requirement to Ndis packet reserved field
	RTMP_SET_PACKET_RTS(skb, RTSRequired);

	// Make sure SendTxWait queue resource won't be used by other threads
	spin_lock_irqsave(&pAd->SendTxWaitQueueLock, irqflag);
	
	//
	// For infrastructure mode, enqueue this frame immediately to sendwaitqueue
	// For Ad-hoc mode, check the DA power state, then decide which queue to enqueue
	//
	if (INFRA_ON(pAd))
	{
		// In infrastructure mode, simply enqueue the packet into Tx waiting queue.
		DBGPRINT(RT_DEBUG_INFO, "Infrastructure -> Enqueue one frame\n");

		// Enqueue Ndis packet to end of Tx wait queue
		skb_queue_tail(&pAd->SendTxWaitQueue, skb);
		Status = NDIS_STATUS_SUCCESS;
	}
	else
	{
		// In IBSS mode, power state of destination should be considered.
		// PsInquiry(pAd, pVirtualAddress, &PsMode, &KeyLen, &PeerKey, &Rate);
		PsMode = PWR_ACTIVE;		// Faked
		if (PsMode == PWR_ACTIVE)
		{
			DBGPRINT(RT_DEBUG_INFO,"Ad-Hoc -> Enqueue one frame\n");
			
			// Enqueue Ndis packet to end of Tx wait queue
			skb_queue_tail(&pAd->SendTxWaitQueue, skb);
			Status = NDIS_STATUS_SUCCESS;
		}
	}

	// Make sure SendTxWait queue resource won't be used by other threads
	spin_unlock_irqrestore(&pAd->SendTxWaitQueueLock, irqflag);
	return (Status);
}

/*
	========================================================================

	Routine	Description:
		To do the enqueue operation and extract the first item of waiting 
		list. If a number of available shared memory segments could meet 
		the request of extracted item, the extracted item will be fragmented
		into shared memory segments.
		
	Arguments:
		pAd			Pointer	to our adapter
		pQueue		Pointer to Waiting Queue
		
	Return Value:
		None

	Note:
	
	========================================================================
*/
VOID	RTMPDeQueuePacket(
    IN	PRTMP_ADAPTER	pAd)
{
	UCHAR			FragmentRequired;
	NDIS_STATUS		Status;
	UCHAR			Count = 0;
	struct sk_buff		*skb;
	unsigned long		irqflag;

	// Make sure SendTxWait queue resource won't be used by other threads
	spin_lock_irqsave(&pAd->SendTxWaitQueueLock, irqflag);
	
	// Check queue before dequeue
	while (!skb_queue_empty(&pAd->SendTxWaitQueue) && (Count < MAX_TX_PROCESS)) 
	{
		// Reset is in progress, stop immediately
		if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS))
		{
			break;
		}

		// Dequeue the first entry from head of queue list
		skb = skb_dequeue(&pAd->SendTxWaitQueue);

		// Total fragment required = number of fragment + RST if required
		FragmentRequired = RTMP_GET_PACKET_FRAGMENTS(skb) + RTMP_GET_PACKET_RTS(skb);
				
		if (RTMPFreeDescriptorRequest(pAd, TX_RING, FragmentRequired) == NDIS_STATUS_SUCCESS)
		{
			// Avaliable ring descriptors are enough for this frame
			// Call hard transmit
			Status = RTMPHardTransmit(pAd, skb, FragmentRequired);
			if (Status == NDIS_STATUS_FAILURE)
			{
				// Packet failed due to various Ndis Packet error
//				NdisMSendComplete(pAdapter->AdapterHandle, pPacket,	NDIS_STATUS_FAILURE);
				dev_kfree_skb_irq(skb);
				break;
			}
			else if (Status == NDIS_STATUS_RESOURCES)
			{
				// Not enough free tx ring, it might happen due to free descriptor inquery might be not correct
				// It also might change to NDIS_STATUS_FAILURE to simply drop the frame
				// Put the frame back into head of queue
				skb_queue_head(&pAd->SendTxWaitQueue, skb);
				break;
			}			
			Count++;
		}	
		else
		{
			skb_queue_head(&pAd->SendTxWaitQueue, skb);
			pAd->PrivateInfo.TxRingFullCnt++;
			DBGPRINT(RT_DEBUG_INFO,"RTMPDequeuePacket --> Not enough free Tx Ring descriptor !!!\n");
			break;
		}
	}

	// Release SendTxWaitQueue resources
	spin_unlock_irqrestore(&pAd->SendTxWaitQueueLock, irqflag);
}    

/*
	========================================================================

	Routine	Description:
		This subroutine will scan through releative ring descriptor to find
		out avaliable free ring descriptor and compare with request size.
		
	Arguments:
		pAd			Pointer	to our adapter
		RingType	Selected Ring
		
	Return Value:
		NDIS_STATUS_FAILURE		Not enough free descriptor
		NDIS_STATUS_SUCCESS		Enough free descriptor

	Note:
	
	========================================================================
*/
NDIS_STATUS	RTMPFreeDescriptorRequest(
	IN		PRTMP_ADAPTER	pAd,
	IN		UCHAR			RingType,
	IN		UCHAR			NumberRequired)
{
	UCHAR			FreeNumber = 0;
	UINT			Index;
	PTXD_STRUC		pTxD;
	NDIS_STATUS		Status = NDIS_STATUS_FAILURE;

	switch (RingType)
	{
		case TX_RING:
			spin_lock(&pAd->TxRingLock);
			Index = pAd->CurTxIndex;
			do
			{
				pTxD  = (PTXD_STRUC) pAd->TxRing[Index].va_addr;
				
				// While Owner bit is NIC, obviously ASIC still need it.
				// If valid bit is TRUE, indicate that TxDone has not process yet
				// We should not use it until TxDone finish cleanup job
				if ((pTxD->Owner == DESC_OWN_HOST) && (pTxD->Valid == FALSE))
				{
					// This one is free
					FreeNumber++;
				}
				else
				{
					break;
				}
					
				Index++;
				if (Index >= TX_RING_SIZE)				// Wrap around issue
				{
					Index = 0;
				}
				
			}	while (FreeNumber < NumberRequired);	// Quit here ! Free number is enough !
			
			if (FreeNumber >= NumberRequired)
			{
				Status = NDIS_STATUS_SUCCESS;
			}
			
			// Make sure to release Tx ring resource
			spin_unlock(&pAd->TxRingLock);
			break;
			
		case PRIO_RING:
			spin_lock(&pAd->PrioRingLock);
			Index = pAd->CurPrioIndex;
			do
			{
				pTxD  = (PTXD_STRUC) pAd->PrioRing[Index].va_addr;
				
				// While Owner bit is NIC, obviously ASIC still need it.
				// If valid bit is TRUE, indicate that TxDone has not process yet
				// We should not use it until TxDone finish cleanup job
				if ((pTxD->Owner == DESC_OWN_HOST) && (pTxD->Valid == FALSE))
				{
					// This one is free
					FreeNumber++;
				}
				else
				{
					break;
				}
					
				Index++;
				if (Index >= PRIO_RING_SIZE)			// Wrap around issue
				{
					Index = 0;
				}
				
			}	while (FreeNumber < NumberRequired);	// Quit here ! Free number is enough !
			
			if (FreeNumber >= NumberRequired)
			{
				Status = NDIS_STATUS_SUCCESS;
			}
			
			// Make sure to release Prio ring resource
			spin_unlock(&pAd->PrioRingLock);
			break;

		default:
			break;
	}
	
	return (Status);
}

VOID	RTMPSendNullFrame(
	IN	PRTMP_ADAPTER	pAd,
	IN	PVOID			pBuffer,
	IN	ULONG			Length)
{
	PUCHAR			pDest;
	PTXD_STRUC		pTxD;
	UCHAR			FrameGap;

	if (pBuffer == NULL)
	{
		return;
	}

	if (RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS))
	{
		MlmeConfirm((PVOID)pAd, pBuffer, 0);		
		return;
	}
	
	FrameGap = IFS_BACKOFF;		// Default frame gap mode

    // outgoing frame always wakeup PHY to prevent frame lost and 
    // turn off PSM bit to improve performance
    AsicForceWakeup(pAd);

	if (!skb_queue_empty(&pAd->SendTxWaitQueue))
	{
		DBGPRINT(RT_DEBUG_TRACE,"Drop Null frame due to Tx queue not empty!\n");
	}
	else
	{
		// Make sure Tx ring resource won't be used by other threads
		spin_lock(&pAd->TxRingLock);
	
		// Get the Tx Ring descriptor & Dma Buffer address
		pDest = (PUCHAR) pAd->TxRing[pAd->CurTxIndex].va_data_addr;
        pTxD  = (PTXD_STRUC) pAd->TxRing[pAd->CurTxIndex].va_addr;
		
		if ((pTxD->Owner == DESC_OWN_HOST) && (pTxD->Valid == FALSE))
		{
		    DBGPRINT(RT_DEBUG_TRACE, "SYNC - send NULL Frame ...\n");
			memcpy(pDest, pBuffer, Length);
			RTMPWriteTxDescriptor(pTxD, TRUE, FALSE, FALSE, SHORT_RETRY, FrameGap, pAd->PortCfg.TxRate, 4, Length, pAd->PortCfg.TxPreamble);
			// Increase & maintain Tx Ring Index
			pAd->CurTxIndex++;
			if (pAd->CurTxIndex >= TX_RING_SIZE)
			{
				pAd->CurTxIndex = 0;
			}
		}
		spin_unlock(&pAd->TxRingLock);
	}
	MlmeConfirm((PVOID)pAd, pBuffer, 0);		
	
	// Kick Tx Control Register at the end of all ring buffer preparation
	RTMP_IO_WRITE32(pAd, TXCSR0, 0x1);	
}

/*
	========================================================================

	Routine	Description:
		Copy frame from waiting queue into relative ring buffer and set 
	appropriate ASIC register to kick hardware transmit function
		
	Arguments:
		pAd				Pointer	to our adapter
		PNDIS_PACKET	Pointer to outgoing Ndis frame
		NumberOfFrag	Number of fragment required
		
	Return Value:
		None

	Note:
	
	========================================================================
*/
NDIS_STATUS	RTMPHardTransmit(
	IN	PRTMP_ADAPTER		pAd,
	IN	struct sk_buff		*skb,
	IN	UCHAR				NumberRequired)
{
	UINT			Length;
	UINT			BytesCopied;
	UINT			TxSize;
	UINT			FreeFragSize;
	UINT			RemainSize;
	UCHAR			FrameGap;
	HEADER_802_11	Header_802_11;
	UCHAR			*pDest;
	UCHAR			*pSrc;
	PTXD_STRUC		pTxD;
	PUCHAR			pEncap;
	BOOLEAN			StartOfFrame;
	BOOLEAN			EAPOLFrame;
	BOOLEAN			Encapped;
	USHORT			Protocol;
	unsigned long		irqflag;

	// Make sure Tx ring resource won't be used by other threads
	spin_lock_irqsave(&pAd->TxRingLock, irqflag);
	
	FrameGap = IFS_BACKOFF;		// Default frame gap mode
	Length = skb->len;

    // outgoing frame always wakeup PHY to prevent frame lost and 
    // turn off PSM bit to improve performance
    if (pAd->PortCfg.Psm == PWR_SAVE)
    {
        MlmeSetPsmBit(pAd, PWR_ACTIVE);
    }
    AsicForceWakeup(pAd);

    // Sequence Number is identical for all fragments belonged to the same frame
    // Sequence is 0 - 4095
    pAd->Sequence = ((pAd->Sequence) + 1) & (MAX_SEQ_NUMBER);
    pAd->TxRing[pAd->CurTxIndex].FrameType = BTYPE_DATA;

	// Initialize 802.11 header for each frame
	memset(&Header_802_11, 0,sizeof(HEADER_802_11));
	
	if ((*(skb->data) & 0x01) != 0)	// Multicast or Broadcast
	{
		pAd->WlanCounters.MulticastTransmittedFrameCount++;
	}

	if (Length < 14)
	{
		printk(KERN_ERR DRV_NAME ": RTMPHardTransmit: Ndis Packet buffer error!\n");
		
		// Make sure to release Tx ring resource
		spin_unlock_irqrestore(&pAd->TxRingLock, irqflag);
		return (NDIS_STATUS_FAILURE);
	}

	//
	// Start making 802.11 frame header
	//
	if (INFRA_ON(pAd))
	{
        // In BSS mode, AP's address(BSSID) is the destination address of all outgoing packets
        // Address 1 - BSSID
        memcpy(&Header_802_11.Controlhead.Addr1, &pAd->PortCfg.Bssid, MAC_ADDR_LEN);
        // Address 3 - DA
        memcpy(&Header_802_11.Addr3, skb->data, MAC_ADDR_LEN);
        Header_802_11.Controlhead.Frame.ToDs = 1;
    }
    else 
    {
		// Address 1 - DA
		memcpy(&Header_802_11.Controlhead.Addr1, skb->data, MAC_ADDR_LEN);
		// Address 3 - BSSID
		memcpy(&Header_802_11.Addr3, &pAd->PortCfg.Bssid, MAC_ADDR_LEN);
    }

    // Address 2 - SA in both infrastructure & ad-hoc modes
    memcpy(&Header_802_11.Controlhead.Addr2, skb->data+MAC_ADDR_LEN, MAC_ADDR_LEN);	
	Header_802_11.Sequence = pAd->Sequence;		// Sequence number
	Header_802_11.Controlhead.Frame.Type = BTYPE_DATA;	// Frame type
	Header_802_11.Controlhead.Frame.PwrMgt = (pAd->PortCfg.Psm == PWR_SAVE);

	if (Header_802_11.Controlhead.Addr1.Octet[0] & 0x01)
	{
		// No ACK expected for multicast frame		
		Header_802_11.Controlhead.Duration = 0;
	}
	else
	{
		// ACK size is 14 include CRC, and its rate is 2Mb
		Header_802_11.Controlhead.Duration = pAd->PortCfg.Dsifs 
			+ RTMPCalcDuration(pAd, pAd->PortCfg.MaxBasicRate, 14);
	}
    	   
    // For the purpose to calculate duration for the second last fragment
	RemainSize = skb->len - LENGTH_802_3 + LENGTH_CRC;

	// Add 802.11x protocol check.
	// For non-WPA network, 802.11x message should not encrypt even
	// the privacy is on.
	if (RTMPEqualMemory(EAPOL, ((PUCHAR) skb->data) + 12, 2))
	{
		EAPOLFrame = TRUE;
	}
	else
	{
		EAPOLFrame = FALSE;
	}
	
	Encapped = FALSE;
	pEncap = NULL;

	pSrc = (UCHAR*) skb->data;

	Protocol = *(pSrc + 12) * 256 + *(pSrc + 13);
	if (Protocol > 1500)	// CHeck for LLC encaped
	{
		INT	Offset;
		
		if (Length <= 22)
		{
			Offset = 20 - Length;
			pSrc = (PUCHAR) skb->data + Offset;
		}
		else
		{
			pSrc = pSrc + 20;
		}

		pEncap = SNAP_802_1H;
		Encapped = TRUE;
		if (RTMPEqualMemory(IPX, pSrc, 2))
		{
			pEncap = SNAP_BRIDGE_TUNNEL;
		}
		
		if (RTMPEqualMemory(APPLE_TALK, pSrc, 2))
		{
			pEncap = SNAP_BRIDGE_TUNNEL;
		}
	}

	//
	// Make RTS frame if required
	//
	if (RTMP_GET_PACKET_RTS(skb))
	{
		PCONTROL_HEADER		pControlHeader;
		UINT				RTSFrameSize;
		
		DBGPRINT(RT_DEBUG_TRACE,"Making RTS Frame\n");
		
		pDest = (PUCHAR) pAd->TxRing[pAd->CurTxIndex].va_data_addr;              
		pTxD  = (PTXD_STRUC) pAd->TxRing[pAd->CurTxIndex].va_addr;
		pTxD->pSk_Buff= NULL;
        		
		if (pTxD->Owner == DESC_OWN_NIC)
		{
			// Descriptor owned by NIC. No descriptor avaliable
			// This should not happen since caller guaranteed.
			// Make sure to release Tx ring resource
			spin_unlock_irqrestore(&pAd->TxRingLock, irqflag);
			return (NDIS_STATUS_RESOURCES);
		}
		if (pTxD->Valid == TRUE)
		{
			// This might happen when HardTransmit process faster than TxDone interrupt.
			// However, Since we did call free descriptor number check before Tx HardTransmit.
			// This case should not happened. We should return to higher caller and requeue this
			// acket until next Tx hardtransmit. Hopefully this situation is corrected then.
			// Ndis packet of last round did not cleared.
			// This should not happen since caller guaranteed.
			// Make sure to release Tx ring resource
			pTxD->Valid = FALSE;
				
			spin_unlock_irqrestore(&pAd->TxRingLock, irqflag);
			return (NDIS_STATUS_RESOURCES);
		}

		pAd->TxRing[pAd->CurTxIndex].FrameType = BTYPE_CNTL;
		pControlHeader = (PCONTROL_HEADER) pDest;
		memset(pControlHeader, 0, sizeof(CONTROL_HEADER));

		//FrameControl
		pControlHeader->Frame.Type    = BTYPE_CNTL;
		pControlHeader->Frame.Subtype = SUBTYPE_RTS;
	
		// Calculate duration
		// 2 SIFS + CTS + Data Frame size
		if (RTMP_GET_PACKET_FRAGMENTS(skb) > 1)
		{
        	// If fragment required, size is maximum fragment size
        	RTSFrameSize = pAd->PortCfg.FragmentThreshold;
		}
		else
		{
			// Size should be frame with 802.11 header & CRC
			RTSFrameSize = skb->len + LENGTH_802_11 + LENGTH_802_1_H + LENGTH_CRC - LENGTH_802_3;

			// Decrease the LLC header size if it's already encapped from protocol layer
			if (Encapped == FALSE)
			{
				RTSFrameSize -= LENGTH_802_11;
			}

			// Check our WEP setting, if WEP is on, Next frame size should include WEP
			if ((pAd->PortCfg.PrivacyInvoked) && 
				(pAd->PortCfg.SharedKey[pAd->PortCfg.DefaultKeyId].KeyLen != 0))
			{
				RTSFrameSize += 8;
			}
		}
		pControlHeader->Duration = 2 * (pAd->PortCfg.Dsifs)
			+ RTMPCalcDuration(pAd, pAd->PortCfg.TxRate, RTSFrameSize)
			+ RTMPCalcDuration(pAd, pAd->PortCfg.MaxBasicRate, sizeof(CONTROL_HEADER)); 

		// RA
		if (INFRA_ON(pAd))
			memcpy(&pControlHeader->Addr1, &pAd->PortCfg.Bssid, MAC_ADDR_LEN);
		else
			memcpy(&pControlHeader->Addr1, skb->data, MAC_ADDR_LEN);
		// TA	
		memcpy(&pControlHeader->Addr2, pAd->CurrentAddress, MAC_ADDR_LEN);

		// Write Tx descriptor
		// Don't kick tx start until all frames are prepared
		// RTS has to set more fragment bit for fragment burst
		RTMPWriteTxDescriptor(pTxD, TRUE, TRUE, FALSE, SHORT_RETRY, FrameGap, pAd->PortCfg.MaxBasicRate, 4, sizeof(CONTROL_HEADER), pAd->PortCfg.TxPreamble);
		pTxD->RTS = 1;
		
        FrameGap = IFS_SIFS;		// Init frame gap for coming data after RTS
		NumberRequired--;
		
		// Increase & maintain Tx Ring Index
		pAd->CurTxIndex++;
		if (pAd->CurTxIndex >= TX_RING_SIZE)
			pAd->CurTxIndex = 0;
	}

	StartOfFrame = TRUE;
	// Start Copy Ndis Packet into Ring buffer.
	// For frame required more than one ring buffer (fragment), all ring buffers
	// have to be filled before kicking start tx bit.
	do
	{
		// Get the Tx Ring descriptor & Dma Buffer address
		pDest = (UCHAR*) pAd->TxRing[pAd->CurTxIndex].va_data_addr;              
        pTxD  = (PTXD_STRUC) pAd->TxRing[pAd->CurTxIndex].va_addr;
		// Maximum allowable payload with one ring buffer, bound by fragment size
		FreeFragSize = pAd->PortCfg.FragmentThreshold - LENGTH_CRC;
		
		if (pTxD->Owner == DESC_OWN_NIC)
		{
			// Descriptor owned by NIC. No descriptor avaliable
			// This should not happen since caller guaranteed.
			// Make sure to release Tx ring resource
			
			spin_unlock_irqrestore(&pAd->TxRingLock, irqflag);
			return (NDIS_STATUS_RESOURCES);
		}
		if (pTxD->Valid == TRUE)
		{
			// Ndis packet of last round did not cleared.
			// This should not happen since caller guaranteed.
			// Make sure to release Tx ring resource

			pTxD->Valid = FALSE;
			spin_unlock_irqrestore(&pAd->TxRingLock, irqflag);
			return (NDIS_STATUS_RESOURCES);
		}

		// Make fragment number & more fragment bit of 802.11 header
		if (StartOfFrame == TRUE)
		{
			Header_802_11.Frag = 0;			// Start of fragment burst / Single Frame
		}
		else
		{
			Header_802_11.Frag++;			// Rest of fragmented frames.
		}
		
		// Turn on with no frames after this one
		if (NumberRequired > 1)
		{
			Header_802_11.Controlhead.Frame.MoreFrag = 1;
			// Duration should include next fragment
			Header_802_11.Controlhead.Duration = 2 * pAd->PortCfg.Dsifs
				+ RTMPCalcDuration(pAd, pAd->PortCfg.MaxBasicRate, 14)		
				+ RTMPCalcDuration(pAd, pAd->PortCfg.MaxTxRate, pAd->PortCfg.FragmentThreshold);
		}
		else
		{
			Header_802_11.Controlhead.Frame.MoreFrag = 0;
			// ACK size is 14 include CRC, and its rate is 2Mb
			Header_802_11.Controlhead.Duration = pAd->PortCfg.Dsifs 
				+ RTMPCalcDuration(pAd, pAd->PortCfg.MaxBasicRate, 14);
			// Save Ndis packet at the last fragment only
			// This will prevent Tx Ring done interrupt free this frame in advance
        	pTxD->pSk_Buff= skb;
		}

		if (NumberRequired == 2)
		{
			RemainSize = RemainSize - pAd->PortCfg.FragmentThreshold + LENGTH_802_11 + LENGTH_802_11 + LENGTH_CRC;
			if ((pAd->PortCfg.PrivacyInvoked) && (EAPOLFrame == FALSE) &&
				(pAd->PortCfg.SharedKey[pAd->PortCfg.DefaultKeyId].KeyLen != 0))
			{
				Header_802_11.Controlhead.Duration = 3 * pAd->PortCfg.Dsifs
					+ 2 * RTMPCalcDuration(pAd, pAd->PortCfg.MaxBasicRate, 14)		
					+ RTMPCalcDuration(pAd, pAd->PortCfg.MaxTxRate, RemainSize  + 8);
			}
			else
			{
				Header_802_11.Controlhead.Duration = 3 * pAd->PortCfg.Dsifs
					+ 2 * RTMPCalcDuration(pAd, pAd->PortCfg.MaxBasicRate, 14)		
					+ RTMPCalcDuration(pAd, pAd->PortCfg.MaxTxRate, RemainSize);
			}
		}

		// Check for WEP enable bit and prepare for software WEP
		if ((pAd->PortCfg.PrivacyInvoked) && (EAPOLFrame == FALSE) &&
			(pAd->PortCfg.SharedKey[pAd->PortCfg.DefaultKeyId].KeyLen != 0))
		{
			Header_802_11.Controlhead.Frame.Wep = 1;
		}
		
		// Copy 802.11 header to Tx ring buffer
		memcpy(pDest, &Header_802_11, sizeof(Header_802_11));
		pDest        += sizeof(Header_802_11);
		FreeFragSize -= sizeof(Header_802_11);

		if ((pAd->PortCfg.PrivacyInvoked) && (EAPOLFrame == FALSE) &&
			(pAd->PortCfg.SharedKey[pAd->PortCfg.DefaultKeyId].KeyLen != 0))
		{
			RTMPInitWepEngine(
				pAd,
				pAd->PortCfg.SharedKey[pAd->PortCfg.DefaultKeyId].Key,
				pAd->PortCfg.DefaultKeyId,
				pAd->PortCfg.SharedKey[pAd->PortCfg.DefaultKeyId].KeyLen,
				pDest);
			// Added WEP header
			pDest += 4;
		}
		//
		// Only the first fragment required LLC-SNAP header !!!
		//
		if (StartOfFrame == TRUE)
		{
			if ((pAd->PortCfg.PrivacyInvoked) && (EAPOLFrame == FALSE) &&
				(pAd->PortCfg.SharedKey[pAd->PortCfg.DefaultKeyId].KeyLen != 0))
				{
				// Copy LLC header
				RTMPEncryptData(pAd, pEncap, pDest, 6);
				pDest += 6;
				// Copy protocol type
				pSrc = (PUCHAR) skb->data;
				RTMPEncryptData(pAd, pSrc + 12, pDest, 2);
				pDest += 2;
				// Exclude 802.3 header size, we will recalculate the size at
				// the end of fragment preparation.
				Length -= 14;
				pSrc += 14;
				FreeFragSize -= LENGTH_802_1_H;
			}
			else
			{
				// Copy LLC header
				memcpy(pDest, pEncap, 6);
				pDest += 6;
				// Copy protocol type
				pSrc = (PUCHAR) skb->data;
				memcpy(pDest, pSrc + 12, 2);
				pDest += 2;
				// Exclude 802.3 header size, we will recalculate the size at
				// the end of fragment preparation.
				Length -= 14;
				pSrc += 14;
				FreeFragSize -= LENGTH_802_1_H;
			}
		}

		// Start copying payload
		BytesCopied = 0;
		do
		{
			if (Length >= FreeFragSize)
			{
				// Copy only the free fragment size, and save the pointer
				// of current buffer descriptor for next fragment buffer.
				if ((pAd->PortCfg.PrivacyInvoked) && (EAPOLFrame == FALSE) &&
					(pAd->PortCfg.SharedKey[pAd->PortCfg.DefaultKeyId].KeyLen != 0))
				{
					RTMPEncryptData(pAd, pSrc, pDest, FreeFragSize);
				}
				else
				{
					memcpy(pDest, pSrc, FreeFragSize);
				}
				BytesCopied += FreeFragSize;
				pSrc        += FreeFragSize;
				pDest       += FreeFragSize;
				Length      -= FreeFragSize;
				break;
			}
			else
			{
				// Copy the rest of this buffer descriptor pointed data
				// into ring buffer.
				if ((pAd->PortCfg.PrivacyInvoked) && (EAPOLFrame == FALSE) &&
					(pAd->PortCfg.SharedKey[pAd->PortCfg.DefaultKeyId].KeyLen != 0))
				{
					RTMPEncryptData(pAd, pSrc, pDest, Length);
				}
				else
				{
					memcpy(pDest, pSrc, Length);
				}
				BytesCopied  += Length;
				pDest        += Length;
				FreeFragSize -= Length;
			}
			
		}	while (FALSE);		// End of copying payload
				
		// Real packet size, No 802.1H header for fragments except the first one.
		if (StartOfFrame == TRUE)
			TxSize = BytesCopied + LENGTH_802_11 + LENGTH_802_1_H;
		else
			TxSize = BytesCopied + LENGTH_802_11;
		
		if ((pAd->PortCfg.PrivacyInvoked) && (EAPOLFrame == FALSE) &&
					(pAd->PortCfg.SharedKey[pAd->PortCfg.DefaultKeyId].KeyLen != 0))
		{
			// Append ICV
			RTMPSetICV(pAd, pDest);
			TxSize += 8;		// WEP header & trailer
		}
			
		// Prepare Tx descriptors before kicking tx.
		// The BBP register index in Tx descriptor has to be configured too.
		if (Header_802_11.Controlhead.Addr1.Octet[0] & 0x01)
		{
			// Multicast, retry bit is off
			RTMPWriteTxDescriptor(pTxD, FALSE, FALSE, FALSE, SHORT_RETRY, FrameGap, pAd->PortCfg.TxRate, 4, TxSize, pAd->PortCfg.TxPreamble);
		}
		else
		{
			RTMPWriteTxDescriptor(pTxD, TRUE, FALSE, FALSE, SHORT_RETRY, FrameGap, pAd->PortCfg.TxRate, 4, TxSize, pAd->PortCfg.TxPreamble);
		}

		// Set frame gap for the rest of fragment burst.
		// It won't matter if there is only one fragment (single fragment frame).
		StartOfFrame = FALSE;
		FrameGap     = IFS_SIFS;
		NumberRequired--;
		
		// Increase & maintain Tx Ring Index
		pAd->CurTxIndex++;
		if (pAd->CurTxIndex >= TX_RING_SIZE)
			pAd->CurTxIndex = 0;
		
	}	while (NumberRequired > 0);

        // Acknowledge protocol send complete of pending packet.
	// NdisMSendComplete(pAdapter->AdapterHandle, pPacket, NDIS_STATUS_SUCCESS);
	dev_kfree_skb_irq(skb);

	// Kick Tx Control Register at the end of all ring buffer preparation
	RTMP_IO_WRITE32(pAd, TXCSR0, 0x1);
        pAd->WlanCounters.TransmittedFragmentCount++;
	
	spin_unlock_irqrestore(&pAd->TxRingLock, irqflag);

	return (NDIS_STATUS_SUCCESS);
}

/*
	========================================================================

	Routine	Description:
		Calculates the duration which is required to transmit out frames 
	with given size and specified rate.
		
	Arguments:
		pAd				Pointer	to our adapter
		Rate			Transmit rate
		Size			Frame size in units of byte
		
	Return Value:
		Duration number in units of usec

	Note:
	
	========================================================================
*/
USHORT	RTMPCalcDuration(
	IN	PRTMP_ADAPTER	pAd,
	IN	UCHAR			Rate,
	IN	ULONG			Size)
{
	ULONG	Duration = 0;

	Duration = pAd->PortCfg.Dplcp + pAd->PortCfg.Dpreamble;
	switch (Rate)
	{
		case RATE_1:
			Duration += (Size * 8);
			break;
			
		case RATE_2:
			Duration += (Size * 4);
			break;
			
		case RATE_5_5:
			Duration += (USHORT) (Size * 16 / 11);
			if (((Size * 16) % 11) != 0)
			{
				Duration++;
			}
			break;
			
		case RATE_11:
			Duration += (USHORT) (Size * 8 / 11);
			if (((Size * 8) % 11) != 0)
			{
				Duration++;
			}
			break;
	}
	return ((USHORT)Duration);
}

/*
	========================================================================

	Routine	Description:
		Calculates the duration which is required to transmit out frames 
	with given size and specified rate.
		
	Arguments:
		pTxD		Pointer to transmit descriptor
		Ack			Setting for Ack requirement bit
		Fragment	Setting for Fragment bit
		RetryMode	Setting for retry mode
		Ifs			Setting for IFS gap
		Rate		Setting for transmit rate
		Service		Setting for service
		Length		Frame length
		
	Return Value:
		None
	========================================================================
*/
VOID	RTMPWriteTxDescriptor(
	IN	PTXD_STRUC	pTxD,
	IN	BOOLEAN		Ack,
	IN	BOOLEAN		Fragment,
	IN	BOOLEAN     InsTimestamp,
	IN	UCHAR		RetryMode,
	IN	UCHAR		Ifs,
	IN	UINT		Rate,
	IN	UCHAR		Service,
	IN	UINT		Length,
	IN  USHORT      TxPreamble)
{
	UINT	Residual;
	
	pTxD->Owner       = DESC_OWN_NIC;
	pTxD->Valid       = TRUE;
	pTxD->MoreFrag    = Fragment;
	pTxD->ACK         = Ack;
	pTxD->Timestamp   = InsTimestamp;
	pTxD->RetryMd     = RetryMode;
	pTxD->IFS         = Ifs;
	pTxD->DataByteCnt = Length;
	
	pTxD->BBCR0 = 0x8500 + (Rate);
	if ((Rate != RATE_1) && (TxPreamble == Rt802_11PreambleShort)) // no short preamble for RATE_1
	{
    	pTxD->BBCR0 |= 0x0008;
	}
	pTxD->BBCR1 = 0x8600 + (Service);
	Length += 4;

	if ((Rate == RATE_1) || ( Rate == RATE_2))
	{
		Length = Length * 8 / (Rate + 1);
	}
	else
	{
		Residual = ((Length * 16) % (11 * (1 + Rate - RATE_5_5)));
		Length = Length * 16 / (11 * (1 + Rate - RATE_5_5));
		if (Residual != 0)
		{
			Length++;
		}
		if ((Residual <= (3 * (1 + Rate - RATE_5_5))) && (Residual != 0))
		{
			pTxD->BBCR1 = 0x8684;
		}
	}
	pTxD->BBCR2 = 0x8700 + (Length / 256);
	pTxD->BBCR3 = 0x8800 + (Length % 256);
}

/*
	========================================================================

	Routine	Description:
		Search tuple cache for receive duplicate frame from unicast frames.
		
	Arguments:
		pAd				Pointer	to our adapter
		pHeader			802.11 header of receiving frame
		
	Return Value:
		TRUE			found matched tuple cache
		FALSE			no matched found

	Note:
	
	========================================================================
*/
BOOLEAN	RTMPSearchTupleCache(
	IN	PRTMP_ADAPTER	pAd,
	IN	PHEADER_802_11	pHeader)
{
	INT	Index;

	for (Index = 0; Index < MAX_CLIENT; Index++)
	{
		// Tuple cache filled sequencially. If not valid, the rest following this entry
		// are invalid too.
		if (pAd->TupleCache[Index].Valid == FALSE)
		{
			return (FALSE);
		}
		
		if (RTMPEqualMemory(&pAd->TupleCache[Index].MAC, &pHeader->Controlhead.Addr2, 6) &&
			(pAd->TupleCache[Index].Sequence == pHeader->Sequence) &&
			(pAd->TupleCache[Index].Frag == pHeader->Frag))
		{
			return (TRUE);
		}
	}
	return (FALSE);
}

/*
	========================================================================

	Routine	Description:
		Update tuple cache for new received unicast frames.
		
	Arguments:
		pAd				Pointer	to our adapter
		pHeader			802.11 header of receiving frame
		
	Return Value:
		None
		
	Note:
	
	========================================================================
*/
VOID	RTMPUpdateTupleCache(
	IN	PRTMP_ADAPTER	pAd,
	IN	PHEADER_802_11	pHeader)
{
	INT	Index;

	for (Index = 0; Index < MAX_CLIENT; Index++)
	{
		if (pAd->TupleCache[Index].Valid == FALSE)
		{
			// Add new entry
			memcpy(&pAd->TupleCache[Index].MAC, &pHeader->Controlhead.Addr2, 6);
			pAd->TupleCache[Index].Sequence = pHeader->Sequence;
			pAd->TupleCache[Index].Frag     = pHeader->Frag;
			pAd->TupleCache[Index].Valid    = TRUE;
			break;
		}
		if (RTMPEqualMemory(&pAd->TupleCache[Index].MAC, &pHeader->Controlhead.Addr2, 6))
		{
			// Update old entry
			pAd->TupleCache[Index].Sequence = pHeader->Sequence;
			pAd->TupleCache[Index].Frag     = pHeader->Frag;
		}
	}
}

/*
	========================================================================

	Routine	Description:
		Suspend MSDU transmission
		
	Arguments:
		pAd			Pointer	to our adapter
		
	Return Value:
		None
		
	Note:
	
	========================================================================
*/
VOID    RTMPSuspendMsduTransmission(
   	IN	PRTMP_ADAPTER	pAd)
{
    DBGPRINT(RT_DEBUG_TRACE,"SCANNING, suspend MSDU transmission ...\n");
    RTMP_SET_FLAG(pAd, fRTMP_ADAPTER_BSS_SCAN_IN_PROGRESS);
}

/*
	========================================================================

	Routine	Description:
		Resume MSDU transmission
		
	Arguments:
		pAd			Pointer	to our adapter
		
	Return Value:
		None
		
	Note:
	
	========================================================================
*/
VOID    RTMPResumeMsduTransmission(
   	IN	PRTMP_ADAPTER	pAd)
{
	DBGPRINT(RT_DEBUG_TRACE,"SCAN done, resume MSDU transmission ...\n");
    RTMP_CLEAR_FLAG(pAd, fRTMP_ADAPTER_BSS_SCAN_IN_PROGRESS);

    // Dequeue Tx queue if Reset is not in progress
	if ((!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RESET_IN_PROGRESS)) &&
		(!RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF)))
	{
		RTMPDeQueuePacket(pAd);
	}
}

/*
	========================================================================

	Routine	Description:
		Check Rx descriptor, return NDIS_STATUS_FAILURE if any error dound
		
	Arguments:
		pRxD		Pointer	to the Rx descriptor
		
	Return Value:
		NDIS_STATUS_SUCCESS		No err
		NDIS_STATUS_FAILURE		Error
		
	Note:
	
	========================================================================
*/
NDIS_STATUS	RTMPCheckRxDescriptor(
	IN	PRXD_STRUC	pRxD)
{
	// Phy errors
	if (pRxD->PhyErr)
	{
		return(NDIS_STATUS_FAILURE);
	}
	
	// CRC errors
	if (pRxD->Crc)
	{
		return(NDIS_STATUS_FAILURE);
	}

	return(NDIS_STATUS_SUCCESS);
}

/*
	========================================================================

	Routine	Description:
		Apply packet filter policy, return NDIS_STATUS_FAILURE if this frame
		should be dropped.
		
	Arguments:
		pAdapter		Pointer	to our adapter
		pRxD			Pointer	to the Rx descriptor
		pHeader			Pointer to the 802.11 frame header
		
	Return Value:
		NDIS_STATUS_SUCCESS		Accept frame
		NDIS_STATUS_FAILURE		Drop Frame
		
	Note:
		Maganement frame should bypass this filtering rule.
	
	========================================================================
*/
NDIS_STATUS	RTMPApplyPacketFilter(
	IN	PRTMP_ADAPTER	pAdapter, 
	IN	PRXD_STRUC		pRxD, 
	IN	PHEADER_802_11	pHeader)
{
	UCHAR	i;
	
	// 0. Management frame should bypass all these filtering rules.
	if (pHeader->Controlhead.Frame.Type == BTYPE_MGMT)
	{
		return(NDIS_STATUS_SUCCESS);
	}
	
	// 1. Drop unicast to me packet if NDIS_PACKET_TYPE_DIRECTED is FALSE
	if (pAdapter->bAcceptPromiscuous == TRUE)
	{
		return(NDIS_STATUS_SUCCESS);
	}
	else if (pRxD->U2M)
	{
		if (pAdapter->bAcceptDirect == FALSE)
		{
			return(NDIS_STATUS_FAILURE);
		}
	}
		
	// 2. Drop broadcast packet if NDIS_PACKET_TYPE_BROADCAST is FALSE
	else if (pRxD->Bcast)
	{
		if (pAdapter->bAcceptBroadcast == FALSE)
		{
			return(NDIS_STATUS_FAILURE);
		}
	}
			
	// 3. Drop multicast packet if NDIS_PACKET_TYPE_ALL_MULTICAST is false
	//    and NDIS_PACKET_TYPE_MULTICAST is false.
	//    If NDIS_PACKET_TYPE_MULTICAST is true, but NDIS_PACKET_TYPE_ALL_MULTICAST is false.
	//    We have to deal with multicast table lookup & drop not matched packets.
	else if (pRxD->Mcast)
	{
		if (pAdapter->bAcceptAllMulticast == FALSE)
		{
			if (pAdapter->bAcceptMulticast == FALSE)
			{
				return(NDIS_STATUS_FAILURE);
			}
			else
			{
				// Selected accept multicast packet based on multicast table
				for (i = 0; i < pAdapter->NumberOfMcAddresses; i++)
				{
					if (RTMPEqualMemory(&pHeader->Controlhead.Addr1, pAdapter->McastTable[i], MAC_ADDR_LEN))
					{
						break;		// Matched
					}
				}

				// Not matched
				if (i == pAdapter->NumberOfMcAddresses)
				{
					DBGPRINT(RT_DEBUG_INFO,"Drop multicast %02x:%02x:%02x:%02x:%02x:%02x\n",
						pHeader->Controlhead.Addr1.Octet[0], pHeader->Controlhead.Addr1.Octet[1],
						pHeader->Controlhead.Addr1.Octet[2], pHeader->Controlhead.Addr1.Octet[3],
						pHeader->Controlhead.Addr1.Octet[4], pHeader->Controlhead.Addr1.Octet[5]);
					return(NDIS_STATUS_FAILURE);
				}
				else
				{
					DBGPRINT(RT_DEBUG_INFO,"Accept multicast %02x:%02x:%02x:%02x:%02x:%02x\n",
						pHeader->Controlhead.Addr1.Octet[0], pHeader->Controlhead.Addr1.Octet[1],
						pHeader->Controlhead.Addr1.Octet[2], pHeader->Controlhead.Addr1.Octet[3],
						pHeader->Controlhead.Addr1.Octet[4], pHeader->Controlhead.Addr1.Octet[5]);
				}
			}
		}
	}

	// 4. Not U2M, not Mcast, not Bcast, must be unicast to other DA.
	//    If promiscuous mode is enabled, accept this packet; otherwise, drop it.

	else
	{
		return(NDIS_STATUS_FAILURE);
	}
	
	return(NDIS_STATUS_SUCCESS);	
}
