/*
	Copyright (C) 2004 - 2006 rt2x00 SourceForge Project
	<http://rt2x00.serialmonkey.com>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Module: rt2500usb
	Abstract: Data structures and registers for the rt2500usb module.
	Supported chipsets: RT2570.
 */

#ifndef RT2500USB_H
#define RT2500USB_H

/*
 * RT chip defines.
 */
#define RT2570				0x1201

/*
 * RF chip defines.
 */
#define RF2522				0x0000
#define RF2523				0x0001
#define RF2524				0x0002
#define RF2525				0x0003
#define RF2525E				0x0005
#define RF5222				0x0010

/*
 * Register layout information.
 */
#define CSR_REG_BASE			0x0400
#define CSR_REG_SIZE			0x00ff
#define EEPROM_BASE			0x0000
#define EEPROM_SIZE			0x006a

/*
 * Control/Status Registers(CSR).
 * Some values are set in TU, whereas 1 TU == 1024 us.
 */

/*
 * MAC_CSR0: ASIC revision number.
 */
#define MAC_CSR0			0x0400

/*
 * MAC_CSR1: System control.
 */
#define MAC_CSR1			0x0402

/*
 * MAC_CSR2: STA MAC register 0.
 */
#define MAC_CSR2			0x0404
#define MAC_CSR2_BYTE0			FIELD16(0x00ff)
#define MAC_CSR2_BYTE1			FIELD16(0xff00)

/*
 * MAC_CSR3: STA MAC register 1.
 */
#define MAC_CSR3			0x0406
#define MAC_CSR3_BYTE2			FIELD16(0x00ff)
#define MAC_CSR3_BYTE3			FIELD16(0xff00)

/*
 * MAC_CSR4: STA MAC register 2.
 */
#define MAC_CSR4			0X0408
#define MAC_CSR4_BYTE4			FIELD16(0x00ff)
#define MAC_CSR4_BYTE5			FIELD16(0xff00)

/*
 * MAC_CSR5: BSSID register 0.
 */
#define MAC_CSR5			0x040a
#define MAC_CSR5_BYTE0			FIELD16(0x00ff)
#define MAC_CSR5_BYTE1			FIELD16(0xff00)

/*
 * MAC_CSR6: BSSID register 1.
 */
#define MAC_CSR6			0x040c
#define MAC_CSR6_BYTE2			FIELD16(0x00ff)
#define MAC_CSR6_BYTE3			FIELD16(0xff00)

/*
 * MAC_CSR7: BSSID register 2.
 */
#define MAC_CSR7			0x040e
#define MAC_CSR7_BYTE4			FIELD16(0x00ff)
#define MAC_CSR7_BYTE5			FIELD16(0xff00)

/*
 * MAC_CSR8: Max frame length.
 */
#define MAC_CSR8			0x0410
#define MAC_CSR8_MAX_FRAME_UNIT		FIELD16(0x0fff)

/*
 * Misc MAC_CSR registers.
 * MAC_CSR9: Timer control.
 * MAC_CSR10: Slot time.
 * MAC_CSR11: IFS.
 * MAC_CSR12: EIFS.
 * MAC_CSR13: Power mode0.
 * MAC_CSR14: Power mode1.
 * MAC_CSR15: Power saving transition0
 * MAC_CSR16: Power saving transition1
 */
#define MAC_CSR9			0x0412
#define MAC_CSR10			0x0414
#define MAC_CSR11			0x0416
#define MAC_CSR12			0x0418
#define MAC_CSR13			0x041a
#define MAC_CSR14			0x041c
#define MAC_CSR15			0x041e
#define MAC_CSR16			0x0420

/*
 * MAC_CSR17: Manual power control / status register.
 * Allowed state: 0 deep_sleep, 1: sleep, 2: standby, 3: awake.
 * SET_STATE: Set state. Write 1 to trigger, self cleared.
 * BBP_DESIRE_STATE: BBP desired state.
 * RF_DESIRE_STATE: RF desired state.
 * BBP_CURRENT_STATE: BBP current state.
 * RF_CURRENT_STATE: RF current state.
 * PUT_TO_SLEEP: Put to sleep. Write 1 to trigger, self cleared.
 */
#define MAC_CSR17			0x0422
#define MAC_CSR17_SET_STATE		FIELD16(0x0001)
#define MAC_CSR17_BBP_DESIRE_STATE	FIELD16(0x0006)
#define MAC_CSR17_RF_DESIRE_STATE	FIELD16(0x0018)
#define MAC_CSR17_BBP_CURR_STATE	FIELD16(0x0060)
#define MAC_CSR17_RF_CURR_STATE		FIELD16(0x0180)
#define MAC_CSR17_PUT_TO_SLEEP		FIELD16(0x0200)

/*
 * MAC_CSR18: Wakeup timer register.
 * DELAY_AFTER_BEACON: Delay after Tbcn expired in units of 1/16 TU.
 * BEACONS_BEFORE_WAKEUP: Number of beacon before wakeup.
 * AUTO_WAKE: Enable auto wakeup / sleep mechanism.
 */
#define MAC_CSR18			0x0424
#define MAC_CSR18_DELAY_AFTER_BEACON	FIELD16(0x00ff)
#define MAC_CSR18_BEACONS_BEFORE_WAKEUP	FIELD16(0x7f00)
#define MAC_CSR18_AUTO_WAKE		FIELD16(0x8000)

/*
 * MAC_CSR19: GPIO control register.
 */
#define MAC_CSR19			0x0426

/*
 * MAC_CSR20: LED control register.
 * ACTIVITY: 0: idle, 1: active.
 * LINK: 0: linkoff, 1: linkup.
 * ACTIVITY_POLARITY: 0: active low, 1: active high.
 */
#define MAC_CSR20			0x0428
#define MAC_CSR20_ACTIVITY		FIELD16(0x0001)
#define MAC_CSR20_LINK			FIELD16(0x0002)
#define MAC_CSR20_ACTIVITY_POLARITY	FIELD16(0x0004)

/*
 * MAC_CSR21: LED control register.
 * ON_PERIOD: On period, default 70ms.
 * OFF_PERIOD: Off period, default 30ms.
 */
#define MAC_CSR21			0x042a
#define MAC_CSR21_ON_PERIOD		FIELD16(0x00ff)
#define MAC_CSR21_OFF_PERIOD		FIELD16(0xff00)

/*
 * Collision window control register.
 */
#define MAC_CSR22			0x042c

/*
 * Transmit related CSRs.
 * Some values are set in TU, whereas 1 TU == 1024 us.
 */

/*
 * TXRX_CSR0: Security control register.
 */
#define TXRX_CSR0			0x0440
#define TXRX_CSR0_ALGORITHM		FIELD16(0x0007)
#define TXRX_CSR0_IV_OFFSET		FIELD16(0x01f8)
#define TXRX_CSR0_KEY_ID		FIELD16(0x1e00)

/*
 * TXRX_CSR1: TX configuration.
 * ACK_TIMEOUT: ACK Timeout in unit of 1-us.
 * TSF_OFFSET: TSF offset in MAC header.
 * AUTO_SEQUENCE: Let ASIC control frame sequence number.
 */
#define TXRX_CSR1			0x0442
#define TXRX_CSR1_ACK_TIMEOUT		FIELD16(0x00ff)
#define TXRX_CSR1_TSF_OFFSET		FIELD16(0x7f00)
#define TXRX_CSR1_AUTO_SEQUENCE		FIELD16(0x8000)

/*
 * TXRX_CSR2: RX control.
 * DISABLE_RX: Disable rx engine.
 * DROP_CRC: Drop crc error.
 * DROP_PHYSICAL: Drop physical error.
 * DROP_CONTROL: Drop control frame.
 * DROP_NOT_TO_ME: Drop not to me unicast frame.
 * DROP_TODS: Drop frame tods bit is true.
 * DROP_VERSION_ERROR: Drop version error frame.
 * DROP_MCAST: Drop multicast frames.
 * DROP_BCAST: Drop broadcast frames.
 */
#define TXRX_CSR2			0x0444
#define	TXRX_CSR2_DISABLE_RX		FIELD16(0x0001)
#define TXRX_CSR2_DROP_CRC		FIELD16(0x0002)
#define TXRX_CSR2_DROP_PHYSICAL		FIELD16(0x0004)
#define TXRX_CSR2_DROP_CONTROL		FIELD16(0x0008)
#define TXRX_CSR2_DROP_NOT_TO_ME	FIELD16(0x0010)
#define TXRX_CSR2_DROP_TODS		FIELD16(0x0020)
#define TXRX_CSR2_DROP_VERSION_ERROR	FIELD16(0x0040)
#define TXRX_CSR2_DROP_MCAST		FIELD16(0x0200)
#define TXRX_CSR2_DROP_BCAST		FIELD16(0x0400)

/*
 * RX BBP ID registers
 * TXRX_CSR3: CCK RX BBP ID.
 * TXRX_CSR4: OFDM RX BBP ID.
 */
#define TXRX_CSR3			0x0446
#define TXRX_CSR4			0x0448

/*
 * TX BBP ID registers
 * TXRX_CSR5: CCK TX BBP ID0.
 * TXRX_CSR5: CCK TX BBP ID1.
 * TXRX_CSR5: OFDM TX BBP ID0.
 * TXRX_CSR5: OFDM TX BBP ID1.
 */
#define TXRX_CSR5			0x044a
#define TXRX_CSR6			0x044c
#define TXRX_CSR7			0x044e
#define TXRX_CSR8			0x0450

/*
 * TXRX_CSR9: TX ACK time-out.
 */
#define TXRX_CSR9			0x0452

/*
 * Auto responder registers.
 * TXRX_CSR10: Auto responder control.
 * TXRX_CSR11: Auto responder basic rate.
 */
#define TXRX_CSR10			0x0454
#define TXRX_CSR11			0x0456

/*
 * ACK/CTS time registers.
 */
#define TXRX_CSR12			0x0458
#define TXRX_CSR13			0x045a
#define TXRX_CSR14			0x045c
#define TXRX_CSR15			0x045e
#define TXRX_CSR16			0x0460
#define TXRX_CSR17			0x0462

/*
 * TXRX_CSR18: Synchronization control register.
 */
#define TXRX_CSR18			0x0464
#define TXRX_CSR18_OFFSET		FIELD16(0x000f)
#define TXRX_CSR18_INTERVAL		FIELD16(0xfff0)

/*
 * TXRX_CSR19: Synchronization control register.
 * TSF_COUNT: Enable TSF auto counting.
 * TSF_SYNC: TSF sync, 0: disable, 1: infra mode, 2: ad-hoc mode.
 * TBCN: Enable Tbcn with reload value.
 * BEACON_GEN: Enable beacon generator.
 */
#define TXRX_CSR19			0x0466
#define TXRX_CSR19_TSF_COUNT		FIELD16(0x0001)
#define TXRX_CSR19_TSF_SYNC		FIELD16(0x0006)
#define TXRX_CSR19_TBCN			FIELD16(0x0008)
#define TXRX_CSR19_BEACON_GEN		FIELD16(0x0010)

/*
 * TXRX_CSR20: Tx BEACON offset time control register.
 * OFFSET: In units of usec.
 * BCN_EXPECT_WINDOW: Default: 2^CWmin
 */
#define TXRX_CSR20			0x0468
#define TXRX_CSR20_OFFSET		FIELD16(0x1fff)
#define TXRX_CSR20_BCN_EXPECT_WINDOW	FIELD16(0xe000)

/*
 * TXRX_CSR21
 */
#define TXRX_CSR21			0x046a

/*
 * Encryption related CSRs.
 *
 */

/*
 * SEC_CSR0-SEC_CSR7: Shared key 0, word 0-7
 */
#define SEC_CSR0			0x0480
#define SEC_CSR1			0x0482
#define SEC_CSR2			0x0484
#define SEC_CSR3			0x0486
#define SEC_CSR4			0x0488
#define SEC_CSR5			0x048a
#define SEC_CSR6			0x048c
#define SEC_CSR7			0x048e

/*
 * SEC_CSR8-SEC_CSR15: Shared key 1, word 0-7
 */
#define SEC_CSR8			0x0490
#define SEC_CSR9			0x0492
#define SEC_CSR10			0x0494
#define SEC_CSR11			0x0496
#define SEC_CSR12			0x0498
#define SEC_CSR13			0x049a
#define SEC_CSR14			0x049c
#define SEC_CSR15			0x049e

/*
 * SEC_CSR16-SEC_CSR23: Shared key 2, word 0-7
 */
#define SEC_CSR16			0x04a0
#define SEC_CSR17			0x04a2
#define SEC_CSR18			0X04A4
#define SEC_CSR19			0x04a6
#define SEC_CSR20			0x04a8
#define SEC_CSR21			0x04aa
#define SEC_CSR22			0x04ac
#define SEC_CSR23			0x04ae

/*
 * SEC_CSR24-SEC_CSR31: Shared key 3, word 0-7
 */
#define SEC_CSR24			0x04b0
#define SEC_CSR25			0x04b2
#define SEC_CSR26			0x04b4
#define SEC_CSR27			0x04b6
#define SEC_CSR28			0x04b8
#define SEC_CSR29			0x04ba
#define SEC_CSR30			0x04bc
#define SEC_CSR31			0x04be

/*
 * PHY control registers.
 */

/*
 * PHY_CSR0: RF switching timing control.
 */
#define PHY_CSR0			0x04c0

/*
 * PHY_CSR1: TX PA configuration.
 */
#define PHY_CSR1			0x04c2

/*
 * MAC configuration registers.
 * PHY_CSR2: TX MAC configuration.
 * PHY_CSR3: RX MAC configuration.
 */
#define PHY_CSR2			0x04c4
#define PHY_CSR3			0x04c6

/*
 * PHY_CSR4: Interface configuration.
 */
#define PHY_CSR4			0x04c8

/*
 * BBP pre-TX registers.
 * PHY_CSR5: BBP pre-TX CCK.
 */
#define PHY_CSR5			0x04ca
#define PHY_CSR5_CCK			FIELD16(0x0003)
#define PHY_CSR5_CCK_FLIP		FIELD16(0x0004)

/*
 * BBP pre-TX registers.
 * PHY_CSR6: BBP pre-TX OFDM.
 */
#define PHY_CSR6			0x04cc
#define PHY_CSR6_OFDM			FIELD16(0x0003)
#define PHY_CSR6_OFDM_FLIP		FIELD16(0x0004)

/*
 * PHY_CSR7: BBP access register 0.
 * BBP_DATA: BBP data.
 * BBP_REG_ID: BBP register ID.
 * BBP_READ_CONTROL: 0: write, 1: read.
 */
#define PHY_CSR7			0x04ce
#define PHY_CSR7_BBP_DATA		FIELD16(0x00ff)
#define PHY_CSR7_BBP_REG_ID		FIELD16(0x7f00)
#define PHY_CSR7_BBP_READ_CONTROL	FIELD16(0x8000)

/*
 * PHY_CSR8: BBP access register 1.
 * BBP_BUSY: ASIC is busy execute BBP programming.
 */
#define PHY_CSR8			0x04d0
#define PHY_CSR8_BBP_BUSY		FIELD16(0x0001)

/*
 * PHY_CSR9: RF access register.
 * RF_VALUE: Register value + id to program into rf/if.
 */
#define PHY_CSR9			0x04d2
#define PHY_CSR9_RF_VALUE		FIELD16(0xffff)

/*
 * PHY_CSR10: RF access register.
 * RF_VALUE: Register value + id to program into rf/if.
 * RF_NUMBER_OF_BITS: Number of bits used in value (i:20, rfmd:22).
 * RF_IF_SELECT: Chip to program: 0: rf, 1: if.
 * RF_PLL_LD: Rf pll_ld status.
 * RF_BUSY: 1: asic is busy execute rf programming.
 */
#define PHY_CSR10			0x04d4
#define PHY_CSR10_RF_VALUE		FIELD16(0x00ff)
#define PHY_CSR10_RF_NUMBER_OF_BITS	FIELD16(0x1f00)
#define PHY_CSR10_RF_IF_SELECT		FIELD16(0x2000)
#define PHY_CSR10_RF_PLL_LD		FIELD16(0x4000)
#define PHY_CSR10_RF_BUSY		FIELD16(0x8000)

/*
 * STA_CSR0: FCS error count.
 * FCS_ERROR: FCS error count, cleared when read.
 */
#define STA_CSR0			0x04e0
#define STA_CSR0_FCS_ERROR		FIELD16(0xffff)

/*
 * Statistic Register.
 * STA_CSR1: PLCP error.
 * STA_CSR2: LONG error.
 * STA_CSR3: CCA false alarm.
 * STA_CSR4: RX FIFO overflow.
 * STA_CSR5: Beacon sent counter.
 */
#define STA_CSR1			0x04e2
#define STA_CSR2			0x04e4
#define STA_CSR3			0x04e6
#define STA_CSR4			0x04e8
#define STA_CSR5			0x04ea
#define STA_CSR6			0x04ec
#define STA_CSR7			0x04ee
#define STA_CSR8			0x04f0
#define STA_CSR9			0x04f2
#define STA_CSR10			0x04f4

/*
 * RF registers.
 */
#define RF1_TUNER			FIELD32(0x00020000)
#define RF3_TUNER			FIELD32(0x00000100)
#define RF3_TXPOWER			FIELD32(0x00003e00)

/*
 * EEPROM contents.
 */

/*
 * HW MAC address.
 */
#define EEPROM_MAC_ADDR			0x0004
#define EEPROM_MAC_ADDR_BYTE0		FIELD16(0x00ff)
#define EEPROM_MAC_ADDR_BYTE1		FIELD16(0xff00)
#define EEPROM_MAC_ADDR_BYTE2		FIELD16(0x00ff)
#define EEPROM_MAC_ADDR_BYTE3		FIELD16(0xff00)
#define EEPROM_MAC_ADDR_BYTE4		FIELD16(0x00ff)
#define EEPROM_MAC_ADDR_BYTE5		FIELD16(0xff00)

/*
 * EEPROM antenna.
 * ANTENNA_NUM: Number of antenna's.
 * TX_DEFAULT: Default antenna 0: diversity, 1: A, 2: B.
 * RX_DEFAULT: Default antenna 0: diversity, 1: A, 2: B.
 * LED_MODE: 0: default, 1: TX/RX activity, 2: Single (ignore link), 3: rsvd.
 * DYN_TXAGC: Dynamic TX AGC control.
 * HARDWARE_RADIO: 1: Hardware controlled radio. Read GPIO0.
 * RF_TYPE: Rf_type of this adapter.
 */
#define EEPROM_ANTENNA			0x0016
#define EEPROM_ANTENNA_NUM		FIELD16(0x0003)
#define EEPROM_ANTENNA_TX_DEFAULT	FIELD16(0x000c)
#define EEPROM_ANTENNA_RX_DEFAULT	FIELD16(0x0030)
#define EEPROM_ANTENNA_LED_MODE		FIELD16(0x01c0)
#define EEPROM_ANTENNA_DYN_TXAGC	FIELD16(0x0200)
#define EEPROM_ANTENNA_HARDWARE_RADIO	FIELD16(0x0400)
#define EEPROM_ANTENNA_RF_TYPE		FIELD16(0xf800)

/*
 * EEPROM NIC config.
 * CARDBUS_ACCEL: 0: enable, 1: disable.
 * DYN_BBP_TUNE: 0: enable, 1: disable.
 * CCK_TX_POWER: CCK TX power compensation.
 */
#define EEPROM_NIC			0x0018
#define EEPROM_NIC_CARDBUS_ACCEL	FIELD16(0x0001)
#define EEPROM_NIC_DISABLE_BBP_TUNE	FIELD16(0x0002)
#define EEPROM_NIC_CCK_TX_POWER		FIELD16(0x000c)

/*
 * EEPROM geography.
 * GEO: Default geography setting for device.
 */
#define EEPROM_GEOGRAPHY		0x001a
#define EEPROM_GEOGRAPHY_GEO		FIELD16(0x0f00)

/*
 * EEPROM BBP.
 */
#define EEPROM_BBP_START		0x001c
#define EEPROM_BBP_SIZE			16
#define EEPROM_BBP_VALUE		FIELD16(0x00ff)
#define EEPROM_BBP_REG_ID		FIELD16(0xff00)

/*
 * EEPROM TXPOWER
 */
#define EEPROM_TXPOWER_START		0x3c
#define EEPROM_TXPOWER_SIZE		7
#define EEPROM_TXPOWER_1		FIELD16(0x00ff)
#define EEPROM_TXPOWER_2		FIELD16(0xff00)

/*
 * Ring defines and data structures.
 */

/*
 * TX descriptor format for TX, PRIO, ATIM and Beacon Ring.
 */
struct txd{
	__le32				word0;
#define TXD_W0_PACKET_ID		FIELD32(0x0000000f)
#define TXD_W0_RETRY_LIMIT		FIELD32(0x000000f0)
#define TXD_W0_MORE_FRAG		FIELD32(0x00000100)
#define TXD_W0_ACK			FIELD32(0x00000200)
#define TXD_W0_TIMESTAMP		FIELD32(0x00000400)
#define TXD_W0_OFDM			FIELD32(0x00000800)
#define TXD_W0_NEW_SEQ			FIELD32(0x00001000)
#define TXD_W0_IFS			FIELD32(0x00006000)
#define TXD_W0_DATABYTE_COUNT		FIELD32(0x0fff0000)
#define TXD_W0_CIPHER			FIELD32(0x20000000)
#define TXD_W0_KEY_ID			FIELD32(0xc0000000)

	__le32				word1;
#define TXD_W1_IV_OFFSET		FIELD32(0x0000003f)
#define TXD_W1_AIFS			FIELD32(0x000000c0)
#define TXD_W1_CWMIN			FIELD32(0x00000f00)
#define TXD_W1_CWMAX			FIELD32(0x0000f000)

	__le32				word2;
#define TXD_W2_PLCP_SIGNAL		FIELD32(0x000000ff)
#define TXD_W2_PLCP_SERVICE		FIELD32(0x0000ff00)
#define TXD_W2_PLCP_LENGTH_LOW		FIELD32(0x00ff0000)
#define TXD_W2_PLCP_LENGTH_HIGH		FIELD32(0xff000000)

	__le32				word3;
#define TXD_W3_IV			FIELD32(0xffffffff)

	__le32				word4;
#define TXD_W4_EIV			FIELD32(0xffffffff)
} __attribute__ ((packed));

/*
 * RX descriptor format for RX Ring.
 */
struct rxd{
	__le32				word0;
#define RXD_W0_UNICAST_TO_ME		FIELD32(0x00000002)
#define RXD_W0_MULTICAST		FIELD32(0x00000004)
#define RXD_W0_BROADCAST		FIELD32(0x00000008)
#define RXD_W0_MY_BSS			FIELD32(0x00000010)
#define RXD_W0_CRC			FIELD32(0x00000020)
#define RXD_W0_OFDM			FIELD32(0x00000040)
#define RXD_W0_PHYSICAL_ERROR		FIELD32(0x00000080)
#define RXD_W0_CIPHER			FIELD32(0x00000100)
#define RXD_W0_CI_ERROR			FIELD32(0x00000200)
#define RXD_W0_DATABYTE_COUNT		FIELD32(0x0fff0000)

	__le32				word1;
#define RXD_W1_SIGNAL			FIELD32(0x000000ff)
#define RXD_W1_RSSI			FIELD32(0x0000ff00)

	__le32				word2;
#define RXD_W2_IV			FIELD32(0xffffffff)

	__le32				word3;
#define RXD_W3_EIV			FIELD32(0xffffffff)
} __attribute__ ((packed));

/*
 * TX ring index number for rt2x00_usb structure.
 */
enum ring_index {
	RING_PRIO			= 0,
	RING_TX				= 1,
	RING_ATIM			= 2,
	RING_BEACON			= 3,
	RING_RX				= 4,
	RING_NUM			= 5,
	RING_NUM_TX			= 2,
};

/*
 * The location of the descriptor is variating and depends
 * on the fact if it is a TX or RX ring and the length of the packet.
 * We need some small handlers to properly access the descriptors.
 */
static inline struct urb* rt2x00usb_urb(struct data_entry *entry)
{
	return (struct urb*)entry->priv;
}

static inline void* rt2x00usb_rxdata_addr(struct data_entry *entry)
{
	return entry->data_addr;
}

static inline void* rt2x00usb_rxdesc_addr(struct data_entry *entry)
{
	return entry->data_addr +
		(rt2x00usb_urb(entry)->actual_length - entry->ring->desc_size);
}

static inline void* rt2x00usb_txdata_addr(struct data_entry *entry)
{
	return entry->data_addr + entry->ring->desc_size;
}

static inline void* rt2x00usb_txdesc_addr(struct data_entry *entry)
{
	return entry->data_addr;
}

/*
 * Macro's for converting txpower from EEPROM to dscape value
 * and from dscape value to register value.
 */
#define MIN_TXPOWER			0
#define MAX_TXPOWER			31
#define DEFAULT_TXPOWER			24

#define TXPOWER_FROM_DEV(__txpower) \
	({ \
		((__txpower) > MAX_TXPOWER) ? DEFAULT_TXPOWER : (__txpower);  \
	})

#define TXPOWER_TO_DEV(__txpower) \
	({ \
		((__txpower) < MIN_TXPOWER) ? MIN_TXPOWER : \
		(((__txpower) > MAX_TXPOWER) ? MAX_TXPOWER : \
		(__txpower)); \
	})

/*
 * IEEE stack callback functions declarations.
 */
static int rt2500usb_tx(struct net_device *net_dev,
	struct sk_buff *skb, struct ieee80211_tx_control *control);
static int rt2500usb_reset(struct net_device *net_dev);
static int rt2500usb_add_interface(struct net_device *net_dev,
	struct ieee80211_if_init_conf *conf);
static void rt2500usb_remove_interface(struct net_device *net_dev,
	struct ieee80211_if_init_conf *conf);
static int rt2500usb_config(struct net_device *net_dev,
	struct ieee80211_conf *conf);
static int rt2500usb_config_interface(struct net_device *net_dev, int if_id,
	struct ieee80211_if_conf *conf);
static void rt2500usb_set_multicast_list(struct net_device *net_dev,
	unsigned short flags, int mc_count);
static int rt2500usb_passive_scan(struct net_device *net_dev,
	int state, struct ieee80211_scan_conf *conf);
static int rt2500usb_get_stats(struct net_device *net_dev,
	struct ieee80211_low_level_stats *stats);
static int rt2500usb_conf_tx(struct net_device *net_dev,
	int queue, const struct ieee80211_tx_queue_params *params);
static int rt2500usb_get_tx_stats(struct net_device *net_dev,
	struct ieee80211_tx_queue_stats *stats);
static int rt2500usb_beacon_update(struct net_device *net_dev,
	struct sk_buff *skb, struct ieee80211_tx_control *control);

/*
 * Radio control functions.
 */
static int rt2500usb_enable_radio(struct rt2x00_usb *rt2x00usb);
static void rt2500usb_disable_radio(struct rt2x00_usb *rt2x00usb);

/*
 * Interrupt functions.
 */
static void rt2500usb_beacondone(void *data);
static void rt2500usb_rxdone(void *data);
static void rt2500usb_txdone(void *data);
static void rt2500usb_interrupt(struct urb *urb, struct pt_regs *regs);

#endif /* RT2500USB_H */
