#!/usr/bin/env python
#
#   XenMan   -  Copyright (c) 2007 Jd & Hap Hazard
#   ======
#
# XenMan is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU Lesser General Public License (LGPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/lgpl.txt
#

import ConfigParser, subprocess, platform
import sys, os, os.path, socket, types
import shutil, urllib,urlparse
import constants
from NodeProxy import Node
import time
import string

class XMConfig(ConfigParser.SafeConfigParser):
    """ XenMan's configuration management class. """

    # the default list of sections in the config
    DEFAULT = 'DEFAULT'
    ENV = 'ENVIRONMENT'
    PATHS = 'PATHS'
    APP_DATA = 'APPLICATION DATA'
    CLIENT_CONFIG = 'CLIENT CONFIGURATION'
    IMAGE_STORE = 'IMAGE STORE'
    DEFAULT_REMOTE_FILE = '/etc/xenman.conf'
    
    def __init__(self, node, searchfiles = None, create_file = None):
        """Looks for xenman.conf in current, user home and /etc directories. If
        it is not found, seeds one in the local directory."""

        ConfigParser.SafeConfigParser.__init__(self)
        self.node = node
        self.std_sections = [self.ENV,self.PATHS,self.APP_DATA,
                             self.CLIENT_CONFIG]
        
        if searchfiles is None:
            # no search path give. apply heuristics
            if not self.node.isRemote:
                # localhost
                filelist = [x for x in [os.path.join(os.getcwd(),'xenman.conf'),
                                        os.path.expanduser('~/.xenman/xenman.conf'),
                                        '/etc/xenman.conf'] if self.node.file_exists(x)]
            else:
                # remote host
                if self.node.file_exists(self.DEFAULT_REMOTE_FILE):
                    filelist =  [self.DEFAULT_REMOTE_FILE]
                else:
                    filelist = []
        else:
            # search path specified
            filelist = [x for x in searchfiles if self.node.file_exists(x)]

        if len(filelist) < 1:
            print 'No Configuration File Found'
            if create_file is None:
                # no default creation file is specified. use heuristics
                if not self.node.isRemote:
                    # localhost. create in cwd
                    print 'Creating default xenman.conf in current directory'            
                    self.configFile = os.path.join(os.getcwd(), 'xenman.conf')
                else:
                    # remote host. create in default remote location
                    print 'Creating default xenman.conf at %s:%s' \
                          % (self.node.hostname,self.DEFAULT_REMOTE_FILE)
                    self.configFile = self.DEFAULT_REMOTE_FILE                
            else:
                # default creation location is specified
                print 'Creating default xenman.conf at %s:%s' \
                      % (self.node.hostname,create_file)                
                self.configFile = create_file            

            # new file created, populate w/ default entries
            self.__createDefaultEntries()
            self.__commit()
            
        else:            
            # config file(s) found. choose a writable file,
            # otherwise create a new default file in the user's
            # home directory (only for localhost)
            self.configFile = None
            for f in filelist:
                try:
                    if self.node.file_is_writable(f):
                        # file is writable
                        if self.__validateFile(f):
                            # file is valid
                            self.configFile = f
                            print 'Valid, writable configuration found, using %s' % f
                        else:
                            # file is writable but not valid
                            # back it up (a new one will get created)                            
                            self.node.rename(f,f+'.bak')
                            print 'Old configuration found. Creating backup: %s.bak' % f                            

                        break
                    else:
                        print 'Confguration File not writable, skipping: %s' % f
                except IOError:
                    print 'Confguration File accessable, skipping: %s' % f
                    continue
                    
            if self.configFile is None:
                # no writable config file found
                print "No writable configuration found ... "
                if not self.node.isRemote:
                    # localhost
                    if not os.path.exists(os.path.expanduser('~/.xenman/')):
                        os.mkdir(os.path.expanduser('~/.xenman/'))
                    self.configFile = os.path.expanduser('~/.xenman/xenman.conf')
                    print "\t Creating %s" % self.configFile                    
                    self.__createDefaultEntries()
                    self.__commit()
                else:
                    # TBD: what to do in the remote case
                    raise Exception('No writable configuration found on remote host: %s' % self.node.hostname)
                
            #self.configFile = filelist[0]
            fp = self.node.open(self.configFile)
            self.readfp(fp)
            fp.close()

        
        self.__commit()


    def __createDefaultEntries(self):

        # cleanup first
        for s in self.sections():
            self.remove_section(s)
            
        # add the standard sections
        for s in self.std_sections:
                self.add_section(s)                


        # seed the defaults
        self.set(self.DEFAULT,constants.prop_default_computed_options,
                 "['arch', 'arch_libdir', 'device_model']") 
        self.set(self.PATHS, constants.prop_disks_dir, '')
        self.set(self.PATHS, constants.prop_snapshots_dir, '')
        self.set(self.PATHS, constants.prop_snapshot_file_ext, '.snapshot.xm')
        self.set(self.PATHS, constants.prop_xenconf_dir,'/etc/xen')
        self.set(self.PATHS, constants.prop_cache_dir, '/var/cache/xenman')
        self.set(self.PATHS, constants.prop_exec_path, '$PATH:/usr/sbin')
        self.set(self.PATHS, constants.prop_image_store,
                 '/var/cache/xenman/image_store')

        self.set(self.PATHS, constants.prop_log_dir,
                 '/var/log/xenman')

        self.set(self.CLIENT_CONFIG, constants.prop_browser, '/usr/bin/yelp')
        #self.set(self.PATHS, constants.prop_staging_location, '')
        #self.set(self.PATHS, constants.prop_kernel, '')
        #self.set(self.PATHS, constants.prop_ramdisk, '')
        #self.set(self.ENV, constants.prop_lvm, 'True')
        #self.__commit()

        # set localhost specific properties
        if not self.node.isRemote:
            #self.add_section(constants.LOCALHOST)
            self.set(self.ENV,constants.prop_dom0_kernel,platform.release())
        
        
    def __commit(self):
        outfile = self.node.open(self.configFile,'w')
        self.write(outfile)
        outfile.close()

    def __validateFile(self, filename):
        temp = ConfigParser.ConfigParser()
        fp = self.node.open(filename)
        temp.readfp(fp)
        fp.close()
        for s in self.std_sections:
            if not temp.has_section(s):
                return False
        return True
    

    def getDefault(self, option):
        """ retrieve a default option/key value """
        return self.get(self.DEFAULT, option)


    def get(self, section, option):

        # does the option exist? return None if not
        if option is None: return None
        if not self.has_option(section, option): return None

        # option is available in the config. get it.
        retVal = ConfigParser.SafeConfigParser.get(self, section, option)
        
        # check if the value is blank. if so, return None
        # otherwise, return the value.
        if retVal == None:
            return retVal
        
        if not retVal.strip():
            return None
        else:
            return retVal
        

    def setDefault(self, option, value):
        """set the default for option to value.
        POSTCONDITION: option, value pair has been set in the default
        configuration, and committed to disk"""
        if option is not None:
            self.set(self.DEFAULT, option, value)


    def set(self, section, option, value):
        ConfigParser.SafeConfigParser.set(self, section, option, value)
        self.__commit()

        
    def getHostProperty(self, option, hostname=constants.LOCALHOST):
        """ retrieve the value for 'option' for 'hostname',
        'None', if the option is not set"""

        # does the option exist? return None if not
        if not self.has_option(hostname, option): return None

        # option is available in the config. get it.
        retVal = self.get(hostname, option)
        
        # check if the value is blank. if so, return None
        # otherwise, return the value.
        if retVal == None:
            return retVal
        
        if not retVal.strip():
            return None
        else:
            return retVal        

    def setHostProperty(self, option, value, hostname=constants.LOCALHOST):
        """ set config 'option' to 'value' for 'hostname'.
        If the a config section for 'hostname' doesn't exit,
        one is created."""
        
        if not self.has_section(hostname): self.add_section(hostname)
        self.set(hostname, option, value)
        self.__commit()

    def removeHost(self, hostname):
        """ remove 'hostname' from the list of configured hosts.
        The configuration is deleted from both memory and filesystem"""

        if self.has_section(hostname):
            self.remove_section(hostname)
            self.__commit()

    def getHosts(self):
        """ return the list configured hosts"""
        hosts=[]
        for sec in self.sections():
            if sec in self.std_sections or sec == self.IMAGE_STORE:
                continue
            else:
                hosts.append(sec)
        #hosts = set(self.sections())-set(self.std_sections)
        #hosts = set(hosts) - set(self.IMAGE_STORE)
        return hosts

    
class LVMProxy:
    """A thin, (os-dependent) wrapper around the shell's LVM
    (Logical Volume Management) verbs"""
    # TODO: move this class to an OSD module
    
    @classmethod
    def isLVMEnabled(cls, node_proxy, exec_path=''):
        retVal = True
        if node_proxy.exec_cmd('vgs 2> /dev/null',exec_path)[1]:
            retVal = False
        return retVal
    
        
    def __init__(self, node_proxy, exec_path=''):
        """ The constructor simply checks if LVM services are available
        for use via the shell at the specified 'node'.
        RAISES: OSError"""
        self.node = node_proxy
        self.exec_path = exec_path

        if node_proxy.exec_cmd('vgs 2> /dev/null',exec_path)[1]:
            raise OSError("LVM facilities not found")


    def listVolumeGroups(self):
        """ Returns the list of existing Volume Groups"""
        try:
            vglist = self.node.exec_cmd('vgs -o vg_name --noheadings', self.exec_path)[0]
            return [s.strip() for s in vglist.splitlines()]
        except OSError, err:
            print err
            return None

    def listLogicalVolumes(self, vgname):
        """ Returns the list of Logical Volumes in a Volume Group"""
        try:            
            lvlist = self.node.exec_cmd('lvs -o lv_name --noheadings '+ vgname,
                                        self.exec_path)[0]
            return [s.strip() for s in lvlist.splitlines()]
        except OSError, err:
            print err
            return None

    def createLogicalVolume(self, lvname, lvsize, vgname):
        """ Create a new LV with in the specified Volume Group.
        'lvsize' denotes size in number of megabytes.
        RETURNS: True on sucees
        RAISES: OSError"""
        error,retcode = self.node.exec_cmd('lvcreate %s -L %sM -n %s'%(vgname,lvsize,lvname),
                                           self.exec_path)
        if retcode:
            raise OSError(error.strip('\n'))
        else:
            return True
        
                
    def removeLogicalVolume(self, lvname, vgname=None):
        """ Remove the logical volume 'lvname' from the
        Volume Group 'vgname'. If the latter is not specified,
        'lvname' is treated as a fully specified path
        RETURNS: True on success
        RAISES: OSError"""
        if (vgname):
            lvpath = vgname + '/' + lvname
        else:
            lvpath = lvname
            
        error,retcode = self.node.exec_cmd('lvremove -f %s'% lvpath, self.exec_path)
        if retcode:
            raise OSError(error.strip('\n'))
        else:
            return True


# Encapsulates some image store structure and convetions.
# 
class ImageStore:
    """A class that manages the list of provisioning images.
      NOTE: This class is intended for use by a client application
      managing multiple DomU images.
      """

    STORE_CONF = "image_store.conf"
    VM_TEMPLATE = "vm_conf.template"
    SCRIPT_NAME  = "provision.sh"
    DEFAULT_STORE_LOCATION = "/var/cache/xenman/image_store"
    COMMON_DIR = 'common'
    IMAGE_CONF = 'image.conf'
    IMAGE_DESC = 'description.htm'
    IMAGE_DESC_HTML = 'description.html'
    
    def __init__(self, config):
        self._config = config
        self._list = []
        self._default = None   # default image for provisioning
        self._excludes = {}  # dirs like common to be excluded
        
        self._store_location = config.get(XMConfig.PATHS,
                                          constants.prop_image_store)
        if self._store_location is None or self._store_location is '':
            self._store_location = self.DEFAULT_STORE_LOCATION
            
        self.__initialize()

    def __initialize(self):
        self.__read_store_conf()

        for file in os.listdir(self._store_location):
            if file not in self._excludes and file[0] != ".":
                full_file = os.path.join(self._store_location, file)
                if os.path.isdir(full_file):
                    self._list.append(file)

    def __read_store_conf(self):
        conf_file = os.path.join(self._store_location, self.STORE_CONF)
        conf = read_python_conf(conf_file)
        if conf is not None:
            if conf.has_key("default"):
                self._default = conf["default"]
            if conf.has_key("excludes"):    
                self._excludes = conf["excludes"]

            
    def get_default_image(self):
        return self._default

    def list(self):
        return self._list

    def get_store_location(self):
        return self._store_location
    
    def get_vm_template(self, image_name):
        if image_name in self.list():
            return os.path.join(self._store_location,
                                image_name,
                                self.VM_TEMPLATE)
        else:
            raise Exception("Invalid image name " + image_name)

    def get_image_conf(self, image_name):
        if image_name in self.list():
            return os.path.join(self._store_location,
                                image_name,
                                self.IMAGE_CONF)
        else:
            raise Exception("Invalid image name " + image_name)
        
    def get_provisioning_script(self, image_name):
        if image_name in self.list():
            return os.path.join(self._store_location,
                                image_name,
                                self.SCRIPT_NAME)
        else:
            raise Exception("Invalid image name " + image_name)

    def get_provisioning_script_dir(self, image_name):
        if image_name in self.list():
            return os.path.join(self._store_location,
                                image_name)
        else:
            raise Exception("Invalid image name " + image_name)
        
    def get_image_desc(self, image_name):
        if image_name in self.list():
            return  os.path.join(self._store_location,
                                image_name,
                                self.IMAGE_DESC)
        else:
            raise Exception("Invalid image name " + image_name)
        
    def get_image_desc_html(self, image_name):
        if image_name in self.list():
            return  os.path.join(self._store_location,
                                image_name,
                                self.IMAGE_DESC_HTML)
        else:
            raise Exception("Invalid image name " + image_name)
        

    def get_common_dir(self):
        return os.path.join(self._store_location, self.COMMON_DIR)

    # Prepare environment for executing script on given managed node
    def prepare_env(self, managed_node, image_name, domconfig, image_conf):
        """ prepare execution environment on the remote node"""

        # prepare directory names
        remote_image_store = managed_node.config.get(XMConfig.PATHS,
                                                     constants.prop_image_store)
        # if not specified, assume similar to client node
        if remote_image_store is None:
            remote_image_store = self.get_store_location()

        scripts_dest = os.path.join(remote_image_store , image_name)
        common_dest  = os.path.join(remote_image_store , self.COMMON_DIR)

        local_image_store = self.get_store_location()
        
        scripts_src_dir = self.get_provisioning_script_dir(image_name)
        common_src  = self.get_common_dir()

        copyToRemote(common_src, managed_node,remote_image_store)
        copyToRemote(scripts_src_dir, managed_node, remote_image_store)
        
        # prepare the log area where the instantiated image.conf and
        # args file would be placed, along with the log dir.

        log_dir = managed_node.config.get(XMConfig.PATHS,
                                          constants.prop_log_dir)
        if log_dir is None or log_dir == '':
            log_dir = constants.DEFAULT_LOG_DIR

        log_location = os.path.join(log_dir, 'image_store', image_name)
        mkdir2(managed_node, log_location)
        
        img_conf_filename = None
        name = domconfig["name"]
        # write the config on the remote node.
        if image_conf is not None:
            img_conf_base = self.get_image_filename(image_name,name)
            img_conf_filename = os.path.join(log_location,img_conf_base)
            # adjust the pyconfig with correct filename and managed_node
            image_conf.set_managed_node(managed_node)
            image_conf.save(img_conf_filename)
        

        return (remote_image_store,
                scripts_dest,
                img_conf_filename,
                log_location)

    def execute_provisioning_script(self,
                                    managed_node,
                                    image_name,
                                    dom_config,
                                    image_conf):

        name = dom_config["name"]
        # prepare the environment to execute script
        (image_store_location,
         script_location,
         img_conf_filename,
         log_location) = \
                               self.prepare_env(managed_node,
                                                image_name,
                                                dom_config,
                                                image_conf)

        # get the script name
        script_name = self.get_provisioning_script(image_name)

        # prepare script args
        script = os.path.join(script_location, script_name)
        script_args_filename = os.path.join(log_location,
                                            self.get_args_filename(image_name,
                                                                     name))
        
        #for now empty
        args = managed_node.node_proxy.open(script_args_filename, "w")
        args.close()

        # update the domconfig to have the reference to the image file
        # kind a kludge: ok for now.
        dom_config["image_conf"] = img_conf_filename
        dom_config.write()

        log_file_base = self.get_log_filename(image_name, name)
        log_filename = os.path.join(log_location,log_file_base)
        script_args=" -x " + dom_config.filename + \
                    " -p " + script_args_filename + \
                    " -s " + image_store_location + \
                    " -i " + image_name + \
                    " -l " + log_filename + \
                    " -c " + img_conf_filename
        
        cmd = script +  script_args

        # execute the script
        (out, exit_code) = managed_node.node_proxy.exec_cmd(cmd)

        #if exit_code != 0:
        #    raise OSError("Provisioning script failed.\n" +  out)

        #managed_node.node_proxy.remove(script_args_filename)
        #managed_node.node_proxy.remove(image_filename)
        #managed_node.node_proxy.remove(log_filename)
        return (out, exit_code, log_filename)

    def get_args_filename(self,image_name, dom_name):
        name = image_name + "_" + dom_name +  ".args"
        #for t in time.gmtime():
        #    name = name + str(t) + "_"
        #name = name[:-1]
        return name

    def get_image_filename(self,image_name, dom_name):
        name = image_name + "_" + dom_name + "_" + "image.conf"
        return name

    def get_log_filename(self,image_name, dom_name):
        name = image_name + "_" + dom_name + ".log"
        return name



from threading import Thread

class Poller(Thread):
    """ A simple poller class representing a thread that wakes
    up at a specified interval and invokes a callback function"""

    def __init__(self, freq, callback, args=[], kwargs={}, max_polls = None):
        Thread.__init__(self)
        self.setDaemon(True)
        self.frequency = freq
        self.callback = callback
        self.args = args
        self.kwargs = kwargs
        self.done = False
        self.remainder = max_polls

    def run(self):
        while not self.done:
            self.callback(*self.args,**self.kwargs)
            time.sleep(self.frequency)
            if self.remainder is not None:
                self.remainder -= 1
                if self.remainder < 0:
                    self.done = True

    def stop(self):
        self.done = True



class PyConfig:
    """
        Class represents python based config file.
        Also, supports instantiating a templatized config file
    """
    default_computed_options = []
    COMPUTED_OPTIONS = "computed_options"
    CUSTOMIZABLE_OPTIONS = "customizable_options"
    def __init__(self,
                 managed_node,
                 filename = None,       # config file
                 signature = None):      # signature to be written as first line

        
        """ filename and file open function to be used """
        self.filename = filename
        self.managed_node = managed_node
        self.lines = []
        self.options = {}
        self.signature = signature
        

        if self.filename is not None:
            (self.lines, self.options) = self.read_conf()

    @classmethod        
    def set_computed_options(cls, computed):
        cls.default_computed_options = computed

    def get_computed_options(self):
        c = []
        if self.default_computed_options is not None:
            c = self.default_computed_options
            
        if self.options.has_key(self.COMPUTED_OPTIONS):
            specific_computed_options = self[self.COMPUTED_OPTIONS]
        else:
            specific_computed_options = None
            
        if specific_computed_options is not None and \
               type(specific_computed_options) == types.ListType:
            for o in specific_computed_options:
                c.append(o)

        if self.COMPUTED_OPTIONS not in c :
            c.append(self.COMPUTED_OPTIONS)
        return c

    def get_customizable_options(self):
        customizable_options = None
        if self.options.has_key(self.CUSTOMIZABLE_OPTIONS):
            customizable_options = self[self.CUSTOMIZABLE_OPTIONS]
        else:
            customizable_options = self.options.keys()

        if customizable_options is not None and \
               self.CUSTOMIZABLE_OPTIONS in customizable_options :
            customizable_options.remove(self.CUSTOMIZABLE_OPTIONS)
            
        return customizable_options

    # set the name of the file associated with the config.
    def set_filename(self, filename):
        """
        set the filename associated with this config.
        subsequent write would use this file name.
        """
        self.filename = filename

    # Allow for changing the managed node.
    # useful in reading a template and then writing modified, instantiated
    # file on a managed node.
    def set_managed_node(self, node):
        self.managed_node = node
        
    def read_conf(self, init_glob=None, init_locs=None):

        if init_glob is not None:
            globs = init_glob
        else:
            globs = {}

        if init_locs is not None:
            locs = init_locs
        else:
            locs  = {}
            
        lines = []
        options = {}
        try:
            f = self.managed_node.node_proxy.open(self.filename)
            lines = f.readlines()
            f.close()
            if len(lines) > 0:
                cmd = "\n".join(lines)
                exec cmd in globs, locs
        except:
            raise
        # Extract the values set by the script and set the corresponding
        # options, if not set on the command line.
        vtypes = [ types.StringType,
                   types.ListType,
                   types.IntType,
                   types.FloatType
                   ]
        for (k, v) in locs.items():
            if not(type(v) in vtypes): continue
            options[k]=v

        return (lines,options)
    
    def write(self, full_edit=False):
        """Writes the settings out to the filename specified during
        initialization"""

        outfile = self.managed_node.node_proxy.open(self.filename, 'w')
            
        if self.signature is not None:
            outfile.write(self.signature)
        
        # Simple write
        if self.lines is None or len(self.lines) == 0:
            for name, value in self.options.iteritems():
                outfile.write("%s = %s\n" % (name, repr(value)))
        else:
            # drive the writing through lines read.
            updated = []
            for line in self.lines:
                if self.signature is not None and \
                       line.find(self.signature) >= 0:
                    continue
                if line[0] == '#' or line[0] == '\n' or \
                        line[0].isspace() or line.strip().endswith(':'):
                    outfile.write(line)
                else:
                    ndx = line.find("=")
                    if ndx > -1:
                        token = line[0:ndx]
                        token = token.strip()
                        if self.options.has_key(token):
                            if token not in self.get_computed_options() and \
                               (token != self.CUSTOMIZABLE_OPTIONS or full_edit) :
                                value = self.options[token]
                                outfile.write("%s=%s\n" % (token, repr(value)))
                                updated.append(token)
                            else:
                                #print "writing computed Token X:" , line
                                if token != self.COMPUTED_OPTIONS:
                                    outfile.write(line)
                        else:
                            if token in self.get_computed_options():
                                outfile.write(line)
                            else:
                                #print "Valid token but removed" , line
                                pass
                    else:
                        #print "writing default Y:" , line
                        outfile.write(line)

            # Add new tokens added
            for name, value in self.options.iteritems():
                if name not in updated and \
                       name not in self.get_computed_options():
                    outfile.write("%s=%s\n" % (name, repr(value)))

        outfile.close()

    def instantiate_config(self, value_map):
        
        # do this so that substitution happens properly
        # we may have to revisit, map interface of PyConfig
        if isinstance(value_map, PyConfig):
            value_map = value_map.options
        
        for key in self.options.keys():
            value = self.options[key]
            if value is not None:
                if type(value) is types.StringType:
                    template_str = string.Template(value)
                    self.options[key] = template_str.safe_substitute(value_map)
                elif type(value) is types.ListType:
                    new_list = []
                    for v in value:
                        if type(v) is types.StringType:
                            template_str = string.Template(v)
                            new_list.append(template_str.safe_substitute(value_map))
                                                
                        else:
                            new_list.append(v)
                    self.options[key] = new_list
                    #print "old %s, new %s", (value, self.options[key])
                            
                    

    def save(self, filename):
        """ save the current state to a file"""
        self.filename = filename
        self.write()

    #access config as hastable 
    def __getitem__(self, name):
        if self.options.has_key(name):
            return self.options[name]
        else:
            attrib = getattr(self,name, None)
            if attrib is not None:
                return attrib
            else:
                return None

    def __setitem__(self, name, item):
        self.options[name] = item

    def __iter__(self):
        return self.options.iterkeys()
    
    def has_key(self, key):
        return self.options.has_key(key)
    
    def keys(self):
        return self.options.keys()
    # debugging dump
    def dump(self):
        if self.filename is not None:
            print self.filename
        for name, value in self.options.iteritems():
            print "%s = %s" % (name, repr(value))


#
# copy directory from local filesystem to remote machine, dest.
# src : source file or directory
# dest_node :node on which the files/directory needs to be copied
# dest_dir : destination directory under which src file/dir would be
#            copied
# dest_name : name of file/directory on the destination node.
# overwrite : would overwrite the files even if they exists.
#
def copyToRemote(src,dest_node,dest_dir, dest_name = None, overwrite=True):
    if not os.path.exists(src):
        raise "%s does not exist." % src
    if os.path.isfile(src):
        mkdir2(dest_node,dest_dir)
        if dest_name is not None:
            dest = os.path.join(dest_dir, dest_name)
        else:
            dest = os.path.join(dest_dir, os.path.basename(src))

        if not dest_node.node_proxy.file_exists(dest) or overwrite:
            mode = os.lstat(src).st_mode
            dest_node.node_proxy.put(src, dest)
            dest_node.node_proxy.chmod(dest, mode)
    elif os.path.isdir(src):
        mkdir2(dest_node,dest_dir)
        if dest_name is not None:
            dest = os.path.join(dest_dir, dest_name)
        else:
            (dirname, basename) = os.path.split(src)
            dest = os.path.join(dest_dir,basename)
            mkdir2(dest_node, dest)
            
        for entry in os.listdir(src):
            s = os.path.join(src, entry)
            copyToRemote(s, dest_node, dest)

# make n level directory.
def mkdir2(dest_node, dir):
    root = dir
    list = []
    while not dest_node.node_proxy.file_exists(root) and root is not '/':
        list.insert(0,root)
        (root, subdir) = os.path.split(root)

    for d in list:
        dest_node.node_proxy.mkdir(d)
    
    

   
def fetchImage(src, dest):
    """ Copies 'src' to 'dest'. 'src' can be an http or ftp URL
    or a filesystem location. dest must be a fully qualified
    filename."""
    
    print "Fetching: "+src
    if src.startswith("http://") or src.startswith("ftp://"):
        # newtwork fetch
        urllib.urlretrieve(src,dest)
    else:
        # filesystem fetch
        shutil.copyfile(src, dest)





def search_tree(tree, key):
    """Retrieve a value from a tree"""

    if tree == None or key == None:
        return None
    
    for elem in tree:
        if elem[0] == key: 
            return elem[1]
        elif type(elem[1]) is list:
            value = search_tree(elem[1], key)
            if value:
                return value
    return None


def is_host_remote(host):
    host_names = []
    try:
        (host_name, host_aliases,host_addrs) = socket.gethostbyaddr(host)
        host_names.append(host_name)
        host_names = host_aliases + host_addrs
    except:
        host_names.append(host)

    return len(set(l_names).intersection(set(host_names))) == 0


# we will be using this at few places. better to keep it in a separate
# function
def read_python_conf(conf_file):
    """ reads a conf file in python format and returns conf as hash table"""
    glob = {}
    loc  = {}
    if not os.path.exists(conf_file):
        print "Image store conf not found :" + conf_file
        return None
    execfile(conf_file, glob, loc)
    
    # filter out everything other than simple values and lists
    vtypes = [ types.StringType,
               types.ListType,
               types.IntType,
               types.FloatType
               ]

    for (k, v) in loc.items():
        if type(v) not in vtypes:
            del loc[k]

    return loc


# need to go in common place
def guess_value(value):
    # check if it is a number
    if value is None or value == '':
        return value

    if value.isdigit():
        return int(value)

    # check if float
    parts = value.split(".")
    if len(parts) == 2:
        if parts[0].isdigit() and parts[1].isdigit():
            return float(value)

    # check if it is a list
    if value[0] == '[':
        g = {}
        l = {}
        cmd = "x = " + value
        exec cmd in g, l
        return l['x']

    # assume it is a string
    return value


#
# Module initialization
#

l_names = []
try:
    (local_name, local_aliases,local_addrs) = \
                 socket.gethostbyaddr(constants.LOCALHOST)

    l_names.append(local_name)
    l_names = local_aliases + local_addrs
except socket.herror:
    print "ERROR : can not resolve localhost"
    pass
    



#########################
# SELF TEST
#########################

if __name__ == '__main__':

    REMOTE_HOST = '192.168.123.155'
    REMOTE_USER = 'root'
    REMOTE_PASSWD = ''

    REMOTE = False    
    
    local_node = Node(hostname=constants.LOCALHOST)
    if not REMOTE:
        remote_node = local_node  # for local-only testing
    else:        
        remote_node = Node(hostname=REMOTE_HOST,
                           username=REMOTE_USER,
                           password = REMOTE_PASSWD,
                           isRemote = True)    


    #
    # LVMProxy tests
    #
    lvm_local = LVMProxy(local_node)
    lvm_remote = LVMProxy(remote_node)
    lvm_remote = lvm_local

    print '\nLVMProxy interface test STARTING'
    for lvm in (lvm_local, lvm_remote):
        vgs =  lvm.listVolumeGroups()
        for g in vgs:
            print g
            print lvm.listLogicalVolumes(g)
            print '\t Creating test LV'
            lvm.createLogicalVolume('selfTest',0.1,g)
            print '\t Deleting test LV'
            lvm.removeLogicalVolume('selfTest',g)
    print 'LVMPRoxy interface test COMPLETED\n'


    #
    # XMConfig tests
    #    

    TEST_CONFIGFILE = '/tmp/xenman.conf'
    
    print "\nXMConfig interface test STARTING\n"
    
    print 'LOCALHOST ...'
    config_local = XMConfig(local_node, searchfiles = [TEST_CONFIGFILE],
                            create_file=TEST_CONFIGFILE)    
    config_local.set(XMConfig.DEFAULT,'TEST_PROP','TEST_VAL')
    print "Default Property TEST_PROP:",config_local.getDefault('TEST_PROP')
    print "Default Sections:",config_local.sections()
    print "Known Hosts", config_local.getHosts()
    config_local2 = XMConfig(local_node, searchfiles = [TEST_CONFIGFILE])
    print "Default Property TEST_PROP:",config_local2.getDefault('test_prop')    
    local_node.remove(TEST_CONFIGFILE)

    print '\nREMOTE HOST ...'
    config_remote = XMConfig(remote_node, searchfiles = [TEST_CONFIGFILE],
                            create_file=TEST_CONFIGFILE)
    config_remote.setDefault('TEST_PROP','TEST_VAL')
    print "Default Property TEST_PROP:",config_remote.get(XMConfig.DEFAULT,'TEST_PROP')
    print "Default Sections:",config_remote.sections()
    print "Known Hosts", config_remote.getHosts()
    config_remote2 = XMConfig(remote_node, searchfiles = [TEST_CONFIGFILE])
    print "Default Property TEST_PROP:",config_remote2.getDefault('test_prop')
    remote_node.remove(TEST_CONFIGFILE)

    print "\nXMConfig interface test COMPLETED"


    #
    # ImageStore tests
    #

    print "\nImageStore interface test STARTING\n"
    config_local = XMConfig(local_node, searchfiles = [TEST_CONFIGFILE],
                            create_file=TEST_CONFIGFILE)
    store = ImageStore(config_local)
    print store.list()
    

##     print "\nImageStore interface test STARTING\n"
##     config_local = XMConfig(local_node, searchfiles = [TEST_CONFIGFILE],
##                             create_file=TEST_CONFIGFILE)
##     store = ImageStore(config_local)
##     print store.list
    
##     store.addImage('test_image','/var/cache/xenman/vmlinuz.default','/var/cache/xenman/initrd.img.default')
##     print store.list
##     print store.getImage('test_image')
##     print store.getFilenames('test_image')

##     #store.addImage('test_image2',
##     #               'http://linux.nssl.noaa.gov/fedora/core/5/i386/os/images/xen/vmlinuz',
##     #               'http://linux.nssl.noaa.gov/fedora/core/5/i386/os/images/xen/initrd.img',
##     #               )
##     #print store.list
##     #print store.getImage('test_image2')
##     #print store.getFilenames('test_image2')

##     store.addImage('test_image2',
##                    'http://localhost/fedora/images/xen/vmlinuz',
##                    'http://localhost/fedora/images/xen/initrd.img',
##                    )
##     print store.list
##     print store.getImage('test_image2')
##     print "First access, should fetch ...\n",store.getFilenames('test_image2')
##     print "Second access, should get from cache ... "
##     kernel, ramdisk = store.getFilenames('test_image2')
##     print (kernel, ramdisk)
##     local_node.remove(kernel)
##     local_node.remove(ramdisk)
##     local_node.rmdir('/var/cache/xenman/test_image2')
    
##     local_node.remove(TEST_CONFIGFILE)
    
    print "\nImageStore interface test COMPLETED"
    sys.exit(0)
    
    
    
        
