#!/usr/bin/env python
#
#   XenMan   -  Copyright (c) 2007 Jd, Hap Hazard & Yves Perrenoud
#   ======
#
# XenMan is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU General Public License (GPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/gpl.txt
#

# adjust the python paths for accomodating xen (similar to xm)
import sys
sys.path.append('/usr/lib/python')
sys.path.append('/usr/lib64/python')

import pango
import types
from TreeViewTooltips import TreeViewTooltips
import gtk, gtk.glade, gobject
from xen.xend.XendProtocol import XendError
import xmlrpclib

#import xen.xm.create
import vte, re, os, platform, glob

import socket
import datetime 

from phelper import PHelper
from NodeProxy import Node
import dialogs
from DomSettings import DomSettings
from dialogs import CreateDialog, InitParamsDialog, AddNodeDialog, NodeSelection
from dialogs import CredentialsHelper, FileViewEditDialog
from dialogs import wtree
from dialogs import cleanupQCDomain, file_selection, checkAndSetDefaultPaths
from dialogs import main_context, show_wait_cursor, hide_wait_cursor

from htmltextview import HtmlTextView


import utils
from utils import is_host_remote
from constants import *

import traceback
import xml.parsers.expat
from XenNode import DomConfig

import subprocess



#
# defaults
#

default_tcp_port = "8005"
default_path = "/RPC2"
default_user = "root"
default_host = 'localhost'

## Add from xenauto or auto_start flag in the domconfig/dom.
# So this is not required
XENAUTO = '/etc/xen/auto'   # pick it up from managed node


#
# class definitions
#

from ManagedNode import ManagedNode,NodeException
from XenNode import XenNode,XenDomain
# _RESTRUCTURING


class LeftNav:
    """
    Keeps track of the left side navigation
    List of Managed Nodes as well as virtual machines running on them.
    """
    # Types of nodes in the tree
    SERVER_POOL = dialogs.SERVER_POOL
    MANAGED_NODE = dialogs.MANAGED_NODE
    DOMAIN  = dialogs.DOMAIN

    IMAGE_STORE = dialogs.IMAGE_STORE
    IMAGE = dialogs.IMAGE

    # NodeList Constructor
    def __init__(self, wtree):
        """
        constructor, initialize the widgets
        Tree populated with nodes from grid manager
        Assumes manager as global variable.
        """
        self.wtree = wtree
        self.left_nav_view = wtree.get_widget("DomView")
        self.left_nav_view.get_selection().connect("changed",
                                               self.handle_selection)
        self.left_nav_view.connect("row-activated",
                                   self.handle_row_activated)

        self.left_nav_view.connect("row-expanded",
                                self.handle_row_expanded)


        # initialize columns
        pbrenderer = gtk.CellRendererPixbuf()
        column = gtk.TreeViewColumn("State", pbrenderer)
        column.set_cell_data_func(pbrenderer, dialogs.get_state_pixbuf)
        self.left_nav_view.append_column(column)

        textrenderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn("Name", textrenderer, text=0)

        self.left_nav_view.append_column(column)

        self.left_nav_tree_model = gtk.TreeStore(gobject.TYPE_STRING,
                                                 gobject.TYPE_STRING,
                                                 gobject.TYPE_PYOBJECT)

        # Add a sorter inbetween
        #sorted_model = gtk.TreeModelSort(self.left_nav_tree_model)
        #sorted_model.set_sort_column_id(0, gtk.SORT_ASCENDING)

        sorted_model = self.left_nav_tree_model

        # This is also behaving strange, when timer is enabled.
        # some multi-threaded problem!!
        #sorted_model.set_sort_column_id(0, gtk.SORT_ASCENDING)
        self.left_nav_view.set_model(sorted_model)
        
        
        ## Does not work..! property not available.!
        #self.left_nav_view.set_property("show-expanders", True)
        #self.left_nav_view.set_show_expanders(True)

        self.refresh_nodes()


    # signal handlers
    def handle_row_activated(self, treeview, path, column):
        """
        handle double-click on the managed node.
        """
        model = treeview.get_model()
        iter = model.get_iter(path)
        if iter:
            node_type = model.get_value(iter,1)
            tree_node = model.get_value(iter,2)
            if node_type == self.MANAGED_NODE:
                if self._refresh_doms(model, tree_node, iter, path, path):
                    path = model.get_path(iter)
                    treeview.expand_row(path, True)
            elif node_type == self.DOMAIN:
                Coordinator.handle_row_activated()

        return True


    # A node in the tree is expanded/collapsed
    def handle_row_expanded(self, treeview, iter, path, userdata=None):
        model = treeview.get_model()
        iter = model.get_iter(path)
        if iter:
            node_type = model.get_value(iter,1)
            tree_node = model.get_value(iter,2)
            
            if node_type == self.MANAGED_NODE:
                self._refresh_doms(model, tree_node, iter, path, path)
                Coordinator.handle_nav_selection(treeview)


        return True

        
    def handle_selection(self, widget=None):
        """Handles the selection of a domain in the domain tree view"""

        Coordinator.handle_nav_selection(widget)


    def refresh_doms(self, auto=False, dom_name = None):
        """ take current selection and refresh dom list."""
        tree_selection = self.left_nav_view.get_selection()
        model, iter = tree_selection.get_selected()
        if not iter :
            # no selection
            return True

        node_type = model.get_value(iter,1)
        selected_node = model.get_value(iter,2)

        if node_type == self.DOMAIN:
            # current selection on dom, get the managed node
            parent_iter = model.iter_parent(iter)
            managed_node = model.get_value(parent_iter,2)
            path = model.get_path(parent_iter)
            current_path = model.get_path(iter)
            return self._refresh_doms(model, managed_node, parent_iter, path,
                                      current_path, auto)
        elif node_type == self.MANAGED_NODE:
            # current selection is managed node itself
            managed_node = selected_node
            path = model.get_path(iter)
            return self._refresh_doms(model, managed_node, iter, path, path, auto)



    # utility methods
    def _append_dummy_row(self, iter):
        self.left_nav_tree_model.append(iter, ["VMs not found",
                                               "DUMMY",None])
    
    def _refresh_doms(self, model, managed_node,
                      iter, path, current_path,
                      auto = False):
        """
        refresh dom list for the managed node, refered by iter
        """
        
        # This is done in the reverse way, as the normal way
        # does not work. We add the new nodes and remove the old
        # ones later.Need to be careful, at some point there
        # would be more than one instance of the same dom here!!
        # Also, the order matters. (careful, when we add sorting!)
        
        #print managed_node.hostname, managed_node.is_authenticated()
        if isinstance(managed_node, ManagedNode) and iter:
            if auto and (not managed_node.is_authenticated() or \
                                            managed_node.is_in_error()):
                return True

            num_old_doms = model.iter_n_children(iter)            
            # get new ones and insert them
            # catch the expact error. Intermittently we get this because
            # server returns an empty string. (dont know why, yet)
            try:
                managed_node.refresh()
                doms = managed_node.get_dom_names()
            except xml.parsers.expat.ExpatError, e:
                #print "_refresh_doms() :", e
                return                
            except Exception, e:
                showmsg('Exception '+str(e))
                return False
            

            doms.sort()
            # add the dummy row back if no current doms
            if len(doms) == 0:
                self._append_dummy_row(iter)

            for dom_name in doms:
                #if not dom.is_resident:
                #    Coordinator.handle_kill_dom(dom)
                node = model.iter_children(iter)
                exists = False
                while node:
                    d_name = model.get_value(node,0)
                    node_type = model.get_value(node,1)
                    m_node = model.get_value(node,2)

                    #update the managed node instance if required.
                    if m_node is not None and m_node is not managed_node:
                        model.set_value(node,2, managed_node)
                        
                    if node_type == self.DOMAIN and d_name == dom_name:
                        exists = True
                        #model.set_value(node,0,dom_name)
                        #path = model.get_path(node)
                        #model.row_changed(path,node)
                        #print "path, iter ", path, iter
                        break

                    if d_name > dom_name:  # keep things in sorted order
                        break

                    node= model.iter_next(node)
                    
                if not exists and dom_name != "Domain-0": #Hard coding ..bad
                    #print "adding new node", dom_name
                    model.insert_before(iter,node,
                                       [dom_name,self.DOMAIN,managed_node,])

            # clear old nodes that do not exist
            next_node = model.iter_children(iter)
            while next_node: 
                this_node = next_node
            
                dom_name = model.get_value(this_node,0)
                node_type = model.get_value(this_node,1)

                if dom_name is not None and dom_name != "" and \
                       node_type == self.DOMAIN:

                    if dom_name in doms:
                        d = managed_node.get_dom(dom_name)
                        if not managed_node.isResident(dom_name):
                            Coordinator.handle_kill_dom(d)
                        else:
                            Coordinator.handle_existing_dom(d)

                        next_node = model.iter_next(this_node)
                            
                    else:
                        #print "removing " , dom_name
                        Coordinator.handle_missing_dom(managed_node,
                                                       dom_name)
                        next_node = model.iter_next(this_node)
                        model.remove(this_node)
                        
                else:
                    #print "removing ", dom_name
                    next_node = model.iter_next(this_node)
                    model.remove(this_node)



            # set  selection to current selection
            try:
                selection = self.left_nav_view.get_selection()
                if auto: # suppress callbacks
                    Coordinator.fromUserAction = False
                selection.select_path(current_path)
            finally:
                Coordinator.fromUserAction = True
        else :
            print "refresh_dom called without Server"
            print managed_node.__class__, iter

        self.left_nav_view.queue_draw()
        return True

    def populate_images(self):
        images = image_store.list()
        

        i_iter = self.left_nav_tree_model.append(None,
                                                 ["Image Store",
                                                  self.IMAGE_STORE,
                                                  None])
        if images:
            images.sort()
            
        for image in images:                                        
            iter = \
                 self.left_nav_tree_model.append(i_iter,
                                                 [image,
                                                  self.IMAGE,
                                                  local_node])

            
    def refresh_nodes(self):
        """
        refresh the nodes.
        Not expected to be called other than constructor.
        
        """
        nodes = manager.getNodeList()
        self.left_nav_tree_model.clear()
        # Append a Global Server Pool
        g_iter = self.left_nav_tree_model.append(None,
                                                 ["Server Pool",
                                                  self.SERVER_POOL,
                                                  None])
                                        
        # Kludge, append a blank node so the expanders show up
        for node in nodes.itervalues():
            iter = \
                 self.left_nav_tree_model.append(g_iter,
                                                [node.hostname,
                                                 self.MANAGED_NODE,
                                                 node,])
            self._append_dummy_row(iter)

        self.left_nav_view.expand_row((0,),False)
        self.populate_images()
        self.left_nav_view.set_cursor((0,))
        

    def add_node(self, node):
        if node.hostname not in manager.getNodeNames():
            manager.addNode(node)
            self.refresh_nodes()


    def remove_node(self, node):
        manager.removeNode(node.hostname)
        client_config.removeHost(node.hostname)
        self.refresh_nodes()

    # returns tuple with name, node_type and managed node
    def get_selection(self):
        """
        return current selection 
        returns either ManagedNode,  Dom or None
        """
        widget = self.left_nav_view.get_selection()
        treemodel, treeiter = widget.get_selected()
        if treeiter:
            return (treemodel.get_value(treeiter, 0),
                    treemodel.get_value(treeiter, 1),
                    treemodel.get_value(treeiter, 2))

        return (None,None,None)

    def get_selection_type(self):
        (name, node_type, managed_node) = self.get_selection()
        return node_type

    def get_managed_node(self):
        """
        return managed node for the currently selected dom
        """
        (name,node_type, managed_node) = self.get_selection()
        return managed_node
    
    def set_selection(self, node_name, node_type, op = None):
        """
        set the current selection to either a Dom or a ManagedNode
        """
        # this may be tricky
        self.left_nav_tree_model.foreach(self.select_managed_node,
                                         (node_name, node_type, op))

    def select_managed_node(self, model, path, iter, context):
        (node_name, node_type, op) = context
        if iter:
            (name, type, managed_node) = (model.get_value(iter, 0),
                                          model.get_value(iter, 1),
                                          model.get_value(iter, 2))
            if name == node_name and type == node_type:
                self.left_nav_view.set_cursor(path)
                if op == "row-activated":
                    # we know that last param is not used. so pass None.
                    self.handle_row_activated(self.left_nav_view, path, None) 
                return True

            
        
    def addDomFile(self, filename):
        """Adds a dom specified by it's file name to the index"""
        try:
            managed_node = Coordinator.get_managed_node()
            dom = managed_node.add_dom_config(filename)
            self.refresh_doms()
        except (Exception, StandardError), e:
            showmsg('FAILED. Invalid File: '+str(e))


    def remDomFile(self, filename):
        """Remove a dom specified by it's file name to the index"""
        managed_node = Coordinator.get_managed_node()
        if managed_node.remove_dom_config(filename):
            self.refresh_doms()
        else:
            return False
        


class Coordinator:
    """ Class that takes care of cordination """
    # currently the toolbar, menu and popup are assumed to be part of
    # cordinator, later can be moved out in to a separate class



    # going with static methods (can make it singleton later)
    # Also, this is not a pure cordinator (yet) but central place
    # for all interactiona.

    nav = None
    notebook = None
    fromUserAction = True

    # timer vars
    skipcount = 0
    stat_refresh_rate = 5
    stat_refresh_count = stat_refresh_rate


    last_focus_in = None
    last_focus_out = None
##  Singleton bit later.
##     @classmethod
##     def instance():
##         return self._instance

##     @classmethod
##     def initialize(nav, notebook):
##         self._instance = Coordinator(nav, notebook)

##     def __init__(self,nav, notebook):
##         self.nav = nav
##         self.notebook = notebook
    
    
    @classmethod
    def set_nav(cls,nav):
        cls.nav = nav

    @classmethod
    def set_notebook(cls, notebook):
        cls.notebook =  notebook
    

    @classmethod
    def handle_row_activated(cls):
        if cls.nav == None:
            return
        
        (name, node_type, managed_node) = cls.nav.get_selection()
        if managed_node is None:
            return

        if node_type == LeftNav.DOMAIN:
            if managed_node.isDomU(name):
                if managed_node.isResident(name):
                    dom = managed_node.get_dom(name)
                    cls.notebook.showTab(dom)

       
    @classmethod
    def handle_nav_selection(cls, widget):
        cls.stateRefresh()
        if cls.nav == None or cls.fromUserAction == False:
            return
        (name, node_type, managed_node) = cls.nav.get_selection()
        
        cls.notebook.summary_tab.enable_tab()
        cls.notebook.info_tab.enable_tab()
        
        if node_type == LeftNav.IMAGE or \
               node_type == LeftNav.IMAGE_STORE:
            cls.notebook.summary_tab.disable_tab()
            cls.notebook.showDomTab(None, None) #uggh
            return

        
        if node_type == LeftNav.SERVER_POOL:
            cls.notebook.info_tab.disable_tab()
            cls.notebook.showSummaryTab()
                
        #if managed_node is None:
        #    return
        
        #Handle if the selection was a Dom on a Node
        if node_type == LeftNav.DOMAIN:
            cls.notebook.showDomTab(managed_node,name)

        # handle node selection
        elif node_type==LeftNav.MANAGED_NODE:
            #managed_node.is_authenticated() and \
            #if not managed_node.is_in_error():
            cls.notebook.showSummaryTab()
            



    @classmethod
    def handle_timer_callback(cls):
        if cls.nav is None : return
        try:
            try:
                cls.fromUserAction = False
                cls.nav.refresh_doms(True)

                cls.stat_refresh_count -= 1
                if cls.stat_refresh_count <= 0:
                   cls.stat_refresh_count = cls.stat_refresh_rate
                   # fetch stats only if home tab is shown
                   if (cls.notebook.getTabPos() == 0):
                       cls.notebook.showSummaryTab()

                if cls.skipcount:
                    cls.skipcount -= 1
                    return True

                #Coordinator.stateRefresh()
            except:
                print "Exception in callback...ignoring ", \
                      traceback.print_exc()
                pass
        finally:
            cls.fromUserAction = True
            cls.stateRefresh()
        return True


    @classmethod
    def handle_save_dom(cls, dom):
        cls.notebook.removeTab(dom)
        
    @classmethod
    def handle_missing_dom(cls, managed_node, dom_name):
        cls.notebook.removeTabUsingName(managed_node,dom_name)

    @classmethod
    def handle_remove_dom(cls, dom):
        cls.notebook.removeTab(dom)

    @classmethod
    def handle_shutdown_dom(cls, dom):
        cls.notebook.removeTab(dom)
        
    @classmethod
    def handle_existing_dom(cls, dom):
        """ Handle refresh of existing dom"""
        # allow notebook to potentially clean up
        cls.notebook.refresh_tab(dom)

    @classmethod
    def handle_kill_dom(cls, dom):
        cls.notebook.removeTab(dom)
        

    @classmethod
    def translate_mode(cls, mode):
        node_type = None
        if mode == SummaryTab.POOL_SUMMARY:
            node_type = LeftNav.MANAGED_NODE
        elif mode == SummaryTab.NODE_SUMMARY:
            node_type = LeftNav.DOMAIN
        return node_type

    @classmethod
    def stateRefresh(cls, selection_widget=None):
        """Refreshes UI elements based on current dom selection"""
        if cls.nav == None: return

        (name, node_type, managed_node) = cls.get_selection()
        if node_type is None:
            return

        pausebutton = wtree.get_widget("pausebutton")
        pausemenu  = wtree.get_widget("menu_pause")
        pausepopup = wtree.get_widget("Popup_pause")

        remove = ("MenuRemove", "PopupRemove", "delete", "delete_domain")
        
        for widget in ("MenuSettings", "PopupSettings",
                       "Menu_edit_config_file",
                       "Popup_edit_config_file"):
            if wtree.get_widget(widget) is not None:
                wtree.get_widget(widget).set_sensitive(True)


        
        restore = node_type == LeftNav.MANAGED_NODE
        for action in ("restore",):
            wtree.get_widget(action + "button").set_sensitive(restore)
            wtree.get_widget("menu_" + action).set_sensitive(restore)
            wtree.get_widget("Popup_" + action).set_sensitive(restore)

        if node_type != LeftNav.DOMAIN or \
           (node_type == LeftNav.DOMAIN and  managed_node.isDom0(name)):

            for action in ("start", "pause", "reboot", "shutdown", "kill",
                           "snapshot", "show_console"):
                wtree.get_widget(action + "button").set_sensitive(False)
                wtree.get_widget("menu_" + action).set_sensitive(False)
                wtree.get_widget("Popup_" + action).set_sensitive(False)

            
            cls.fromUserAction = False
            pausebutton.set_active(False)
            cls.fromUserAction = True

            pausemenu.set_sensitive(False)
            pausepopup.set_sensitive(False)

            for widget in remove:
                wtree.get_widget(widget).set_sensitive(False)

            return
        else:
            dom_is_resident = managed_node.isResident(name)

            for widget in ("startbutton", "menu_start", "Popup_start"):
                wtree.get_widget(widget)\
                                          .set_sensitive(not dom_is_resident)

            for action in ("pause", "reboot", "shutdown", "kill", "snapshot",
                           "show_console"):
                wtree.get_widget(action + "button")\
                                        .set_sensitive(dom_is_resident)
                wtree.get_widget("menu_" + action)\
                                         .set_sensitive(dom_is_resident)
                wtree.get_widget("Popup_" + action)\
                                          .set_sensitive(dom_is_resident)

            isauto = False
            #_RESTRUCT
            dom = managed_node.get_dom(name)
            if dom.get_config() is not None and \
                   dom.get_config().filename.startswith(XENAUTO):
                isauto = True
            

            for widget in remove:
                if isauto or dom_is_resident:
                    wtree.get_widget(widget).set_sensitive(False)
                else:
                    wtree.get_widget(widget).set_sensitive(True)
            

            if dom_is_resident:
                state = dom["state"]
            else:
                state = None

            cls.fromUserAction = False
            if state:
                pausemenu.set_sensitive(True)
                pausepopup.set_sensitive(True)

                if (state[2] == 'p'):
                    pausebutton.set_active(True)
                    cls.notebook.pausedTab(dom, True)
                    
                    for widget in (pausebutton, pausemenu.child,
                                   pausepopup.child):
                        widget.set_label("Resume")
                else:
                    pausebutton.set_active(False)
                    cls.notebook.pausedTab(dom, False)
                    
                    for widget in (pausebutton, pausemenu.child,
                                   pausepopup.child):
                        widget.set_label("Pause")
            else:
                pausebutton.set_active(False)
                for widget in (pausemenu.child, pausepopup.child):
                    widget.set_sensitive(False)
            cls.fromUserAction = True

    # Track focus between summary table and left nav table
    @classmethod
    def focus_in(cls,widget, event):
        cls.last_focus_in = widget
        cls.stateRefresh()

        
    @classmethod
    def focus_out(cls,widget, event):
        cls.last_focus_out = widget

    @classmethod
    def get_selection_type(cls):
        if cls.last_focus_in == cls.nav.left_nav_view:
            return cls.nav.get_selection_type()
        else:
            (name, mode,managed_node)=cls.notebook.summary_tab.get_selection()
            node_type = cls.translate_mode(mode)
            return node_type
        
    @classmethod
    def get_selection(cls):
        if cls.last_focus_in == cls.nav.left_nav_view:
            return cls.nav.get_selection()
        else:
            (name, mode,managed_node)=cls.notebook.summary_tab.get_selection()
            node_type = cls.translate_mode(mode)
            return (name, node_type, managed_node)
        
    @classmethod
    def get_managed_node(cls):
        if cls.last_focus_in == cls.nav.left_nav_view:
            return cls.nav.get_managed_node()
        else:
            (name, mode,managed_node)=cls.notebook.summary_tab.get_selection()
            return managed_node


class NoteBook:
    """This class manages the gtk.NoteBook"""

    
    def __init__(self, left_nav):
        self.notebook = left_nav.wtree.get_widget("MainNotebook")
        self.left_nav = left_nav
        self.summary_tab = SummaryTab(wtree, left_nav)
        self.info_tab = InfoTab(wtree, left_nav)
        
        self.nbooklist = [0,0]
        self.terms = {}
        
        # add handlers : connecting through glade not working.
        self.notebook.connect("switch_page", \
                              self.on_notebook_switch_page)

    def _getDomToken(self, managed_node,dom_name):
        if managed_node is None or dom_name is None:
            return None
        else:
            return managed_node.hostname + ","  + dom_name


    def __getDomToken(self, dom):
        """ std way of generating unique id for a given dom"""
        if dom is None:
            #print "ERROR : getDomToken :dom is None"
            return None
        if dom.node is None :
            #print "ERROR : getDomToken :dom.node is None"
            return None
        return self._getDomToken(dom.node,dom.name)


    def showDomTab(self,managed_node, dom_name):
        #if self.getTabPos()== 1: 
        self.showInfoTab()


    def showInfoTab(self,widget=None,dom=None):
        if self.notebook.get_current_page() == 1:
            self.info_tab.refresh()
        else:
            self.notebook.set_current_page(1)
        

    def showSummaryTab(self, widget=None, manged_node=None):
        if self.notebook.get_current_page() == 0:
            self.summary_tab.refresh()
        else:
            self.notebook.set_current_page(0)
            
    def showTab(self, dom):
        """Displays the tab for the requested Dom or Dom ID"""
        dom_token = self.__getDomToken(dom)

        if dom_token in self.terms.keys():
            self.notebook.set_current_page(self.nbooklist.index(dom_token))
        else:
            self.__addTab(dom)
            self.pausedTab(dom, False)

    def removeTabUsingName(self, managed_node, dom_name):
        dom_token = self._getDomToken(managed_node,dom_name)
        if dom_token:
            self.__removeTab(dom_token)

    def removeTab(self, dom):
        dom_token = self.__getDomToken(dom)
        if dom_token is not None:
            self.__removeTab(dom_token)

    def __removeTab(self, dom_token):
        """Removes the tab for the requested Dom ID"""
        if dom_token in self.terms.keys():
            self.notebook.remove_page(self.nbooklist.index(dom_token))
            del self.terms[dom_token]
            self.nbooklist.remove(dom_token)


    def getTabPos(self):
        return self.notebook.get_current_page()

    def pausedTab(self, dom, state):
        """Set the sensitive state of the contents of a tab specified by
        the requested Dom or Dom ID"""

        dom_token = self.__getDomToken(dom)
        
        if dom_token in self.terms.keys():
            term_tab = self.terms[dom_token]
            term_tab.showPaused(state)

    def __addTab(self, dom):
        """Adds a console tab to the notebook for the specified Dom"""

        # assume that dom is only passed.
        #if not isinstance(dom, Dom):
        #    return

        dom_token = self.__getDomToken(dom)
        if dom_token:
            term_tab = TerminalTab(self, dom, dom_token)
            self.terms[dom_token] = term_tab
            self.nbooklist.append(dom_token)

        
    def closeTab(self, widget, dom_token):
        """Handles the request to close a notebook tab"""
        self.__removeTab(dom_token)


    # TODO : add the tab at the same position as itwas.
    def refresh_tab(self, dom):
        """ If a tab for the same dom with old id exists, clean it up"""
        
        if dom:
            dom_token = self.__getDomToken(dom)
            if dom_token in self.terms.keys():
                tab = self.terms[dom_token]
                if tab.dom_id != dom.id:
                    self.__removeTab(dom_token)
                    self.__addTab(dom)
                

    #### NOTE THIS IS NOT CALLED FROM ANYWHERE.
    #### MAY BE RETIRED (NOT CUTOVER FOR MULTINODE)
    def cleanup(self):
        """Goes through the list of notebook tabs and checks if there is a
        corresponding Dom and what it's state is, in order to cleanup tabs
        that no longer should be displayed"""

        changes = 0

        
        for dom_token in self.terms.keys():
            try:
                dom = self.terms[dom_token]
                dom.refresh()
            except NodeException:
                self.__removeTab(dom_token)
                changes += 1
                continue
            #except Exception, e:
            #    print 'domid: ',domid
            #    continue

        if changes:
            # create new tab if required (after reboot)
            self.left_nav.domselected()

            
    def on_notebook_switch_page(self, widget, page, page_no):
        """if the user switched to info tab, populate with the
        dom information """
        if page_no == 1:
            self.info_tab.refresh()
        else:
            self.info_tab.set_default_title()
        if page_no == 0:
            self.summary_tab.refresh()
        else:
            self.summary_tab.set_default_title()

        return True



class TerminalTab:
    """ Class represents terminal tab for a dom """

    def __init__(self, notebook, dom, dom_token):
        
        self.notebook = notebook
        
        
        self.term = vte.Terminal()
        self.savedTerm = None

        #self.dom = dom
        self.host_name = dom.node.hostname
        self.dom_id = dom.id 
        self.dom_name = dom.name
        # for now simple check
        self.is_hvm = (dom["kernel"] is not None) and \
                      dom["kernel"].endswith("hvmloader")
        self.vnc=dom["vnc"]
        self.vnc_display = dom.id


        # get the display if found in dom info
        if dom["vncdisplay"] is not None:
            self.vnc_display = dom["vncdisplay"]
            
        #cfg = dom.get_config()
        #if cfg is not None and cfg["vncdisplay"]:
        #    self.vnc_display = cfg["vncdisplay"]
        
        self.dom_token = dom_token
        
        managed_node = dom.node

        # determine the architecture for the node        
        if re.search('64', managed_node.environ[prop_env_MACHINE_TYPE]):
            arch_libdir = 'lib64'
        else:
            arch_libdir = 'lib'


        if self.vnc == 1:
            if not managed_node.is_remote():
                # try vnc
                self.term.fork_command("vncviewer",
                                      ("vncviewer", 
                                       ":" + str(self.vnc_display)))
            else:
                # try it through SSH (should we try without SSH ?)
                self.term.fork_command("ssh",
                                       ("vncviewer","-X",
                                        managed_node.username+ "@" +
                                        managed_node.hostname,
                                        ":"+str(self.vnc_display)))
        else:
        
           if not managed_node.is_remote():            
               self.term.fork_command("/usr/"+arch_libdir+"/xen/bin/xenconsole",
                                      ("xenconsole", str(self.dom_id)))
           else:
               hostname = managed_node.hostname
               if managed_node.username is not None:
                   hostname = managed_node.username + "@" + hostname
               self.term.fork_command("ssh",
                                      ("xenconsole","-t",
                                       managed_node.username+ "@" +
                                       managed_node.hostname,
                                       "/usr/"+arch_libdir+"/xen/bin/xenconsole",
                                       str(self.dom_id)))

        
        self.tbox = gtk.VBox()
        self.tbox.add(self.term)
        self.tbox.show_all()

        self.nbox = gtk.HBox()
        self.event_box = gtk.EventBox()
        self.tab_name = gtk.Label(self.dom_name)
        self.event_box.add(self.tab_name)
        self.nbox.pack_start(self.event_box)
        self.closebutton = gtk.Button()
        self.closebutton.set_relief(gtk.RELIEF_NONE)
        self.closebutton.set_size_request(20, 17)
        self.closeimage = gtk.Image()
        self.closeimage.set_from_stock(gtk.STOCK_CLOSE,
                                       gtk.ICON_SIZE_MENU)
        self.closebutton.add(self.closeimage)
        self.closebutton.connect("clicked", self.__closeTab, self.dom_token)

        """ Some other time...
        infobutton = gtk.Button()
        infobutton.set_relief(gtk.RELIEF_NONE)
        infobutton.set_size_request(28, 28)
        info_image = gtk.Image()
        info_image.set_from_stock(gtk.STOCK_INFO,
                                  gtk.ICON_SIZE_MENU)
        infobutton.add(info_image)
        infobutton.connect("clicked", self.notebook.showInfoTab, dom)
        nbox.pack_start(infobutton)
        """
        self.nbox.pack_start(self.closebutton)
        self.tool_tip = gtk.Tooltips()

        self.tool_tip.set_tip(self.event_box, managed_node.hostname)
        self.nbox.show_all()

        # add the tab to the notebook
        # Kludge : done like this because append page requires
        # tab content (tbox etc)
        page_id = self.notebook.notebook.append_page(self.tbox, self.nbox)
        self.notebook.notebook.set_current_page(page_id)


    def __closeTab(self, widget, dom_token):
        self.notebook.closeTab(widget, dom_token)

    def showPaused(self,paused):
        """
        show paused or normal console for a running dom
        """
        
        if paused:
            if not self.savedTerm:
                self.savedTerm = self.term
                self.tbox.remove(self.term)
                
                label = gtk.Label()
                label.set_markup('<span size="xx-large">Paused</span>')
                self.tbox.add(label)
                self.tbox.show_all()
                
        elif self.savedTerm:
            self.savedTerm = None
            self.tbox.remove(self.tbox.get_children()[0])
            self.tbox.add(self.term)


class InfoTab:
    """ Class representing information tab for current selection """
    
    def __init__(self, wtree, left_nav):
        self.left_nav = left_nav
        self.info_tree_model = gtk.TreeStore(gobject.TYPE_STRING,
                                             gobject.TYPE_STRING,
                                             gobject.TYPE_BOOLEAN
                                             )
        self.info_text = wtree.get_widget("info_text_view")
        self.info_text_window=wtree.get_widget("info_text_view_window")

        # replace the text view the htmlviewer
        self.info_text_window.remove(self.info_text)
        self.info_text = HtmlTextView()
        self.info_text_window.add(self.info_text)
        
        self.info_tree_view = wtree.get_widget("info_tree")
        self.tab_name = wtree.get_widget("info_tree_tab_label")
        textrenderer = gtk.CellRendererText()

        textrenderer.set_property("weight",700)

        column = gtk.TreeViewColumn("Configuration", textrenderer, text=0,
                                    weight_set=2)
        self.info_tree_view.append_column(column)
        
        textrenderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn("", textrenderer, text=1, weight_set=2)
        self.info_tree_view.append_column(column)

        """
        column = gtk.TreeViewColumn("Value", textrenderer, text=2)
        self.info_tree_view.append_column(column)
        """
        self.info_tree_view.set_model(self.info_tree_model)
        self.info_tree_view.show()


    def disable_tab(self):
        # just disable label. actual disabling does not work
        self.tab_name.set_sensitive(False)

    def enable_tab(self):
        # just disable label. actual disabling does not work
        self.tab_name.set_sensitive(True)


    def refresh(self):
        (name, node_type, managed_node) = self.left_nav.get_selection()
        


        if node_type == LeftNav.IMAGE or node_type == LeftNav.IMAGE_STORE:
            self.info_text.set_property("visible",True)
            self.info_text.set_wrap_mode(gtk.WRAP_WORD)
            self.info_text.set_editable(False)
            self.info_text_window.set_property("visible",True)
            self.info_tree_view.set_property("visible", False)
            text = ""
            if node_type == LeftNav.IMAGE:
                # populate image description
                image_name=name
                filename = image_store.get_image_desc(image_name) 
                text = "No description file (%s) for %s " % (filename,name)
                if os.path.exists(filename):
                    file = local_node.node_proxy.open(filename)
                    file = local_node.node_proxy.open(filename)
                    lines = file.readlines()
                    text = "".join(lines)
                    file.close()

            elif node_type == LeftNav.IMAGE_STORE:
                text = """<body>
                <span style=\"font-size: 200%; font-family: serif; color:blue;text-align: 
                center;background-color:lightgray\">
                The Image Store
                </span>
                <br/>
                <span style=\"font-size: 125%; font-family: serif; 
                color:#0000FF;text-align: left\">
                Store Location:
                </span>"""
                text = text + image_store.get_store_location()
                text = text + """<br/>
                <span style=\"font-size: 125%; font-family: serif;color:#0000FF;text-align: left\">
                Available Images:
                </span>
                <ol >
                """
                images = image_store.list()
                if images:
                    images.sort()
                for image in images :
                    text = text + "<li>" + image + "</li>" + "\n"
                footer = '</ol></body>'
                text = text + footer
                
            #text_buffer = gtk.TextBuffer()
            #text_buffer.set_text(text)
            #self.info_text.set_buffer(text_buffer)
            if text is not None:
                if text.find("<body>") == -1:
                    #text= "<body>\n%s</body>\n" % text
                    #pass
                    text_buffer = gtk.TextBuffer()
                    text_buffer.set_text(text)
                    self.info_text.set_buffer(text_buffer)
                else:
                    self.info_text.display_html(text)
            return

        self.info_text.set_property("visible",False)
        self.info_text_window.set_property("visible",False)
        self.info_tree_view.set_property("visible", True)
        
        if managed_node is None:
            return

        self.info_tree_model.clear()
        if  node_type != LeftNav.DOMAIN and node_type !=LeftNav.IMAGE:
            env_iter  = self.info_tree_model.append(None,["Environment","", True])
            self.populateEnvInfo(env_iter, managed_node)
            self.info_tree_view.expand_all()
            return

        # self.tab_name.set_text("Information" + "\n" + name)
        gen_iter  = self.info_tree_model.append(None,["General","", True])
        boot_iter = self.info_tree_model.append(None,["Boot","", True])
        res_iter  = self.info_tree_model.append(None,["Resource","", True])

        dom = managed_node.get_dom(name)
        self.populateGeneralInfo(gen_iter, dom)
        self.populateBootInfo(boot_iter, dom)
        self.populateResourceInfo(res_iter,dom)
        self.info_tree_view.expand_all()

    def set_default_title(self):
        pass
        # self.tab_name.set_text("Information")

    def populateEnvInfo(self, iter, managed_node):
        for name in managed_node.environ:
            value = managed_node.environ[name]
            if type(value) == int:
                value = str(value)
            self.info_tree_model.append(iter, [name,value, False])
            

    def append_info_node(self, iter, dom,  name, param):
        """ get the value from dom index and append Name / Value in the
        tree at 'iter' position."""
        
        value = dom[param]

        if value:
            if type(value) == int:
                value = str(value)
            self.info_tree_model.append(iter, [name,value,False])
        else:
            config = dom.get_config()
            if config:
                value = config[param]
                if value:
                    if type(value) == int:
                        value = str(value)
                    self.info_tree_model.append(iter, [name,value,False])
                    return

            self.info_tree_model.append(iter, [name,"N/A", False])

    
            
    def populateGeneralInfo(self, iter, dom):
        for name, param in ( ("Name", "name"),
                             ("FileName", "filename"),
                             ):
            self.append_info_node(iter, dom, name, param)
        

    def populateBootInfo(self, iter, dom):
        for name, param in ( 
                             ("Kernel", "kernel"),
                             ("Ramdisk", "ramdisk"),
                             ("Bootloader", "bootloader"),

                             ("On Crash", "on_crash"),
                             ("On Reboot", "on_reboot"),):
            self.append_info_node(iter, dom,name, param)


    def populateResourceInfo(self, iter, dom):
        for name, param in ( 
                             ("Memory", "memory"),
                             ("CPU", "vcpus"),
                             ("Network", "vif"),
                             ("Disks", "disk"),
                             ):
            self.append_info_node(iter, dom, name, param)


class CustomTreeViewColumn(gtk.TreeViewColumn):
    def __init__(self,title=None,
                 renderer=None,
                 display_modes = [],
                 data_source = None,
                 data_key=None,format=None, hide_if_snapshot=False):
    
        gtk.TreeViewColumn.__init__(self,title, renderer)
        self.data_source=data_source
        self.data_key = data_key
        self.format = format
        self.display_modes = display_modes
        self.hide_if_snapshot = hide_if_snapshot # this is lame
        self.set_visible(False)
        self.set_resizable(True)
        self.x_props = {}

    def set_x_porp(self, key, value):
        self.x_props[key] = value

    def get_x_prop(self,key):
        if self.x_props.has_key(key):
            return self.x_props[key]
        else:
            return None

    def get_x_props(self):
        return self.x_props

class ColumnToolTip(TreeViewTooltips):
    
    def __init__(self, tip_column):
        self.tip_col = tip_column
        # call base class init
        TreeViewTooltips.__init__(self)
        
    def get_tooltip(self, view, column, path):
        if column is self.tip_col:
            return '<i>Total/Running/Paused/Crashed</i>'

    def XX_location(self, x, y, w, h):
        # rename me to "location" so I override the base class
        # method.  This will demonstrate being able to change
        # where the tooltip window popups, relative to the
        # pointer.

        # this will place the tooltip above and to the right
        return x + 10, y - (h + 10)

class SummaryTab:
    """ Summary information of all running doms"""
    (M_NODE,DOM_INFO,SNAPSHOT, MODE) = range(4)
    (POOL_SUMMARY, NODE_SUMMARY, ALL) = range(3)
    (COL_ID_MEM_PCT, COL_ID_CPU_PCT,COL_ID_VM_MEM_PCT, COL_ID_VM_CPU_PCT,
     COL_ID_SERVER_NAME, COL_ID_DOM_NAME, COL_ID_CONN_STATUS,
     COL_ID_SERVER_MEM) = range(8)
    
    # Can add some more information about the node.
    def __init__(self, wtree, left_nav):
        self.left_nav = left_nav

        self.statmodel = gtk.ListStore(gobject.TYPE_PYOBJECT,#manged_node
                                       gobject.TYPE_PYOBJECT,#dom
                                       gobject.TYPE_PYOBJECT,#snapshot
                                       gobject.TYPE_INT
                                       )
        
        self.statview = wtree.get_widget("statTable")
        self.tab_name = wtree.get_widget("summaryTabLabel")

        self.statview.connect("row-activated",
                              self.handle_row_activated)

        self.statview.get_selection().connect("changed",
                                              self.handle_selection)


        # did not work. gtk 2.10 and later.
        #self.statview.set_grid_lines(gtk.TREE_VIEW_GRID_LINES_BOTH)
                                     
        # Renderers
        textrenderer = gtk.CellRendererText()
        bold_textrenderer = gtk.CellRendererText()
                
        bold_textrenderer.set_property("weight",700)
        bold_textrenderer.set_property("weight-set",True)
        #bold_textrenderer.set_property("scale",pango.SCALE_LARGE)
        

        ct_textrenderer = gtk.CellRendererText()
        ct_textrenderer.set_property("xalign", 0.5)

        rt_textrenderer = gtk.CellRendererText()
        rt_textrenderer.set_property("xalign", 1.0)

        bar_renderer = gtk.CellRendererProgress()

        pix_renderer = gtk.CellRendererPixbuf()

        column = CustomTreeViewColumn("Server", bold_textrenderer,
                                      [self.POOL_SUMMARY],
                                      self.M_NODE, 'hostname')
        column.set_cell_data_func(bold_textrenderer,
                                  self.summary_cell_data)
        self.statmodel.set_sort_func(self.COL_ID_SERVER_NAME,
                                     self.summary_cmp, column)
        #self.statmodel.set_sort_column_id(self.COL_ID_SERVER_NAME,
        #                                  gtk.SORT_ASCENDING)
        column.set_sort_column_id(self.COL_ID_SERVER_NAME)
        self.statview.append_column(column)

        column = CustomTreeViewColumn("Connection", pix_renderer,
                                      [self.POOL_SUMMARY],
                                      self.SNAPSHOT, 'NODE_STATUS')
        column.set_cell_data_func(pix_renderer,
                                  self.summary_cell_data)
        self.statmodel.set_sort_func(self.COL_ID_CONN_STATUS,
                                     self.summary_cmp, column)
        column.set_sort_column_id(self.COL_ID_CONN_STATUS)
        self.statview.append_column(column)

        column = CustomTreeViewColumn("VM Summary", ct_textrenderer,
                                      [self.POOL_SUMMARY],
                                      self.SNAPSHOT,
                                      'VM_SUMMARY'
                                      )
        column.set_cell_data_func(ct_textrenderer,
                                  self.summary_cell_data)

        
        self.stat_view_tt = ColumnToolTip(column)
        self.stat_view_tt.add_view(self.statview)
        self.statview.append_column(column)

        
        column = CustomTreeViewColumn("VM CPU(%)", bar_renderer,
                                      [self.POOL_SUMMARY],
                                      self.SNAPSHOT, 'VM_TOTAL_CPU(%)',
                                      'float%')
        column.set_cell_data_func(bar_renderer,
                                  self.summary_cell_data)
        column.set_clickable(True)
        self.statmodel.set_sort_func(self.COL_ID_VM_CPU_PCT,
                                     self.summary_cmp, column)
        #self.statmodel.set_sort_column_id(self.COL_ID_VM_CPU_PCT,
        #                                  gtk.SORT_DESCENDING)
        column.set_sort_column_id(self.COL_ID_VM_CPU_PCT)

        self.statview.append_column(column)

        column = CustomTreeViewColumn("VM Mem(%)", bar_renderer,
                                      [self.POOL_SUMMARY],
                                      self.SNAPSHOT, 'VM_TOTAL_MEM(%)',
                                      'float%')
        column.set_cell_data_func(bar_renderer,
                                  self.summary_cell_data)
        column.set_clickable(True)
        self.statmodel.set_sort_func(self.COL_ID_VM_MEM_PCT,
                                     self.summary_cmp, column)
        #self.statmodel.set_sort_column_id(self.COL_ID_VM_MEM_PCT,
        #                                  gtk.SORT_ASCENDING)
        column.set_sort_column_id(self.COL_ID_VM_MEM_PCT)

        self.statview.append_column(column)


##         column = CustomTreeViewColumn("Total VMs", textrenderer,
##                                       [self.POOL_SUMMARY],
##                                       self.SNAPSHOT, 'TOTAL_VMS')
##         column.set_cell_data_func(textrenderer,
##                                   self.summary_cell_data)
##         self.statview.append_column(column)

##         column = CustomTreeViewColumn("Running VMs", textrenderer,
##                                       [self.POOL_SUMMARY],
##                                       self.SNAPSHOT, 'RUNNING_VMs')
##         column.set_cell_data_func(textrenderer,
##                                   self.summary_cell_data)
##         self.statview.append_column(column)

##         column = CustomTreeViewColumn("Paused VMs", textrenderer,
##                                       [self.POOL_SUMMARY],
##                                       self.SNAPSHOT, 'PAUSED_VMs')
##         column.set_cell_data_func(textrenderer,
##                                   self.summary_cell_data)
##         self.statview.append_column(column)


##         column = CustomTreeViewColumn("Crashed VMs", textrenderer,
##                                       [self.POOL_SUMMARY],
##                                       self.SNAPSHOT, 'CRASHED_VMs')
##         column.set_cell_data_func(textrenderer,
##                                   self.summary_cell_data)
##         self.statview.append_column(column)


        column = CustomTreeViewColumn("Server CPU(s)", textrenderer,
                                      [self.POOL_SUMMARY],
                                      self.SNAPSHOT, 'SERVER_CPUs')
        column.set_cell_data_func(textrenderer,
                                  self.summary_cell_data)
        self.statview.append_column(column)


        
        column = CustomTreeViewColumn("Server Mem", rt_textrenderer,
                                      [self.POOL_SUMMARY],
                                      self.SNAPSHOT, 'SERVER_MEM')
        column.set_cell_data_func(rt_textrenderer,
                                  self.summary_cell_data)

        column.set_clickable(True)
        self.statmodel.set_sort_func(self.COL_ID_SERVER_MEM,
                                     self.summary_cmp, column)
        #self.statmodel.set_sort_column_id(self.COL_ID_SERVER_MEM,
        #                                  gtk.SORT_ASCENDING)
        column.set_sort_column_id(self.COL_ID_SERVER_MEM)



        self.statview.append_column(column)

        column = CustomTreeViewColumn("Version", textrenderer,
                                      [self.POOL_SUMMARY],
                                      self.SNAPSHOT, 'XEN_VER')
        column.set_cell_data_func(textrenderer,
                                  self.summary_cell_data)
        self.statview.append_column(column)


        column = CustomTreeViewColumn("Id", textrenderer,
                                      [self.NODE_SUMMARY],
                                      self.DOM_INFO, 'domid')
        column.set_cell_data_func(textrenderer,
                                  self.summary_cell_data)
        self.statview.append_column(column)

        column = CustomTreeViewColumn("Name", bold_textrenderer,
                                      [self.NODE_SUMMARY],
                                      self.DOM_INFO, 'name')
        column.set_cell_data_func(bold_textrenderer,
                                  self.summary_cell_data)
        self.statmodel.set_sort_func(self.COL_ID_DOM_NAME,
                                     self.summary_cmp, column)
        #self.statmodel.set_sort_column_id(self.COL_ID_DOM_NAME,
        #                                  gtk.SORT_ASCENDING)
        column.set_sort_column_id(self.COL_ID_DOM_NAME)
        self.statview.append_column(column)

        column = CustomTreeViewColumn("State", textrenderer,
                                      [self.NODE_SUMMARY],
                                      self.DOM_INFO, 'state')
        column.set_cell_data_func(textrenderer,
                                  self.summary_cell_data)
        self.statview.append_column(column)


        column = CustomTreeViewColumn("CPU Time(sec)", rt_textrenderer,
                                      [self.NODE_SUMMARY],
                                      self.DOM_INFO, 'cpu_time','float', True)
        column.set_cell_data_func(rt_textrenderer,
                                  self.summary_cell_data)
        self.statview.append_column(column)

        column = CustomTreeViewColumn("Memory", rt_textrenderer,
                                      [self.NODE_SUMMARY],
                                      self.DOM_INFO, 'memory', 'float',True)
        column.set_cell_data_func(rt_textrenderer,
                                  self.summary_cell_data)
        self.statview.append_column(column)



        column = CustomTreeViewColumn("CPU (%)", bar_renderer,
                                      [self.NODE_SUMMARY],
                                      self.SNAPSHOT, 'CPU(%)','float%')
        column.set_cell_data_func(bar_renderer,
                                  self.summary_cell_data)
        column.set_clickable(True)
        #column.connect("clicked",self.col_clicked)
        self.statmodel.set_sort_func(self.COL_ID_CPU_PCT,
                                     self.summary_cmp, column)
        #self.statmodel.set_sort_column_id(self.COL_ID_CPU_PCT,
        #                                  gtk.SORT_ASCENDING)
        column.set_sort_column_id(self.COL_ID_CPU_PCT)
        self.statview.append_column(column)


        column = CustomTreeViewColumn("MEM (%)", bar_renderer,
                                      [self.NODE_SUMMARY],
                                      self.SNAPSHOT, 'MEM(%)','float%')
        column.set_clickable(True)
        self.statmodel.set_sort_func(self.COL_ID_MEM_PCT,
                                     self.summary_cmp, column)
        #self.statmodel.set_sort_column_id(self.COL_ID_MEM_PCT,
        #                                  gtk.SORT_ASCENDING)
        column.set_sort_column_id(self.COL_ID_MEM_PCT)
        #column.connect("clicked",self.col_clicked)
        column.set_cell_data_func(bar_renderer,
                                  self.summary_cell_data)
        self.statview.append_column(column)


        #dummy column for alignment.. may not be required.
        column = CustomTreeViewColumn("", textrenderer,
                                      [self.ALL],
                                      self.DOM_INFO, 'Empty')
        column.set_cell_data_func(textrenderer,
                                  self.summary_cell_data)
        self.statview.append_column(column)

        self.statview.set_model(self.statmodel)
        self.statview.show()


    def get_selection(self):
        selection = self.statview.get_selection()
        (model, iter) = selection.get_selected()
        if iter:
            mode = model.get_value(iter, self.MODE)
            if mode == self.POOL_SUMMARY:
                m_node = model.get_value(iter, self.M_NODE)
                name = m_node.hostname
            elif mode == SummaryTab.NODE_SUMMARY:
                dom_info = model.get_value(iter, self.DOM_INFO)
                name = dom_info.name
            managed_node = model.get_value(iter, self.M_NODE)

            return (name, mode, managed_node)
        return (None, None, None)

    def handle_selection(self, widget = None):
        Coordinator.stateRefresh(widget)

    def handle_row_activated(self,treeview,path,column):
        # need to fig out if we are in server pool or at a managed node.
        selection = treeview.get_selection()
        (model, iter) = selection.get_selected()
        if iter:
            mode = model.get_value(iter, self.MODE)
            name = None
            if mode == self.POOL_SUMMARY:
                m_node = model.get_value(iter, self.M_NODE)
                name = m_node.hostname
                left_nav.set_selection(name,
                                       left_nav.MANAGED_NODE,
                                       "row-activated")
            else:
                dom_info = model.get_value(iter, self.DOM_INFO)
                if not dom_info.isDom0():
                    name = dom_info.name
                    left_nav.set_selection(name,
                                       left_nav.DOMAIN,
                                       "row-activated")

    def disable_tab(self):
        self.tab_name.set_sensitive(False)

    def enable_tab(self):
        self.tab_name.set_sensitive(True)



    def summary_cmp(self, model, iter1, iter2, column=None):
        #print "summary cmp called"
        try:
            v1 = self.get_text(model, column, iter1)
            v2 = self.get_text(model, column, iter2)
        except:
            print traceback.print_exc()
            raise
            
        if v1 < v2:
            return -1
        elif v1 == v2:
            return 0
        else:
            return 1
        

    def col_clicked(self, column):
        print "Column clicked"

    def get_state_string(self,state_str):
        state = XenDomain.get_state(state_str)
                                   
        if state == XenDomain.RUNNING: return "Running"
        elif state ==  XenDomain.BLOCKED : return "Blocked"
        elif state ==  XenDomain.PAUSED : return "Paused"
        elif state == XenDomain.SHUTDOWN: return "Shutdown"
        elif state == XenDomain.CRASHED: return "Crashed"
        elif state == XenDomain.UNKNOWN: return "Unknown"
        
        return "Unknown"
        


    # Call back function to get the data and show it.
    def summary_cell_data(self,column,cell,model,iter):
        text = self.get_text(model, column, iter, cell)
        r = None    
        renderers = column.get_cell_renderers()
        r =renderers[0]
        if r is not None and isinstance(r, gtk.CellRendererProgress):
            cell.set_property('value',0)
            if text is not None and text is not '':
                cell.set_property('value',float(text))

        if text == None or text == '':
            text='N/A'

        if r is not None and isinstance(r, gtk.CellRendererPixbuf):
            pb = self.get_connection_pb(text)
            cell.set_property('pixbuf', pb)
        else:
            cell.set_property('text',text)

    def get_connection_pb(self,state):
        if state == "Connected":
            pb = dialogs.connected_pb
        else:
            pb = dialogs.disconnected_pb

        return pb
    
    def get_text(self, model, column, iter, cell=None):
        format = None
        (pos, key, format) = (column.data_source,column.data_key,column.format)
        dp = model.get_value(iter, pos)
        text = ''
        if dp is not None:
            if pos == 1: # dom access
                text = dp[key]
            elif pos == 2: # snapshot access
                if dp.has_key(key):
                    text = dp[key]
            elif pos == 0:
                text = getattr(dp, key)


        if text is None:
            text = ''

        if key == 'Empty':
            text = ' ' # a space so N/A is not displayed
        # special handling for name
        if (key=='name' or key=='NAME') and text=='Domain-0':#uggh,hard coding
            managed_node = model.get_value(iter,0)
            text = managed_node.hostname
            
        if key=='state' or key=='STATE':
            text = self.get_state_string(text)

        if key=='VM_SUMMARY' and dp is not None: # construct the string
            text = ''
            for k in ('TOTAL_VMs', 'RUNNING_VMs', 'PAUSED_VMs', 'CRASHED_VMs'):
                if dp.has_key(k):
                    if k == 'RUNNING_VMs':
                        v = dp[k]
                        n = int(v) - 1  # exclude domain-0
                        text = text + str(n)
                    else:
                        text = text + str(dp[k])
                else:
                    text = text + '?'
                text = text + '/'
            text = text[:-1]
                
        if text is not None and text != '' and format is not None:
            if format.find('float') == 0:
                text= "%12.2f" % float(text)            

        return text

    # refresh can be done smartly(in place update) to reduce flicker
    
    def refresh(self):
        """Refreshes the statistics on the home tab"""
        (name, node_type,managed_node)  = self.left_nav.get_selection()
        # save current selection and restore it once the refresh is done.
        path = None
        old_mode = None
        new_mode = None
        old_managed_node = None
        selection = self.statview.get_selection()
        if selection:
            (tm, ti) = selection.get_selected()
            if ti is not None:
                path = tm.get_path(ti)
                # special check to see if the view is changing
                old_mode = tm.get_value(ti, self.MODE)
                old_managed_node = tm.get_value(ti, self.M_NODE)

        self.statmodel.clear()        
        if node_type == left_nav.SERVER_POOL:
            self._populate_pool_summary()
            new_mode = self.POOL_SUMMARY
        else:
            # old call
            self._populate_node_summary(managed_node)
            new_mode = self.NODE_SUMMARY
            
        if selection is not None and path is not None:
            if old_mode == new_mode:
                #if managed_node is old_managed_node:
                selection.select_path(path)
                

    def _populate_pool_summary(self):
        node_list = manager.getNodeNames()
        node_list.sort()
        for m_name in node_list:
            m_node = manager.getNode(m_name)
            if m_node is None : continue
            node_status = "Unknown"
            if m_node.is_authenticated():
                node_status = "Connected"
            else:
                node_status = "Not Connected"

            #if m_node.is_in_error():
            #    node_status = "Error"
            
        
            if m_node is not None:
                if m_node.is_authenticated() and not m_node.is_in_error():
                    node_snapshot = m_node.get_metrics()
                    if node_snapshot is None:
                        node_snapshot = {}
                        node_snapshot["NODE_NAME"] = m_node.hostname
                        
                    node_snapshot["NODE_STATUS"]= node_status
                    node_snapshot["TOTAL_VMs"] = len(m_node.get_dom_names())-1
                else:
                    node_snapshot = {"NODE_NAME":m_node.hostname,
                                     "NODE_STATUS":node_status
                                     }
                self.adjust_columns(self.POOL_SUMMARY,
                                    len(node_snapshot.keys()) > 0)
                
                iter = self.statmodel.insert_before(None, None)
                self.statmodel.set(iter,
                                   self.M_NODE, m_node,
                                   self.DOM_INFO, None,
                                   self.SNAPSHOT, node_snapshot,
                                   self.MODE, self.POOL_SUMMARY
                                   )
                

    def adjust_columns(self,mode, snapshot_available):
        
        for col in self.statview.get_columns():
            show_col = (mode in col.display_modes or \
                       self.ALL in col.display_modes)
            col.set_visible(show_col)
            if not show_col:
                continue
            
            if snapshot_available : # snapshot available
                if col.data_source == self.SNAPSHOT:
                    col.set_visible(True)
                    
                if col.hide_if_snapshot==True:
                    col.set_visible(False)
                    
            else: # no snapshot
                if col.data_source == self.SNAPSHOT:
                    col.set_visible(False)
                    
                if col.hide_if_snapshot==True:
                    col.set_visible(True)



    def _populate_node_summary(self, managed_node):
        
        if managed_node is None : return
        if not managed_node.is_authenticated(): return
        if managed_node.is_in_error():
            return
        
        
        snapshot = managed_node.get_metrics()
        if snapshot is not None:
            snapshot_keys = snapshot.keys()
            snapshot_len = len(snapshot_keys)
        else:
            snapshot_keys = []
            snapshot_len = 0

        self.adjust_columns(self.NODE_SUMMARY, snapshot_len > 0)
        doms = managed_node.get_dom_names()
        doms.sort()

        for dom_name in doms:
            dom = managed_node.get_dom(dom_name)
            if dom and dom.is_resident:
                if snapshot_len > 0 and dom_name in snapshot_keys:
                    snapshot_dom = snapshot[dom_name]
                else:
                    snapshot_dom = None

                iter = self.statmodel.insert_before(None, None)
                self.statmodel.set(iter,
                                   self.M_NODE, managed_node,
                                   self.DOM_INFO, dom,
                                   self.SNAPSHOT, snapshot_dom,
                                   self.MODE, self.NODE_SUMMARY
                                       )
            


    def set_default_title(self):
        pass
        #self.tab_name.set_text("Summary")
        

class Timer:
    """This class initializes a GTK call back loop that runs one of it's
    methods on a regular basis to perform refreshes and such operations"""



    def __init__(self):
        self.timeoutid = gobject.timeout_add(2000, self.__callback)

    def __callback(self):
        """This is the callback that handles the refresh"""
        return Coordinator.handle_timer_callback()



#
# function definitions
#

def validate_node_selected():
    if Coordinator.get_selection_type() == LeftNav.MANAGED_NODE:
        return True
    else:
        showmsg("Select a host for this operation")
        return False


def quitProgram(*args):
    """Quit the execution of the program"""

    gtk.main_quit()

def show_about(*args):
    """handler that shows the about dialog"""

    wtree.get_widget('AboutDialog').show()

def hide_about(widget, *args):
    """handler that hides the about dialog"""

    widget.hide()
    return True


def savedom(widget):
    """Take Snapshot of a running dom. (a.k.a save)"""

    (name,node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.DOMAIN:
        checkAndSetDefaultPaths(wtree, managed_node)
        dom = managed_node.get_dom(name)
        snap_dir = managed_node.config.get(utils.XMConfig.PATHS,prop_snapshots_dir)
        default_ext = managed_node.config.get(utils.XMConfig.PATHS,prop_snapshot_file_ext);
        snap_file = dom.name
        if default_ext is not None:
            snap_file += default_ext
        mainwin = wtree.get_widget('MainWindow')
        (res, fname) = file_selection(managed_node,
                                      "Save Snapshot", "save", snap_dir,
                                      snap_file,
                                      parentwin = mainwin)
        if res and fname:
            try:
                show_wait_cursor()
                try:
                    dom._save(fname)
                except socket.error, e:
                    print "error ", e
            finally:
                hide_wait_cursor()
                
        Coordinator.handle_save_dom(dom)
        left_nav.refresh_doms()

def restoredom(widget):
    """Restore a dom from a file."""

    if not validate_node_selected():
        return
    managed_node = Coordinator.get_managed_node()
    checkAndSetDefaultPaths(wtree, managed_node)
    snap_dir = managed_node.config.get(utils.XMConfig.PATHS,prop_snapshots_dir)
    mainwin = wtree.get_widget('MainWindow')
    (res, fname) = file_selection(managed_node,
                                  "Restore from Snapshot","open",
                                  snap_dir,
                                  parentwin = mainwin)
    if res and fname:
        if managed_node is not None:
            try:
                show_wait_cursor()
                managed_node.restore_dom(fname)
                managed_node.refresh()
                left_nav.refresh_doms()
            finally:
                hide_wait_cursor()

def rebootdom(widget):
    """Reboot the currently selected domain"""
    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)
        if confirmation("Reboot " + dom.name + " on " + managed_node.hostname + "?"):
            try:
                show_wait_cursor()
                dom._reboot()
                managed_node = dom.node
                managed_node.refresh()
                left_nav.refresh_doms()
            finally:
                hide_wait_cursor()
            

def shutdowndom(widget):
    """shutdown the currently selected domain"""

    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)

        if confirmation("Shutdown " + dom.name+ " on " + managed_node.hostname + "?"):
            try:
                show_wait_cursor()
                dom._shutdown()
                managed_node = dom.node
                left_nav.refresh_doms()
            finally:
                hide_wait_cursor()

            

def startdom(widget):
    """Start the selected domain"""
    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)
    else:
        return
    

    if not managed_node.isResident(dom.name):
        #_RESTRUCT
        #errmsgs = left_nav.cursel.validate()

        #if errmsgs:
        #    errmsg = "Can't start the domain because of the following " + \
        #             "problems:\n"

        #    for msg in errmsgs:
        #        errmsg += " - " + msg + "\n"

        #    showmsg(errmsg)
        #    return
        try:
            try:
                show_wait_cursor()
                dom._start()
            finally:
                hide_wait_cursor()
        except Exception, e:
            showmsg("Error starting " + dom.name +". "+ str(e))
            return


        # _RESTRUCT
        managed_node = dom.node
        left_nav.refresh_doms()


def killdom(widget):
    """Kills the currently selected domain"""
    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)

        if (confirmation("Kill " + dom.name + " on " + managed_node.hostname+ "?")):
            # sometime the xen destroy gives socket error after destorying vm
            # lets ignore it for now.
            try:
                show_wait_cursor()
                try:
                    dom._destroy()                
                    #left_nav.refresh()
                except (socket.error, xmlrpclib.Fault):
                    pass
            finally:
                hide_wait_cursor()

            
            # _RESTRUCT
            managed_node = dom.node
            left_nav.refresh_doms()
            Coordinator.handle_kill_dom(dom)

def pausedom(widget):
    """Pause the currently selected domain"""

    # We use set_active on the pause button, which may fire
    # unnecessary pasuedom calls, to prevent this the callback from the
    # UI actually sets fromUserAction to True.
    # Note : There must be a better way of doing this. 
    if not Coordinator.fromUserAction: return

    need_refresh = False
    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)
        
        state = dom._state()
        if state == XenDomain.PAUSED:
            if confirmation("Resume " + dom.name + " on " + managed_node.hostname + "?"):
                try:
                    show_wait_cursor()
                    dom._resume()
                    need_refresh = True
                finally:
                    hide_wait_cursor()


        else:
            if confirmation("Pause " + dom.name + " on " + managed_node.hostname+ "?"):
                try:
                    show_wait_cursor()
                    dom._pause()
                    need_refresh = True
                finally:
                    hide_wait_cursor()
                    

        if need_refresh:
            managed_node = dom.node
            left_nav.refresh_doms()

            Coordinator.stateRefresh()


def summary_popupmenu(widget, event):
    """Checks whether the right mouse button was clicked
       and displays a popup menu"""

    if event.type == gtk.gdk.BUTTON_PRESS and event.button == 3:
        x = int(event.x)
        y = int(event.y)
        time = event.time
        pthinfo = widget.get_path_at_pos(x, y)
        if pthinfo is not None:
            path, col, cellx, celly = pthinfo
            widget.grab_focus()
            widget.set_cursor( path, col, 0)

            selection = widget.get_selection()
            (model,iter) = selection.get_selected()

            if iter:
                mode = model.get_value(iter, SummaryTab.MODE)

            if mode == SummaryTab.POOL_SUMMARY:
                wtree.get_widget("NodePopup").popup(None, None, None,
                                                    event.button,
                                                    event.time)
            if mode == SummaryTab.NODE_SUMMARY:
                wtree.get_widget("PopupMenu").popup(None, None, None,
                                                    event.button,
                                                    event.time)
            return 1
            

                

def popupmenu(widget, event):
    """Checks whether the right mouse button was clicked within the domain
    TreeView and if so, displays a popup menu"""

    if event.type == gtk.gdk.BUTTON_PRESS and event.button == 3:
        x = int(event.x)
        y = int(event.y)
        time = event.time
        pthinfo = left_nav.left_nav_view.get_path_at_pos(x, y)
        if pthinfo is not None:
            path, col, cellx, celly = pthinfo
            left_nav.left_nav_view.grab_focus()
            left_nav.left_nav_view.set_cursor( path, col, 0)

            (name, node_type, managed_node) = left_nav.get_selection()
            if node_type == LeftNav.DOMAIN:
                wtree.get_widget("PopupMenu").popup(None, None, None,
                                                    event.button,
                                                    event.time)
            elif node_type == LeftNav.MANAGED_NODE:
                wtree.get_widget("NodePopup").popup(None, None, None,
                                                    event.button,
                                                    event.time)
            elif node_type == LeftNav.SERVER_POOL:
                wtree.get_widget("PoolPopup").popup(None, None, None,
                                                    event.button,
                                                    event.time)
            elif node_type == LeftNav.IMAGE:
                wtree.get_widget("ImagePopup").popup(None, None, None,
                                                     event.button,
                                                     event.time)
            return 1
            

def opendom(widget):
    """Handles the open domain menu item which adds a dom by file name to
    the list of doms in the UI"""
    if not validate_node_selected():
        return
    managed_node = Coordinator.get_managed_node()
    
    mainwin = wtree.get_widget('MainWindow')
    result, filename = \
            file_selection(managed_node,
                           "Please select a VM configuration file", "open",
                           managed_node.config.get(utils.XMConfig.PATHS,prop_xenconf_dir),
                           parentwin = mainwin)

    if result and filename:
        left_nav.addDomFile(filename)

def deletedom(widget):
    """Handles the menu items requesting to delete of a dom from the
    left_nav"""
    
    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)
        msg = "Delete " + dom.name + " on " + managed_node.hostname+ "?"
        msg += "\n" + \
               "WARNING : Related VBDs/Volumes would also be deleted."
        if confirmation(msg):
            try:
                show_wait_cursor()
                host = dom.node.hostname
                cleanupQCDomain(managed_node, dom.name)
                left_nav.remDomFile(dom.get_config().filename)
                Coordinator.handle_remove_dom(dom)
            finally:
                hide_wait_cursor()

def removedom(widget):
    """Handles the menu items requesting to removal of a dom from the
    left_nav"""
    
    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)
        msg = "Remove " + dom.name + " from list under " + \
              managed_node.hostname+ "?"

        if confirmation(msg):
            host = dom.node.hostname
            left_nav.remDomFile(dom.get_config().filename)
            Coordinator.handle_remove_dom(dom)


def show_create_dialog(widget):
    if not validate_node_selected():
        return
    managed_node = Coordinator.get_managed_node()
    checkAndSetDefaultPaths(wtree, managed_node)
    create_dlg.show(managed_node)


def show_settings(widget, blank = False):
    """ get the current context and show the settings dialog"""
    if blank and not validate_node_selected():
        return
    (name, node_type, managed_node) = Coordinator.get_selection()
    if managed_node is not None:
        dom = managed_node.get_dom(name)
        if dom == None:
            blank = True
        if blank:
            settings.show(widget, managed_node, dom,blank = blank)
        else:
            if dom.get_config() is not None:
                settings.show(widget, managed_node, dom, blank = blank)
##                 if dom.get_config().is_xenman_generated():
##                     settings.show(widget, managed_node, dom, blank)
##                 else:
##                     dlg = FileViewEditDialog(wtree).show(widget,
##                                                          managed_node,
##                                                         dom.get_config().filename)
            else:
                if dom.isDom0():
                    settings.show(widget, managed_node, dom, blank = blank)
                


def show_file_view_edit_dlg(widget):
    (name, node_type, managed_node) = Coordinator.get_selection()
    if managed_node is not None:
        dom = managed_node.get_dom(name)
        if dom is not None and dom.get_config() is not None:
            
            file_editor.show(widget,
                             managed_node,
                             dom.get_config().filename)

    

def kill_all_domains(widget):
    """ kill all running domains on a node """
    if not validate_node_selected():
        return

    err_doms = []
    (name, node_type, managed_node) = Coordinator.get_selection()
    if managed_node is not None:
        if confirmation("Kill all running domains on " +
                        managed_node.hostname + "?"):
            domains = managed_node.get_dom_names()
            for dom in domains:
                try:
                    show_wait_cursor()
                    try:
                        if managed_node.isResident(dom) and \
                               managed_node.isDomU(dom):
                            managed_node.destroy_dom(dom)
                    except Exception ,ex:
                        traceback.print_exc()
                        print "kill all : Error killing ",dom,str(ex) 
                        # do best effort start, do not bother user.
                        err_doms.append(dom)
                finally:
                    hide_wait_cursor()
            if len(err_doms) > 0:
                showmsg(str(err_doms) + " could not be killed.\n" +
                        "Please try killing individually.")




def shutdown_all_domains(widget):
    """ shutdown all running domains on a node """
    if not validate_node_selected():
        return

    err_doms = []
    (name, node_type, managed_node) = Coordinator.get_selection()
    if managed_node is not None:
        if confirmation("Shutdown all running domains on " +
                        managed_node.hostname + "?"):
            domains = managed_node.get_dom_names()
            for dom in domains:
                try:
                    show_wait_cursor()
                    try:
                        if managed_node.isResident(dom) and \
                               managed_node.isDomU(dom):
                            managed_node.shutdown_dom(dom)
                    except Exception ,ex:
                        traceback.print_exc()
                        print "shutdown  all : Error shutting down ",dom,str(ex) 
                        # do best effort start, do not bother user.
                        err_doms.append(dom)
                finally:
                    hide_wait_cursor()
                    
            if len(err_doms) > 0:
                showmsg(str(err_doms) + " did not shutdown.\n" +
                        "Please try shutting them down individually.")


def start_all_domains(widget):
    """ start all domains on a node"""
    if not validate_node_selected():
        return

    err_doms = []
    (name, node_type, managed_node) = Coordinator.get_selection()
    if managed_node is not None:
        if confirmation("Start  all domains on " +
                        managed_node.hostname + "?"):
            domains = managed_node.get_dom_names()
            for dom in domains:
                try:
                    show_wait_cursor()
                    try:
                        if not managed_node.isResident(dom): 
                            managed_node.start_dom(dom)
                    except Exception ,ex:
                        print "Start all : Error starting " , dom, str(ex) 
                        # do best effort start, do not bother user.
                        err_doms.append(dom)
                finally:
                    hide_wait_cursor()

            if len(err_doms) > 0:
                showmsg(str(err_doms) + " could not be started.\n" +
                        "Please try starting them individually.")
        

def add_node(widget):
    """ add a new managed node to xenman """
    add_node_dlg.show()


def remove_node(widget):
    """ add a new managed node to xenman """
    if not validate_node_selected():
        return

    (name, node_type, managed_node) = Coordinator.get_selection()
    if confirmation("Remove " + managed_node.hostname + "?"):
        left_nav.remove_node(managed_node)

def show_console(widget):
    """ show cpnsole """
    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.DOMAIN:
        dom = managed_node.get_dom(name)
        nbook.showTab(dom)

def provision_image(widget):
    (image_name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.IMAGE:
        node_selection_dlg.show(widget, manager)
        (name, m_node) = node_selection_dlg.get_selection()
        if m_node is not None:
            settings.show(widget, m_node, dom=None,
                          image_name=image_name,
                          image_edit_mode = False, blank=True)
            left_nav.set_selection(managed_node.hostname, left_nav.MANAGED_NODE)
            left_nav.refresh_doms()

def edit_image(widget):
    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.IMAGE:
        settings.show(widget, local_node, dom=None, image_name=name,
                      image_edit_mode = True, blank=True)


def edit_image_script(widget):
    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.IMAGE:
        file_editor.show(widget,
                         local_node,
                         image_store.get_provisioning_script(name))
        

def edit_image_desc(widget):
    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.IMAGE:
        desc_file = image_store.get_image_desc(name)
        if not local_node.node_proxy.file_exists(desc_file):
            file = local_node.node_proxy.open(desc_file,"w")
            file.write("Add description for %s image here\n" % name)
            file.close()
        file_editor.show(widget,
                         local_node,
                         desc_file)
        Coordinator.handle_nav_selection(widget)

def connect_node(widget):
    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.MANAGED_NODE:
        if managed_node is not None:
            names = managed_node.get_dom_names()
        

def disconnect_node(widget):
    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.MANAGED_NODE:
        if managed_node is not None:
            showmsg("Not implemented")
    

def connect_all_nodes(widget):
    (name, node_type, managed_node) = Coordinator.get_selection()
    if node_type == LeftNav.SERVER_POOL:
        names = manager.getNodeNames()
        names.sort()
        for name in names:
            node = manager.getNode(name)
            if not node.is_authenticated():
                doms = node.get_dom_names()
        


def wrapper(widget, funct, *args, **cargs):
    try:
        funct(widget, *args, **cargs)
        left_nav.left_nav_view.queue_draw()
    except Exception , ex:
        traceback.print_exc()
        showmsg("Wrapper : Exception : " + str(ex))

def display_help(widget):
    """ display help """
    
   ##     import gnome
   ##     gnome.help_display_uri("./errata.html")
   ##     return
    browser = client_config.get(utils.XMConfig.CLIENT_CONFIG,
                                prop_browser)
    if browser is None:
        browser ="/usr/bin/yelp"
        
    
    if os.path.exists('doc/manual.html'):
        help_file = 'doc/manual.html'
    elif len(glob.glob('/usr/share/doc/xenman*/manual.html'))>0:
        help_file = glob.glob('/usr/share/doc/xenman*/manual.html')[0]
    else:
        help_file = None
        
    cmd = browser  + " " +  help_file
    p1 = subprocess.Popen(cmd,
                          stdout=subprocess.PIPE,stderr=subprocess.STDOUT,
                          universal_newlines=True,
                          shell=True, close_fds=True)
    #out = p1.communicate()[0]
    #exit_code   = p1.returncode
    #return out, exit_code



def checkLocalEnvironment():
    """Checks a number of required pre-conditions for the tool to be usable,
    if those aren't met, displays an appropriate dialog box and exits"""

    isXen = False
    isRoot = False
    
    # check whether the local kernel is recognisable as xen
    if platform.release().find('xen') != -1:
        isXen = True
    # check if running as root
    if os.getuid() == 0:
        isRoot = True

    return isXen, isRoot


    #ts = rpm.TransactionSet()
    #fedora = ts.dbMatch(rpm.RPMTAG_NAME, "fedora-release").count()
    # Check that we're running a Dom0 Xen kernel locally        

    #if platform.release().find('xen') == -1:
    #    msg = "You must be running a Xen Dom0 kernel to manage Xen domains."
    #
    #    if fedora:
    #        if ts.dbMatch(rpm.RPMTAG_NAME, "kernel-xen0").count():
    #            msg += "\nYou seem to have an appropriate Xen Dom0 kernel " + \
    #                   "installed, so restart your system and boot into it."
    #        else:
    #            msg += "\nYou don't seem to have an appropriate kernel " + \
    #                   'installed, try running "yum install "'
    #
    #            if not ts.dbMatch(rpm.RPMTAG_NAME, "xen").count():
    #                msg += "xen "
    #
    #            msg += 'kernel-xen0" to deploy the appropriate kernel, ' + \
    #                   "then restart your system and boot into that newly " + \
    #                   "installed kernel."
    #
    #    showmsg(msg)
    #    sys.exit(1)

    # Check that we're running as root
    #if os.getuid() != 0:
    #    showmsg("You must be running as root to interact with the hypervisor")
    #    sys.exit(1)
        
    #try:
        #_RESTRUCT
        #server.xend.domains()
        ### With remoting .. this is not a must.
        ### Tbis check would be made when adding local host.
        
        ### manager.getNode(default_host).get_doms()
    #    pass
    #except NodeException:
    #    if not ts.dbMatch(rpm.RPMTAG_NAME, "xen").count():
    #        msg = "You must install the Xen daemon in order manage Xen " + \
    #              "domains."
    #        if fedora:
    #            msg += 'Try running "yum install xen", followed by ' + \
    #                   '"service xend start" as root.'
    #    else:
    #        msg = "The Xen daemon isn't responding, "

    #        if fedora:
    #            msg += 'try starting it using "service xend start" as root.'
    #        else:
    #            msg += "please ensure it is installed and started."
    #
    #    showmsg(msg)
    #    sys.exit(1)


                
#############################################
# Main block                                #
#############################################

# set default timeout to 5 sec. This applies to setting connections.
# Fixes the problem when I tried to connect to my machine booted in
# Windows!!!.
socket.setdefaulttimeout(5)

showmsg = dialogs.showmsg
confirmation = dialogs.confirmation
file_selection = dialogs.file_selection


## initialize the phelper credentials helper
creds_helper = CredentialsHelper(wtree)
#PHelper.set_credentials_helper(creds_helper)

# Look at the command line arguments  and set up the server.
# for now simple checks for arguments

timer = None

# _RESTRUCTURING
from GridManager import GridManager
manager = GridManager()
remote_host = None


# Create a local node
# TODO: change this to a ManagedNode object
local_node = XenNode(hostname = LOCALHOST, isRemote = False, helper = creds_helper)

# 
xen_config_computed_options = local_node.config.get(utils.XMConfig.DEFAULT,
                                            prop_default_computed_options)
if xen_config_computed_options is None:
    defaults = ["arch", "arch_libdir", "device_model"]
else:
    defaults = eval(xen_config_computed_options)
    
DomConfig.set_computed_options(defaults)



# Sanity check the environment before performing any Xen interactions
isXen,isSuperUser = checkLocalEnvironment()
if isXen and not isSuperUser:
    if not local_node.config.get(utils.XMConfig.CLIENT_CONFIG,prop_init_confirmation):
        if not confirmation('WARNING: XenMan must be run from a superuser account '+
                            'in order to manage the local Xen environment.\n\n'+
                            'Would you like to manage remote hosts only? \n\n'+
                            'Click YES to proceed \n'+
                            'Click CANCEL to exit and launch XenMan from a '+
                            'superuser account'):        
            sys.exit(0)
        else:
            local_node.config.set(utils.XMConfig.CLIENT_CONFIG,prop_init_confirmation,'False')


if isXen and isSuperUser:
    manager.addNode(local_node)
    local_dom_names = local_node.get_dom_names() # force authentication

if len(sys.argv) > 1:
    remote_host = sys.argv[1]


# Initialise the configuration objects to gain access to global defaults
client_config = local_node.config
image_store = utils.ImageStore(local_node.config)

# testing only #
## image_store.addImage('Fedora Installer',
##                      '/var/cache/xenman/vmlinuz.default',
##                      '/var/cache/xenman/initrd.img.default',
##                      '' #'/usr/bin/pygrub'
##                      )
## image_store.setDefault('Fedora Installer')


#KLUDGE :populate the main context so dialogs and utils can share.
main_context["client_config"] = client_config
main_context["local_node"] = local_node

host_names = client_config.getHosts()

for host in host_names:
    
    _remote = client_config.getHostProperty(prop_isRemote, host)
    if not _remote:
        remote = is_host_remote(host)
        client_config.setHostProperty(prop_isRemote,str(remote),host)
    else:
        remote = eval(_remote)

    username = client_config.getHostProperty(prop_login, host)
    xen_port = client_config.getHostProperty(prop_xen_port, host)
    print host, remote
    node = XenNode(hostname=host,
                   username = username,
                   isRemote = remote,
                   tcp_port = xen_port,
                   helper = creds_helper)
    
    manager.addNode(node)
    
# Add the host on the command line
if remote_host:
    if remote_host not in manager.getNodeNames():
        remote = is_host_remote(remote_host)
        r_nd = XenNode(hostname = remote_host, isRemote = remote)
        manager.addNode(r_nd)
        
# _RESTRUCTURING



# Let's instantiate the dom management object

#left_nav = Left_nav(wtree)
left_nav = LeftNav(wtree)
nbook = NoteBook(left_nav)
Coordinator.set_nav(left_nav)
Coordinator.set_notebook(nbook)



settings = DomSettings(wtree, image_store)

signals = {"on_MainWindow_delete_event": quitProgram,
           "on_startbutton_clicked": (wrapper,startdom),
           "on_killbutton_clicked": (wrapper,killdom),
           "on_pausebutton_toggled": (wrapper,pausedom),
           "on_new_domain_activate": (wrapper,show_settings, True),
           "on_add_node_activate": (wrapper,add_node),
           "on_start_all_domains_activate": (wrapper,start_all_domains),
           "on_shutdown_all_domains_activate" : (wrapper,shutdown_all_domains),
           "on_kill_all_domains_activate" : (wrapper,kill_all_domains),
           "on_show_console_activate": (wrapper,show_console),
           "on_remove_node_activate": (wrapper,remove_node),
           "on_open_domain_activate": (wrapper,opendom),
           "on_Remove_activate": (wrapper,removedom),
           "on_delete_dom_activate": (wrapper,deletedom),
           "on_create_domain_wizard_activate": (wrapper,show_create_dialog),
           "on_DomView_button_press_event": popupmenu,
           "on_statTable_button_press_event": summary_popupmenu,
           "on_DomView_focus_in_event": Coordinator.focus_in,
           "on_DomView_focus_out_event":Coordinator.focus_out,
           "on_statTable_focus_in_event":Coordinator.focus_in,
           "on_statTable_focus_out_event":Coordinator.focus_out,
           "on_Settings_activate": (wrapper,show_settings),
           "on_edit_config_file_activate": (wrapper,show_file_view_edit_dlg),
           "on_quit_activate": quitProgram,
           "on_rebootbutton_clicked": (wrapper,rebootdom),
           "on_shutdownbutton_clicked": (wrapper,shutdowndom),
           "on_snapshotbutton_clicked": (wrapper,savedom),
           "on_restorebutton_clicked": (wrapper,restoredom),
           'on_about_activate': show_about,
           'on_AboutDialog_response': hide_about,
           'on_AboutDialog_delete_event': hide_about,
           'on_help_activate': (wrapper,display_help),
           'on_image_popup_provision_activate' : (wrapper, provision_image),
           'on_image_popup_edit_activate' : (wrapper, edit_image),
           'on_image_popup_edit_script_activate' : (wrapper, edit_image_script),
           'on_image_popup_edit_desc_activate' : (wrapper, edit_image_desc),
           'on_connect_activate': (wrapper, connect_node),
           'on_disconnect_activate': (wrapper, disconnect_node),
           'on_connect_all_nodes_activate': (wrapper,connect_all_nodes),
           }

wtree.signal_autoconnect(signals)

# Let's define the parent for the dialog windows (glade should do it but
# doesn't)

mainwin = wtree.get_widget('MainWindow')

# Wait cursor window

## attribs = { 'wmclass': gtk.gdk.INPUT_ONLY,
##             'window_type': gtk.gdk.WINDOW_CHILD,
##             'event_mask': 0,
##             'x': 0,
##             'y': 0,
##             'width': gtk.gdk.screen_width(),
##             'height': gtk.gdk.screen_height() }
## cursor_win = gtk.window_new(mainwin, attribs)
## gdk_cursor = gtk.cursor_new(gtk.gdk.WATCH)
## cursor_win.set_cursor(gdk_cursor)

## main_context["cursor_win"] = cursor_win
main_context["main_window"] = mainwin

for dname in ("SettingsDialog", "StatusDialog", "ConfirmationDialog",
              "AboutDialog", "CreateDialog", "initParamsDialog",
              "AddNode", "Credentials", "NodeSelection"):
    dialog = wtree.get_widget(dname)
    dialog.set_transient_for(mainwin)

node_selection_dlg = NodeSelection(wtree)
file_editor =  FileViewEditDialog(wtree)
create_dlg = CreateDialog(wtree, image_store)
add_node_dlg = AddNodeDialog(wtree,client_config, left_nav)

# Setup the timer

# kick off population of summary tab
#TRICK
Coordinator.stat_refresh_count = 1

timer = Timer()

# add the threads init as recommended in pygtk FAQ
gobject.threads_init()

# Run the main GUI loop
gtk.main()

# destroy wait window
#cursor_win.destroy()
#cursor_win = None
