/* $Id: xdg-desktop-support.c,v 1.9 2004/11/05 10:50:36 bmeurer Exp $ */
/*-
 * Copyright (c) 2004 os-cillation
 * All rights reserved.
 *
 * Written by Benedikt Meurer <bm@os-cillation.de>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <libxfce4util/libxfce4util.h>

#include "xdg-desktop-support.h"


static void xdg_desktop_cache_class_init      (XdgDesktopCacheClass *klass);
static void xdg_desktop_cache_init            (XdgDesktopCache      *cache);
static void xdg_desktop_cache_finalize        (GObject              *object);
static void xdg_desktop_cache_entry_removed   (XdgDesktopDir            *appdir,
                                               XdgDesktopEntry      *entry,
                                               XdgDesktopCache      *cache);
static void xdg_desktop_cache_dir_destroyed   (gpointer              user_data,
                                               GObject              *object);


static XdgDesktopDir *xdg_desktop_dir_new           (XdgDesktopCache    *cache,
                                                     const gchar        *path,
                                                     const gchar        *prefix);
static void           xdg_desktop_dir_add           (XdgDesktopDir      *dir,
                                                     XdgDesktopEntry    *entry);
static void           xdg_desktop_dir_remove        (XdgDesktopDir      *dir,
                                                     XdgDesktopEntry    *entry);
static void           xdg_desktop_dir_class_init    (XdgDesktopDirClass *klass);
static void           xdg_desktop_dir_finalize      (GObject        *object);
static void           xdg_desktop_dir_schedule_sync (XdgDesktopDir      *dir);
static void           xdg_desktop_dir_scan          (XdgDesktopDir      *dir,
                                                     const gchar        *path,
                                                     const gchar        *prefix);
static void           xdg_desktop_dir_scan_entry    (XdgDesktopDir      *dir,
                                                     const gchar        *id,
                                                     const gchar        *fullpath,
                                                     time_t              mtime);
static gboolean       xdg_desktop_dir_idle_load     (gpointer            user_data);
static gboolean       xdg_desktop_dir_timed_check   (gpointer            user_data);
static gboolean       xdg_desktop_dir_idle_sync     (gpointer            user_data);


static XdgDesktopEntry  *xdg_desktop_entry_new           (const gchar          *id,
                                                          const gchar          *name,
                                                          const gchar          *comment,
                                                          const gchar          *command,
                                                          const gchar          *categories,
                                                          const gchar          *icon,
                                                          gboolean              use_sn,
                                                          gboolean              use_term,
                                                          const gchar          *fullpath,
                                                          time_t                mtime);
static XdgDesktopEntry  *xdg_desktop_entry_load          (const gchar          *id,
                                                          const gchar          *fullpath,
                                                          time_t                mtime);
static gboolean          xdg_desktop_entry_recheck       (XdgDesktopEntry      *entry);
static void              xdg_desktop_entry_to_cache_line (XdgDesktopEntry      *entry,
                                                          gchar                *line,
                                                          gsize                 size);
static void              xdg_desktop_entry_class_init    (XdgDesktopEntryClass *klass);
static void              xdg_desktop_entry_finalize      (GObject              *object);
static void              xdg_desktop_entry_expand_exec   (XdgDesktopEntry *entry);
static gboolean          xdg_desktop_entry_parse         (XdgDesktopEntry      *entry,
                                                          XfceDesktopEntry     *dentry);
static gboolean          xdg_check_valid_exec            (const gchar          *exec);


enum
{
  DIR_SIGNAL_ADDED,
  DIR_SIGNAL_REMOVED,
  DIR_SIGNAL_LAST,
};

enum
{
  ENTRY_SIGNAL_CHANGED,
  ENTRY_SIGNAL_LAST,
};



static GObjectClass *parent_class;
static guint         dir_signals[DIR_SIGNAL_LAST];
static guint         entry_signals[ENTRY_SIGNAL_LAST];


static const gchar *desktop_keywords[] =
{
  "Icon",
  "Exec",
  "TryExec",
  "Name",
  "Comment",
  "GenericName",
  "Categories",
  "OnlyShowIn",
  "Hidden",
  "Terminal",
  "StartupNotify",
  "NoDisplay",
};



/*
   XdgDesktopCache
 */


GType
xdg_desktop_cache_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo type_info = {
        sizeof (XdgDesktopCacheClass),
        NULL,
        NULL,
        (GClassInitFunc) xdg_desktop_cache_class_init,
        NULL,
        NULL,
        sizeof (XdgDesktopCache),
        0,
        (GInstanceInitFunc) xdg_desktop_cache_init,
      };

      type = g_type_register_static (G_TYPE_OBJECT,
                                     "XdgDesktopCache",
                                     &type_info,
                                     0);
    }

  return type;
}



XdgDesktopCache*
xdg_desktop_cache_new (void)
{
  XdgDesktopCache *cache;

  cache      = g_object_new (XDG_TYPE_DESKTOP_CACHE, NULL);
  cache->dir = xfce_resource_save_location (XFCE_RESOURCE_CACHE,
                                            "xfce4/menu-cache/", TRUE);

  return cache;
}



/**
 * AppDir entries should be added in the reverse order that they appear
 * in the menu file, cause last one wins.
 */
XdgDesktopDir*
xdg_desktop_cache_append_dir (XdgDesktopCache *cache,
                              const gchar     *path,
                              const gchar     *prefix)
{
  XdgDesktopDir *dir;
  GList         *lp;

  for (lp = cache->dirs; lp != NULL; lp = lp->next)
    if (strcmp (XDG_DESKTOP_DIR (lp->data)->path, path) == 0)
      return XDG_DESKTOP_DIR (g_object_ref (G_OBJECT (lp->data)));

  dir = xdg_desktop_dir_new (cache, path, prefix);
  cache->dirs = g_list_append (cache->dirs, dir);

  g_signal_connect (G_OBJECT (dir), "removed",
                    G_CALLBACK (xdg_desktop_cache_entry_removed), cache);
  g_object_weak_ref (G_OBJECT (dir), xdg_desktop_cache_dir_destroyed, cache);

  return dir;
}



static void
xdg_desktop_cache_class_init (XdgDesktopCacheClass *klass)
{
  GObjectClass *gobject_class;

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  gobject_class->finalize = xdg_desktop_cache_finalize;
}



static void
xdg_desktop_cache_init (XdgDesktopCache *cache)
{
  cache->id_to_entry = g_hash_table_new (g_str_hash, g_str_equal);
}



static void
xdg_desktop_cache_finalize (GObject *object)
{
  XdgDesktopCache *cache;

  cache = XDG_DESKTOP_CACHE (object);

  /* all dirs are deleted, cause else finalize wouldn't have been called */
  g_list_free (cache->dirs);

  g_hash_table_destroy (cache->id_to_entry);
  g_free (cache->dir);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}



static void
xdg_desktop_cache_entry_removed (XdgDesktopDir   *dir,
                                 XdgDesktopEntry *entry,
                                 XdgDesktopCache *cache)
{
  g_hash_table_remove (cache->id_to_entry, entry->id);
}



static void
xdg_desktop_cache_dir_destroyed (gpointer user_data,
                                 GObject *object)
{
  XdgDesktopCache *cache;
  XdgDesktopDir   *dir;

  dir         = XDG_DESKTOP_DIR (object);
  cache       = XDG_DESKTOP_CACHE (user_data);
  cache->dirs = g_list_remove (cache->dirs, dir);
}



/*
   XdgDesktopDir
 */


GType
xdg_desktop_dir_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo type_info = {
        sizeof (XdgDesktopDirClass),
        NULL,
        NULL,
        (GClassInitFunc) xdg_desktop_dir_class_init,
        NULL,
        NULL,
        sizeof (XdgDesktopDir),
        0,
        NULL,
      };

      type = g_type_register_static (G_TYPE_OBJECT,
                                     "XdgDesktopDir",
                                     &type_info,
                                     0);
    }

  return type;
}



static XdgDesktopDir*
xdg_desktop_dir_new (XdgDesktopCache *cache,
                     const gchar     *path,
                     const gchar     *prefix)
{
  XdgDesktopDir *appdir;
  gchar         *namep;
  gchar          name[PATH_MAX];

  g_return_val_if_fail (path != NULL, NULL);
  g_return_val_if_fail (*path != '\0', NULL);

  /* get name from path ('/' -> '_') */
  g_strlcpy (name, path + 1, PATH_MAX);
  for (namep = name; *namep != '\0'; ++namep)
    if (*namep == '/')
      {
        if (*(namep + 1) == '\0')
          {
            *namep = '\0';
            break;
          }
        else
          {
            *namep = '_';
          }
      }

  appdir              = g_object_new (XDG_TYPE_DESKTOP_DIR, NULL);
  appdir->path        = g_strdup (path);
  appdir->prefix      = g_strdup ((prefix != NULL) ? prefix : "");
  appdir->cache       = XDG_DESKTOP_CACHE (g_object_ref (G_OBJECT (cache)));
  appdir->cache_file  = g_build_filename (cache->dir, name, NULL);

  /* load cached entries ASAP */
  appdir->timer_id    = g_idle_add_full (G_PRIORITY_HIGH,
                                         xdg_desktop_dir_idle_load,
                                         appdir,
                                         NULL);

  return appdir;
}



static void
xdg_desktop_dir_add (XdgDesktopDir   *dir,
                     XdgDesktopEntry *entry)
{
  g_object_ref (G_OBJECT (entry));

  entry->appdir = dir;
  dir->entries  = g_list_append (dir->entries, entry);

  g_hash_table_insert (dir->cache->id_to_entry, entry->id, entry);

  g_signal_emit (dir, dir_signals[DIR_SIGNAL_ADDED], 0, entry);
}



static void
xdg_desktop_dir_remove (XdgDesktopDir   *appdir,
                        XdgDesktopEntry *entry)
{
  g_return_if_fail (XDG_IS_DESKTOP_DIR (appdir));

  if (G_LIKELY (XDG_DESKTOP_DIR (entry->appdir) == appdir))
    {
      entry->appdir = NULL;

      g_signal_emit (appdir, dir_signals[DIR_SIGNAL_REMOVED], 0, entry);

      appdir->entries = g_list_remove (appdir->entries, entry);
      g_object_unref (entry);
    }
}



static void
xdg_desktop_dir_class_init (XdgDesktopDirClass *klass)
{
  G_OBJECT_CLASS (klass)->finalize = xdg_desktop_dir_finalize;

  dir_signals[DIR_SIGNAL_ADDED] = g_signal_new ("added",
                                                G_OBJECT_CLASS_TYPE (klass),
                                                G_SIGNAL_RUN_LAST,
                                                G_STRUCT_OFFSET (XdgDesktopDirClass, added),
                                                NULL, NULL,
                                                g_cclosure_marshal_VOID__OBJECT,
                                                G_TYPE_NONE, 1,
                                                XDG_TYPE_DESKTOP_ENTRY);

  dir_signals[DIR_SIGNAL_REMOVED] = g_signal_new ("removed",
                                                  G_OBJECT_CLASS_TYPE (klass),
                                                  G_SIGNAL_RUN_LAST,
                                                  G_STRUCT_OFFSET
                                                  (XdgDesktopDirClass, removed),
                                                  NULL, NULL,
                                                  g_cclosure_marshal_VOID__OBJECT,
                                                  G_TYPE_NONE, 1,
                                                  XDG_TYPE_DESKTOP_ENTRY);
}


static void
xdg_desktop_dir_finalize (GObject *object)
{
  XdgDesktopDir *dir;
  GObjectClass  *parent_class;

  dir = XDG_DESKTOP_DIR (object);

  g_object_unref (dir->cache);

  if (dir->timer_id != 0)
    g_source_remove (dir->timer_id);
  if (dir->sync_id != 0)
    g_source_remove (dir->sync_id);

  /* FIXME!!!!! */
  g_list_free (dir->entries);
  /* FIXME!!!!! */

  g_free (dir->cache_file);
  g_free (dir->prefix);
  g_free (dir->path);

  parent_class = g_type_class_peek_parent (XDG_DESKTOP_DIR_GET_CLASS (dir));
  parent_class->finalize (object);
}



static void
xdg_desktop_dir_schedule_sync (XdgDesktopDir *dir)
{
  if (dir->sync_id == 0)
    dir->sync_id = g_idle_add_full (G_PRIORITY_LOW, xdg_desktop_dir_idle_sync, dir, NULL);
}



static void
xdg_desktop_dir_scan (XdgDesktopDir *dir,
                      const gchar   *path,
                      const gchar   *prefix)
{
  const gchar *entry;
  struct stat  sb;
  gchar        buffer[128];
  gchar        fullpath[PATH_MAX];
  gsize        offset;
  GList       *list;
  GList       *lp;
  GDir        *dp;

  dp = g_dir_open (path, 0, NULL);
  if (G_LIKELY (dp != NULL))
    {
      offset = g_snprintf (fullpath, PATH_MAX, "%s/", path);

      for (;;)
        {
          entry = g_dir_read_name (dp);
          if (G_UNLIKELY (entry == NULL))
            break;

          g_strlcpy (fullpath + offset, entry, PATH_MAX - offset);
          if (G_UNLIKELY (stat (fullpath, &sb) < 0))
            continue;

          if (S_ISDIR (sb.st_mode) && *entry != '.')
            {
              g_snprintf (buffer, 128, "%s%s-", prefix, entry);
              xdg_desktop_dir_scan (dir, fullpath, buffer);
            }
          else if (S_ISREG (sb.st_mode) && g_str_has_suffix (entry, ".desktop"))
            {
              g_snprintf (buffer, 128, "%s%s", prefix, entry);
              xdg_desktop_dir_scan_entry (dir, buffer, fullpath, sb.st_mtime);
            }
        }

      g_dir_close (dp);
    }

  list = g_list_copy (dir->entries);
  for (lp = list; lp != NULL; lp = lp->next)
    if (xdg_desktop_entry_recheck (lp->data))
      xdg_desktop_dir_schedule_sync (dir);
  g_list_free (list);
}



static void
xdg_desktop_dir_scan_entry (XdgDesktopDir *dir,
                            const gchar   *id,
                            const gchar   *fullpath,
                            time_t         mtime)
{
  XdgDesktopEntry *entry;

  /* check if a matching desktop entry is already listed */
  entry = g_hash_table_lookup (dir->cache->id_to_entry, id);
  if (entry == NULL)
    {
      entry = xdg_desktop_entry_load (id, fullpath, mtime);
      if (G_UNLIKELY (entry == NULL))
        return;

      xdg_desktop_dir_add (dir, entry);
      g_object_unref (entry);

      xdg_desktop_dir_schedule_sync (dir);
    }
  else
    {
      /* recheck the matched entry */
      if (xdg_desktop_entry_recheck (entry))
        xdg_desktop_dir_schedule_sync (dir);
    }
}



static gboolean
xdg_desktop_dir_idle_load (gpointer user_data)
{
  XdgDesktopEntry *entry;
  XdgDesktopDir   *dir;
  gchar            line[4096];
  gchar          **fields;
  FILE            *fp;

  dir = XDG_DESKTOP_DIR (user_data);

  /* try to load desktop entries from cache */
  fp = fopen (dir->cache_file, "r");
  if (fp != NULL)
    {
      while (fgets (line, 4096, fp) != NULL)
        {
          /* split and validate line */
          fields = g_strsplit (line, "|", -1);
          if (G_UNLIKELY (fields[0] == NULL || fields[1] == NULL || fields[2] == NULL
                       || fields[3] == NULL || fields[4] == NULL || fields[5] == NULL
                       || fields[6] == NULL || fields[7] == NULL || fields[8] == NULL
                       || fields[9] == NULL || fields[10] != NULL))
            {
              g_strfreev (fields);
              continue;
            }

          /* check if a entry of this desktop-id is already present */
          entry = g_hash_table_lookup (dir->cache->id_to_entry, fields[0]);
          if (G_LIKELY (entry == NULL))
            {
              entry = xdg_desktop_entry_new (fields[0], fields[1],
                                             fields[2], fields[3],
                                             fields[4], fields[5],
                                             atoi (fields[6]), atoi (fields[7]),
                                             fields[8], atoi (fields[9]));

              if (G_LIKELY (entry != NULL))
                {
                  xdg_desktop_dir_add (dir, entry);
                  g_object_unref (entry);
                }
            }

          g_strfreev (fields);
        }

      fclose (fp);
    }
  else
    {
      /* no cache file, perform a scan immediatly */
      xdg_desktop_dir_scan (dir, dir->path, dir->prefix);
    }

  /* start normal dir checking */
  dir->timer_id = g_timeout_add (60 * 1000, xdg_desktop_dir_timed_check, dir);

  return FALSE;
}



static gboolean
xdg_desktop_dir_timed_check (gpointer user_data)
{
  XdgDesktopDir *dir;

  /* rescan for new desktop files */
  dir = XDG_DESKTOP_DIR (user_data);
  xdg_desktop_dir_scan (dir, dir->path, dir->prefix);

  /* keep the timer running */
  return TRUE;
}



static gboolean
xdg_desktop_dir_idle_sync (gpointer user_data)
{
  XdgDesktopDir *dir;
  gchar          temp_file[PATH_MAX];
  gchar          line[4096];
  GList         *lp;
  FILE          *fp;

  dir = XDG_DESKTOP_DIR (user_data);

  g_snprintf (temp_file, PATH_MAX, "%s.tmp", dir->cache_file);

  fp  = fopen (temp_file, "w");
  if (G_LIKELY (fp != NULL))
    {
      for (lp = dir->entries; lp != NULL; lp = lp->next)
        {
          xdg_desktop_entry_to_cache_line (XDG_DESKTOP_ENTRY (lp->data), line, 4096);
          fprintf (fp, "%s\n", line);
        }

      fclose (fp);

      if (rename (temp_file, dir->cache_file) < 0)
        unlink (temp_file);
    }

  /* ok, we did it */
  dir->sync_id = 0;
  return FALSE;
}




/*
   XdgDesktopEntry
 */


GType
xdg_desktop_entry_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo type_info = {
        sizeof (XdgDesktopEntryClass),
        NULL,
        NULL,
        (GClassInitFunc) xdg_desktop_entry_class_init,
        NULL,
        NULL,
        sizeof (XdgDesktopEntry),
        0,
        NULL,
      };

      type = g_type_register_static (G_TYPE_OBJECT,
                                     "XdgDesktopEntry",
                                     &type_info,
                                     0);
    }

  return type;
}



static XdgDesktopEntry*
xdg_desktop_entry_new (const gchar          *id,
                       const gchar          *name,
                       const gchar          *comment,
                       const gchar          *command,
                       const gchar          *categories,
                       const gchar          *icon,
                       gboolean              use_sn,
                       gboolean              use_term,
                       const gchar          *fullpath,
                       time_t                mtime)
{
  XdgDesktopEntry *entry;

  if (!xdg_check_valid_exec (command))
    return NULL;

  entry             = g_object_new (XDG_TYPE_DESKTOP_ENTRY, NULL);
  entry->id         = g_strdup (id);
  entry->name       = g_strdup (name);
  entry->comment    = (strlen (comment) > 0) ? g_strdup (comment) : NULL;
  entry->command    = g_strdup (command);
  entry->categories = g_strsplit (categories, ";", -1);
  entry->icon       = (strlen (icon) > 0) ? g_strdup (icon) : NULL;
  entry->use_sn     = use_sn;
  entry->use_term   = use_term;
  entry->fullpath   = g_strdup (fullpath);
  entry->filename   = strrchr (entry->fullpath, '/') + 1;
  entry->mtime      = mtime;

  return entry;
}



static XdgDesktopEntry*
xdg_desktop_entry_load (const gchar *id,
                        const gchar *fullpath,
                        time_t       mtime)
{
  XfceDesktopEntry *dentry;
  XdgDesktopEntry  *entry;
  gboolean          result;
  gchar           **only_show_in;
  gchar            *tmp;
  gsize             n;

  /* parse .desktop file */
  dentry = xfce_desktop_entry_new (fullpath,
                                   desktop_keywords,
                                   G_N_ELEMENTS (desktop_keywords));
  if (G_UNLIKELY (dentry == NULL))
    return NULL;

  /* check if this should be displayed in XFCE */
  if (xfce_desktop_entry_get_string (dentry, "OnlyShowIn", FALSE, &tmp))
    {
      only_show_in = g_strsplit (tmp, ";", -1);
      g_free (tmp);

      for (n = 0; only_show_in[n] != NULL; ++n)
        if (strcmp (only_show_in[n], "XFCE") == 0)
          break;

      if (only_show_in[n] == NULL)
        {
          g_strfreev (only_show_in);
          g_object_unref (dentry);
          return NULL;
        }

      g_strfreev (only_show_in);
    }

  /* check for NoDisplay (old style KDE crap) */
  if (xfce_desktop_entry_get_string (dentry, "NoDisplay", FALSE, &tmp))
    {
      if (strcmp (tmp, "false") != 0)
        {
          g_object_unref (dentry);
          g_free (tmp);
          return NULL;
        }

      g_free (tmp);
    }

  /* check for Hidden */
  if (xfce_desktop_entry_get_string (dentry, "Hidden", FALSE, &tmp))
    {
      if (strcmp (tmp, "false") != 0)
        {
          g_object_unref (dentry);
          g_free (tmp);
          return NULL;
        }

      g_free (tmp);
    }

  /* create the desktop entry object */
  entry             = g_object_new (XDG_TYPE_DESKTOP_ENTRY, NULL);
  entry->id         = g_strdup (id);
  entry->fullpath   = g_strdup (fullpath);
  entry->filename   = strrchr (entry->fullpath, '/') + 1;
  entry->mtime      = mtime;

  /* try to parse the desktop entry file */
  result = xdg_desktop_entry_parse (entry, dentry);
  g_object_unref (dentry);

  if (G_UNLIKELY (!result))
    {
      g_object_unref (entry);
      return NULL;
    }

  return entry;
}



static gboolean
xdg_desktop_entry_recheck (XdgDesktopEntry *entry)
{
  XfceDesktopEntry *dentry;
  struct stat       sb;
  gboolean          result;

  /* check if the file is still present */
  if (stat (entry->fullpath, &sb) < 0)
    goto failed;

  /* now on with the modification time */
  if (G_LIKELY (sb.st_mtime <= entry->mtime))
    return FALSE;

  dentry = xfce_desktop_entry_new (entry->fullpath,
                                   desktop_keywords,
                                   G_N_ELEMENTS (desktop_keywords));
  if (G_UNLIKELY (dentry == NULL))
    goto failed;

  result = xdg_desktop_entry_parse (entry, dentry);
  g_object_unref (dentry);

  if (G_UNLIKELY (!result))
    goto failed;

  entry->mtime = sb.st_mtime;

  /* notify listeners that we got new data */
  g_signal_emit (entry, entry_signals[ENTRY_SIGNAL_CHANGED], 0);

  return TRUE;

failed:
  if (G_LIKELY (entry->appdir != NULL))
    xdg_desktop_dir_remove (entry->appdir, entry);
  return TRUE;
}



static void
xdg_desktop_entry_to_cache_line (XdgDesktopEntry *entry,
                                 gchar           *line,
                                 gsize            size)
{
  gchar *categories;
  gchar *time;

  categories = g_strjoinv (";", entry->categories);
  time       = g_strdup_printf ("%d", entry->mtime);

  /* FIXME: use one g_snprintf! */
  g_strlcpy (line, entry->id, size);
  g_strlcat (line, "|", size);
  g_strlcat (line, entry->name, size);
  g_strlcat (line, "|", size);
  if (entry->comment != NULL)
    g_strlcat (line, entry->comment, size);
  g_strlcat (line, "|", size);
  g_strlcat (line, entry->command, size);
  g_strlcat (line, "|", size);
  g_strlcat (line, categories, size);
  g_strlcat (line, "|", size);
  if (entry->icon != NULL)
    g_strlcat (line, entry->icon, size);
  g_strlcat (line, "|", size);
  g_strlcat (line, entry->use_sn ? "1" : "0", size);
  g_strlcat (line, "|", size);
  g_strlcat (line, entry->use_term ? "1" : "0", size);
  g_strlcat (line, "|", size);
  g_strlcat (line, entry->fullpath, size);
  g_strlcat (line, "|", size);
  g_strlcat (line, time, size);

  g_free (categories);
  g_free (time);
}



static void
xdg_desktop_entry_class_init (XdgDesktopEntryClass *klass)
{
  G_OBJECT_CLASS (klass)->finalize = xdg_desktop_entry_finalize;

  entry_signals[ENTRY_SIGNAL_CHANGED] = g_signal_new ("changed",
                                                      G_OBJECT_CLASS_TYPE (klass),
                                                      G_SIGNAL_RUN_LAST,
                                                      G_STRUCT_OFFSET
                                                      (XdgDesktopEntryClass,
                                                       changed),
                                                      NULL, NULL,
                                                      g_cclosure_marshal_VOID__VOID,
                                                      G_TYPE_NONE, 0);
}



static void
xdg_desktop_entry_finalize (GObject *object)
{
  XdgDesktopEntry *entry;
  GObjectClass    *parent_class;

  entry = XDG_DESKTOP_ENTRY (object);

  if (entry->appdir != NULL)
    xdg_desktop_dir_remove (entry->appdir, entry);

  if (G_LIKELY (entry->id != NULL))
    g_free (entry->id);

  if (G_LIKELY (entry->name != NULL))
    g_free (entry->name);

  if (G_LIKELY (entry->comment != NULL))
    g_free (entry->comment);

  if (G_LIKELY (entry->command != NULL))
    g_free (entry->command);

  if (G_LIKELY (entry->categories != NULL))
    g_strfreev (entry->categories);

  if (G_LIKELY (entry->icon != NULL))
    g_free (entry->icon);

  if (G_LIKELY (entry->fullpath != NULL))
    g_free (entry->fullpath);

  parent_class = g_type_class_peek_parent (XDG_DESKTOP_ENTRY_GET_CLASS (entry));
  parent_class->finalize (object);
}



static void
xdg_desktop_entry_expand_exec (XdgDesktopEntry *entry)
{
  gchar  buffer[4096];
  gchar *tp;
  gchar *sp;

  if (strchr (entry->command, '%') == NULL)
    return;

  for (sp = entry->command, tp = buffer; *sp != '\0'; )
    {
      if (*sp != '%')
        {
          *tp++ = *sp++;
          continue;
        }

      ++sp;

      switch (*sp)
        {
        case 'c':
          memcpy (tp, entry->name, strlen (entry->name));
          tp += strlen (entry->name);
          break;

        case 'i':
          if (G_LIKELY (entry->icon != NULL))
            {
              memcpy (tp, "-icon ", strlen ("-icon "));
              tp += strlen ("-icon ");
              memcpy (tp, entry->icon, strlen (entry->icon));
              tp += strlen (entry->icon);
            }
          break;

        case 'm':
          if (G_LIKELY (entry->icon != NULL))
            {
              memcpy (tp, "-miniicon ", strlen ("-miniicon "));
              tp += strlen ("-miniicon ");
              memcpy (tp, entry->icon, strlen (entry->icon));
              tp += strlen (entry->icon);
            }
          break;

        case '%':
          *tp++ = '%';
          break;

        default:
          break;
        }

      ++sp;
    }

  *tp = '\0';

  g_free (entry->command);
  entry->command = g_strdup (buffer);
}



static gboolean
xdg_desktop_entry_parse (XdgDesktopEntry  *entry,
                         XfceDesktopEntry *dentry)
{
  gboolean result;
  gchar   *categories;
  gchar   *tmp;

  if (G_UNLIKELY (entry->command != NULL))
    {
      g_free (entry->command);
      entry->command = NULL;
    }

  if (xfce_desktop_entry_get_string (dentry, "TryExec", FALSE, &tmp))
    {
#if 0
      p = strchr (tmp, ' ');
      if (G_UNLIKELY (p != NULL))
        *p = '\0';

      p = g_find_program_in_path (tmp);
      g_free (tmp);

      if (G_UNLIKELY (p == NULL))
        {
          return FALSE;
        }
      else
        {
          result = access (p, X_OK);
          g_free (p);

          if (result < 0)
            return FALSE;
        }
#else
      result = xdg_check_valid_exec (tmp);
      g_free (tmp);

      if (!result)
        return FALSE;
#endif
    }

  if (!xfce_desktop_entry_get_string (dentry, "Exec", FALSE, &entry->command))
    return FALSE;

  /* check if Exec is valid */
  if (!xdg_check_valid_exec (entry->command))
    return FALSE;

#if 0
  if (*entry->command != '\0')
    {
      tmp = g_strdup (entry->command);
      p = strchr (tmp, ' ');
      if (G_UNLIKELY (p != NULL))
        *p = '\0';

      p = g_find_program_in_path (tmp);
      g_free (tmp);

      if (G_UNLIKELY (p == NULL))
        {
          return FALSE;
        }
      else
        {
          result = access (p, X_OK);
          g_free (p);

          if (result < 0)
            return FALSE;
        }
    }
#endif

  if (G_UNLIKELY (entry->name != NULL))
    {
      g_free (entry->name);
      entry->name = NULL;
    }

  if (!xfce_desktop_entry_get_string (dentry, "Name", TRUE, &entry->name))
    return FALSE;

  if (G_UNLIKELY (entry->comment != NULL))
    {
      g_free (entry->comment);
      entry->comment = NULL;
    }

  if (!xfce_desktop_entry_get_string (dentry, "Comment", TRUE, &entry->comment))
    {
      /* try KDE crap */
      xfce_desktop_entry_get_string (dentry, "GenericName", TRUE, &entry->comment);
    }

  if (G_UNLIKELY (entry->icon != NULL))
    {
      g_free (entry->icon);
      entry->icon = NULL;
    }

  xfce_desktop_entry_get_string (dentry, "Icon", TRUE, &entry->icon);

  if (G_UNLIKELY (entry->categories != NULL))
    {
      g_strfreev (entry->categories);
      entry->categories = NULL;
    }

  if (!xfce_desktop_entry_get_string (dentry, "Categories", FALSE, &categories))
    {
      /* treat the entry as a legacy entry */
      entry->categories     = g_new (gchar *, 2);
      entry->categories[0]  = g_strdup ("Legacy");
      entry->categories[1]  = NULL;
    }
  else
    {
      entry->categories = g_strsplit (categories, ";", -1);
      g_free (categories);
    }

  if (xfce_desktop_entry_get_string (dentry, "StartupNotify", FALSE, &tmp))
    {
      entry->use_sn = (strcmp (tmp, "false") != 0 && *tmp != '0');
      g_free (tmp);
    }
  else
    {
      entry->use_sn = FALSE;
    }

  if (xfce_desktop_entry_get_string (dentry, "Terminal", FALSE, &tmp))
    {
      entry->use_term = (strcmp (tmp, "false") != 0 && *tmp != '0');
      g_free (tmp);
    }
  else
    {
      entry->use_term = FALSE;
    }

  xdg_desktop_entry_expand_exec (entry);

  return TRUE;
}



static gboolean
xdg_check_valid_exec (const gchar *exec)
{
  gboolean result = TRUE;
  gchar   *tmp;
  gchar   *p;

  if (*exec == '/')
    {
      result = (access (exec, X_OK) == 0);
    }
  else
    {
      tmp = g_strdup (exec);
      p = strchr (tmp, ' ');
      if (G_UNLIKELY (p != NULL))
        *p = '\0';

      p = g_find_program_in_path (tmp);
      g_free (tmp);

      if (G_UNLIKELY (p == NULL))
        {
          result = FALSE;
        }
      else
        {
          result = (access (p, X_OK) == 0);
          g_free (p);
        }
    }

  return result;
}



