#include "config.h"
#include "i18n.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <fox/fx.h>
#include <fox/fxkeys.h>
#include <fox/FXPNGIcon.h>

#include "icons.h"
#include "DirPanel.h"
#include "DirList.h"
#include "Properties.h"
#include "HistInputDialog.h"
#include "File.h"
#include "XFileExplorer.h"
#include "MessageBox.h"


// Root directory string
#ifndef ROOTDIR
#define ROOTDIR "/"
#endif

// History size
#ifndef ADD_TO_ARCH_HIST_SIZE
#define ADD_TO_ARCH_HIST_SIZE 30
#endif

// Maximum length of a file name
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

// Global variables
extern FXbool confirm_del;
extern FXString clipboard;
extern FXuint clptype;
#if defined(linux)
extern FXStringDict* devices;
#endif

// Map
FXDEFMAP(DirPanel) DirPanelMap[]={
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_EXPANDDIR,DirPanel::onExpandDir),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_EXPANDTREE,DirPanel::onExpandTree),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_COLLAPSEDIR,DirPanel::onCollapseDir),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_COLLAPSETREE,DirPanel::onCollapseTree),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_PROPERTIES,DirPanel::onCmdProperties),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_ARCHIVE,DirPanel::onCmdAddToArch),
                                   FXMAPFUNC(SEL_RIGHTBUTTONPRESS,DirPanel::ID_DIR_PANEL,DirPanel::onPopupMenu),
                               	   FXMAPFUNC(SEL_CLICKED,DirPanel::ID_DIR_PANEL,DirPanel::onCmdDirectory),
                               	   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_COPY,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_MOVE,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_RENAME,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_SLINK,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_DELETE,DirPanel::onCmdDirDel),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_NEW_DIR,DirPanel::onCmdNewDir),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_XTERM,DirPanel::onCmdXTerm),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_COPY_CLP,DirPanel::onCmdClp),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_CUT_CLP,DirPanel::onCmdClp),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_PASTE_CLP,DirPanel::onCmdPaste),
#if defined(linux)
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_MOUNT,DirPanel::onCmdMount),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_UMOUNT,DirPanel::onCmdMount),
#endif								   
                               	   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_PASTE_CLP,DirPanel::onUpdPaste),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_TOGGLE_HIDDEN,DirPanel::onUpdToggleHidden),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_TOGGLE_HIDDEN,DirPanel::onCmdToggleHidden),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_TOGGLE_TREE,DirPanel::onCmdToggleTree),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_TOGGLE_TREE,DirPanel::onUpdToggleTree),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_STATUS,DirPanel::onUpdStatus),
                          		};

// Object implementation 
FXIMPLEMENT(DirPanel,FXVerticalFrame,DirPanelMap,ARRAYNUMBER(DirPanelMap))

// Construct Directory Panel
DirPanel::DirPanel(FXComposite *p,FXuint opts,FXint x,FXint y,FXint w,FXint h):
        FXVerticalFrame(p,opts,x,y,w,h,0,0,0,0)
{
	// Create some icons
	attricon=new FXPNGIcon(getApp(),attrib);
    archicon=new FXPNGIcon(getApp(),arch);
    newfoldericon=new FXPNGIcon(getApp(),newfolder);
	delicon=new FXPNGIcon(getApp(),filedelete);	
	copyicon=new FXPNGIcon(getApp(),copy_clp);	
	cuticon=new FXPNGIcon(getApp(),cut_clp);	
	pasteicon=new FXPNGIcon(getApp(),paste_clp);	
	mapicon=new FXPNGIcon(getApp(),maphost);	
	unmapicon=new FXPNGIcon(getApp(),unmaphost);	
	xtermicon=new FXPNGIcon(getApp(),shell);
	symlinkicon=new FXPNGIcon(getApp(),minilink);
	renameicon=new FXPNGIcon(getApp(),renameit);
	moveicon=new FXPNGIcon(getApp(),moveit);
	exptreeicon=new FXPNGIcon(getApp(),exptree);
	expfoldericon=new FXPNGIcon(getApp(),expfolder);
	colltreeicon=new FXPNGIcon(getApp(),colltree);
	collfoldericon=new FXPNGIcon(getApp(),collfolder);

	FXIcon *showhiddenicon=new FXPNGIcon(getApp(),showhidden);
	FXIcon *hidehiddenicon=new FXPNGIcon(getApp(),hidehidden);

	// Construct directory panel
    FXVerticalFrame* cont=new FXVerticalFrame(this,LAYOUT_FILL_Y|LAYOUT_FILL_X|FRAME_SUNKEN,0,0,0,0, 0,0,0,0);    
	FXPacker* packer=new FXHorizontalFrame(cont,JUSTIFY_LEFT|LAYOUT_FILL_X|FRAME_SUNKEN,0,0,0,0, 0,0,0,0);

    list=new DirList(cont,this,ID_DIR_PANEL,LAYOUT_FILL_X|LAYOUT_FILL_Y|TREELIST_BROWSESELECT|TREELIST_SHOWS_LINES|TREELIST_SHOWS_BOXES|ICONLIST_BIG_ICONS);

	FXPacker* packer2=new FXHorizontalFrame(cont,JUSTIFY_LEFT|LAYOUT_FILL_X,0,0,0,0, 0,0,0,0);
    new FXToggleButton(packer2,_("\tShow hidden folders (Ctrl-F6)"),_("\tHide hidden folders (Ctrl-F6)"),showhiddenicon,hidehiddenicon,this,ID_TOGGLE_HIDDEN,BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_LEFT|ICON_BEFORE_TEXT);
    status=new FXLabel(packer2,_("Status"),NULL,JUSTIFY_LEFT|LAYOUT_LEFT|LAYOUT_FILL_X|FRAME_SUNKEN);
    status->setTarget(this);
    status->setSelector(MKUINT(DirPanel::ID_STATUS,SEL_UPDATE));

    new FXButton(packer,_("\tHide"),new FXPNGIcon(getApp(),small_x),this,FXWindow::ID_HIDE,BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_RIGHT,0,0,0,0, 3,3,0,0);  
	new FXLabel(packer,_("Folders"),NULL,JUSTIFY_LEFT|LAYOUT_LEFT|LAYOUT_FILL_X);
}


// Destructor
DirPanel::~DirPanel()
{
	delete attricon;
	delete archicon;
	delete newfoldericon;
	delete delicon;
	delete copyicon;
	delete cuticon;
	delete pasteicon;
	delete mapicon;
	delete unmapicon;
	delete xtermicon;
	delete symlinkicon;
	delete renameicon;
	delete moveicon;
	delete exptreeicon;
	delete expfoldericon;
	delete colltreeicon;
	delete collfoldericon;
	delete list;
	delete status;
}

// Create X window
void DirPanel::create()
{
	FXVerticalFrame::create();
	list->create();
}


// Change the directory when clicking on the tree list
long DirPanel::onCmdDirectory(FXObject*,FXSelector,void* ptr)
{
    TreeItem *item=(TreeItem*)ptr;
    if(item)
    {
        FXString directory = list->getItemPathname(item);
        if(access(directory.text(),R_OK|X_OK))
        {
            MessageBox::error(this,MBOX_OK,_("Error"),_(" Permission to: %s denied."), directory.text());
            return 0;
        }
		FilePanel* currentpanel=((XFileExplorer*) getApp()->getMainWindow())->lpanel->current;
		FXComboBox* address=((XFileExplorer*) getApp()->getMainWindow())->address;
        currentpanel->list->setDirectory(directory);
        
		// Remember latest directory in the location address
       	FXString item;
		FXint i=0;
		FXint count=address->getNumItems();
       	FXString p=currentpanel->list->getDirectory();

		if(!count)
		{
			count++;
			address->insertItem(0,address->getText());
		}
       	while(i < count)
		{
       		item = address->retrieveItem(i++);
       		if(!strcmp((const char*)&p[0],(const char*)&item[0]))
			{
				i--;
				break;
			}
       	}
       	if(i==count)
			address->insertItem(0,currentpanel->list->getDirectory());
    }
    return 1;
}

// Toggle hidden files
long DirPanel::onCmdToggleHidden(FXObject*,FXSelector,void*)
{
    list->showHiddenFiles(!list->showHiddenFiles());
    return 1;
}


// Update toggle hidden files widget
long DirPanel::onUpdToggleHidden(FXObject* sender,FXSelector,void*)
{
    if(list->showHiddenFiles())
        sender->handle(this,MKUINT(ID_CHECK,SEL_COMMAND),NULL);
    else
        sender->handle(this,MKUINT(ID_UNCHECK,SEL_COMMAND),NULL);
    return 1;
}


long DirPanel::onCmdToggleTree(FXObject* sender,FXSelector sel,void* ptr)
{
	return this->handle(sender,MKUINT(FXWindow::ID_TOGGLESHOWN,SEL_COMMAND),ptr);
}

long DirPanel::onUpdToggleTree(FXObject* sender,FXSelector sel,void* ptr)
{
    if(this->shown())
        sender->handle(this,MKUINT(FXWindow::ID_CHECK,SEL_COMMAND),ptr);
    else
        sender->handle(this,MKUINT(FXWindow::ID_UNCHECK,SEL_COMMAND),ptr);
    return 1;
}


// File list context menu
long DirPanel::onPopupMenu(FXObject* o,FXSelector s,void* p)
{
	// Current item becomes item under cursor
	FXint x, y, xitem, yitem;
    FXuint state;
	getRoot()->getCursorPosition(x,y,state);
	list->getCursorPosition(xitem,yitem,state);   	    
	DirItem* item=(DirItem*)list->getItemAt(xitem,yitem);

	// If no item under cursor, then return
	if (item==NULL) return 0;
	
	// If item, then set it current and set directory in DirList and FileList
	list->setCurrentItem(item,TRUE);
	FXString dir=list->getItemPathname((TreeItem*)item);
	list->setDirectory(dir);
	((XFileExplorer*) getApp()->getMainWindow())->lpanel->current->list->setDirectory(dir);

    // Popup menu pane
    static FXMenuPane *menu = NULL;
    if(menu)
        delete menu;
    menu = new FXMenuPane(this);
    
	// Menu
    new FXMenuCommand(menu,_("Terminal"),xtermicon,this,ID_XTERM);
	new FXMenuCommand(menu,_("New folder..."),newfoldericon,this,ID_NEW_DIR);
	new FXMenuCommand(menu,_("Add to archive..."),archicon,this,ID_ARCHIVE);
#if defined(linux)
	if (isLink(dir))
		dir=readlink(dir);
    if(devices->find(dir.text()))
	{
    	new FXMenuSeparator(menu);
		new FXMenuCommand(menu,_("Mount"),mapicon,this,ID_MOUNT);
		new FXMenuCommand(menu,_("Unmount"),unmapicon,this,ID_UMOUNT);
	}
#endif
    new FXMenuSeparator(menu);
	new FXMenuCommand(menu,_("Copy"),copyicon,this,ID_COPY_CLP);
	new FXMenuCommand(menu,_("Cut"),cuticon,this,ID_CUT_CLP);
	if(!clipboard.empty())
		new FXMenuCommand(menu,_("Paste"),pasteicon,this,ID_PASTE_CLP);
    new FXMenuSeparator(menu);
	new FXMenuCommand(menu,_("Move..."),moveicon,this,ID_DIR_MOVE);
	new FXMenuCommand(menu,_("Rename..."),renameicon,this,ID_DIR_RENAME);
	new FXMenuCommand(menu,_("Symlink..."),symlinkicon,this,ID_DIR_SLINK);
	new FXMenuCommand(menu,_("Delete"),delicon,this,ID_DIR_DELETE);
    new FXMenuSeparator(menu);
	new FXMenuCommand(menu,_("Hidden folders"),NULL,this,ID_TOGGLE_HIDDEN);
	new FXMenuCommand(menu,_("Expand"),expfoldericon,this,ID_EXPANDDIR);
	new FXMenuCommand(menu,_("Collapse"),collfoldericon,this,ID_COLLAPSEDIR);
	new FXMenuCommand(menu,_("Expand tree"),exptreeicon,this,ID_EXPANDTREE);
	new FXMenuCommand(menu,_("Collapse tree"),colltreeicon,this,ID_COLLAPSETREE);
    new FXMenuSeparator(menu);
	new FXMenuCommand(menu,_("Properties"),attricon,this,ID_PROPERTIES);
	
    menu->create();
    menu->popup(NULL,x,y);

    return 1;
}


// Expand a directory in the tree
long DirPanel::onExpandDir(FXObject* sender,FXSelector sel,void* ptr)
{
    // Current item
	DirItem* item=(DirItem*)list->getCurrentItem();

    // Expand tree item
    list->expandTree(item,TRUE);

	return 1;
}


// Helper function used to explore the directory tree and expand or collapse it
long DirPanel::exploreUp(DirItem *item, const DirItem *rootitem, const FXint task)
{
	DirItem* parentitem=item;
	
	if (task==ID_EXPANDTREE)
		list->expandTree(item,TRUE);
	else
		list->collapseTree(item,TRUE);
	
	item=(DirItem*)item->getFirst();
	if (!item)
		exploreDown(parentitem,rootitem,task);
	else
		exploreUp(item,rootitem,task);
	return 1;	
}


// Helper function used to explore the directory tree and expand or collapse it
long DirPanel::exploreDown(DirItem *item, const DirItem *rootitem, const FXint task)
{
	if (item==rootitem)
		return 1;
	
	DirItem* parentitem=(DirItem*)item->getParent();
	
	if (task==ID_EXPANDTREE)
		list->expandTree(item,TRUE);
	else
		list->collapseTree(item,TRUE);
	item=(DirItem*)item->getNext();
	
	if (!item)
		exploreDown(parentitem,rootitem,task);
	else
	{
		if (task==ID_EXPANDTREE)
			list->expandTree(item,TRUE);
		else
			list->collapseTree(item,TRUE);
		if (!list->isItemLeaf(item))
			exploreUp(item,rootitem,task);
		else
			exploreDown(item,rootitem,task);
	}
	return 1;		
}
				

// Expand the directory tree under cursor
long DirPanel::onExpandTree(FXObject* sender,FXSelector sel,void* ptr)
{
    DirItem* rootitem=(DirItem*)list->getCurrentItem();
	getApp()->beginWaitCursor();
	exploreUp(rootitem,rootitem,ID_EXPANDTREE);
	getApp()->endWaitCursor();
	
	return 1;
}


// Collapse the directory tree under cursor
long DirPanel::onCollapseTree(FXObject* sender,FXSelector sel,void* ptr)
{
    DirItem* rootitem=(DirItem*)list->getCurrentItem();
	getApp()->beginWaitCursor();
	exploreUp(rootitem,rootitem,ID_COLLAPSETREE);
	getApp()->endWaitCursor();
	
	return 1;
}


// Collapse a directory in the tree
long DirPanel::onCollapseDir(FXObject* sender,FXSelector sel,void* ptr)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();

    // Collapse tree item
    list->collapseTree(item,TRUE);
	
	return 1;
}

// Directory properties
long DirPanel::onCmdProperties(FXObject* sender,FXSelector,void* ptr)
{
    // Current item
	DirItem* item=(DirItem*)list->getCurrentItem();
	FXString pathname=list->getItemPathname((TreeItem*)item);
	
	PropertiesBox* attrdlg = new PropertiesBox(this,FXFile::name(pathname),FXFile::directory(pathname));	
   	if(attrdlg->execute())
        list->setDirectory(FXFile::directory(pathname));
    delete attrdlg;
   	return 1;
}


// Add files or directory to an archive
long DirPanel::onCmdAddToArch(FXObject* o,FXSelector sel,void* ptr)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString name=list->getItemText(item);
	FXString pathname=list->getItemPathname((TreeItem*)item);	
	FXString parentdir=FXFile::directory(pathname);

    static char history[ADD_TO_ARCH_HIST_SIZE][100];
    static int n=0;
    int i;
	File *f;	
    HistInputDialog* dialog;

    // Default archive is tar.gz 
	FXString command="/bin/tar -zcvf ";
    
	dialog = new HistInputDialog(this,pathname,_("Create new archive: (e.g. archive.tar.gz)"),_("Add To Archive"));
    for(int i=0;i<n;i++)
        dialog->appendItem(history[i]);
    dialog->CursorEnd();
    if(dialog->execute())
    {
        FXString archive=dialog->getText();

        // Handle zip and bz2 archives
        if (strstr(archive.text(),".zip") != NULL || strstr(archive.text(),".ZIP") != NULL)
            command="/usr/bin/zip -r ";
        else if (strstr(archive.text(),".bz2") != NULL || strstr(archive.text(),".tbz2") != NULL)
            command="/bin/tar -jcvf ";

		// Archive command name
		command+=archive+" "+"\""+name+"\"";
					
		// File object
		f=new File(getApp(),_("Create archive"),ARCHIVE);
		f->create();
				   
		// Create archive
		chdir(parentdir.text());
		f->archive(archive,command);
 
		// If action is cancelled in progress dialog
		if (f->isCancelled)
		{
			f->hide();
			MessageBox::error(this,MBOX_OK,_("Error"),_("Archive creation cancelled!"));
		}					
		delete f; 

        // Manage dialog history size
		n=dialog->getHistorySize();
        n++;
        command=dialog->getText();
        if(n>ADD_TO_ARCH_HIST_SIZE)
            n=ADD_TO_ARCH_HIST_SIZE;
        for(i=0;i<n-1;i++)
            if(!strcmp(history[i],command.text()))
                break;
        if(i==n-1)
        {
            strcpy(history[0],command.text());
            for(i=1;i<n;i++)
                strcpy(history[i],dialog->getHistoryItem(i-1).text());
        }
        else
            n--;

		// Display parent directory in DirList and FileList
		list->setDirectory(parentdir);
		((XFileExplorer*) getApp()->getMainWindow())->lpanel->current->list->setDirectory(parentdir);
    }
    delete dialog;
    return 1;
}


// Copy/Move/Rename/SymLink directory
long DirPanel::onCmdDirMan(FXObject* sender,FXSelector sel,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	
	// Parameters passed to XFileExplorer::onCmdFileMan
	// -1 is used to tell that we are calling from DirPanel
	FXString source=list->getItemPathname((TreeItem*)item);	
    FXString target = FXFile::directory(source);
	FXString param = target+"\n"+"-1"+"\n"+source;

    switch(SELID(sel))
    {
    case ID_DIR_COPY:
        sel = MKUINT(XFileExplorer::ID_FILE_COPY,SEL_COMMAND);
        break;
    case ID_DIR_MOVE:
        sel = MKUINT(XFileExplorer::ID_FILE_MOVE,SEL_COMMAND);
        break;
    case ID_DIR_RENAME:
        sel = MKUINT(XFileExplorer::ID_FILE_RENAME,SEL_COMMAND);
        break;
    case ID_DIR_SLINK:
        sel = MKUINT(XFileExplorer::ID_FILE_SLINK,SEL_COMMAND);
        break;
    }
    getApp()->getMainWindow()->handle(this,sel,strdup(param.text()));

	// Force dirpanel refresh
    list->onRefresh(0,0,0);
    return 1;
}


// Delete directory
long DirPanel::onCmdDirDel(FXObject*,FXSelector,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString pathname=list->getItemPathname((TreeItem*)item);	
	FXString parentdir=FXFile::directory(pathname);	

    // If we don't have permission to write to the parent directory
    if(access(parentdir.text(), W_OK))
    {
        MessageBox::error(this,MBOX_OK,_("Error"),_("Can't write to %s: Permission denied"), parentdir.text());
        return 0;
    }

    if(confirm_del)
    {
        FXPNGIcon icon(getApp(),delete_big);
		FXString message=_("Delete folder ") + pathname + " ?";
		MessageBox box(this,_("Confirm Delete"),message,&icon,MBOX_YES_NO|DECOR_TITLE|DECOR_BORDER);
        if(box.execute() != MBOX_CLICKED_YES)
            return 0;
    }

	// File object
   	File* f=new File(getApp(),_("File Delete"),DELETE);
   	f->create();

    // If we don't have permission to write to the directory
    if(access(pathname.text(), W_OK) && !access(pathname.text(), F_OK))
    {
		// Get process uid and directory uid
		FXuint uid=getuid();
		struct stat status;
		lstat(pathname.text(),&status);
		FXuint diruid=status.st_uid;

		// If we are owner of the directory
		if (diruid==uid)
		{
			// Dialog to confirm directory deletion
			f->allowPdialog=FALSE;
			f->hide();
			FXPNGIcon icon(getApp(),errorIcon);
			FXString str=_("Folder ") + pathname + _(" is write-protected, delete it anyway?");
        	MessageBox box(this,_("Confirm Delete"),str,&icon,MBOX_YES_NO|DECOR_TITLE|DECOR_BORDER);        	
			FXuint answer=box.execute();
			if(answer == MBOX_CLICKED_YES)
			{
				f->remove(pathname); 	

				// If action is cancelled in progress dialog
				if (f->isCancelled)
					MessageBox::error(this,MBOX_OK,_("Error"),_("Delete folder operation cancelled!"));
				
				// Return to parent directory in DirList and FileList
				list->setDirectory(parentdir);
				((XFileExplorer*) getApp()->getMainWindow())->lpanel->current->list->setDirectory(parentdir);
			}
		}
					
		// If we are not owner of the directory
		else
		{
			f->allowPdialog=FALSE;
			f->hide();
        	MessageBox::error(this,MBOX_OK,_("Error"),_("Can't delete folder %s: Permission denied"), pathname.text());
    	}
	}

	// If we have permission to write
	else
	{
		f->remove(pathname); 	

		// If action is cancelled in progress dialog
		if (f->isCancelled)
		{
			f->hide();
			MessageBox::error(this,MBOX_OK,_("Error"),_("Delete file operation cancelled!"));
		}
		// Return to parent directory in DirList and FileList
		list->setDirectory(parentdir);
		((XFileExplorer*) getApp()->getMainWindow())->lpanel->current->list->setDirectory(parentdir);
	}

	delete f;

	// Force DirPanel and FilePanel refresh
    list->onRefresh(0,0,0);
	((XFileExplorer*) getApp()->getMainWindow())->lpanel->onRefresh(0,0,0);
    return 1;
}


// Create new directory
long DirPanel::onCmdNewDir(FXObject*,FXSelector,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();

	FXString buf=list->getItemPathname((TreeItem*)item);
	FXString oldbuf=buf;
    FXString cmd = "mkdir ";
    
    InputDialog* dialog = new InputDialog(this,buf,_("Create new folder..."),_("New Folder"));
    dialog->CursorEnd();
    if(dialog->execute())
    {
        buf=dialog->getText();
		
		// Test some error conditions
		if (exists(dequote(buf)))
        {
			MessageBox::error(this,MBOX_OK,_("Error"),_("Folder %s already exists"), buf.text());
        	return 0;
		}
		FXString parentdir;
		if (FXFile::name(buf)=="")
			parentdir=FXFile::directory(FXFile::directory(buf));
		else			
			parentdir=FXFile::directory(buf);
		if (!exists(dequote(parentdir)))
        {
			MessageBox::error(this,MBOX_OK,_("Error"),_("Folder %s doesn't exist"), parentdir.text());
        	return 0;
		}
		if (!FXFile::isWritable(parentdir))
		{
         	MessageBox::error(this,MBOX_OK,_("Error"),_("Can't write to %s : Permission denied"), parentdir.text());
        	return 0;
		}
        if(oldbuf!=buf)
        {
            cmd+=buf;
            cmd+=" &";
            system(cmd.text());
        }
    }
    delete dialog;
	
	// Force dirpanel refresh
	list->onRefresh(0,0,0);
    return 1;
}


// Run Terminal in the selected directory
long  DirPanel::onCmdXTerm(FXObject*,FXSelector,void*)
{
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString buf=list->getItemPathname((TreeItem*)item);
	chdir(buf.text());
    FXString cmd = getApp()->reg().readStringEntry("PROGS","xterm","xterm -sb");
    cmd += " &";
    system(cmd.text());
    return 1;
}

// Paste file(s)
long DirPanel::onCmdPaste(FXObject*,FXSelector sel,void*)
{
    if(clipboard.empty())
        return 1;

    FXString param;
    FXString targetdir;

	targetdir=((XFileExplorer*) getApp()->getMainWindow())->lpanel->current->list->getDirectory();
    param = targetdir+"\n"+clipboard;

    switch(clptype)
    {
    case COPY_CLP:
        sel = MKUINT(XFileExplorer::ID_FILE_COPY,SEL_COMMAND);
        break;
    case CUT_CLP:
        sel = MKUINT(XFileExplorer::ID_FILE_CUT,SEL_COMMAND);
        break;
    }
    getApp()->getMainWindow()->handle(this,sel,strdup(param.text()));
    clipboard = "";

	// Force dirpanel refresh
	list->onRefresh(0,0,0);

    return 1;
}


// Clipboard
long DirPanel::onCmdClp(FXObject*,FXSelector sel,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString pathname=list->getItemPathname((TreeItem*)item);

	// Clipboard type
    clipboard = "";
	if (SELID(sel)==ID_CUT_CLP)
		clptype=CUT_CLP;
	else
		clptype=COPY_CLP;

	clipboard += pathname+"\n";
    clipboard = FXStringVal(1)+"\n"+clipboard;

    return 1;
}


#if defined(linux)
// Mount/Unmount directory
long DirPanel::onCmdMount(FXObject*,FXSelector sel,void*)
{
    FXString cmd, msg, text;
	unsigned int op;
	File *f;
	
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString dir=list->getItemPathname((TreeItem*)item);

	// If symbolic link
	if (isLink(dir))
		dir=FXFile::symlink(dir);

    // Select the command and set the appropriate message
    if(SELID(sel)==ID_MOUNT)
    {
        op=MOUNT;
		msg="Mount";
        cmd="/bin/mount ";
    }
    else
    {
        op=UNMOUNT;
		msg=_("Unmount");
        cmd="/bin/umount ";
    }
    cmd+=dir;
    cmd+=" 2>&1";
    chdir(ROOTDIR);

	// Wait cursor
	getApp()->beginWaitCursor();

	// File object
	text=msg + _(" file system...");
	f=new File(getApp(),text.text(),op);
	f->create();
				   
	// Mount/unmount file system
	text=msg + _(" the folder :");
	f->mount(dir,text,cmd,op);
 
	// If action is cancelled in progress dialog
	if (f->isCancelled)
	{
		f->hide();
		text=msg + _(" operation cancelled!");
		MessageBox::error(this,MBOX_OK,_("Error"),text.text());
		delete f;
		return 0;
	}					
	
	getApp()->endWaitCursor();
	delete f; 

	// Force dirpanel refresh
    list->onRefresh(0,0,0);

    return 1;
}
#endif


// Is it necessary?
long DirPanel::onUpdPaste(FXObject* o,FXSelector,void*)
{
    FXButton *button=(FXButton*)o;
    int style=button->getButtonStyle();
    if(style & BUTTON_TOOLBAR)
    {
        if(!clipboard.empty())
            o->handle(this,MKUINT(FXWindow::ID_ENABLE,SEL_COMMAND),NULL);
        else
            o->handle(this,MKUINT(FXWindow::ID_DISABLE,SEL_COMMAND),NULL);
    }
    else
        o->handle(this,MKUINT(FXWindow::ID_ENABLE,SEL_COMMAND),NULL);

    return 1;
}

// Update the status bar and the directory path name
long DirPanel::onUpdStatus(FXObject* sender,FXSelector,void*)
{
	// Name of the current selected item
	TreeItem* item=(TreeItem*)list->getCurrentItem();
	
	// There is no selected item
	if (item==NULL)
	{
		status->setText("Total");
		return 0;
	}
	
	// Path name of the selected item
	FXString path=list->getItemPathname(item);	
	
	// File doesn't exist
	if (!exists(dequote(path)))
	{
		status->setText("Total");
		return 0;
	}

	// Update the path name
	path=quote(path,FALSE);
	//chdir(path.text());

	// Size of the files in this directory (not recursive!)
	FILE *p;
	char buf[MAXPATHLEN+1];
	char size[64], hsize[64];
	sprintf(buf,"ls -sk %s 2> /dev/null",path.text());
	p=popen(buf,"r");
	if(!p)
	{
		MessageBox::error(this,MBOX_OK,_("Error"),_("Failed command : %s"),buf);
		return -1;
	}
	fgets(buf,sizeof(buf),p);
	strcpy(size,strtok(buf,"total\n"));
	pclose(p);

	// Size in human readable form
	sprintf(size,"%lu",atol(size)*1024);
	::Size(size,hsize);

	// Update the status label
	sprintf(buf,_("Total %s"),hsize);
	status->setText(buf);

    return 1;
}
