// File list. Taken from the FOX library and slightly modified.
#include "config.h"
#include "i18n.h"

#include <ctype.h>
#include <time.h>


#include <fox-1.4/fx.h>
#include <fox-1.4/fxkeys.h>
#include <fox-1.4/FXPNGIcon.h>
#include <fox-1.4/FXJPGIcon.h>
#include <fox-1.4/FXTIFIcon.h>
#if defined(linux)
#include <mntent.h>
#endif

#include "xfedefs.h"
#include "icons.h"
#include "FileList.h"
#include "InputDialog.h"
#include "File.h"
#include "FileDict.h"
#include "MessageBox.h"


// Minimum file name size in detailed view
#ifndef MIN_NAME_SIZE
#define MIN_NAME_SIZE 75
#endif

// For thumbnail image preview
#define MAX_BIG_ICON_SIZE 64
#define MAX_MINI_ICON_SIZE 20

// Time interval before refreshing the view
#define REFRESH_INTERVAL     1000
#define REFRESH_FREQUENCY    30

// Time interval before opening a folder
#define OPEN_INTERVAL		1000

#define HASH1(x,n) (((unsigned int)(x)*13)%(n))           // Probe Position [0..n-1]
#define HASH2(x,n) (1|(((unsigned int)(x)*17)%((n)-1)))   // Probe Distance [1..n-1]


#if defined(linux)
#define FSTAB_PATH "/etc/fstab"
#define MTAB_PATH "/proc/mounts"
FXStringDict* fsdevices=NULL; // Devices from fstab
FXStringDict* mtdevices=NULL; // Mounted devices
FXStringDict* updevices=NULL; // Responding devices
#endif


#define SELECT_MASK   (ICONLIST_EXTENDEDSELECT|ICONLIST_SINGLESELECT|ICONLIST_BROWSESELECT|ICONLIST_MULTIPLESELECT)


// The function below is taken from the FXIconList class
// and hacked to right justify the second column of the file list (file size)
// It replaces the normal function...

#define SIDE_SPACING             4    // Left or right spacing between items
#define DETAIL_TEXT_SPACING      2    // Spacing between text and icon in detail icon mode


void FXIconItem::drawDetails(const FXIconList* list,FXDC& dc,FXint x,FXint y,FXint,FXint h) const
{
    register FXHeader *header=list->getHeader();
    register FXFont *font=list->getFont();
    register FXint iw=0,ih=0,tw=0,th=0,yt,beg,end,hi,drw,space,used,dw,xx;
    if(header->getNumItems()==0)
        return;
    if(isSelected())
    {
        dc.setForeground(list->getSelBackColor());
        dc.fillRectangle(x,y,header->getTotalSize(),h);
    }
    if(hasFocus())
    {
        dc.drawFocusRectangle(x+1,y+1,header->getTotalSize()-2,h-2);
    }
    xx=x+SIDE_SPACING/2;
    if(miniIcon)
    {
        iw=miniIcon->getWidth();
        ih=miniIcon->getHeight();
        dc.setClipRectangle(x,y,header->getItemSize(0),h);
        dc.drawIcon(miniIcon,xx,y+(h-ih)/2);
        dc.clearClipRectangle();
        xx+=iw+DETAIL_TEXT_SPACING;
    }
    if(!label.empty())
    {
        th=font->getFontHeight();
        dw=font->getTextWidth("...",3);
        yt=y+(h-th-4)/2;
        if(!isEnabled())
            dc.setForeground(makeShadowColor(list->getBackColor()));
        else if(isSelected())
            dc.setForeground(list->getSelTextColor());
        else
            dc.setForeground(list->getTextColor());
        used=iw+DETAIL_TEXT_SPACING+SIDE_SPACING/2;
        for(hi=beg=0; beg<label.length() && hi<header->getNumItems(); hi++,beg=end+1)
        {
            space=header->getItemSize(hi)-used;
            for(end=beg; end<label.length() && label[end]!='\t'; end++)
                ;
            if(end>beg)
            {
                drw=end-beg;
                tw=font->getTextWidth(&label[beg],drw);
                if(tw>space-4)
                {
                    while((tw=font->getTextWidth(&label[beg],drw))+dw>space-4 && drw>1)
                        drw--;
                    dc.setClipRectangle(xx,y,space,h);
                    dc.drawText(xx+2,yt+font->getFontAscent()+2,&label[beg],drw);
                    dc.drawText(xx+tw+2,yt+font->getFontAscent()+2,"...",3);
                    dc.clearClipRectangle();
                }
                else
                {
                    // !!!! Hack to right justify the second column (file size) !!!!
                    if (hi==1)
                        dc.drawText(xx+space-tw-20,yt+font->getFontAscent()+2,&label[beg],drw);
                    else
                        dc.drawText(xx+2,yt+font->getFontAscent()+2,&label[beg],drw);
                    // !!!! End of hack
                }
            }
            xx+=space;
            used=0;
        }
    }
}


// Object implementation
FXIMPLEMENT(FileItem,FXIconItem,NULL,0)


// Map
FXDEFMAP(FileList) FileListMap[]={
                                     FXMAPFUNC(SEL_DRAGGED,0,FileList::onDragged),
                                     FXMAPFUNC(SEL_TIMEOUT,FileList::ID_REFRESH_TIMER,FileList::onRefreshTimer),
                                     FXMAPFUNC(SEL_TIMEOUT,FileList::ID_OPEN_TIMER,FileList::onOpenTimer),
                                     FXMAPFUNC(SEL_DND_ENTER,0,FileList::onDNDEnter),
                                     FXMAPFUNC(SEL_DND_LEAVE,0,FileList::onDNDLeave),
                                     FXMAPFUNC(SEL_DND_DROP,0,FileList::onDNDDrop),
                                     FXMAPFUNC(SEL_DND_MOTION,0,FileList::onDNDMotion),
                                     FXMAPFUNC(SEL_DND_REQUEST,0,FileList::onDNDRequest),
                                     FXMAPFUNC(SEL_BEGINDRAG,0,FileList::onBeginDrag),
                                     FXMAPFUNC(SEL_ENDDRAG,0,FileList::onEndDrag),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_DIRECTORY_UP,FileList::onCmdDirectoryUp),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_NAME,FileList::onCmdSortByName),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_TYPE,FileList::onCmdSortByType),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_SIZE,FileList::onCmdSortBySize),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_TIME,FileList::onCmdSortByTime),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_USER,FileList::onCmdSortByUser),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_BY_GROUP,FileList::onCmdSortByGroup),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_REVERSE,FileList::onCmdSortReverse),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SORT_CASE,FileList::onCmdSortCase),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SET_PATTERN,FileList::onCmdSetPattern),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SET_DIRECTORY,FileList::onCmdSetDirectory),
                                     FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETVALUE,FileList::onCmdSetValue),
                                     FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETSTRINGVALUE,FileList::onCmdSetStringValue),
                                     FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETSTRINGVALUE,FileList::onCmdGetStringValue),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_SHOW_HIDDEN,FileList::onCmdShowHidden),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_HIDE_HIDDEN,FileList::onCmdHideHidden),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_TOGGLE_HIDDEN,FileList::onCmdToggleHidden),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_TOGGLE_THUMBNAILS,FileList::onCmdToggleThumbnails),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_HEADER_CHANGE,FileList::onCmdHeader),
                                     FXMAPFUNC(SEL_COMMAND,FileList::ID_REFRESH,FileList::onCmdRefresh),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_HEADER_CHANGE,FileList::onUpdHeader),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_DIRECTORY_UP,FileList::onUpdDirectoryUp),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_NAME,FileList::onUpdSortByName),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_TYPE,FileList::onUpdSortByType),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_SIZE,FileList::onUpdSortBySize),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_TIME,FileList::onUpdSortByTime),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_USER,FileList::onUpdSortByUser),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_BY_GROUP,FileList::onUpdSortByGroup),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_REVERSE,FileList::onUpdSortReverse),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SORT_CASE,FileList::onUpdSortCase),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SET_PATTERN,FileList::onUpdSetPattern),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SET_DIRECTORY,FileList::onUpdSetDirectory),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_SHOW_HIDDEN,FileList::onUpdShowHidden),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_HIDE_HIDDEN,FileList::onUpdHideHidden),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_TOGGLE_HIDDEN,FileList::onUpdToggleHidden),
                                     FXMAPFUNC(SEL_UPDATE,FileList::ID_TOGGLE_THUMBNAILS,FileList::onUpdToggleThumbnails),
                                 };


// Object implementation
FXIMPLEMENT(FileList,FXIconList,FileListMap,ARRAYNUMBER(FileListMap))


// For serialization
FileList::FileList()
{
	flags|=FLAG_ENABLED|FLAG_DROPTARGET;
	sortfunc=ascendingCase;
	associations=NULL;
	list=NULL;
	timestamp=0;
	dropaction=DRAG_MOVE;
	counter=0;
};


// File List
FileList::FileList(FXComposite *p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
        FXIconList(p,tgt,sel,opts,x,y,w,h),directory(ROOTDIR),orgdirectory(ROOTDIR),pattern("*")
{
    flags|=FLAG_ENABLED|FLAG_DROPTARGET;
    associations=NULL;
    appendHeader(_("Name"),NULL,200);
    appendHeader(_("Size"),NULL,60);
    appendHeader(_("Type"),NULL,100);
    appendHeader(_("Modified Date"),NULL,150);
    appendHeader(_("User"),NULL,50);
    appendHeader(_("Group"),NULL,50);
    appendHeader(_("Permissions"),NULL,100);

    big_folder=new FXPNGIcon(getApp(),bigfolder);
    mini_folder=new FXPNGIcon(getApp(),minifolder);
    big_doc=new FXPNGIcon(getApp(),bigdoc);
    mini_doc=new FXPNGIcon(getApp(),minidoc);
    big_exec=new FXPNGIcon(getApp(),bigexec);
    mini_exec=new FXPNGIcon(getApp(),miniexec);
    big_folderopen=new FXPNGIcon(getApp(),bigfolderopen);
    mini_folderopen=new FXPNGIcon(getApp(),minifolderopen);
    big_folderup=new FXPNGIcon(getApp(),bigfolderup);
    mini_folderup=new FXPNGIcon(getApp(),minifolderup);
    big_lockfolder=new FXPNGIcon(getApp(),bigfolderlocked);
    mini_lockfolder=new FXPNGIcon(getApp(),minifolderlocked);
    big_cdev=new FXPNGIcon(getApp(),bigchardev);
    mini_cdev=new FXPNGIcon(getApp(),minichardev);
    big_bdev=new FXPNGIcon(getApp(),bigblockdev);
    mini_bdev=new FXPNGIcon(getApp(),miniblockdev);
    big_sock=new FXPNGIcon(getApp(),bigsocket);
    mini_sock=new FXPNGIcon(getApp(),minisocket);
    big_pipe=new FXPNGIcon(getApp(),bigpipe);
    mini_pipe=new FXPNGIcon(getApp(),minipipe);
    big_link=new FXPNGIcon(getApp(),biglink);
    mini_link=new FXPNGIcon(getApp(),minilink);
	big_brokenlink=new FXPNGIcon(getApp(),bigbrokenlink);
    mini_brokenlink=new FXPNGIcon(getApp(),minibrokenlink);
	big_harddisk=new FXPNGIcon(getApp(),bigharddisk);
    mini_harddisk=new FXPNGIcon(getApp(),harddisk);
    big_nfs=new FXPNGIcon(getApp(),bignfsdrive);
    mini_nfs=new FXPNGIcon(getApp(),nfsdrive);
    big_floppy=new FXPNGIcon(getApp(),bigfloppy);
    mini_floppy=new FXPNGIcon(getApp(),floppy);
    big_cdrom=new FXPNGIcon(getApp(),bigcdrom);
    mini_cdrom=new FXPNGIcon(getApp(),cdrom);
    big_zip=new FXPNGIcon(getApp(),bigzip);
    mini_zip=new FXPNGIcon(getApp(),zip);

    // Initializations
	matchmode=FILEMATCH_FILE_NAME|FILEMATCH_NOESCAPE;
	associations=new FileDict(getApp());
    dropaction=DRAG_MOVE;
    sortfunc=ascendingCase;
    ignorecase=TRUE;
	allowrefresh=TRUE;
    timestamp=0;
    counter=0;
    prevIndex=-1;
    list=NULL;
	displaythumbnails=FALSE;

#if defined(linux)

	// Initialize the fsdevices, mtdevices and updevices lists
	// if it was not done in DirList (useful for XFileView, XFileQuery and XFileImage) 
    struct mntent *mnt;
    if (fsdevices==NULL)
	{
		// To list file system devices
    	fsdevices=new FXStringDict();
		FILE *fstab=setmntent(FSTAB_PATH,"r");
    	if(fstab)
    	{
        	while((mnt=getmntent(fstab)))
        	{
            	if(!::streq(mnt->mnt_type,MNTTYPE_IGNORE) && !::streq(mnt->mnt_type,MNTTYPE_SWAP) && !::streq(mnt->mnt_type,"proc") && !::streq(mnt->mnt_dir,"/"))
            	{
                	if(!strncmp(mnt->mnt_fsname,"/dev/fd",7))
                    	fsdevices->insert(mnt->mnt_dir,"floppy");
                	else if (!strncmp(mnt->mnt_type,"iso",3))
                    	fsdevices->insert(mnt->mnt_dir,"cdrom");
                	else if (!strncmp(mnt->mnt_fsname,"/dev/zip",8))
                    	fsdevices->insert(mnt->mnt_dir,"zip");
                	else if (::streq(mnt->mnt_type,"nfs"))
                    	fsdevices->insert(mnt->mnt_dir,"nfsdisk");
                	else if (::streq(mnt->mnt_type,"smbfs"))
                    	fsdevices->insert(mnt->mnt_dir,"smbdisk");
                	else
                    	fsdevices->insert(mnt->mnt_dir,"harddisk");
            	}
        	}
        	endmntent(fstab);
    	}
	}
	if (mtdevices==NULL)
	{
		// To list mounted devices
    	mtdevices=new FXStringDict();
		FILE *mtab=setmntent(MTAB_PATH,"r");
		if(mtab)
		{
			while((mnt=getmntent(mtab)))
				mtdevices->insert(mnt->mnt_dir,mnt->mnt_type);
			endmntent(mtab);
		}
	}
	if (updevices==NULL)
	{
		// To mark mount points that are up or down
    	updevices=new FXStringDict();
 		struct stat statbuf;
		FXString mtstate;
		FILE *mtab=setmntent(MTAB_PATH,"r");
		if(mtab)
    	{
        	while((mnt=getmntent(mtab)))
			{
				if (mtlstat(mnt->mnt_dir,&statbuf)==-1)
					mtstate="down";
				else
					mtstate="up";	
				updevices->insert(mnt->mnt_dir,mtstate.text());
			}
        	endmntent(mtab);
    	}
	}
#endif
}


// Create the file list
void FileList::create()
{
    FXIconList::create();
    if(!deleteType)
        deleteType=getApp()->registerDragType(deleteTypeName);
    if(!urilistType)
        urilistType=getApp()->registerDragType(urilistTypeName);
	getApp()->addTimeout(this,ID_REFRESH_TIMER,REFRESH_INTERVAL);

    big_folder->create();
    mini_folder->create();
    big_doc->create();
    mini_doc->create();
    big_exec->create();
    mini_exec->create();
    big_folderopen->create();
    mini_folderopen->create();
    big_lockfolder->create();
    mini_lockfolder->create();
    big_cdev->create();
    mini_cdev->create();
    big_bdev->create();
    mini_bdev->create();
    big_sock->create();
    mini_sock->create();
    big_pipe->create();
    mini_pipe->create();
    big_folderup->create();
    mini_folderup->create();
    big_harddisk->create();
    mini_harddisk->create();
    big_link->create();
    mini_brokenlink->create();
    big_brokenlink->create();
    mini_link->create();
    big_nfs->create();
    mini_nfs->create();
    big_floppy->create();
    mini_floppy->create();
    big_cdrom->create();
    mini_cdrom->create();
    big_zip->create();
    mini_zip->create();
}


// Open up folder when hovering long over a folder
long FileList::onOpenTimer(FXObject*,FXSelector,void*)
{
    FXint xx,yy,index;
    FXuint state;
    getCursorPosition(xx,yy,state);
    index=getItemAt(xx,yy);
    if(0<=index && isItemDirectory(index))
    {
        dropdirectory=getItemPathname(index);
        setDirectory(dropdirectory);
        getApp()->addTimeout(this,ID_OPEN_TIMER,OPEN_INTERVAL);
        prevIndex=-1;
    }
    return 1;
}

// Handle drag-and-drop enter
long FileList::onDNDEnter(FXObject* sender,FXSelector sel,void* ptr)
{
    FXIconList::onDNDEnter(sender,sel,ptr);

    // Keep original directory
    orgdirectory=getDirectory();

    return 1;
}


// Handle drag-and-drop leave
long FileList::onDNDLeave(FXObject* sender,FXSelector sel,void* ptr)
{
    FXIconList::onDNDLeave(sender,sel,ptr);

    if(prevIndex!=-1)
    {
        setItemMiniIcon(prevIndex,mini_folder);
        setItemBigIcon(prevIndex,big_folder);
        prevIndex=-1;
    }

    // Cancel open up timer
	getApp()->removeTimeout(this,ID_OPEN_TIMER);

    // Stop scrolling
    stopAutoScroll();

    // Restore original directory
    setDirectory(orgdirectory);
    return 1;
}


// Handle drag-and-drop motion
long FileList::onDNDMotion(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent *event=(FXEvent*)ptr;
    FXint index=-1;

    // Cancel open up timer
	getApp()->removeTimeout(this,ID_OPEN_TIMER);

    // Start autoscrolling
    if(startAutoScroll(event,FALSE))
        return 1;

    // Give base class a shot
    if(FXIconList::onDNDMotion(sender,sel,ptr))
        return 1;

    // Dropping list of filenames
    if(offeredDNDType(FROM_DRAGNDROP,urilistType))
    {
        index=getItemAt(event->win_x,event->win_y);
        if(prevIndex!=-1 && prevIndex != index)
        {
            setItemMiniIcon(prevIndex,mini_folder);
            setItemBigIcon(prevIndex,big_folder);
            prevIndex=-1;
        }

        // Drop in the background
        dropdirectory=getDirectory();

        // What is being done (move,copy,link)
        dropaction=inquireDNDAction();

        // We will open up a folder if we can hover over it for a while
        if(0<=index && isItemDirectory(index))
        {
            // Set open up timer
            getApp()->addTimeout(this,ID_OPEN_TIMER,OPEN_INTERVAL);

            prevIndex=index;
            setItemMiniIcon(index,mini_folderopen);
            setItemBigIcon(index,big_folderopen);

            // Directory where to drop, or directory to open up
            dropdirectory=getItemPathname(index);
        }

        // See if dropdirectory is writable
        if(::isWritable(dropdirectory))
            acceptDrop(DRAG_ACCEPT);
        return 1;
    }
    return 0;
}


// Handle drag-and-drop drop
long FileList::onDNDDrop(FXObject* sender,FXSelector sel,void* ptr)
{
    FXuchar *data;
    FXuint len;
    FXbool showdialog=TRUE;
    File *f=NULL;

    FXbool ask_before_copy=getApp()->reg().readUnsignedEntry("OPTIONS","ask_before_copy",TRUE);
	
	if(prevIndex!=-1)
    {
        setItemMiniIcon(prevIndex,mini_folder);
        setItemBigIcon(prevIndex,big_folder);
        prevIndex=-1;
    }

    // Cancel open up timer
	getApp()->removeTimeout(this,ID_OPEN_TIMER);

    // Stop scrolling
    stopAutoScroll();

    // Restore original directory
    setDirectory(orgdirectory);

    // Perhaps target wants to deal with it
    if(FXIconList::onDNDDrop(sender,sel,ptr))
        return 1;

    // Get uri-list of files being dropped
    if(getDNDData(FROM_DRAGNDROP,urilistType,data,len))
    {
        FXRESIZE(&data,FXuchar,len+1);
        data[len]='\0';
        FXchar *p,*q;
        p=q=(FXchar*)data;
        
		// Number of selected items
		FXString buf=p;
        int num=buf.count('\n')+1;
		
		// Eventually correct the number of selected items
		// because sometimes there is another '\n' at the end of the string
		FXint pos=buf.rfind('\n');
		if (pos==buf.length()-1)
			num=num-1;

        // File object
		if (dropaction==DRAG_COPY)
            f=new File(this,_("File copy"),COPY);
        else if (dropaction==DRAG_MOVE)
				f=new File(this,_("File move"),MOVE);
		else if ((dropaction==DRAG_LINK) && (num==1))
            f=new File(this,_("File symlink"),SYMLINK);
        else if ((dropaction==DRAG_LINK) && (num>1))
		{
			MessageBox::error(this,MBOX_OK,_("Error"),_("Can't symlink multiple selection!"));
			return 0;
		}
        else
            return 0;

        while(*p)
        {
            while(*q && *q!='\r')
                q++;
            FXString url(p,q-p);
            FXString source(FXURL::fileFromURL(url));
            FXString target(dropdirectory+PATHSEPSTRING+FXFile::name(source));
            FXString sourcedir=FXFile::directory(source);
            FXString targetdir=FXFile::directory(target);

            // File operation dialog, if needed
            if(ask_before_copy & showdialog)
            {
                FXIcon *icon=NULL;
                FXString title,message;
                if (dropaction==DRAG_COPY)
                {
                    title=_("Copy ");
                    icon=new FXPNGIcon(getApp(),copy_big);
                    if (num==1)
                        message=title+source;
                    else
                        message=_("Copy ")+FXStringVal(num)+_(" files/folders.\nFrom: ")+sourcedir;
                }
                else if (dropaction==DRAG_MOVE)
                {
                    title=_("Move ");
                    icon=new FXPNGIcon(getApp(),move_big);
                    if (num==1)
                        message=title+source;
                    else
                        message=_("Move ")+FXStringVal(num)+_(" files/folders.\nFrom: ")+sourcedir;
                }
                else if ((dropaction==DRAG_LINK) && (num==1))
                {
                    title=_("Symlink ");
                    icon=new FXPNGIcon(getApp(),link_big);
					message=title+source;
                }

                InputDialog* dialog=new InputDialog(this,targetdir,message,title,_("To:"),icon);
                dialog->CursorEnd();
                int rc=1;
                rc=dialog->execute();
                target=dialog->getText();
                target=::filePath(target);
                if (num>1)
                    showdialog=FALSE;
                delete dialog;
                delete icon;
                if (!rc)
                    return 0;
            }

            // Move the source file
            if(dropaction==DRAG_MOVE)
            {
                // Move file
                f->create();
                f->move(source,target);

                // If action is cancelled in progress dialog
                if (f->isCancelled)
                {
                    f->hide();
                    MessageBox::error(this,MBOX_OK,_("Error"),_("Move file operation cancelled!"));
                }
            }
            // Copy the source file
            else if(dropaction==DRAG_COPY)
            {
                // Copy file
                f->create();
                f->copy(source,target);

                // If action is cancelled in progress dialog
                if (f->isCancelled)
                {
                    f->hide();
                    MessageBox::error(this,MBOX_OK,_("Error"),_("Copy file operation cancelled!"));
                }
            }
            // Link the source file (no progress dialog in this case)
            else if(dropaction==DRAG_LINK)
            {
                // Link file
                f->create();
                f->symlink(source,target);
            }
            if(*q=='\r')
                q+=2;
            p=q;
        }
        delete f;
        FXFREE(&data);
        return 1;
    }

    return 0;
}


// Somebody wants our dragged data
long FileList::onDNDRequest(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent *event=(FXEvent*)ptr;
    FXuchar *data;
    FXuint len;

    // Perhaps the target wants to supply its own data
    if(FXIconList::onDNDRequest(sender,sel,ptr))
        return 1;

    // Return list of filenames as a uri-list
    if(event->target==urilistType)
    {
        if(!dragfiles.empty())
        {
            len=dragfiles.length();
            FXMEMDUP(&data,dragfiles.text(),FXuchar,len);
            setDNDData(FROM_DRAGNDROP,event->target,data,len);
        }
        return 1;
    }

    // Delete selected files
    if(event->target==deleteType)
        return 1;

    return 0;
}



// Start a drag operation
long FileList::onBeginDrag(FXObject* sender,FXSelector sel,void* ptr)
{
	register FXint i;
	if(FXIconList::onBeginDrag(sender,sel,ptr))
		return 1;
  	if(beginDrag(&urilistType,1))
	{
    	dragfiles=FXString::null;
    	for(i=0; i<getNumItems(); i++)
		{
      		if(isItemSelected(i))
			{
        		if (getItemFilename(i)=="..")
					deselectItem(i);
				else
				{
					if(!dragfiles.empty())
						dragfiles+="\r\n";
        			dragfiles+=FXURL::fileToURL(getItemPathname(i));
				}
        	}
      	}
    	return 1;
	}
	return 0;
}


// End drag operation
long FileList::onEndDrag(FXObject* sender,FXSelector sel,void* ptr)
{
    if(FXIconList::onEndDrag(sender,sel,ptr))
        return 1;
    endDrag((didAccept()!=DRAG_REJECT));
    setDragCursor(getDefaultCursor());
    dragfiles=FXString::null;
    return 1;
}


// Dragged stuff around
long FileList::onDragged(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent* event=(FXEvent*)ptr;
    FXDragAction action;
    if(FXIconList::onDragged(sender,sel,ptr))
        return 1;
    action=DRAG_MOVE;
    if(event->state&CONTROLMASK)
        action=DRAG_COPY;
    if(event->state&SHIFTMASK)
        action=DRAG_MOVE;
	if((event->state&CONTROLMASK) && (event->state&SHIFTMASK))
		action=DRAG_LINK;
	
	// If source directory is read-only, convert move action to copy
	if (!::isWritable(orgdirectory) && (action==DRAG_MOVE))
		action=DRAG_COPY;
	
    handleDrag(event->root_x,event->root_y,action);
    if(didAccept()!=DRAG_REJECT)
    {
        if(action==DRAG_MOVE)
            setDragCursor(getApp()->getDefaultCursor(DEF_DNDMOVE_CURSOR));
    	else if(action==DRAG_LINK)
      		setDragCursor(getApp()->getDefaultCursor(DEF_DNDLINK_CURSOR));
        else
            setDragCursor(getApp()->getDefaultCursor(DEF_DNDCOPY_CURSOR));
    }
    else
        setDragCursor(getApp()->getDefaultCursor(DEF_DNDSTOP_CURSOR));
    return 1;
}


// Set value from a message
long FileList::onCmdSetValue(FXObject*,FXSelector,void* ptr)
{
    if(ptr)
        setCurrentFile((const FXchar*)ptr);
    return 1;
}


// Set current directory from dir part of filename
long FileList::onCmdSetStringValue(FXObject*,FXSelector,void* ptr)
{
    if(ptr==NULL)
        fxerror("%s::onCmdSetStringValue: NULL pointer.\n",getClassName());
    setCurrentFile(*((FXString*)ptr));
    return 1;
}


// Get current file name (NULL if no current file)
long FileList::onCmdGetStringValue(FXObject*,FXSelector,void* ptr)
{
    if(ptr==NULL)
        fxerror("%s::onCmdGetStringValue: NULL pointer.\n",getClassName());
    *((FXString*)ptr)=getCurrentFile();
    return 1;
}


// Toggle hidden files display
long FileList::onCmdToggleHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(!showHiddenFiles());
    return 1;
}


// Toggle thumbnails display
long FileList::onCmdToggleThumbnails(FXObject*,FXSelector,void*)
{
    showThumbnails(!displaythumbnails);
    return 1;
}


// Update toggle thumbnails button
long FileList::onUpdToggleThumbnails(FXObject* sender,FXSelector,void*)
{
    if(showThumbnails())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Update toggle hidden files button
long FileList::onUpdToggleHidden(FXObject* sender,FXSelector,void*)
{
    if(showHiddenFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Show hidden files
long FileList::onCmdShowHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(TRUE);
    return 1;
}


// Update show hidden files widget
long FileList::onUpdShowHidden(FXObject* sender,FXSelector,void*)
{
    if(showHiddenFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Hide hidden files
long FileList::onCmdHideHidden(FXObject*,FXSelector,void*)
{
    showHiddenFiles(FALSE);
    return 1;
}


// Update hide hidden files widget
long FileList::onUpdHideHidden(FXObject* sender,FXSelector,void*)
{
    if(!showHiddenFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Move up one level
long FileList::onCmdDirectoryUp(FXObject*,FXSelector,void*)
{
	setDirectory(FXFile::upLevel(directory));
    return 1;
}


// Determine if we can still go up more
long FileList::onUpdDirectoryUp(FXObject* sender,FXSelector,void* ptr)
{
    FXuint msg=FXFile::isTopDirectory(directory) ? ID_DISABLE : ID_ENABLE;
    sender->handle(this,FXSEL(SEL_COMMAND,msg),ptr);
    return 1;
}


// Change pattern
long FileList::onCmdSetPattern(FXObject*,FXSelector,void* ptr)
{
    if(!ptr)
        return 0;
    setPattern((const char*)ptr);
    return 1;
}


// Update pattern
long FileList::onUpdSetPattern(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETVALUE),(void*)pattern.text());
    return 1;
}


// Change directory
long FileList::onCmdSetDirectory(FXObject*,FXSelector,void* ptr)
{
    if(!ptr)
        return 0;
    setDirectory((const char*)ptr);
    return 1;
}


// Update directory
long FileList::onUpdSetDirectory(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_SETVALUE),(void*)directory.text());
    return 1;
}

// Sort by name
long FileList::onCmdSortByName(FXObject*,FXSelector,void*)
{
    if(sortfunc==ascending)
        sortfunc=descending;
    else if(sortfunc==ascendingCase)
        sortfunc=descendingCase;
    else if(sortfunc==descending)
        sortfunc=ascending;
    else
    {
        if (ignorecase)
            sortfunc=ascendingCase;
        else
            sortfunc=ascending;
    }
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortByName(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascending || sortfunc==descending || sortfunc==ascendingCase || sortfunc==descendingCase) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Sort by type
long FileList::onCmdSortByType(FXObject*,FXSelector,void*)
{
    sortfunc=(sortfunc==ascendingType) ? descendingType : ascendingType;
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortByType(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascendingType || sortfunc==descendingType) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Sort by size
long FileList::onCmdSortBySize(FXObject*,FXSelector,void*)
{
    sortfunc=(sortfunc==ascendingSize) ? descendingSize : ascendingSize;
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortBySize(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascendingSize || sortfunc==descendingSize) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Sort by time
long FileList::onCmdSortByTime(FXObject*,FXSelector,void*)
{
    sortfunc=(sortfunc==ascendingTime) ? descendingTime : ascendingTime;
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortByTime(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascendingTime || sortfunc==descendingTime) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Sort by user
long FileList::onCmdSortByUser(FXObject*,FXSelector,void*)
{
    sortfunc=(sortfunc==ascendingUser) ? descendingUser : ascendingUser;
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortByUser(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascendingUser || sortfunc==descendingUser) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Sort by group
long FileList::onCmdSortByGroup(FXObject*,FXSelector,void*)
{
    sortfunc=(sortfunc==ascendingGroup) ? descendingGroup : ascendingGroup;
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortByGroup(FXObject* sender,FXSelector,void*)
{
    sender->handle(this,(sortfunc==ascendingGroup || sortfunc==descendingGroup) ? FXSEL(SEL_COMMAND,ID_CHECK) : FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


// Reverse sort order
long FileList::onCmdSortReverse(FXObject*,FXSelector,void*)
{
    if(sortfunc==ascending)
        sortfunc=descending;
    else if(sortfunc==descending)
        sortfunc=ascending;
    else if(sortfunc==ascendingCase)
        sortfunc=descendingCase;
    else if(sortfunc==descendingCase)
        sortfunc=ascendingCase;
    else if(sortfunc==ascendingType)
        sortfunc=descendingType;
    else if(sortfunc==descendingType)
        sortfunc=ascendingType;
    else if(sortfunc==ascendingSize)
        sortfunc=descendingSize;
    else if(sortfunc==descendingSize)
        sortfunc=ascendingSize;
    else if(sortfunc==ascendingTime)
        sortfunc=descendingTime;
    else if(sortfunc==descendingTime)
        sortfunc=ascendingTime;
    else if(sortfunc==ascendingUser)
        sortfunc=descendingUser;
    else if(sortfunc==descendingUser)
        sortfunc=ascendingUser;
    else if(sortfunc==ascendingGroup)
        sortfunc=descendingGroup;
    else if(sortfunc==descendingGroup)
        sortfunc=ascendingGroup;
    scan(TRUE);
    return 1;
}


// Update sender
long FileList::onUpdSortReverse(FXObject* sender,FXSelector,void*)
{
    FXSelector selector=FXSEL(SEL_COMMAND,ID_UNCHECK);
    if(sortfunc==descending)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    else if(sortfunc==descendingCase)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    else if(sortfunc==descendingType)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    else if(sortfunc==descendingSize)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    else if(sortfunc==descendingTime)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    else if(sortfunc==descendingUser)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    else if(sortfunc==descendingGroup)
        selector=FXSEL(SEL_COMMAND,ID_CHECK);
    sender->handle(this,selector,NULL);
    return 1;
}


// Toggle case sensitivity
long FileList::onCmdSortCase(FXObject*,FXSelector,void*)
{
    if(sortfunc==ascending)
    {
        sortfunc=ascendingCase;
        ignorecase=TRUE;
    }
    else if(sortfunc==ascendingCase)
    {
        sortfunc=ascending;
        ignorecase=FALSE;
    }
    else if(sortfunc==descending)
    {
        sortfunc=descendingCase;
        ignorecase=TRUE;
    }
    else if(sortfunc==descendingCase)
    {
        sortfunc=descending;
        ignorecase=FALSE;
    }
    scan(TRUE);
    return 1;
}


// Update case sensitivity
long FileList::onUpdSortCase(FXObject* sender,FXSelector,void* ptr)
{
    if (sortfunc==ascending || sortfunc==descending)
    {
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),ptr);
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),ptr);
        ignorecase=FALSE;
    }
    else if (sortfunc==ascendingCase || sortfunc==descendingCase)
    {
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),ptr);
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),ptr);
        ignorecase=TRUE;
    }
    else
    {
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),ptr);
    }
    return 1;
}





// Clicked header button
long FileList::onCmdHeader(FXObject*,FXSelector,void* ptr)
{
    if(((FXuint)(FXuval)ptr)<6)
        handle(this,FXSEL(SEL_COMMAND,(ID_SORT_BY_NAME+(FXuint)(FXuval)ptr)),NULL);
    if (getHeaderSize(0)<MIN_NAME_SIZE)
        setHeaderSize(0,MIN_NAME_SIZE);
    return 1;
}


// Clicked header button
long FileList::onUpdHeader(FXObject*,FXSelector,void*)
{
    header->setArrowDir(0,(sortfunc==ascending || sortfunc==ascendingCase)  ? FALSE : (sortfunc==descending || sortfunc==descendingCase) ? TRUE : MAYBE);   // Name
    header->setArrowDir(1,(sortfunc==ascendingSize)  ? FALSE : (sortfunc==descendingSize) ? TRUE : MAYBE);   // Size
    header->setArrowDir(2,(sortfunc==ascendingType)  ? FALSE : (sortfunc==descendingType) ? TRUE : MAYBE);   // Type
    header->setArrowDir(3,(sortfunc==ascendingTime)  ? FALSE : (sortfunc==descendingTime) ? TRUE : MAYBE);   // Date
    header->setArrowDir(4,(sortfunc==ascendingUser)  ? FALSE : (sortfunc==descendingUser) ? TRUE : MAYBE);   // User
    header->setArrowDir(5,(sortfunc==ascendingGroup) ? FALSE : (sortfunc==descendingGroup)? TRUE : MAYBE);   // Group
    if (getHeaderSize(0)<MIN_NAME_SIZE)
        setHeaderSize(0,MIN_NAME_SIZE);
    return 1;
}


// Compare file names
FXint FileList::ascending(const FXIconItem* pa,const FXIconItem* pb)
{

    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return 0;
}


// Compare file names, case insensitive
FXint FileList::ascendingCase(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    while(1)
    {
        if(tolower(*p) > tolower(*q))
            return 1;
        if(tolower(*p) < tolower(*q))
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return 0;
}


// Compare file types
FXint FileList::ascendingType(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;
    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;

    register int i;
    for(i=2; *p && i; i-=(*p++=='\t'))
        ;
    for(i=2; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return FXIconList::ascendingCase(pa,pb);
}


// Compare file size - Warning : only returns the sign of the comparison!!!
FXint FileList::ascendingSize(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    register long long l=a->size - b->size;
    if(l)
    {
        if (l>=0)
            return 1;
        else
            return -1;
    }
    return FXIconList::ascendingCase(pa,pb);
}


// Compare file time
FXint FileList::ascendingTime(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    register long l=(long)a->date - (long)b->date;
    if(l)
        return l;
    return FXIconList::ascendingCase(pa, pb);
}


// Compare file user
FXint FileList::ascendingUser(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    for(i=4; *p && i; i-=(*p++=='\t'))
        ;
    for(i=4; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return FXIconList::ascendingCase(pa,pb);
}


// Compare file group
FXint FileList::ascendingGroup(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return diff;
    for(i=5; *p && i; i-=(*p++=='\t'))
        ;
    for(i=5; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return 1;
        if(*p < *q)
            return -1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return FXIconList::ascendingCase(pa,pb);
}


// Reversed compare file name, case insensitive
FXint FileList::descendingCase(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;
    while(1)
    {
        if(tolower(*p) > tolower(*q))
            return -1;
        if(tolower(*p) < tolower(*q))
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return 0;
}


// Reversed compare file name
FXint FileList::descending(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return 0;
}


// Reversed compare file type
FXint FileList::descendingType(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;

    register int i;
    for(i=2; *p && i; i-=(*p++=='\t'))
        ;
    for(i=2; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return -FXIconList::ascendingCase(pa,pb);
}


// Reversed compare file size
FXint FileList::descendingSize(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;
    register long long l=a->size - b->size;
    if(l)
    {
        if (l>=0)
            return -1;
        else
            return 1;
    }
    return -FXIconList::ascendingCase(pa,pb);
}


// Reversed compare file time
FXint FileList::descendingTime(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;
    register long l=(long)a->date - (long)b->date;
    if(l)
        return -l;
    return -FXIconList::ascendingCase(pa, pb);
}


// Reversed compare file user
FXint FileList::descendingUser(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;
    for(i=4; *p && i; i-=(*p++=='\t'))
        ;
    for(i=4; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return -FXIconList::ascendingCase(pa,pb);
}


// Reversed compare file group
FXint FileList::descendingGroup(const FXIconItem* pa,const FXIconItem* pb)
{
    register const FileItem *a=(FileItem*)pa;
    register const FileItem *b=(FileItem*)pb;

    register const unsigned char *p=(const unsigned char*)a->label.text();
    register const unsigned char *q=(const unsigned char*)b->label.text();

    // Directory '..' should always be on top
    if (p[0]=='.' && p[1]=='.' && p[2]=='\t')
        return -1;
    if (q[0]=='.' && q[1]=='.' && q[2]=='\t')
        return 1;

    register int i;
    register FXint diff=(FXint)b->isDirectory() - (FXint)a->isDirectory();
    if(diff)
        return -diff;
    for(i=5; *p && i; i-=(*p++=='\t'))
        ;
    for(i=5; *q && i; i-=(*q++=='\t'))
        ;
    while(1)
    {
        if(*p > *q)
            return -1;
        if(*p < *q)
            return 1;
        if(*p<='\t')
            break;
        p++;
        q++;
    }
    return -FXIconList::ascendingCase(pa,pb);
}


// Scan items to see if listing is necessary
void FileList::scan(FXbool force)
{
    struct stat info;

    // Stat the current directory
    if(::info(directory,info))
    {
        // New date of directory
        FXTime newdate=(FXTime)FXMAX(info.st_mtime,info.st_ctime);

        // Forced, date was changed, or failed to get proper date or counter expired
        if(force || (timestamp!=newdate) || (counter==0))
        {
            // And do the refresh
            listItems();
            sortItems();

            // Remember when we did this
            timestamp=newdate;
        }
    }

    // Move to higher directory
    else
        setDirectory(FXFile::upLevel(directory));
}


// Force an immediate update of the list
long FileList::onCmdRefresh(FXObject*,FXSelector,void*)
{
    allowrefresh=TRUE;
	scan(TRUE);
    return 1;
}

// Allow or forbid file list refresh
void FileList::setAllowRefresh(const FXbool allow)
{
	if (allow==FALSE)
		allowrefresh=FALSE;
	else
		allowrefresh=TRUE;
}


// Refresh; don't update if user is interacting with the list
long FileList::onRefreshTimer(FXObject*,FXSelector,void*)
{
    if(flags&FLAG_UPDATE && allowrefresh)
    {
        scan(FALSE);
        counter=(counter+1)%REFRESH_FREQUENCY;
    }

    // Reset timer again
    getApp()->addTimeout(this,ID_REFRESH_TIMER,REFRESH_INTERVAL);
    return 0;
}


// Set current filename
void FileList::setCurrentFile(const FXString& pathname)
{    // FIXME notify argument?
    if(!pathname.empty())
    {
        setDirectory(FXFile::directory(pathname));
        setCurrentItem(findItem(FXFile::name(pathname)));
    }
}


// Get pathname to current file, if any
FXString FileList::getCurrentFile() const
{
    if(current<0)
        return FXString::null;
    return getItemPathname(current);
}


// Set directory being displayed
void FileList::setDirectory(const FXString& pathname)
{
	// FIXME notify argument?
    if(!pathname.empty())
    {
        FXString path=FXFile::absolute(directory,pathname);
        while(!FXFile::isTopDirectory(path) && !::isDirectory(path))
            path=FXFile::upLevel(path);
        if(directory!=path)
        {
            directory=path;
            clearItems();
            list=NULL;
            scan(TRUE);
        }
    }	
}


// Set the pattern to filter
void FileList::setPattern(const FXString& ptrn)
{
    if(ptrn.empty())
        return;
    if(pattern!=ptrn)
    {
        pattern=ptrn;
		scan(TRUE);
    }
}


// Change file match mode
void FileList::setMatchMode(FXuint mode)
{
    if(matchmode!=mode)
    {
        matchmode=mode;
		scan(TRUE);
    }
}


// Return TRUE if showing hidden files
FXbool FileList::showHiddenFiles() const
{
    return (options&FILELIST_SHOWHIDDEN)!=0;
}


// Change show hidden files mode
void FileList::showHiddenFiles(FXbool shown)
{
    FXuint opts=shown ? (options|FILELIST_SHOWHIDDEN) : (options&~FILELIST_SHOWHIDDEN);
    if(opts!=options)
    {
        options=opts;
		scan(TRUE);
    }
}


// Return show thumbnails
FXbool FileList::showThumbnails() const
{
	return displaythumbnails;
}

// Change show thumbnails mode
void FileList::showThumbnails(FXbool display)
{
	displaythumbnails=display;
		
	// Refresh to display or hide thumbnails 
	scan(TRUE);
}


// Return TRUE if showing directories only
FXbool FileList::showOnlyDirectories() const
{
    return (options&FILELIST_SHOWDIRS)!=0;
}


// Change show directories only mode
void FileList::showOnlyDirectories(FXbool shown)
{
    FXuint opts=shown ? (options|FILELIST_SHOWDIRS) : (options&~FILELIST_SHOWDIRS);
    if(opts!=options)
    {
        options=opts;
		scan(TRUE);
    }
}


// Compare till '\t' or '\0'
static FXbool fileequal(const FXString& a,const FXString& b)
{
  register const FXuchar *p1=(const FXuchar *)a.text();
  register const FXuchar *p2=(const FXuchar *)b.text();
  register FXint c1,c2;
  do
  {
    c1=*p1++;
    c2=*p2++;
	}
  while(c1!='\0' && c1!='\t' && c1==c2);
  return (c1=='\0' || c1=='\t') && (c2=='\0' || c2=='\t');
}


// Create custom item
FXIconItem *FileList::createItem(const FXString& text,FXIcon *big,FXIcon* mini,void* ptr)
{
    return new FileItem(text,big,mini,ptr);
}


// List directory (from Fox library)
void FileList::listItems()
{
    FileItem *oldlist, *newlist;
    FileItem **po, **pn, **pp;
    FXString grpid, usrid, atts, mod;
    FXString name, dirname, pathname;
    FileItem *curitem=NULL;
    FileItem *item, *link;
    FileAssoc *fileassoc;
    FXString extension;
    FXIcon *big, *mini;
	FXbool image;
    time_t filetime;
    struct stat info;
    struct dirent *dp;
    DIR *dirp;
    int islink;

	// Start wait cursor
	getApp()->beginWaitCursor();

  	// Build old and new insert-order lists
  	oldlist=list;
  	newlist=NULL;

  	// Head of old and new lists
  	po=&oldlist;
  	pn=&newlist;
  
    // Remember current item
    if(0<=current)
        curitem=(FileItem*)items[current];

    // Start inserting
    items.clear();

    // Get info about directory
    if(statout(directory.text(),&info)==0)
    {
        // Need latest change no matter what actually changed!
        timestamp=FXMAX(info.st_mtime,info.st_ctime);

        // Set path to stat with
		dirname=directory.text();

    	if(dirname!=ROOTDIR) 
			dirname+=PATHSEPSTRING;

        // Get directory stream pointer
        dirp=opendir(directory.text());

        // Managed to open directory
        if(dirp)
        {
       		// Loop over directory entries
#ifdef FOX_THREAD_SAFE
        	struct fxdirent dirresult;
        	while(!readdir_r(dirp,&dirresult,&dp) && dp)
        	{
#else
        	while((dp=readdir(dirp))!=NULL)
			{
#endif
                name=dp->d_name;

                // Hidden file (.xxx) or directory (. or .yyy) normally not shown,
                // but directory .. is always shown so we can navigate up or down
                if(name[0]=='.' && (name[1]==0 || (!(name[1]=='.'&& name[2]==0) && !(options&FILELIST_SHOWHIDDEN))))
                    continue;

                // Build full pathname
				pathname=dirname+name;

				// Get file/link info
                if(lstatout(pathname.text(),&info)!=0)
                    continue;

                // If its a link, get the info on file itself
                islink=S_ISLNK(info.st_mode);

 				// If it is a link, modify the extension
                if(islink)
                {
                    // If it is a broken link
                    if (statout(pathname.text(),&info)!=0)
                        extension=_("Broken link to ");
                    else
                        extension=_("Link to ");
                }
                else
                    extension="";

                // If not a directory and we want only directories, skip it
                if(!S_ISDIR(info.st_mode) && (options&FILELIST_SHOWDIRS))
                    continue;

                // Is it a directory or does it match the pattern?
                if(!S_ISDIR(info.st_mode) && !FXFile::match(pattern,name,matchmode))
                    continue;

                // Anything about file has changed
                filetime=info.st_mtime;

            	// Find it, and take it out from the old list if found
            	for(pp=po; (item=*pp)!=NULL; pp=&item->link)
            	{
                	if(fileequal(item->label,name))
                	{
                    	*pp=item->link;
                    	item->link=NULL;
                    	po=pp;
                    	goto fnd;
                	}
            	}

                // Make new item if we have to
                item=(FileItem*)createItem(NULL,NULL,NULL,NULL);

                // Append item in list
fnd:          	*pn=item;
            	pn=&item->link;

            	// Append
      			if(item==curitem)
					current=items.no();
      			items.append(item);

				// Obtain user name
                usrid=FXFile::owner(info.st_uid);

                // Obtain group name
                grpid=FXFile::group(info.st_gid);

                // Permissions
                atts=FXFile::permissions(info.st_mode);

                // Mod time
                mod=FXFile::time("%x %X",filetime);

				// Item flags
                if(S_ISDIR(info.st_mode))
                    item->state|=FileItem::FOLDER;
                else
                    item->state&=~FileItem::FOLDER;
                if(S_ISLNK(info.st_mode))
                    item->state|=FileItem::SYMLINK;
                else
                    item->state&=~FileItem::SYMLINK;
                if(S_ISCHR(info.st_mode))
                    item->state|=FileItem::CHARDEV;
                else
                    item->state&=~FileItem::CHARDEV;
                if(S_ISBLK(info.st_mode))
                    item->state|=FileItem::BLOCKDEV;
                else
                    item->state&=~FileItem::BLOCKDEV;
                if(S_ISFIFO(info.st_mode))
                    item->state|=FileItem::FIFO;
                else
                    item->state&=~FileItem::FIFO;
                if(S_ISSOCK(info.st_mode))
                    item->state|=FileItem::SOCK;
                else
                    item->state&=~FileItem::SOCK;
                if((info.st_mode&(S_IXUSR|S_IXGRP|S_IXOTH)) && !(S_ISDIR(info.st_mode)||S_ISCHR(info.st_mode)||S_ISBLK(info.st_mode)||S_ISFIFO(info.st_mode)||S_ISSOCK(info.st_mode)))
                    item->state|=FileItem::EXECUTABLE;
                else
                    item->state&=~FileItem::EXECUTABLE;

                // We can drag items
                item->state|=FileItem::DRAGGABLE;

                // Assume no associations
                fileassoc=NULL;

                // Determine icons and type
                if(item->state&FileItem::FOLDER)
                {
				    if(!::isReadExecutable(pathname))
                    {
                        big=big_lockfolder;
                        mini=mini_lockfolder;
                        extension+=_("Folder");
                    }
                    else
                    {
                        big=big_folder;
                        mini=mini_folder;
                        extension+=_("Folder");
                    }
                    if(associations)
                        fileassoc=associations->findDirBinding(pathname.text());
                }
                else if(item->state&FileItem::CHARDEV)
    			{
    				big=big_cdev;
    				mini=mini_cdev;
    				extension+=_("Character Device");
    			}
    			else if(item->state&FileItem::BLOCKDEV)
    			{
    				big=big_bdev;
    				mini=mini_bdev;
    				extension+=_("Block Device");
    			}
    			else if(item->state&FileItem::FIFO)
    			{
    				big=big_pipe;
    				mini=mini_pipe;
    				extension+=_("Named Pipe");
    			}
    			else if(item->state&FileItem::SOCK)
                {
                	big=big_sock;
                	mini=mini_sock;
                	extension+=_("Socket");
                }
                else if(item->state&FileItem::EXECUTABLE)
                {
                	big=big_exec;
                	mini=mini_exec;
                	extension+=_("Executable");
                	if(associations)
					{
						fileassoc=associations->findFileBinding(pathname.text());					
						
						// In some special cases where the file is executable and its file name
						// is equal to the short extension (ex: zip, cc, ...) we don't want to use the association
						if ( fileassoc && (name==fileassoc->key) )
							fileassoc=NULL;
					}					
                }
                else
                {
                	big=big_doc;
               		mini=mini_doc;
                	extension+=_("Document");
                	if(associations)
                		fileassoc=associations->findFileBinding(pathname.text());
                }

                // If association is found, use it
				if (fileassoc)
				{														
					// If it is a link, modify the extension
                	if(islink)
                	{
                    	// If it is a broken link
                		if (statout(pathname.text(),&info)!=0)
                        	extension=_("Broken link to ");
                    	else
                        	extension=_("Link to ");

                    	// Complete the extension
                    	extension+=fileassoc->extension.text();
                	}
                	else
                    	extension=fileassoc->extension.text();

                	if(fileassoc->bigicon)
                    	big=fileassoc->bigicon;
                	if(fileassoc->miniicon)
                    	mini=fileassoc->miniicon;
												
					// Eventually display thumbnails
					if (displaythumbnails)
					{
						image=FALSE;
						if(extension=="GIF Image")
						{
							big=new FXGIFIcon(getApp());
							mini=new FXGIFIcon(getApp());
							image=TRUE;
						}
						else if (extension=="BMP Image")
						{
							big=new FXBMPIcon(getApp());
							mini=new FXBMPIcon(getApp());
							image=TRUE;
						}
						else if (extension=="X Pixmap")
						{
							big=new FXXPMIcon(getApp());
							mini=new FXXPMIcon(getApp());
							image=TRUE;
						}
						else if (extension=="TIFF Image")
						{
							big=new FXTIFIcon(getApp());
							mini=new FXTIFIcon(getApp());
							image=TRUE;
						}
						else if (extension=="PNG Image")
						{
							big=new FXPNGIcon(getApp());
							mini=new FXPNGIcon(getApp());
							image=TRUE;
						}
						else if (extension=="JPEG Image")
						{
							big=new FXJPGIcon(getApp());
							mini=new FXJPGIcon(getApp());
							image=TRUE;
						}
						if (image)
						{
							FXFileStream str;

							if(str.open(pathname,FXStreamLoad))
							{
           						big->loadPixels(str);
           						str.close();
 								
								// Eventually scale the big icon (best quality)
								if((big->getWidth()>MAX_BIG_ICON_SIZE) || (big->getHeight()>MAX_BIG_ICON_SIZE))
								{
             						if(big->getWidth()>big->getHeight())
               							big->scale(MAX_BIG_ICON_SIZE,(MAX_BIG_ICON_SIZE*big->getHeight())/big->getWidth(),1);
              						else
               							big->scale((MAX_BIG_ICON_SIZE*big->getWidth())/big->getHeight(),MAX_BIG_ICON_SIZE,1);
             					}
								
 								// Copy the mini icon to the big icon (faster!)
   								if (!FXMEMDUP(&tmpdata,big->getData(),FXColor,big->getWidth()*big->getHeight()))
									throw FXMemoryException(_("Unable to load image"));
								mini->setData(tmpdata,IMAGE_OWNED,big->getWidth(),big->getHeight());	

								// Eventually scale the mini icon (best quality)
								if((mini->getWidth()>MAX_MINI_ICON_SIZE) || (mini->getHeight()>MAX_MINI_ICON_SIZE))
								{
             						if(mini->getWidth()>mini->getHeight())
               							mini->scale(MAX_MINI_ICON_SIZE,(MAX_MINI_ICON_SIZE*mini->getHeight())/mini->getWidth(),1);
              						else
               							mini->scale((MAX_MINI_ICON_SIZE*mini->getWidth())/mini->getHeight(),MAX_MINI_ICON_SIZE,1);
             					}
								
							}
						}
					}
				}

                // Update item information
				FXchar size[64];
                snprintf(size,sizeof(size)-1,"%llu",(unsigned long long)info.st_size);
                FXString hsize=::hSize(size);
                item->label.format("%s\t%s\t%s\t%s\t%s\t%s\t%s",name.text(),hsize.text(),extension.text(),mod.text(),usrid.text(),grpid.text(),atts.text());
                item->bigIcon=big;
                item->miniIcon=mini;
                item->size=(unsigned long long)info.st_size;
                item->assoc=fileassoc;
                item->date=filetime;

#if defined(linux)
                 // Devices have a specific icon
				if(fsdevices->find(pathname.text()))
                {
					if(::streq(fsdevices->find(pathname.text()),"harddisk"))
                	{
                		item->bigIcon=big_harddisk;
                		item->miniIcon=mini_harddisk;
                	}
					else if(::streq(fsdevices->find(pathname.text()),"nfsdisk"))
                	{
                		item->bigIcon=big_nfs;
                		item->miniIcon=mini_nfs;
                	}
					else if(::streq(fsdevices->find(pathname.text()),"smbdisk"))
                	{
                		item->bigIcon=big_nfs;
                		item->miniIcon=mini_nfs;
                	}
                	else if(::streq(fsdevices->find(pathname.text()),"floppy"))
                	{
                		item->bigIcon=big_floppy;
                		item->miniIcon=mini_floppy;
                	}
                	else if(::streq(fsdevices->find(pathname.text()),"cdrom"))
                	{
                		item->bigIcon=big_cdrom;
                		item->miniIcon=mini_cdrom;
					}
                	else if(::streq(fsdevices->find(pathname.text()),"zip"))
                	{
                		item->bigIcon=big_zip;
                		item->miniIcon=mini_zip;
                	}
				}
#endif
                // Dotdot folders have a specific icon
                if(name[0]=='.' && name[1]=='.' && name[2]==0)
                {
                	item->bigIcon=big_folderup;
                	item->miniIcon=mini_folderup;
                }
 
                // Symbolic links have a specific icon
                if (islink)
                {
                	
					// Broken links
					if (statout(pathname.text(),&info)!=0)
					{
						item->bigIcon=big_brokenlink;
						item->miniIcon=mini_brokenlink;	
					}
					else
					{
						item->bigIcon=big_link;
						item->miniIcon=mini_link;
					}
                }
                // Create item
                if(id())
                	item->create();
			}
            closedir(dirp);
		}
	}

    // Wipe items remaining in list:- they have disappeared!!
    for(item=oldlist; item; item=link)
    {
    	link=item->link;
    	delete item;
    }

    // Validate
    if(current>=items.no())
    	current=-1;
    if(anchor>=items.no())
    	anchor=-1;
    if(extent>=items.no())
    	extent=-1;

    // Remember new list
    list=newlist;

    // Gotta recalc size of content
    recalc();

	// Stop wait cursor
	getApp()->endWaitCursor();
}


// Is directory
FXbool FileList::isItemDirectory(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::isItemDirectory: index out of range.\n",getClassName());
    return (((FileItem*)items[index])->state&FileItem::FOLDER)!=0;
}


// Is file
FXbool FileList::isItemFile(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::isItemFile: index out of range.\n",getClassName());
    return (((FileItem*)items[index])->state&(FileItem::FOLDER|FileItem::CHARDEV|FileItem::BLOCKDEV|FileItem::FIFO|FileItem::SOCK))==0;
}


// Is executable
FXbool FileList::isItemExecutable(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::isItemExecutable: index out of range.\n",getClassName());
    return (((FileItem*)items[index])->state&FileItem::EXECUTABLE)!=0;
}

// Get number of selected items
FXint FileList::getNumSelectedItems(void) const
{
    FXint num=0;
    for (int u=0; u<getNumItems(); u++)
        if (isItemSelected(u))
            num++;
    return num;
}


// Get number of selected items and index of first selected item
FXint FileList::getNumSelectedItems(FXint *index) const
{
    FXint num=0, itm=-1;
    for (int u=0; u<getNumItems(); u++)
        if (isItemSelected(u))
        {
            if (itm== -1)
                itm= u;
            num++;
        }
    (*index)=itm;
    return num;
}


// Get file name from item
FXString FileList::getItemFilename(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::getItemFilename: index out of range.\n",getClassName());
    FXString label=items[index]->getText();
    return label.section('\t',0);
}


// Get full pathname to item
FXString FileList::getItemPathname(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::getItemPathname: index out of range.\n",getClassName());
    FXString label=items[index]->getText();
    return FXFile::absolute(directory,label.section('\t',0));
}


// Get associations (if any) from the file
FileAssoc* FileList::getItemAssoc(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::getItemAssoc: index out of range.\n",getClassName());
    return ((FileItem*)items[index])->assoc;
}

// Return file size of the item
unsigned long long FileList::getItemFileSize(FXint index) const
{
    if(index<0 || items.no()<=index)
        fxerror("%s::getItemFileSize: index out of range.\n",getClassName());
    return ((FileItem*)items[index])->size;
}


// Change associations table; force a rescan so as to
// update the bindings in each item to the new associations
void FileList::setAssociations(FileDict* assocs)
{
    if(associations!=assocs)
    {
        associations=assocs;
		scan(TRUE);
    }
}


// Cleanup
FileList::~FileList()
{
	getApp()->removeTimeout(this,ID_REFRESH_TIMER);
	getApp()->removeTimeout(this,ID_OPEN_TIMER);
	delete associations;

    associations=(FileDict*)-1;
	list=(FileItem*)-1L;

    delete big_folder;
    delete mini_folder;
    delete big_doc;
    delete mini_doc;
    delete big_exec;
    delete mini_exec;
    delete big_folderopen;
    delete mini_folderopen;
    delete big_folderup;
    delete mini_folderup;
    delete big_lockfolder;
    delete mini_lockfolder;
    delete big_sock;
    delete mini_sock;
    delete big_pipe;
    delete mini_pipe;
    delete big_cdev;
    delete mini_cdev;
    delete big_bdev;
    delete mini_bdev;
    delete big_link;
    delete mini_link;
    delete big_brokenlink;
    delete mini_brokenlink;
    delete big_harddisk;
    delete mini_harddisk;
    delete big_nfs;
    delete mini_nfs;
    delete big_floppy;
    delete mini_floppy;
    delete big_cdrom;
    delete mini_cdrom;
    delete big_zip;
    delete mini_zip;

    big_folder=(FXIcon*)-1;
    mini_folder=(FXIcon*)-1;
    big_doc=(FXIcon*)-1;
    mini_doc=(FXIcon*)-1;
    big_exec=(FXIcon*)-1;
    mini_exec=(FXIcon*)-1;
    big_folderopen=(FXIcon*)-1;
    mini_folderopen=(FXIcon*)-1;
    big_folderup=(FXIcon*)-1;
    mini_folderup=(FXIcon*)-1;
    big_lockfolder=(FXIcon*)-1;
    mini_lockfolder=(FXIcon*)-1;
    big_sock=(FXIcon*)-1;
    mini_sock=(FXIcon*)-1;
    big_pipe=(FXIcon*)-1;
    mini_pipe=(FXIcon*)-1;
    big_cdev=(FXIcon*)-1;
    mini_cdev=(FXIcon*)-1;
    big_bdev=(FXIcon*)-1;
    mini_bdev=(FXIcon*)-1;
    big_link=(FXIcon*)-1;
    mini_link=(FXIcon*)-1;
    big_harddisk=(FXIcon*)-1;
    mini_harddisk=(FXIcon*)-1;
    big_nfs=(FXIcon*)-1;
    mini_nfs=(FXIcon*)-1;
    big_floppy=(FXIcon*)-1;
    mini_floppy=(FXIcon*)-1;
    big_cdrom=(FXIcon*)-1;
    mini_cdrom=(FXIcon*)-1;
    big_zip=(FXIcon*)-1;
    mini_zip=(FXIcon*)-1;
}


