

/*
 * Copyright (C) 2002-4 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/wait.h>
#include <limits.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#include <libxfcegui4/dialogs.h>

#include "glade_find_gui.h"
#include "glade_support.h"

#include "constants.h"
#include "types.h"
#include "modules.h"

#include "basic_row.h"

#include "find.h"

#include "entry.h"
#include "icons.h"
#include "misc.h"
#include "monitor.h"
#include "tubo.h"
#include "widgets.h"
#include "string_option_menu.h"

#define HISTORY_LIMIT 20
#define GLOB "fgr"
#define FIND "find"

/*#define DEBUG*/

static time_t initial,deltatime;
static xffind_functions *xffind_fun=NULL;

static xfc_combo_info_t *find_combo_info=NULL;
static xfc_combo_info_t *findpath_combo_info=NULL;
static xfc_combo_info_t *findgrep_combo_info=NULL;


static gint stop_watch;
static short int findCount;	/* how many files found */
static short int fileLimit = 64;
static gboolean cancelled;
static pid_t Gpid;

static GList *type_list = NULL;
static GList *find_results_list = NULL;
static xfdir_t find_xfdir;
/*static char *find_path;*/

static GtkTreeView *find_treeview;
static GtkTreeIter results_iter;

static void on_find_clicked_wrapper(GtkEntry *entry,gpointer data);

static void destroy(GtkWidget * widget, gpointer data);
static void on_find_clicked(GtkButton * button, gpointer user_data);
static void on_find_close(GtkButton * button, gpointer data);
static void on_help_filter(GtkToggleButton * button, gpointer user_data);
static void on_help_grep(GtkToggleButton * button, gpointer user_data);
static char *ftypes[] = {
    N_("Any kind"),
    N_("Regular"),
    N_("Directory"),
    N_("Symlink"),
    N_("Socket"),
    N_("Block device"),
    N_("Character device"),
    N_("FIFO"),
    NULL
};
static char *ft[] = {
    "any",
    "f",
    "d",
    "l",
    "s",
    "b",
    "c",
    "p",
    NULL
};

#if 0

static char *find_ft[] = {
    "any",
    "reg",
    "dir",
    "sym",
    "sock",
    "blk",
    "chr",
    "fifo",
    NULL
};
static gint on_key_press(GtkWidget * entry, GdkEventKey * event, gpointer data)
{
    void ok_input(GtkTreeView * treeview);
    GtkTreeView *treeview = (GtkTreeView *) data;
    GtkEditable *editable=(GtkEditable *)entry;

    /*printf("DBG(1):got key= 0x%x\n",event->keyval);*/
 if (event->keyval == GDK_BackSpace && (event->state&GDK_CONTROL_MASK)){
	gtk_editable_delete_text (editable,0,-1);
	return TRUE;
 }
    if(event->keyval == GDK_Escape)
    {
	destroy(lookup_widget((GtkWidget *) entry, "find_dialog"),(GtkWidget *) treeview);
	gtk_widget_grab_focus((GtkWidget *) treeview);
	return (TRUE);
    }

    return (FALSE);
}
#endif

static void save_ff_text(char *p)
{				
    gchar *fname;
    if (!p || !strlen(p)) return;
    {
	gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
	fname=g_build_filename(xdg_dir,FIND_DBH_FILE,NULL);
	g_free(xdg_dir);
    }
    XFC_save_to_history(fname,p);
    g_free(fname);
}
static void save_fpath_text(char *p)
{				
    gchar *fname;
    if (!p || !strlen(p)) return;
    {
	gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
	fname=g_build_filename(xdg_dir,FIND_PATHS_DBH_FILE,NULL);
	g_free(xdg_dir);
    }
    XFC_save_to_history(fname,p);
    g_free(fname);
}
static void save_fgrep_text(char *p)
{				
    gchar *fname;
    if (!p || !strlen(p)) return;
    {
	gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
	fname=g_build_filename(xdg_dir,FIND_GREP_DBH_FILE,NULL);
	g_free(xdg_dir);
    }
    XFC_save_to_history(fname,p);
    g_free(fname);
}


static void find_over(void)
{
    gint tree_id = get_active_tree_id();
    GtkTreeView *treeview = tree_details->treestuff[tree_id].treeview;
    GtkTreeModel *treemodel = tree_details->treestuff[tree_id].treemodel;

    GtkTreePath *treepath;
    GtkTreeIter iter;
    tree_entry_t *en;

    unset_load_wait();

    get_find_root(treeview, &iter, &en);
    erase_dummy_row(treemodel, &iter,NULL);
    treepath = gtk_tree_model_get_path(treemodel, &iter);
    gtk_tree_view_expand_row(treeview, treepath, FALSE);
    gtk_tree_path_free(treepath);

    erase_dummy_row(treemodel, &results_iter,NULL);
    if(!findCount){
	insert_dummy_row(treemodel,  &results_iter,NULL,NULL,"xfce/info",_("Nothing found"));	
    }
    

    treepath = gtk_tree_model_get_path(treemodel, &results_iter);
    gtk_tree_view_expand_row(treeview, treepath, FALSE);

    /* scroll is in callbacks.c, open_dir(), 
     * but only called if preference enabled */
    gdk_flush();
    gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.0, 0.0);
    gtk_tree_path_free(treepath);
    
    g_source_remove(stop_watch);
    {
	gchar *g=g_strdup_printf(_("%ld seconds"),(long)(time(NULL) - initial));
	print_status("xfce/info","fgr ",g,NULL);
	g_free(g);
    }
} 

static GtkTreeIter add_find_results(GtkTreeView * treeview, char *name)
{
    GtkTreeIter iter;
    tree_entry_t *en, *c_en;
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);

    get_find_root(treeview, &iter, &en);

    c_en = mk_entry(en->type);
    c_en->path = g_strdup(name);
    SET_XF_FND(c_en->type);
    if (!add_row(treemodel,&iter,NULL,&results_iter,c_en,(gchar *)abreviate(name)))
    {
	set_icon(treemodel, &results_iter);
	insert_dummy_row(treemodel, &results_iter,NULL,c_en,NULL,NULL);
	g_warning("could not add row");
    }

    return results_iter;
}

static void add_find_results_content(void)
{
    GList *tmp;
    int i;
    tree_entry_t *en;
    GtkTreeIter iter, child;
    GtkTreePath *treepath;
    GtkTreeModel *treemodel = gtk_tree_view_get_model(find_treeview);


    gtk_tree_model_get(treemodel, &results_iter, ENTRY_COLUMN, &en, -1);

    if(findCount)
    {
	find_xfdir.pathc = findCount;
	find_xfdir.gl = (dir_t *) malloc(findCount * sizeof(dir_t));
	if(!find_xfdir.gl)
	    assert_not_reached();
	for(i = 0; i < find_xfdir.pathc; i++)
	    find_xfdir.gl[i].pathv = NULL;

	tmp = find_results_list;
	for(i = 0; i < find_xfdir.pathc; i++)
	{
	    char *name;
	    if(!tmp)
		assert_not_reached();

	    if(!strchr((char *)tmp->data, '/'))
		assert_not_reached();
	    name = strrchr((char *)tmp->data, '/');
	    if(strlen(name) == 1)
		name = "/";
	    else
		name++;
	    find_xfdir.gl[i].pathv = g_strdup(name);
	    find_xfdir.gl[i].en = stat_entry((char *)tmp->data, en->type);
	    if(!find_xfdir.gl[i].en){
		/* file has dissappeared ! */  
		find_xfdir.pathc--;
		g_free(find_xfdir.gl[i].pathv);
		find_xfdir.gl[i].pathv=NULL;
	        i--;
		/*assert_not_reached();*/
	    }
	    g_free((char *)tmp->data);
	    tmp->data=NULL;
	    tmp = tmp->next;
	}
	if(find_results_list)
	    g_list_free(find_results_list);
	find_results_list = NULL;
	
	hide_stop();

    	if (find_xfdir.pathc){
	  add_contents_row(treemodel, &results_iter, &find_xfdir);
	  get_find_root(find_treeview, &iter, &en);
	  erase_dummy_row(treemodel, &iter,NULL);
	  /* collapse all */
	  if(gtk_tree_model_iter_children(treemodel, &child, &iter))
	  {
	    do
	    {
		treepath = gtk_tree_model_get_path(treemodel, &child);
		gtk_tree_view_collapse_row(find_treeview, treepath);
		gtk_tree_path_free(treepath);
	    }
	    while(gtk_tree_model_iter_next(treemodel, &child));
	  }
	}
	
	xfdirfree(&find_xfdir);
    }
}



/****************************************************************/

static void abort_because_of_limit(GtkWidget * widget)
{
    if(tree_details->tubo_object)
    {
	gchar *message;
	gchar *m=g_strdup_printf("%d", Gpid);
	print_diagnostics(NULL , strerror(ECANCELLED)," pid=", m,"\n", NULL);
	g_free(m);
	
	cancelled = TRUE;
	TuboCancel(tree_details->tubo_object, NULL);
	tree_details->stop = FALSE;
	tree_details->tubo_object = NULL;
	Gpid = 0;
	message = g_strdup_printf(_("Results limit reached: %d"), fileLimit);
	print_diagnostics("xfce/warning", message, "\n",NULL);
	g_free(message);
	hide_stop();
	D(printf("find over by abort limit\n");) 
    }
}

static gint watch_stop(gpointer data)
{

    time_t current=time(NULL);
    if (current - initial != deltatime) {
	gchar *g;
	deltatime = current - initial;
	g=g_strdup_printf(_("%ld seconds"),(long)(deltatime));
	print_status("xfce/warning","fgr ",g,NULL);
	g_free(g);
    }

    if(!tree_details->tubo_object) return FALSE;
    if(tree_details->stop)
    {
	char m[32];
	sprintf(m, "%d\n", Gpid);
	print_diagnostics(NULL, strerror(ECANCELLED), " pid=", m, "\n",NULL);
	cancelled = TRUE;
	TuboCancel(tree_details->tubo_object, NULL);
	tree_details->stop = FALSE;
	tree_details->tubo_object = NULL;
	print_status("xfce/info", strerror(ECANCELLED), NULL);
	hide_stop();
	Gpid = 0;
	return FALSE;
    }
    {
	
    }
    set_progress(-1, -1);
    return TRUE;
}

static int operate_stderr(int n, void *data)
{
    char *line;
    if(n)
	return TRUE;		/* this would mean binary data */
    line = (char *)data;
    print_diagnostics("xfce/error", line, NULL);
    return TRUE;
}

static int operate_stdout(int n, void *data)
{
    char *line;
    char *filename;

    if(n)
	return TRUE;		/* this would mean binary data */
    line = (char *)data;



    if(tree_details->stop || cancelled)
    {
	return (TRUE);
    }
#ifdef DEBUG
    printf("line=%s\n",line);
#endif
    if(line[0] == G_DIR_SEPARATOR)
    {
	if(findCount >= fileLimit){
	    abort_because_of_limit(NULL);
	}
	else
	{
	    char *path, *linecount = NULL;

	    path =line;
	    if(strstr(path, "\n")) path = strtok(path, "\n");
	    if(strstr(path, ":"))
	    {
		/* some filenames with colons exist! */
		if(!g_file_test(path, G_FILE_TEST_EXISTS))
		{
		    /* but not this one */
		    linecount = strrchr(path, ':') + 1;
		    *(strrchr(path, ':')) = 0;
		    if(strcmp(linecount, "0") == 0)
		    {
			linecount = NULL;
			return TRUE;
		    }
		}
	    }


	    findCount++;
	    find_results_list = g_list_append(find_results_list, g_strdup(path));
	    filename = strrchr(path, G_DIR_SEPARATOR) + 1;

	    print_diagnostics(NULL, path, "\n", NULL);
	}
    }
    else
    {
	print_diagnostics(NULL, line, NULL);
    }
    return TRUE;
}


static void fork_finished_function(pid_t pid)
{
    gchar *m=g_strdup_printf("%d", pid);
    print_diagnostics(NULL, strerror(ESRCH)," pid=", m,"\n", NULL);
    g_free(m);
    tree_details->tubo_object = NULL;
    hide_stop();
    cursor_reset();

    
    if(findCount)
    {
	gchar *mess;
	if (findCount >= fileLimit) {
	  gchar *message = g_strdup_printf(_("Results limit reached: %d"), fileLimit);
	  print_diagnostics("xfce/warning", message, "\n",NULL);
	  g_free(message);
	}
	mess = g_strdup_printf(_("Files found=%d"), findCount);
	print_diagnostics(NULL,  mess, "\n", NULL);
	g_free(mess);
	add_find_results_content();

    }
    else
    {
	print_diagnostics( NULL, strerror(ENOENT), "\n",NULL);
	print_status( "xfce/info", strerror(ENOENT), NULL);
    }
    D(printf("find over by fork_finished_function\n");) 
    find_over();
}

static gchar *get_combo_entry(GtkEntry * entry)
{
    int i, j;
    static gchar *s = NULL;
    if(s){
	g_free(s);
	s=NULL;
    }
    s = g_strdup((char *)gtk_entry_get_text(entry));
    while((s[0] == ' ') || (s[0] == '\t'))
	s++;
    j = strlen(s) - 1;
    for(i = j; i >= 0; i--)
    {
	if((s[i] == ' ') || (s[i] == '\t'))
	    s[i] = (char)0;
	else
	    break;
    }
    return s;
}


   
G_MODULE_EXPORT
void do_find_path(GtkWidget * widget,const char *path){
    GtkWidget *find_dialog;
    GtkCombo *combo;
    GtkCombo *combopath;
    GtkCombo *combogrep;
    gint tree_id = get_active_tree_id();
    find_treeview = tree_details->treestuff[tree_id].treeview;
     
    find_dialog = create_find_dialog();
    combo=(GtkCombo *)lookup_widget(find_dialog, "filter_combo");
    combopath=(GtkCombo *)lookup_widget(find_dialog, "path_combo");
    combogrep=(GtkCombo *)lookup_widget(find_dialog, "grep_combo");
    gtk_widget_show(find_dialog);
    gtk_window_set_transient_for(GTK_WINDOW(find_dialog), GTK_WINDOW(tree_details->window));

    g_signal_connect(G_OBJECT(lookup_widget(find_dialog, "find_button")), "clicked", G_CALLBACK(on_find_clicked), (gpointer) find_treeview);
    g_signal_connect(G_OBJECT(lookup_widget(find_dialog, "close_button")), "clicked", G_CALLBACK(on_find_close), (gpointer) find_treeview);    
    g_signal_connect(G_OBJECT(lookup_widget(find_dialog, "togglebutton2")), "toggled", G_CALLBACK(on_help_filter), NULL);
    g_signal_connect(G_OBJECT(lookup_widget(find_dialog, "togglebutton3")), "toggled", G_CALLBACK(on_help_grep), NULL);

    g_signal_connect(G_OBJECT(find_dialog), "destroy_event", G_CALLBACK(destroy), (gpointer) find_treeview);
    g_signal_connect(G_OBJECT(find_dialog), "delete_event", G_CALLBACK(destroy), (gpointer) find_treeview);

    
    if (findgrep_combo_info) findgrep_combo_info=XFC_destroy_combo(findgrep_combo_info);
    {
      gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
      gchar *f=g_build_filename(xdg_dir,FIND_GREP_DBH_FILE,NULL);

      g_free(xdg_dir);
      findgrep_combo_info = XFC_init_combo(combogrep);

      findgrep_combo_info->activate_func = on_find_clicked_wrapper;
      findgrep_combo_info->activate_user_data = (gpointer) find_treeview;
      
      XFC_read_history(findgrep_combo_info,f);
      g_free (f);    
    }
    XFC_set_blank(findgrep_combo_info);
    
    if (find_combo_info) find_combo_info=XFC_destroy_combo(find_combo_info);
    {
      gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
      gchar *f=g_build_filename(xdg_dir,FIND_DBH_FILE,NULL);

      g_free(xdg_dir);
      find_combo_info = XFC_init_combo(combo);

      find_combo_info->activate_func = on_find_clicked_wrapper;
      find_combo_info->activate_user_data = (gpointer) find_treeview;
      
      XFC_read_history(find_combo_info,f);
      g_free (f);    
    }
    XFC_set_blank(find_combo_info);
    
    if (findpath_combo_info) findpath_combo_info=XFC_destroy_combo(findpath_combo_info);
    {
      gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
      gchar *f=g_build_filename(xdg_dir,FIND_PATHS_DBH_FILE,NULL);

      g_free(xdg_dir);
      findpath_combo_info = XFC_init_combo(combopath);

      findpath_combo_info->activate_func = on_find_clicked_wrapper;
      findpath_combo_info->activate_user_data = (gpointer) find_treeview;
      
      XFC_read_history(findpath_combo_info,f);
      g_free (f);    
    }  
    if (path && g_file_test(path,G_FILE_TEST_EXISTS | G_FILE_TEST_IS_DIR)) 
	XFC_set_entry(findpath_combo_info,(char *)path);
    else XFC_set_combo (findpath_combo_info,NULL);

    if(!type_list)
    {
	int i;
	for(i = 0; ftypes[i] != NULL; i++)
	{
	    type_list = g_list_append(type_list, _(ftypes[i]));
	}

    }
    fill_string_option_menu ((GtkOptionMenu *) lookup_widget(find_dialog, "file_type_om"), type_list);
#if 0
    gtk_option_menu_set_history ((GtkOptionMenu *) lookup_widget(find_dialog, "file_type_om"), 1); /* preselects "Normal File" */
#endif
    
    gtk_label_set_text((GtkLabel *) lookup_widget(find_dialog, "filter_help"), _("Basic rules:\n" "\n" "*  Will match any character zero or more times.\n" "?  Will match any character exactly one time\n"));
    gtk_label_set_text((GtkLabel *) lookup_widget(find_dialog, "regexp_help"),
		       _("Reserved characters for extended regexp are\n"
			 ". ^ $ [ ] ? * + { } | \\ ( ) : \n"
			 "In  basic regular expressions the metacharacters\n"
			 "?, +, {, |, (, and ) lose their special meaning.\n"
			 "\n"
			 "The  period  .   matches  any  single  character.\n"
			 "The caret ^ matches at the start of line.\n"
			 "The dollar $ matches at the end of line.\n" "\n"
			 "Characters within [ ] matches any single \n"
			 "       character in the list.\n"
			 "Characters within [^ ] matches any single\n"
			 "       character *not* in the list.\n"
			 "Characters inside [ - ] matches a range of\n"
			 "       characters (ie [0-9] or [a-z]).\n" "\n"
			 "A regular expression may be followed by one\n"
			 "       of several repetition operators:\n"
			 "?      The preceding item is optional and matched\n"
			 "       at most once.\n"
			 "*      The preceding item will be matched zero\n"
			 "       or more times.\n"
			 "+      The preceding item will be matched one or\n"
			 "       more times.\n"
			 "{n}    The preceding item is matched exactly n times.\n"
			 "{n,}   The preceding item is matched n or more times.\n" "{n,m}  The preceding item is matched at least n times,\n" "       but not more than m times.\n" "\n" "To match any reserved character, precede it with \\. \n" "\n" "Two regular expressions may be joined by the logical or\n" "       operator |.\n" "Two regular expressions may be concatenated.\n" "\n" "More information is available by typing \"man grep\"\n" "       at the command prompt.\n"));

}


G_MODULE_EXPORT
void do_find(GtkWidget * widget){
    tree_entry_t *en;
    gchar *path=NULL;
    GtkTreeIter iter;
    if(tree_details->tubo_object)
    {
	print_status("xfce/error", strerror(EALREADY), NULL);
	return;
    }
    if (GTK_WIDGET_VISIBLE(tree_details->window)){
      en = get_selected_entry(&iter);
      if(en && en->path && g_file_test(en->path,G_FILE_TEST_EXISTS | G_FILE_TEST_IS_DIR)) {
	  if (g_file_test(en->path,G_FILE_TEST_IS_DIR)) path=g_strdup(en->path);
	  else path=g_path_get_dirname(en->path);
      }
    }
    /* use last path from history */
    /*if (!path) path=g_strdup(g_get_home_dir());
    if (!path) path=g_strdup("/");*/
    do_find_path(widget,(const gchar *)path);
    g_free(path);
}


/******  internal callbacks   *******/

static void destroy(GtkWidget * widget, gpointer data)
{
#ifdef DEBUG
    printf("destroy(find window)\n");
#endif
    if(tree_details->tubo_object)
	TuboCancel(tree_details->tubo_object, NULL);
    tree_details->tubo_object = NULL;
    /* kill(Gpid, SIGHUP) is done inside TuboCancel. 
     * What could happen if Gpid is
     * a valid pid when instruction given, but before signal reaches
     * Gpid, the one from TuboCancel kills Gpid first?
     * Where does that signal end up? */
    /* dirty instruction: if(Gpid) kill(Gpid, SIGHUP);*/
    Gpid = 0;
    cursor_reset();
    gtk_widget_destroy(widget);
}


static void on_find_clicked(GtkButton * button, gpointer data)
{
    short int i, j;
    char *s;
    char *argument[MAX_ARGS];
    GtkWidget *w = (GtkWidget *) button;
    char *find_filter = NULL, *filter = NULL, *path = NULL, *token = NULL;
    gint tree_id = get_active_tree_id();
    GtkTreeView *treeview = tree_details->treestuff[tree_id].treeview;


    if (!GTK_WIDGET_VISIBLE(tree_details->window)){
	gtk_widget_show(tree_details->window);
    }
    gtk_widget_grab_focus((GtkWidget *) treeview);

    if(!set_load_wait())
    {
	assert_not_reached();
    }
    /* get options */

    cancelled = FALSE;

    if(tree_details->tubo_object)
	TuboCancel(tree_details->tubo_object, NULL);
    tree_details->tubo_object = NULL;
    /* again as above, 
     * dirty instruction:  if(Gpid) kill(Gpid, SIGTERM);
     * */
    Gpid = 0;

    findCount = 0;
/* get the parameters set by the user... *****/

/* limit */
    fileLimit = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(lookup_widget(w, "spinbutton1")));

/* the rest */

    s = get_combo_entry((GtkEntry *) (lookup_widget(w, "path_entry")));
    if(!s || strlen(s) == 0)
	s = "/";
    /* tilde expansion */
    else if(s[strlen(s) - 1] == '~')
	s = "~/";
    /* environment variables */
    else if(s[0] == '$')
    {
	char *p = getenv(s + 1);
	if(p)
	    s = p;
	else
	    s = "/";
    }
    path = g_strdup(s);	
    if (g_file_test(s,G_FILE_TEST_EXISTS | G_FILE_TEST_IS_DIR)){
        save_fpath_text(path);
    }


    s = get_combo_entry((GtkEntry *) lookup_widget(w, "filter_entry"));
    if(s)
    {
	filter=g_strdup(s);
        save_ff_text(filter);
    }
    

    s = get_combo_entry((GtkEntry *) lookup_widget(w, "grep_entry"));
    if(s && strlen(s))
    {
	token = g_strdup(s);
	save_fgrep_text(token);
    }
    /* select list */
    /*s = get_combo_entry((GtkEntry *) lookup_widget(w, "file_type_entry"));*/
    s = string_option_menu_get_selected((GtkOptionMenu *)lookup_widget(w, "file_type_om"));
    /* s: this could probably probably very low probably be null, that is when the option menu itself is empty. */

    i = 0;
/*#define USE_FIND   */
#ifdef USE_FIND
    /* special case find: if no grep token needed, best to use FIND */
    if((!token || !strlen(token)) && filter && strlen(filter))
    /*if((!token || !strlen(token)) && (filter && strlen(filter) && !strchr(filter, '?') && !strchr(filter, '*')))*/
    {
 char *find_ft[] = {
    "",
    "f",
    "d",
    "l",
    "s",
    "b",
    "c",
    "p",
    NULL
};
	argument[i++] = FIND;
	/* first argument to FIND must be the path */
	/* but, oho, since symlinks are not followed because of
	 * infinite loop posibilities, if the path is a symlink,
	 * expand to where it points. */
	{
	    struct stat st;
	    lstat(path,&st);
	    if (S_ISLNK(st.st_mode)){
		 static char lpath[_POSIX_PATH_MAX+1];
		 memset(lpath,0,_POSIX_PATH_MAX+1);
		 if (readlink(path, lpath, _POSIX_PATH_MAX)<0) {
		     argument[i++] = path;
		 } else {
		     argument[i++] = lpath;
		     chdir(path); /* the expanded link may be relative... */
		 }
	    } else {
		argument[i++] = path;
	    }
	}
	if(!gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "recursive")))
	    argument[i++] = "-prune";
	/* no filter if none specified  */
	if(filter && strlen(filter))
	{
	    if(find_filter){
		g_free(find_filter);
		find_filter=NULL;
	    }
	    if(!gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "case_sensitive"))){
		/*argument[i++] = "-ipath";*/
		argument[i++] = "-iname";
	    } else {
		/*argument[i++] = "-path";*/
		argument[i++] = "-name";
	    }
#if 0
	    find_filter = g_strconcat("*/",filter,NULL);
#endif
	    find_filter = g_strdup(filter);
	    if(!find_filter) assert_not_reached();

	    argument[i++] = find_filter;
	}
	if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "xdev")))
	    argument[i++] = "-xdev";

	/* OjO: "any" type is equal to no type specified in FIND 
	 *      so begin with j=1 */
	for(j = 1; ftypes[j] != NULL; j++)
	{
	    if(s && strcmp(s, _(ftypes[j])) == 0 && strlen(find_ft[j]))
	    {
		argument[i++] = "-type";
		argument[i++] = find_ft[j];
		break;
	    }
	}
	{
	  gchar *m=g_strdup_printf(_("Searching for %s at %s"),filter, path);
	  print_diagnostics(NULL, m, "\n", NULL);
	  g_free(m);
	}
    }
    else
#endif
    {
	argument[i++] = GLOB;

	/*argument[i++] = "-v"; (verbose output from glob for debugging) */
	/*argument[i++] = "-P"; */

	if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "recursive")))
	    argument[i++] = "-r";
	if(!gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "case_sensitive")))
	    argument[i++] = "-i";
	if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "xdev")))
	    argument[i++] = "-a";
	if(token)
	{
	    if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "ext_regexp")))
		argument[i++] = "-E";
	    else
		argument[i++] = "-e";
	    argument[i++] = token;


	    /* options for grep: ***** */
	    if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "no_look_in_binaries")))
		argument[i++] = "-I";
	    /*
	       if (gtk_toggle_button_get_active(
	       (GtkToggleButton *)lookup_widget(w,"output_count")))
	       argument[i++] = "-c"; */
	    /* radio buttons */
	    if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "anywhere")))
		;
	    else if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "match_words")))
		argument[i++] = "-w";
	    else if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "match_lines")))
		argument[i++] = "-x";
	    else if(gtk_toggle_button_get_active((GtkToggleButton *) lookup_widget(w, "match_no_match")))
		argument[i++] = "-L";
	}

	for(j = 0; ftypes[j] != NULL; j++)
	{
	    if(s && strcmp(s, _(ftypes[j])) == 0)
	    {
		argument[i++] = "-t";
		argument[i++] = ft[j];
		break;
	    }
	}


	/* apply wildcard filter if not specified  */
	argument[i++] = "-f";
	if(filter && strlen(filter))
	{
	    argument[i++] = filter;
	}
	else
	{
	    argument[i++] = "*";
	}

	/* last argument to GLOB must be the path */
	argument[i++] = path;
    }
    print_status("xfce/warning",strerror(EBUSY), NULL);
    {
	GtkTreeIter iter;
	tree_entry_t *en;
	GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
	if (get_find_root(treeview, &iter, &en)) 
	    reset_dummy_row(treemodel, &iter,NULL,en,"xfce/info",_("Search begun"));
    }
    if (s) { /* type, I presume */
    	g_free (s);
    	s = NULL;
    }

    argument[i] = (char *)0;

    /*for (j=0;j<i;j++) printf ("%s ",argument[j]);printf ("\n"); */


    cursor_wait();
    /* pango is bugged. Do not print icons if pipe output is to be written
     * to the diagnostics window... */
    print_diagnostics(NULL, "$", NULL);
    for(j = 0; j < i; j++)
    {
	print_diagnostics(NULL, argument[j], " ", NULL);
    }
    print_diagnostics(NULL, "\n", NULL);

    tree_details->stop = FALSE;

    /*printf("DBG: show stop at find.c\n"); */

    show_stop();

    { /* map the find branch if its not shown */
      GtkTreeIter iter;
      tree_entry_t *en;
      get_find_root(treeview, &iter, &en);
    }
    
    initial=time(NULL);

    tree_details->tubo_object = Tubo(
		    fork_function, 
		    (void *)argument, 
		    fork_finished_function, 
		    NULL, 
		    operate_stdout, operate_stderr,0,
		    TRUE);
    if (!tree_details->tubo_object){
            gtk_widget_destroy(lookup_widget(w, "find_dialog"));
	    print_diagnostics("xfce/error",strerror(ECHILD),"\n", NULL);
	    return;
    }
    stop_watch = g_timeout_add_full(0,260, (GtkFunction) watch_stop, (gpointer) treeview,NULL);
    Gpid = TuboPID(tree_details->tubo_object);

    gtk_widget_destroy(lookup_widget(w, "find_dialog"));
    chdir(GETWD); 
    {
	int size = 5;
	char *name;

	size += ((path) ? strlen(path) : 0);
	size += ((filter) ? strlen(filter) : 0);
	size += ((token) ? strlen(token) : 0);

	name = (char *)malloc(size);
	sprintf(name, "%s", path);
	if(strcmp(path, "/") != 0)
	    strcat(name, "/");
	if(filter && strlen(filter))
	    strcat(name, filter);
	if(token && strlen(token))
	{
	    strcat(name, "(");
	    strcat(name, token);
	    strcat(name, ")");
	}
#ifdef DEBUG
	printf("DBG:size=%d, name=%s\n",size,name); 
#endif

	add_find_results(treeview, name);

	g_free(name);
	name=NULL;
    }
    if (token) g_free(token);
    if (filter) g_free(filter);
    if (find_filter) g_free(find_filter);
    if (path) g_free(path);
}

static void on_find_clicked_wrapper(GtkEntry *entry,gpointer data){
      on_find_clicked((GtkButton *)entry, data);
}

static void on_find_close(GtkButton * button, gpointer data)
{
    destroy(lookup_widget((GtkWidget *) button, "find_dialog"), NULL);
    if (!GTK_WIDGET_VISIBLE(tree_details->window)) exit(1);
}

static void on_help_filter(GtkToggleButton * button, gpointer user_data)
{
    if(gtk_toggle_button_get_active(button))
	showit((GtkWidget *) button, "scrolledwindow8");
    else
	hideit((GtkWidget *) button, "scrolledwindow8");

}

static void on_help_grep(GtkToggleButton * button, gpointer user_data)
{
    if(gtk_toggle_button_get_active(button))
	showit((GtkWidget *) button, "scrolledwindow9");
    else
	hideit((GtkWidget *) button, "scrolledwindow9");

}



G_MODULE_EXPORT
xffind_functions *module_init(void){
#ifdef ENABLE_NLS
    /* This is required for UTF-8 at least - Please don't remove it */
    bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
    bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
    textdomain (GETTEXT_PACKAGE);
#endif
    xffind_fun = g_new0 (xffind_functions,1);
    if (!xffind_fun) g_assert_not_reached();
    xffind_fun->do_find=do_find;
    xffind_fun->do_find_path=do_find_path; 
    return xffind_fun;
}



