/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  magic.c

  (c) Juan D. Martin 1995
  (c) CNM-US 1995
                                                                           
  modified 2004,2005,2006 by Bernhard R. Link (see Changelog)

  magic headers related code.
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <xfmconfig.h>

#include <assert.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "magic.h"
#include "suffix.h"

#ifdef HAVE_ZLIB
#include "zlib.h"
#endif
#ifdef HAVE_LIBBZ2
#include "bzlib.h"
#endif

#include <regex.h>

#define M_LONG		0x0
#define M_SHORT		0x1
#define M_BYTE		0x2
#define M_STRING	0x3
#define M_MODE		0x4
#define M_LMODE		0x5
#define M_BUILTIN	0x6		/* no longer supported */
#define M_REGEXP	0x7
#define M_LELONG	0x8
#define M_LESHORT	0x9
#define M_TYPE		0xF		/* Mask for all types. */
#define M_MASKED	0x10		/* Value is masked. */
#define M_EQ		(0x0<<5)
#define M_LT		(0x1<<5)
#define M_GT		(0x2<<5)
#define M_SET		(0x3<<5)
#define M_CLEAR		(0x4<<5)
#define M_ANY		(0x5<<5)
#define M_OP		(0x7<<5)	/* Mask for operations. */

typedef union
{
    long number;
    char *string;
    regex_t *expr;
} m_un;

typedef struct
{
    off_t offset;
    m_un value;
    long mask;
    int flags;
    char *message;
    struct mime_filetype *mime;
    int subtypes;
} hmagic;


/* ULTRIX apparently doesn't define these */
#ifdef ultrix
#define S_ISLNK(mode) (mode & S_IFMT) == S_IFLNK
#define S_ISSOCK(mode) (mode & S_IFMT) == S_IFSOCK
#endif

#define MAXLIN 1024
#define ASCLEN 512
#define REGLEN 256

static char    linebuf[MAXLIN];
static hmagic *mtypes = 0;	/* Array of file-types. */
static int     count = 0;	/* Number of types registered. */
static int     allocated = 0;	/* Room in the array. */
static int     top = 0;		/* Current top for sub-types. */
static size_t  maxhdr = ASCLEN;	/* Maximum header size required. */
#ifdef HAVE_ZLIB
#if ASCLEN < 21
#error "ASCLEN too short for .gz header"
#endif
#endif
static char   *hdrbuf = 0;	/* Header buffer allocated. */
static size_t  hdrbufsiz = 0;	/* Size of header buffer. */
static struct stat stbuf;	/* Stat buffer. */
static struct stat lstbuf;	/* Lstat buffer. */
static int bytes;		/* Bytes read from the file. */
static const char   *namep;           /* pointer to filename */
static bool mmatch(int,const char **,struct mime_filetype **);
static const char *builtin_test(struct mime_filetype **mimep);
static char *parse_string(char **s, long *size);

void magic_parse_singlefile(char *name, typeinitfunc *initfunc, size_t maxlen)
{
    FILE *fh;
    size_t hsiz;

    if(!hdrbuf)
    {
	hdrbuf = (char *) malloc(maxhdr + 1);
	hdrbufsiz = maxhdr;
    }

    if(!name || !(fh = fopen(name, "r")))
	return;

    while(fgets(linebuf, MAXLIN, fh))
    {
	char *cptr, *sptr;
	int cnt = 0;
	int l;

	l = strlen(linebuf);
	while(l < MAXLIN && linebuf[--l] == '\n' && linebuf[--l] == '\\')
	{
	    if(!fgets(linebuf + l, MAXLIN - l, fh))
		break;
	    l = strlen(linebuf);
	}

	if(count >= allocated)
	{
	    if(!mtypes)
	    {
		allocated = 25;
		mtypes = (hmagic *) malloc(allocated * sizeof(hmagic));
	    }
	    else
	    {
		allocated += 25;
		mtypes = (hmagic *) realloc((char *) mtypes,
					    allocated * sizeof(hmagic));
	    }
	}

	cptr = linebuf;
	if(cptr[0] == '>')	/* Sub-type */
	{
	    while( cptr[0] == '>' )
		    cptr++;
	    mtypes[top].subtypes++;
	}
	else if(cptr[0] == '#' || cptr[0] == '\n' || cptr[0] == '\r')
	{
	    continue;
	}
	else if(strncmp("include:",cptr,8) == 0)
	{
		size_t cl;

		cptr += 8;
		while(isspace(*cptr))
			cptr++;
		cl = strlen(cptr);
		while( cl>0 && isspace(cptr[cl-1]))
		{
			cptr[--cl]=(char)0;
		}
		magic_parse_singlefile(cptr, initfunc, maxlen);
		continue;
	}
	else
	{
	    top = count;
	    mtypes[top].subtypes = 0;
	}

	mtypes[count].offset = strtol(cptr, &cptr, 0);
	while(isspace(*cptr))
	    cptr++;

	while(islower(cptr[cnt]))
	    cnt++;

	if(!strncmp("string", cptr, cnt))
	{
	    char *s;
	    mtypes[count].flags = M_STRING;
	    cptr += cnt;

	    /* allow string/{bcB} now, implement that later */
	    if( cptr[0] == '/' ) {
		    cptr++;
		    while(cptr[0] != '\0' && !isspace(cptr[0]))
			    cptr++;
	    }

	    sptr = parse_string(&cptr,&mtypes[count].mask);

	    hsiz = mtypes[count].offset + mtypes[count].mask;
	    /* ignore if too long */
	    if(hsiz >= maxlen)
		// TODO: print warning?
                continue;
	    if(hsiz > maxhdr)
		maxhdr = hsiz;

	    /* here it gets a bit ugly, as there can be \000 in
	     * those strings, that are already unescaped into '\0',
	     * so strcpy is not the way to go. - brl */

	    s = malloc(mtypes[count].mask + 1);
	    if( s == NULL )
		    abort();
	    memcpy(s,sptr,mtypes[count].mask);
	    s[mtypes[count].mask] = '\0';
	    mtypes[count].value.string = s;

	}
	else if(!strncmp("builtin", cptr, cnt))
	{
		// TODO: report errors
		continue;
	}
	else if(!strncmp("regexp", cptr, cnt))
	{
	    mtypes[count].flags = M_REGEXP;
	    cptr += cnt;


	    if(*cptr == '&')
	    {
		cptr++;
		mtypes[count].mask = strtol(cptr, &cptr, 0);
	    }
	    else
		mtypes[count].mask = REGLEN;
	    hsiz = mtypes[count].offset + mtypes[count].mask;
	    if(hsiz > maxhdr)
		maxhdr = hsiz;

	    sptr = parse_string(&cptr,NULL);
	    mtypes[count].value.expr = calloc(1,sizeof(regex_t));
	    if( mtypes[count].value.expr == NULL )
		    // TODO: report errors
		    continue;
	    // TODO: decice what other flags to give:
	    if( regcomp(mtypes[count].value.expr,sptr,REG_NOSUB|REG_EXTENDED) != 0 ) {
		    free(mtypes[count].value.expr);
		    // TODO: report errors
		    continue;
	    }
	}
	else
	{
	    long vmask;

	    if(!strncmp("byte", cptr, cnt))
	    {
		mtypes[count].flags = M_BYTE;
		hsiz = mtypes[count].offset + 1;
		if(hsiz > maxhdr)
		    maxhdr = hsiz;
		vmask = 0xFF;
	    }
	    else if(!strncmp("short", cptr, cnt) || !strncmp("beshort", cptr, cnt))
	    {
		mtypes[count].flags = M_SHORT;
		hsiz = mtypes[count].offset + 2;
		if(hsiz > maxhdr)
		    maxhdr = hsiz;
		vmask = 0xFFFF;
	    }
	    else if(!strncmp("long", cptr, cnt) || !strncmp("belong", cptr, cnt))
	    {
		mtypes[count].flags = M_LONG;
		hsiz = mtypes[count].offset + 4;
		if(hsiz > maxhdr)
		    maxhdr = hsiz;
		vmask = 0xFFFFFFFF;
	    }
	    else if( strncmp("leshort", cptr, cnt) == 0 )
	    {
		mtypes[count].flags = M_LESHORT;
		hsiz = mtypes[count].offset + 2;
		if(hsiz > maxhdr)
		    maxhdr = hsiz;
		vmask = 0xFFFF;
	    }
	    else if( strncmp("lelong", cptr, cnt) == 0 )
	    {
		mtypes[count].flags = M_LELONG;
		hsiz = mtypes[count].offset + 4;
		if(hsiz > maxhdr)
		    maxhdr = hsiz;
		vmask = 0xFFFFFFFF;
	    }
	    else if(!strncmp("mode", cptr, cnt))
	    {
		mtypes[count].flags = M_MODE;
		vmask = 0xFFFFFFFF;
	    }
	    else if(!strncmp("lmode", cptr, cnt))
	    {
		mtypes[count].flags = M_LMODE;
		vmask = 0xFFFFFFFF;
	    }
	    else
		continue;		/* Error. Skip line. */

	    cptr += cnt;

	    if(*cptr == '&')
	    {
		mtypes[count].flags |= M_MASKED;
		cptr++;
		mtypes[count].mask = strtol(cptr, &cptr, 0);
	    }

	    while(isspace(*cptr))
		cptr++;
	    switch(*cptr)
	    {
	    case '=':
		mtypes[count].flags |= M_EQ;
		cptr++;
		break;
	    case '<':
		mtypes[count].flags |= M_LT;
		cptr++;
		break;
	    case '>':
		mtypes[count].flags |= M_GT;
		cptr++;
		break;
	    case '&':
		mtypes[count].flags |= M_SET;
		cptr++;
		break;
	    case '^':
		mtypes[count].flags |= M_CLEAR;
		cptr++;
		break;
	    case 'x':
		mtypes[count].flags |= M_ANY;
		cptr++;
		break;
	    default:
		mtypes[count].flags |= M_EQ;
	    }
	    mtypes[count].value.number = strtol(cptr, &cptr, 0) & vmask;
	}
	while(isspace(*cptr))
	    cptr++;
	sptr = cptr;
	while(*cptr != '\n' && *cptr != '\r' && *cptr != ' ' && *cptr != '\t' && *cptr != '\0')
	    cptr++;
	*cptr = '\0';
	mtypes[count].message = strdup(sptr);
	if( initfunc == NULL )
		mtypes[count].mime = NULL;
	else
		mtypes[count].mime = initfunc(mtypes[count].message);
	count++;
    }
    fclose(fh);

    if(maxhdr > hdrbufsiz)
    {
	free(hdrbuf);
	hdrbuf = (char *) malloc(maxhdr + 1);
	hdrbufsiz = maxhdr;
    }
}

#ifdef HAVE_ZLIB
// TODO: add resource to not look into those 
// (for too slow computers)
static char *ungzip(int fd, unsigned char *inbuf, size_t maxhdr, size_t *bytes) {
	z_stream z;
	unsigned char *outbuf;
	unsigned char flags = hdrbuf[3];
	size_t len;
	int err;

	if( *bytes <= 12 )
		return NULL;
	if( inbuf[0] != 0x1f || inbuf[1] != 0x8b
		|| inbuf[2] != Z_DEFLATED || (inbuf[3] & 0xE0) != 0)
		return NULL;

	outbuf = malloc(maxhdr+1);
	if( outbuf == NULL )
		return NULL;
	z.zalloc = NULL;
	z.zfree = NULL;
	z.opaque = NULL;
	z.next_out = outbuf;
	z.avail_out = maxhdr;
	z.next_in = inbuf + 10;
	z.avail_in = *bytes - 10;
	/* skip optional fields */
	if( (flags & 0x04 )!= 0 ){
		z.avail_in -= 2;
		len = *(z.next_in++);
		len += ((size_t)(*(z.next_in++)))<<8;
		/* TODO: reload instead */
		if( z.avail_in-2 < len ) {
			free(outbuf);
			return NULL;
		}
		z.avail_in -= len;
		z.next_in += len;
	}
	if( (flags & 0x08 )!= 0 ){
		/* perhaps use that as filename instead for
		 * suffix checking? */
		while( z.avail_in > 2 && *z.next_in != '\0' ) {
			z.avail_in--;
			z.next_in++;
		}
		if( *z.next_in != '\0' ) {
			free(outbuf);
			return NULL;
		}
		z.avail_in--;
		z.next_in++;
	}
	if( (flags & 0x10 )!= 0 ){
		while( z.avail_in > 2 && *z.next_in != '\0' ) {
			z.avail_in--;
			z.next_in++;
		}
		if( *z.next_in != '\0' ) {
			free(outbuf);
			return NULL;
		}
		z.avail_in--;
		z.next_in++;
	}
	if( (flags & 0x02 )!= 0 ){
		if( z.avail_in <= 2 ) {
			free(outbuf);
			return NULL;
		}
		z.avail_in -= 2;
		z.next_in += 2;
	}
	err = inflateInit2(&z,-15);
	if( err != Z_OK ) {
		free(outbuf);
		return NULL;
	}
	while( err == Z_OK ) {
		err = inflate(&z,Z_SYNC_FLUSH);
		if( err == Z_OK ) {
			if( z.avail_out == 0 ) {
				/* all produced */
				break;
			} else {
				/* needs more data */
				// TODO: read from file instead
				fprintf(stderr,"Warning: not looking into .gz file as too bad compressed!\n");
				err = Z_STREAM_ERROR;
				break;
			}
		}
	}
	if( err == Z_OK || err == Z_STREAM_END) {
		*bytes = z.next_out - outbuf;
	} else {
		free(outbuf);
		outbuf = NULL;
	}
	inflateEnd(&z);
	return outbuf;
}
#endif
#ifdef HAVE_LIBBZ2
static char *unbzip2(int fd, unsigned char *inbuf, size_t maxhdr, size_t *bytes) {
	bz_stream bz;
	char *outbuf;
	char *extrabuf = NULL;
	int err;

	if( *bytes < 10 )
		return NULL;
	if( inbuf[0] != 'B' || inbuf[1] != 'Z' || inbuf[2] != 'h' )
		return NULL;

	bz.bzalloc = NULL;
	bz.bzfree = NULL;
	bz.opaque = NULL;
	err = BZ2_bzDecompressInit(&bz, 0, 0);
	if( err != BZ_OK )
		return NULL;

	outbuf = malloc(maxhdr+1);
	if( outbuf == NULL ) {
		BZ2_bzDecompressEnd(&bz);
		return NULL;
	}
	bz.next_out = outbuf;
	bz.avail_out = maxhdr;
	bz.next_in = inbuf;
	bz.avail_in = *bytes;
	while( err == BZ_OK ) {
		err = BZ2_bzDecompress(&bz);
		if( err == BZ_OK ) {
			if( bz.avail_out == 0 ) {
				/* we got all we need */
				err = BZ_STREAM_END;
				break;
			} else if( bz.avail_in == 0) {
				if( extrabuf == NULL)
					extrabuf = malloc(4096);
				if( extrabuf == NULL )
					break;
				bz.avail_in = read(fd, extrabuf, 4096);
				bz.next_in = extrabuf;
				fprintf(stderr,"reading extradata for .bz2 (%u was not enough)!\n",bz.total_in_lo32);
			} else {
				assert(42==7*8);
			}
		}
	}

	free(extrabuf);
	if( err == BZ_STREAM_END ) {
		*bytes = bz.next_out - outbuf;
	} else {
		free(outbuf);
		outbuf = NULL;
	}

	BZ2_bzDecompressEnd(&bz);
	return outbuf;
}
#endif


void magic_get_type(const char *name, const char **messagep, struct mime_filetype **mimep,struct magic_modifiers *modifiers,struct stat *statp)
{
    int i;
    int fd;

    modifiers->symlink = false;
    modifiers->gzipped = false;
    modifiers->bzipped = false;

    namep = name;

    /* added hdrbuf initialization in case magic_get_type() is invoked
       before magic_parse_file() - AG */
    if(!hdrbuf)
    {
       hdrbuf = (char *) malloc(maxhdr + 1);
       hdrbufsiz = maxhdr;
    }

    if(lstat(name, &lstbuf) < 0)
    {
	stbuf.st_mode = 0;
	lstbuf.st_mode = 0;
	statp->st_mode = 0;
	bytes = -1;
    }
    else
    {
	if(S_ISLNK(lstbuf.st_mode))
	{
	    modifiers->symlink = true;
	    if(stat(name, &stbuf) < 0)
	    {
		stbuf.st_mode = 0;
		*statp = lstbuf;
		bytes = -1;
	    } else
		*statp = stbuf;
	}
	else
	{
	    stbuf = lstbuf;
	    *statp = lstbuf;
	}
	if(S_ISREG(stbuf.st_mode))
	{
	    if(stbuf.st_size == 0)
		bytes = 0;
            /* don't open zero-size files (causes hang on /proc) -- AG */
            else if((fd = open(name, O_RDONLY, 0)) < 0)
		bytes = -1;
	    else
	    {
		char *newbuffer;

		bytes = read(fd, hdrbuf, maxhdr);
#ifdef HAVE_ZLIB
    		/* special treatment to look inside .gz files */
		newbuffer = ungzip(fd,hdrbuf,maxhdr,&bytes);
		if( newbuffer != NULL ) {
			size_t l;

			free(hdrbuf);
			hdrbuf = newbuffer;
    			modifiers->gzipped = true;

			l = strlen(namep);
			if( l > 3 && strcmp(namep+l-3,".gz") == 0 ) {
				char *n = alloca(l-2);
				memcpy(n, namep, l-3);
				n[l-3] = '\0';
				namep = n;
			}

		} else {
#endif
#ifdef HAVE_LIBBZ2
		newbuffer = unbzip2(fd,hdrbuf,maxhdr,&bytes);
		if( newbuffer != NULL ) {
			size_t l;

			free(hdrbuf);
			hdrbuf = newbuffer;
    			modifiers->bzipped = true;

			l = strlen(namep);
			if( l > 4 && strcmp(namep+l-4,".bz2") == 0 ) {
				char *n = alloca(l-3);
				memcpy(n, namep, l-4);
				n[l-4] = '\0';
				namep = n;
			}
		}
#endif
#ifdef HAVE_ZLIB
		}
#endif
		close(fd);
		/* Make sure it is nul-terminated. */
		if(bytes >= 0)
		    hdrbuf[bytes] = '\0'; 
	    }
	}
	else
	    bytes = 0;
    }

    /* TODO: check for gzip, bzip here and give them
     * special treatment to look inside */

    for(i = 0; i < count; i++)
    {
	const char *message = NULL;
	struct mime_filetype *mime = NULL;

	if(mmatch(i, &message, &mime))
	{
	    if(mtypes[i].subtypes != 0)
	    {
		int n;

		n = i + mtypes[i].subtypes + 1;
		for(++i; i < n; i++)
		{
		    const char *nmessage = NULL;
		    struct mime_filetype *nmime = NULL;

		    if(mmatch(i, &nmessage, &nmime) )
		    {
		        message = nmessage;
			mime = nmime;
		    }
		}
	    }
	    if( message != NULL ) 
	    {
		    if( messagep != NULL)
			    *messagep = message;
		    if( mimep != NULL )
		    	 *mimep = mime;
		    return;
	    }
	}
	else
	    i += mtypes[i].subtypes;
    }
    *messagep = builtin_test(mimep);
}

static bool mmatch(int i,const char **message,struct mime_filetype **mime)
{
    int t;
    int o;
    long v;
    unsigned char *h;

    h = (unsigned char *) (hdrbuf + mtypes[i].offset);
    t = mtypes[i].flags & M_TYPE;

    switch(t)
    {
    case M_STRING:
	if(bytes >= mtypes[i].offset + mtypes[i].mask &&
	   memcmp(hdrbuf + mtypes[i].offset, 
		    mtypes[i].value.string,
		    mtypes[i].mask) == 0 )
	{
	    *message = mtypes[i].message;
	    *mime = mtypes[i].mime;
	    return 1;
	}
	return 0;

    case M_BUILTIN:
	return 0;

    case M_REGEXP:
    {
#ifdef STUPIDCC
	char sav='\0';
#else
	char sav;
#endif
	int len;
	regex_t *prog = mtypes[i].value.expr;

	if(bytes <= mtypes[i].offset || !prog)
	    return 0;

	if(bytes > mtypes[i].offset + mtypes[i].mask)
	{
	    len = mtypes[i].mask;
	    sav = h[len];
	    h[len] = '\0';
	}
	else
	    len = -1;
	
	if(regexec(prog , (char*)h, 0, NULL, REG_NOTEOL) == 0)
	{
	    *message = mtypes[i].message;
	    *mime = mtypes[i].mime;
	    if(len >= 0)
		h[len] = sav;
	    return 1;
	}
	if(len >= 0)
	    h[len] = sav;
    }
    return 0;

    case M_BYTE:
	if(bytes < mtypes[i].offset + 1)
	    return 0;
	else
	    v = h[0];
	break;

    case M_LESHORT:
	if(bytes < mtypes[i].offset + 2)
	    return 0;
	else
	    v = ((long) h[1] << 8) | h[0];
	break;

    case M_SHORT:
	if(bytes < mtypes[i].offset + 2)
	    return 0;
	else
	    v = ((long) h[0] << 8) | h[1];
	break;

    case M_LONG:
	if(bytes < mtypes[i].offset + 4)
	    return 0;
	else
	    v = ((long) h[0] << 24) | ((long) h[1] << 16) |
	        ((long) h[2] << 8) | h[3];
	break;

    case M_LELONG:
	if(bytes < mtypes[i].offset + 4)
	    return 0;
	else
	    v = ((long) h[3] << 24) | ((long) h[2] << 16) |
	        ((long) h[1] << 8) | h[0];
	break;

    case M_MODE:
	v = stbuf.st_mode;
	break;

    case M_LMODE:
	v = lstbuf.st_mode;
	break;

    default:
	return 0;		/* Should never happen. */
    }

    if (mtypes[i].flags & M_MASKED)
	v &= mtypes[i].mask;

    o = mtypes[i].flags & M_OP;
    switch(o)
    {
    case M_EQ:
	if(v == mtypes[i].value.number)
	{
	    *message = mtypes[i].message;
	    *mime = mtypes[i].mime;
	    return 1;
	}
	else
	    return 0;
    case M_LT:
	if(v < mtypes[i].value.number)
	{
	    *message = mtypes[i].message;
	    *mime = mtypes[i].mime;
	    return 1;
	}
	else
	    return 0;
    case M_GT:
	if(v > mtypes[i].value.number)
	{
	    *message = mtypes[i].message;
	    *mime = mtypes[i].mime;
	    return 1;
	}
	else
	    return 0;
    case M_SET:
	if((v & mtypes[i].value.number) == mtypes[i].value.number)
	{
	    *message = mtypes[i].message;
	    *mime = mtypes[i].mime;
	    return 1;
	}
	else
	    return 0;
    case M_CLEAR:
        /* AG fix: this must be "any clear" instead of "or" */
        if((~v & mtypes[i].value.number) != 0)
	{
	    *message = mtypes[i].message;
	    *mime = mtypes[i].mime;
	    return 1;
	}
	else
	    return 0;
    case M_ANY:
	*message = mtypes[i].message;
	*mime = mtypes[i].mime;
	return 1;
    }
    return 0;
}

static struct { const char *message; struct mime_filetype *mime; } 
builtins[] = {
	{ "application/octet-stream", NULL },
	{ "text/plain", NULL },
	{ "inode/x-empty", NULL },
	{ "inode/directory", NULL },
	{ "inode/chardevice", NULL },
	{ "inode/blockdevice", NULL },
	{ "inode/pipe", NULL },
	{ "inode/socket", NULL },
	{ "inode/default", NULL },
	{ "inode/x-unreadable", NULL },
	{ "inode/x-parent-directory", NULL },
};

static const char *builtin_test(struct mime_filetype **mimep)
{
#define RETURN(j) {\
		if(mimep!=NULL)\
			*mimep=builtins[j].mime;\
		return builtins[j].message;\
	}
    int i;

    if(bytes > 0)
    {
	const char *message = suffix_search(namep,mimep);
	if(message != NULL)
		return message;
	if(bytes > ASCLEN)
	    bytes = ASCLEN;
	for(i = 0; i < bytes; i++)
	    if( hdrbuf[i] == '\0' || !isascii(hdrbuf[i]))
		RETURN(0);
	RETURN(1);
    }
    if (bytes == 0)
    {
	if(S_ISREG(stbuf.st_mode))
		RETURN(2)
	else if(S_ISDIR(stbuf.st_mode)) {
		if( strcmp(namep,"..") == 0 ) 
			RETURN(10)
		else

			RETURN(3)
	} else if(S_ISCHR(stbuf.st_mode))
		RETURN(4)
	else if(S_ISBLK(stbuf.st_mode))
		RETURN(5)
	else if(S_ISFIFO(stbuf.st_mode))
		RETURN(6)
#ifdef S_ISSOCK
	else if(S_ISSOCK(stbuf.st_mode))
		RETURN(7)
#endif /* S_ISSOCK */
	else
		RETURN(8)
    }
    RETURN(9);
}

static char *parse_string(char **s, long *size)
{
    char cbuf[4];
    int i;
    char *vptr, *sptr, *cptr;

    cptr = *s;
    while(isspace(*cptr))
	    cptr++;
    vptr = sptr = cptr;

    while(!isspace(*cptr))
    {
	if(*cptr != '\\')
	{
	    *vptr++ = *cptr++;
	    continue;
	}
	cptr++;
	switch(*cptr)
	{
	case '\\':
	    *vptr++ = '\\';
	    cptr++;
	    break;
	case 'n':
	    *vptr++ = '\n';
	    cptr++;
	    break;
	case 'r':
	    *vptr++ = '\r';
	    cptr++;
	    break;
	case 't':
	    *vptr++ = '\t';
	    cptr++;
	    break;
	case 'v':
	    *vptr++ = '\v';
	    cptr++;
	    break;
	case 'b':
	    *vptr++ = '\b';
	    cptr++;
	    break;
	case 'f':
	    *vptr++ = '\f';
	    cptr++;
	    break;
	case 'x':
	    cptr++;
	    for(i = 0; i < 3 && isxdigit(*cptr); i++, cptr++)
		cbuf[i] = *cptr;
	    cbuf[i] = '\0';
	    *vptr++ = (char) strtol(cbuf, 0, 16);
	    break;
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	    for(i = 0; i < 3 && *cptr >= '0' && *cptr <= '7';
		i++, cptr++)
		cbuf[i] = *cptr;
	    cbuf[i] = '\0';
	    *vptr++ = (char) strtol(cbuf, 0, 8);
	    break;
	default:
	    *vptr++ = *cptr++;
	}
    }
    if( size != NULL )
    	*size = vptr-sptr;
    *vptr = '\0';
    cptr++;
    *s = cptr;
    return sptr;
}

void magic_parse_init(typeinitfunc *initfunc) {
    size_t i;
    count = 0; /* Added by O. Mai to allow reparsing of magic file */
    if( initfunc == NULL )
	    return;
    for( i = 0 ; i < sizeof(builtins)/sizeof(builtins[0]) ; i ++ ) {
	    builtins[i].mime = initfunc(builtins[i].message);
    }
}

void magic_parse_file(char *name, typeinitfunc *initfunc, size_t maxlen)
{
    magic_parse_init(initfunc);
    magic_parse_singlefile(name,initfunc,maxlen);
}
