/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifndef PATH_MAX
#define PATH_MAX 4096
#endif

#include <glib.h>
#include <glib-object.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#if !GTK_CHECK_VERSION(2, 6, 0)
#define GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID (-2)
#endif

#include <libxfcegui4/libxfcegui4.h>

#define EXO_API_SUBJECT_TO_CHANGE
#include <exo/exo.h>

#include <xfmedia/xfmedia-playlist.h>
#include "playlist-files.h"
#include "jumptofilewin.h"

enum {
	PLAYLIST_COL_NUM = 0,
	PLAYLIST_COL_TITLE,
	PLAYLIST_COL_LENGTH_S,
	PLAYLIST_COL_FILENAME,
	PLAYLIST_COL_LENGTH,
	PLAYLIST_COL_METADATA_LOADED,
	PLAYLIST_N_COLS
};

enum {
    PLIST_SIG_CLEARED = 0,
    PLIST_SIG_OPENED,
    PLIST_SIG_SAVED,
    PLIST_SIG_SCROLLED,
    PLIST_SIG_ENTRY_ADDED,
    PLIST_SIG_ENTRY_CHANGED,
    PLIST_SIG_ENTRY_REMOVED,
    PLIST_SIG_ENTRY_ACTIVATED,
    PLIST_SIG_SHUFFLE_TOGGLED,
    PLIST_SIG_REPEAT_TOGGLED,
    PLIST_N_SIGS
};

/* NOTE: keep in sync with mediamarks.c.  though i don't remember why. */
enum {
	DROP_TGT_REORDER = 0,
	DROP_TGT_URILIST,
	DROP_TGT_STRING
};

static void xfmedia_playlist_class_init(XfMediaPlaylistClass *klass);
static void xfmedia_playlist_init(XfMediaPlaylist *plist);
static void xfmedia_playlist_finalize(GObject *object);

static gboolean xfmedia_playlist_save_as_cb(GtkWidget *w, gpointer user_data);

struct _XfMediaPlaylistEntryRef
{
	XfMediaPlaylist *plist;
	GtkTreeRowReference *rref;
};

struct _XfMediaPlaylistPriv
{
	GtkWidget *treeview;
	GtkListStore *file_list;
	GtkTreeSelection *tree_sel;
	GtkTreeRowReference *cur_playing;
	gint cur_playing_len;
	GQueue *playlist_queue;
	
    GtkTooltips *ttips;
    
	GtkWidget *popup_menu;
	GtkWidget *shuffle_btn;
	GtkWidget *repeat_btn;
    
    GtkWidget *add_menu;
    GtkWidget *external_menu;
	
	guint8 *shuffle_bitmap;
	guint shuffle_bitmap_len;
	
	gchar *chooser_last_dir;
	gboolean is_dirty;
    
    XfMediaPlaylistEntryRef *bold_ref;
    XfMediaPlaylistEntryRef *italic_ref;
};

static GtkVBoxClass *parent_class = NULL;
static guint __signals[PLIST_N_SIGS] = { 0 };

GType
xfmedia_playlist_get_type()
{
    static GType plist_type = 0;
    
    if(!plist_type) {
        static const GTypeInfo plist_info = {
            sizeof(XfMediaPlaylistClass),
            NULL,
            NULL,
            (GClassInitFunc)xfmedia_playlist_class_init,
            NULL,
            NULL,
            sizeof(XfMediaPlaylist),
            0,
            (GInstanceInitFunc)xfmedia_playlist_init,
        };
        
        plist_type = g_type_register_static(GTK_TYPE_VBOX, "XfMediaPlaylist",
                &plist_info, 0);
    }
    
    return plist_type;
}

static void
xfmedia_playlist_class_init(XfMediaPlaylistClass *klass)
{
    GObjectClass *gobject_class;
    
    gobject_class = (GObjectClass *)klass;
    
    parent_class = g_type_class_peek_parent(klass);
    
    gobject_class->finalize = xfmedia_playlist_finalize;
    
    __signals[PLIST_SIG_CLEARED] = g_signal_new("playlist-cleared",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfMediaPlaylistClass, playlist_cleared), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    __signals[PLIST_SIG_OPENED] = g_signal_new("playlist-opened",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfMediaPlaylistClass, playlist_opened), NULL, NULL,
            g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);
    
    __signals[PLIST_SIG_SAVED] = g_signal_new("playlist-saved",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfMediaPlaylistClass, playlist_saved), NULL, NULL,
            g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);
    
    __signals[PLIST_SIG_SCROLLED] = g_signal_new("playlist-scrolled",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfMediaPlaylistClass, playlist_scrolled), NULL,
            NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    __signals[PLIST_SIG_ENTRY_ADDED] = g_signal_new("entry-added",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfMediaPlaylistClass, entry_added), NULL, NULL,
            g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);
    
    __signals[PLIST_SIG_ENTRY_CHANGED] = g_signal_new("entry-changed",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfMediaPlaylistClass, entry_changed), NULL, NULL,
            g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);
    
    __signals[PLIST_SIG_ENTRY_REMOVED] = g_signal_new("entry-removed",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfMediaPlaylistClass, entry_removed), NULL, NULL,
            g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);
    
    __signals[PLIST_SIG_ENTRY_ACTIVATED] = g_signal_new("entry-activated",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfMediaPlaylistClass, entry_activated), NULL, NULL,
            g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);
    
    __signals[PLIST_SIG_SHUFFLE_TOGGLED] = g_signal_new("shuffle-toggled",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfMediaPlaylistClass, shuffle_toggled), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    __signals[PLIST_SIG_REPEAT_TOGGLED] = g_signal_new("repeat-toggled",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfMediaPlaylistClass, repeat_toggled), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
}

static void
xfmedia_playlist_init(XfMediaPlaylist *plist)
{
    plist->priv = g_new0(XfMediaPlaylistPriv, 1);
}

static void
xfmedia_playlist_finalize(GObject *object)
{
    XfMediaPlaylist *plist;
    
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(object));
    plist = XFMEDIA_PLAYLIST(object);
    
    if(plist->priv->chooser_last_dir)
        g_free(plist->priv->chooser_last_dir);
    
    g_object_unref(G_OBJECT(plist->priv->ttips));
    
    g_free(plist->priv);
    plist->priv = NULL;
    
    G_OBJECT_CLASS(parent_class)->finalize(object);
}


static void
xfmedia_playlist_renumber_file_list(GtkListStore *ls, gint from_index,
        gint to_index)
{
    GtkTreeIter itr;
    gint i, nrows;
    
    DBG("renumbering from %d to %d", from_index, to_index);
    
    if(from_index < 0)
        from_index = 0;
    i = from_index + 1;
    nrows = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(ls), NULL);
    if(to_index >= nrows || to_index < 0)
        to_index = nrows - 1;
    
    if(from_index == 0 && gtk_tree_model_get_iter_first(GTK_TREE_MODEL(ls), &itr)) {
        gtk_list_store_set(ls, &itr, PLAYLIST_COL_NUM, 1, -1);
        i++;
    } else if(from_index == 0 || (from_index > 0
            && !gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(ls), &itr,
                                             NULL, from_index-1)))
    {
        return;
    }
    
    for(; gtk_tree_model_iter_next(GTK_TREE_MODEL(ls), &itr); i++) {
        gtk_list_store_set(ls, &itr, PLAYLIST_COL_NUM, i, -1);
        if(i == to_index + 1)
            break;
    }
}

static void
xfmedia_playlist_cell_render_title(GtkTreeViewColumn *tree_column,
        GtkCellRenderer *cell, GtkTreeModel *tree_model, GtkTreeIter *iter,
        gpointer data)
{
    XfMediaPlaylist *plist = data;
    GtkTreePath *cur_path = NULL;
    gint *indices, cur_index = -1;
    
    cur_path = gtk_tree_model_get_path(tree_model, iter);
    if(!cur_path)
        return;
    indices = gtk_tree_path_get_indices(cur_path);
    if(indices)
        cur_index = indices[0];
    gtk_tree_path_free(cur_path);
    if(!indices)
        return;
    
    if(plist->priv->bold_ref && xfmedia_playlist_entry_ref_get_index(plist->priv->bold_ref) == cur_index)
        g_object_set(G_OBJECT(cell), "weight", PANGO_WEIGHT_BOLD, NULL);
    else
        g_object_set(G_OBJECT(cell), "weight", PANGO_WEIGHT_NORMAL, NULL);
    
    if(plist->priv->italic_ref && xfmedia_playlist_entry_ref_get_index(plist->priv->italic_ref) == cur_index)
        g_object_set(G_OBJECT(cell), "style", PANGO_STYLE_ITALIC, "style-set", TRUE, NULL);
    else
        g_object_set(G_OBJECT(cell), "style-set", FALSE, NULL);
}

static void
xfmedia_playlist_treeview_scroll_cb(GtkAdjustment *adj, gpointer user_data)
{
    g_signal_emit(G_OBJECT(user_data), __signals[PLIST_SIG_SCROLLED], 0);
}

static void
xfmedia_playlist_shuffle_toggle_cb(GtkToggleButton *b, gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    gboolean set;
    
    set = gtk_toggle_button_get_active(b);
    if(set)
        xfmedia_playlist_shuffle_unset_all(plist);
    
    g_signal_emit(G_OBJECT(user_data), __signals[PLIST_SIG_SHUFFLE_TOGGLED], 0);
}

static void
xfmedia_playlist_repeat_toggle_cb(GtkToggleButton *b, gpointer user_data)
{
    g_signal_emit(G_OBJECT(user_data), __signals[PLIST_SIG_REPEAT_TOGGLED], 0);
}

static void
playlist_zero_pointer(GtkWidget **w)
{
    if(w)
        *w = NULL;
}

static void
playlist_add_url_dialog_response_cb(GtkWidget *dlg, gint response,
        gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    GtkWidget *entry;
    gchar *url;
    
    entry = g_object_get_data(G_OBJECT(dlg), "xfmedia-url-entry");
    
    url = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);
    if(url && *url)
        xfmedia_playlist_append_entry(plist, NULL, -1, url, FALSE);
    g_free(url);
    
    gtk_widget_destroy(dlg);
    xfmedia_playlist_treeview_scroll_cb(NULL, plist);
}

static void
xfmedia_playlist_add_url_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    static GtkWidget *dlg = NULL;
    GtkWidget *lbl, *entry, *topvbox, *hbox, *toplevel;
    
    if(dlg) {
        gtk_window_present(GTK_WINDOW(dlg));
        return;
    }
    
    toplevel = gtk_widget_get_toplevel(GTK_WIDGET(plist));
    dlg = gtk_dialog_new_with_buttons(_("Add URL"), GTK_WINDOW(toplevel),
            GTK_DIALOG_NO_SEPARATOR|GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OK,
            GTK_RESPONSE_ACCEPT, NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dlg), GTK_RESPONSE_ACCEPT);
    gtk_container_set_border_width(GTK_CONTAINER(dlg), BORDER);
    g_signal_connect(G_OBJECT(dlg), "response",
            G_CALLBACK(playlist_add_url_dialog_response_cb), plist);
    g_signal_connect_swapped(G_OBJECT(dlg), "destroy",
            G_CALLBACK(playlist_zero_pointer), &dlg);
    
    topvbox = GTK_DIALOG(dlg)->vbox;
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(topvbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(_("_URL:"));
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    gtk_entry_set_activates_default(GTK_ENTRY(entry), TRUE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), entry);
    g_object_set_data(G_OBJECT(dlg), "xfmedia-url-entry", entry);
    
    gtk_widget_set_size_request(dlg, 400, -1);
    gtk_widget_show(dlg);
}

static guint
recurse_add_dirs(XfMediaPlaylist *plist, const gchar *dirname)
{
    GDir *dir;
    const gchar *file;
    gchar fullpath[PATH_MAX];
    GList *file_list = NULL, *dir_list = NULL, *l;
    guint nadded = 0;
    
    dir = g_dir_open(dirname, 0, NULL);
    if(!dir)
        return 0;
    
    while((file = g_dir_read_name(dir))) {
        g_snprintf(fullpath, PATH_MAX, "%s%s%s", dirname, G_DIR_SEPARATOR_S, file);
        if(g_file_test(fullpath, G_FILE_TEST_IS_DIR)) {
            dir_list = g_list_insert_sorted(dir_list, g_strdup(fullpath),
                    (GCompareFunc)g_ascii_strcasecmp);
        } else {
            file_list = g_list_insert_sorted(file_list, g_strdup(fullpath),
                    (GCompareFunc)g_ascii_strcasecmp);
        }
    }
    g_dir_close(dir);
    
    for(l = dir_list; l; l = l->next) {
        nadded += recurse_add_dirs(plist, (const gchar *)l->data);
        g_free(l->data);
    }
    if(dir_list)
        g_list_free(dir_list);
    
    gdk_threads_enter();
    for(l = file_list; l; l = l->next) {
        xfmedia_playlist_append_entry(plist, NULL, -1, (const gchar *)l->data, FALSE);
        nadded++;
        g_free(l->data);
    }
    gdk_threads_leave();
    
    if(file_list)
        g_list_free(file_list);
    
    return nadded;
}

struct RecurseData {
    XfMediaPlaylist *plist;
    gchar *basedir;
};

static gpointer
recurse_add_dirs_th(gpointer data)
{
    struct RecurseData *recurse_data = data;
    guint nadded;
    
    nadded = recurse_add_dirs(recurse_data->plist, recurse_data->basedir);
    
    if(nadded > 0) {
        gdk_threads_enter();
        //xfmedia_playlist_shuffle_add_entries(recurse_data->mwin->plist,
        //        -1, nadded);
        xfmedia_playlist_treeview_scroll_cb(NULL, recurse_data->plist);
        gdk_threads_leave();
    }
    
    g_free(recurse_data->basedir);
    g_free(recurse_data);
    
    return NULL;
}

#define MY_FOLDER_SELECT_RESPONSE 749837
static void
playlist_add_dir_chooser_response_cb(GtkWidget *chooser, gint response,
        gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    struct RecurseData *recurse_data;
    gchar *dirname;
    
    dirname = xfce_file_chooser_get_filename(XFCE_FILE_CHOOSER(chooser));
    if(dirname) {
        recurse_data = g_new0(struct RecurseData, 1);
        recurse_data->plist = plist;
        recurse_data->basedir = dirname;
        plist->priv->is_dirty = TRUE;
        g_thread_create((GThreadFunc)recurse_add_dirs_th, recurse_data,
                FALSE, NULL);
    }
    
    gtk_widget_destroy(chooser);
}

static void
xfmedia_playlist_add_dir_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    static GtkWidget *chooser = NULL;
    GtkWidget *toplevel;
    
    if(chooser) {
        gtk_window_present(GTK_WINDOW(chooser));
        return;
    }
    
    toplevel = gtk_widget_get_toplevel(GTK_WIDGET(plist));
    chooser = xfce_file_chooser_new(_("Add Directory"), GTK_WINDOW(toplevel),
            XFCE_FILE_CHOOSER_ACTION_SELECT_FOLDER, GTK_STOCK_CANCEL,
            GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN,
            MY_FOLDER_SELECT_RESPONSE, NULL);
    if(plist->priv->chooser_last_dir)
        xfce_file_chooser_set_current_folder(XFCE_FILE_CHOOSER(chooser),
                plist->priv->chooser_last_dir);
    gtk_dialog_set_default_response(GTK_DIALOG(chooser), GTK_RESPONSE_ACCEPT);
    g_signal_connect(G_OBJECT(chooser), "response",
            G_CALLBACK(playlist_add_dir_chooser_response_cb), plist);
    g_signal_connect_swapped(G_OBJECT(chooser), "destroy",
            G_CALLBACK(playlist_zero_pointer), &chooser);
    
    gtk_widget_show(chooser);
}
#undef MY_FOLDER_SELECT_RESPONSE

static void
playlist_add_file_chooser_response_cb(GtkWidget *chooser, gint response,
        gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    GSList *filenames, *l;
    gboolean defer = TRUE;
    gint nadded = 0;
    
    if(response == GTK_RESPONSE_ACCEPT) {
        gtk_widget_hide(chooser);
        while(gtk_events_pending())
            gtk_main_iteration();
    }
    
    if(response == GTK_RESPONSE_ACCEPT || response == GTK_RESPONSE_APPLY) {
        filenames = xfce_file_chooser_get_filenames(XFCE_FILE_CHOOSER(chooser));
        if(filenames && g_slist_length(filenames) < 10)
            defer = FALSE;
        for(l = filenames; l; l = l->next) {
            if(xfmedia_playlists_is_playlist_file(l->data))
                xfmedia_playlists_load(plist, l->data);
            else {
                gchar *title_utf8 = xfmedia_filename_to_name(l->data);
                xfmedia_playlist_append_entry(plist, title_utf8, -1, l->data, FALSE);
                g_free(title_utf8);
                nadded++;
            }
            g_free(l->data);
        }
        if(filenames)
            g_slist_free(filenames);
        if(plist->priv->chooser_last_dir)
            g_free(plist->priv->chooser_last_dir);
        plist->priv->chooser_last_dir = xfce_file_chooser_get_current_folder(XFCE_FILE_CHOOSER(chooser));
        plist->priv->is_dirty = TRUE;
        
        //if(nadded > 0)
        //    xfmedia_playlist_shuffle_add_entries(mwin->plist, -1, nadded);
    }
    
    if(response != GTK_RESPONSE_APPLY)
        gtk_widget_destroy(chooser);
    if(response != GTK_RESPONSE_CANCEL)
        xfmedia_playlist_treeview_scroll_cb(NULL, plist);
}

static void
xfmedia_playlist_add_file_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    static GtkWidget *chooser = NULL;
    GtkWidget *btn, *toplevel;
    XfceFileFilter *filter;
    
    if(chooser) {
        gtk_window_present(GTK_WINDOW(chooser));
        return;
    }
    
    toplevel = gtk_widget_get_toplevel(GTK_WIDGET(plist));
    chooser = xfce_file_chooser_new(_("Add Media File"), GTK_WINDOW(toplevel),
            XFCE_FILE_CHOOSER_ACTION_OPEN, NULL);
    btn = xfmedia_custom_button_new(_("Add _Selected"), GTK_STOCK_ADD);
    gtk_widget_show(btn);
    gtk_dialog_add_action_widget(GTK_DIALOG(chooser), btn, GTK_RESPONSE_APPLY);
    gtk_dialog_add_button(GTK_DIALOG(chooser), GTK_STOCK_CLOSE, GTK_RESPONSE_CANCEL);
    btn = xfmedia_custom_button_new(_("A_dd and Close"), GTK_STOCK_ADD);
    GTK_WIDGET_SET_FLAGS(btn, GTK_CAN_DEFAULT);
    gtk_widget_show(btn);
    gtk_dialog_add_action_widget(GTK_DIALOG(chooser), btn, GTK_RESPONSE_ACCEPT);
    xfce_file_chooser_set_select_multiple(XFCE_FILE_CHOOSER(chooser), TRUE);
    if(plist->priv->chooser_last_dir)
        xfce_file_chooser_set_current_folder(XFCE_FILE_CHOOSER(chooser),
                plist->priv->chooser_last_dir);
    gtk_dialog_set_default_response(GTK_DIALOG(chooser), GTK_RESPONSE_ACCEPT);
    g_signal_connect(G_OBJECT(chooser), "response",
            G_CALLBACK(playlist_add_file_chooser_response_cb), plist);
    g_signal_connect_swapped(G_OBJECT(chooser), "destroy",
            G_CALLBACK(playlist_zero_pointer), &chooser);
    
    filter = xfce_file_filter_new();
    xfce_file_filter_set_name(filter, _("All Files"));
    xfce_file_filter_add_pattern(filter, "*");
    xfce_file_chooser_add_filter(XFCE_FILE_CHOOSER(chooser), filter);
    filter = xfce_file_filter_new();
    xfce_file_filter_set_name(filter, _("Music Files"));
    xfce_file_filter_add_pattern(filter, "*.aif*");
    xfce_file_filter_add_pattern(filter, "*.au");
    xfce_file_filter_add_pattern(filter, "*.flac");
    xfce_file_filter_add_pattern(filter, "*.mka");
    xfce_file_filter_add_pattern(filter, "*.mpa");
    xfce_file_filter_add_pattern(filter, "*.mp2");
    xfce_file_filter_add_pattern(filter, "*.mp3");
    xfce_file_filter_add_pattern(filter, "*.ogg");
    xfce_file_filter_add_pattern(filter, "*.wav");
    xfce_file_filter_add_pattern(filter, "*.wma");
    xfce_file_chooser_add_filter(XFCE_FILE_CHOOSER(chooser), filter);
    filter = xfce_file_filter_new();
    xfce_file_filter_set_name(filter, _("Playlists"));
    xfce_file_filter_add_pattern(filter, "*.m3u");
    xfce_file_filter_add_pattern(filter, "*.pls");
    xfce_file_filter_add_pattern(filter, "*.asx");
    xfce_file_chooser_add_filter(XFCE_FILE_CHOOSER(chooser), filter);
    filter = xfce_file_filter_new();
    xfce_file_filter_set_name(filter, _("Video Files"));
    xfce_file_filter_add_pattern(filter, "*.avi");
    xfce_file_filter_add_pattern(filter, "*.mkv");
    xfce_file_filter_add_pattern(filter, "*.mpeg");
    xfce_file_filter_add_pattern(filter, "*.mpg");
    xfce_file_filter_add_pattern(filter, "*.ogg");
    xfce_file_filter_add_pattern(filter, "*.ogm");
    xfce_file_filter_add_pattern(filter, "*.vob");
    xfce_file_filter_add_pattern(filter, "*.wmv");
    xfce_file_chooser_add_filter(XFCE_FILE_CHOOSER(chooser), filter);
    
    gtk_widget_show(chooser);
}

static void
add_menu_position(GtkMenu *menu, gint *x, gint *y, gboolean *push_in, 
        gpointer user_data)
{
    GtkRequisition req;
    gint xpointer, ypointer;
    
    gtk_widget_size_request(GTK_WIDGET(menu), &req);
    
    /* get pointer position */
    gdk_display_get_pointer(
            gdk_screen_get_display(gtk_widget_get_screen(GTK_WIDGET(menu))),
            NULL, &xpointer, &ypointer, NULL);
    
    /* set x to the current position */
    *x = xpointer;
    /* ...and y to the current position minus the menu height */
    *y = ypointer - req.height;
    
    /* FIXME: huh? */
    *push_in = FALSE;
}

static void
xfmedia_playlist_add_menu_deactivate_cb(GtkWidget *w, gpointer user_data)
{
	guint sig_id;
	
	sig_id = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(w),
			"xfmedia-plist-deactivate-sigid"));
	g_signal_handler_disconnect(G_OBJECT(w), sig_id);
    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(user_data), FALSE);
}

static void
xfmedia_playlist_add_cb(GtkToggleButton *tb, gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    guint sig_id;
    
    if(!gtk_toggle_button_get_active(tb))
        return;
    
    sig_id = g_signal_connect(G_OBJECT(plist->priv->add_menu), "deactivate",
    		G_CALLBACK(xfmedia_playlist_add_menu_deactivate_cb), tb);
    g_object_set_data(G_OBJECT(plist->priv->add_menu),
    		"xfmedia-plist-deactivate-sigid", GUINT_TO_POINTER(sig_id));
    
    gtk_menu_popup(GTK_MENU(plist->priv->add_menu), NULL, NULL,
            (GtkMenuPositionFunc)add_menu_position, tb, 0,
            gtk_get_current_event_time());
}

static void
xfmedia_playlist_create_add_menu(XfMediaPlaylist *plist)
{
    GtkWidget *menu, *mi, *img;
    
    plist->priv->add_menu = menu = gtk_menu_new();
    gtk_widget_show(menu);
    
    img = gtk_image_new_from_stock(GTK_STOCK_NETWORK, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    mi = gtk_image_menu_item_new_with_mnemonic(_("_URL..."));
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_playlist_add_url_cb), plist);
    
    img = gtk_image_new_from_stock(GTK_STOCK_OPEN, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    mi = gtk_image_menu_item_new_with_mnemonic(_("_Directory..."));
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_playlist_add_dir_cb), plist);
    
    img = gtk_image_new_from_stock(GTK_STOCK_HARDDISK, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    mi = gtk_image_menu_item_new_with_mnemonic(_("_File..."));
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_widget_show(mi);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_playlist_add_file_cb), plist);
}


static void
xfmedia_playlist_remove_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    GList *selected, *l, *rrefs = NULL;
    guint low_index = -1, index;
    
    selected = gtk_tree_selection_get_selected_rows(plist->priv->tree_sel, NULL);
    if(selected) {
        for(l = selected; l; l = l->next) {
            GtkTreePath *path = l->data;
            gint *indices;
            
            indices = gtk_tree_path_get_indices(path);
            index = indices[0];
            if(index < low_index)
                low_index = index;
            
            rrefs = g_list_prepend(rrefs,
                gtk_tree_row_reference_new(GTK_TREE_MODEL(plist->priv->file_list), path));
            gtk_tree_path_free(path);
        }
        g_list_free(selected);
        
        for(l = rrefs; l; l = l->next) {
            GtkTreeRowReference *rref = l->data;
            GtkTreePath *path = gtk_tree_row_reference_get_path(rref);
            gint *indices, path_index;
            
            indices = gtk_tree_path_get_indices(path);
            path_index = indices[0];
            
            xfmedia_playlist_remove_entry(plist, path_index);
            gtk_tree_path_free(path);
            gtk_tree_row_reference_free(rref);
        }
        g_list_free(rrefs);
    }
    
    DBG("renumbering from %d", low_index);
    if(low_index != -1)
        xfmedia_playlist_renumber_file_list(plist->priv->file_list, low_index, -1);        
}

static void
xfmedia_playlist_go_up_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    GtkTreeIter me, above_me;
    gint index, path_index;
    GList *selected;
    
    selected = gtk_tree_selection_get_selected_rows(plist->priv->tree_sel, NULL);
    if(selected && g_list_length(selected) == 1) {
        if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list), &me,
                (GtkTreePath *)selected->data))
        {
            if(gtk_tree_path_prev((GtkTreePath *)selected->data)) {
                if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                        &above_me, (GtkTreePath *)selected->data))
                {
                    gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list),
                            &above_me, PLAYLIST_COL_NUM, &index, -1);
                    gtk_list_store_set(plist->priv->file_list, &above_me,
                            PLAYLIST_COL_NUM, index+1, -1);
                    gtk_list_store_set(plist->priv->file_list, &me,
                            PLAYLIST_COL_NUM, index, -1);
                    gtk_list_store_swap(plist->priv->file_list, &me, &above_me);
                    path_index = xfmedia_tree_path_to_index((GtkTreePath *)selected->data);
                    xfmedia_playlist_shuffle_swap_entries(plist, path_index, path_index-1);
                    plist->priv->is_dirty = TRUE;
                }
            }
        }
    }
    
    if(selected) {
        g_list_foreach(selected, (GFunc)gtk_tree_path_free, NULL);
        g_list_free(selected);
    }
}

static void
xfmedia_playlist_go_down_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    GtkTreeIter me, below_me;
    gint index, path_index;
    GList *selected;
    
    selected = gtk_tree_selection_get_selected_rows(plist->priv->tree_sel, NULL);
    if(selected && g_list_length(selected) == 1) {
        if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list), &me,
                (GtkTreePath *)selected->data))
        {
            gtk_tree_path_next((GtkTreePath *)selected->data);
            if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                    &below_me, (GtkTreePath *)selected->data))
            {
                gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list),
                        &below_me, PLAYLIST_COL_NUM, &index, -1);
                gtk_list_store_set(plist->priv->file_list, &below_me,
                        PLAYLIST_COL_NUM, index-1, -1);
                gtk_list_store_set(plist->priv->file_list, &me,
                        PLAYLIST_COL_NUM, index, -1);
                gtk_list_store_swap(plist->priv->file_list, &me, &below_me);
                path_index = xfmedia_tree_path_to_index((GtkTreePath *)selected->data);
                xfmedia_playlist_shuffle_swap_entries(plist, path_index, path_index+1);
                plist->priv->is_dirty = TRUE;
            }
        }
    }
    
    if(selected) {
        g_list_foreach(selected, (GFunc)gtk_tree_path_free, NULL);
        g_list_free(selected);
    }
}

static gint
playlist_dirty_confirm(GtkWindow *parent)
{
    const gchar *primary = _("Save changes to this playlist before closing?");
    const gchar *secondary = _("You've changed this playlist since it was last saved.");
    
    return xfce_message_dialog(parent, "Xfmedia", GTK_STOCK_DIALOG_WARNING,
            primary, secondary,
            XFCE_CUSTOM_STOCK_BUTTON, _("Do_n't save"),
                GTK_STOCK_NO, GTK_RESPONSE_NO,
            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
            GTK_STOCK_SAVE, GTK_RESPONSE_YES, NULL);
}

static void
xfmedia_playlist_new_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    GtkWidget *toplevel;
    gint resp;
    
    if(plist->priv->is_dirty && xfmedia_playlist_get_n_entries(plist) > 0) {
        toplevel = gtk_widget_get_toplevel(GTK_WIDGET(plist));
        resp = playlist_dirty_confirm(GTK_WINDOW(toplevel));
        switch(resp) {
            case GTK_RESPONSE_CANCEL:
                return;
            case GTK_RESPONSE_YES:
                if(!xfmedia_playlist_save_as_cb(GTK_WIDGET(plist), plist))
                    return;
                break;
            case GTK_RESPONSE_NO:
                break;
        }
    }
    
    xfmedia_playlist_clear(plist);
    plist->priv->is_dirty = FALSE;
    xfmedia_playlist_shuffle_init_entries(plist, 0);
    
    g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_CLEARED], 0);
}

static gboolean
xfmedia_playlist_save_as_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    GtkWidget *topwin, *chooser;
    gchar *filename;
    gboolean ret = FALSE;
    
    if(!xfmedia_playlist_get_n_entries(plist))
        return TRUE;
    
    topwin = gtk_widget_get_toplevel(w);
    
    chooser = xfce_file_chooser_new(_("Save Playlist"), GTK_WINDOW(topwin),
            XFCE_FILE_CHOOSER_ACTION_SAVE, GTK_STOCK_CANCEL,
            GTK_RESPONSE_CANCEL, GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, NULL);
    xfce_file_chooser_set_select_multiple(XFCE_FILE_CHOOSER(chooser), FALSE);
    if(plist->priv->chooser_last_dir)
        xfce_file_chooser_set_current_folder(XFCE_FILE_CHOOSER(chooser),
                plist->priv->chooser_last_dir);
    gtk_dialog_set_default_response(GTK_DIALOG(chooser), GTK_RESPONSE_ACCEPT);
    gtk_widget_show(chooser);
    
    if(gtk_dialog_run(GTK_DIALOG(chooser)) == GTK_RESPONSE_ACCEPT) {
        gtk_widget_hide(chooser);
        filename = xfce_file_chooser_get_filename(XFCE_FILE_CHOOSER(chooser));
        if(filename) {
            ret = xfmedia_playlists_save(plist, filename);
            g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_SAVED], 0, filename);
            g_free(filename);
        }
        if(plist->priv->chooser_last_dir)
            g_free(plist->priv->chooser_last_dir);
        plist->priv->chooser_last_dir = xfce_file_chooser_get_current_folder(XFCE_FILE_CHOOSER(chooser));

    }
    gtk_widget_destroy(chooser);
    
    return ret;
}

static void
xfmedia_playlist_open_cb(GtkWidget *w, gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    GtkWidget *topwin, *chooser;
    XfceFileFilter *filter;
    gchar *filename;
    gint resp;
    
    topwin = gtk_widget_get_toplevel(w);
    
    if(plist->priv->is_dirty && xfmedia_playlist_get_n_entries(plist) > 0) {
        resp = playlist_dirty_confirm(GTK_WINDOW(w));
        switch(resp) {
            case GTK_RESPONSE_CANCEL:
                return;
            case GTK_RESPONSE_YES:
                if(!xfmedia_playlist_save_as_cb(w, plist))
                    return;
                break;
            case GTK_RESPONSE_NO:
                break;
        }
    }
    
    chooser = xfce_file_chooser_new(_("Open Playlist"), GTK_WINDOW(topwin),
            XFCE_FILE_CHOOSER_ACTION_OPEN, GTK_STOCK_CANCEL,
            GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);
    xfce_file_chooser_set_select_multiple(XFCE_FILE_CHOOSER(chooser), FALSE);
    if(plist->priv->chooser_last_dir)
        xfce_file_chooser_set_current_folder(XFCE_FILE_CHOOSER(chooser),
                plist->priv->chooser_last_dir);
    gtk_dialog_set_default_response(GTK_DIALOG(chooser), GTK_RESPONSE_ACCEPT);
    
    filter = xfce_file_filter_new();
    xfce_file_filter_set_name(filter, _("All Files"));
    xfce_file_filter_add_pattern(filter, "*");
    xfce_file_chooser_add_filter(XFCE_FILE_CHOOSER(chooser), filter);
    filter = xfce_file_filter_new();
    xfce_file_filter_set_name(filter, _("Playlists"));
    xfce_file_filter_add_pattern(filter, "*.m3u");
    xfce_file_filter_add_pattern(filter, "*.pls");
    xfce_file_filter_add_pattern(filter, "*.asx");
    xfce_file_chooser_add_filter(XFCE_FILE_CHOOSER(chooser), filter);
    
    gtk_widget_show(chooser);
    
    if(gtk_dialog_run(GTK_DIALOG(chooser)) == GTK_RESPONSE_ACCEPT) {
        gtk_widget_hide(chooser);
        filename = xfce_file_chooser_get_filename(XFCE_FILE_CHOOSER(chooser));
        if(filename) {
            xfmedia_playlist_clear(plist);
            if(xfmedia_playlists_load(plist, filename)) {
                plist->priv->is_dirty = FALSE;
                g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_OPENED], 0, filename);
            }
            g_free(filename);
        }
        if(plist->priv->chooser_last_dir)
            g_free(plist->priv->chooser_last_dir);
        plist->priv->chooser_last_dir = xfce_file_chooser_get_current_folder(XFCE_FILE_CHOOSER(chooser));
    }
    gtk_widget_destroy(chooser);
    xfmedia_playlist_treeview_scroll_cb(NULL, plist);
}

static gboolean
xfmedia_playlist_treeview_keypress_cb(GtkWidget *w, GdkEventKey *evt,
        gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    
    if((evt->keyval == GDK_Return || evt->keyval == GDK_KP_Enter) ) {
        GList *selected;
        
        selected = gtk_tree_selection_get_selected_rows(plist->priv->tree_sel, NULL);
        if(selected) {
            gint *indices;
            
            xfmedia_playlist_shuffle_unset_all(plist);
            
            indices = gtk_tree_path_get_indices((GtkTreePath *)selected->data);
            g_signal_emit(G_OBJECT(plist),
                    __signals[PLIST_SIG_ENTRY_ACTIVATED], 0, indices[0]);
            
            g_list_foreach(selected, (GFunc)gtk_tree_path_free, NULL);
            g_list_free(selected);
        }
        
        return TRUE;
    } else if(evt->keyval == GDK_Delete || evt->keyval == GDK_KP_Delete) {
        xfmedia_playlist_remove_cb(w, plist);
        return TRUE;
    }
    
    return FALSE;
}

static gboolean
xfmedia_playlist_treeview_buttonpress_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    GList *selected;
    
    if(evt->button == 1 && evt->type == GDK_2BUTTON_PRESS) {
        selected = gtk_tree_selection_get_selected_rows(plist->priv->tree_sel, NULL);
        if(selected) {
            gint *indices;
            
            xfmedia_playlist_shuffle_unset_all(plist);
            
            indices = gtk_tree_path_get_indices((GtkTreePath *)selected->data);
            g_signal_emit(G_OBJECT(plist),
                    __signals[PLIST_SIG_ENTRY_ACTIVATED], 0, indices[0]);
            
            /* free selection stuff */
            g_list_foreach(selected, (GFunc)gtk_tree_path_free, NULL);
            g_list_free(selected);
        }
    } else if(evt->button == 3) {
        GtkTreePath *mouse_over = NULL;
        
        if(gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(plist->priv->treeview),
                evt->x, evt->y, &mouse_over, NULL, NULL, NULL))
        {
            gtk_tree_selection_unselect_all(plist->priv->tree_sel);
            gtk_tree_selection_select_path(plist->priv->tree_sel, mouse_over);
            gtk_tree_path_free(mouse_over);
            
            gtk_menu_popup(GTK_MENU(plist->priv->popup_menu), NULL, NULL, NULL,
                    NULL, evt->button, evt->time);
            return TRUE;  /* eat event so DnD doesn't kick in */
        }
    }
    
    return FALSE;
}

static void
xfmedia_playlist_treeview_drag_get_cb(GtkWidget *widget,
        GdkDragContext *drag_context, GtkSelectionData *data, guint info,
        guint time, gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    GdkAtom playlist_item = gdk_atom_intern("PLAYLIST_ITEM", FALSE);
    
    if(data->target == playlist_item) {
        GtkTreeRowReference *srcrref = g_object_get_data(G_OBJECT(drag_context),
                "gtk-tree-view-source-row");
        GtkTreePath *sourcerow = gtk_tree_row_reference_get_path(srcrref);
        GtkTreeIter *itr;
        
        if(!sourcerow)
            return;
        
        itr  = g_new0(GtkTreeIter, 1);
        gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                itr, sourcerow);
        
        gtk_selection_data_set(data, playlist_item, 8,
                (gpointer)&itr, sizeof(itr));
        
        gtk_tree_path_free(sourcerow);
    }
}

static void
xfmedia_playlist_treeview_drag_rcv_cb(GtkWidget *widget,
        GdkDragContext *drag_context, gint x, gint y,  GtkSelectionData *data,
        guint info, guint time, gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    GdkAtom _PLAYLIST_ITEM = gdk_atom_intern("PLAYLIST_ITEM", FALSE);
    GdkAtom _URI_LIST = gdk_atom_intern("text/uri-list", FALSE);
    GdkAtom _STRING = gdk_atom_intern("STRING", FALSE);
    GtkTreePath *dest_path = NULL;
    GtkTreeViewDropPosition drop_pos;
    GtkTreeIter *src_itr, dest_itr;
    gboolean drag_success = FALSE;
    
    DBG("got DnD target: %s", gdk_atom_name(data->target));
    
    if(data->target == _PLAYLIST_ITEM) {
        src_itr = *(GtkTreeIter **)data->data;
        if(!src_itr) {
            gtk_drag_finish(drag_context, FALSE, FALSE, time);
            return;
        }
        if(gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(plist->priv->treeview),
                    x, y, &dest_path, &drop_pos))
        {
            if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                                       &dest_itr, dest_path)
                    && gtk_list_store_iter_is_valid(plist->priv->file_list,
                                                    src_itr))
            {
                gint src_index = 0, dest_index = 0;
                
                gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list),
                        src_itr, PLAYLIST_COL_NUM, &src_index, -1);
                gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list),
                        &dest_itr, PLAYLIST_COL_NUM, &dest_index, -1);
                
				if(src_index != dest_index) {
					src_index--;
					dest_index--;
					
					DBG("src_index=%d, dest_index=%d", src_index, dest_index);
					
					if(drop_pos == GTK_TREE_VIEW_DROP_BEFORE
							|| drop_pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE)
					{
						DBG("putting src before dest");
						/* gtk oddity: if dest is the first item in the list, then
						 * gtk_list_store_move_before(ls, src, dest) will move src
						 * to the _end_ of the list.  somehow it thinks the item
						 * before the zeroth item is the last item. this sucks. */
						if(dest_index == 0) {
							GtkTreeModel *model = GTK_TREE_MODEL(plist->priv->file_list);
							/* first convert our iters into row refs */
							GtkTreePath *srcpath = gtk_tree_model_get_path(model, src_itr);
							GtkTreePath *destpath = gtk_tree_model_get_path(model, &dest_itr);
							GtkTreeRowReference *srcref =
								gtk_tree_row_reference_new(model, srcpath);
							GtkTreeRowReference *destref =
								gtk_tree_row_reference_new(model, destpath);
							gtk_tree_path_free(srcpath);
							gtk_tree_path_free(destpath);
							
							/* then move the src _after_ the dest */
							gtk_list_store_move_after(plist->priv->file_list,
									src_itr, &dest_itr);
							
							/* now get new iters from the row refs */
							srcpath = gtk_tree_row_reference_get_path(srcref);
							destpath = gtk_tree_row_reference_get_path(destref);
							gtk_tree_model_get_iter(model, src_itr, srcpath);
							gtk_tree_model_get_iter(model, &dest_itr, destpath);
							gtk_tree_path_free(srcpath);
							gtk_tree_path_free(destpath);
							
							/* swap the two items, so now dest is before src */
							gtk_list_store_swap(plist->priv->file_list, src_itr,
									&dest_itr);
							
							/* ditch the row refs */
							gtk_tree_row_reference_free(srcref);
							gtk_tree_row_reference_free(destref);
							
							/* man, that was stupid */
						} else {
							gtk_list_store_move_before(plist->priv->file_list,
									src_itr, &dest_itr);
						}
					} else {
						DBG("putting src after dest");
						gtk_list_store_move_after(plist->priv->file_list, src_itr,
								&dest_itr);
					}
					
					if(src_index < dest_index)
						xfmedia_playlist_renumber_file_list(plist->priv->file_list,
								src_index, dest_index);
					else
						xfmedia_playlist_renumber_file_list(plist->priv->file_list,
								dest_index, src_index);
					xfmedia_playlist_shuffle_swap_entries(plist, src_index, dest_index);
					
					plist->priv->is_dirty = TRUE;
					drag_success = TRUE;
				}
            } else {
                DBG("one of the itrs is invalid");
            }
            
            gtk_tree_path_free(dest_path);
        } else {
            DBG("gtk_tree_view_get_dest_row_at_pos() failed");
        }
        
        g_free(src_itr);
    } else if(data->target == _URI_LIST) {
        GList *uris, *l;
        gint from_index = -1, index, num_added = 0;
        
        DBG("we have an uri list: '%s'", (gchar *)data->data);
        
        if(gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(plist->priv->treeview),
                    x, y, &dest_path, &drop_pos))
        {
            if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                    &dest_itr, dest_path))
            {
                gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list),
                        &dest_itr, PLAYLIST_COL_NUM, &from_index, -1);
                if(drop_pos == GTK_TREE_VIEW_DROP_BEFORE
                        || drop_pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE)
                {
                    from_index--;
                }
            }
            gtk_tree_path_free(dest_path);
        }
        
        index = from_index;
        uris = xfmedia_uri_list_extract_uris((const gchar *)data->data);
        for(l = uris; l; l = l->next) {
            gchar *filename = l->data, *title_utf8;
            
            if(!g_ascii_strncasecmp(filename, "file://", 7))
                filename += 7;
            DBG("adding '%s'", filename);
            title_utf8 = xfmedia_filename_to_name(filename);
            if(from_index == -1)
                xfmedia_playlist_append_entry(plist, title_utf8, -1, filename, FALSE);
            else {
                xfmedia_playlist_insert_entry(plist, index, title_utf8, -1, filename, FALSE);
                if(index != -1)
                    index++;
            }
            g_free(title_utf8);
            g_free(l->data);
            
            num_added++;
        }
        if(uris) {
            if(from_index == -1) {
                from_index = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(plist->priv->file_list), NULL);
                from_index -= num_added - 1;
            }
            xfmedia_playlist_renumber_file_list(plist->priv->file_list,
                    from_index, -1);
            g_list_free(uris);
            
            xfmedia_playlist_treeview_scroll_cb(NULL, plist);
            //xfmedia_playlist_shuffle_add_entries(plist, from_index, num_added);
        }
        
        drag_success = TRUE;
    } else if(data->target == _STRING) {
        gchar *filename = data->data;
        gint from_index = -1;
        
        if(gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(plist->priv->treeview),
                    x, y, &dest_path, &drop_pos))
        {
            if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                    &dest_itr, dest_path))
            {
                gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list),
                        &dest_itr, PLAYLIST_COL_NUM, &from_index, -1);
                if(drop_pos == GTK_TREE_VIEW_DROP_BEFORE
                        || drop_pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE)
                {
                    from_index--;
                }
            }
            gtk_tree_path_free(dest_path);
        }
        
        if(filename) {
            gchar *title_utf8;
            
            if(!g_ascii_strncasecmp(filename, "file://", 7))
                filename += 7;
            
            title_utf8 = xfmedia_filename_to_name(filename);
            xfmedia_playlist_insert_entry(plist, from_index, title_utf8, -1,
                    filename, FALSE);
            g_free(title_utf8);
            
            xfmedia_playlist_renumber_file_list(plist->priv->file_list,
                    from_index, -1);
            xfmedia_playlist_treeview_scroll_cb(NULL, plist);
            //xfmedia_playlist_shuffle_add_entries(plist, from_index, 1);
            drag_success = TRUE;
        }
    }
    
    gtk_drag_finish(drag_context, drag_success, FALSE, time);
}

static gboolean
xfmedia_playlist_external_menu_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    
    if(evt->button == 3 && plist->priv->external_menu) {
        gtk_menu_popup(GTK_MENU(plist->priv->external_menu), NULL, NULL, NULL,
                NULL, evt->button, evt->time);
        return TRUE;
    }
    
    return FALSE;
}

static void
xfmedia_playlist_jtf_activate_cb(GtkTreePath *activated_path,
        gpointer user_data)
{
    XfMediaPlaylist *plist = user_data;
    gint *indices;
    
    indices = gtk_tree_path_get_indices(activated_path);
    g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_ENTRY_ACTIVATED], 0, indices[0]);
}
    



/******
 * public api
 ******/

GtkWidget *
xfmedia_playlist_new()
{
    XfMediaPlaylist *plist;
    GtkWidget *topvbox, *sw, *hbox, *treeview, *btn, *img, *evtbox, *spacer,
        *menu, *mi;
    GdkPixbuf *pix;
    GtkListStore *ls;
    GtkTreeViewColumn *col;
    GtkCellRenderer *render;
    GtkTooltips *ttips;
    gint iw, ih;
    const GtkTargetEntry tv_targets[] = {
        { "PLAYLIST_ITEM", GTK_TARGET_SAME_WIDGET, DROP_TGT_REORDER },
        { "text/uri-list", 0, DROP_TGT_URILIST },
        { "STRING", 0, DROP_TGT_STRING },
    };
    
    plist = g_object_new(XFMEDIA_TYPE_PLAYLIST, NULL);
    topvbox = GTK_WIDGET(plist);
    gtk_box_set_spacing(GTK_BOX(topvbox), BORDER);
    
    ttips = plist->priv->ttips = gtk_tooltips_new();
    g_object_ref(G_OBJECT(ttips));
    gtk_object_sink(GTK_OBJECT(ttips));
    
    sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_NEVER,
            GTK_POLICY_ALWAYS);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
            GTK_SHADOW_ETCHED_IN);
    gtk_widget_show(sw);
    gtk_box_pack_start(GTK_BOX(topvbox), sw, TRUE, TRUE, 0);
    
    plist->priv->file_list = ls = gtk_list_store_new(PLAYLIST_N_COLS,
            G_TYPE_INT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
            G_TYPE_INT, G_TYPE_BOOLEAN);
    
    plist->priv->treeview = treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(ls));
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), TRUE);
    gtk_tree_view_set_headers_clickable(GTK_TREE_VIEW(treeview), FALSE);
    gtk_tree_view_set_reorderable(GTK_TREE_VIEW(treeview), TRUE);
    gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), TRUE);
    gtk_tree_view_set_enable_search(GTK_TREE_VIEW(treeview), FALSE);
    
    render = gtk_cell_renderer_text_new();
    col = gtk_tree_view_column_new_with_attributes(_("#"), render, "text",
            PLAYLIST_COL_NUM, NULL);
    gtk_tree_view_column_set_resizable(col, FALSE);
    gtk_tree_view_column_set_cell_data_func(col, render,
            xfmedia_playlist_cell_render_title, plist, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    render = exo_cell_renderer_ellipsized_text_new();
	if(gtk_major_version == 2 && gtk_minor_version >= 6)
		g_object_set(G_OBJECT(render), "ellipsize", EXO_PANGO_ELLIPSIZE_END, NULL);
	else {
		g_object_set(G_OBJECT(render), "ellipsize", EXO_PANGO_ELLIPSIZE_END,
				"ellipsize-set", TRUE, NULL);
	}
    col = gtk_tree_view_column_new_with_attributes(_("Title"), render, "text",
            PLAYLIST_COL_TITLE, NULL);
    g_object_set(G_OBJECT(col), "resizable", TRUE, "expand", TRUE, NULL);
    gtk_tree_view_column_set_cell_data_func(col, render,
            xfmedia_playlist_cell_render_title, plist, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    render = gtk_cell_renderer_text_new();
    col = gtk_tree_view_column_new_with_attributes(_("Length"), render, "text",
            PLAYLIST_COL_LENGTH_S, NULL);
    gtk_tree_view_column_set_resizable(col, FALSE);
    gtk_tree_view_column_set_cell_data_func(col, render,
            xfmedia_playlist_cell_render_title, plist, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    plist->priv->tree_sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
    gtk_tree_selection_set_mode(plist->priv->tree_sel, GTK_SELECTION_MULTIPLE);
    gtk_widget_show(treeview);
    gtk_container_add(GTK_CONTAINER(sw), treeview);
    gtk_widget_add_events(treeview, GDK_BUTTON_PRESS|GDK_KEY_PRESS);
    g_signal_connect(G_OBJECT(treeview), "button-press-event",
            G_CALLBACK(xfmedia_playlist_treeview_buttonpress_cb), plist);
    g_signal_connect(G_OBJECT(treeview), "key-press-event",
            G_CALLBACK(xfmedia_playlist_treeview_keypress_cb), plist);
    g_signal_connect(G_OBJECT(gtk_tree_view_get_vadjustment(GTK_TREE_VIEW(treeview))),
            "value-changed", G_CALLBACK(xfmedia_playlist_treeview_scroll_cb), plist);
    gtk_tree_view_enable_model_drag_source(GTK_TREE_VIEW(treeview),
            GDK_BUTTON1_MASK, tv_targets, 3, GDK_ACTION_MOVE);
    gtk_tree_view_enable_model_drag_dest(GTK_TREE_VIEW(treeview), tv_targets,
            3, GDK_ACTION_COPY);
    g_signal_connect(G_OBJECT(treeview), "drag-data-received",
            G_CALLBACK(xfmedia_playlist_treeview_drag_rcv_cb), plist);
    g_signal_connect(G_OBJECT(treeview), "drag-data-get",
            G_CALLBACK(xfmedia_playlist_treeview_drag_get_cb), plist);
    
    evtbox = gtk_event_box_new();
    gtk_event_box_set_visible_window(GTK_EVENT_BOX(evtbox), FALSE);
    gtk_widget_show(evtbox);
    gtk_box_pack_start(GTK_BOX(topvbox), evtbox, FALSE, FALSE, 0);
    gtk_widget_add_events(evtbox, GDK_BUTTON_PRESS);
    g_signal_connect(G_OBJECT(evtbox), "button-press-event",
            G_CALLBACK(xfmedia_playlist_external_menu_cb), plist);
    
    hbox = gtk_hbox_new(FALSE, BORDER-4);
    gtk_widget_show(hbox);
    gtk_container_add(GTK_CONTAINER(evtbox), hbox);
    
    img = gtk_image_new_from_stock(GTK_STOCK_ADD, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_toggle_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled",
            G_CALLBACK(xfmedia_playlist_add_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Add entry to playlist"), NULL);
    
    xfmedia_playlist_create_add_menu(plist);
    
    img = gtk_image_new_from_stock(GTK_STOCK_REMOVE, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playlist_remove_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Remove entry from playlist"), NULL);
    
    img = gtk_image_new_from_stock(GTK_STOCK_GO_UP, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playlist_go_up_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Move selected playlist entry up"), NULL);
    
    img = gtk_image_new_from_stock(GTK_STOCK_GO_DOWN, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playlist_go_down_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Move selected playlist entry down"), NULL);
    
    spacer = gtk_alignment_new(0, 0, 0, 0);
    gtk_widget_show(spacer);
    gtk_box_pack_start(GTK_BOX(hbox), spacer, TRUE, TRUE, 0);
    
    gtk_icon_size_lookup(GTK_ICON_SIZE_SMALL_TOOLBAR, &iw, &ih);
    
    pix = xfce_icon_theme_load(icon_theme, "media-shuffle", iw);
    if(!pix)
        pix = xfce_icon_theme_load(icon_theme, "stock_media-shuffle", iw);
    if(!pix)
        pix = xfce_icon_theme_load(icon_theme, "xfmedia-shuffle", iw);
    if(pix) {
        img = gtk_image_new_from_pixbuf(pix);
        gtk_widget_show(img);
        g_object_unref(G_OBJECT(pix));
        plist->priv->shuffle_btn = btn = gtk_toggle_button_new();
        gtk_container_add(GTK_CONTAINER(btn), img);
    } else
        plist->priv->shuffle_btn = btn = gtk_toggle_button_new_with_label("S");
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled",
            G_CALLBACK(xfmedia_playlist_shuffle_toggle_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Randomize play order"), NULL);
    
    pix = xfce_icon_theme_load(icon_theme, "repeat", iw);
    if(!pix)
        pix = xfce_icon_theme_load(icon_theme, "stock_repeat", iw);
    if(!pix)
        pix = xfce_icon_theme_load(icon_theme, "xfmedia-repeat", iw);
    if(pix) {
        img = gtk_image_new_from_pixbuf(pix);
        gtk_widget_show(img);
        g_object_unref(G_OBJECT(pix));
        plist->priv->repeat_btn = btn = gtk_toggle_button_new();
        gtk_container_add(GTK_CONTAINER(btn), img);
    } else
        plist->priv->repeat_btn = btn = gtk_toggle_button_new_with_label("R");
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled",
            G_CALLBACK(xfmedia_playlist_repeat_toggle_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Repeat after playlist finishes"), NULL);
    
    spacer = gtk_alignment_new(0, 0, 0, 0);
    //gtk_widget_set_size_request(spacer, 25, -1);
    gtk_widget_show(spacer);
    gtk_box_pack_start(GTK_BOX(hbox), spacer, TRUE, TRUE, 0);
    
    img = gtk_image_new_from_stock(GTK_STOCK_SAVE_AS, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playlist_save_as_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Save current playlist"), NULL);
    
    img = gtk_image_new_from_stock(GTK_STOCK_OPEN, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playlist_open_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Open a saved playlist"), NULL);
    
    img = gtk_image_new_from_stock(GTK_STOCK_NEW, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playlist_new_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Start a new playlist"), NULL);
    
    plist->priv->popup_menu = menu = gtk_menu_new();
    gtk_widget_show(menu);
    gtk_icon_size_lookup(GTK_ICON_SIZE_MENU, &iw, &ih);
    
    img = gtk_image_new_from_stock(GTK_STOCK_REMOVE, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    mi = gtk_image_menu_item_new_with_mnemonic(_("_Remove Entry"));
    gtk_widget_show(mi);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_playlist_remove_cb), plist);
    
    img = gtk_image_new_from_stock(GTK_STOCK_GO_UP, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    mi = gtk_image_menu_item_new_with_mnemonic(_("Move _Up"));
    gtk_widget_show(mi);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_playlist_go_up_cb), plist);
    
    img = gtk_image_new_from_stock(GTK_STOCK_GO_DOWN, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    mi = gtk_image_menu_item_new_with_mnemonic(_("Move _Down"));
    gtk_widget_show(mi);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_playlist_go_down_cb), plist);
    
    return GTK_WIDGET(plist);
}

guint
xfmedia_playlist_append_entry(XfMediaPlaylist *plist, const gchar *title,
        gint length, const gchar *filename, gboolean metadata_loaded)
{
    GtkTreeIter itr;
    gint index;
    gchar length_str[32];
    
    index = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(plist->priv->file_list), NULL);
    
    gtk_list_store_append(plist->priv->file_list, &itr);
    
    if(length >= 0)
        g_snprintf(length_str, 32, "%02d:%02d", length/60, length%60);
    
    gtk_list_store_set(plist->priv->file_list, &itr,
            PLAYLIST_COL_NUM, index + 1,
            PLAYLIST_COL_TITLE, title,
            PLAYLIST_COL_LENGTH, length,
            PLAYLIST_COL_FILENAME, filename,
            PLAYLIST_COL_LENGTH_S, (length >= 0) ? length_str : "??:??",
            PLAYLIST_COL_METADATA_LOADED, metadata_loaded,
            -1);
    
    xfmedia_playlist_shuffle_add_entries(plist, -1, 1);
    g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_ENTRY_ADDED], 0, index);
    
    return index;
}

guint
xfmedia_playlist_insert_entry(XfMediaPlaylist *plist, guint index,
        const gchar *title, gint length, const gchar *filename,
        gboolean metadata_loaded)
{
    GtkTreeIter itr;
    gchar length_str[32];
    
    gtk_list_store_insert(plist->priv->file_list, &itr, index);
    
    if(length >= 0)
        g_snprintf(length_str, 32, "%02d:%02d", length/60, length%60);
    
    gtk_list_store_set(plist->priv->file_list, &itr,
            PLAYLIST_COL_NUM, index + 1,
            PLAYLIST_COL_TITLE, title,
            PLAYLIST_COL_LENGTH, length,
            PLAYLIST_COL_FILENAME, filename,
            PLAYLIST_COL_LENGTH_S, (length >= 0) ? length_str : "??:??",
            PLAYLIST_COL_METADATA_LOADED, metadata_loaded,
            -1);
    
    xfmedia_playlist_shuffle_add_entries(plist, index, 1);
    g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_ENTRY_ADDED], 0, index);
    
    return index;
}

void
xfmedia_playlist_modify_entry(XfMediaPlaylist *plist, guint index,
        const gchar *title, gint length, const gchar *filename)
{
    GtkTreeIter itr;
    
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    if(!gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(plist->priv->file_list),
            &itr, NULL, index))
    {
        return;
    }
    
    if(title) {
        gtk_list_store_set(plist->priv->file_list, &itr,
                PLAYLIST_COL_TITLE, title, -1);
    }
    
    if(length >= 0) {
        gchar *length_str = g_strdup_printf("%02d:%02d", length/60, length%60);
        gtk_list_store_set(plist->priv->file_list, &itr,
                PLAYLIST_COL_LENGTH, length,
                PLAYLIST_COL_LENGTH_S, length_str, -1);
        g_free(length_str);
    }
    
    if(filename) {
        gtk_list_store_set(plist->priv->file_list, &itr,
                PLAYLIST_COL_FILENAME, filename, -1);
    }
    
    g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_ENTRY_CHANGED], 0, index);
}

gboolean
xfmedia_playlist_remove_entry(XfMediaPlaylist *plist, guint index)
{
    gchar *filename = NULL;
    GtkTreeIter itr;
    
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), FALSE);
    
    if(!gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(plist->priv->file_list),
            &itr, NULL, index))
    {
        return FALSE;
    }
    
    gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
            PLAYLIST_COL_FILENAME, &filename, -1);
    
    gtk_list_store_remove(plist->priv->file_list, &itr);
    xfmedia_playlist_shuffle_remove_entries(plist, index, 1);
    
    g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_ENTRY_REMOVED], 0,
            filename ? filename : "");
    
    if(filename)
        g_free(filename);
    
    return TRUE;
}

void
xfmedia_playlist_clear(XfMediaPlaylist *plist)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    gtk_list_store_clear(plist->priv->file_list);
    g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_CLEARED], 0);
}


gboolean
xfmedia_playlist_get(XfMediaPlaylist *plist, guint index, gchar **title,
        gint *length, gchar **filename)
{
    GtkTreeIter itr;
    
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist) && (title || length || filename), FALSE);
    
    if(!gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(plist->priv->file_list),
            &itr, NULL, index))
    {
        return FALSE;
    }
    
    if(title)
        *title = NULL;
    if(length)
        *length = -1;
    if(filename)
        *filename = NULL;
    
    if(title && length && filename) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_TITLE, title,
                PLAYLIST_COL_LENGTH, length,
                PLAYLIST_COL_FILENAME, filename, -1);
    } else if(title && length) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_TITLE, title,
                PLAYLIST_COL_LENGTH, length, -1);
    } else if(title && filename) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_TITLE, title,
                PLAYLIST_COL_FILENAME, filename, -1);
    } else if(length && filename) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_LENGTH, length,
                PLAYLIST_COL_FILENAME, filename, -1);
    } else if(title) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_TITLE, title, -1);
    } else if(length) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_LENGTH, length, -1);
    } else if(filename) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_FILENAME, filename, -1);
    } else
        return FALSE;
    
    return TRUE;
}

guint
xfmedia_playlist_get_n_entries(XfMediaPlaylist *plist)
{
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), 0);
    
    return gtk_tree_model_iter_n_children(GTK_TREE_MODEL(plist->priv->file_list), NULL);
}

gint
xfmedia_playlist_get_selected(XfMediaPlaylist *plist)
{
    gint *indices, index;
    GList *selected;
    
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), -1);
    
    selected = gtk_tree_selection_get_selected_rows(plist->priv->tree_sel, NULL);
    if(!selected)
        return -1;
    
    indices = gtk_tree_path_get_indices((GtkTreePath *)selected->data);
    index = indices[0];
    
    g_list_foreach(selected, (GFunc)gtk_tree_path_free, NULL);
    g_list_free(selected);
    
    return index;
}

void
xfmedia_playlist_set_bold_entry(XfMediaPlaylist *plist,
        XfMediaPlaylistEntryRef *ref)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist) && xfmedia_playlist_entry_ref_valid(ref));
    
    if(plist->priv->bold_ref)
        xfmedia_playlist_entry_ref_destroy(plist->priv->bold_ref);
    
    plist->priv->bold_ref = xfmedia_playlist_entry_ref_copy(ref);
}
void
xfmedia_playlist_set_italic_entry(XfMediaPlaylist *plist,
        XfMediaPlaylistEntryRef *ref)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist) && xfmedia_playlist_entry_ref_valid(ref));
    
    if(plist->priv->italic_ref)
        xfmedia_playlist_entry_ref_destroy(plist->priv->italic_ref);
    
    plist->priv->italic_ref = xfmedia_playlist_entry_ref_copy(ref);
}

void
xfmedia_playlist_set_metadata_loaded(XfMediaPlaylist *plist, guint index,
        gboolean metadata_loaded)
{
    GtkTreeIter itr;
    
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    if(!gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(plist->priv->file_list),
            &itr, NULL, index))
    {
        return;
    }
    
    gtk_list_store_set(plist->priv->file_list, &itr,
            PLAYLIST_COL_METADATA_LOADED, metadata_loaded, -1);
}

gboolean
xfmedia_playlist_get_metadata_loaded(XfMediaPlaylist *plist, guint index)
{
    GtkTreeIter itr;
    gboolean metadata_loaded = FALSE;
    
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), FALSE);
    
    if(!gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(plist->priv->file_list),
            &itr, NULL, index))
    {
        return FALSE;
    }
    
    gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
            PLAYLIST_COL_METADATA_LOADED, &metadata_loaded, -1);
    
    return metadata_loaded;
}    

void
xfmedia_playlist_context_menu_prepend_item(XfMediaPlaylist *plist,
        GtkMenuItem *menu_item)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    gtk_menu_shell_prepend(GTK_MENU_SHELL(plist->priv->popup_menu),
            GTK_WIDGET(menu_item));
}

void
xfmedia_playlist_context_menu_append_item(XfMediaPlaylist *plist,
        GtkMenuItem *menu_item)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    gtk_menu_shell_append(GTK_MENU_SHELL(plist->priv->popup_menu),
            GTK_WIDGET(menu_item));
}

gboolean
xfmedia_playlist_is_dirty(XfMediaPlaylist *plist)
{
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), FALSE);
    return plist->priv->is_dirty;
}

void
xfmedia_playlist_set_shuffle_state(XfMediaPlaylist *plist, gboolean state)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    g_signal_handlers_block_by_func(G_OBJECT(plist->priv->shuffle_btn),
            G_CALLBACK(xfmedia_playlist_shuffle_toggle_cb), plist);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plist->priv->shuffle_btn), state);
    g_signal_handlers_unblock_by_func(G_OBJECT(plist->priv->shuffle_btn),
            G_CALLBACK(xfmedia_playlist_shuffle_toggle_cb), plist);
    xfmedia_playlist_shuffle_unset_all(plist);
}
    
gboolean
xfmedia_playlist_get_shuffle_state(XfMediaPlaylist *plist)
{
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), FALSE);
    
    return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(plist->priv->shuffle_btn));
}

void
xfmedia_playlist_set_repeat_state(XfMediaPlaylist *plist, gboolean state)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    g_signal_handlers_block_by_func(G_OBJECT(plist->priv->repeat_btn),
            G_CALLBACK(xfmedia_playlist_repeat_toggle_cb), plist);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plist->priv->repeat_btn), state);
    g_signal_handlers_unblock_by_func(G_OBJECT(plist->priv->repeat_btn),
            G_CALLBACK(xfmedia_playlist_repeat_toggle_cb), plist);
}

gboolean
xfmedia_playlist_get_repeat_state(XfMediaPlaylist *plist)
{
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), FALSE);
    
    return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(plist->priv->repeat_btn));
}

void
xfmedia_playlist_scroll_to_index(XfMediaPlaylist *plist, gint index,
        gdouble align)
{
    GtkTreePath *path;
    
    path = gtk_tree_path_new_from_indices(index, -1);
    gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(plist->priv->treeview), path,
                NULL, TRUE, align, 0.0);
    gtk_tree_path_free(path);
}

gboolean
xfmedia_playlist_get_visible_range(XfMediaPlaylist *plist, guint *start,
        guint *end)
{
    GtkTreeView *treeview;
    GdkRectangle rect;
    gint cell_x = 0, cell_y = 0;
    gint wx = 0, wy = 0;
    GtkTreeIter itr;
    GtkTreePath *path_start = NULL, *path_end = NULL;
    gboolean path_start_good = FALSE, path_end_good = FALSE, ret = FALSE;
    
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist) && start && end, FALSE);
    
    if(!GTK_WIDGET_REALIZED(plist->priv->treeview))
        return FALSE;
    treeview = GTK_TREE_VIEW(plist->priv->treeview);
    
    /* get visible rectangle */
    gtk_tree_view_get_visible_rect(treeview, &rect);
    
    /* get a tree path and iter for the bottommost visible item */
    gtk_tree_view_tree_to_widget_coords(treeview, rect.x,
            rect.y+rect.height-1, &wx, &wy);
    if(gtk_tree_view_get_path_at_pos(treeview, wx, wy, &path_end, NULL,
            &cell_x, &cell_y))
    {
        if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                &itr, path_end))
        {
            path_end_good = TRUE;
        }
    }
    
    /* get a tree path and iter for the topmost visible item */
    gtk_tree_view_tree_to_widget_coords(treeview, rect.x, rect.y, &wx, &wy);
    if(gtk_tree_view_get_path_at_pos(treeview, wx, wy, &path_start, NULL,
            &cell_x, &cell_y))
    {
        if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                &itr, path_start))
        {
            path_start_good = TRUE;
        }
    }
    
    if(path_start_good && path_end_good) {
        gint *i_start, *i_end;
        
        i_start = gtk_tree_path_get_indices(path_start);
        i_end = gtk_tree_path_get_indices(path_end);
        
        *start = i_start[0];
        *end = i_end[0];
        
        ret = TRUE;
    } else if(path_start_good) {
        GtkTreePath *path_1 = gtk_tree_path_new_first();
        GdkRectangle bg_rect = { 0, 0, 0, 0 };
        
        gtk_tree_view_get_background_area(GTK_TREE_VIEW(plist->priv->treeview),
                path_1, NULL, &bg_rect);
        gtk_tree_path_free(path_1);
        
        if(bg_rect.height > 0) {
            /* move back a bit and try to see where the last item is */
            gint y_coord = rect.y + rect.height - 1 - bg_rect.height;
            while(y_coord > rect.y) {
                gtk_tree_view_tree_to_widget_coords(treeview, rect.x,
                        y_coord, &wx, &wy);
                if(gtk_tree_view_get_path_at_pos(treeview, wx, wy, &path_end,
                        NULL, &cell_x, &cell_y))
                {
                    if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                            &itr, path_end))
                    {
                        path_end_good = TRUE;
                        break;
                    }
                }
                y_coord -= bg_rect.height;
            }
        }
        
        if(path_end_good) {
            gint *i_start, *i_end;
        
            i_start = gtk_tree_path_get_indices(path_start);
            i_end = gtk_tree_path_get_indices(path_end);
            
            *start = i_start[0];
            *end = i_end[0];
            
            ret = TRUE;
        }
    }
    
    if(path_start)
        gtk_tree_path_free(path_start);
    if(path_end)
        gtk_tree_path_free(path_end);
    
    return ret;
}

static gint
plist_sort_string(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
        gpointer user_data)
{
    gint ret, type = GPOINTER_TO_UINT(user_data);
    gchar *str_a = NULL, *str_b = NULL;
    
    switch(type) {
        case XFMEDIA_PLAYLIST_SORT_TITLE:
            gtk_tree_model_get(model, a, PLAYLIST_COL_TITLE, &str_a, -1);
            gtk_tree_model_get(model, b, PLAYLIST_COL_TITLE, &str_b, -1);
            break;
        case XFMEDIA_PLAYLIST_SORT_FILENAME:
            gtk_tree_model_get(model, a, PLAYLIST_COL_FILENAME, &str_a, -1);
            gtk_tree_model_get(model, b, PLAYLIST_COL_FILENAME, &str_b, -1);
            break;
        default:
            g_return_val_if_reached(0);
    }
    
    if(!str_a && !str_b)
        ret = 0;
    else if(!str_a) {
        ret = -1;
        g_free(str_b);
    } else if(!str_b) {
        ret = 1;
        g_free(str_a);
    } else {
        ret = g_utf8_collate(str_a, str_b);
        g_free(str_a);
        g_free(str_b);
    }
    
    return ret;
}

static void
plist_sort_random(XfMediaPlaylist *plist)
{
    gint total, *new_order, i, tmp;
    gchar *order_map;
    
    total = xfmedia_playlist_get_n_entries(plist);
    if(total < 2)
        return;
    new_order = g_malloc(sizeof(gint)*total);
    order_map = g_malloc0(sizeof(gchar)*total);
    
    for(i = 0; i < total; i++) {
        do {
#ifdef HAVE_SRANDOM
            tmp = random() % total;
#else
            tmp = rand() % total;
#endif
        } while(order_map[tmp]);
        
        order_map[tmp] = 0xff;
        new_order[i] = tmp;
    }
    
    gtk_list_store_reorder(plist->priv->file_list, new_order);
    
    g_free(new_order);
    g_free(order_map);
}

void
xfmedia_playlist_sort(XfMediaPlaylist *plist, XfMediaPlaylistSortType type)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    switch(type) {
        case XFMEDIA_PLAYLIST_SORT_TITLE:
            gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(plist->priv->file_list),
                    PLAYLIST_COL_TITLE, plist_sort_string,
                    GUINT_TO_POINTER(type), NULL);
            gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(plist->priv->file_list),
                    PLAYLIST_COL_TITLE, GTK_SORT_ASCENDING);
            /* then ditch the sort column so we don't continually sort */
            gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(plist->priv->file_list),
                    GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID, GTK_SORT_ASCENDING);
            break;
        
        case XFMEDIA_PLAYLIST_SORT_FILENAME:
            gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(plist->priv->file_list),
                    PLAYLIST_COL_FILENAME, plist_sort_string,
                    GUINT_TO_POINTER(type), NULL);
            gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(plist->priv->file_list),
                    PLAYLIST_COL_FILENAME, GTK_SORT_ASCENDING);
            /* then ditch the sort column so we don't continually sort */
            gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(plist->priv->file_list),
                    GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID, GTK_SORT_ASCENDING);
            break;
        
        case XFMEDIA_PLAYLIST_SORT_RANDOM:
            plist_sort_random(plist);
            break;
        
        default:
            g_return_if_reached();
    }
    
    xfmedia_playlist_renumber_file_list(plist->priv->file_list, 0, -1);
    xfmedia_playlist_treeview_scroll_cb(NULL, plist);
}


void
xfmedia_playlist_set_external_menu(XfMediaPlaylist *plist,
        GtkMenu *menu)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist) && GTK_IS_MENU(menu));
    
    plist->priv->external_menu = GTK_WIDGET(menu);
}

void
xfmedia_playlist_augment_add_menu(XfMediaPlaylist *plist,
        GtkMenuItem *menu_item)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    gtk_menu_shell_prepend(GTK_MENU_SHELL(plist->priv->add_menu),
            GTK_WIDGET(menu_item));
}

void
xfmedia_playlist_trigger_add_file(XfMediaPlaylist *plist)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    xfmedia_playlist_add_file_cb(GTK_WIDGET(plist), plist);
}

void
xfmedia_playlist_trigger_add_directory(XfMediaPlaylist *plist)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    xfmedia_playlist_add_dir_cb(GTK_WIDGET(plist), plist);
}

void
xfmedia_playlist_trigger_add_url(XfMediaPlaylist *plist)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    xfmedia_playlist_add_url_cb(GTK_WIDGET(plist), plist);
}

void
xfmedia_playlist_trigger_new_playlist(XfMediaPlaylist *plist)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    xfmedia_playlist_new_cb(GTK_WIDGET(plist), plist);
}

void
xfmedia_playlist_trigger_open_playlist(XfMediaPlaylist *plist)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    xfmedia_playlist_open_cb(GTK_WIDGET(plist), plist);
}

void
xfmedia_playlist_trigger_save_playlist(XfMediaPlaylist *plist)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    xfmedia_playlist_save_as_cb(GTK_WIDGET(plist), plist);
}

void
xfmedia_playlist_trigger_jump_to_file(XfMediaPlaylist *plist)
{
    GtkWidget *toplevel;
    
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    toplevel = gtk_widget_get_toplevel(GTK_WIDGET(plist));
    xfmedia_jump_to_file_show(GTK_WINDOW(toplevel),
            GTK_TREE_MODEL(plist->priv->file_list), PLAYLIST_COL_TITLE,
            xfmedia_playlist_jtf_activate_cb, plist);
}

gboolean
xfmedia_playlist_has_focus(XfMediaPlaylist *plist)
{
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), FALSE);
    return GTK_WIDGET_HAS_FOCUS(plist->priv->treeview);
}

/*
 * playlist shuffle-play management
 */

gboolean
xfmedia_playlist_shuffle_check(XfMediaPlaylist *plist, guint index)
{
    g_return_val_if_fail(plist->priv->shuffle_bitmap
            && index < plist->priv->shuffle_bitmap_len, FALSE);
    
    return (plist->priv->shuffle_bitmap[index] == 0x0);
}

gboolean
xfmedia_playlist_shuffle_check_set(XfMediaPlaylist *plist, guint index)
{
    g_return_val_if_fail(plist->priv->shuffle_bitmap
            && index < plist->priv->shuffle_bitmap_len, FALSE);
    
    if(plist->priv->shuffle_bitmap[index])
        return FALSE;
    
    return (plist->priv->shuffle_bitmap[index] = 0x1);  /* yes, a single '=' is correct */
}

void
xfmedia_playlist_shuffle_unset(XfMediaPlaylist *plist, guint index)
{
    g_return_if_fail(index < plist->priv->shuffle_bitmap_len);
    
    plist->priv->shuffle_bitmap[index] = 0x0;
}
    
void
xfmedia_playlist_shuffle_unset_all(XfMediaPlaylist *plist)
{
    if(!plist->priv->shuffle_bitmap)
        return;
    
    memset(plist->priv->shuffle_bitmap, 0x0, plist->priv->shuffle_bitmap_len);
}

void
xfmedia_playlist_shuffle_add_entries(XfMediaPlaylist *plist, gint start,
        guint len)
{
    g_return_if_fail(len > 0);
    
    plist->priv->shuffle_bitmap = g_realloc(plist->priv->shuffle_bitmap,
            plist->priv->shuffle_bitmap_len + len);
    
    if(start == -1 || start >= plist->priv->shuffle_bitmap_len)
        memset(plist->priv->shuffle_bitmap+plist->priv->shuffle_bitmap_len, 0x0, len);
    else {
        memmove(plist->priv->shuffle_bitmap+start+len, plist->priv->shuffle_bitmap+start, len);
        memset(plist->priv->shuffle_bitmap+start, 0x0, len);
    }
        
    plist->priv->shuffle_bitmap_len += len;
}

void
xfmedia_playlist_shuffle_remove_entries(XfMediaPlaylist *plist, guint start,
        gint len)
{
    g_return_if_fail(len != 0 && start < plist->priv->shuffle_bitmap_len);
    
    if(len < 0)
        len = plist->priv->shuffle_bitmap_len - start;
    
    if(len == plist->priv->shuffle_bitmap_len) {
        g_free(plist->priv->shuffle_bitmap);
        plist->priv->shuffle_bitmap = NULL;
    } else {
        memmove(plist->priv->shuffle_bitmap+start, plist->priv->shuffle_bitmap+start+len, len);
        plist->priv->shuffle_bitmap = g_realloc(plist->priv->shuffle_bitmap,
                plist->priv->shuffle_bitmap_len - len);
    }
    
    plist->priv->shuffle_bitmap_len -= len;
}

void
xfmedia_playlist_shuffle_swap_entries(XfMediaPlaylist *plist, guint a, guint b)
{
    guint8 tmp;
    
    g_return_if_fail(a < plist->priv->shuffle_bitmap_len
            && b < plist->priv->shuffle_bitmap_len && a != b);
    
    tmp = plist->priv->shuffle_bitmap[a];
    plist->priv->shuffle_bitmap[a] = plist->priv->shuffle_bitmap[b];
    plist->priv->shuffle_bitmap[b] = tmp;
}

void
xfmedia_playlist_shuffle_init_entries(XfMediaPlaylist *plist, guint len)
{
    if(len == 0) {
        if(plist->priv->shuffle_bitmap) {
            g_free(plist->priv->shuffle_bitmap);
            plist->priv->shuffle_bitmap = NULL;
            plist->priv->shuffle_bitmap_len = 0;
        }
    } else {
        plist->priv->shuffle_bitmap = g_realloc(plist->priv->shuffle_bitmap, len);
        memset(plist->priv->shuffle_bitmap, 0x0, len);
        plist->priv->shuffle_bitmap_len = len;
    }
}


/* row references */

XfMediaPlaylistEntryRef *
xfmedia_playlist_entry_ref_new_for_index(XfMediaPlaylist *plist, guint index)
{
	XfMediaPlaylistEntryRef *ref;
	GtkTreePath *path;
	
	g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist)
			&& index < xfmedia_playlist_get_n_entries(plist), NULL);
	
	ref = g_new0(XfMediaPlaylistEntryRef, 1);
	ref->plist = plist;
	
	path = gtk_tree_path_new_from_indices(index, -1);
	ref->rref = gtk_tree_row_reference_new(GTK_TREE_MODEL(plist->priv->file_list),
			path);
	gtk_tree_path_free(path);
	
	return ref;
}

XfMediaPlaylistEntryRef *
xfmedia_playlist_entry_ref_copy(XfMediaPlaylistEntryRef *ref)
{
    XfMediaPlaylistEntryRef *new_ref;
    
    g_return_val_if_fail(ref, NULL);
    
    new_ref = g_new(XfMediaPlaylistEntryRef, 1);
    new_ref->plist = ref->plist;
    new_ref->rref = gtk_tree_row_reference_copy(ref->rref);
    
    return new_ref;
}

gboolean
xfmedia_playlist_entry_ref_valid(XfMediaPlaylistEntryRef *ref)
{
    g_return_val_if_fail(ref, FALSE);
    
    return gtk_tree_row_reference_valid(ref->rref);
}

gint
xfmedia_playlist_entry_ref_get_index(XfMediaPlaylistEntryRef *ref)
{
	GtkTreePath *path;
	gint *indices, index;
	
	g_return_val_if_fail(ref, -1);
	
	if(!gtk_tree_row_reference_valid(ref->rref))
		return -1;
	
	path = gtk_tree_row_reference_get_path(ref->rref);
	indices = gtk_tree_path_get_indices(path);
	index = indices[0];
	gtk_tree_path_free(path);
	
	return index;
}

void
xfmedia_playlist_entry_ref_destroy(XfMediaPlaylistEntryRef *ref)
{
	g_return_if_fail(ref);
	
	gtk_tree_row_reference_free(ref->rref);
	g_free(ref);
}
