/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_MMAP
#include <sys/mman.h>
#ifndef MAP_FILE
#define MAP_FILE (0)
#endif
#endif

#include <libxfce4util/libxfce4util.h>

#include <xfmedia/xfmedia-settings.h>
#include "xfmedia-common.h"
#include "playlist-files.h"

#define PLAYLISTFILE             "xfmedia/playlist.m3u"
#define XFMEDIA_CONFIG           "xfmedia/settings.xml"
#define XFMEDIA_CONFIG_NEW       "xfmedia/settings.xml.new"
#define XFMEDIA_CONFIG_WRITEFREQ 7500

typedef struct _XfMediaConfigEntry
{
    gchar *key;
    gint type;
    union {
        gchar *s;
        gint i;
        gboolean b;
        gdouble f;
    } value;
} XfMediaConfigEntry;

enum {
    XFMEDIA_SETTING_BRANCH = 0,
    XFMEDIA_SETTING_STRING,
    XFMEDIA_SETTING_INT,
    XFMEDIA_SETTING_BOOL,
    XFMEDIA_SETTING_FLOAT,
    XFMEDIA_N_SETTING_TYPES
};

static gboolean xfmedia_settings_xml_parse(const gchar *filename);

static GHashTable *settings_hash = NULL;
static GList *settings_list = NULL;
static gboolean settings_write_pending = FALSE;

gint
xfmedia_settings_glist_compare(gconstpointer a, gconstpointer b)
{
    const XfMediaConfigEntry *xa = a, *xb = b;
    
    return strcmp(xa->key, xb->key);
}

static void
ght_val_destroy(gpointer data)
{
    XfMediaConfigEntry *entry = data;
    
    g_free(entry->key);
    if(entry->type == XFMEDIA_SETTING_STRING && entry->value.s)
        g_free(entry->value.s);
    g_free(entry);
}

static inline gint
count_slashes(const gchar *str)
{
    gint slashes = 1;
    gchar *p;
    
    p = strstr(str, "/");
    if(!p)
        return 0;
    while((p=strstr(p+1, "/")))
        slashes++;
    
    return slashes;
}

static void
settings_write_xml_tree(FILE *fp, GList *prefs)
{
    gint depth = 1, new_depth;
    gchar tabs[128], *name_esc, *val_esc, *p;
    GList *l;
    XfMediaConfigEntry *entry;
    
    if(depth > 127)
        depth = 127;
    memset(tabs, '\t', depth);
    tabs[depth] = 0;
    
    for(l = prefs; l; l = l->next) {
        entry = l->data;
        if(!entry)
            continue;
        
        new_depth = count_slashes(entry->key);
        if(new_depth == 0)
            continue;
        
        if(new_depth < depth) {
            while(depth-- > new_depth) {
                tabs[depth] = 0;
                fprintf(fp, "%s</branch>\n", tabs);
            }
            depth = new_depth;
        } else if(new_depth > depth) {
            tabs[depth] = '\t';
            tabs[new_depth] = 0;
            depth = new_depth;
        }
        
        p = g_strrstr(entry->key, "/");
        if(!p)
            continue;
        name_esc = g_markup_escape_text(p+1, strlen(p+1));
        
        switch(entry->type) {
            case XFMEDIA_SETTING_BRANCH:
                fprintf(fp, "%s<branch name=\"%s\">\n", tabs, name_esc);
                break;
            
            case XFMEDIA_SETTING_STRING:
                val_esc = g_markup_escape_text(entry->value.s, strlen(entry->value.s));
                fprintf(fp, "%s<setting name=\"%s\" type=\"%d\" value=\"%s\" />\n",
                        tabs, name_esc, entry->type, val_esc);
                g_free(val_esc);
                break;
            
            case XFMEDIA_SETTING_INT:
                fprintf(fp, "%s<setting name=\"%s\" type=\"%d\" value=\"%d\" />\n",
                        tabs, name_esc, entry->type, entry->value.i);
                break;
            
            case XFMEDIA_SETTING_BOOL:
                fprintf(fp, "%s<setting name=\"%s\" type=\"%d\" value=\"%s\" />\n",
                        tabs, name_esc, entry->type,
                        entry->value.b ? "true" : "false");
                break;
            
            case XFMEDIA_SETTING_FLOAT:
                fprintf(fp, "%s<setting name=\"%s\" type=\"%d\" value=\"%f\" />\n",
                        tabs, name_esc, entry->type, entry->value.f);
                break;
            
            default:
                g_warning("Unknown config type for key %s", entry->key);
                break;
        }
        
        g_free(name_esc);
    }
    
    while(--depth > 0) {
        tabs[depth] = 0;
        fprintf(fp, "%s</branch>\n", tabs);
    }
}

static gboolean
settings_write(gpointer user_data)
{
    gchar *cfgfile = NULL, *cfgfile_new = NULL;
    FILE *fp = NULL;
    
    settings_write_pending = FALSE;
    
    g_return_val_if_fail(settings_hash, FALSE);
    
    cfgfile = xfce_resource_save_location(XFCE_RESOURCE_CONFIG,
            XFMEDIA_CONFIG, TRUE);
    if(!cfgfile)
        goto cleanup;
    cfgfile_new = xfce_resource_save_location(XFCE_RESOURCE_CONFIG,
            XFMEDIA_CONFIG_NEW, TRUE);
    if(!cfgfile_new)
        goto cleanup;
    
    fp = fopen(cfgfile_new, "w");
    if(!fp)
        goto cleanup;
    
    fputs("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n\n<branch name=\"/\">\n", fp);
    settings_write_xml_tree(fp, settings_list);
    fputs("</branch>\n", fp);
    
    fclose(fp);
    
    if(g_file_test(cfgfile, G_FILE_TEST_EXISTS)) {
        if(!unlink(cfgfile)) {
            if(!link(cfgfile_new, cfgfile)) {
                if(unlink(cfgfile_new))
                    g_warning("Unable to delete temp file when writing settings");
            } else
                g_warning("Unable to copy temp file to settings file");
        } else
            g_warning("Unable to remove old config file");
    } else {
        if(!link(cfgfile_new, cfgfile)) {
            if(unlink(cfgfile_new))
                g_warning("Unable to delete temp file when writing settings");
        } else
            g_warning("Unable to copy temp file to settings file");
    }
    
    cleanup:
    
    if(cfgfile)
        g_free(cfgfile);
    if(cfgfile_new)
        g_free(cfgfile_new);
    
    return FALSE;
}    

static void
settings_sched_write()
{
    g_return_if_fail(settings_hash);
    
    if(settings_write_pending)
        return;
    
    settings_write_pending = TRUE;
    g_timeout_add(XFMEDIA_CONFIG_WRITEFREQ, settings_write, NULL);
}

static void
settings_ensure_branch(const gchar *branch)
{
    gchar buf[2048], *p;
    XfMediaConfigEntry *entry;
    
    g_strlcpy(buf, branch, 2048);
    for(;;) {
        entry = g_hash_table_lookup(settings_hash, branch);
        if(!entry) {
            entry = g_new(XfMediaConfigEntry, 1);
            entry->key = g_strdup(branch);
            entry->type = XFMEDIA_SETTING_BRANCH;
            g_hash_table_insert(settings_hash, entry->key, entry);
            settings_list = g_list_insert_sorted(settings_list,
                    entry, xfmedia_settings_glist_compare);
        } else
            break;
        
        p = g_strrstr(branch, "/");
        if(!p)
            break;
        else
            *p = 0;
    }
}

static XfMediaConfigEntry *
settings_get_entry_for_set(const gchar *key)
{
    XfMediaConfigEntry *entry;
    gchar *branch = g_strdup(key), *p;
    
    p = g_strrstr(branch, "/");
    if(p) {
        *p = 0;
        settings_ensure_branch(branch);
    }
    g_free(branch);
    
    entry = g_hash_table_lookup(settings_hash, key);
    if(!entry) {
        entry = g_new0(XfMediaConfigEntry, 1);
        entry->key = g_strdup(key);
        settings_list = g_list_insert_sorted(settings_list, entry,
                xfmedia_settings_glist_compare);
        g_hash_table_insert(settings_hash, entry->key, entry);
    }
    if(entry->type == XFMEDIA_SETTING_STRING && entry->value.s)
        g_free(entry->value.s);
    
    return entry;
}

static XfMediaConfigEntry *
settings_get_entry_for_add(const gchar *key)
{
    XfMediaConfigEntry *entry;
    gchar *branch, *p;
    
    entry = g_hash_table_lookup(settings_hash, key);
    if(entry)
        return NULL;
    
    branch = g_strdup(key);
    p = g_strrstr(branch, "/");
    if(p) {
        *p = 0;
        settings_ensure_branch(branch);
    }
    g_free(branch);
    
    entry = g_new0(XfMediaConfigEntry, 1);
    entry->key = g_strdup(key);
    settings_list = g_list_insert_sorted(settings_list, entry,
            xfmedia_settings_glist_compare);
    g_hash_table_insert(settings_hash, entry->key, entry);
        
    return entry;
}

void
xfmedia_settings_cleanup()
{
    g_return_if_fail(settings_hash);
    
    settings_write(NULL);
    
    g_hash_table_destroy(settings_hash);
    settings_hash = NULL;
    if(settings_list) {
        g_list_free(settings_list);
        settings_list = NULL;
    }
}

void
xfmedia_settings_add_string(const gchar *key,
        const gchar *value)
{
    XfMediaConfigEntry *entry;
    
    g_return_if_fail(key && settings_hash);
    
    entry = settings_get_entry_for_add(key);
    if(!entry)
        return;
    
    entry->type = XFMEDIA_SETTING_STRING;
    entry->value.s = g_strdup(value);
    
    settings_sched_write();
}

void
xfmedia_settings_add_int(const gchar *key, gint value)
{
    XfMediaConfigEntry *entry;
    
    g_return_if_fail(key && settings_hash);
    
    entry = settings_get_entry_for_add(key);
    if(!entry)
        return;
    
    entry->type = XFMEDIA_SETTING_INT;
    entry->value.i = value;
    
    settings_sched_write();
}

void
xfmedia_settings_add_bool(const gchar *key, gboolean value)
{
    XfMediaConfigEntry *entry;
    
    g_return_if_fail(key && settings_hash);
    
    entry = settings_get_entry_for_add(key);
    if(!entry)
        return;
    
    entry->type = XFMEDIA_SETTING_BOOL;
    entry->value.b = value;
    
    settings_sched_write();
}

void
xfmedia_settings_add_float(const gchar *key, gdouble value)
{
    XfMediaConfigEntry *entry;
    
    g_return_if_fail(key && settings_hash);
    
    entry = settings_get_entry_for_add(key);
    if(!entry)
        return;
    
    entry->type = XFMEDIA_SETTING_FLOAT;
    entry->value.f = value;
    
    settings_sched_write();
}

G_CONST_RETURN gchar *
xfmedia_settings_get_string(const gchar *key)
{
    XfMediaConfigEntry *entry;
    
    g_return_val_if_fail(key && settings_hash, NULL);
    
    entry = g_hash_table_lookup(settings_hash, key);
    if(entry && entry->type == XFMEDIA_SETTING_STRING)
        return entry->value.s;
    
    return NULL;
}

gint
xfmedia_settings_get_int(const gchar *key)
{
    XfMediaConfigEntry *entry;
    
    g_return_val_if_fail(key && settings_hash, 0);    
    
    entry = g_hash_table_lookup(settings_hash, key);
    if(entry && entry->type == XFMEDIA_SETTING_INT)
        return entry->value.i;
    
    return 0;
}

gboolean
xfmedia_settings_get_bool(const gchar *key)
{
    XfMediaConfigEntry *entry;
    
    g_return_val_if_fail(key && settings_hash, FALSE);
    
    entry = g_hash_table_lookup(settings_hash, key);
    if(entry && entry->type == XFMEDIA_SETTING_BOOL)
        return entry->value.b;
    
    return FALSE;
}

gdouble
xfmedia_settings_get_float(const gchar *key)
{
    XfMediaConfigEntry *entry;
    
    g_return_val_if_fail(key && settings_hash, 0.0);
    
    entry = g_hash_table_lookup(settings_hash, key);
    if(entry && entry->type == XFMEDIA_SETTING_FLOAT)
        return entry->value.f;
    
    return 0.0;
}

void
xfmedia_settings_set_string(const gchar *key, const gchar *value)
{
    XfMediaConfigEntry *entry;
    
    g_return_if_fail(key && settings_hash);
    
    entry = settings_get_entry_for_set(key);    
    entry->type = XFMEDIA_SETTING_STRING;
    entry->value.s = g_strdup(value);
    settings_sched_write();
}

void
xfmedia_settings_set_int(const gchar *key, gint value)
{
    XfMediaConfigEntry *entry;
    
    g_return_if_fail(key && settings_hash);
    
    entry = settings_get_entry_for_set(key);    
    entry->type = XFMEDIA_SETTING_INT;
    entry->value.i = value;
    settings_sched_write();
}

void
xfmedia_settings_set_bool(const gchar *key, gboolean value)
{
    XfMediaConfigEntry *entry;
    
    g_return_if_fail(key && settings_hash);
    
    entry = settings_get_entry_for_set(key);    
    entry->type = XFMEDIA_SETTING_BOOL;
    entry->value.b = value;
    settings_sched_write();
}

void
xfmedia_settings_set_float(const gchar *key, gdouble value)
{
    XfMediaConfigEntry *entry;
    
    g_return_if_fail(key && settings_hash);
    
    entry = settings_get_entry_for_set(key);    
    entry->type = XFMEDIA_SETTING_FLOAT;
    entry->value.f = value;
    settings_sched_write();
}

void
xfmedia_settings_init()
{
    gchar *cfgfile;
    
    g_return_if_fail(settings_hash == NULL && settings_list == NULL);
    
    settings_list = NULL;
    settings_hash = g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
            (GDestroyNotify)ght_val_destroy);
    
    cfgfile = xfce_resource_lookup(XFCE_RESOURCE_CONFIG, XFMEDIA_CONFIG);
    if(cfgfile) {
        if(!xfmedia_settings_xml_parse(cfgfile)) {
            g_hash_table_destroy(settings_hash);
            settings_hash = g_hash_table_new_full(g_str_hash, g_str_equal,
                    NULL, (GDestroyNotify)ght_val_destroy);
            if(settings_list) {
                g_list_free(settings_list);
                settings_list = NULL;
            }
        }
        g_free(cfgfile);
    }
    
    xfmedia_settings_add_int("/xfmedia/general/height_large", -1);
    xfmedia_settings_add_int("/xfmedia/general/height_small", -1);
    xfmedia_settings_add_int("/xfmedia/general/width", -1);
    xfmedia_settings_add_int("/xfmedia/general/pos_x", -1);
    xfmedia_settings_add_int("/xfmedia/general/pos_y", -1);
    xfmedia_settings_add_bool("/xfmedia/general/window_sticky", FALSE);
    xfmedia_settings_add_bool("/xfmedia/general/window_above", FALSE);
    xfmedia_settings_add_bool("/xfmedia/general/show_audio_vis", FALSE);
    xfmedia_settings_add_string("/xfmedia/general/selected_vis", "goom");
    xfmedia_settings_add_bool("/xfmedia/general/dock_autohide", FALSE);
    xfmedia_settings_add_int("/xfmedia/general/vwin_pos_x", -1);
    xfmedia_settings_add_int("/xfmedia/general/vwin_pos_y", -1);
    
#if 0
    xfmedia_settings_add_int("/xfmedia/general/eq_band_0", 0);
    xfmedia_settings_add_int("/xfmedia/general/eq_band_1", 0);
    xfmedia_settings_add_int("/xfmedia/general/eq_band_2", 0);
    xfmedia_settings_add_int("/xfmedia/general/eq_band_3", 0);
    xfmedia_settings_add_int("/xfmedia/general/eq_band_4", 0);
    xfmedia_settings_add_int("/xfmedia/general/eq_band_5", 0);
    xfmedia_settings_add_int("/xfmedia/general/eq_band_6", 0);
    xfmedia_settings_add_int("/xfmedia/general/eq_band_7", 0);
    xfmedia_settings_add_int("/xfmedia/general/eq_band_8", 0);
    xfmedia_settings_add_int("/xfmedia/general/eq_band_9", 0);
#endif
    
    xfmedia_settings_add_string("/xfmedia/plugins/libxine/audio_out", "auto");
    xfmedia_settings_add_string("/xfmedia/plugins/libxine/video_out", "auto");
    
    xfmedia_settings_add_bool("/xfmedia/playlist/playlist_expanded", FALSE);
    xfmedia_settings_add_string("/xfmedia/playlist/playlist_format", "(%p) %t");
    xfmedia_settings_add_int("/xfmedia/playlist/playlist_position", -1);
    xfmedia_settings_add_bool("/xfmedia/playlist/shuffle", FALSE);
    xfmedia_settings_add_bool("/xfmedia/playlist/repeat", FALSE);
    xfmedia_settings_add_bool("/xfmedia/playlist/auto_advance", TRUE);
    xfmedia_settings_add_bool("/xfmedia/playlist/save_on_exit_always", FALSE);
    xfmedia_settings_add_int("/xfmedia/playlist/metadata_load_impetus", 0);
    xfmedia_settings_add_bool("/xfmedia/playlist/repeat_one", FALSE);
    
    xfmedia_settings_add_int("/xfmedia/jtf/win_x", 25);
    xfmedia_settings_add_int("/xfmedia/jtf/win_y", 25);
    xfmedia_settings_add_int("/xfmedia/jtf/win_width", 425);
    xfmedia_settings_add_int("/xfmedia/jtf/win_height", 200);
    
    xfmedia_settings_add_int("/xfmedia/mediamarks/manager_x", 25);
    xfmedia_settings_add_int("/xfmedia/mediamarks/manager_y", 25);
    xfmedia_settings_add_int("/xfmedia/mediamarks/manager_width", 615);
    xfmedia_settings_add_int("/xfmedia/mediamarks/manager_height", 400);
}

static inline gchar *
get_default_playlist_file()
{
    gchar *playlist_file;
    
    playlist_file = xfce_resource_save_location(XFCE_RESOURCE_CACHE,
            PLAYLISTFILE, TRUE);
    
    return playlist_file;
}    

gboolean
xfmedia_settings_load_playlist(XfMediaMainwin *mwin)
{
    gchar *file = get_default_playlist_file();
    gboolean ret = FALSE;
    
    if(file) {
        xfmedia_playlist_clear(mwin->plist);
        ret = xfmedia_playlists_load(mwin->plist, file);
        g_free(file);
    }
    
    return ret;
}

gboolean
xfmedia_settings_save_playlist(XfMediaMainwin *mwin)
{
    gchar *file = get_default_playlist_file();
    gboolean ret = FALSE;
    
    if(file) {
        ret = xfmedia_playlists_save(mwin->plist, file);
        g_free(file);
    }
    
    return ret;
}


/* XML file parser */

struct ConfigParserState
{
    gchar cur_path[2048];
};

static void
config_xml_start(GMarkupParseContext *context, const gchar *element_name,
        const gchar **attribute_names, const gchar **attribute_values,
        gpointer user_data, GError **error)
{
    struct ConfigParserState *state = user_data;
    gint i;
    const gchar *name = NULL, *type = NULL, *value = NULL;
    XfMediaConfigEntry *entry;
    
    for(i = 0; attribute_names[i]; i++) {
        if(!strcmp(attribute_names[i], "name"))
            name = attribute_values[i];
        else if(!strcmp(attribute_names[i], "type"))
            type = attribute_values[i];
        else if(!strcmp(attribute_names[i], "value"))
            value = attribute_values[i];
    }
    
    if(!strcmp(element_name, "branch")) {
        if(!name)
            return;
        
        if(!strcmp(name, "/")) {
            state->cur_path[0] = '/';
            state->cur_path[1] = 0;
            return;
        }
        
        entry = g_new(XfMediaConfigEntry, 1);
        entry->type = XFMEDIA_SETTING_BRANCH;
        entry->key = g_strconcat(state->cur_path,
                (!strcmp(state->cur_path, "/")?"":"/"), name, NULL);
        
        g_hash_table_insert(settings_hash, entry->key, entry);
        settings_list = g_list_insert_sorted(settings_list, entry,
                xfmedia_settings_glist_compare);
        
        g_strlcpy(state->cur_path, entry->key, 2048);
    } else if(!strcmp(element_name, "setting")) {
        if(!name || !type || !value)
            return;
        entry = g_new(XfMediaConfigEntry, 1);
        entry->type = atoi(type);
        if(entry->type <= 0 || entry->type >= XFMEDIA_N_SETTING_TYPES) {
            g_free(entry);
            return;
        }
        entry->key = g_strconcat(state->cur_path,
                (!strcmp(state->cur_path, "/")?"":"/"), name, NULL);
        switch(entry->type) {
            case XFMEDIA_SETTING_STRING:
                entry->value.s = g_strdup(value);
                break;
            case XFMEDIA_SETTING_INT:
                entry->value.i = atoi(value);
                break;
            case XFMEDIA_SETTING_BOOL:
                entry->value.b = (!strcmp(value, "true") ? TRUE : FALSE);
                break;
            case XFMEDIA_SETTING_FLOAT:
                entry->value.f = g_ascii_strtod(value, NULL);
                break;
            default:
                break;
        }
        
        g_hash_table_insert(settings_hash, entry->key, entry);
        settings_list = g_list_insert_sorted(settings_list, entry,
                xfmedia_settings_glist_compare);
    }
}

static void
config_xml_end(GMarkupParseContext *context, const gchar *element_name,
        gpointer user_data, GError **error)
{
    struct ConfigParserState *state = user_data;
    gchar *p;
    
    if(!strcmp(element_name, "branch")) {
        p = g_strrstr(state->cur_path, "/");
        if(p)
            *p = 0;
    }
}

static gboolean
xfmedia_settings_xml_parse(const gchar *filename)
{
    gchar *file_contents = NULL;
    GMarkupParseContext *gpcontext = NULL;
    int fd = -1;
    struct stat st;
    GMarkupParser gmparser = {
        config_xml_start,
        config_xml_end,
        NULL,
        NULL,
        NULL
    };
    struct ConfigParserState state;
    gboolean ret = FALSE;
    GError *err = NULL;
#ifdef HAVE_MMAP
    void *maddr = NULL;
#endif
    
    if(stat(filename, &st) < 0)
        return TRUE;
    
    fd = open(filename, O_RDONLY, 0);
    if(fd < 0)
        goto cleanup;
    
#ifdef HAVE_MMAP
    maddr = mmap(NULL, st.st_size, PROT_READ, MAP_FILE|MAP_SHARED, fd, 0);
    if(maddr)
        file_contents = maddr;
    else {
#endif
        file_contents = g_malloc(st.st_size);
        if(!file_contents)
            goto cleanup;
        
        if(read(fd, file_contents, st.st_size) != st.st_size)
            goto cleanup;
#ifdef HAVE_MMAP
    }
#endif
    
    *state.cur_path = 0;
    
    gpcontext = g_markup_parse_context_new(&gmparser, 0, &state, NULL);

    if(!g_markup_parse_context_parse(gpcontext, file_contents, st.st_size, &err)) {
        g_warning("Error parsing Xfmusic4 settings file (%d): %s\n",
                err->code, err->message);
        g_error_free(err);
        goto cleanup;
    }

    if(!g_markup_parse_context_end_parse(gpcontext, &err)) {
        g_warning("Error finishing settings file parsing (%d): %s\n",
                err->code, err->message);
        g_error_free(err);
        goto cleanup;
    }
    
    ret = TRUE;
    
    cleanup:
    
    if(gpcontext)
        g_markup_parse_context_free(gpcontext);
#ifdef HAVE_MMAP
    if(maddr)
        munmap(maddr, st.st_size);
    else if(file_contents)
#endif
        g_free(file_contents);
    if(fd > -1)
        close(fd);
    
    return ret;
}
