/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#include <libxfcegui4/netk-trayicon.h>

#include "trayicon.h"
#include "xfmedia-common.h"
#include <xfmedia/xfmedia-settings.h>

struct _XfMediaTrayIcon
{
    GtkWidget *tray_icon;
    GtkWidget *event_box;
    GtkMenu *menu;
    GtkTooltips *tooltip;
    gchar *tooltip_str;
    XfMediaMainwin *mwin;
};

static GtkWidget *_xfmedia_tray_icon_create(GdkScreen *gscreen, gpointer cb_data);

static gboolean
xfmedia_tray_icon_button_press_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
    XfMediaTrayIcon *tray_icon = user_data;
    XfMediaMainwin *mwin = tray_icon->mwin;
    
    if(evt->button == 1) {
        if(GTK_WIDGET_VISIBLE(mwin->window))
            gtk_widget_hide(mwin->window);
        else if(GTK_WIDGET_VISIBLE(mwin->autohide_window))
            gtk_widget_hide(mwin->autohide_window);
        else {
            gtk_window_move(GTK_WINDOW(mwin->window),
                    xfmedia_settings_get_int("/xfmedia/general/pos_x"),
                    xfmedia_settings_get_int("/xfmedia/general/pos_y"));
            if(xfmedia_settings_get_bool("/xfmedia/general/window_sticky"))
                gtk_window_stick(GTK_WINDOW(mwin->window));
            else
                gtk_window_unstick(GTK_WINDOW(mwin->window));
            if(xfmedia_settings_get_bool("/xfmedia/general/window_above"))
                gtk_window_set_keep_above(GTK_WINDOW(mwin->window), TRUE);
            gtk_widget_show(mwin->window);
        }
        return TRUE;
    } else if(evt->button == 3) {
        if(tray_icon->menu) {
            gtk_menu_popup(GTK_MENU(tray_icon->menu), NULL, NULL, NULL,
                    NULL, evt->button, evt->time);
        }
        return TRUE;
    }
    
    return FALSE;
}

static void
xfmedia_tray_icon_destroy_cb(GtkObject *object, gpointer user_data)
{
    XfMediaTrayIcon *tray_icon = user_data;
    GdkScreen *gscreen = gtk_widget_get_screen(GTK_WIDGET(object));
    
    tray_icon->tray_icon = _xfmedia_tray_icon_create(gscreen, tray_icon);
    if(tray_icon->tooltip_str) {
        gtk_tooltips_set_tip(tray_icon->tooltip, tray_icon->event_box,
                tray_icon->tooltip_str, NULL);
    }
}

static GtkWidget *
_xfmedia_tray_icon_create(GdkScreen *gscreen, gpointer cb_data)
{
    GtkWidget *tray_icon, *evtbox, *img;
    GdkPixbuf *pix, *tmp;
    gint iw, ih;
    
    tray_icon = netk_tray_icon_new(GDK_SCREEN_XSCREEN(gscreen));
    
    evtbox = gtk_event_box_new();
    gtk_widget_add_events(evtbox, GDK_BUTTON_PRESS);
    gtk_widget_show(evtbox);
    gtk_container_add(GTK_CONTAINER(tray_icon), evtbox);
    
    gtk_icon_size_lookup(GTK_ICON_SIZE_LARGE_TOOLBAR, &iw, &ih);
    pix = xfce_icon_theme_load(icon_theme, "xfmedia", iw);
    if(!pix) {
        tmp = gdk_pixbuf_from_pixdata(&xfmedia_icon, TRUE, NULL);
        if(tmp && gdk_pixbuf_get_width(tmp) == iw && gdk_pixbuf_get_height(tmp) == ih)
            pix = tmp;
        else if(tmp) {
            pix = gdk_pixbuf_scale_simple(tmp, iw, ih, GDK_INTERP_BILINEAR);
            g_object_unref(G_OBJECT(tmp));
        } else {
            pix = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, iw, ih);
            gdk_pixbuf_fill(pix, 0);
        }
    }
    
    img = gtk_image_new_from_pixbuf(pix);
    g_object_unref(G_OBJECT(pix));
    gtk_widget_show(img);
    gtk_container_add(GTK_CONTAINER(evtbox), img);
    
    g_signal_connect(G_OBJECT(evtbox), "button-press-event",
            G_CALLBACK(xfmedia_tray_icon_button_press_cb), cb_data);
    g_signal_connect(G_OBJECT(tray_icon), "destroy",
            G_CALLBACK(xfmedia_tray_icon_destroy_cb), cb_data);
    
    gtk_widget_show(tray_icon);
    
    return tray_icon;
}

XfMediaTrayIcon *
xfmedia_tray_icon_create(GdkScreen *gscreen, XfMediaMainwin *mwin,
        GtkMenu *menu)
{
    XfMediaTrayIcon *tray_icon;
    
    
    tray_icon = g_new0(XfMediaTrayIcon, 1);
    tray_icon->tray_icon = _xfmedia_tray_icon_create(gscreen, tray_icon);
    tray_icon->event_box = gtk_bin_get_child(GTK_BIN(tray_icon->tray_icon));
    tray_icon->mwin = mwin;
    tray_icon->menu = menu;
    tray_icon->tooltip = gtk_tooltips_new();
    g_object_ref(G_OBJECT(tray_icon->tooltip));
    gtk_object_sink(GTK_OBJECT(tray_icon->tooltip));
    
    return tray_icon;
}

void
xfmedia_tray_icon_set_tooltip(XfMediaTrayIcon *tray_icon, const gchar *tip)
{
    g_return_if_fail(tray_icon);
    
    if(tray_icon->tooltip_str)
        g_free(tray_icon->tooltip_str);
    tray_icon->tooltip_str = g_strdup(tip);
    
    gtk_tooltips_set_tip(tray_icon->tooltip, tray_icon->event_box, tip, NULL);
}

void
xfmedia_tray_icon_destroy(XfMediaTrayIcon *tray_icon)
{
    g_return_if_fail(tray_icon && NETK_IS_TRAY_ICON(tray_icon->tray_icon));
    
    g_signal_handlers_disconnect_by_func(G_OBJECT(tray_icon->tray_icon),
            xfmedia_tray_icon_destroy_cb, tray_icon);
    
    g_object_unref(G_OBJECT(tray_icon->tooltip));
    if(tray_icon->tooltip_str)
        g_free(tray_icon->tooltip_str);
    
    gtk_widget_destroy(tray_icon->tray_icon);
    g_free(tray_icon);
}
