/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gmodule.h>

#include <xfmedia/xfmedia-plugin.h>
#include "mainwin.h"

static void xfmedia_plugin_class_init(XfmediaPluginClass *klass);
static void xfmedia_plugin_init(XfmediaPlugin *plugin);
static void xfmedia_plugin_finalize(GObject *object);

enum {
    SIG_STREAM_LOADED = 0,
    SIG_STREAM_STARTED,
    SIG_STREAM_ENDED,
    SIG_SPEED_CHANGED,
    SIG_UNLOADING,
    PLIST_SIG_CLEARED,
    PLIST_SIG_OPENED,
    PLIST_SIG_SAVED,
    PLIST_SIG_SCROLLED,
    PLIST_SIG_ENTRY_ADDED,
    PLIST_SIG_ENTRY_CHANGED,
    PLIST_SIG_ENTRY_REMOVED,
    PLIST_SIG_ENTRY_ACTIVATED,
    PLIST_SIG_SHUFFLE_TOGGLED,
    PLIST_SIG_REPEAT_TOGGLED,
    XFMP_N_SIGNALS
};

static guint xfmp_signals[XFMP_N_SIGNALS] = { 0 };
static GObjectClass *parent_class = NULL;

struct _XfmediaPluginPriv
{
    gboolean enabled;
    
    gchar *name;
    gchar *description;
    gchar *version;
    gchar *author;
    gchar *website;
    gchar *license;
    gchar *date;
    
    gchar *icon_file;
    GdkPixbuf *icon;
    
    PluginConfigureFunc configure_func;
};


GType
xfmedia_plugin_get_type()
{
    static GType plugin_type = 0;
    
    if(!plugin_type) {
        static const GTypeInfo plugin_info = {
            sizeof(XfmediaPluginClass),
            NULL,  /* base_init */
            NULL,  /* base_finalize */
            (GClassInitFunc)xfmedia_plugin_class_init,
            NULL,  /* class_finalize */
            NULL,  /* class_data */
            sizeof(XfmediaPlugin),
            0,  /* n_preallocs */
            (GInstanceInitFunc)xfmedia_plugin_init,
        };
        
        plugin_type = g_type_register_static(G_TYPE_OBJECT,
                "XfmediaPlugin", &plugin_info, 0);
    }
    
    return plugin_type;
}

static void
xfmedia_plugin_class_init(XfmediaPluginClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    
    parent_class = g_type_class_peek_parent(klass);
    
    gobject_class->finalize = xfmedia_plugin_finalize;
    
    xfmp_signals[SIG_STREAM_LOADED] = g_signal_new("stream-loaded",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, stream_loaded), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    xfmp_signals[SIG_STREAM_STARTED] = g_signal_new("stream-started",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, stream_started), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    xfmp_signals[SIG_STREAM_ENDED] = g_signal_new("stream-ended",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, stream_ended), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);\
    
    xfmp_signals[SIG_SPEED_CHANGED] = g_signal_new("speed-changed",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, speed_changed), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    xfmp_signals[SIG_UNLOADING] = g_signal_new("unloading",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, unloading), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    
    xfmp_signals[PLIST_SIG_CLEARED] = g_signal_new("playlist-cleared",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, playlist_cleared), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    xfmp_signals[PLIST_SIG_OPENED] = g_signal_new("playlist-opened",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, playlist_opened), NULL, NULL,
            g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);
    
    xfmp_signals[PLIST_SIG_SAVED] = g_signal_new("playlist-saved",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, playlist_saved), NULL, NULL,
            g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);
    
    xfmp_signals[PLIST_SIG_SCROLLED] = g_signal_new("playlist-scrolled",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, playlist_scrolled), NULL,
            NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    xfmp_signals[PLIST_SIG_ENTRY_ADDED] = g_signal_new("playlist-entry-added",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, playlist_entry_added), NULL, NULL,
            g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);
    
    xfmp_signals[PLIST_SIG_ENTRY_CHANGED] = g_signal_new("playlist-entry-changed",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, playlist_entry_changed), NULL, NULL,
            g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);
    
    xfmp_signals[PLIST_SIG_ENTRY_REMOVED] = g_signal_new("playlist-entry-removed",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, playlist_entry_removed), NULL, NULL,
            g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);
    
    xfmp_signals[PLIST_SIG_ENTRY_ACTIVATED] = g_signal_new("playlist-entry-activated",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, playlist_entry_activated), NULL, NULL,
            g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);
    
    xfmp_signals[PLIST_SIG_SHUFFLE_TOGGLED] = g_signal_new("playlist-shuffle-toggled",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, playlist_shuffle_toggled), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    xfmp_signals[PLIST_SIG_REPEAT_TOGGLED] = g_signal_new("playlist-repeat-toggled",
            XFMEDIA_TYPE_PLUGIN, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPluginClass, playlist_repeat_toggled), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
}

static void
xfmedia_plugin_finalize(GObject *object)
{
    XfmediaPlugin *plugin = XFMEDIA_PLUGIN(object);
    XfmediaPluginPriv *priv;
    
    g_return_if_fail(plugin);
    priv = plugin->priv;
    
    if(priv->name)
        g_free(priv->name);
    if(priv->description)
        g_free(priv->description);
    if(priv->version)
        g_free(priv->version);
    if(priv->author)
        g_free(priv->author);
    if(priv->website)
        g_free(priv->website);
    if(priv->license)
        g_free(priv->license);
    
    if(priv->icon_file)
        g_free(priv->icon_file);
    if(priv->icon)
        g_object_unref(G_OBJECT(priv->icon));
    
    g_free(priv);
    
    G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
xfmedia_plugin_init(XfmediaPlugin *plugin)
{
    plugin->priv = g_new0(XfmediaPluginPriv, 1);
    plugin->priv->enabled = TRUE;
}


XfmediaPlugin *
xfmedia_plugin_new()
{
    return g_object_new(XFMEDIA_TYPE_PLUGIN, NULL);
}

void
xfmedia_plugin_set_name(XfmediaPlugin *plugin, const gchar *name)
{
    g_return_if_fail(XFMEDIA_IS_PLUGIN(plugin) && name);
    
    if(plugin->priv->name)
        g_free(plugin->priv->name);
    plugin->priv->name = g_strdup(name);
}

G_CONST_RETURN gchar *
xfmedia_plugin_get_name(XfmediaPlugin *plugin)
{
    g_return_val_if_fail(XFMEDIA_IS_PLUGIN(plugin), NULL);
    
    return plugin->priv->name;
}

void
xfmedia_plugin_set_description(XfmediaPlugin *plugin,const gchar *description)
{
    g_return_if_fail(XFMEDIA_IS_PLUGIN(plugin) && description);
    
    if(plugin->priv->description)
        g_free(plugin->priv->description);
    plugin->priv->description = g_strdup(description);
}

G_CONST_RETURN gchar *
xfmedia_plugin_get_description(XfmediaPlugin *plugin)
{
    g_return_val_if_fail(XFMEDIA_IS_PLUGIN(plugin), NULL);
    
    return plugin->priv->description;
}

void
xfmedia_plugin_set_version(XfmediaPlugin *plugin, const gchar *version)
{
    g_return_if_fail(XFMEDIA_IS_PLUGIN(plugin) && version);
    
    if(plugin->priv->version)
        g_free(plugin->priv->version);
    plugin->priv->version = g_strdup(version);
}

G_CONST_RETURN gchar *
xfmedia_plugin_get_version(XfmediaPlugin *plugin)
{
    g_return_val_if_fail(XFMEDIA_IS_PLUGIN(plugin), NULL);
    
    return plugin->priv->version;
}

void
xfmedia_plugin_set_icon_file(XfmediaPlugin *plugin, const gchar *icon_file)
{
    g_return_if_fail(XFMEDIA_IS_PLUGIN(plugin));
    
    if(plugin->priv->icon) {
        g_object_unref(G_OBJECT(plugin->priv->icon));
        plugin->priv->icon = NULL;
    }
    
    if(plugin->priv->icon_file)
        g_free(plugin->priv->icon_file);
    plugin->priv->icon_file = g_strdup(icon_file);
}
    
G_CONST_RETURN gchar *
xfmedia_plugin_get_icon_file(XfmediaPlugin *plugin)
{
    g_return_val_if_fail(XFMEDIA_IS_PLUGIN(plugin), NULL);
    
    return plugin->priv->icon_file;
}

void
xfmedia_plugin_set_icon(XfmediaPlugin *plugin, GdkPixbuf *icon)
{
    g_return_if_fail(XFMEDIA_IS_PLUGIN(plugin));
    
    if(plugin->priv->icon_file) {
        g_free(plugin->priv->icon_file);
        plugin->priv->icon_file = NULL;
    }
    
    if(plugin->priv->icon)
        g_object_unref(G_OBJECT(plugin->priv->icon));
    plugin->priv->icon = g_object_ref(G_OBJECT(icon));
}

GdkPixbuf *
xfmedia_plugin_get_icon(XfmediaPlugin *plugin)
{
    g_return_val_if_fail(XFMEDIA_IS_PLUGIN(plugin), NULL);
    
    return plugin->priv->icon;
}

void
xfmedia_plugin_set_author(XfmediaPlugin *plugin, const gchar *author)
{
    g_return_if_fail(XFMEDIA_IS_PLUGIN(plugin) && author);
    
    if(plugin->priv->author)
        g_free(plugin->priv->author);
    plugin->priv->author = g_strdup(author);
}

G_CONST_RETURN gchar *
xfmedia_plugin_get_author(XfmediaPlugin *plugin)
{
    g_return_val_if_fail(XFMEDIA_IS_PLUGIN(plugin), NULL);
    
    return plugin->priv->author;
}

void
xfmedia_plugin_set_website(XfmediaPlugin *plugin, const gchar *website)
{
    g_return_if_fail(XFMEDIA_IS_PLUGIN(plugin) && website);
    
    if(plugin->priv->website)
        g_free(plugin->priv->website);
    plugin->priv->website = g_strdup(website);
}

G_CONST_RETURN gchar *
xfmedia_plugin_get_website(XfmediaPlugin *plugin)
{
    g_return_val_if_fail(XFMEDIA_IS_PLUGIN(plugin), NULL);
    
    return plugin->priv->website;
}

void
xfmedia_plugin_set_license(XfmediaPlugin *plugin, const gchar *license)
{
    g_return_if_fail(XFMEDIA_IS_PLUGIN(plugin) && license);
    
    if(plugin->priv->license)
        g_free(plugin->priv->license);
    plugin->priv->license = g_strdup(license);
}

G_CONST_RETURN gchar *
xfmedia_plugin_get_license(XfmediaPlugin *plugin)
{
    g_return_val_if_fail(XFMEDIA_IS_PLUGIN(plugin), NULL);
    
    return plugin->priv->license;
}

void
xfmedia_plugin_set_date(XfmediaPlugin *plugin, const gchar *date)
{
    g_return_if_fail(XFMEDIA_IS_PLUGIN(plugin) && date);
    
    if(plugin->priv->date)
        g_free(plugin->priv->date);
    plugin->priv->date = g_strdup(date);
}

G_CONST_RETURN gchar *
xfmedia_plugin_get_date(XfmediaPlugin *plugin)
{
    g_return_val_if_fail(XFMEDIA_IS_PLUGIN(plugin), NULL);
    
    return plugin->priv->date;
}

void
xfmedia_plugin_set_configure_func(XfmediaPlugin *plugin,
        PluginConfigureFunc func)
{
    g_return_if_fail(XFMEDIA_IS_PLUGIN(plugin));
    
    plugin->priv->configure_func = func;
}

PluginConfigureFunc
xfmedia_plugin_get_configure_func(XfmediaPlugin *plugin)
{
    g_return_val_if_fail(XFMEDIA_IS_PLUGIN(plugin), NULL);
    
    return plugin->priv->configure_func;
}

void
xfmedia_plugin_set_enabled(XfmediaPlugin *plugin, gboolean enabled)
{
    g_return_if_fail(XFMEDIA_IS_PLUGIN(plugin));
    
    if(enabled == plugin->priv->enabled)
        return;
    
    plugin->priv->enabled = enabled;
}
    
gboolean
xfmedia_plugin_get_enabled(XfmediaPlugin *plugin)
{
    g_return_val_if_fail(XFMEDIA_IS_PLUGIN(plugin), FALSE);
    
    return plugin->priv->enabled;
}

XfmediaPlaylist *
xfmedia_plugin_get_playlist(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    
    g_return_val_if_fail(XFMEDIA_IS_PLUGIN(plugin), NULL);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return mwin->plist;
}


/**************************
 * xfmedia internal stuff *
 **************************/

#define N_PLUGIN_DIRS 2
static GList *xfmedia_plugins = NULL;

void
xfmedia_plugins_load(gpointer mwin)  /* gpointer to avoid an include */
{
    gchar **plugin_dirs, fullpath[PATH_MAX];
    gint i;
    const gchar *file;
    
    plugin_dirs = g_malloc(sizeof(gchar *) * (N_PLUGIN_DIRS+1));
    plugin_dirs[0] = g_build_path(G_DIR_SEPARATOR_S, ".local", "lib",
            "xfmedia", "plugins", NULL);
    plugin_dirs[1] = g_build_path(G_DIR_SEPARATOR_S, LIBDIR, "xfmedia",
            "plugins", NULL);
    plugin_dirs[2] = NULL;
    
    for(i = 0; plugin_dirs[i]; i++) {
        GDir *dir = g_dir_open(plugin_dirs[i], 0, NULL);
        if(!dir)
            continue;
        
        while((file = g_dir_read_name(dir))) {
            GModule *gm;
            XfmediaPlugin *plugin;
            GError *err = NULL;
            gboolean (*xfmedia_plugin_check_version)(gint, gint, gint);
            gboolean (*xfmedia_plugin_get)(XfmediaPlugin *, GError **);
            
            if(!g_str_has_suffix(file, G_MODULE_SUFFIX))
                continue;
            
            g_snprintf(fullpath, PATH_MAX, "%s/%s", plugin_dirs[i], file);
            gm = g_module_open(fullpath, 0);
            if(!gm)
                continue;
            
            if(!g_module_symbol(gm, "xfmedia_plugin_check_version",
                                (gpointer)&xfmedia_plugin_check_version)
                    || !g_module_symbol(gm, "xfmedia_plugin_get",
                                        (gpointer)&xfmedia_plugin_get))
            {
                g_module_close(gm);
                continue;
            }
            
            if(!xfmedia_plugin_check_version(XFMEDIA_PLUGIN_API_MAJOR,
                    XFMEDIA_PLUGIN_API_MINOR, XFMEDIA_PLUGIN_API_MICRO))
            {
                g_module_close(gm);
                continue;
            }
            
            plugin = xfmedia_plugin_new();
            g_object_set_data(G_OBJECT(plugin), "__xfmedia-mainwin", mwin);
            if(!xfmedia_plugin_get(plugin, &err)) {
                g_object_unref(G_OBJECT(plugin));
                g_module_close(gm);
                if(err) {
                    g_warning("Failed to init plugin '%s' (%d): %s",
                            fullpath, err->code, err->message);
                    g_error_free(err);
                }
                continue;
            }
            
            g_object_set_data(G_OBJECT(plugin), "__xfmedia-plugin-gmodule", gm);
            xfmedia_plugins = g_list_prepend(xfmedia_plugins, plugin);
        }
    }
    
    g_strfreev(plugin_dirs);
}

GList *
xfmedia_plugins_enumerate()
{
    GList *newlist = NULL, *l;
    
    for(l = xfmedia_plugins; l; l = l->next)
        newlist = g_list_append(newlist, l->data);
    
    return newlist;
}

void
xfmedia_plugins_forward_signal_VOID(const gchar *signal_name)
{
    GList *l;
    
    for(l = xfmedia_plugins; l; l = l->next)
        g_signal_emit_by_name(G_OBJECT(l->data), signal_name, NULL);
}

void
xfmedia_plugins_forward_signal_STRING(const gchar *signal_name,
        const gchar *string)
{
    GList *l;
    
    for(l = xfmedia_plugins; l; l = l->next)
        g_signal_emit_by_name(G_OBJECT(l->data), signal_name, string, NULL);
}

void
xfmedia_plugins_forward_signal_INT(const gchar *signal_name, gint integer)
{
    GList *l;
    
    for(l = xfmedia_plugins; l; l = l->next)
        g_signal_emit_by_name(G_OBJECT(l->data), signal_name, integer, NULL);
}

void
xfmedia_plugins_unload()
{
    GList *l;
    
    for(l = xfmedia_plugins; l; l = l->next) {
        XfmediaPlugin *plugin = l->data;
        GModule *gm = g_object_get_data(G_OBJECT(plugin),
                "__xfmedia-plugin-gmodule");
        
        g_signal_emit(G_OBJECT(plugin), xfmp_signals[SIG_UNLOADING], 0);
        
        g_object_unref(G_OBJECT(plugin));
        g_module_close(gm);
    }
}
